/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#ifndef PEEKABOT_CLIENT_OCCUPANCY_GRID_3D_PROXY_HH_INCLUDED
#define PEEKABOT_CLIENT_OCCUPANCY_GRID_3D_PROXY_HH_INCLUDED


#include "ObjectProxy.hh"

#include <cstddef>
#include <boost/scoped_ptr.hpp>


namespace peekabot
{
    namespace client
    {
        /**
         * \brief Class used to feed data to
         * OccupancyGrid3DProxyBase::set_cells().
         *
         * \warning This API is EXPERIMENTAL.
         *
         * \remark This class is not thread safe.
         */
        class PEEKABOT_API OccupancySet3D
        {
            friend class OccupancyGrid3DProxyBase;

        public:
            OccupancySet3D();

            OccupancySet3D(const OccupancySet3D &other);

            ~OccupancySet3D();

            OccupancySet3D &operator=(const OccupancySet3D &other);

            /**
             * \brief Add a cell and its occupancy value to the set.
             *
             * \param x An x-coordinate in the cell to update.
             * \param y A y-coordinate in the cell to update.
             * \param z A z-coordinate in the cell to update.
             * \param belief The occupancy value of the cell, in the
             * range [0,1], where 1 means fully occupied.
             */
            void set_cell(float x, float y, float z, float belief);

            void clear();

            std::size_t size() const;

            bool empty() const;

        private:
            class Impl;
            boost::scoped_ptr<Impl> m_impl;
        };

        /**
         * \brief Base class for OccupancyGrid3DProxy.
         *
         * \warning This API is EXPERIMENTAL.
         */
        class PEEKABOT_API OccupancyGrid3DProxyBase : public ObjectProxyBase
        {
        public:
            OccupancyGrid3DProxyBase();

            OccupancyGrid3DProxyBase(OccupancyGrid3DProxyBase &p);

            /**
             * \brief Set/update occupancy values.
             *
             * Note that cell \em coordinates are used, not cell indices. See
             * OccupancyGrid3DProxy for more information.
             */
            DelayedDispatch set_cells(const OccupancySet3D &cells);

            /**
             * \brief Enable coloring of the grid voxels based on their height.
             *
             * \param z_min The height to which the lower end of the color
             * spectrum is mapped.
             * \param z_max The height to which the upper end of the color
             * spectrum is mapped.
             */
            DelayedDispatch enable_color_mapping(float z_min, float z_max);

            /**
             * \brief Disable height color mapping.
             */
            DelayedDispatch disable_color_mapping();
        };


        /**
         * \brief Proxy class used to add and manipulate 3D occupancy grids.
         *
         * Cells in the occupancy grid are addressed by coordinates, rather
         * than indices.
         *
         * \par Usage example
         *
         * \code
         * // Add an occupancy grid to the scene
         * peekabot::OccupancyGrid3DProxy og;
         * og.add(client, "root.og", 0.2);
         *
         * // Set a bunch of cells
         * peekabot::OccupancySet3D cells;
         * cells.set_cell(7.2, 9.8, 2.3, 0.5);
         * cells.set_cell(7.4, 9.8, 2,5, 0.4);
         * cells.set_cell(7.6, 10.0, 2.5, 0.55);
         * og.set_cells(cells);
         * \endcode
         */
        class PEEKABOT_API OccupancyGrid3DProxy :
            public OccupancyGrid3DProxyBase
        {
        public:
            OccupancyGrid3DProxy();

            OccupancyGrid3DProxy(OccupancyGrid3DProxy &p);

            OccupancyGrid3DProxy &operator=(const OccupancyGrid3DProxy &p);

            OccupancyGrid3DProxy &operator=(const OccupancyGrid3DProxyBase &p);

            /**
             * \copydoc ScalableProxy::assign(const ObjectProxyBase&)
             */
            DelayedDispatch assign(const ObjectProxyBase &p);

            /**
             * \copydoc ObjectProxy::assign(PeekabotClient&,const std::string&)
             */
            DelayedDispatch assign(
                PeekabotClient &client,
                const std::string &path);

            /**
             * \copydoc
             * ObjectProxy::assign(const ObjectProxyBase&,const std::string&)
             */
            DelayedDispatch assign(
                const ObjectProxyBase &parent,
                const std::string &rel_path);

            /**
             * \brief Add a 3D occupancy grid to the scene.
             *
             * Adds a occupancy grid to the scene at the given path and assigns
             * the proxy to point to it. If an object already exists at the
             * given path the outcome is governed by \a conflict_policy.
             *
             * \param path The path where the object will be added, including
             * the name of the object.
             * \param cell_size The length of the sides, in meters, of the
             * (square) occupancy grid cells.
             * \param conflict_policy Determines how name conflicts are handled.
             */
            DelayedDispatch add(
                PeekabotClient &client,
                const std::string &path,
                float cell_xy_size, float cell_z_size,
                NameConflictPolicy conflict_policy = AUTO_ENUMERATE_ON_CONFLICT);

            /**
             * \brief Add a 3D occupancy grid to the scene.
             *
             * Adds a occupancy grid with the given name under the the object
             * referred to by \a parent and assigns the proxy to point to it.
             * If an object already exists at the given path the outcome is
             * governed by \a conflict_policy.
             *
             * \param parent The parent under which the object will be added.
             * \param name The name to assign to the created object.
             * \param cell_size The length of the sides, in meters, of the
             * (square) occupancy grid cells.
             * \param conflict_policy Determines how name conflicts are handled.
             */
            DelayedDispatch add(
                const ObjectProxyBase &parent,
                const std::string &name,
                float cell_xy_size, float cell_z_size,
                NameConflictPolicy conflict_policy = AUTO_ENUMERATE_ON_CONFLICT);
        };
    }
}


#endif // PEEKABOT_CLIENT_OCCUPANCY_GRID_3D_PROXY_HH_INCLUDED
