/*
 * Copyright Staffan Gimåker 2008-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <stdexcept>
#include <boost/algorithm/string/case_conv.hpp>

#include "LineBasedProxy.hh"
#include "../PeekabotClient.hh"
#include "../../PropKeys.hh"
#include "../../ObjectTypes.hh"

#include "../../actions/Assign.hh"
#include "../../actions/MiniBundle.hh"
#include "../../actions/SetProp.hh"


using namespace peekabot;
using namespace peekabot::client;


//
// ------------------ LineBasedProxyBase implementation -------------------
//

LineBasedProxyBase::LineBasedProxyBase()
{
}


LineBasedProxyBase::LineBasedProxyBase(const LineBasedProxyBase &p)
    : ObjectProxyBase(p)
{
}

//

DelayedDispatch LineBasedProxyBase::set_line_width(float width)
{
    return DelayedDispatch(
        get_client_impl(),
        new SetProp(get_object_id(), LINE_WIDTH_PROP, width));
}

DelayedDispatch LineBasedProxyBase::set_line_style(LineStyle style, int factor)
{
    MiniBundle *b = new MiniBundle;
    b->add_action(new SetProp(get_object_id(), LINE_STYLE_PROP, style));
    b->add_action(
        new SetProp(
            get_object_id(), LINE_STIPPLE_FACTOR_PROP,
            boost::uint32_t(factor)));

    return DelayedDispatch(get_client_impl(), b);
}


DelayedDispatch LineBasedProxyBase::set_line_style(
    const std::string &style, int factor)
{
    std::string str = style;
    boost::to_lower(str);

    if( str == "solid" )
        return set_line_style(LINE_STYLE_SOLID, factor);
    else if( str == "dashed" )
        return set_line_style(LINE_STYLE_DASHED, factor);
    else if( str == "dotted" )
        return set_line_style(LINE_STYLE_DOTTED, factor);
    else if( str == "dash-dot" )
        return set_line_style(LINE_STYLE_DASH_DOT, factor);
    else if( str == "dash-dot-dot" )
        return set_line_style(LINE_STYLE_DASH_DOT_DOT, factor);
    else
        throw std::runtime_error("Invalid line style");
}


//
// ------------------ LineBasedProxy implementation -------------------
//


LineBasedProxy::LineBasedProxy()
{
}

LineBasedProxy::LineBasedProxy(const LineBasedProxyBase &p)
    : ObjectProxyBase(p),
      LineBasedProxyBase(p)
{
}


LineBasedProxy &LineBasedProxy::operator=(const LineBasedProxy &p)
{
    return *this = (LineBasedProxyBase &)p;
}


LineBasedProxy &LineBasedProxy::operator=(const LineBasedProxyBase &p)
{
    unchecked_assign(unchecked_get_client_impl(p), get_pseudonym(p));
    return *this;
}

DelayedDispatch LineBasedProxy::assign(const ObjectProxyBase &p)
{
    unchecked_assign(get_client_impl(p), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(get_object_id(p)),
                   get_object_id(), LINE_BASED_CLASS));
}


DelayedDispatch LineBasedProxy::assign(
    PeekabotClient &client,
    const std::string &path)
{
    unchecked_assign(get_client_impl(client), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(PathIdentifier(path), get_object_id(), LINE_BASED_CLASS));
}


DelayedDispatch LineBasedProxy::assign(
    const ObjectProxyBase &parent,
    const std::string &rel_path)
{
    unchecked_assign(get_client_impl(parent), allocate_pseudonym());

    return DelayedDispatch(
        get_client_impl(),
        new Assign(
            PathIdentifier(get_object_id(parent), rel_path),
            get_object_id(), LINE_BASED_CLASS));
}
