/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <cassert>
#include <boost/bind.hpp>

#include "ActionRecorder.hh"
#include "../PbarWriter.hh"


using namespace peekabot;
using namespace peekabot::client;


ActionRecorder::ActionRecorder(
    boost::shared_ptr<ClientImpl> client,
    const std::string &filename)
    : Transport(client),
      m_thread(0),
      m_stop_signal(false),
      m_t0(boost::posix_time::microsec_clock::local_time()),
      m_writer(0)
{
    m_ofs.open(filename.c_str(), std::ios::binary);
    if( !m_ofs )
        throw std::runtime_error(
            "Failed to open file '" + filename + "' for writing");

    m_writer = new PbarWriter(m_ofs, false);
    m_writer->flush();
    m_thread = new boost::thread(boost::bind(&ActionRecorder::run, this));
}


ActionRecorder::~ActionRecorder()
{
    assert( m_thread != 0 );
    m_stop_signal = true;
    m_queue_cond.notify_all();
    m_thread->join();
    delete m_thread;
    m_thread = 0;

    if( m_writer )
    {
        delete m_writer;
        m_writer = 0;
    }

    m_ofs.close();
}


void ActionRecorder::dispatch_action(boost::shared_ptr<Action> action)
{
    boost::recursive_mutex::scoped_lock lock(m_queue_mutex);
    m_queue.push(
        std::make_pair(
            boost::posix_time::microsec_clock::local_time(), action));
    m_queue_cond.notify_all();
}


void ActionRecorder::flush()
{
    boost::recursive_mutex tmp;
    boost::recursive_mutex::scoped_lock lock(tmp);
    m_flush_cond.wait(lock);
}


void ActionRecorder::run()
{
    while( !m_stop_signal )
    {
        TimestampedAction a;
        a.second = boost::shared_ptr<Action>();

        {
            boost::recursive_mutex::scoped_lock lock(m_queue_mutex);
            if( m_queue.empty() )
            {
                m_flush_cond.notify_all();
                m_queue_cond.wait(lock);
                continue;
            }
            else
            {
                a = m_queue.front();
                m_queue.pop();
            }
        }

        if( a.second )
        {
            boost::posix_time::time_duration td = a.first - m_t0;
            m_writer->write(a.second, td, 0 /* ignored */);
            m_writer->flush();
        }
    }
}
