/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "SetHingePivot.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../Hinge.hh"
#endif

#include <boost/math/fpclassify.hpp>
#include <Eigen/Core>
#include <stdexcept>


using namespace peekabot;


SetHingePivot::SetHingePivot(
    ObjectID target,
    float x, float y, float z,
    CoordinateSystem coord_sys)
    : m_target(target),
      m_x(x), m_y(y), m_z(z),
      m_coord_sys(coord_sys)
{
}


SetHingePivot::SetHingePivot() throw()
{
}


SetHingePivot::SetHingePivot(const SetHingePivot &other) throw()
    : m_target(other.m_target),
      m_x(other.m_x), m_y(other.m_y), m_z(other.m_z),
      m_coord_sys(other.m_coord_sys)
{
}


SetHingePivot::~SetHingePivot() throw()
{
}


Action *SetHingePivot::clone() const
{
    return new SetHingePivot(*this);
}


void SetHingePivot::execute(ServerExecutionContext *context) const
{
#ifdef __PEEKABOT_SERVER
    Hinge *ptr = dynamic_cast<Hinge *>(context->get_object(m_target));

    if( !ptr )
        throw std::runtime_error(
            "Failed to set hinge pivot, target object not found or "
            "of non-hinge type");

    // Check for infinity and NaN
    if( !boost::math::isfinite(m_x) ||
        !boost::math::isfinite(m_y) ||
        !boost::math::isfinite(m_z) )
        throw std::logic_error(
                "Failed to set hinge pivot: pivot "
                "cannot contain infinity or NaN");


    ptr->set_pivot(Eigen::Vector3f(m_x, m_y, m_z), m_coord_sys);
#endif
}


void SetHingePivot::save(SerializationInterface &ar) const
{
    ar << m_target << m_x << m_y << m_z << m_coord_sys;
}

void SetHingePivot::load(DeserializationInterface &ar)
{
    ar >> m_target >> m_x >> m_y >> m_z >> m_coord_sys;
}
