/*
 * Copyright Anders Boberg 2006.
 * Copyright Staffan Gimåker 2006-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <stdexcept>
#include <Eigen/LU>

#include "RearrangeObject.hh"

#ifdef __PEEKABOT_SERVER
#  include "../ServerExecutionContext.hh"
#  include "../SceneObject.hh"
#endif


using namespace peekabot;


RearrangeObject::RearrangeObject() throw()
{
}


RearrangeObject::RearrangeObject(
    PathIdentifier object, 
    PathIdentifier new_parent,
    bool retain_world_pose,
    NameConflictPolicy conflict_policy) throw()
    : m_object_id(object),
      m_new_parent_id(new_parent),
      m_retain_world_pose(retain_world_pose),
      m_conflict_policy(conflict_policy)
{
}


RearrangeObject::~RearrangeObject() throw()
{
}


Action *RearrangeObject::clone() const
{
    return new RearrangeObject(
        m_object_id, m_new_parent_id, 
        m_retain_world_pose, m_conflict_policy);
}


void RearrangeObject::execute(
    ServerExecutionContext *context) 
    const throw(std::exception)
{
#ifdef __PEEKABOT_SERVER
    SceneObject *ptr = m_object_id.get_leaf(context);
    SceneObject *old_parent = ptr->get_parent();

    SceneObject *new_parent;
    try
    {
        new_parent = m_new_parent_id.get_leaf(context);
    }
    catch(...)
    {
        throw std::runtime_error("Designated (new) parent object not found");
    }

    // Parent-less nodes (i.e. the root node) cannot be rearranged!
    if( !old_parent )
        throw std::runtime_error("The root node canot be rearranged");

    // Check if the *new parent* is a descendant of the object being
    // rearranged -- committing such an operation would create a small
    // orphaned graph, eek!
    if( ptr->is_descendant(new_parent) )
    {
        throw std::runtime_error(
            "The new parent cannot be a descendant of the object being "
            "rearranged");
    }

    if( old_parent == new_parent )
        return;

    Eigen::Transform3f mtow = ptr->get_mtow();
    ptr->detach();

    try
    {
        new_parent->attach(ptr, m_conflict_policy);

        if( m_retain_world_pose )
            ptr->set_transformation(
                Eigen::Transform3f(
                    new_parent->get_mtow().inverse(Eigen::Isometry)) *
                mtow);
    }
    catch(...)
    {
        old_parent->attach(ptr);
        throw;
    }
#endif
}


void RearrangeObject::save(SerializationInterface &ar) const
{
    ar << m_object_id << m_new_parent_id 
       << m_retain_world_pose << m_conflict_policy;
}

void RearrangeObject::load(DeserializationInterface &ar)
{
    ar >> m_object_id >> m_new_parent_id 
       >> m_retain_world_pose >> m_conflict_policy;
}
