/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cassert>
#include <fstream>
#include <boost/cstdint.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/filesystem/operations.hpp>
#include <boost/thread/mutex.hpp>

#include "UploadCache.hh"
#include "FsToolbox.hh"


using namespace peekabot;


namespace
{
    boost::uint32_t allocate_cache_no()
    {
        static boost::uint32_t s_cache_no = 0;
        static boost::mutex s_mutex;
        boost::mutex::scoped_lock lock(s_mutex);
        return s_cache_no++;
    }
}


boost::shared_ptr<UploadCache> UploadCache::create()
{
    return boost::shared_ptr<UploadCache>(new UploadCache);
}


UploadCache::UploadCache()
    : m_counter(0),
      m_cache_no(allocate_cache_no())
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    if( !boost::filesystem::exists(get_upload_directory()) )
        boost::filesystem::create_directory(get_upload_directory());

    if( boost::filesystem::exists(get_cache_directory()) )
        boost::filesystem::remove_all(get_cache_directory());

    boost::filesystem::create_directory(get_cache_directory());

    assert( boost::filesystem::exists(get_cache_directory()) );
}


UploadCache::~UploadCache()
{
    clear();
}


void UploadCache::add_file(
    const std::string &remote_filename,
    const void *data, std::size_t n)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    UploadMap::const_iterator it = m_files.find(remote_filename);
    if( it == m_files.end() )
        m_files[remote_filename] = generate_local_filename(remote_filename);

    boost::filesystem::path path =
        get_cache_directory() / m_files[remote_filename];
    std::ofstream ofs(path.string().c_str(), std::ios::binary);

    if( !ofs )
        throw std::runtime_error(
            "Failed to open file '" + path.string() + "' for writing");

    if( n > 0 )
        ofs.write((const char *)data, n);

    ofs.close();
}


void UploadCache::remove_file(const std::string &remote_filename)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    UploadMap::iterator it = m_files.find(remote_filename);
    if( it == m_files.end() )
    {
        return;
    }
    else
    {
        boost::filesystem::remove(get_cache_directory() / it->second);
        m_files.erase(it);
    }
}


void UploadCache::clear()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    boost::filesystem::remove_all(get_cache_directory());
    m_files.clear();
}


bool UploadCache::get_local_path(
    const std::string &remote_filename,
    boost::filesystem::path &local_path) const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    UploadMap::const_iterator it = m_files.find(remote_filename);
    if( it == m_files.end() )
    {
        return true;
    }
    else
    {
        local_path = get_cache_directory() / it->second;
        return false;
    }
}


std::string UploadCache::generate_local_filename(
    const std::string &remote_filename)
{
    return boost::lexical_cast<std::string>(m_counter++);
}


boost::filesystem::path UploadCache::get_upload_directory()
{
    return fs::get_temp_path() / "peekabot" / "uploads";
}


boost::filesystem::path UploadCache::get_cache_directory() const
{
    return get_upload_directory() /
        boost::lexical_cast<std::string>(m_cache_no);
}
