/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "TriMesh.hh"
#include "ObjectVisitor.hh"
#include "ObjectTypes.hh"
#include "PropKeys.hh"

#include <limits>
#include <stdexcept>
#include <Eigen/Geometry>


using namespace peekabot;


HandlerInformer TriMesh::ms_handler_informer(
    "tri_mesh", &TriMesh::start_handler);


TriMesh::TriMesh()
    : SceneObject("tri_mesh"),
      VertexObject(1)
{
}


TriMesh::TriMesh(ScopedHandler *handler)
    : SceneObject("tri_mesh", handler),
      ScalableObject(handler),
      VertexObject(handler, 1)
{
    // Register the vertices tag start handler
    handler->add_start_handler(
        "triangles",
        boost::bind(&TriMesh::triangles_start_handler,
                    this, _1, _2, _3));
}


void TriMesh::accept(ObjectVisitor *visitor) throw()
{
    visitor->visit(this);
}


ObjectType TriMesh::get_object_type() const
{
    return TRI_MESH_OBJECT;
}


const TriMesh::Indices &TriMesh::get_indices() const
{
    return m_indices;
}


void TriMesh::set_indices(const Indices &indices)
{
    if( indices.size()%3 != 0 )
        throw std::length_error(
            "The number of indices must be a multiple of three");

    m_indices = indices;
    m_indices_set_signal();
}


void TriMesh::add_indices(const Indices &indices)
{
    if( indices.size()%3 != 0 )
        throw std::length_error(
            "The number of indices must be a multiple of three");

    for( std::size_t i = 0; i < indices.size(); ++i )
        m_indices.push_back(indices[i]);

    m_indices_added_signal(indices);
}


PropMap &TriMesh::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(
            *s_prop_adapters, SceneObject::get_prop_adapters());
        merge_prop_adapters(
            *s_prop_adapters, ScalableObject::get_prop_adapters());
        merge_prop_adapters(
            *s_prop_adapters, VertexObject::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void TriMesh::create_prop_adapters(PropMap &adapters)
{
}


// ---------------------- XML handler functions ----------------------


void TriMesh::start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject* tmp = new TriMesh(handler);

    // Set the new object as the current object
    ScopedMap & variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}


void TriMesh::triangles_start_handler(
    const std::string &tag_name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
{
    ScopedHandler::TagScope vertices_scope;

    vertices_scope.start_functors["triangle"] = boost::bind(
        &TriMesh::triangle_start_handler, _1, _2, _3);

    vertices_scope.end_functors.push_back(
        boost::bind(&TriMesh::triangles_end_handler, this, _1, _2));

    handler->get_variables().push_variable<Indices>("indices", Indices());

    handler->enter_scope(vertices_scope);
}


void TriMesh::triangles_end_handler(
    const std::string &name,
    ScopedHandler *handler)
{
    ScopedMap &variables = handler->get_variables();
    assert( variables.exists<Indices>("indices") );
    set_indices(variables.get_variable<Indices>("indices"));
    variables.pop_variable<Indices>("indices");
}


void TriMesh::triangle_start_handler(
    const std::string &tag_name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler)
{
    ScopedHandler::TagScope vertex_scope;

    vertex_scope.cdata_functor = boost::bind(
        &TriMesh::triangle_cdata_handler, _1, _2);

    handler->enter_scope(vertex_scope);
}


void TriMesh::triangle_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler)
{
    // Read the index triple
    std::stringstream s(cdata);
    boost::uint32_t i1, i2, i3;
    s >> i1 >> i2 >> i3;

    ScopedMap &variables = handler->get_variables();
    Indices &inds = variables.get_variable<Indices>("indices");
    inds.push_back(i1);
    inds.push_back(i2);
    inds.push_back(i3);
}
