/*
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __PEEKABOT_SUBSCRIBER_HH
#define __PEEKABOT_SUBSCRIBER_HH


#include <set>
#include <boost/thread/recursive_mutex.hpp>

#include "Message.hh"


namespace peekabot
{


    /**
     * \internal
     *
     * \brief An interface that can be implemented to gain message recieving
     * capability, as in messages published by a MessageHub.
     */
    class Subscriber
    {
        friend class MessageHub;

    public:
        /**
         * \brief Called when a message is received.
         */
        virtual void handle_message(const Message &) throw() = 0;

        /**
         * \brief Automatically cancels any active subscriptions when the
         * Subscriber-object is destroyed.
         */
        virtual ~Subscriber();

    private:
        /**
         * \brief Add a subscription to the set internally keeping track
         * of all active subscriptions.
         *
         * \param type The message category the subscription pertains to.
         * \param hub The originator of messages in the subscription.
         */
        void add_subscription(MessageType type, MessageHub *hub) throw();

        /**
         * \brief Remove a subscription from the set internally keeping track
         * of all active subscriptions.
         *
         * \param type The message category the subscription pertains to.
         * \param hub The originator of messages in the subscription.
         */
        void remove_subscription(MessageType type, MessageHub *hub) throw();

    private:
        /**
         * \brief Typedef for a unique-value container containing 
         * \<type, publisher\> pairs for all active subscriptions.
         */
        typedef std::set<std::pair<MessageType, MessageHub *> > SubscriptionSet;

        /**
         * \brief Contains information about all current subscriptions where
         * the object is the recipient. The information consists of pairs
         * containing message category and publisher.
         */
        SubscriptionSet m_subscriptions;

        /**
         * \brief Governs all (read \e and write) access all members.
         */
        mutable boost::recursive_mutex m_mutex;
    };

}


#endif // __PEEKABOT_SUBSCRIBER_HH
