/*
 * Copyright Staffan Gimåker 2007-2010.
 * Copyright Anders Boberg 2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#ifndef PEEKABOT_SLIDER_HH_INCLUDED
#define PEEKABOT_SLIDER_HH_INCLUDED


#include <boost/any.hpp>
#include <boost/function.hpp>
#include <boost/lexical_cast.hpp>
#include <Eigen/Core>
#include <Eigen/Geometry>

#include "Joint.hh"
#include "HandlerInformer.hh"
#include "ScopedHandler.hh"


namespace peekabot
{

    /**
     * \internal
     *
     * \brief A prismatic joint.
     *
     * The amount of displacement is governed by the track's value.
     * For example, a value of 5.6 means that all children will be
     * displaced 5.6 meters along the track's displacement axis.
     */
    class Slider : public Joint
    {
    public:
        Slider();

        Slider(ScopedHandler *handler);

        virtual ~Slider() throw();

        /**
         * \brief Create a slider DOF that displaces its children along the
         * given vector. The amount of displacement, in standard units, is equal
         * to its value.
         *
         * \param axis A direction vector, in parent coordinates, that will be
         * used as the displacement axis for the DOF.
         * It's magnitude should be non-zero, and as long as it is the
         * magnitude doesn't affect the behaviour of the DOF.
         */
        Slider(const Eigen::Vector3f &axis);

        /**
         * \brief Set the track's axis of displacement.
         *
         * The DOF will be updated to utilize the new axis when the method
         * is called.
         *
         * \param axis A non-zero direction vector, in the given coordinate
         * system, that will be used as the displacement axis for the joint.
         */
        void set_axis(
            const Eigen::Vector3f &axis,
            CoordinateSystem coord_sys = PARENT_COORDINATES) throw();

        /**
         * \brief Get the track's normalized axis of displacement.
         */
        const Eigen::Vector3f &get_axis() const throw();

        virtual void accept(ObjectVisitor *visitor) throw();

        virtual ObjectType get_object_type() const;

        virtual Eigen::Transform3f calculate_dof_transform(float value)
            const throw();

        /// \name Signals
        /// @{

        typedef boost::signals2::signal<void ()> AxisSetSignal;

        inline AxisSetSignal &axis_set_signal()
        {
            return m_axis_set_signal;
        }

        /// @}

    protected:
        virtual PropMap &get_prop_adapters();

    private:
        static void create_prop_adapters(PropMap &adapters);

        /// \name XML handler methods
        /// @{

        static void start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw();

        void axis_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        void axis_cdata_handler(
            CoordinateSystem coord_sys,
            const std::string &cdata,
            ScopedHandler *handler)
            throw(std::domain_error, std::runtime_error, boost::bad_any_cast);

        /**
         * \brief Min tag start handler for XML parsing.
         */
        void min_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        /**
         * \brief Max tag start handler for XML parsing.
         */
        void max_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        /**
         * \brief Initial tag start handler for XML parsing.
         */
        void initial_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        /**
         * \brief Offset tag start handler for XML parsing.
         */
        void offset_start_handler(
            const std::string & name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler)
            throw();

        /**
         * \brief Generic CDATA handler that simply casts the cdata and invokes
         * the supplied function with that value.
         */
        template<class T>
        static void generic_cdata_handler(
            const std::string &cdata,
            ScopedHandler *handler,
            boost::function<void (T)> f)
        {
            f( boost::lexical_cast<T>(cdata) );
        }

        /**
         * \brief Set up the supplied function \a f to be invoked with the
         * argument \a val when the end element of the DOF is encountered.
         *
         * \param dof_scope The DOF elements scope.
         * \param f The function to invoke when the end element is encountered.
         * \param val The value \a f is invoked with.
         */
        static void invoke_on_end_element(
            ScopedHandler::TagScope &dof_scope,
            boost::function<void (float)> f, float val);

        /// @}


    private:
        /**
         * \brief The tracks's normalized displacement axis,
         * in parent coordinates.
         *
         * The initial value is \f$(0,0,1)\f$.
         *
         * \invariant \c m_axis must always be of unit length.
         */
        Eigen::Vector3f m_axis;

        AxisSetSignal m_axis_set_signal;

        static HandlerInformer ms_handler_informer;
    };
}

#endif // PEEKABOT_SLIDER_HH_INCLUDED
