/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "Server.hh"
#include "ClientConnection.hh"
#include "SceneTree.hh"
#include "MessageHub.hh"

#include <boost/bind.hpp>


using namespace peekabot;


Server::Server(const Config &config)
    : m_config(config),
      m_work(0),
      m_asio_thread(0),
      m_server_thread(0)
{
    m_data.m_scene = new SceneTree;
}


Server::~Server()
{
    // TODO
}


const Config &Server::get_config() const
{
    return m_config;
}


void Server::start()
{
    // TODO: error handling

    m_stop_signal = false;

    m_work = new boost::asio::io_service::work(m_io_service);

    m_asio_thread = new boost::thread(
        boost::bind(&boost::asio::io_service::run, &m_io_service));

    m_server_thread = new boost::thread(
        boost::bind(&Server::server_thread, this));
}


void Server::stop()
{
    delete m_work;
    m_work = 0;

    m_io_service.stop();
    m_asio_thread->join();
    delete m_asio_thread;
    m_asio_thread = 0;

    m_stop_signal = true;
    m_requests_cond.notify_all();
    m_server_thread->join();
    delete m_server_thread;
    m_server_thread = 0;
}


bool Server::is_stopped() const
{
    // TODO
}


void Server::post(boost::function<void (ServerData &)> handler)
{
    boost::mutex::scoped_lock lock(m_requests_mutex);
    m_requests.push(handler);
    m_requests_cond.notify_all();
}


void Server::listen_ipv4(unsigned int port)
{
    boost::shared_ptr<boost::asio::ip::tcp::acceptor> acceptor(
        new boost::asio::ip::tcp::acceptor(
            m_io_service,
            boost::asio::ip::tcp::endpoint(boost::asio::ip::tcp::v4(), port)));

    acceptor->set_option(
        boost::asio::socket_base::reuse_address(true));

    start_accept(acceptor);
}


void Server::start_accept(
    boost::shared_ptr<boost::asio::ip::tcp::acceptor> acceptor)
{
    boost::shared_ptr<ClientConnection> new_connection =
        ClientConnection::create(*this, m_io_service);

    acceptor->async_accept(
        new_connection->get_socket(),
        boost::bind(
            &Server::handle_accept, this, _1, acceptor, new_connection));
}


void Server::handle_accept(
    const boost::system::error_code &e,
    boost::shared_ptr<boost::asio::ip::tcp::acceptor> acceptor,
    boost::shared_ptr<ClientConnection> c)
{
    if( !e )
    {
        c->start();
    }

    start_accept(acceptor);
}


void Server::server_thread()
{
    while( !m_stop_signal )
    {
        RequestQueue::value_type handler;
        {
            boost::mutex::scoped_lock lock(m_requests_mutex);
            if( m_requests.empty() )
                m_requests_cond.wait(lock);

            if( m_requests.empty() )
                continue;

            handler = m_requests.front();
            m_requests.pop();
        }

        try
        {
            handler(m_data);
        }
        catch(std::exception &e)
        {
            TheMessageHub::instance().publish(
                ERROR_MESSAGE,
                std::string(e.what()));
        }
        catch(...)
        {
            TheMessageHub::instance().publish(
                ERROR_MESSAGE,
                "Server caught an unknown exception");
        }
    }
}

