/*
 * Copyright Staffan Gimåker 2006-2010.
 * Copyright Anders Boberg 2006-2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "SceneObject.hh"
#include "SceneFileLoader.hh"
#include "Prop.hh"
#include "PropKeys.hh"
#include "props/LayerPropAdapter.hh"
#include "props/StringPropBase.hh"
#include "props/BoolPropBase.hh"
#include "props/FloatPropBase.hh"
#include "props/ColorPropBase.hh"
#include "props/MtowPropAdapter.hh"

#include <stack>
#include <iostream>
#include <sstream>
#include <stdexcept>
#include <boost/bind.hpp>
#include <boost/function.hpp>
#include <boost/format.hpp>
#include <boost/foreach.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/algorithm/string/classification.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/math/fpclassify.hpp>
#include <Eigen/LU>


using namespace peekabot;


namespace
{
    // Do one normalization for every 10 transforms (25 breaks it, 15 almost
    // breaks it..)
    int TRANSFORMS_PER_NORMALIZATION = 10;

    void do_nothing(const std::string &, ScopedHandler *) {}

    void end_handler_wrapper(
        const std::string &, ScopedHandler *,
        boost::function<void (void)> f)
    {
        f();
    }

    /*
     * XML handler that ignores all contained cdata and elements.
     */
    void external_handler(
        const std::string &name,
        ScopedHandler::AttributeMap &attributes,
        ScopedHandler *handler) throw()
    {
        ScopedHandler::TagScope scope;
        scope.cdata_functor = &do_nothing;
        scope.unhandled_functor = &external_handler;

        handler->enter_scope(scope);
    }


    boost::bimap<PropKey, std::string> make_prop_adapters_names()
    {
        typedef boost::bimap<PropKey, std::string> R;
        R ret;

        ret.insert( R::value_type(NAME_PROP, "Name") );
        ret.insert( R::value_type(LAYER_PROP, "Layer") );
        ret.insert( R::value_type(HIDDEN_PROP, "Hidden") );
        ret.insert( R::value_type(VISIBILITY_PROP, "Visible") );
        ret.insert( R::value_type(COLOR_PROP, "Color") );
        ret.insert( R::value_type(OPACITY_PROP, "Opacity") );
        ret.insert( R::value_type(SCALE_PROP, "Scale") );
        ret.insert( R::value_type(MTOW_PROP, "Transformation") );

        ret.insert( R::value_type(CAMERA_ORTHO_PROP, "Orthographic") );
        ret.insert( R::value_type(CAMERA_FOV_PROP, "Vertical FOV") );
        ret.insert( R::value_type(CAMERA_ZOOM_DISTANCE_PROP, "Zoom distance") );

        ret.insert( R::value_type(CAMERA_NEAR_PROP, "Near plane") );
        ret.insert( R::value_type(CAMERA_FAR_PROP, "Far plane") );

        ret.insert( R::value_type(GRID_TYPE_PROP, "Grid type") );
        ret.insert( R::value_type(GRID_CENTRAL_ANGLE_PROP, "Central angle") );
        ret.insert( R::value_type(GRID_SEGMENTS_PROP, "Segments") );

        ret.insert( R::value_type(LABEL_TEXT_PROP, "Text") );
        ret.insert( R::value_type(LABEL_ALIGNMENT_PROP, "Alignment") );

        ret.insert( R::value_type(LINE_WIDTH_PROP, "Line width") );
        ret.insert( R::value_type(LINE_STYLE_PROP, "Line style") );
        ret.insert( R::value_type(LINE_STIPPLE_FACTOR_PROP, "Stipple factor") );

        ret.insert( R::value_type(MAX_VERTICES_PROP, "Max vertices") );
        ret.insert( R::value_type(VERTEX_OVERFLOW_POLICY_PROP, "Overflow policy") );

        ret.insert( R::value_type(JOINT_VALUE_PROP, "Joint value") );
        ret.insert( R::value_type(JOINT_MIN_PROP, "Joint min") );
        ret.insert( R::value_type(JOINT_MAX_PROP, "Joint max") );
        ret.insert( R::value_type(JOINT_OFFSET_PROP, "Joint offset") );

        ret.insert( R::value_type(HINGE_AXIS_PROP, "Hinge axis") );
        ret.insert( R::value_type(HINGE_PIVOT_PROP, "Hinge pivot") );

        ret.insert( R::value_type(SLIDER_AXIS_PROP, "Slider axis") );

        ret.insert( R::value_type(OG2D_CELL_SIZE_PROP, "Cell size") );
        ret.insert( R::value_type(OG2D_UNOCCUPIED_COLOR_PROP, "Free color") );
        ret.insert( R::value_type(OG2D_OCCUPIED_COLOR_PROP, "Occupied color") );

        ret.insert( R::value_type(OG3D_CELL_XY_SIZE_PROP, "Cell XY size") );
        ret.insert( R::value_type(OG3D_CELL_Z_SIZE_PROP, "Cell height") );
        ret.insert(
            R::value_type(OG3D_COLOR_MAPPING_ENABLED_PROP, "Color mapping") );
        ret.insert(R::value_type(OG3D_COLOR_MAPPING_Z_MIN_PROP, "Z min") );
        ret.insert(R::value_type(OG3D_COLOR_MAPPING_Z_MAX_PROP, "Z max") );

        return ret;
    }
}




// Keys below 10000 are reserved for prop adapters
PropKey SceneObject::ms_next_prop_key = 10000;

ObjectID SceneObject::ms_next_object_id = 1;

SceneObject::PropNames SceneObject::ms_prop_names = make_prop_adapters_names();


SceneObject::SceneObject(const std::string &default_name)
    : m_is_hidden(false),
      m_is_visible(true),
      m_parent(0),
      m_layer(0),
      m_opacity(1.0f),
      m_name(default_name),
      m_id(ms_next_object_id++),
      m_transforms_since_normalization(0),
      m_color(0, 0.5, 0),
      m_is_selected(false)
{
    Eigen::Transform3f tmp;
    tmp.setIdentity();
    set_transformation(tmp);
}

SceneObject::~SceneObject()
{
    // TODO: Who removes objects from the parent's child list?
    while( !m_children.empty() )
    {
        delete *m_children.begin();
        m_children.erase(m_children.begin());
    }

    for( PropIterator it = props_begin(); it != props_end(); ++it )
        delete it->second;
    m_props.clear();
}


ObjectID SceneObject::get_object_id() const throw()
{
    return m_id;
}


boost::uint8_t SceneObject::get_layer() const throw()
{
    return m_layer;
}


void SceneObject::set_layer(boost::uint8_t layer)
{
    if( layer >= NUMBER_OF_LAYERS )
        throw std::range_error("Layer out of range");

    if( layer != m_layer )
    {
        m_layer = layer;
        m_layer_set_signal();
    }
}


const std::string& SceneObject::get_name() const throw()
{
    return m_name;
}


void SceneObject::set_name(const std::string& name)
{
    if( name == get_name() )
        return;

    // In order for the new name to be valid, we must make sure there are
    // no siblings of this object who have the same name.
    if(m_parent != 0)
    {
        Children::iterator child_it = m_parent->m_children.begin();

        for( ; child_it != m_parent->m_children.end(); ++child_it )
        {
            if( (*child_it)->get_name() == name &&
                (*child_it) != this )
                throw std::runtime_error(
                    "Failed to change name - name uniqueness constraint violated");
        }
    }

    m_name = name;
    m_name_set_signal();
}


float SceneObject::get_opacity() const throw()
{
    return m_opacity;
}


void SceneObject::set_opacity(float opacity)
{
    if( opacity > 1.0f || opacity < 0.0f )
        throw std::range_error("Opacity must be in the range [0,1]");

    m_opacity = opacity;
    m_opacity_set_signal();

    // Notify the relevant parts (i.e. the subtrees that doesn't have a
    // node of absolute opacity at its root) of the subtree that the
    // accumulated opacity has changed
    std::stack<SceneObject *> s;
    s.push( this );

    while( !s.empty() )
    {
        SceneObject *obj = s.top();
        s.pop();

        obj->m_accum_opacity_changed_signal();

        for( Children::iterator it = obj->m_children.begin();
             it != obj->m_children.end(); it++ )
        {
            /*if( opacity_is_absolute() )*/
            s.push(*it);
        }
    }
}


void SceneObject::set_hidden(bool hidden)
{
    if( m_is_hidden != hidden )
    {
        m_is_hidden = hidden;
        recalc_visibility();

        m_hidden_set_signal();
    }
}


bool SceneObject::is_hidden() const throw()
{
    return m_is_hidden;
}


bool SceneObject::is_visible() const throw()
{
    return m_is_visible;
}


float SceneObject::get_accumulated_opacity() const throw()
{
    std::stack<const SceneObject*> path;
    float accumulated_opacity;

    // Trace a path up to the first object with absolute opacity.
    const SceneObject *current = this;
    path.push(current);
    while(/*!(current->get_opacity().is_absolute) &&*/ current->m_parent != 0)
    {
        path.push(current->m_parent);
        current = current->m_parent;
    }

    // Start the accumulated opacity to the value of the absolute one.
    current = path.top();
    path.pop();
    accumulated_opacity = current->m_opacity;

    // Start accumulating...
    while(!path.empty())
    {
        current = path.top();
        path.pop();
        accumulated_opacity *= current->m_opacity;
    }

    return accumulated_opacity;
}


bool SceneObject::attach(
    SceneObject *subtree, NameConflictPolicy conflict_policy)
    throw(std::runtime_error)
{
    bool ret = false;

    // Check for and/or resolve name conflicts...
    if( has_child(subtree->get_name()) )
    {
        if( conflict_policy == AUTO_ENUMERATE_ON_CONFLICT )
        {
            // Do a binary search to find a suitable suffixed name, we assume
            // that names are continously distributed... so it might fail, if
            // it does we resort to using a linear search
            std::string name;
            boost::uint32_t lb = 1;
            boost::uint32_t ub = std::numeric_limits<boost::uint32_t>::max();

            while( ub-lb > 1 )
            {
                boost::uint32_t n = lb + (ub-lb)/2;
                name = (boost::format("%1%%2%") % subtree->get_name() % n).str();

                if( has_child(name) )
                    lb = n+1;
                else
                    ub = n;
            }

            if( ub-lb == 1 )
            {
                name = (boost::format("%1%%2%") % subtree->get_name() % lb).str();
                if( has_child(name) )
                    name = (boost::format("%1%%2%") % subtree->get_name() % ub).str();
            }

            // Did the binary search fail? If so, resort to a linear search
            if( has_child(name) )
            {
                boost::uint32_t n = 0;
                do
                {
                    name = (boost::format("%1%%2%") % subtree->get_name() % ++n).str();
                }
                while( has_child(name) );
            }

            assert( !has_child(name) );

            subtree->set_name(name);
            ret = true;
        }
        else if( conflict_policy == REPLACE_ON_CONFLICT )
        {
            SceneObject *conflicting = get_child(subtree->get_name());
            conflicting->detach();
            delete conflicting;
        }
        else
            throw std::runtime_error(
                "Failed to attach subtree due to a name uniqueness violation");
    }

    subtree->m_parent = this;
    m_children.insert(subtree);

    subtree->recalc_mtows();
    subtree->recalc_visibility();

    m_child_attached_signal(subtree);

    return ret;
}


void SceneObject::detach()
{
    if( get_parent() == 0 )
        throw std::logic_error("Can't detach parentless nodes");

    m_parent->m_children.erase(this);
    m_parent = 0;

    recalc_mtows();
    recalc_visibility();

    m_detached_signal();
}


void SceneObject::rearrange(SceneObject *new_parent, bool retain_mtow)
{
    if( !get_parent() )
        throw std::logic_error("Parentless objects cannot be rearranged");

    if( get_parent() == new_parent )
        return;

    if( is_descendant(new_parent) )
        throw std::runtime_error(
            "Cannot move an object to one of its descendants");

    Eigen::Transform3f mtow = get_mtow();

    detach();
    new_parent->attach(this, AUTO_ENUMERATE_ON_CONFLICT);

    if( retain_mtow )
        set_transformation(mtow, WORLD_COORDINATES);
}


bool SceneObject::is_descendant(SceneObject *object) const throw()
{
    // An object is a descendant of this object if it is a child of
    // this object, or if it is a descendant of one of this object's
    // children.
    for(Children::const_iterator it = m_children.begin();
        it != m_children.end(); ++it)
    {
        if(*it == object)
            return true;
        else
            return (*it)->is_descendant(object);
    }
    return false;
}


void SceneObject::set_transformation(
    const Eigen::Transform3f &m, CoordinateSystem system)
{
    if( system == PARENT_COORDINATES )
    {
        m_mtop = m;
    }
    else if( system == WORLD_COORDINATES )
    {
        // Transform m (given in word coordinates) to the coordinate system of
        // the parent, i.e. m' = inv(P) * m, where P is the parent's
        // model-to-world transformation.
        m_mtop = get_parent_mtow().inverse(Eigen::Isometry) * m;
    }
    else if( system == LOCAL_COORDINATES )
    {
        m_mtop = m * m_mtop;
    }

    // TODO: remove this...? set_transformation should get already valid matrices
    if( ++m_transforms_since_normalization >= TRANSFORMS_PER_NORMALIZATION )
        orthonormalize_transform();

    // Re-compute mtows
    recalc_mtows();

    // NOTE: Order is important here - mtows must be re-calculated before
    // firing any signals, or we'll risk introducing very subtle
    m_transformation_set_signal();
}


void SceneObject::recalc_mtows()
{
    if( get_parent() == 0 )
    {
        // Special case: It's a root node
        m_mtow = m_mtop;
    }
    else
    {
        m_mtow = m_parent->m_mtow * m_mtop;
    }

    // Update child nodes
    for( Children::iterator it = m_children.begin();
        it != m_children.end(); it++ )
    {
        (*it)->recalc_mtows();
    }

    // Notify any interested parties that the mtow changed
    //
    // Note that order is important here! If we fire the signal
    // before updating all the children, any slot that traverses the
    // children and uses their mtows will be using old data.
    m_mtow_changed_signal();
}


void SceneObject::recalc_visibility()
{
    if( get_parent() == 0 )
        m_is_visible = !m_is_hidden; // Special case: It's a root node
    else
        m_is_visible = !m_is_hidden && get_parent()->m_is_visible;

    for( Children::iterator it = m_children.begin();
        it != m_children.end(); it++ )
        (*it)->recalc_visibility();

    // NOTE: order is important! do this last
    m_visibility_changed_signal();
}


void SceneObject::apply_transformation(
    const Eigen::Transform3f &transform) throw()
{
    set_transformation(transform * m_mtop);
}


void SceneObject::orthonormalize_transform() throw()
{
    m_mtop.linear().col(0).normalize();
    m_mtop.linear().col(1).normalize();

    // The cross product of two normalized vectors is a normalized vector, thus
    // we don't need to normalize the third basis vector
    m_mtop.linear().col(2) = m_mtop.linear().col(0).cross(
        m_mtop.linear().col(1));
    m_mtop.linear().col(1) = m_mtop.linear().col(2).cross(
        m_mtop.linear().col(0));

    m_transforms_since_normalization = 0;
}


SceneObject *SceneObject::get_parent() throw()
{
    return m_parent;
}


const SceneObject *SceneObject::get_parent() const throw()
{
    return m_parent;
}


SceneObject::Children &SceneObject::get_children() throw()
{
    return m_children;
}


const SceneObject::Children &SceneObject::get_children() const throw()
{
    return m_children;
}


SceneObject::ChildIterator SceneObject::begin()
{
    return m_children.begin();
}


SceneObject::ChildIterator SceneObject::end()
{
    return m_children.end();
}


void SceneObject::clear()
{
    Children tmp = m_children;
    for( ChildIterator it = tmp.begin(); it != tmp.end(); ++it )
    {
        (*it)->detach();
        delete *it;
    }
}


SceneObject::ConstChildIterator SceneObject::begin() const
{
    return m_children.begin();
}


SceneObject::ConstChildIterator SceneObject::end() const
{
    return m_children.end();
}


SceneObject *SceneObject::get_child(const std::string &name)
    throw(ChildNotFound)
{
    for(Children::iterator it = m_children.begin();
        it != m_children.end(); ++it)
        if((*it)->get_name() == name)
            return *it;

    throw ChildNotFound(name);
}


const SceneObject *SceneObject::get_child(const std::string &name)
    const throw(ChildNotFound)
{
    for(Children::const_iterator it = m_children.begin();
        it != m_children.end(); ++it)
        if((*it)->get_name() == name)
            return *it;

    throw ChildNotFound(name);
}


std::size_t SceneObject::child_count() const throw()
{
    return m_children.size();
}

bool SceneObject::has_child(const std::string &name) const throw()
{
    for(Children::const_iterator it = m_children.begin();
        it != m_children.end(); ++it)
        if((*it)->get_name() == name)
            return true;
    return false;
}

bool SceneObject::has_descendent(const std::string &rel_path) const throw()
{
    std::vector<std::string> components;
    boost::split( components, rel_path, boost::is_any_of(".") );

    return _get_descendent(components.begin(), components.end()) != 0;
}


SceneObject *SceneObject::get_descendent(const std::string &rel_path)
    throw(std::runtime_error)
{
    std::vector<std::string> components;
    boost::split( components, rel_path, boost::is_any_of(".") );

    SceneObject *ret = _get_descendent(
        components.begin(), components.end());

    if( ret == 0 )
        throw std::runtime_error("Descendent '" + rel_path + "' not found");

    return ret;
}


const SceneObject *SceneObject::get_descendent(const std::string &rel_path)
    const throw(std::runtime_error)
{
    std::vector<std::string> components;
    boost::split( components, rel_path, boost::is_any_of(".") );

    const SceneObject *ret = _get_descendent(
        components.begin(), components.end());

    if( ret == 0 )
        throw std::runtime_error("Descendent '" + rel_path + "' not found");

    return ret;
}


PropMap &SceneObject::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
    }

    return *s_prop_adapters;
}


const PropMap &SceneObject::get_prop_adapters() const
{
    return const_cast<SceneObject *>(this)->get_prop_adapters();
}


void SceneObject::create_prop_adapters(PropMap &adapters)
{
    class NamePropAdapter : public StringPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            obj->set_name(any_cast<std::string>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            return Any(obj->get_name());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            return obj->name_set_signal();
        }
    };

    class HiddenPropAdapter : public BoolPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            obj->set_hidden(any_cast<bool>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            return Any(obj->is_hidden());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            return obj->hidden_set_signal();
        }
    };

    class VisibilityPropAdapter : public BoolPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj) {}

        virtual Any get(const SceneObject *obj) const
        {
            return Any(obj->is_visible());
        }

        virtual bool is_read_only() const
        {
            return true;
        }

        SignalType &signal(SceneObject *obj)
        {
            return obj->visibility_changed_signal();
        }
    };

    class ColorPropAdapter : public ColorPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            obj->set_color(any_cast<RGBColor>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            return Any(obj->get_color());
        }

        SignalType &signal(SceneObject *obj)
        {
            return obj->color_set_signal();
        }
    };

    class OpacityPropAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            obj->set_opacity(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            return Any(obj->get_opacity());
        }

        SignalType &signal(SceneObject *obj)
        {
            return obj->opacity_set_signal();
        }

        virtual float get_min_value() const
        {
            return 0.0f;
        }

        virtual float get_max_value() const
        {
            return 1.0f;
        }

        virtual bool display_as_slider() const
        {
            return true;
        }
    };

    adapters.insert(
        PropMap::value_type(
            NAME_PROP, new NamePropAdapter));

    adapters.insert(
        PropMap::value_type(
            LAYER_PROP, new LayerPropAdapter));

    adapters.insert(
        PropMap::value_type(
            HIDDEN_PROP, new HiddenPropAdapter));

    /*adapters.insert(
        PropMap::value_type(
        VISIBILITY_PROP, new VisibilityPropAdapter));*/

    adapters.insert(
        PropMap::value_type(
            OPACITY_PROP, new OpacityPropAdapter));

    adapters.insert(
        PropMap::value_type(
            COLOR_PROP, new ColorPropAdapter));

    adapters.insert(
        PropMap::value_type(
            MTOW_PROP,new MtowPropAdapter));
}


void SceneObject::merge_prop_adapters(PropMap &to, const PropMap &from)
{
    for( PropMap::const_iterator it = from.begin(); it != from.end(); ++it )
        to.insert(*it);
}


void SceneObject::set_position(const Eigen::Vector3f &pos) throw()
{
    Eigen::Transform3f tmp = m_mtop;
    tmp(0,3) = pos(0);
    tmp(1,3) = pos(1);
    tmp(2,3) = pos(2);
    set_transformation(tmp);
}


void SceneObject::set_world_position(const Eigen::Vector3f &pos) throw()
{
    if( get_parent() )
    {
        Eigen::Vector4f tmp(pos(0), pos(1), pos(2), 1);
        tmp = get_parent_mtow().inverse(Eigen::Isometry) * tmp;
        set_position(Eigen::Vector3f(tmp(0), tmp(1), tmp(2)));
    }
    else
    {
        // No parent found, e.g. we might just say the parent's
        // transformation is the identity:
        set_position(pos);
    }
}


void SceneObject::set_rpy(
    float roll, float pitch, float yaw,
    CoordinateSystem system)
{
    Eigen::Transform3f m;
    m = Eigen::AngleAxisf(yaw, Eigen::Vector3f::UnitZ()) *
        Eigen::AngleAxisf(pitch, Eigen::Vector3f::UnitY()) *
        Eigen::AngleAxisf(roll, Eigen::Vector3f::UnitX());

    if( system == WORLD_COORDINATES )
        m.translation() = m_mtow.translation();
    else if( system == PARENT_COORDINATES )
        m.translation() = m_mtop.translation();

    set_transformation(m, system);
}


void SceneObject::rotate_x(float angle) throw()
{
    Eigen::Transform3f T;
    T = Eigen::AngleAxisf(angle, Eigen::Vector3f::UnitX());
    apply_transformation(T);
}


void SceneObject::rotate_y(float angle) throw()
{
    Eigen::Transform3f T;
    T = Eigen::AngleAxisf(angle, Eigen::Vector3f::UnitY());
    apply_transformation(T);
}


void SceneObject::rotate_z(float angle) throw()
{
    Eigen::Transform3f T;
    T = Eigen::AngleAxisf(angle, Eigen::Vector3f::UnitZ());
    apply_transformation(T);
}


void SceneObject::set_orientation(
    const Eigen::Vector3f &v, CoordinateSystem system)
{
    if( v.isApprox(Eigen::Vector3f(0,0,0)) )
        throw std::logic_error("Orientation must be a non-zero vector");

    // Check for infinity and NaN
    for( int i = 0; i < 3; i++ )
        if( !boost::math::isfinite(v(i)) )
            throw std::logic_error(
                "Orientation vector cannot contain infinity or NaN");

    Eigen::Vector3f orient;
    if( system == LOCAL_COORDINATES )
    {
        orient = get_transformation().linear() * v;
    }
    else if( system == PARENT_COORDINATES )
    {
        orient = v;
    }
    else if( system == WORLD_COORDINATES )
    {
        orient = Eigen::Transform3f(
            get_parent_mtow().inverse(Eigen::Isometry)).linear() * v;
    }
    else
        assert( false );

    orient.normalize(); // Important! The equality test will fail otherwise
    const Eigen::Transform3f &m = get_transformation();
    const Eigen::Vector3f &old_orient = m.linear().col(0);

    if( old_orient.isApprox(orient, 0.001) )
    {
        // It's already correctly oriented
        return;
    }
    else if( old_orient.isApprox(-orient, 0.001) )
    {
        // The current and requested orientation are parallel but with opposite
        // directions; in this case the there is not a unique solution. The
        // requested orientation can be achieved by rotating +-180 degrees
        // about either local Z or local Y. By convention we will choose the
        // rotate-around-local-Z solution.
        set_transformation(Eigen::AngleAxisf(M_PI, m.linear().col(2)) * m);
    }
    else
    {
        Eigen::Vector3f axis = orient.cross(old_orient);
        float theta = acosf(old_orient.dot(orient));
        float x = m(0,3), y = m(1,3), z = m(2,3);
        set_transformation(
            Eigen::Translation3f(x, y, z) *
            Eigen::AngleAxisf(-theta, axis) *
            Eigen::Translation3f(-x, -y, -z) * m);
    }
}


void SceneObject::set_color(const RGBColor &color, bool recursive)
{
    m_color = color;
    m_color_set_signal();

    if( recursive )
    {
        for( Children::iterator it = m_children.begin();
             it != m_children.end(); it++ )
        {
            (*it)->set_color(color, true);
        }
    }
}


RGBColor SceneObject::get_color() const throw()
{
    return m_color;
}


void SceneObject::set_selected(bool select)
{
    if( select != m_is_selected )
    {
        m_is_selected = select;
        m_selected_set_signal();

        if( !has_selected_ancestor() )
        {
            std::stack<SceneObject *> temp_stack;
            BOOST_FOREACH( SceneObject *s, m_children )
            {
                temp_stack.push(s);
            }

            while( !temp_stack.empty() )
            {
                SceneObject *s = temp_stack.top();
                temp_stack.pop();

                s->m_has_selected_ancestor_changed_signal();

                if( !s->is_selected() )
                {
                    BOOST_FOREACH( SceneObject *ch, s->m_children )
                    {
                        temp_stack.push(ch);
                    }
                }
            }
        }
    }
}


bool SceneObject::is_selected() const throw()
{
    return m_is_selected;
}


bool SceneObject::has_selected_ancestor() const throw()
{
    const SceneObject *p = get_parent();
    while( p )
    {
        if( p->is_selected() )
            return true;
        p = p->get_parent();
    }

    return false;
}


Prop *SceneObject::get_prop(PropKey key)
{
    PropIterator it = get_prop_adapters().find(key);
    if( it != get_prop_adapters().end() )
    {
        return it->second;
    }
    else
    {
        it = m_props.find(key);
        if( it == m_props.end() )
            throw std::runtime_error("No such property");
        return it->second;
    }
}


const Prop *SceneObject::get_prop(PropKey key) const
{
    ConstPropIterator it = get_prop_adapters().find(key);
    if( it != get_prop_adapters().end() )
    {
        return it->second;
    }
    else
    {
        it = m_props.find(key);
        if( it == m_props.end() )
            throw std::runtime_error("No such property");
        return it->second;
    }
}


Any SceneObject::get_prop_value(PropKey key) const
{
    return get_prop(key)->get(this);
}


void SceneObject::set_prop_value(PropKey key, const Any &val)
{
    get_prop(key)->set(val, this);
}


std::size_t SceneObject::prop_adapter_count() const
{
    return get_prop_adapters().size();
}


SceneObject::PropIterator SceneObject::prop_adapters_begin()
{
    return get_prop_adapters().begin();
}


SceneObject::ConstPropIterator SceneObject::prop_adapters_begin() const
{
    return get_prop_adapters().begin();
}


SceneObject::PropIterator SceneObject::prop_adapters_end()
{
    return get_prop_adapters().end();
}


SceneObject::ConstPropIterator SceneObject::prop_adapters_end() const
{
    return get_prop_adapters().end();
}

std::size_t SceneObject::prop_count() const
{
    return m_props.size();
}


SceneObject::PropIterator SceneObject::props_begin()
{
    return m_props.begin();
}


SceneObject::ConstPropIterator SceneObject::props_begin() const
{
    return m_props.begin();
}


SceneObject::PropIterator SceneObject::props_end()
{
    return m_props.end();
}


SceneObject::ConstPropIterator SceneObject::props_end() const
{
    return m_props.end();
}


void SceneObject::erase_prop(PropKey key)
{
    PropIterator it = m_props.find(key);
    if( it == m_props.end() )
        throw std::runtime_error("No such property");
    erase_prop(it);
}


void SceneObject::erase_prop(PropIterator it)
{
    PropKey key = it->first;
    Prop *prop = it->second;
    m_props.erase(it);
    m_prop_erased_signal(key, prop);
    delete prop;
}


bool SceneObject::has_prop(PropKey key) const
{
    return m_props.find(key) != m_props.end();
}


PropKey SceneObject::get_prop_key(const std::string &name) const
{
    PropNames::right_map::iterator it = ms_prop_names.right.find(name);
    if( it == ms_prop_names.right.end() )
        throw std::runtime_error("No such property");
    return it->second;
}


PropKey SceneObject::add_prop(const std::string &name, Prop *prop)
{
    PropKey key;
    PropNames::right_map::iterator it = ms_prop_names.right.find(name);
    if( it == ms_prop_names.right.end() )
    {
        key = ms_next_prop_key++;
        ms_prop_names.insert( PropNames::value_type(key, name) );
    }
    else
    {
        key = it->second;
    }

    if( has_prop(key) )
        throw std::runtime_error("Property already exists");

    m_props.insert(std::make_pair(key, prop));

    m_prop_added_signal(key, prop);

    return key;
}


const std::string &SceneObject::get_prop_name(PropKey key) const
{
    PropNames::left_map::const_iterator it = ms_prop_names.left.find(key);
    if( it == ms_prop_names.left.end() )
    {
        static const std::string s_unknown = "<unknown>";
        return s_unknown;
    }
    else
    {
        return it->second;
    }
}


SceneObject::SceneObject(
    const std::string &default_name, ScopedHandler *handler)
    : m_is_hidden(false),
      m_is_visible(true),
      m_parent(0),
      m_layer(0),
      m_opacity(1.0f),
      m_name(default_name),
      m_id(ms_next_object_id++),
      m_transforms_since_normalization(0),
      m_color(0, 0.5, 0),
      m_is_selected(false)
{
    Eigen::Transform3f tmp;
    tmp.setIdentity();
    set_transformation(tmp);

    // If there is a layer or color variable set, change the color and/or
    // layer of this object accordingly.
    ScopedMap & variables = handler->get_variables();

    if(variables.exists<boost::uint8_t>("layer"))
        m_layer = variables.get_variable<boost::uint8_t>("layer") - 1;

    // Save the original name of the object for later (overriden by
    // the name_cdata_handler, when a name tag is present).
    variables.get_variable<std::map<SceneObject *, std::string> >(
        "orig_names")[this] = m_name;

    // Attach the object to its parent.
    SceneObject* parent = variables.get_variable<SceneObject*>("current_object");
    parent->attach(this, AUTO_ENUMERATE_ON_CONFLICT);

    ScopedHandler::TagScope scope;

    // Specify the scene object end handler for the new scope.
    scope.end_functors.push_back(
        boost::bind(&SceneObject::end_handler, this, _1, _2));

    // If the object's color is being overriden, set up a function
    // that will set the appropriate color when the end tag is encountered -
    // we can't set it right away since ctors higher up in the hierarchy
    // might change m_color (if they have special defaults, like models and
    // cameras)
    if(variables.exists<RGBColor>("color"))
    {
        const RGBColor &color = variables.get_variable<RGBColor>("color");

        boost::function<void (void)> f = boost::bind(
            &SceneObject::set_color, this, color, false);

        scope.end_functors.push_back(
            boost::bind(&end_handler_wrapper, _1, _2, f));
    }


    handler->enter_scope(scope);


    // Register the name tag start handler
    handler->add_start_handler(
        "name",
        boost::bind(&SceneObject::name_start_handler, this, _1, _2, _3));

    // Register the opacity tag start handler
    handler->add_start_handler(
        "opacity",
        boost::bind(&SceneObject::opacity_start_handler, this, _1, _2, _3));

    // Register the hidden tag start handler
    handler->add_start_handler(
        "hidden",
        boost::bind(&SceneObject::hidden_start_handler, this, _1, _2, _3));

    // Register the transform tag start handler
    handler->add_start_handler(
        "transform",
        boost::bind(&SceneObject::transform_start_handler, this, _1, _2, _3));

    // Register the children tag start handler
    handler->add_start_handler(
        "children",
        boost::bind(&SceneObject::children_start_handler, this, _1, _2, _3));

    // Register handler for ignoring all cdata/elements wrapped in an
    // <external>-element.
    handler->add_start_handler("external", &external_handler);
}


void SceneObject::end_handler(
    const std::string &name, ScopedHandler *handler) throw(std::runtime_error)
{
    // Pop the current_object variable since we're exiting an object element
    ScopedMap & variables = handler->get_variables();
    variables.pop_variable<SceneObject*>("current_object");

    ++variables.get_variable<unsigned int>("object_counter");
}


void SceneObject::name_start_handler(
    const std::string &name,
    ScopedHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope name_scope;
    name_scope.cdata_functor = boost::bind(&SceneObject::name_cdata_handler,
                                           this, _1, _2);

    // Enter a new scope where only cdata is valid, and is handled by
    // the name_cdata_handler() method.
    handler->enter_scope(name_scope);
}


void SceneObject::name_cdata_handler(
    const std::string &cdata, ScopedHandler *handler) throw(std::runtime_error)
{
    std::string name = cdata;
    boost::trim(name);

    try
    {
        set_name(name);
        // Override the default original name
        handler->get_variables().get_variable<
            std::map<SceneObject *, std::string> >("orig_names")[this] = name;
    }
    catch(std::exception &e)
    {
        std::string error = "Name conflict during parsing: ";
        error += name;
        throw(std::runtime_error(error));
    }
}


void SceneObject::opacity_start_handler(
    const std::string &name,
    ScopedHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope opacity_scope;

    bool absolute = false;

    // Check if the opacity is absolute
    absolute = attributes["absolute"] == "true";

    opacity_scope.cdata_functor = boost::bind(
        &SceneObject::opacity_cdata_handler, this, absolute, _1, _2);

    // Enter a new scope where only cdata is valid, and is handled by
    // the meta_cdata_handler() method.
    handler->enter_scope(opacity_scope);
}


void SceneObject::opacity_cdata_handler(
    bool absolute,
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    set_opacity(boost::lexical_cast<float>(cdata));
}


void SceneObject::hidden_start_handler(
    const std::string &name,
    ScopedHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{

    m_is_hidden = true;

    // Enter a new scope to make sure the scope stack
    // won't be f***ed up (empty tags have end tags,
    // even if they're not explicit, so the top
    // end handler will get called).
    handler->enter_scope();
}


void SceneObject::transform_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    // Add transformation tag handlers to the scope.
    scope.start_functors["rotate"] = boost::bind(
        &SceneObject::rotate_start_handler, this, _1, _2, _3);

    scope.start_functors["translate"] = boost::bind(
        &SceneObject::translate_start_handler, this, _1, _2, _3);

    scope.start_functors["matrix"] = boost::bind(
        &SceneObject::matrix_start_handler, this, _1, _2, _3);

    handler->enter_scope(scope);
}


void SceneObject::rotate_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    if(name == "rotate")
    {
        ScopedHandler::TagScope rotation_scope;

        rotation_scope.start_functors["axis"] = boost::bind(
            &SceneObject::rotate_start_handler, this, _1, _2, _3);

        rotation_scope.start_functors["pivot"] = boost::bind(
            &SceneObject::rotate_start_handler, this, _1, _2, _3);

        rotation_scope.start_functors["radians"] = boost::bind(
            &SceneObject::rotate_start_handler, this, _1, _2, _3);

        rotation_scope.start_functors["degrees"] = boost::bind(
            &SceneObject::rotate_start_handler, this, _1, _2, _3);

        rotation_scope.end_functors.push_back(
            boost::bind(&SceneObject::rotate_end_handler, this, _1, _2));

        handler->enter_scope(rotation_scope);
    }
    else if(name == "axis")
    {
        // Check which coordinate system we're using
        CoordinateSystem system;
        if(attributes["system"] == "world")
            system = WORLD_COORDINATES;
        else if(attributes["system"] == "parent")
            system = PARENT_COORDINATES;
        else
            system = LOCAL_COORDINATES;

        ScopedHandler::TagScope axis_scope;

        axis_scope.cdata_functor = boost::bind(
            &SceneObject::rotate_cdata_handler, this, 1, // 1 -> tag = axis
            system, _1, _2);

        handler->enter_scope(axis_scope);
    }
    else if(name == "pivot")
    {
        // Check which coordinate system we're using
        CoordinateSystem system;
        if(attributes["system"] == "world")
            system = WORLD_COORDINATES;
        else if(attributes["system"] == "parent")
            system = PARENT_COORDINATES;
        else
            system = LOCAL_COORDINATES;

        ScopedHandler::TagScope pivot_scope;

        pivot_scope.cdata_functor = boost::bind(
            &SceneObject::rotate_cdata_handler, this, 2, // 2 -> tag = pivot
            system, _1, _2);

        handler->enter_scope(pivot_scope);
    }
    else if(name == "radians")
    {
        ScopedHandler::TagScope radians_scope;

        radians_scope.cdata_functor = boost::bind(
            &SceneObject::rotate_cdata_handler, this, 3, // 3 -> tag = radians
            LOCAL_COORDINATES, _1, _2); // bogus system

        handler->enter_scope(radians_scope);
    }
    else if(name == "degrees")
    {
        ScopedHandler::TagScope degrees_scope;

        degrees_scope.cdata_functor = boost::bind(
            &SceneObject::rotate_cdata_handler, this, 4, // 4 -> tag = degrees
            LOCAL_COORDINATES, _1, _2); // bogus system

        handler->enter_scope(degrees_scope);
    }
}


void SceneObject::rotate_cdata_handler(
    boost::uint32_t tag,
    CoordinateSystem system,
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    ScopedMap &variables = handler->get_variables();

    std::stringstream s(cdata);

    if(tag == 1 || tag == 2) // axis or pivot tag
    {
        Eigen::Vector4f vec;
        s >> vec(0) >> vec(1) >> vec(2);

        if(tag == 1) // If we're transforming an axis, vec(3) needs to be 0
            vec(3) = 0;
        if(tag == 2) // For a point coordinate, it needs to be 1.
            vec(3) = 1;

        // If we're using parent or local coordinates we need to transform
        // the vector.
        if(system == PARENT_COORDINATES)
        {
            vec = get_parent_mtow() * vec;
        }
        else if(system == LOCAL_COORDINATES)
        {
            vec = get_parent_mtow() * m_mtop * vec;
        }

        // Always zero for axes, also zero when we translate to pivot!
        vec(3) = 0;

        if(tag == 1)
            variables.push_variable("rotation_axis", vec);
        else
            variables.push_variable("rotation_pivot", vec);
    }
    else // radians or degrees tag
    {
        float rad;
        s >> rad;

        if(tag == 4) // If it's in degrees, convert into radians
            rad *= 2.0 * 3.141592654 / 360.0;

        variables.push_variable("rotation_rad", rad);
    }
}


void SceneObject::rotate_end_handler(
    const std::string &name, ScopedHandler *handler) throw()
{
    // Fetch rotation parameters
    ScopedMap &variables = handler->get_variables();
    Eigen::Vector4f axis;
    if(variables.exists<Eigen::Vector4f>("rotation_axis"))
    {
        axis = variables.get_variable<Eigen::Vector4f>("rotation_axis");
        variables.pop_variable<Eigen::Vector4f>("rotation_axis");
    }
    else
        axis = Eigen::Vector4f(1, 0, 0, 0);

    Eigen::Vector4f pivot;
    if(variables.exists<Eigen::Vector4f>("rotation_pivot"))
    {
        pivot = variables.get_variable<Eigen::Vector4f>("rotation_pivot");
        variables.pop_variable<Eigen::Vector4f>("rotation_pivot");
    }
    else
        pivot = Eigen::Vector4f(0, 0, 0, 0);

    float rad =  variables.get_variable<float>("rotation_rad");
    variables.pop_variable<float>("rotation_rad");

    set_transformation(
        Eigen::Transform3f(get_parent_mtow().inverse(Eigen::Isometry)) *
        Eigen::Translation3f(pivot(0), pivot(1), pivot(2)) *
        Eigen::AngleAxisf(rad, Eigen::Vector3f(axis(0), axis(1), axis(2))) *
        Eigen::Translation3f(-pivot(0), -pivot(1), -pivot(2)) *
        get_mtow());
}


void SceneObject::translate_start_handler(
            const std::string &name,
            XMLHandler::AttributeMap &attributes,
            ScopedHandler *handler) throw()
{
    // Check which coordinate system we're using
    CoordinateSystem system;
    if(attributes["system"] == "world")
        system = WORLD_COORDINATES;
    else if(attributes["system"] == "parent")
        system = PARENT_COORDINATES;
    else
        system = LOCAL_COORDINATES;

    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &SceneObject::translate_cdata_handler, this, system, _1, _2);

    handler->enter_scope(scope);
}


void SceneObject::translate_cdata_handler(
    CoordinateSystem system,
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    // Read coordinate information
    std::stringstream s(cdata);
    float x, y, z;
    s >> x >> y >> z;

    Eigen::Transform3f m;

    if(system == PARENT_COORDINATES)
    {
        m = Eigen::Translation3f(x, y, z) *
            m_mtop;
    }
    else if(system == WORLD_COORDINATES)
    {
        m = Eigen::Transform3f(get_parent_mtow().inverse(Eigen::Isometry)) *
            Eigen::Translation3f(x, y, z) *
            get_mtow();
    }
    else if(system == LOCAL_COORDINATES)
    {
        m = m_mtop;
        m.translation() +=
            x * m.linear().col(0) +
            y * m.linear().col(1) +
            z * m.linear().col(2);
    }
    else
    {
        assert( false );
    }

    set_transformation(m);
}


void SceneObject::matrix_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Check which coordinate system we're using
    CoordinateSystem system;
    if(attributes["system"] == "world")
        system = WORLD_COORDINATES;
    else if(attributes["system"] == "parent")
        system = PARENT_COORDINATES;
    else
        system = LOCAL_COORDINATES;

    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &SceneObject::matrix_cdata_handler, this, system, _1, _2);

    handler->enter_scope(scope);
}


void SceneObject::matrix_cdata_handler(
    CoordinateSystem system,
    const std::string &cdata,
    ScopedHandler *handler) throw()
{
    std::stringstream s(cdata);
    Eigen::Transform3f m;

    // Read our matrix.
    s >> m.matrix()(0, 0) >> m.matrix()(0, 1) >> m.matrix()(0, 2) >> m.matrix()(0, 3)
      >> m.matrix()(1, 0) >> m.matrix()(1, 1) >> m.matrix()(1, 2) >> m.matrix()(1, 3)
      >> m.matrix()(2, 0) >> m.matrix()(2, 1) >> m.matrix()(2, 2) >> m.matrix()(2, 3)
      >> m.matrix()(3, 0) >> m.matrix()(3, 1) >> m.matrix()(3, 2) >> m.matrix()(3, 3);

    // Normalize to make sure it's good.
    m.linear().col(0).normalize();
    m.linear().col(1).normalize();
    m.linear().col(2).normalize();

    if(system == WORLD_COORDINATES)
    {
        // Here we need to apply the inverse parent to translate into local
        m = Eigen::Transform3f(get_parent_mtow().inverse(Eigen::Isometry)) * m;
    }

    // If the system was PARENT_COORDINATES, we're safe.

    set_transformation(m);
}


void SceneObject::children_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // children scope - allow all global start handlers
    ScopedHandler::TagScope children_scope;
    children_scope.start_functors = SceneFileLoader::get_start_handlers();
    handler->enter_scope(children_scope);
}


SceneObject *SceneObject::_get_descendent(
    const std::vector<std::string>::const_iterator first,
    const std::vector<std::string>::const_iterator last) throw()
{
    if( first == last )
        return 0; // Path is empty <=> no such descendent!

    for( Children::iterator it = m_children.begin();
         it != m_children.end(); ++it )
    {
        if( (*it)->get_name() == *first )
        {
            if( std::distance(first, last) == 1 )
            {
                return *it;
            }
            else
            {
                return (*it)->_get_descendent(first+1,last);
            }
        }
    }

    return 0;
}


const SceneObject *SceneObject::_get_descendent(
    const std::vector<std::string>::const_iterator first,
    const std::vector<std::string>::const_iterator last) const throw()
{
    if( first == last )
        return 0; // Path is empty <=> no such descendent!

    for( Children::const_iterator it = m_children.begin();
         it != m_children.end(); ++it )
    {
        if( (*it)->get_name() == *first )
        {
            if( std::distance(first, last) == 1 )
            {
                return *it;
            }
            else
            {
                return (*it)->_get_descendent(first+1,last);
            }
        }
    }

    return 0;
}
