/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <boost/scoped_array.hpp>

#include "PbarWriter.hh"
#include "Action.hh"
#include "Bytesex.hh"
#include "serialization/SerializationInterface.hh"
#include "serialization/StreambufAdapter.hh"
#include "serialization/MemAdapters.hh"
#include "Version.hh"

extern "C" {
#include <liblzf/lzf.h>
}


using namespace peekabot;


namespace
{
    const uint32_t PBAR_VERSION = 3;
    const char PBAR_MAGIC_MARKER[] = "pbar";
}


PbarWriter::PbarWriter(std::ostream &os, bool multi_client)
    : m_os(os),
      m_action_count(0),
      m_is_multi_client(multi_client)
{
    // Write PBAR preamble
    m_os.write(PBAR_MAGIC_MARKER, 4);

    uint8_t is_be = (PEEKABOT_BYTE_ORDER == PEEKABOT_BIG_ENDIAN);
    StreambufAdapter adapter(*(m_os.rdbuf()));
    SerializationInterface buf(adapter);

    buf << is_be
        << PBAR_VERSION
        << boost::uint32_t(PEEKABOT_VERSION | (PEEKABOT_RELEASE_STATUS << 24))
        << uint8_t(m_is_multi_client ? 1:0);

    std::streampos cpos = m_os.tellp();

    // Write PBAR epilogue
    buf << (uint32_t)0  // action count
        << (uint32_t)0  // duration, s
        << (uint16_t)0; // duration, ms

    m_os.seekp(cpos);
}


void PbarWriter::write(
    const boost::shared_ptr<Action> a,
    const boost::posix_time::time_duration &timestamp,
    int32_t client_id)
{
    MemSerializationBuffer uncomp;
    SerializationInterface ar(uncomp);
    ar << a.get();

    uint32_t uncomp_len = (uint32_t)uncomp.size(), comp_len = uncomp_len;
    boost::scoped_array<boost::uint8_t> comp;

    // Compress data?
    if( uncomp_len > 128 )
    {
        // require at least 5% compression, or it's not worth the effort
        comp.reset(new boost::uint8_t[95*uncomp_len/100]);
        comp_len = (uint32_t)lzf_compress(
            uncomp.get(), uncomp_len, comp.get(), 95*uncomp_len/100-1);

        if( comp_len == 0 )
        {
            // Failed to compress... fall back to uncompressed
            comp.reset();
            comp_len = uncomp_len;
        }
    }


    StreambufAdapter adapter(*(m_os.rdbuf()));
    SerializationInterface buf(adapter);

    assert( !timestamp.is_negative() );
    uint32_t s = timestamp.total_seconds();
    uint16_t ms = (uint16_t)(
        timestamp-boost::posix_time::seconds(
            timestamp.total_seconds())).total_milliseconds();

    // Write action header + action body
    uint8_t ctrl_byte = comp ? 1:0;
    buf << s << ms << (comp ? comp_len : uncomp_len) << ctrl_byte;

    if( comp )
    {
        buf << uncomp_len;
        buf.save_binary(comp.get(), comp_len);
    }
    else
    {
        buf << a.get();
    }

    std::streampos cpos = m_os.tellp();

    // Write epilogue
    buf << ++m_action_count << s << ms;

    m_os.seekp(cpos, std::ios_base::beg);
}


void PbarWriter::flush()
{
    m_os.flush();
}
