/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <stdexcept>
#include <boost/scoped_array.hpp>

#include "PbarReader.hh"
#include "Action.hh"
#include "serialization/DeserializationInterface.hh"
#include "serialization/StreambufAdapter.hh"
#include "serialization/MemAdapters.hh"
#include "Version.hh"

extern "C" {
#include <liblzf/lzf.h>
}


using namespace peekabot;


namespace
{
    const size_t EPILOGUE_LEN = 10;
    const uint32_t CURRENT_PBAR_VERSION = 3;
    const uint32_t MIN_PBAR_VERSION = 2;
}



PbarReader::PbarReader(const std::string &filename)
    : m_ifs(filename.c_str(), std::ios::binary)
{
    if( !m_ifs )
        throw std::runtime_error(
            "Failed to open recording file '" + filename + "' for reading");

    // Determine file size
    m_ifs.seekg(0, std::ios_base::end);
    m_file_size = m_ifs.tellg();
    m_ifs.seekg(0, std::ios_base::beg);

    read_preamble();
    read_timestamp();
    read_epilogue();
}


void PbarReader::get_peekabot_version(
    uint8_t &major, uint8_t &minor, uint8_t &revision, uint8_t &rc) const
{
    major = (m_peekabot_version>>16) & 0xFF;
    minor = (m_peekabot_version>>8) & 0xFF;
    revision = m_peekabot_version & 0xFF;
    rc = (m_peekabot_version>>24) & 0xFF;
}


uint32_t PbarReader::get_format_version() const
{
    return m_pbar_version;
}


bool PbarReader::is_multi_client_recording() const
{
    return m_multi_client_recording;
}


boost::shared_ptr<Action> PbarReader::get_next_action()
{
    if( eof() )
        throw std::runtime_error("EOF");

    StreambufAdapter adapter(*m_ifs.rdbuf());
    DeserializationInterface buf(adapter, m_file_is_be);

    Action *action = 0;

    if( m_pbar_version == 2 )
    {
        buf >> action;
    }
    else if( m_pbar_version == 3 )
    {
        uint32_t body_size;
        uint8_t ctrl_byte;
        buf >> body_size >> ctrl_byte;

        if( ctrl_byte == 1 )
        {
            uint32_t comp_len = body_size;
            uint32_t uncomp_len;
            buf >> uncomp_len;

            boost::scoped_array<boost::uint8_t> comp(
                new boost::uint8_t[comp_len]);

            buf.load_binary(comp.get(), comp_len);

            boost::scoped_array<boost::uint8_t> uncomp(
                new boost::uint8_t[uncomp_len]);

            if( lzf_decompress(comp.get(), comp_len, uncomp.get(), uncomp_len)
                != uncomp_len )
                throw std::runtime_error("Failed to decompress action");

            MemDeserializationBuffer membuf(uncomp.get(), uncomp_len);
            DeserializationInterface ar(membuf, m_file_is_be);

            ar >> action;
        }
        else
        {
            buf >> action;
        }
    }
    else
        throw std::runtime_error("Unknown PBAR format version");

    // Read timestamp for next action
    read_timestamp();

    return boost::shared_ptr<Action>(action);
}


const boost::posix_time::time_duration &
PbarReader::get_next_action_time() const
{
    if( eof() )
        throw std::runtime_error("EOF");
    else
        return m_next_t;
}


void PbarReader::read_preamble()
{
    char uid[4];
    m_ifs.read(uid, 4);
    if( m_ifs.gcount() != 4 )
        throw std::runtime_error("Error reading PBAR preamble");

    if( uid[0] != 'p' || uid[1] != 'b' || uid[2] != 'a' || uid[3] != 'r' )
        throw std::runtime_error(
            "The given file does not appear to be a PBAR file");

    uint8_t is_big_endian;
    m_ifs.read(reinterpret_cast<char *>(&is_big_endian), 1);
    if( m_ifs.gcount() != 1 )
        throw std::runtime_error("Error reading PBAR preamble");

    if( is_big_endian != 0 && is_big_endian != 1 )
    {
        throw std::runtime_error(
            "Invalid preamble - LE/BE field contains invalid data");
    }

    m_file_is_be = is_big_endian;

    StreambufAdapter adapter(*m_ifs.rdbuf());
    DeserializationInterface buf(adapter, m_file_is_be);

    try
    {
        buf >> m_pbar_version
            >> m_peekabot_version
            >> m_multi_client_recording;
    }
    catch(...)
    {
        throw std::runtime_error("Error reading PBAR preamble");
    }

    if( m_pbar_version < MIN_PBAR_VERSION )
        throw std::runtime_error("Unsupported PBAR format version");
    else if( m_pbar_version > CURRENT_PBAR_VERSION )
        throw std::runtime_error(
            "The PBAR format version is newer than the current format, "
            "please upgrade to a later version of peekabot to read this file");

    if( (m_peekabot_version & 0x00FFFFFF) < PEEKABOT_PBAR_COMPATIBLE_VERSION )
        throw std::runtime_error(
            "PBAR file produced by an older, incompatible peekabot version");

    if( (m_peekabot_version & 0x00FFFFFF) > PEEKABOT_VERSION )
        throw std::runtime_error(
            "PBAR file produced by an newer peekabot version, please "
            "upgrade to a later version of peekabot to read this file");
}


void PbarReader::read_timestamp()
{
    if( eof() )
        return;

    StreambufAdapter adapter(*m_ifs.rdbuf());
    DeserializationInterface buf(adapter, m_file_is_be);

    // Read timestamp (seconds, milliseconds)
    uint32_t s;
    uint16_t ms;
    buf >> s >> ms;

    m_next_t = boost::posix_time::seconds((long)s) +
        boost::posix_time::milliseconds((long)ms);
}


void PbarReader::read_epilogue()
{
    const std::streampos orig_pos = m_ifs.tellg();

    try
    {
        StreambufAdapter adapter(*m_ifs.rdbuf());
        DeserializationInterface buf(adapter, m_file_is_be);

        // Seek to end, read epilogue
        m_ifs.seekg(-EPILOGUE_LEN, std::ios_base::end);
        uint32_t s;
        uint16_t ms;
        buf >> m_action_count >> s >> ms;

        m_duration = boost::posix_time::seconds((long)s) +
            boost::posix_time::milliseconds((long)ms);

        // Restore original position in stream
        m_ifs.seekg(orig_pos);
    }
    catch(...)
    {
        m_ifs.seekg(orig_pos);
        throw std::runtime_error("Error reading PBAR epilogue");
    }
}


bool PbarReader::eof() const
{
    // There are no more actions to be read if we're at the epilogue
    return (size_t)const_cast<std::ifstream &>(m_ifs).tellg() >=
        m_file_size-EPILOGUE_LEN;
}


const boost::posix_time::time_duration &PbarReader::duration() const
{
    return m_duration;
}


size_t PbarReader::action_count() const
{
    return m_action_count;
}
