/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include <cassert>
#include <stdexcept>
#include <boost/bind.hpp>
#include <boost/thread/thread.hpp>

#include "PbarPlayer.hh"


using namespace peekabot;


namespace
{
    void sleep(const boost::posix_time::time_duration &d)
    {
        const boost::system_time timeout = boost::get_system_time() + d;
        while( boost::get_system_time() < timeout )
        {
            try
            {
                boost::thread::sleep(timeout);
            }
            catch(boost::thread_interrupted &) {}
        }
    }
}


PbarPlayer::PbarPlayer(
    const std::string &filename,
    boost::function<void (boost::shared_ptr<Action>)> dispatch_fun)
    : m_dispatch_fun(dispatch_fun),
      m_reader(filename),
      m_paused(true),
      m_playback_spd(1),
      m_elapsed(boost::posix_time::milliseconds(0)),
      m_thread(0),
      m_stop_signal(false)
{
    m_thread = new boost::thread(boost::bind(&PbarPlayer::run, this));
}


PbarPlayer::~PbarPlayer()
{
    if( m_thread )
    {
        m_stop_signal = true;
        m_play_cond.notify_all(); // Wake run-loop if paused
        m_thread->join();
        delete m_thread;
        m_thread = 0;
    }
}


void PbarPlayer::play()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    m_paused = false || m_reader.eof();
    if( !m_paused )
        m_play_cond.notify_all();
}


void PbarPlayer::pause()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    m_paused = true;
}


bool PbarPlayer::is_paused() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    return m_paused;
}


bool PbarPlayer::is_finished() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    return m_reader.eof();
}


void PbarPlayer::step(size_t n)
{
    if( n < 1 )
        return;

    boost::recursive_mutex::scoped_lock lock(m_mutex);
    if( !is_paused() )
        throw std::runtime_error("Can only step when paused");

    if( is_finished() )
        return;

    for( size_t i = 0; i < n; ++i )
    {
        if( is_finished() )
            return;

        if( m_buffer.empty() )
            buffer();

        m_elapsed = m_buffer.front().first;
        m_dispatch_fun(m_buffer.front().second);
        m_buffer.pop();
    }
}


void PbarPlayer::step(const boost::posix_time::time_duration &td)
{
    if( td.is_negative() )
        return;

    boost::recursive_mutex::scoped_lock lock(m_mutex);
    if( !is_paused() )
        throw std::runtime_error("Can only step when paused");

    if( is_finished() )
        return;

    m_elapsed += td;

    while( true )
    {
        if( is_finished() )
            return;

        if( m_buffer.empty() )
            buffer();

        if( m_buffer.front().first > m_elapsed )
            return;
        else
        {
            m_dispatch_fun(m_buffer.front().second);
            m_buffer.pop();
        }
    }
}


void PbarPlayer::set_playback_speed(double factor)
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    if( factor <= 0 )
        throw std::runtime_error(
            "The playback speed factor must be greater than zero");
    m_playback_spd = factor;
}


double PbarPlayer::get_playback_speed() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    return m_playback_spd;
}


bool PbarPlayer::is_multi_client_recording() const
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);
    return m_reader.is_multi_client_recording();
}


boost::posix_time::time_duration PbarPlayer::elapsed() const
{
    return m_elapsed;
}


boost::posix_time::time_duration PbarPlayer::duration() const
{
    return m_reader.duration();
}


size_t PbarPlayer::action_count() const
{
    return m_reader.action_count();
}


void PbarPlayer::run()
{
    boost::posix_time::ptime t2, t1 =
        boost::posix_time::microsec_clock::local_time();

    while( !m_stop_signal )
    {
        boost::posix_time::time_duration rem;

        {
            boost::recursive_mutex::scoped_lock lock(m_mutex);

            // Advance time
            t2 = boost::posix_time::microsec_clock::local_time();

            // Note: In Boost 1.35, multiplying with a value less than 1
            // seems to trigger some strange bug in date-time, thus the
            // eyesore below
            if( m_playback_spd >= 1 )
                m_elapsed += (t2-t1) * m_playback_spd;
            else
                m_elapsed += (t2-t1) / (1/m_playback_spd);

            t1 = t2;

            if( m_reader.eof() )
            {
                m_stop_signal = true;
            }
            else if( m_paused )
            {
                m_play_cond.wait(lock);
                t1 = boost::posix_time::microsec_clock::local_time();
            }
            else
            {
                // Process action? Buffer? Sleep?
                if( m_buffer.empty() )
                {
                    buffer();
                }
                else if( !m_buffer.empty() )
                {
                    if( m_elapsed >= m_buffer.front().first )
                    {
                        m_dispatch_fun(m_buffer.front().second);
                        m_buffer.pop();
                    }
                    else
                        rem = m_buffer.front().first - m_elapsed;
                }
            }
        }

        if( !rem.is_special() )
        {
            const boost::posix_time::time_duration tmp =
                boost::posix_time::milliseconds(20);
            sleep(std::min(tmp, rem));
        }
    }
}


void PbarPlayer::buffer()
{
    boost::recursive_mutex::scoped_lock lock(m_mutex);

    // Get next action + timestamp
    boost::posix_time::time_duration td = m_reader.get_next_action_time();
    boost::shared_ptr<Action> a = m_reader.get_next_action();

    m_buffer.push(std::make_pair(td, a));
}
