/*
 * Copyright Staffan Gimåker 2008, 2010.
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

#include "PathIdentifier.hh"

#ifdef __PEEKABOT_SERVER
#  include <boost/algorithm/string/split.hpp>
#  include <boost/algorithm/string/classification.hpp>

#  include "SceneObject.hh"
#  include "ServerExecutionContext.hh"
#endif


using namespace peekabot;


PathIdentifier::PathIdentifier() throw()
    : m_parent_id((ObjectID)-1),
      m_path("")
{
}


PathIdentifier::PathIdentifier(ObjectID parent_id, const std::string &rel_path)
    throw(std::runtime_error)
    : m_parent_id(parent_id),
      m_path(rel_path)
{
}


PathIdentifier::PathIdentifier(const std::string &path)
    throw(std::runtime_error)
    : m_parent_id((ObjectID)-1),
      m_path(path)
{
}


PathIdentifier::PathIdentifier(ObjectID id) throw()
    : m_parent_id(id),
      m_path("")
{
}


#ifdef __PEEKABOT_SERVER
SceneObject *PathIdentifier::get_leaf(ServerExecutionContext *context)
    const throw(std::runtime_error)
{
    SceneObject *obj = 0;

    if( m_parent_id != (ObjectID)-1 )
    {
        SceneObject *parent = context->get_object(m_parent_id);

        if( !parent )
            throw std::runtime_error(
                "Path parent not found, relative path: " + m_path);

        if( m_path == "" )
        {
            obj = parent;
        }
        else if( parent )
        {
            obj = context->get_object(m_path, parent);

            if( !obj )
                throw std::runtime_error(
                    "Object not found, relative path: " + m_path);
        }
    }
    else
    {
        obj = context->get_object(m_path);

        if( !obj )
            throw std::runtime_error("Object not found: " + m_path);
    }

    if( !obj )
        throw std::runtime_error("Object not found");

    return obj;
}


SceneObject *PathIdentifier::get_parent(ServerExecutionContext *context)
    const throw(std::runtime_error)
{
    SceneObject *parent = 0;

    if( m_parent_id != (ObjectID)-1 )
    {
        if( m_path == "" )
        {
            parent = context->get_object(m_parent_id)->get_parent();
        }
        else
        {
            // Path is relative m_parent_id

            SceneObject *path_parent = context->get_object(m_parent_id);
            if( !path_parent )
                throw std::runtime_error(
                    "Path parent not found, relative path: " + m_path);

            std::string::size_type pos = m_path.rfind('/');
            if( pos != std::string::npos )
                parent = context->get_object(m_path.substr(0,pos), path_parent);
            else
                parent = path_parent;

            if( !path_parent )
                throw std::runtime_error(
                    "Parent not found, relative path: " + m_path);
        }
    }
    else
    {
        std::string::size_type pos = m_path.rfind('/');
        if( pos != std::string::npos )
            parent = context->get_object(m_path.substr(0,pos));
        else
            parent = context->get_root();

        if( !parent )
            throw std::runtime_error("Parent not found: " + m_path);
    }

    if( !parent )
        throw std::runtime_error("Parent not found");

    return parent;
}


std::string PathIdentifier::get_leaf_name(ServerExecutionContext *context)
    const throw(std::runtime_error)
{
    if( m_parent_id != (ObjectID)-1 )
    {
        if( m_path == "" )
        {
            return get_leaf(context)->get_name();
        }
        else
        {
            // Path is relative m_parent_id

            std::vector<std::string> components;
            boost::split( components, m_path, boost::is_any_of("/") );
            if( components.empty() )
                components.push_back(m_path);
            return components.back();
        }
    }
    else
    {
        std::vector<std::string> components;
        boost::split( components, m_path, boost::is_any_of(".") );

        if( components.empty() )
            return "";
        else
            return components.back();
    }
}
#endif


void peekabot::save(SerializationInterface &ar, const PathIdentifier &path_ident)
{
    ar << path_ident.get_path() << path_ident.get_parent_id();
}

void peekabot::load(
    DeserializationInterface &ar, PathIdentifier &path_ident, int version)
{
    std::string path;
    ObjectID parent_id;
    ar >> path >> parent_id;
    path_ident.set_path(path);
    path_ident.set_parent_id(parent_id);
}
