/*
 * Copyright Staffan Gimåker 2010.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ClientConnection.hh"
#include "Server.hh"
#include "ServerExecutionContext.hh"
#include "Action.hh"
#include "UploadCache.hh"
#include "Config.hh"
#include "Path.hh"
#include "MessageHub.hh"

#include <boost/bind.hpp>
#include <stdexcept>


using namespace peekabot;


boost::shared_ptr<ClientConnection> ClientConnection::create(
    Server &server,
    boost::asio::io_service &io_service)
{
    return boost::shared_ptr<ClientConnection>(
        new ClientConnection(server, io_service));
}


ClientConnection::ClientConnection(
    Server &server,
    boost::asio::io_service &io_service)
    : Connection(io_service),
      ActionSource(server.get_config()),
      m_server(server)
{
}


void ClientConnection::start()
{
    authenticate();
}


void ClientConnection::execute_action(
    boost::shared_ptr<Action> action,
    ServerData &sd)
{
    ServerExecutionContext context(
        m_server.get_config(), *sd.m_scene, *this);

    try
    {
        action->execute(&context);
    }
    catch(std::exception &e)
    {
        TheMessageHub::instance().publish(
            ERROR_MESSAGE, e.what());
    }
    catch(...)
    {
        TheMessageHub::instance().publish(
            ERROR_MESSAGE,
            "Caught an unknown exception during action execution");
    }

    // Starting receiving the next action
    read_action();
}


boost::shared_ptr<Connection> ClientConnection::ptr()
{
    return shared_from_this();
}


void ClientConnection::action_read(
    const boost::shared_ptr<Action> &action)
{
    m_server.post(
        boost::bind(&ClientConnection::execute_action,
                    shared_from_this(), action, _1));
}


void ClientConnection::on_authenticated()
{
    // TODO: log a notification that a client connected?
    read_action();
}


void ClientConnection::on_authentication_failed(
    protocol::AuthenticationResult reason)
{
    // TODO: log a notification about this
}


void ClientConnection::send_response(const boost::shared_ptr<Action> &action)
{
    write_action(action);
}
