/*
 * Copyright Staffan Gimåker 2006-2010.
 * Copyright Anders Boberg 2006-2007.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cstring>
#include <boost/bind.hpp>

#include "SceneObject.hh"
#include "CameraObject.hh"
#include "ObjectVisitor.hh"
#include "ScopedHandler.hh"
#include "ScopedMap.hh"
#include "PropKeys.hh"
#include "props/BoolPropBase.hh"
#include "props/FloatPropBase.hh"
#include "ObjectTypes.hh"


using namespace peekabot;


namespace
{
    const RGBColor DEFAULT_COLOR(0, 0, 0);
}


HandlerInformer CameraObject::ms_handler_informer(
    "camera", &CameraObject::start_handler);


CameraObject::CameraObject() :
    SceneObject("camera"),
    m_fov(45),
    m_is_orthographic(false),
    m_near(1.0f),
    m_far(100.0f),
    m_zoom_distance(10)
{
    set_color(DEFAULT_COLOR);
}


CameraObject::CameraObject(ScopedHandler* handler) throw()
    : SceneObject("camera", handler),
      m_fov(45),
      m_is_orthographic(false),
      m_near(1.0f),
      m_far(100.0f),
      m_zoom_distance(10)
{
    // Unless the color has been explicitly set by a color tag, set
    // it to the default color for grids
    if( !handler->get_variables().exists<RGBColor>("color") )
        set_color(DEFAULT_COLOR);

    // Register the orthographic tag start handler
    handler->add_start_handler(
        "orthographic",
        boost::bind(&CameraObject::orthographic_start_handler,
                    this, _1, _2, _3));

    // Register the fov tag start handler
    handler->add_start_handler(
        "fov",
        boost::bind(&CameraObject::fov_start_handler,
                    this, _1, _2, _3));
}


bool CameraObject::set_fov(float fov)
{
    if(fov > 0 && fov < 180)
    {
        m_fov = fov;
        m_fov_set_signal();

        return false;
    }
    else
        return true;
}


float CameraObject::get_fov() const throw()
{
    return m_fov;
}


void CameraObject::set_orthographic(bool enable)
{
    m_is_orthographic = enable;
    m_ortho_set_signal();
}


bool CameraObject::is_orthographic() const throw()
{
    return m_is_orthographic;
}


void CameraObject::accept(ObjectVisitor* visitor) throw()
{
    visitor->visit(this);
}


ObjectType CameraObject::get_object_type() const
{
    return CAMERA_OBJECT;
}


void CameraObject::set_zoom_distance(float zoom_distance) throw()
{
    if( zoom_distance != m_zoom_distance )
    {
        m_zoom_distance = zoom_distance;
        m_zoom_distance_set_signal();

        float near = std::max(
            std::min(1.0f, 0.2f*fabsf(m_zoom_distance)),
            0.1f);

        set_near_plane(near);
        set_far_plane(100*near);
    }
}


float CameraObject::get_zoom_distance() const throw()
{
    return m_zoom_distance;
}


void CameraObject::set_near_plane(float y_distance) throw()
{
    m_near = y_distance;
    m_near_plane_set_signal();
}


float CameraObject::get_near_plane() const throw()
{
    return m_near;
}


void CameraObject::set_far_plane(float y_distance) throw()
{
    m_far = y_distance;
    m_far_plane_set_signal();
}


float CameraObject::get_far_plane() const throw()
{
    return m_far;
}


PropMap &CameraObject::get_prop_adapters()
{
    static PropMap *s_prop_adapters = 0;
    if( !s_prop_adapters )
    {
        s_prop_adapters = new PropMap;
        create_prop_adapters(*s_prop_adapters);
        merge_prop_adapters(*s_prop_adapters, SceneObject::get_prop_adapters());
    }

    return *s_prop_adapters;
}


void CameraObject::create_prop_adapters(PropMap &adapters)
{
    class OrthoAdapter : public BoolPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            cam->set_orthographic(any_cast<bool>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const CameraObject *cam =
                dynamic_cast<const CameraObject *>(obj);
            assert( cam );
            return Any(cam->is_orthographic());
        }

        virtual SignalType &signal(SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            return cam->ortho_set_signal();
        }
    };

    class FovAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            cam->set_fov(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const CameraObject *cam =
                dynamic_cast<const CameraObject *>(obj);
            assert( cam );
            return Any(cam->get_fov());
        }

        SignalType &signal(SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            return cam->fov_set_signal();
        }

        virtual float get_min_value() const { return 0.1f; }

        virtual float get_max_value() const { return 179.9f; }
    };

    class ZoomDistanceAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            cam->set_zoom_distance(any_cast<float>(val));
        }

        virtual Any get(const SceneObject *obj) const
        {
            const CameraObject *cam =
                dynamic_cast<const CameraObject *>(obj);
            assert( cam );
            return Any(cam->get_zoom_distance());
        }

        SignalType &signal(SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            return cam->zoom_distance_set_signal();
        }

        virtual float get_min_value() const { return 0.01f; }
    };

    class NearAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj) {}

        virtual Any get(const SceneObject *obj) const
        {
            const CameraObject *cam =
                dynamic_cast<const CameraObject *>(obj);
            assert( cam );
            return Any(cam->get_near_plane());
        }

        SignalType &signal(SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            return cam->near_plane_set_signal();
        }

        virtual bool is_read_only() const { return true; }
    };

    class FarAdapter : public FloatPropBase
    {
    public:
        virtual void set(const Any &val, SceneObject *obj) {}

        virtual Any get(const SceneObject *obj) const
        {
            const CameraObject *cam =
                dynamic_cast<const CameraObject *>(obj);
            assert( cam );
            return Any(cam->get_far_plane());
        }

        SignalType &signal(SceneObject *obj)
        {
            CameraObject *cam = dynamic_cast<CameraObject *>(obj);
            assert( cam );
            return cam->far_plane_set_signal();
        }

        virtual bool is_read_only() const { return true; }
    };

    adapters.insert(
        PropMap::value_type(
            CAMERA_ORTHO_PROP, new OrthoAdapter));
    adapters.insert(
        PropMap::value_type(
            CAMERA_FOV_PROP, new FovAdapter));
    adapters.insert(
        PropMap::value_type(
            CAMERA_ZOOM_DISTANCE_PROP, new ZoomDistanceAdapter));
    adapters.insert(
        PropMap::value_type(
            CAMERA_NEAR_PROP, new NearAdapter));
    adapters.insert(
        PropMap::value_type(
            CAMERA_FAR_PROP, new FarAdapter));
}


//
// --------------- XML handler methods --------------------
//

void CameraObject::start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    // Creating a scene object by passing a ScopedHandler will cause it to
    // enter a new scope with all registered tag start handlers plus any
    // specific handlers for SceneObject properties which are registered
    // by the parent constructor.
    SceneObject* tmp = new CameraObject(handler);

    // Set the new object as the current object
    ScopedMap & variables = handler->get_variables();
    variables.push_variable("current_object", tmp);
}


void CameraObject::orthographic_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    set_orthographic(true);
}


void CameraObject::fov_start_handler(
    const std::string & name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler) throw()
{
    ScopedHandler::TagScope scope;

    scope.start_functors["radians"] = boost::bind(
        &CameraObject::fov_value_start_handler, this, _1, _2, _3, false);

    scope.start_functors["degrees"] = boost::bind(
        &CameraObject::fov_value_start_handler, this, _1, _2, _3, true);

    handler->enter_scope(scope);
}


void CameraObject::fov_value_start_handler(
    const std::string &name,
    XMLHandler::AttributeMap &attributes,
    ScopedHandler *handler,
    bool value_in_degrees) throw()
{
    ScopedHandler::TagScope scope;

    scope.cdata_functor = boost::bind(
        &CameraObject::fov_value_cdata_handler, this, _1, _2, value_in_degrees);

    handler->enter_scope(scope);
}


void CameraObject::fov_value_cdata_handler(
    const std::string &cdata,
    ScopedHandler *handler,
    bool value_in_degrees) throw()
{
    float value = atof(cdata.c_str());

    if( !value_in_degrees )
    {
        // convert to degrees
        value = value/M_PI*180;
    }

    set_fov(value);
}
