/*
 * Copyright Staffan Gimåker 2009-2010.
 *
 * ---
 *
 * Distributed under the Boost Software License, Version 1.0.
 * (See accompanying file LICENSE_1_0.txt or copy at
 * http://www.boost.org/LICENSE_1_0.txt)
 */

/*
 * This is simple demo that exemplifies how to use results to do various
 * useful things - like controlling a robot.
 */

#include <iostream>
#include <cassert>
#include <cmath>
#include <boost/date_time/posix_time/posix_time.hpp>
#include <peekabot.hh>


namespace
{
    double normalize_rad(double rad)
    {
        rad = fmod(rad, 2*M_PI);

        if( rad > M_PI )
            rad = rad-2*M_PI;
        else if( rad < -M_PI )
            rad = rad+2*M_PI;

        assert( fabs(rad) <= M_PI );

        return rad;
    }
}


int main(int argc, char *argv[])
{
    peekabot::PeekabotClient client;
    client.connect("localhost");

    peekabot::GroupProxy grp;
    grp.add(client, "grp", peekabot::REPLACE_ON_CONFLICT);

    peekabot::SphereProxy marker;
    marker.add(grp, "marker");
    marker.set_scale(0.1);
    marker.set_opacity(0.3);
    marker.set_position(0,0,1);

    peekabot::ModelProxy robot;
    client.upload_file("../data/pioneer_3dx.pbmf");
    robot.add(grp, "robot", "../data/pioneer_3dx.pbmf");
    double x = 0, y = 0, theta = 0;
    double v = 0, w = 0;


    peekabot::PolylineProxy path;
    path.add(grp, "path");
    path.set_color(0,0,1);
    std::size_t path_segments = 0;
    peekabot::LineCloudProxy dir;
    dir.add(robot, "direction");
    dir.set_color(0,0,0);
    dir.set_line_width(2);

    std::cout << "Select the 'marker' object in the GUI and move it around "
              << "in the XY-plane, and watch the robot follow it around!"
              << std::endl;

    // Control parameters
    const double k_rho = 3;
    const double k_alpha = 0.59;
    const double k_v = 1;
    const double v_max = 2;

    boost::posix_time::ptime t1(
        boost::posix_time::microsec_clock::local_time());

    while( client.is_connected() )
    {
        peekabot::Result<peekabot::Vector3> marker_pos =
            marker.get_position();

        if( !marker_pos.succeeded() )
        {
            std::cerr << "Failed to get position of marker!" << std::endl;
            return -1;
        }

        double xT = marker_pos.get_result().x - x;
        double yT = marker_pos.get_result().y - y;
        double alpha = normalize_rad(atan2(yT, xT)-theta);
        double rho = sqrt(xT*xT + yT*yT);

        if( rho > 0.1 )
        {
            marker.set_color(0,1,0);
            v = k_v * v_max*cos(alpha) * tanh(rho/k_rho);
            w = k_alpha * alpha + k_v * v_max * tanh(rho/k_rho) * sin(2*alpha)/(2*rho);
        }
        else
        {
            marker.set_color(0,0,1);
            v = w = 0;
        }

        boost::posix_time::ptime t2(
            boost::posix_time::microsec_clock::local_time());
        double dt = (t2-t1).total_milliseconds() / 1000.0;
        t1 = t2;

        if( ++path_segments > 5000 )
        {
            path_segments = 1;
            path.clear_vertices();
        }

        x += v*dt*cos(theta);
        y += v*dt*sin(theta);
        theta = theta + w*dt;

        peekabot::VertexSet vs;
        vs.add_vertex(x, y, 0);
        path.add_vertices(vs);

        robot.set_pose(x,y,0,theta);
        vs.clear();
        vs.add_vertex(0,0,0);
        vs.add_vertex(2*v,0,0);
        dir.set_vertices(vs);
    }

    return 0;
}

