/*
 * Copyright Staffan Gimåker 2009.
 *
 * ---
 *
 * This file is part of peekabot.
 *
 * peekabot is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * peekabot is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cmath>
#include <cassert>
#include <iostream>
#include <boost/numeric/ublas/io.hpp>

#include "BearingOnlySLAM2D.hh"


namespace
{
    // Matrix inversion for 1x1 and 2x2 matrices :)
    ublas::matrix<double> inv(const ublas::matrix<double> &m)
    {
        if( m.size1() == m.size2() == 2 )
        {
            double det = m(0,0)*m(1,1) - m(1,0)*m(0,1);
            ublas::matrix<double> ret(2,2);
            ret(0,0) = m(1,1)/det;
            ret(1,1) = m(0,0)/det;
            ret(0,1) = -m(0,1)/det;
            ret(1,0) = -m(1,0)/det;
            return ret;
        }
        else if( m.size1() == m.size2() == 1 )
        {
            ublas::matrix<double> ret(1,1);
            ret(0,0) = 1/m(0,0);
            return ret;
        }
        else
            assert( false );
    }
}


BearingOnlySLAM2D::BearingOnlySLAM2D(
    double sigma1, double sigma2, double sigma3)
    : m_x(ublas::zero_vector<double>(3)),
      m_P(ublas::zero_matrix<double>(3,3)),
      m_sigma1(sigma1),
      m_sigma2(sigma2),
      m_sigma3(sigma3)
{
}


void BearingOnlySLAM2D::update(
    double ds, double dtheta,
    std::vector<Observation> &obs_)
{
    //
    // PREDICT STEP
    //

    // Predict covariance:
    //   P = F*P*F' + G*Q*G

    ublas::matrix<double> Q(ublas::zero_matrix<double>(3));
    Q(0,0) = fabs(ds) * pow(m_sigma1, 2);
    Q(1,1) = fabs(ds) * pow(m_sigma2, 2);
    Q(2,2) = fabs(dtheta) * pow(m_sigma3, 2);

    ublas::matrix<double> F(ublas::identity_matrix<double>(m_x.size()));
    F(ROBOT_X, ROBOT_THETA) = -ds*sin(m_x(ROBOT_THETA));
    F(ROBOT_Y, ROBOT_THETA) = ds*cos(m_x(ROBOT_THETA));

    ublas::matrix<double> G(ublas::zero_matrix<double>(m_x.size(), 3));
    G(ROBOT_X, 0) = cos(m_x(ROBOT_THETA));
    G(ROBOT_Y, 0) = sin(m_x(ROBOT_THETA));
    G(ROBOT_THETA, 1) = 1;
    G(ROBOT_THETA, 2) = 1;

    m_P = prod(ublas::matrix<double>(prod(F, m_P)), trans(F)) +
        prod(ublas::matrix<double>(prod(G, Q)), trans(G));

    // Predict mean
    //   x = f(x, u, Sigma)
    m_x(ROBOT_X) += ds*cos(m_x(ROBOT_THETA));
    m_x(ROBOT_Y) += ds*sin(m_x(ROBOT_THETA));
    m_x(ROBOT_THETA) = normalize_rad(m_x(ROBOT_THETA) + dtheta);

    //
    // UPDATE STEP
    //
    for( std::size_t i = 0; i < obs_.size(); ++i )
    {
        Observation &obs = obs_[i];

        // Observation noise
        ublas::matrix<double> R(1,1);
        R(0,0) = pow(obs.m_bearing_std, 2);

        if( !has_feature(obs.m_feat_id) )
        {
            // Extend state with new feature

            std::size_t idx = m_x.size();
            m_feats[obs.m_feat_id] = idx;

            // Augment x
            //   x_aug = j(X=(x, bearing, rho0)) = [ ... x_i y_i phi_i rho_i ]'
            //     = [ ... x_r y_r bearing+theta_r rho0 ]'
            m_x.resize(m_x.size()+4);
            m_x(idx+0) = m_x(ROBOT_X);
            m_x(idx+1) = m_x(ROBOT_Y);
            m_x(idx+2) = normalize_rad(obs.m_bearing + m_x(ROBOT_THETA));
            double d_min = 1;
            double rho_min = 1/d_min;
            double rho0 = rho_min/2;
            double rho0_std = rho_min/4;
            m_x(idx+3) = rho0; // rho_i, 1/d_i

            // Augment P
            ublas::matrix<double> J(
                ublas::zero_matrix<double>(m_P.size1()+4, m_P.size2()+2));

            for( std::size_t i = 0; i < m_P.size1(); ++i )
                J(i,i) = 1;

            J(m_P.size1()+2, m_P.size1()+0) = 1; // dj/dbearing
            J(m_P.size1()+3, m_P.size1()+1) = 1; // dj/drho0

            J(m_P.size1()+0, 0) = 1; // dj/dx_r
            J(m_P.size1()+1, 0) = 0;
            J(m_P.size1()+2, 0) = 0;
            J(m_P.size1()+3, 0) = 0;

            J(m_P.size1()+0, 1) = 0; // dj/dy_r
            J(m_P.size1()+1, 1) = 1;
            J(m_P.size1()+2, 1) = 0;
            J(m_P.size1()+3, 1) = 0;

            J(m_P.size1()+0, 2) = 0; // dj/dtheta_r
            J(m_P.size1()+1, 2) = 0;
            J(m_P.size1()+2, 2) = 1;
            J(m_P.size1()+3, 2) = 0;

            ublas::matrix<double> ext(
                ublas::zero_matrix<double>(m_P.size1() + R.size1() + 1));
            ublas::matrix_range<ublas::matrix<double> > Pr(
                ext, ublas::range (0, m_P.size1()),
                ublas::range (0, m_P.size1()));
            ublas::matrix_range<ublas::matrix<double> > Rr(
                ext, ublas::range (m_P.size1(), m_P.size1()+R.size1()),
                ublas::range (m_P.size1(), m_P.size1()+R.size2()));

            Pr = m_P;
            Rr = R;

            ext(ext.size1()-1, ext.size2()-1) = pow(rho0_std, 2);
            m_P = prod(ublas::matrix<double>(prod(J, ext)), trans(J));
        }
        else
        {
            std::size_t idx = m_feats[obs.m_feat_id];

            double phi_i = m_x(idx+2);
            double rho_i = m_x(idx+3);
            // Estimated feature position in Cartesian coordinates
            double mx = m_x(idx+0) + 1/m_x(idx+3)*cos(m_x(idx+2));
            double my = m_x(idx+1) + 1/m_x(idx+3)*sin(m_x(idx+2));
            double dx = mx - m_x(ROBOT_X);
            double dy = my - m_x(ROBOT_Y);

            double y_pred = normalize_rad(atan2(dy, dx) - m_x(ROBOT_THETA));
            double y_meas = obs.m_bearing;
            ublas::vector<double> nu(1);
            nu(0) = normalize_rad(y_meas - y_pred); // innovation

            ublas::matrix<double> H(ublas::zero_matrix<double>(1, m_x.size()));
            double tmp = dx*dx + dy*dy;
            H(0, ROBOT_X) = dy / tmp;
            H(0, ROBOT_Y) = -dx / tmp;
            H(0, ROBOT_THETA) = -1;
            H(0, idx+0) = -H(0, ROBOT_X);
            H(0, idx+1) = -H(0, ROBOT_Y);
            H(0, idx+2) = (dx*cos(phi_i) + dy*sin(phi_i)) / (tmp * rho_i);
            H(0, idx+3) = (-dx*sin(phi_i) + dy*cos(phi_i)) / (tmp * rho_i*rho_i);

            // S = H*P*H' + D*R*D', where H is the Jacobian of the
            // measurement function wrt the state, and D wrt the noise
            ublas::matrix<double> S = prod(
                ublas::matrix<double>(prod(H, m_P)), trans(H)) + R;

            // Kalman gain: K = P*H'*inv(S)
            ublas::matrix<double> K =
                prod(ublas::matrix<double>(prod(m_P, trans(H))), inv(S));

            // Normalized innovation squared:
            double maha = inner_prod(prod(nu, inv(S)), nu);

            if( maha < 3.84 ) // 95%
            {
                m_x += prod(K, nu);
                m_x(ROBOT_THETA) = normalize_rad(m_x(ROBOT_THETA));
                m_P -= prod(ublas::matrix<double>(prod(K, H)), m_P);
                obs.m_is_outlier = false;
            }
            else
            {
                obs.m_is_outlier = true;
            }
        }
    }
}


std::size_t BearingOnlySLAM2D::feature_count() const
{
    return m_feats.size();
}


bool BearingOnlySLAM2D::has_feature(FeatureID feat_id) const
{
    return m_feats.find(feat_id) != m_feats.end();
}


double BearingOnlySLAM2D::normalize_rad(double rad) const
{
    rad = fmod(rad, 2*M_PI);

    if( rad > M_PI )
        rad = rad-2*M_PI;
    else if( rad < -M_PI )
        rad = rad+2*M_PI;

    assert( fabs(rad) <= M_PI );

    return rad;
}


void BearingOnlySLAM2D::get_pose(double &x, double &y, double &theta) const
{
    x = m_x(ROBOT_X);
    y = m_x(ROBOT_Y);
    theta = m_x(ROBOT_THETA);
}


void BearingOnlySLAM2D::get_pose(
    double &x, double &y, double &theta,
    ublas::matrix<double> &cov) const
{
    x = m_x(ROBOT_X);
    y = m_x(ROBOT_Y);
    theta = m_x(ROBOT_THETA);
    cov = ublas::matrix_range<const ublas::matrix<double> >(
        m_P, ublas::range (0, 3), ublas::range (0, 3));
}


BearingOnlySLAM2D::FeatureID BearingOnlySLAM2D::get_cartesian_feature(
    std::size_t i,
    double &x, double &y,
    ublas::matrix<double> &xy_cov) const
{
    std::size_t idx = 4*i + 3;

    double phi = m_x(idx+2);
    double rho = m_x(idx+3);

    x = m_x(idx+0) + 1/rho*cos(phi);
    y = m_x(idx+1) + 1/rho*sin(phi);

    ublas::matrix<double> C(2,4);
    C(0,0) = 1;
    C(0,1) = 0;
    C(0,2) = -sin(phi) / rho;
    C(0,3) = -cos(phi) / (rho*rho);

    C(1,0) = 0;
    C(1,1) = 1;
    C(1,2) = cos(phi) / rho;
    C(1,3) = -sin(phi) / (rho*rho);

    ublas::matrix_range<const ublas::matrix<double> > Pv(
        m_P, ublas::range (idx, idx+4),
        ublas::range (idx, idx+4));

    //std::cout << Pv.size1() << "x" << Pv.size2() << std::endl;
    //assert( Pv.size1() == Pv.size2() == 4 );

    xy_cov = prod(ublas::matrix<double>(prod(C, Pv)), trans(C));
    //std::cout << xy_cov.size1() << "x" << xy_cov.size2() << std::endl;
    //assert( xy_cov.size1() == xy_cov.size2() == 2 );

    return -1;
}
