<?php
// ----------------------------------------------------------------------------------
// Class: RDF_N3_Serializer
// ----------------------------------------------------------------------------------
/**
 * PHP Notation3 Serializer
 *
 * This class serialises models to N3 Syntax.
 *
 * Supported N3 features:
 * <ul>
 *    <li>Using [ ] for blank nodes, or _: if nescessary</li>
 *    <li>Literal datytype- and xmlLanguageTag support</li>
 * </ul>
 *
 * Un-supported N3 Features include:
 * <ul>
 *    <li>Reification</li>
 * </ul>
 *
 * TODO: * added namespace prefixes are persisent...
 *
 * @author Gunnar AA. Grimnes <ggrimnes@csd.abdn.ac.uk>, Daniel Westphal <mail@d-westphal.de>
 * @version V0.7
 * @package syntax
 * @access public
 */

define('MAGIC_STRING', '~~~');

class RDF_N3_Serializer extends RDF_Object
{
    var $debug;

    var $prefixes;

    var $done; // keeps track of already serialized resources
    var $resourcetext;
    var $resourcetext_taken;
    var $model;

    var $res;

    var $anon;

    /**
     * @access public
     */
    function RDF_N3_Serializer()
    {
        $this->debug = false;
    }

    /**
     * Adds a new namespace prefix to use.
     * Unknown namespaces will become ns0, ns1 etc.
     *
     * @access public
     * @param string $s
     * @returns void
     */

    function addNSPrefix($ns, $prefix)
    {
        $this->prefixes[$ns] = $prefix;
    }

    /**
     * Serializes a model to N3 syntax.
     *
     * @param object Model $model
     * @return string
     * @access public
     */
    function &serialize(&$m)
    {
        if (!is_a($m, 'RDF_Model_Memory')) {
            $m =& $m->getMemModel();
        }

        $this->reset();
        $this->model = $m;
        $this->res = "";

        $namespaces = array();
        $count = array();
        $resources = array();
        foreach ($this->model->triples as $t) {
            $s = $t->getSubject();
            if (is_a($s, 'RDF_Resource')) {
                $namespaces[$s->getNamespace()] = 1;
            }

            $p = $t->getPredicate();
            if (is_a($p, 'RDF_Resource')) {
                $namespaces[$p->getNamespace()] = 1;
            }

            $o = $t->getObject();
            if (is_a($o, 'RDF_Resource')) {
                $namespaces[$o->getNamespace()] = 1;
            }

            $uri = $s->getURI();

            if (isset($count[$uri])) {
                $count[$uri]++;
            } else {
                $count[$uri] = 0;
                $resources[$uri] = $s;
            }
        }

        if (!RDF_HIDE_ADVERTISE) {
            $this->res .= '# Generated by N3Serializer.php from RDF RAP.' .
                RDF_LINEFEED . '# http://www.wiwiss.fu-berlin.de/suhl/bizer/rdfapi/index.html'.
                RDF_LINEFEED . RDF_LINEFEED;
        }

        $this->doNamespaces($namespaces);

        $this->res .= RDF_LINEFEED . RDF_LINEFEED;

        arsort($count);

        foreach ($count as $k => $v) {
            $this->doResource($resources[$k]);
            // $this->res.=" .\n";
        }

        $c = 0;
        foreach ($this->resourcetext as $r => $t) {
            $ms = array();
            if (preg_match_all('/'.RDF_MAGIC_STRING.'([^ ]+)'.RDF_MAGIC_STRING.'/', $t, $ms, RDF_PREG_SET_ORDER)) {
                foreach($ms as $mseach) {
                    $rp = $this->resourcetext[$mseach[1]];
                    $t = preg_replace(
                        '/' . RDF_MAGIC_STRING . $mseach[1] . RDF_MAGIC_STRING . '/',
                        $rp, $t
                    );
                }
            }

            if ($this->debug) {
                $this->res .= $c . ': ';
            }
            if (!(isset($this->resourcetext_taken[$r])
                && $this->resourcetext_taken[$r] > 0)
            ) {
                $this->res .= $t . ' .' . RDF_LINEFEED;
            } else if ($this->debug) {
                $this->res .= ' Skipping : ' . $t . RDF_LINEFEED;
            }
            $c++;
        }
        // $max=-1111;
        // $maxkey="";
        // foreach ($count as $k=>$c) {
        // if ( $c>$max) { $maxkey=$k; $max=$c; }
        // }
        // if ($this->debug) {
        // print "$maxkey is subject of most triples! ($max) \n";
        // }
        return $this->res;
    }

    /**
     * Serializes a model and saves it into a file.
     * Returns FALSE if the model couldn't be saved to the file.
     *
     * @access public
     * @param object Model_Memory $model
     * @param string $filename
     * @return boolean
     * @access public
     */
    function saveAs(&$model, $filename)
    {
        // serialize model
        $n3 = $this->serialize($model);
        // write serialized model to file
        $file_handle = @fopen($filename, 'w');
        if ($file_handle) {
            fwrite($file_handle, $n3);
            fclose($file_handle);
            return true;
        } else {
            return false;
        } ;
    }

    /* ==================== protected Methods from here ==================== */

    /**
     * Readies this object for serializing another model
     *
     * @access protected
     * @param void
     * @returns void
     */
    function reset()
    {
        $this->anon = 0;
        $this->done = array();
        $this->resourcetext_taken = array();
        $this->resourcetext = array();
        $this->res = '';
        $this->model = null;

        $this->prefixes['http://www.w3.org/1999/02/22-rdf-syntax-ns#'] = 'rdf';
        $this->prefixes['http://www.w3.org/2000/01/rdf-schema#'] = 'rdfs';
        $this->prefixes['http://www.w3.org/2002/07/owl#'] = 'owl';
        $this->prefixes['http://xmlns.com/foaf/0.1/'] = 'foaf';
        $this->prefixes['http://purl.org/dc/elements/1.1/'] = 'dc';
        $this->prefixes['http://www.w3.org/2000/10/XMLSchema#'] = 'xmls';
        $this->prefixes['http://www.w3.org/2000/10/swap/pim/contact#'] = 'contact';
    }

    /**
     * Makes ns0, ns1 etc. prefixes for unknown prefixes.
     * Outputs
     *
     * @prefix lines.
     * @access protected
     * @param array $n
     * @returns void
     */
    function doNamespaces(&$n)
    {
        $c = 0;
        foreach ($n as $ns => $v) {
            if (!$ns) {
                continue;
            }
            if (isset($this->prefixes[$ns])) {
                $p = $this->prefixes[$ns];
            } else {
                $p = 'ns' . $c;
                $this->prefixes[$ns] = $p;
                $c++;
            }
            $this->res .= "@prefix $p: <" . $ns . '> .' . RDF_LINEFEED;
        }
    }

    /**
     * Fill in $resourcetext for a single resource.
     * Will recurse into Objects of triples, but should never look ? (really?)
     *
     * @access protected
     * @param object Resource $r
     * @returns boolean
     */

    function doResource(&$r)
    {
        $ts = $this->model->find($r, null, null);
        if (PEAR::isError($ts)) {
            return $ts;
        }
        if (count($ts->triples) == 0) {
            return;
        }

        $out = '';

        if (isset($this->done[$r->getURI()]) && $this->done[$r->getURI()]) {
            if (is_a($r, 'RDF_BlankNode')) {
                if ($this->resourcetext_taken[$r->getURI()] == 1) {
                    // Oh bother, we must use the _:blah construct.
                    $a = $this->resourcetext[$r->getURI()];
                    $this->resourcetext[$r->getURI()] = '_:anon' . $this->anon;

                    $this->resourcetext['_:anon' . $this->anon] =
                        $this->fixAnon($a, '_:anon' . $this->anon);
                    $this->resourcetext_taken[$r->getURI()] = 2;
                    $this->anon++;
                }
            }
            return false;
        }
        $this->done[$r->getURI()] = true;

        if ( is_a($r, 'RDF_Resource') ) {
            if ( is_a($r, 'RDF_BlankNode') ) { 
                //test, if this blanknode is referenced somewhere
                $rbn = $this->model->find(null, null, $r);
                if (PEAR::isError($rbn)) {
                    return $rbn;
                }

                if (count($rbn->triples) > 0 | !RDF_N3SER_BNODE_SHORT) {
                    $out.='_:'.$r->getLabel();
                } else {
                    $out.='[ ';
                };
            } else { 
                $this->doURI($r, $out);
            }
        }

        usort($ts->triples, 'statementsorter');
        $lastp = '';

        $out .= ' ';

        foreach ($ts->triples as $t) {
            $p = $t->getPredicate();

            if ($p == $lastp) {
                $out .= ' , ';
            } else {
                if ($lastp != '') $out .= ' ; ';
                $this->doURI($p, $out);
                $lastp = $p;
            }

            $out .= ' ';

            $o = $t->getObject();

            if (is_a($o, 'RDF_Literal')) {
                $l = $o->getLabel();
                if (strpos($l, RDF_LINEFEED) === false) {
                    $out .= "\"$l\"";
                } else {
                    $out .= "\"\"\"$l\"\"\"";
                }
                if ($o->getLanguage() != '') {
                    $out .= '@' . $o->getLanguage();
                }
                if ($o->getDatatype() != '') {
                    $out .= '^^<' . $o->getDatatype() . '>';
                }
            }

            if (is_a($o, 'RDF_Resource')) {
                if ($this->debug) {
                    print 'Doing object: ' . $o->getURI() . RDF_LINEFEED;
                }
                if (is_a($o, 'RDF_BlankNode')) {
                    $out .='_:'.$o->getLabel();
                } else {
                    $this->doURI($o, $out);
                }
            }
        }

        if (isset($rbn) && !count($rbn->triples) > 0 && RDF_N3SER_BNODE_SHORT) {
            $out.=' ] ';
        }

        $this->resourcetext[$r->getURI()] = $out;

        return true;
    }

    /**
     * Format a single URI
     *
     * @param string $s
     * @access protected
     * @return void
     */
    function doURI(&$r, &$out)
    {
        if ($r->getURI() == 'http://www.w3.org/1999/02/22-rdf-syntax-ns#type') {
            $out .= 'a';
            return;
        }
        if ($r->getNamespace() != '') {
            $out .= $this->prefixes[$r->getNamespace()] . ':' . $r->getLocalName();
        } else {
            // Will this ever happen?
            $out .= $r->getURI();
        }
    }

    /**
     * Fix the resourcetext for a blanknode where the _: construct was used
     *
     * @param string $s
     * @param string $a
     * @access protected
     * @return void
     */
    function fixAnon($t, $a)
    {
        $t = preg_replace('/( \] $|^\[ )/', '', $t);

        return $a . $t;
    }
}

?>