<?php
/**
 * A package to make adding self updating functionality to other
 * PHP-GTK 2 packages easy.
 *
 * This is a PHP-GTK 2 driver for PEAR_PackageUpdate.
 *
 * The interface for this package must allow for the following
 * functionality:
 * - check to see if a new version is available for a given 
 *   package on a given channel
 *   - check minimum state
 * - present information regarding the upgrade (version, size)
 *   - inform user about dependencies
 * - allow user to confirm or cancel upgrade
 * - download and install the package
 * - track preferences on a per package basis
 *   - don't ask again
 *   - don't ask until next release
 *   - only ask for state XXXX or higher
 *   - bug/minor/major updates only
 * - update channel automatically
 * - force application to exit when upgrade complete
 *   - PHP-GTK/CLI apps must exit to allow classes to reload
 *   - web front end could send headers to reload certain page
 *
 * This class is simply a wrapper for PEAR classes that actually
 * do the work. 
 *
 * EXAMPLE:
 * <code>
 * <?php
 *  class Goo {
 *      function __construct()
 *      {
 *          // Check for updates...
 *          require_once 'PEAR/PackageUpdate.php';
 *          $ppu =& PEAR_PackageUpdate::factory('Gtk2', '@package@', '@channel@');
 *          if ($ppu !== false) {
 *              if ($ppu->checkUpdate()) {
 *                  // Use a dialog window to ask permission to update.
 *                  if ($ppu->presentUpdate()) {
 *                      if ($ppu->update()) {
 *                          // If the update succeeded, the application should
 *                          // be restarted.
 *                          $ppu->forceRestart();
 *                      }
 *                  }
 *              }
 *          }
 *          // ...
 *      }
 *      // ...
 *  }
 * ?>
 * </code>
 *
 * @author    Scott Mattocks
 * @package   PEAR_PackageUpdate_Gtk2
 * @version   0.3.2
 * @license   PHP License
 * @copyright Copyright 2006 Scott Mattocks
 */
require_once 'PEAR/PackageUpdate.php';
class PEAR_PackageUpdate_Gtk2 extends PEAR_PackageUpdate {

    /**
     * The GtkDialog widget.
     *
     * @access public
     * @var    object
     */
    public $widget;

    /**
     * A GtkTable holding the detail information.
     *
     * @access public
     * @var    object
     */
    public $detailsTable;

    /**
     * Creates the dialog that will ask the user if it is ok to update.
     *
     * @access protected
     * @return void
     */
    protected function createDialog()
    {    
        // Don't do anything if the dialog already exists.
        if ($this->widget instanceof GtkDialog) {
            return;
        }

        // Create a title string.
        $title = 'Update available for: ' . $this->packageName;
        
        // Make the dialog modal don't show the separator.
        $flags = Gtk::DIALOG_MODAL | Gtk::DIALOG_NO_SEPARATOR;

        // Add the buttons later.
        $buttons = array(//Gtk::STOCK_NO,  Gtk::RESPONSE_NO,
                         //Gtk::STOCK_YES, Gtk::RESPONSE_YES
                         );

        // Create the dialog
        $this->widget = new GtkDialog($title, null, $flags, $buttons);

        // Create an image for the dialog.
        $img = GtkImage::new_from_stock(Gtk::STOCK_DIALOG_INFO,
                                        Gtk::ICON_SIZE_DIALOG
                                        );

        // Create the message for the dialog.
        $msg     = 'A new version of ' . $this->packageName . ' ';
        $msg    .= " is available.\n\nWould you like to upgrade?";
        $message = new GtkLabel($msg);
        
        // Pack the image and message into a table.
        $table = new GtkTable(2, 2);
        $table->attach($img, 0, 1, 0, 2);
        $table->attach($message, 1, 2, 0, 1);

        // Attach a table holding the update details.
        $table->attach($this->getDetailsTable(), 1, 2, 1, 2);

        // Pack the imange and message into the vbox area.
        $this->widget->vbox->pack_start($table);


        // Create a button for preferences.
        $prefButton   = GtkButton::new_from_stock(Gtk::STOCK_PREFERENCES);

        // Pack the button into the start of the dialog's action area.
        $this->widget->action_area->pack_start($prefButton);

        // Connect the pref button to a preferences dialog.
        $prefButton->connect_simple('clicked', array($this, 'prefDialog'));
        
        // Add the yes/no buttons.
        $this->widget->add_action_widget(GtkButton::new_from_stock(Gtk::STOCK_NO),
                                         Gtk::RESPONSE_NO
                                         );
        $this->widget->add_action_widget(GtkButton::new_from_stock(Gtk::STOCK_YES),
                                         Gtk::RESPONSE_YES
                                         );
    }

    /**
     * Creates and runs a dialog for setting preferences.
     *
     * @access public
     * @return void
     */
    public function prefDialog()
    {
        // Create the preferences dialog title.
        $title = $this->packageName . ' Update Preferences';

        // Create the flags.
        // Make the dialog modal don't show the separator.
        $flags = Gtk::DIALOG_MODAL | Gtk::DIALOG_NO_SEPARATOR;

        // Add two buttons.
        $buttons = array(Gtk::STOCK_CANCEL, Gtk::RESPONSE_CANCEL,
                         Gtk::STOCK_OK,     Gtk::RESPONSE_OK
                         );

        // Create the dialog.
        $prefDialog = new GtkDialog($title, null, $flags, $buttons);

        // The preferences dialog needs to have some inputs for the user.
        // Get the current preferences so that defaults can be set.
        $prefs = $this->getPackagePreferences();

        // It needs a check box for "Don't ask again"
        $dontAsk = new GtkCheckButton('Don\'t ask me again');

        // Set the default.
        if (isset($prefs[PEAR_PACKAGEUPDATE_PREF_NOUPDATES])) {
            $dontAsk->set_active($prefs[PEAR_PACKAGEUPDATE_PREF_NOUPDATES]);
        }

        // It needs a check box for the next release.
        $nextRelease = new GtkCheckButton('Don\'t ask again until the next release.');

        // Set the default.
        if (isset($prefs[PEAR_PACKAGEUPDATE_PREF_NEXTRELEASE])) {
            $nextRelease->set_active($prefs[PEAR_PACKAGEUPDATE_PREF_NEXTRELEASE]);
        }

        // It needs a radio group for the state.
        $stateLabel = new GtkLabel('Only ask when the state is at least:');
        $allStates  = new GtkRadioButton(null,       'All states');
        $devel      = new GtkRadioButton($allStates, 'devel');
        $alpha      = new GtkRadioButton($allStates, 'alpha');
        $beta       = new GtkRadioButton($allStates, 'beta');
        $stable     = new GtkRadioButton($allStates, 'stable');
        
        // Set the default.
        if (!isset($prefs[PEAR_PACKAGEUPDATE_PREF_STATE])) {
            // Don't do anything. It is better to check this once here
            // than it is to check it in each of the else's below.
        } elseif ($prefs[PEAR_PACKAGEUPDATE_PREF_STATE] == PEAR_PACKAGEUPDATE_STATE_DEVEL) {
            $devel->set_active(true);
        } elseif ($prefs[PEAR_PACKAGEUPDATE_PREF_STATE] == PEAR_PACKAGEUPDATE_STATE_ALPHA) {
            $alpha->set_active(true);
        } elseif ($prefs[PEAR_PACKAGEUPDATE_PREF_STATE] == PEAR_PACKAGEUPDATE_STATE_BETA) {
            $beta->set_active(true);
        } elseif ($prefs[PEAR_PACKAGEUPDATE_PREF_STATE] == PEAR_PACKAGEUPDATE_STATE_STABLE) {
            $stable->set_active(true);
        }

        // It needs a radio group for the type.
        $typeLabel = new GtkLabel('Only ask when the type is at least:');
        $allTypes  = new GtkRadioButton(null,      'All Release Types');
        $bug       = new GtkRadioButton($allTypes, 'Bug Fix');
        $minor     = new GtkRadioButton($allTypes, 'Minor');
        $major     = new GtkRadioButton($allTypes, 'Major');

        // Set the default.
        if (!isset($prefs[PEAR_PACKAGEUPDATE_PREF_TYPE])) {
            // Don't do anything. It is better to check this once here
            // than it is to check it in each of the else's below.
        } elseif ($prefs[PEAR_PACKAGEUPDATE_PREF_TYPE] == PEAR_PACKAGEUPDATE_TYPE_BUG) {
            $bug->set_active(true);
        } elseif ($prefs[PEAR_PACKAGEUPDATE_PREF_TYPE] == PEAR_PACKAGEUPDATE_TYPE_MINOR) {
            $minor->set_active(true);
        } elseif ($prefs[PEAR_PACKAGEUPDATE_PREF_TYPE] == PEAR_PACKAGEUPDATE_TYPE_MAJOR) {
            $major->set_active(true);
        }

        // The don't ask button should make other options insensitive.
        $inactivate = array($nextRelease,
                            $allStates,
                            $devel,
                            $alpha,
                            $beta,
                            $stable,
                            $allTypes,
                            $bug,
                            $minor,
                            $major
                            );
        $dontAsk->connect('toggled',
                          array($this, 'toggleSensitiveArray'),
                          $inactivate
                          );

        // The next release button should make other options insensitive excpet
        // for the don't ask button.
        $inactivate = array($allStates,
                            $devel,
                            $alpha,
                            $beta,
                            $stable,
                            $allTypes,
                            $bug,
                            $minor,
                            $major
                            );
        $nextRelease->connect('toggled',
                              array($this, 'toggleSensitiveArray'),
                              $inactivate
                              );

        // Any response should destroy the dialog.
        $prefDialog->connect_simple_after('response',
                                          array($prefDialog, 'destroy')
                                          );
        
        // Attach everything to a table.
        $table = new GtkTable(14, 1);
        
        // Everthing should be wrapped in GtkAlignment objects.
        $expandFill = Gtk::EXPAND | Gtk::FILL;
        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($dontAsk);
        $table->attach($align, 0, 1, 1, 2, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($nextRelease);
        $table->attach($align, 0, 1, 2, 3, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($stateLabel);
        $table->attach($align, 0, 1, 3, 4, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($allStates);
        $table->attach($align, 0, 1, 4, 5, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($devel);
        $table->attach($align, 0, 1, 5, 6, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($alpha);
        $table->attach($align, 0, 1, 6, 7, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($beta);
        $table->attach($align, 0, 1, 7, 8, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($stable);
        $table->attach($align, 0, 1, 8, 9, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($typeLabel);
        $table->attach($align, 0, 1, 9, 10, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($allTypes);
        $table->attach($align, 0, 1, 10, 11, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($bug);
        $table->attach($align, 0, 1, 11, 12, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($minor);
        $table->attach($align, 0, 1, 12, 13, $expandFill, 0, 0, 0);

        $align = new GtkAlignment(0, .5, 0, 0);
        $align->add($major);
        $table->attach($align, 0, 1, 13, 14, $expandFill, 0, 0, 0);

        // Add the table to the dialog's vbox.
        $prefDialog->vbox->pack_start($table);
        
        // Run the dialog.
        $prefDialog->show_all();
        if ($prefDialog->run() == Gtk::RESPONSE_OK) {
            // Get all of the preferences.
            $prefs = array();
            
            // Check for the don't ask preference.
            $prefs[PEAR_PACKAGEUPDATE_PREF_NOUPDATES]   = $dontAsk->get_active();
            
            // Check for next version.
            $prefs[PEAR_PACKAGEUPDATE_PREF_NEXTRELEASE] = $nextRelease->get_active();

            // Check for type.
            if ($bug->get_active()) {
                $prefs[PEAR_PACKAGEUPDATE_PREF_TYPE] = PEAR_PACKAGEUPDATE_TYPE_BUG;
            } elseif ($minor->get_active()) {
                $prefs[PEAR_PACKAGEUPDATE_PREF_TYPE] = PEAR_PACKAGEUPDATE_TYPE_MINOR;
            } elseif ($major->get_active()) {
                $prefs[PEAR_PACKAGEUPDATE_PREF_TYPE] = PEAR_PACKAGEUPDATE_TYPE_MAJOR;
            } else {
                unset($prefs[PEAR_PACKAGEUPDATE_PREF_TYPE]);
            }

            // Check for state.
            if ($devel->get_active()) {
                $prefs[PEAR_PACKAGEUPDATE_PREF_STATE] = PEAR_PACKAGEUPDATE_STATE_DEVEL;
            } elseif ($alpha->get_active()) {
                $prefs[PEAR_PACKAGEUPDATE_PREF_STATE] = PEAR_PACKAGEUPDATE_STATE_ALPHA;
            } elseif ($beta->get_active()) {
                $prefs[PEAR_PACKAGEUPDATE_PREF_STATE] = PEAR_PACKAGEUPDATE_STATE_BETA;
            } elseif ($stable->get_active()) {
                $prefs[PEAR_PACKAGEUPDATE_PREF_STATE] = PEAR_PACKAGEUPDATE_STATE_STABLE;
            } else {
                unset($prefs[PEAR_PACKAGEUPDATE_PREF_STATE]);
            }

            // Save the preferences.
            $this->setPreferences($prefs);
        }
    }

    /**
     * Makes the given array sensitive or insensitive depending on the state of
     * the given button.
     *
     * @access public
     * @param  object $button   A GtkToggleButton instance.
     * @param  array  $elements The elements to make (in)sensitive.
     * @return void
     */
    public function toggleSensitiveArray(GtkToggleButton $button, $elements)
    {
        // Determine whether or not the elements should be sensitive.
        $sensitive = !$button->get_active();
        foreach ($elements as $widget) {
            $widget->set_sensitive($sensitive);
        }
    }

    /**
     * Creates and/or returns a GtkTable holding the detail information.
     *
     * @access public
     * @return object A GtkTable.
     */
    public function getDetailsTable()
    {
        if (!($this->detailsTable instanceof GtkTable)) {
            $table = new GtkTable(6, 2);
            
            // The first column should have static labels.
            $currentVer   = new GtkLabel('Current Version:');
            $releaseVer   = new GtkLabel('Release Version:');
            $releaseDate  = new GtkLabel('Release Date:');
            $releaseState = new GtkLabel('Release State:');
            $releaseNotes = new GtkLabel('Release Notes:');
            $releasedBy   = new GtkLabel('Released By:');

            // The second column had info from the info array.
            $cVer  = new GtkLabel($this->instVersion);
            $rVer  = new GtkLabel($this->latestVersion);
            $date  = new GtkLabel($this->info['releasedate']);
            $state = new GtkLabel($this->info['state']);
            $notes = new GtkLabel($this->info['releasenotes']);
            $by    = new GtkLabel($this->info['doneby']);

            // Add all of these to a GtkAlign and then attach them.
            $expandFill = Gtk::EXPAND | Gtk::FILL;
            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($currentVer);
            $table->attach($align, 0, 1, 0, 1, $expandFill, 0, 0, 0);
            
            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($releaseVer);
            $table->attach($align, 0, 1, 1, 2, $expandFill, 0, 0, 0);
            
            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($releaseDate);
            $table->attach($align, 0, 1, 2, 3, $expandFill, 0, 0, 0);
            
            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($releaseState);
            $table->attach($align, 0, 1, 3, 4, $expandFill, 0, 0, 0);

            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($releaseNotes);
            $table->attach($align, 0, 1, 4, 5, $expandFill, 0, 0, 0);

            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($releasedBy);
            $table->attach($align, 0, 1, 5, 6, $expandFill, 0, 0, 0);
            

            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($cVer);
            $table->attach($align, 1, 2, 0, 1, $expandFill, 0, 0, 0);

            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($rVer);
            $table->attach($align, 1, 2, 1, 2, $expandFill, 0, 0, 0);

            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($date);
            $table->attach($align, 1, 2, 2, 3, $expandFill, 0, 0, 0);
            
            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($state);
            $table->attach($align, 1, 2, 3, 4, $expandFill, 0, 0, 0);
            
            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($notes);
            $table->attach($align, 1, 2, 4, 5, $expandFill, 0, 0, 0);
            
            $align = new GtkAlignment(0, .5, 0, 0);
            $align->add($by);
            $table->attach($align, 1, 2, 5, 6, $expandFill, 0, 0, 0);
            
            // The release notes should be allowed to wrap.
            $notes->set_line_wrap(true);

            $this->detailsTable = $table;
        }

        return $this->detailsTable;
    }

    /**
     * Redirects or exits to force the user to restart the application.
     *
     * @access public
     * @return void
     */
    public function forceRestart()
    {
        // Just exit.
        exit;
    }

    /**
     * Presents the user with the option to update.
     *
     * @access public
     * @return boolean true if the user would like to update the package.
     */
    public function presentUpdate()
    {
        // Make sure the info has been grabbed.
        // This will just return if the info has already been grabbed.
        $this->getPackageInfo();

        // Create the dialog widget.
        $this->createDialog();

        // Run the dialog here and return whether or not the use clicked "Yes".
        $this->widget->show_all();
        $update = ($this->widget->run() == Gtk::RESPONSE_YES);

        // Hide the widget.
        $this->widget->hide();

        return $update;
    }

    /**
     * Presents an error in a dialog window.
     *
     * @access public
     * @param  array   $error An optional PEAR::ErrorStack error array.
     * @return boolean true if an error was displayed.
     */
    public function errorDialog($error = null)
    {
        // Check to see if a PEAR_Error was given.
        if (!is_array($error)) {
            if ($this->hasErrors()) {
                $error = $this->popError();
            } else {
                // Nothing to do.
                return false;
            }
        }
  
        // Create a simple message dialog.
        $flags   = 0;
        $type    = Gtk::MESSAGE_WARNING;
        $buttons = Gtk::BUTTONS_CLOSE;
        
        $dialog = new GtkMessageDialog(null, $flags, $type, $buttons,
                                       $error['message']);
        
        // Run the dialog.
        $dialog->run();

        return true;
    }
}
/*
 * Local variables:
 * tab-width: 4
 * c-basic-offset: 4
 * End:
 */
?>