<?php
// Call File_FortuneTest::main() if this source file is executed directly.
if (!defined("PHPUnit_MAIN_METHOD")) {
    set_include_path(dirname(dirname(__FILE__)) . ':' . get_include_path());
    define("PHPUnit_MAIN_METHOD", "File_FortuneTest::main");
}

require_once "PHPUnit/Framework/TestCase.php";
require_once "PHPUnit/Framework/TestSuite.php";

require_once 'File/Fortune.php';

/**
 * Test class for File_Fortune.
 */
class File_FortuneTest extends PHPUnit_Framework_TestCase 
{
    /**
     * Array of fortune files to use in testing (including header files)
     * @var array
     */
    public $fortuneFiles = array('ftest', 'ftest.dat', 'ftest2', 'ftest2.dat');

    /**
     * @var File_Fortune
     */
    public $fortunes;

    /**
     * Base files directory
     * @var string
     */
    public $base;

    /**
     * Runs the test methods of this class.
     */
    public static function main()
    {
        require_once "PHPUnit/TextUI/TestRunner.php";

        $suite  = new PHPUnit_Framework_TestSuite("File_FortuneTest");
        $result = PHPUnit_TextUI_TestRunner::run($suite);
    }

    /**
     * Sets up the fixture, for example, open a network connection.
     * This method is called before a test is executed.
     */
    public function setUp()
    {
        $this->base = $base = dirname(__FILE__) . '/_files/';
        $origBase = $base . '/orig/';
        foreach ($this->fortuneFiles as $file) {
            copy($origBase . $file, $base . $file);
        }

        $this->fortunes = new File_Fortune();
    }

    /**
     * Tears down the fixture, for example, close a network connection.
     * This method is called after a test is executed.
     */
    public function tearDown()
    {
        foreach ($this->fortuneFiles as $file) {
            unlink($this->base . $file);
        }

        if (file_exists($this->base . 'footest')) {
            unlink($this->base . 'footest');
        }
        if (file_exists($this->base . 'footest.dat')) {
            unlink($this->base . 'footest.dat');
        }
    }

    public function testIteratorAccess()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        try {
            foreach ($this->fortunes as $key => $fortune) {
                $this->assertTrue(is_int($key));
                $this->assertTrue(is_string($fortune));
            }
        } catch (Exception $e) {
            $this->fail('Iterating through fortunes should not throw exceptions');
        }
    }

    public function testCountable()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        try {
            $count = count($this->fortunes);
        } catch (Exception $e) {
            $this->fail('count() on fortunes object should not throw exceptions');
        }

        $this->assertTrue(is_int($count));
        $this->assertTrue($count > 0);
    }

    public function testGetFileInitiallyNull()
    {
        $this->assertNull($this->fortunes->getFile());
    }

    public function testSetAndGetFile()
    {
        $file = $this->base . $this->fortuneFiles[0];
        $this->fortunes->setFile($file);
        $this->assertEquals($file, $this->fortunes->getFile());
    }

    public function testSetFileSetsHeaderFileWhenProvided()
    {
        $file   = $this->base . $this->fortuneFiles[0];
        $header = $this->base . $this->fortuneFiles[1];
        $this->fortunes->setFile($file, $header);
        $this->assertEquals($header, $this->fortunes->getHeaderFile());
    }

    public function testSetFilesInitiallyNull()
    {
        $this->assertNull($this->fortunes->getFiles());
    }

    public function testSetAndGetFilesAsIndividualArguments()
    {
        $this->fortunes->setFiles(dirname(__FILE__) . '/foo', dirname(__FILE__) . '/bar');
        $files = $this->fortunes->getFiles();
        $expected = array(dirname(__FILE__) . '/foo', dirname(__FILE__) . '/bar');
        $this->assertEquals($expected, $files);
    }

    public function testSetAndGetFilesAsArray()
    {
        $expected = array(dirname(__FILE__) . '/foo', dirname(__FILE__) . '/bar');
        $this->fortunes->setFiles($expected);
        $files = $this->fortunes->getFiles();
        $this->assertEquals($expected, $files);
    }

    public function testGetFilesAfterDirectorySet()
    {
        $this->fortunes->setDirectory($this->base);
        $files = $this->fortunes->getFiles();
        $this->assertTrue(count($files) >= 1);
    }

    public function testSetFilesUnsetsFilename()
    {
        $file = $this->base . $this->fortuneFiles[0];
        $this->fortunes->setFile($file);

        $files = array(dirname(__FILE__) . '/foo', dirname(__FILE__) . '/bar');
        $this->fortunes->setFiles($files);
        $this->assertNull($this->fortunes->getFile());
    }

    public function testSetFilesUnsetsDirectory()
    {
        $this->fortunes->setDirectory($this->base);

        $files = array(dirname(__FILE__) . '/foo', dirname(__FILE__) . '/bar');
        $this->fortunes->setFiles($files);
        $this->assertNull($this->fortunes->getDirectory());
    }

    public function testGetHeaderFileInitiallyNull()
    {
        $this->assertNull($this->fortunes->getHeaderFile());
    }

    public function testSetAndGetHeaderFile()
    {
        $file = $this->base . $this->fortuneFiles[1];
        $this->fortunes->setHeaderFile($file);
        $this->assertEquals($file, $this->fortunes->getHeaderFile());
    }

    public function testGetDirectoryInitiallyNull()
    {
        $this->assertNull($this->fortunes->getDirectory());
    }

    public function testGetAndSetDirectory()
    {
        $this->fortunes->setDirectory($this->base);
        $this->assertEquals($this->base, $this->fortunes->getDirectory());
    }

    public function testSetDirectoryUnsetsFilename()
    {
        $file = $this->base . $this->fortuneFiles[0];
        $this->fortunes->setFile($file);
        $this->fortunes->setDirectory($this->base);
        $this->assertNull($this->fortunes->getFile());
    }

    public function testAddFortuneAppends()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $count = count($this->fortunes);
        $this->fortunes->add('This is a new fortune');
        $this->assertEquals($count + 1, count($this->fortunes));
        $fortune = $this->fortunes[$count];
        $this->assertTrue(is_string($fortune));
    }

    public function testUpdate()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $text  = 'Updated fortune';
        $count = count($this->fortunes);
        $this->assertNotEquals($text, $this->fortunes[2]);
        $this->fortunes->update(2, $text);
        $this->assertEquals($text, $this->fortunes[2]);
        $this->assertEquals($count, count($this->fortunes));
    }

    public function testDelete()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $count = count($this->fortunes);
        $text  = $this->fortunes[2];
        $this->assertFalse(empty($text));
        $this->fortunes->delete(2);
        $this->assertEquals($count - 1, count($this->fortunes));
        $fortune = $this->fortunes[2];
        $this->assertNotEquals($text, $fortune);
    }

    public function testOffsetExists()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $this->assertTrue($this->fortunes->offsetExists(2));
    }

    public function testOffsetExistsViaArrayAccess()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $this->assertTrue(isset($this->fortunes[2]));
    }

    public function testOffsetGet()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $fortune = $this->fortunes->offsetGet(2);
        $this->assertTrue(is_string($fortune));
        $this->assertTrue(0 < strlen($fortune));
    }

    public function testRetrieveByIndexSameWhenReadingFromFileAsFromArray()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $readFortune = $this->fortunes->offsetGet(2);
        $this->fortunes->getAll();
        $arrayFortune = $this->fortunes->offsetGet(2);
        $this->assertEquals($readFortune, $arrayFortune);
    }

    public function testOffsetGetViaArrayAccess()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $fortune = $this->fortunes[2];
        $this->assertTrue(is_string($fortune));
        $this->assertTrue(0 < strlen($fortune));
    }

    public function testOffsetSet()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $text  = 'Updated fortune';
        $count = count($this->fortunes);
        $this->assertNotEquals($text, $this->fortunes[2]);
        $this->fortunes->offsetSet(2, $text);
        $this->assertEquals($text, $this->fortunes[2]);
        $this->assertEquals($count, count($this->fortunes));
    }

    public function testOffsetSetViaArrayAccess()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $text  = 'Updated fortune';
        $count = count($this->fortunes);
        $this->assertNotEquals($text, $this->fortunes[2]);
        $this->fortunes[2] = $text;
        $this->assertEquals($text, $this->fortunes[2]);
        $this->assertEquals($count, count($this->fortunes));
    }

    public function testOffsetUnset()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $count = count($this->fortunes);
        $text  = $this->fortunes[2];
        $this->fortunes->offsetUnset(2);
        $this->assertEquals($count - 1, count($this->fortunes));
        $fortune = $this->fortunes[2];
        $this->assertNotEquals($text, $fortune);
    }

    public function testOffsetUnsetViaArrayAccess()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $count = count($this->fortunes);
        $text  = $this->fortunes[2];
        unset($this->fortunes[2]);
        $this->assertEquals($count - 1, count($this->fortunes));
        $fortune = $this->fortunes[2];
        $this->assertNotEquals($text, $fortune);
    }

    public function testGetRandom()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $fortune = $this->fortunes->getRandom();
        $this->assertTrue(is_string($fortune));
        $this->assertFalse(empty($fortune));
    }

    public function testGetRandomWithDirectorySetShouldWork()
    {
        $this->fortunes->setDirectory($this->base);
        $fortune = $this->fortunes->getRandom();
        $this->assertTrue(is_string($fortune));
        $this->assertFalse(empty($fortune));
    }

    public function testGetAll()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $fortunes = $this->fortunes->getAll();
        $this->assertTrue(is_array($fortunes));
        $this->assertFalse(empty($fortunes));
    }

    public function testGetAllWhenDirectorySetRetrievesAllFortunes()
    {
        $count = $this->_countAll();
        $this->fortunes->setDirectory($this->base);
        $discovered = $this->fortunes->getAll();
        $found      = count($discovered);
        $this->assertEquals($count, $found, "Found $found fortunes; discovered $count");
    }

    public function testSavingChangesToFile()
    {
        $this->fortunes->setFile($this->base . $this->fortuneFiles[0]);
        $expected = 'Changed fortune';
        $this->fortunes[2] = $expected;
        $this->fortunes->save();
        $this->assertEquals($expected, $this->fortunes[2]);
    }

    public function testCreateNewFortuneFile()
    {
        $fortunes = array(
            'Fortune 1',
            'Fortune 2',
            'Fortune 3',
        );
        $this->fortunes->setFile($this->base . 'footest');
        $this->fortunes->create($fortunes);
        $this->assertTrue(file_exists($this->base . 'footest'));
        $this->assertTrue(file_exists($this->base . 'footest.dat'));
        $this->assertEquals(3, count($this->fortunes));
        $this->assertEquals($fortunes, $this->fortunes->getAll());
        $this->assertEquals('%', $this->fortunes->delim);
    }

    public function testIteratorOverDirectoryThrowsException()
    {
        $this->fortunes->setDirectory($this->base);
        try {
            foreach ($this->fortunes as $file) {}
            $this->fail('Iteration over a directory should throw exception');
        } catch (File_Fortune_Exception $e) {}
    }

    public function testIteratorOverFilesThrowsException()
    {
        $this->fortunes->setFiles($this->base . $this->fortuneFiles[0], $this->base . $this->fortuneFiles[2]);
        try {
            foreach ($this->fortunes as $file) {}
            $this->fail('Iteration over multiple files should throw exception');
        } catch (File_Fortune_Exception $e) {}
    }

    public function testCountAccumulatesAllWhenDirectorySet()
    {
        $expected = $this->_countAll();
        $this->fortunes->setDirectory($this->base);
        $this->assertEquals($expected, count($this->fortunes));
    }

    public function testCountAccumulatesAllWhenMultipleFilesSet()
    {
        $expected = $this->_countAll();
        $files    = array();
        $di       = new DirectoryIterator($this->base);
        foreach ($di as $file) {
            if ($file->isDot() || $file->isDir() || ('.dat' == substr($file->getFilename(), -4))) {
                continue;
            }
            $files[] = $file->getPathName();
        }
        $this->fortunes->setFiles($files);
        $this->assertEquals($expected, count($this->fortunes));
    }

    public function testArrayAccessThrowsExceptionWithDirectorySet()
    {
        $this->fortunes->setDirectory($this->base);
        $this->_tryArrayAccess('directory');
    }

    public function testArrayAccessThrowsExceptionWithMultipleFilesSet()
    {
        $this->fortunes->setDirectory($this->base);
        $this->_tryArrayAccess('multiple files');
    }

    protected function _tryArrayAccess($type)
    {
        try {
            isset($this->fortunes[1]);
            $this->fail("Array access (exists) when $type set should throw exception");
        } catch (File_Fortune_Exception $e) {}
        try {
            $fortune = $this->fortunes[1];
            $this->fail("Array access (read) when $type set should throw exception");
        } catch (File_Fortune_Exception $e) {}
        try {
            $this->fortunes[1] = 'This should fail';
            $this->fail("Array access (update) when $type set should throw exception");
        } catch (File_Fortune_Exception $e) {}
        try {
            unset($this->fortunes[1]);
            $this->fail("Array access (delete) when $type set should throw exception");
        } catch (File_Fortune_Exception $e) {}
    }

    protected function _countAll()
    {
        $count = 0;
        $di    = new DirectoryIterator($this->base);
        foreach ($di as $file) {
            if ($file->isDot() || $file->isDir() || ('.dat' == substr($file->getFilename(), -4))) {
                continue;
            }
            $this->fortunes->setFile($file->getPathName());
            $count += count($this->fortunes);
        }
        return $count;
    }
}

// Call File_FortuneTest::main() if this source file is executed directly.
if (PHPUnit_MAIN_METHOD == "File_FortuneTest::main") {
    File_FortuneTest::main();
}
