#!/usr/bin/env pythonwrapper
"""
Contains implementation of the CreateDoxyfile script.
"""
import os
import re

import shellscript
import utils



class CreateDoxyfileScript(shellscript.ShellScript):
  """
  Shell script that creates a configuration file for Doxygen, based on a
  template.
  """

  def __init__(self,
         argv):
    """
    Constructor.

    @argument argv: Argument vector.
    """
    shellscript.ShellScript.__init__(self, argv)
    assert len(self.arguments) > 2, "filename arguments+"

  def _argumentsToListOfTuples(self):
    def updateArgument(
         match):
      return "\n%s" % (match.string[match.start():match.end()])

    # Filename patterns in Makefiles must be escaped. Here we have to un-escape
    # them again. Currently only '\*' is handled.
    arguments = (" ".join(self.arguments[1:])).replace("\\*", "*")
    regex = re.compile("\w+\s*=")
    return [tuple.split("=") for
         tuple in regex.sub(updateArgument, arguments).split("\n")[1:]]

  def _addDefaultSettings(self,
         tuples):
    excludeFound = excludePatternsFound = False
    defaultExcludes = [".svn"]
    defaultExcludePatterns = ["*/moc_*"]

    for tuple in tuples:
      if tuple[0] == "EXCLUDE":
        excludeFound = True
        tuple[1] += " %s" % (" ".join(defaultExcludes))
      elif tuple[0] == "EXCLUDE_PATTERNS":
        excludePatternsFound = True
        tuple[1] += " %s" % (" ".join(defaultExcludePatterns))

    if not excludeFound:
      tuples.append(["EXCLUDE", " ".join(defaultExcludes)])

    if not excludePatternsFound:
      tuples.append(["EXCLUDE_PATTERNS", " ".join(defaultExcludePatterns)])

    return tuples

  def _configureDoxyfile(self,
         tuples):
    def makeOptionReplacer(
         tuple):
      return lambda match: "%s = %s" % (tuple[0], tuple[1])

    configuration = open(os.path.join(os.path.split(__file__)[0],
      "Doxyfile")).read()

    for tuple in tuples:
      regex = re.compile("^%s\s*=\s*\w*\s*$" % (tuple[0]), re.MULTILINE)
      configuration = regex.sub(makeOptionReplacer(tuple), configuration)

    return configuration

  def _run(self):
    # Tuples is a list of KEYWORD, VALUE lists:
    # [[KEYWORD1, VALUE1], [KEYWORD2, VALUE2]]
    tuples = self._argumentsToListOfTuples()
    tuples = self._addDefaultSettings(tuples)
    header = "# WARNING: THIS FILE IS GENERATED BY CREATEDOXYFILE, DO NOT EDIT"
    configuration = "%s\n%s" % (header, self._configureDoxyfile(tuples))
    open(self.arguments[0], "w").write(configuration)

if __name__ == "__main__": 
  import sys
  sys.exit(CreateDoxyfileScript(sys.argv).run())
