
/*
    pbrt source code Copyright(c) 1998-2012 Matt Pharr and Greg Humphreys.

    This file is part of pbrt.

    Redistribution and use in source and binary forms, with or without
    modification, are permitted provided that the following conditions are
    met:

    - Redistributions of source code must retain the above copyright
      notice, this list of conditions and the following disclaimer.

    - Redistributions in binary form must reproduce the above copyright
      notice, this list of conditions and the following disclaimer in the
      documentation and/or other materials provided with the distribution.

    THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
    IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
    TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
    PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
    HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
    SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
    LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
    DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
    THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
    (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
    OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

 */


// core/shrots.cpp*
#include "stdafx.h"
#include "sh.h"

// Spherical Harmonics Rotations Definitions
void SHRotateXMinus(const Spectrum *c_in, Spectrum *c_out, int lmax) {
    // -x rotations are the same as +x rotations, just with a negation
    // factor thrown in for some of the terms.
    SHRotateXPlus(c_in, c_out, lmax);

    // l = 0 band is a no op...
    for (int l = 1; l <= lmax; ++l) {
        float s = (l & 0x1) ? -1.f : 1.f;
        c_out[SHIndex(l, 0)] *= s;
        for (int m = 1; m <= l; ++m) {
            s = -s;
            c_out[SHIndex(l, m)] *= s;
            c_out[SHIndex(l, -m)] *= -s;
        }
    }
}


void SHRotateXPlus(const Spectrum *c_in, Spectrum *c_out, int lmax) {
#define O(l, m)  c_in[SHIndex(l, m)]

    // first band is a no-op
    *c_out++ = c_in[0];

    if (lmax < 1) return;
    *c_out++ = (O(1,0));
    *c_out++ = (-1.*O(1,-1));
    *c_out++ = (O(1,1));

    if (lmax < 2) return;
    *c_out++ = (O(2,1));
    *c_out++ = (-1.*O(2,-1));
    *c_out++ = (-0.5*O(2,0) - 0.8660254037844386*O(2,2));
    *c_out++ = (-1.*O(2,-2));
    *c_out++ = (-0.8660254037844386*O(2,0) + 0.5*O(2,2));

    // Remainder of SH $x+$ rotation definition
    if (lmax < 3) return;
    *c_out++ = (-0.7905694150420949*O(3,0) + 0.6123724356957945*O(3,2));
    *c_out++ = (-1.*O(3,-2));
    *c_out++ = (-0.6123724356957945*O(3,0) - 0.7905694150420949*O(3,2));
    *c_out++ = (0.7905694150420949*O(3,-3) + 0.6123724356957945*O(3,-1));
    *c_out++ = (-0.25*O(3,1) - 0.9682458365518543*O(3,3));
    *c_out++ = (-0.6123724356957945*O(3,-3) + 0.7905694150420949*O(3,-1));
    *c_out++ = (-0.9682458365518543*O(3,1) + 0.25*O(3,3));
    
    if (lmax < 4) return;
    *c_out++ = (-0.9354143466934853*O(4,1) + 0.35355339059327373*O(4,3));
    *c_out++ = (-0.75*O(4,-3) + 0.6614378277661477*O(4,-1));
    *c_out++ = (-0.35355339059327373*O(4,1) - 0.9354143466934853*O(4,3));
    *c_out++ = (0.6614378277661477*O(4,-3) + 0.75*O(4,-1));
    *c_out++ = (0.375*O(4,0) + 0.5590169943749475*O(4,2) + 0.739509972887452*O(4,4));
    *c_out++ = (0.9354143466934853*O(4,-4) + 0.35355339059327373*O(4,-2));
    *c_out++ = (0.5590169943749475*O(4,0) + 0.5*O(4,2) - 0.6614378277661477*O(4,4));
    *c_out++ = (-0.35355339059327373*O(4,-4) + 0.9354143466934853*O(4,-2));
    *c_out++ = (0.739509972887452*O(4,0) - 0.6614378277661477*O(4,2) + 0.125*O(4,4));
    
    if (lmax < 5) return;
    *c_out++ = (0.701560760020114*O(5,0) - 0.6846531968814576*O(5,2) +
                0.19764235376052372*O(5,4));
    *c_out++ = (-0.5*O(5,-4) + 0.8660254037844386*O(5,-2));
    *c_out++ = (0.5229125165837972*O(5,0) + 0.30618621784789724*O(5,2) -
                0.795495128834866*O(5,4));
    *c_out++ = (0.8660254037844386*O(5,-4) + 0.5*O(5,-2));
    *c_out++ = (0.4841229182759271*O(5,0) + 0.6614378277661477*O(5,2) +
                0.57282196186948*O(5,4));
    *c_out++ = (-0.701560760020114*O(5,-5) - 0.5229125165837972*O(5,-3) -
                0.4841229182759271*O(5,-1));
    *c_out++ = (0.125*O(5,1) + 0.4050462936504913*O(5,3) + 0.9057110466368399*O(5,5));
    *c_out++ = (0.6846531968814576*O(5,-5) - 0.30618621784789724*O(5,-3) -
                0.6614378277661477*O(5,-1));
    *c_out++ = (0.4050462936504913*O(5,1) + 0.8125*O(5,3) - 0.4192627457812106*O(5,5));
    *c_out++ = (-0.19764235376052372*O(5,-5) + 0.795495128834866*O(5,-3) -
                0.57282196186948*O(5,-1));
    *c_out++ = (0.9057110466368399*O(5,1) - 0.4192627457812106*O(5,3) + 0.0625*O(5,5));
    
    if (lmax < 6) return;
    *c_out++ = (0.879452954966893*O(6,1) - 0.46351240544347894*O(6,3) +
                0.10825317547305482*O(6,5));
    *c_out++ = (-0.3125*O(6,-5) + 0.8028270361665706*O(6,-3) - 0.5077524002897476*O(6,-1));
    *c_out++ = (0.4330127018922193*O(6,1) + 0.6846531968814576*O(6,3) -
                0.5863019699779287*O(6,5));
    *c_out++ = (0.8028270361665706*O(6,-5) - 0.0625*O(6,-3) - 0.5929270612815711*O(6,-1));
    *c_out++ = (0.19764235376052372*O(6,1) + 0.5625*O(6,3) + 0.8028270361665706*O(6,5));
    *c_out++ = (-0.5077524002897476*O(6,-5) - 0.5929270612815711*O(6,-3) -
                0.625*O(6,-1));
    *c_out++ = (-0.3125*O(6,0) - 0.45285552331841994*O(6,2) - 0.49607837082461076*O(6,4) -
                0.6716932893813962*O(6,6));
    *c_out++ = (-0.879452954966893*O(6,-6) - 0.4330127018922193*O(6,-4) -
                0.19764235376052372*O(6,-2));
    *c_out++ = (-0.45285552331841994*O(6,0) - 0.53125*O(6,2) - 0.1711632992203644*O(6,4) +
                0.6952686081652184*O(6,6));
    *c_out++ = (0.46351240544347894*O(6,-6) - 0.6846531968814576*O(6,-4) -
                0.5625*O(6,-2));
    *c_out++ = (-0.49607837082461076*O(6,0) - 0.1711632992203644*O(6,2) +
                0.8125*O(6,4) - 0.2538762001448738*O(6,6));
    *c_out++ = (-0.10825317547305482*O(6,-6) + 0.5863019699779287*O(6,-4) -
                0.8028270361665706*O(6,-2));
    *c_out++ = (-0.6716932893813962*O(6,0) + 0.6952686081652184*O(6,2) -
                0.2538762001448738*O(6,4) + 0.03125*O(6,6));
    
    if (lmax < 7) return;
    *c_out++ = (-0.6472598492877494*O(7,0) + 0.6991205412874092*O(7,2) -
                0.2981060004427955*O(7,4) + 0.05846339666834283*O(7,6));
    *c_out++ = (-0.1875*O(7,-6) + 0.6373774391990981*O(7,-4) - 0.7473912964438374*O(7,-2));
    *c_out++ = (-0.47495887979908324*O(7,0) - 0.07328774624724109*O(7,2) +
                0.78125*O(7,4) - 0.3983608994994363*O(7,6));
    *c_out++ = (0.6373774391990981*O(7,-6) - 0.5*O(7,-4) - 0.5863019699779287*O(7,-2));
    *c_out++ = (-0.42961647140211*O(7,0) - 0.41984465132951254*O(7,2) +
                0.10364452469860624*O(7,4) + 0.7927281808728639*O(7,6));
    *c_out++ = (-0.7473912964438374*O(7,-6) - 0.5863019699779287*O(7,-4) -
                0.3125*O(7,-2));
    *c_out++ = (-0.41339864235384227*O(7,0) - 0.5740991584648073*O(7,2) -
                0.5385527481129402*O(7,4) - 0.4576818286211503*O(7,6));
    *c_out++ = (0.6472598492877494*O(7,-7) + 0.47495887979908324*O(7,-5) +
                0.42961647140211*O(7,-3) + 0.41339864235384227*O(7,-1));
    *c_out++ = (-0.078125*O(7,1) - 0.24356964481437335*O(7,3) - 0.4487939567607835*O(7,5) -
                0.8562442974262661*O(7,7));
    *c_out++ = (-0.6991205412874092*O(7,-7) + 0.07328774624724109*O(7,-5) +
                0.41984465132951254*O(7,-3) + 0.5740991584648073*O(7,-1));
    *c_out++ = (-0.24356964481437335*O(7,1) - 0.609375*O(7,3) - 0.5700448858423344*O(7,5) +
                0.4943528756111367*O(7,7));
    *c_out++ = (0.2981060004427955*O(7,-7) - 0.78125*O(7,-5) - 0.10364452469860624*O(7,-3) +
                0.5385527481129402*O(7,-1));
    *c_out++ = (-0.4487939567607835*O(7,1) - 0.5700448858423344*O(7,3) + 0.671875*O(7,5) -
                0.14905300022139775*O(7,7));
    *c_out++ = (-0.05846339666834283*O(7,-7) + 0.3983608994994363*O(7,-5) -
                0.7927281808728639*O(7,-3) + 0.4576818286211503*O(7,-1));
    *c_out++ = (-0.8562442974262661*O(7,1) + 0.4943528756111367*O(7,3) -
                0.14905300022139775*O(7,5) + 0.015625*O(7,7));
    
    if (lmax < 8) return;
    *c_out++ = (-0.8356088723200586*O(8,1) + 0.516334738808072*O(8,3) -
                0.184877493221863*O(8,5) + 0.03125*O(8,7));
    *c_out++ = (-0.109375*O(8,-7) + 0.4621937330546575*O(8,-5) - 0.774502108212108*O(8,-3) +
                0.4178044361600293*O(8,-1));
    *c_out++ = (-0.4576818286211503*O(8,1) - 0.47134697278119864*O(8,3) +
                0.7088310138883598*O(8,5) - 0.2567449488305466*O(8,7));
    *c_out++ = (0.4621937330546575*O(8,-7) - 0.703125*O(8,-5) - 0.2181912506838897*O(8,-3) +
                0.4943528756111367*O(8,-1));
    *c_out++ = (-0.27421763710600383*O(8,1) - 0.6051536478449089*O(8,3) -
                0.33802043207474897*O(8,5) + 0.6665852814906732*O(8,7));
    *c_out++ = (-0.774502108212108*O(8,-7) - 0.2181912506838897*O(8,-5) +
                0.265625*O(8,-3) + 0.5310201708739509*O(8,-1));
    *c_out++ = (-0.1307281291459493*O(8,1) - 0.38081430021731066*O(8,3) -
                0.5908647000371574*O(8,5) - 0.6991205412874092*O(8,7));
    *c_out++ = (0.4178044361600293*O(8,-7) + 0.4943528756111367*O(8,-5) +
                0.5310201708739509*O(8,-3) + 0.546875*O(8,-1));
    *c_out++ = (0.2734375*O(8,0) + 0.3921843874378479*O(8,2) + 0.4113264556590057*O(8,4) +
                0.4576818286211503*O(8,6) + 0.626706654240044*O(8,8));
    *c_out++ = (0.8356088723200586*O(8,-8) + 0.4576818286211503*O(8,-6) +
                0.27421763710600383*O(8,-4) + 0.1307281291459493*O(8,-2));
    *c_out++ = (0.3921843874378479*O(8,0) + 0.5*O(8,2) + 0.32775276505317236*O(8,4) -
                0.6991205412874092*O(8,8));
    *c_out++ = (-0.516334738808072*O(8,-8) + 0.47134697278119864*O(8,-6) +
                0.6051536478449089*O(8,-4) + 0.38081430021731066*O(8,-2));
    *c_out++ = (0.4113264556590057*O(8,0) + 0.32775276505317236*O(8,2) -
                0.28125*O(8,4) - 0.7302075903467452*O(8,6) + 0.3332926407453366*O(8,8));
    *c_out++ = (0.184877493221863*O(8,-8) - 0.7088310138883598*O(8,-6) +
                0.33802043207474897*O(8,-4) + 0.5908647000371574*O(8,-2));
    *c_out++ = (0.4576818286211503*O(8,0) - 0.7302075903467452*O(8,4) + 0.5*O(8,6) -
                0.0855816496101822*O(8,8));
    *c_out++ = (-0.03125*O(8,-8) + 0.2567449488305466*O(8,-6) - 0.6665852814906732*O(8,-4) +
                0.6991205412874092*O(8,-2));
    *c_out++ = (0.626706654240044*O(8,0) - 0.6991205412874092*O(8,2) +
                0.3332926407453366*O(8,4) - 0.0855816496101822*O(8,6) + 0.0078125*O(8,8));
    
    if (lmax < 9) return;
    *c_out++ = (0.6090493921755238*O(9,0) - 0.6968469725305549*O(9,2) +
                0.3615761395439417*O(9,4) - 0.11158481919598204*O(9,6) + 0.016572815184059706*O(9,8));
    *c_out++ = (-0.0625*O(9,-8) + 0.3156095293238149*O(9,-6) - 0.6817945071647321*O(9,-4) +
                0.656993626300895*O(9,-2));
    *c_out++ = (0.44314852502786806*O(9,0) - 0.05633673867912483*O(9,2) - 0.6723290616859425*O(9,4) +
                0.5683291712335379*O(9,6) - 0.1594400908746762*O(9,8));
    *c_out++ = (0.3156095293238149*O(9,-8) - 0.71875*O(9,-6) + 0.20252314682524564*O(9,-4) +
                0.5854685623498499*O(9,-2));
    *c_out++ = (0.39636409043643195*O(9,0) + 0.25194555463432966*O(9,2) - 0.3921843874378479*O(9,4) -
                0.6051536478449089*O(9,6) + 0.509312687906457*O(9,8));
    *c_out++ = (-0.6817945071647321*O(9,-8) + 0.20252314682524564*O(9,-6) + 0.5625*O(9,-4) +
                0.4215855488510013*O(9,-2));
    *c_out++ = (0.3754879637718099*O(9,0) + 0.42961647140211*O(9,2) + 0.13799626353637262*O(9,4) -
                0.2981060004427955*O(9,6) - 0.7526807559068452*O(9,8));
    *c_out++ = (0.656993626300895*O(9,-8) + 0.5854685623498499*O(9,-6) + 0.4215855488510013*O(9,-4) +
                0.21875*O(9,-2));
    *c_out++ = (0.36685490255855924*O(9,0) + 0.5130142237306876*O(9,2) + 0.4943528756111367*O(9,4) +
                0.4576818286211503*O(9,6) + 0.38519665736315783*O(9,8));
    *c_out++ = (-0.6090493921755238*O(9,-9) - 0.44314852502786806*O(9,-7) - 0.39636409043643195*O(9,-5) -
                0.3754879637718099*O(9,-3) - 0.36685490255855924*O(9,-1));
    *c_out++ = (0.0546875*O(9,1) + 0.16792332234534904*O(9,3) + 0.2954323500185787*O(9,5) +
                0.4624247721758373*O(9,7) + 0.8171255055356398*O(9,9));
    *c_out++ = (0.6968469725305549*O(9,-9) + 0.05633673867912483*O(9,-7) - 0.25194555463432966*O(9,-5) -
                0.42961647140211*O(9,-3) - 0.5130142237306876*O(9,-1));
    *c_out++ = (0.16792332234534904*O(9,1) + 0.453125*O(9,3) + 0.577279787559724*O(9,5) +
                0.387251054106054*O(9,7) - 0.5322256665703469*O(9,9));
    *c_out++ = (-0.3615761395439417*O(9,-9) + 0.6723290616859425*O(9,-7) + 0.3921843874378479*O(9,-5) -
                0.13799626353637262*O(9,-3) - 0.4943528756111367*O(9,-1));
    *c_out++ = (0.2954323500185787*O(9,1) + 0.577279787559724*O(9,3) + 0.140625*O(9,5) -
                0.7162405240429014*O(9,7) + 0.21608307321780204*O(9,9));
    *c_out++ = (0.11158481919598204*O(9,-9) - 0.5683291712335379*O(9,-7) + 0.6051536478449089*O(9,-5) +
                0.2981060004427955*O(9,-3) - 0.4576818286211503*O(9,-1));
    *c_out++ = (0.4624247721758373*O(9,1) + 0.387251054106054*O(9,3) - 0.7162405240429014*O(9,5) +
                0.34765625*O(9,7) - 0.048317644050206957*O(9,9));
    *c_out++ = (-0.016572815184059706*O(9,-9) + 0.1594400908746762*O(9,-7) - 0.509312687906457*O(9,-5) +
                0.7526807559068452*O(9,-3) - 0.38519665736315783*O(9,-1));
    *c_out++ = (0.8171255055356398*O(9,1) - 0.5322256665703469*O(9,3) + 0.21608307321780204*O(9,5) -
                0.048317644050206957*O(9,7) + 0.00390625*O(9,9));
    Assert(lmax < 10);
    
#if 0
    if (lmax < 10) return;
    *c_out++ = (0.800447720175637*O(10,1) - 0.5437971423529642*O(10,3) + 0.24319347525427157*O(10,5) -
                0.06594508990677396*O(10,7) + 0.008734640537108554*O(10,9));
    *c_out++ = (-0.03515625*O(10,-9) + 0.20644078533943455*O(10,-7) - 0.5437971423529642*O(10,-5) +
                0.7295804257628147*O(10,-3) - 0.35797110294949086*O(10,-1));
    *c_out++ = (0.4645646483537033*O(10,1) + 0.3156095293238149*O(10,3) - 0.7057243619147634*O(10,5) +
                0.4210060495408587*O(10,7) - 0.09631896879639025*O(10,9));
    *c_out++ = (0.20644078533943455*O(10,-9) - 0.62890625*O(10,-7) + 0.5041938375082048*O(10,-5) +
                0.3543293897015177*O(10,-3) - 0.4267298778757763*O(10,-1));
    *c_out++ = (0.31049159295683837*O(10,1) + 0.5390625*O(10,3) - 0.017469281074217108*O(10,5) -
                0.6925528980529664*O(10,7) + 0.36479021288140734*O(10,9));
    *c_out++ = (-0.5437971423529642*O(10,-9) + 0.5041938375082048*O(10,-7) + 0.484375*O(10,-5) -
                0.034938562148434216*O(10,-3) - 0.46285353886245695*O(10,-1));
    *c_out++ = (0.19515618744994995*O(10,1) + 0.48613591206575135*O(10,3) + 0.4941058844013093*O(10,5) +
                0.09110862335695782*O(10,7) - 0.6878550219704731*O(10,9));
    *c_out++ = (0.7295804257628147*O(10,-9) + 0.3543293897015177*O(10,-7) - 0.034938562148434216*O(10,-5) -
                0.328125*O(10,-3) - 0.4829869223773042*O(10,-1));
    *c_out++ = (0.09472152853892297*O(10,1) + 0.2788526296496054*O(10,3) + 0.44538102542935076*O(10,5) +
                0.5748694230132831*O(10,7) + 0.6200241379499873*O(10,9));
    *c_out++ = (-0.35797110294949086*O(10,-9) - 0.4267298778757763*O(10,-7) - 0.46285353886245695*O(10,-5) -
                0.4829869223773042*O(10,-3) - 0.4921875*O(10,-1));
    *c_out++ = (-0.24609375*O(10,0) - 0.3512368283228746*O(10,2) - 0.3618292555284191*O(10,4) -
                0.3837778803237379*O(10,6) - 0.4306629552848579*O(10,8) - 0.5936279171365733*O(10,10));
    *c_out++ = (-0.800447720175637*O(10,-10) - 0.4645646483537033*O(10,-8) - 0.31049159295683837*O(10,-6) -
                0.19515618744994995*O(10,-4) - 0.09472152853892297*O(10,-2));
    *c_out++ = (-0.3512368283228746*O(10,0) - 0.46484375*O(10,2) - 0.36618880141431137*O(10,4) -
                0.18922142726223223*O(10,6) + 0.10058119679362292*O(10,8) + 0.6932080600734395*O(10,10));
    *c_out++ = (0.5437971423529642*O(10,-10) - 0.3156095293238149*O(10,-8) - 0.5390625*O(10,-6) -
                0.48613591206575135*O(10,-4) - 0.2788526296496054*O(10,-2));
    *c_out++ = (-0.3618292555284191*O(10,0) - 0.36618880141431137*O(10,2) + 0.015625*O(10,4) +
                0.4585145534256519*O(10,6) + 0.6137165055779013*O(10,8) - 0.38452264694764726*O(10,10));
    *c_out++ = (-0.24319347525427157*O(10,-10) + 0.7057243619147634*O(10,-8) + 0.017469281074217108*O(10,-6) -
                0.4941058844013093*O(10,-4) - 0.44538102542935076*O(10,-2));
    *c_out++ = (-0.3837778803237379*O(10,0) - 0.18922142726223223*O(10,2) + 0.4585145534256519*O(10,4) +
                0.462890625*O(10,6) - 0.6114934630648914*O(10,8) + 0.13594928558824104*O(10,10));
    *c_out++ = (0.06594508990677396*O(10,-10) - 0.4210060495408587*O(10,-8) + 0.6925528980529664*O(10,-6) -
                0.09110862335695782*O(10,-4) - 0.5748694230132831*O(10,-2));
    *c_out++ = (-0.4306629552848579*O(10,0) + 0.10058119679362292*O(10,2) + 0.6137165055779013*O(10,4) -
                0.6114934630648914*O(10,6) + 0.23046875*O(10,8) - 0.026921970218926214*O(10,10));
    *c_out++ = (-0.008734640537108554*O(10,-10) + 0.09631896879639025*O(10,-8) - 0.36479021288140734*O(10,-6) +
                0.6878550219704731*O(10,-4) - 0.6200241379499873*O(10,-2));
    *c_out++ = (-0.5936279171365733*O(10,0) + 0.6932080600734395*O(10,2) - 0.38452264694764726*O(10,4) +
                0.13594928558824104*O(10,6) - 0.026921970218926214*O(10,8) + 0.001953125*O(10,10));
#endif
#undef O
}


#if 0
void SHSwapYZ(const Spectrum *c_in, Spectrum *c_out, int lmax) {
    for (int i = 0; i < SHTerms(lmax); ++i)
        c_out[i] = 0.f;
    c_out[SHIndex(0, 0)] += c_in[SHIndex(0, 0)] * 1.;
    if (lmax == 0) return;

    c_out[SHIndex(1, -1)] += c_in[SHIndex(1, 0)] * -1.;
    c_out[SHIndex(1, 0)] += c_in[SHIndex(1, -1)] * -1.;
    c_out[SHIndex(1, 1)] += c_in[SHIndex(1, 1)] * 1.;
    if (lmax == 1) return;
    // Remainder of _SHSwapYZ()_ implementation
    c_out[SHIndex(2, -2)] += c_in[SHIndex(2, 1)] * -1.;
    c_out[SHIndex(2, -1)] += c_in[SHIndex(2, -1)] * 1.;
    c_out[SHIndex(2, 0)] += c_in[SHIndex(2, 0)] * -0.5;
    c_out[SHIndex(2, 0)] += c_in[SHIndex(2, 2)] * -0.86602540378444;
    c_out[SHIndex(2, 1)] += c_in[SHIndex(2, -2)] * -1.;
    c_out[SHIndex(2, 2)] += c_in[SHIndex(2, 0)] * -0.86602540378444;
    c_out[SHIndex(2, 2)] += c_in[SHIndex(2, 2)] * 0.5;
    if (lmax == 2) return;
    
    c_out[SHIndex(3, -3)] += c_in[SHIndex(3, 0)] * 0.79056941504209;
    c_out[SHIndex(3, -3)] += c_in[SHIndex(3, 2)] * -0.61237243569579;
    c_out[SHIndex(3, -2)] += c_in[SHIndex(3, -2)] * 1.;
    c_out[SHIndex(3, -1)] += c_in[SHIndex(3, 0)] * 0.61237243569579;
    c_out[SHIndex(3, -1)] += c_in[SHIndex(3, 2)] * 0.79056941504209;
    c_out[SHIndex(3, 0)] += c_in[SHIndex(3, -3)] * 0.79056941504209;
    c_out[SHIndex(3, 0)] += c_in[SHIndex(3, -1)] * 0.61237243569579;
    c_out[SHIndex(3, 1)] += c_in[SHIndex(3, 1)] * -0.25;
    c_out[SHIndex(3, 1)] += c_in[SHIndex(3, 3)] * -0.96824583655185;
    c_out[SHIndex(3, 2)] += c_in[SHIndex(3, -3)] * -0.61237243569579;
    c_out[SHIndex(3, 2)] += c_in[SHIndex(3, -1)] * 0.79056941504209;
    c_out[SHIndex(3, 3)] += c_in[SHIndex(3, 1)] * -0.96824583655185;
    c_out[SHIndex(3, 3)] += c_in[SHIndex(3, 3)] * 0.25;
    if (lmax == 3) return;
    
    c_out[SHIndex(4, -4)] += c_in[SHIndex(4, 1)] * 0.93541434669349;
    c_out[SHIndex(4, -4)] += c_in[SHIndex(4, 3)] * -0.35355339059327;
    c_out[SHIndex(4, -3)] += c_in[SHIndex(4, -3)] * 0.75;
    c_out[SHIndex(4, -3)] += c_in[SHIndex(4, -1)] * -0.66143782776615;
    c_out[SHIndex(4, -2)] += c_in[SHIndex(4, 1)] * 0.35355339059327;
    c_out[SHIndex(4, -2)] += c_in[SHIndex(4, 3)] * 0.93541434669349;
    c_out[SHIndex(4, -1)] += c_in[SHIndex(4, -3)] * -0.66143782776615;
    c_out[SHIndex(4, -1)] += c_in[SHIndex(4, -1)] * -0.75;
    c_out[SHIndex(4, 0)] += c_in[SHIndex(4, 0)] * 0.375;
    c_out[SHIndex(4, 0)] += c_in[SHIndex(4, 2)] * 0.55901699437495;
    c_out[SHIndex(4, 0)] += c_in[SHIndex(4, 4)] * 0.73950997288745;
    c_out[SHIndex(4, 1)] += c_in[SHIndex(4, -4)] * 0.93541434669349;
    c_out[SHIndex(4, 1)] += c_in[SHIndex(4, -2)] * 0.35355339059327;
    c_out[SHIndex(4, 2)] += c_in[SHIndex(4, 0)] * 0.55901699437495;
    c_out[SHIndex(4, 2)] += c_in[SHIndex(4, 2)] * 0.5;
    c_out[SHIndex(4, 2)] += c_in[SHIndex(4, 4)] * -0.66143782776615;
    c_out[SHIndex(4, 3)] += c_in[SHIndex(4, -4)] * -0.35355339059327;
    c_out[SHIndex(4, 3)] += c_in[SHIndex(4, -2)] * 0.93541434669349;
    c_out[SHIndex(4, 4)] += c_in[SHIndex(4, 0)] * 0.73950997288745;
    c_out[SHIndex(4, 4)] += c_in[SHIndex(4, 2)] * -0.66143782776615;
    c_out[SHIndex(4, 4)] += c_in[SHIndex(4, 4)] * 0.125;
    if (lmax == 4) return;
    
    
    c_out[SHIndex(5, -5)] += c_in[SHIndex(5, 0)] * -0.70156076002011;
    c_out[SHIndex(5, -5)] += c_in[SHIndex(5, 2)] * 0.68465319688146;
    c_out[SHIndex(5, -5)] += c_in[SHIndex(5, 4)] * -0.19764235376052;
    c_out[SHIndex(5, -4)] += c_in[SHIndex(5, -4)] * 0.5;
    c_out[SHIndex(5, -4)] += c_in[SHIndex(5, -2)] * -0.86602540378444;
    c_out[SHIndex(5, -3)] += c_in[SHIndex(5, 0)] * -0.5229125165838;
    c_out[SHIndex(5, -3)] += c_in[SHIndex(5, 2)] * -0.3061862178479;
    c_out[SHIndex(5, -3)] += c_in[SHIndex(5, 4)] * 0.79549512883487;
    c_out[SHIndex(5, -2)] += c_in[SHIndex(5, -4)] * -0.86602540378444;
    c_out[SHIndex(5, -2)] += c_in[SHIndex(5, -2)] * -0.5;
    c_out[SHIndex(5, -1)] += c_in[SHIndex(5, 0)] * -0.48412291827593;
    c_out[SHIndex(5, -1)] += c_in[SHIndex(5, 2)] * -0.66143782776615;
    c_out[SHIndex(5, -1)] += c_in[SHIndex(5, 4)] * -0.57282196186948;
    c_out[SHIndex(5, 0)] += c_in[SHIndex(5, -5)] * -0.70156076002011;
    c_out[SHIndex(5, 0)] += c_in[SHIndex(5, -3)] * -0.5229125165838;
    c_out[SHIndex(5, 0)] += c_in[SHIndex(5, -1)] * -0.48412291827593;
    c_out[SHIndex(5, 1)] += c_in[SHIndex(5, 1)] * 0.125;
    c_out[SHIndex(5, 1)] += c_in[SHIndex(5, 3)] * 0.40504629365049;
    c_out[SHIndex(5, 1)] += c_in[SHIndex(5, 5)] * 0.90571104663684;
    c_out[SHIndex(5, 2)] += c_in[SHIndex(5, -5)] * 0.68465319688146;
    c_out[SHIndex(5, 2)] += c_in[SHIndex(5, -3)] * -0.3061862178479;
    c_out[SHIndex(5, 2)] += c_in[SHIndex(5, -1)] * -0.66143782776615;
    c_out[SHIndex(5, 3)] += c_in[SHIndex(5, 1)] * 0.40504629365049;
    c_out[SHIndex(5, 3)] += c_in[SHIndex(5, 3)] * 0.8125;
    c_out[SHIndex(5, 3)] += c_in[SHIndex(5, 5)] * -0.41926274578121;
    c_out[SHIndex(5, 4)] += c_in[SHIndex(5, -5)] * -0.19764235376052;
    c_out[SHIndex(5, 4)] += c_in[SHIndex(5, -3)] * 0.79549512883487;
    c_out[SHIndex(5, 4)] += c_in[SHIndex(5, -1)] * -0.57282196186948;
    c_out[SHIndex(5, 5)] += c_in[SHIndex(5, 1)] * 0.90571104663684;
    c_out[SHIndex(5, 5)] += c_in[SHIndex(5, 3)] * -0.41926274578121;
    c_out[SHIndex(5, 5)] += c_in[SHIndex(5, 5)] * 0.0625;
    if (lmax == 5) return;
    
    
    c_out[SHIndex(6, -6)] += c_in[SHIndex(6, 1)] * -0.87945295496689;
    c_out[SHIndex(6, -6)] += c_in[SHIndex(6, 3)] * 0.46351240544348;
    c_out[SHIndex(6, -6)] += c_in[SHIndex(6, 5)] * -0.10825317547305;
    c_out[SHIndex(6, -5)] += c_in[SHIndex(6, -5)] * 0.3125;
    c_out[SHIndex(6, -5)] += c_in[SHIndex(6, -3)] * -0.80282703616657;
    c_out[SHIndex(6, -5)] += c_in[SHIndex(6, -1)] * 0.50775240028975;
    c_out[SHIndex(6, -4)] += c_in[SHIndex(6, 1)] * -0.43301270189222;
    c_out[SHIndex(6, -4)] += c_in[SHIndex(6, 3)] * -0.68465319688146;
    c_out[SHIndex(6, -4)] += c_in[SHIndex(6, 5)] * 0.58630196997793;
    c_out[SHIndex(6, -3)] += c_in[SHIndex(6, -5)] * -0.80282703616657;
    c_out[SHIndex(6, -3)] += c_in[SHIndex(6, -3)] * 0.0625;
    c_out[SHIndex(6, -3)] += c_in[SHIndex(6, -1)] * 0.59292706128157;
    c_out[SHIndex(6, -2)] += c_in[SHIndex(6, 1)] * -0.19764235376052;
    c_out[SHIndex(6, -2)] += c_in[SHIndex(6, 3)] * -0.5625;
    c_out[SHIndex(6, -2)] += c_in[SHIndex(6, 5)] * -0.80282703616657;
    c_out[SHIndex(6, -1)] += c_in[SHIndex(6, -5)] * 0.50775240028975;
    c_out[SHIndex(6, -1)] += c_in[SHIndex(6, -3)] * 0.59292706128157;
    c_out[SHIndex(6, -1)] += c_in[SHIndex(6, -1)] * 0.625;
    c_out[SHIndex(6, 0)] += c_in[SHIndex(6, 0)] * -0.3125;
    c_out[SHIndex(6, 0)] += c_in[SHIndex(6, 2)] * -0.45285552331842;
    c_out[SHIndex(6, 0)] += c_in[SHIndex(6, 4)] * -0.49607837082461;
    c_out[SHIndex(6, 0)] += c_in[SHIndex(6, 6)] * -0.6716932893814;
    c_out[SHIndex(6, 1)] += c_in[SHIndex(6, -6)] * -0.87945295496689;
    c_out[SHIndex(6, 1)] += c_in[SHIndex(6, -4)] * -0.43301270189222;
    c_out[SHIndex(6, 1)] += c_in[SHIndex(6, -2)] * -0.19764235376052;
    c_out[SHIndex(6, 2)] += c_in[SHIndex(6, 0)] * -0.45285552331842;
    c_out[SHIndex(6, 2)] += c_in[SHIndex(6, 2)] * -0.53125;
    c_out[SHIndex(6, 2)] += c_in[SHIndex(6, 4)] * -0.17116329922036;
    c_out[SHIndex(6, 2)] += c_in[SHIndex(6, 6)] * 0.69526860816522;
    c_out[SHIndex(6, 3)] += c_in[SHIndex(6, -6)] * 0.46351240544348;
    c_out[SHIndex(6, 3)] += c_in[SHIndex(6, -4)] * -0.68465319688146;
    c_out[SHIndex(6, 3)] += c_in[SHIndex(6, -2)] * -0.5625;
    c_out[SHIndex(6, 4)] += c_in[SHIndex(6, 0)] * -0.49607837082461;
    c_out[SHIndex(6, 4)] += c_in[SHIndex(6, 2)] * -0.17116329922036;
    c_out[SHIndex(6, 4)] += c_in[SHIndex(6, 4)] * 0.8125;
    c_out[SHIndex(6, 4)] += c_in[SHIndex(6, 6)] * -0.25387620014487;
    c_out[SHIndex(6, 5)] += c_in[SHIndex(6, -6)] * -0.10825317547305;
    c_out[SHIndex(6, 5)] += c_in[SHIndex(6, -4)] * 0.58630196997793;
    c_out[SHIndex(6, 5)] += c_in[SHIndex(6, -2)] * -0.80282703616657;
    c_out[SHIndex(6, 6)] += c_in[SHIndex(6, 0)] * -0.6716932893814;
    c_out[SHIndex(6, 6)] += c_in[SHIndex(6, 2)] * 0.69526860816522;
    c_out[SHIndex(6, 6)] += c_in[SHIndex(6, 4)] * -0.25387620014487;
    c_out[SHIndex(6, 6)] += c_in[SHIndex(6, 6)] * 0.03125;
    if (lmax == 6) return;
    
    
    c_out[SHIndex(7, -7)] += c_in[SHIndex(7, 0)] * 0.64725984928775;
    c_out[SHIndex(7, -7)] += c_in[SHIndex(7, 2)] * -0.69912054128741;
    c_out[SHIndex(7, -7)] += c_in[SHIndex(7, 4)] * 0.2981060004428;
    c_out[SHIndex(7, -7)] += c_in[SHIndex(7, 6)] * -0.058463396668343;
    c_out[SHIndex(7, -6)] += c_in[SHIndex(7, -6)] * 0.1875;
    c_out[SHIndex(7, -6)] += c_in[SHIndex(7, -4)] * -0.6373774391991;
    c_out[SHIndex(7, -6)] += c_in[SHIndex(7, -2)] * 0.74739129644384;
    c_out[SHIndex(7, -5)] += c_in[SHIndex(7, 0)] * 0.47495887979908;
    c_out[SHIndex(7, -5)] += c_in[SHIndex(7, 2)] * 0.073287746247241;
    c_out[SHIndex(7, -5)] += c_in[SHIndex(7, 4)] * -0.78125;
    c_out[SHIndex(7, -5)] += c_in[SHIndex(7, 6)] * 0.39836089949944;
    c_out[SHIndex(7, -4)] += c_in[SHIndex(7, -6)] * -0.6373774391991;
    c_out[SHIndex(7, -4)] += c_in[SHIndex(7, -4)] * 0.5;
    c_out[SHIndex(7, -4)] += c_in[SHIndex(7, -2)] * 0.58630196997793;
    c_out[SHIndex(7, -3)] += c_in[SHIndex(7, 0)] * 0.42961647140211;
    c_out[SHIndex(7, -3)] += c_in[SHIndex(7, 2)] * 0.41984465132951;
    c_out[SHIndex(7, -3)] += c_in[SHIndex(7, 4)] * -0.10364452469861;
    c_out[SHIndex(7, -3)] += c_in[SHIndex(7, 6)] * -0.79272818087286;
    c_out[SHIndex(7, -2)] += c_in[SHIndex(7, -6)] * 0.74739129644384;
    c_out[SHIndex(7, -2)] += c_in[SHIndex(7, -4)] * 0.58630196997793;
    c_out[SHIndex(7, -2)] += c_in[SHIndex(7, -2)] * 0.3125;
    c_out[SHIndex(7, -1)] += c_in[SHIndex(7, 0)] * 0.41339864235384;
    c_out[SHIndex(7, -1)] += c_in[SHIndex(7, 2)] * 0.57409915846481;
    c_out[SHIndex(7, -1)] += c_in[SHIndex(7, 4)] * 0.53855274811294;
    c_out[SHIndex(7, -1)] += c_in[SHIndex(7, 6)] * 0.45768182862115;
    c_out[SHIndex(7, 0)] += c_in[SHIndex(7, -7)] * 0.64725984928775;
    c_out[SHIndex(7, 0)] += c_in[SHIndex(7, -5)] * 0.47495887979908;
    c_out[SHIndex(7, 0)] += c_in[SHIndex(7, -3)] * 0.42961647140211;
    c_out[SHIndex(7, 0)] += c_in[SHIndex(7, -1)] * 0.41339864235384;
    c_out[SHIndex(7, 1)] += c_in[SHIndex(7, 1)] * -0.078125;
    c_out[SHIndex(7, 1)] += c_in[SHIndex(7, 3)] * -0.24356964481437;
    c_out[SHIndex(7, 1)] += c_in[SHIndex(7, 5)] * -0.44879395676078;
    c_out[SHIndex(7, 1)] += c_in[SHIndex(7, 7)] * -0.85624429742627;
    c_out[SHIndex(7, 2)] += c_in[SHIndex(7, -7)] * -0.69912054128741;
    c_out[SHIndex(7, 2)] += c_in[SHIndex(7, -5)] * 0.073287746247241;
    c_out[SHIndex(7, 2)] += c_in[SHIndex(7, -3)] * 0.41984465132951;
    c_out[SHIndex(7, 2)] += c_in[SHIndex(7, -1)] * 0.57409915846481;
    c_out[SHIndex(7, 3)] += c_in[SHIndex(7, 1)] * -0.24356964481437;
    c_out[SHIndex(7, 3)] += c_in[SHIndex(7, 3)] * -0.609375;
    c_out[SHIndex(7, 3)] += c_in[SHIndex(7, 5)] * -0.57004488584233;
    c_out[SHIndex(7, 3)] += c_in[SHIndex(7, 7)] * 0.49435287561114;
    c_out[SHIndex(7, 4)] += c_in[SHIndex(7, -7)] * 0.2981060004428;
    c_out[SHIndex(7, 4)] += c_in[SHIndex(7, -5)] * -0.78125;
    c_out[SHIndex(7, 4)] += c_in[SHIndex(7, -3)] * -0.10364452469861;
    c_out[SHIndex(7, 4)] += c_in[SHIndex(7, -1)] * 0.53855274811294;
    c_out[SHIndex(7, 5)] += c_in[SHIndex(7, 1)] * -0.44879395676078;
    c_out[SHIndex(7, 5)] += c_in[SHIndex(7, 3)] * -0.57004488584233;
    c_out[SHIndex(7, 5)] += c_in[SHIndex(7, 5)] * 0.671875;
    c_out[SHIndex(7, 5)] += c_in[SHIndex(7, 7)] * -0.1490530002214;
    c_out[SHIndex(7, 6)] += c_in[SHIndex(7, -7)] * -0.058463396668343;
    c_out[SHIndex(7, 6)] += c_in[SHIndex(7, -5)] * 0.39836089949944;
    c_out[SHIndex(7, 6)] += c_in[SHIndex(7, -3)] * -0.79272818087286;
    c_out[SHIndex(7, 6)] += c_in[SHIndex(7, -1)] * 0.45768182862115;
    c_out[SHIndex(7, 7)] += c_in[SHIndex(7, 1)] * -0.85624429742627;
    c_out[SHIndex(7, 7)] += c_in[SHIndex(7, 3)] * 0.49435287561114;
    c_out[SHIndex(7, 7)] += c_in[SHIndex(7, 5)] * -0.1490530002214;
    c_out[SHIndex(7, 7)] += c_in[SHIndex(7, 7)] * 0.015625;
    if (lmax == 7) return;
    
    
    c_out[SHIndex(8, -8)] += c_in[SHIndex(8, 1)] * 0.83560887232006;
    c_out[SHIndex(8, -8)] += c_in[SHIndex(8, 3)] * -0.51633473880807;
    c_out[SHIndex(8, -8)] += c_in[SHIndex(8, 5)] * 0.18487749322186;
    c_out[SHIndex(8, -8)] += c_in[SHIndex(8, 7)] * -0.03125;
    c_out[SHIndex(8, -7)] += c_in[SHIndex(8, -7)] * 0.109375;
    c_out[SHIndex(8, -7)] += c_in[SHIndex(8, -5)] * -0.46219373305466;
    c_out[SHIndex(8, -7)] += c_in[SHIndex(8, -3)] * 0.77450210821211;
    c_out[SHIndex(8, -7)] += c_in[SHIndex(8, -1)] * -0.41780443616003;
    c_out[SHIndex(8, -6)] += c_in[SHIndex(8, 1)] * 0.45768182862115;
    c_out[SHIndex(8, -6)] += c_in[SHIndex(8, 3)] * 0.4713469727812;
    c_out[SHIndex(8, -6)] += c_in[SHIndex(8, 5)] * -0.70883101388836;
    c_out[SHIndex(8, -6)] += c_in[SHIndex(8, 7)] * 0.25674494883055;
    c_out[SHIndex(8, -5)] += c_in[SHIndex(8, -7)] * -0.46219373305466;
    c_out[SHIndex(8, -5)] += c_in[SHIndex(8, -5)] * 0.703125;
    c_out[SHIndex(8, -5)] += c_in[SHIndex(8, -3)] * 0.21819125068389;
    c_out[SHIndex(8, -5)] += c_in[SHIndex(8, -1)] * -0.49435287561114;
    c_out[SHIndex(8, -4)] += c_in[SHIndex(8, 1)] * 0.274217637106;
    c_out[SHIndex(8, -4)] += c_in[SHIndex(8, 3)] * 0.60515364784491;
    c_out[SHIndex(8, -4)] += c_in[SHIndex(8, 5)] * 0.33802043207475;
    c_out[SHIndex(8, -4)] += c_in[SHIndex(8, 7)] * -0.66658528149067;
    c_out[SHIndex(8, -3)] += c_in[SHIndex(8, -7)] * 0.77450210821211;
    c_out[SHIndex(8, -3)] += c_in[SHIndex(8, -5)] * 0.21819125068389;
    c_out[SHIndex(8, -3)] += c_in[SHIndex(8, -3)] * -0.265625;
    c_out[SHIndex(8, -3)] += c_in[SHIndex(8, -1)] * -0.53102017087395;
    c_out[SHIndex(8, -2)] += c_in[SHIndex(8, 1)] * 0.13072812914595;
    c_out[SHIndex(8, -2)] += c_in[SHIndex(8, 3)] * 0.38081430021731;
    c_out[SHIndex(8, -2)] += c_in[SHIndex(8, 5)] * 0.59086470003716;
    c_out[SHIndex(8, -2)] += c_in[SHIndex(8, 7)] * 0.69912054128741;
    c_out[SHIndex(8, -1)] += c_in[SHIndex(8, -7)] * -0.41780443616003;
    c_out[SHIndex(8, -1)] += c_in[SHIndex(8, -5)] * -0.49435287561114;
    c_out[SHIndex(8, -1)] += c_in[SHIndex(8, -3)] * -0.53102017087395;
    c_out[SHIndex(8, -1)] += c_in[SHIndex(8, -1)] * -0.546875;
    c_out[SHIndex(8, 0)] += c_in[SHIndex(8, 0)] * 0.2734375;
    c_out[SHIndex(8, 0)] += c_in[SHIndex(8, 2)] * 0.39218438743785;
    c_out[SHIndex(8, 0)] += c_in[SHIndex(8, 4)] * 0.41132645565901;
    c_out[SHIndex(8, 0)] += c_in[SHIndex(8, 6)] * 0.45768182862115;
    c_out[SHIndex(8, 0)] += c_in[SHIndex(8, 8)] * 0.62670665424004;
    c_out[SHIndex(8, 1)] += c_in[SHIndex(8, -8)] * 0.83560887232006;
    c_out[SHIndex(8, 1)] += c_in[SHIndex(8, -6)] * 0.45768182862115;
    c_out[SHIndex(8, 1)] += c_in[SHIndex(8, -4)] * 0.274217637106;
    c_out[SHIndex(8, 1)] += c_in[SHIndex(8, -2)] * 0.13072812914595;
    c_out[SHIndex(8, 2)] += c_in[SHIndex(8, 0)] * 0.39218438743785;
    c_out[SHIndex(8, 2)] += c_in[SHIndex(8, 2)] * 0.5;
    c_out[SHIndex(8, 2)] += c_in[SHIndex(8, 4)] * 0.32775276505317;
    c_out[SHIndex(8, 2)] += c_in[SHIndex(8, 8)] * -0.69912054128741;
    c_out[SHIndex(8, 3)] += c_in[SHIndex(8, -8)] * -0.51633473880807;
    c_out[SHIndex(8, 3)] += c_in[SHIndex(8, -6)] * 0.4713469727812;
    c_out[SHIndex(8, 3)] += c_in[SHIndex(8, -4)] * 0.60515364784491;
    c_out[SHIndex(8, 3)] += c_in[SHIndex(8, -2)] * 0.38081430021731;
    c_out[SHIndex(8, 4)] += c_in[SHIndex(8, 0)] * 0.41132645565901;
    c_out[SHIndex(8, 4)] += c_in[SHIndex(8, 2)] * 0.32775276505317;
    c_out[SHIndex(8, 4)] += c_in[SHIndex(8, 4)] * -0.28125;
    c_out[SHIndex(8, 4)] += c_in[SHIndex(8, 6)] * -0.73020759034675;
    c_out[SHIndex(8, 4)] += c_in[SHIndex(8, 8)] * 0.33329264074534;
    c_out[SHIndex(8, 5)] += c_in[SHIndex(8, -8)] * 0.18487749322186;
    c_out[SHIndex(8, 5)] += c_in[SHIndex(8, -6)] * -0.70883101388836;
    c_out[SHIndex(8, 5)] += c_in[SHIndex(8, -4)] * 0.33802043207475;
    c_out[SHIndex(8, 5)] += c_in[SHIndex(8, -2)] * 0.59086470003716;
    c_out[SHIndex(8, 6)] += c_in[SHIndex(8, 0)] * 0.45768182862115;
    c_out[SHIndex(8, 6)] += c_in[SHIndex(8, 4)] * -0.73020759034675;
    c_out[SHIndex(8, 6)] += c_in[SHIndex(8, 6)] * 0.5;
    c_out[SHIndex(8, 6)] += c_in[SHIndex(8, 8)] * -0.085581649610182;
    c_out[SHIndex(8, 7)] += c_in[SHIndex(8, -8)] * -0.03125;
    c_out[SHIndex(8, 7)] += c_in[SHIndex(8, -6)] * 0.25674494883055;
    c_out[SHIndex(8, 7)] += c_in[SHIndex(8, -4)] * -0.66658528149067;
    c_out[SHIndex(8, 7)] += c_in[SHIndex(8, -2)] * 0.69912054128741;
    c_out[SHIndex(8, 8)] += c_in[SHIndex(8, 0)] * 0.62670665424004;
    c_out[SHIndex(8, 8)] += c_in[SHIndex(8, 2)] * -0.69912054128741;
    c_out[SHIndex(8, 8)] += c_in[SHIndex(8, 4)] * 0.33329264074534;
    c_out[SHIndex(8, 8)] += c_in[SHIndex(8, 6)] * -0.085581649610182;
    c_out[SHIndex(8, 8)] += c_in[SHIndex(8, 8)] * 0.0078125;
    if (lmax == 8) return;
    
    
    c_out[SHIndex(9, -9)] += c_in[SHIndex(9, 0)] * -0.60904939217552;
    c_out[SHIndex(9, -9)] += c_in[SHIndex(9, 2)] * 0.69684697253055;
    c_out[SHIndex(9, -9)] += c_in[SHIndex(9, 4)] * -0.36157613954394;
    c_out[SHIndex(9, -9)] += c_in[SHIndex(9, 6)] * 0.11158481919598;
    c_out[SHIndex(9, -9)] += c_in[SHIndex(9, 8)] * -0.01657281518406;
    c_out[SHIndex(9, -8)] += c_in[SHIndex(9, -8)] * 0.0625;
    c_out[SHIndex(9, -8)] += c_in[SHIndex(9, -6)] * -0.31560952932381;
    c_out[SHIndex(9, -8)] += c_in[SHIndex(9, -4)] * 0.68179450716473;
    c_out[SHIndex(9, -8)] += c_in[SHIndex(9, -2)] * -0.65699362630089;
    c_out[SHIndex(9, -7)] += c_in[SHIndex(9, 0)] * -0.44314852502787;
    c_out[SHIndex(9, -7)] += c_in[SHIndex(9, 2)] * 0.056336738679125;
    c_out[SHIndex(9, -7)] += c_in[SHIndex(9, 4)] * 0.67232906168594;
    c_out[SHIndex(9, -7)] += c_in[SHIndex(9, 6)] * -0.56832917123354;
    c_out[SHIndex(9, -7)] += c_in[SHIndex(9, 8)] * 0.15944009087468;
    c_out[SHIndex(9, -6)] += c_in[SHIndex(9, -8)] * -0.31560952932381;
    c_out[SHIndex(9, -6)] += c_in[SHIndex(9, -6)] * 0.71875;
    c_out[SHIndex(9, -6)] += c_in[SHIndex(9, -4)] * -0.20252314682525;
    c_out[SHIndex(9, -6)] += c_in[SHIndex(9, -2)] * -0.58546856234985;
    c_out[SHIndex(9, -5)] += c_in[SHIndex(9, 0)] * -0.39636409043643;
    c_out[SHIndex(9, -5)] += c_in[SHIndex(9, 2)] * -0.25194555463433;
    c_out[SHIndex(9, -5)] += c_in[SHIndex(9, 4)] * 0.39218438743785;
    c_out[SHIndex(9, -5)] += c_in[SHIndex(9, 6)] * 0.60515364784491;
    c_out[SHIndex(9, -5)] += c_in[SHIndex(9, 8)] * -0.50931268790646;
    c_out[SHIndex(9, -4)] += c_in[SHIndex(9, -8)] * 0.68179450716473;
    c_out[SHIndex(9, -4)] += c_in[SHIndex(9, -6)] * -0.20252314682525;
    c_out[SHIndex(9, -4)] += c_in[SHIndex(9, -4)] * -0.5625;
    c_out[SHIndex(9, -4)] += c_in[SHIndex(9, -2)] * -0.421585548851;
    c_out[SHIndex(9, -3)] += c_in[SHIndex(9, 0)] * -0.37548796377181;
    c_out[SHIndex(9, -3)] += c_in[SHIndex(9, 2)] * -0.42961647140211;
    c_out[SHIndex(9, -3)] += c_in[SHIndex(9, 4)] * -0.13799626353637;
    c_out[SHIndex(9, -3)] += c_in[SHIndex(9, 6)] * 0.2981060004428;
    c_out[SHIndex(9, -3)] += c_in[SHIndex(9, 8)] * 0.75268075590685;
    c_out[SHIndex(9, -2)] += c_in[SHIndex(9, -8)] * -0.65699362630089;
    c_out[SHIndex(9, -2)] += c_in[SHIndex(9, -6)] * -0.58546856234985;
    c_out[SHIndex(9, -2)] += c_in[SHIndex(9, -4)] * -0.421585548851;
    c_out[SHIndex(9, -2)] += c_in[SHIndex(9, -2)] * -0.21875;
    c_out[SHIndex(9, -1)] += c_in[SHIndex(9, 0)] * -0.36685490255856;
    c_out[SHIndex(9, -1)] += c_in[SHIndex(9, 2)] * -0.51301422373069;
    c_out[SHIndex(9, -1)] += c_in[SHIndex(9, 4)] * -0.49435287561114;
    c_out[SHIndex(9, -1)] += c_in[SHIndex(9, 6)] * -0.45768182862115;
    c_out[SHIndex(9, -1)] += c_in[SHIndex(9, 8)] * -0.38519665736316;
    c_out[SHIndex(9, 0)] += c_in[SHIndex(9, -9)] * -0.60904939217552;
    c_out[SHIndex(9, 0)] += c_in[SHIndex(9, -7)] * -0.44314852502787;
    c_out[SHIndex(9, 0)] += c_in[SHIndex(9, -5)] * -0.39636409043643;
    c_out[SHIndex(9, 0)] += c_in[SHIndex(9, -3)] * -0.37548796377181;
    c_out[SHIndex(9, 0)] += c_in[SHIndex(9, -1)] * -0.36685490255856;
    c_out[SHIndex(9, 1)] += c_in[SHIndex(9, 1)] * 0.0546875;
    c_out[SHIndex(9, 1)] += c_in[SHIndex(9, 3)] * 0.16792332234535;
    c_out[SHIndex(9, 1)] += c_in[SHIndex(9, 5)] * 0.29543235001858;
    c_out[SHIndex(9, 1)] += c_in[SHIndex(9, 7)] * 0.46242477217584;
    c_out[SHIndex(9, 1)] += c_in[SHIndex(9, 9)] * 0.81712550553564;
    c_out[SHIndex(9, 2)] += c_in[SHIndex(9, -9)] * 0.69684697253055;
    c_out[SHIndex(9, 2)] += c_in[SHIndex(9, -7)] * 0.056336738679125;
    c_out[SHIndex(9, 2)] += c_in[SHIndex(9, -5)] * -0.25194555463433;
    c_out[SHIndex(9, 2)] += c_in[SHIndex(9, -3)] * -0.42961647140211;
    c_out[SHIndex(9, 2)] += c_in[SHIndex(9, -1)] * -0.51301422373069;
    c_out[SHIndex(9, 3)] += c_in[SHIndex(9, 1)] * 0.16792332234535;
    c_out[SHIndex(9, 3)] += c_in[SHIndex(9, 3)] * 0.453125;
    c_out[SHIndex(9, 3)] += c_in[SHIndex(9, 5)] * 0.57727978755972;
    c_out[SHIndex(9, 3)] += c_in[SHIndex(9, 7)] * 0.38725105410605;
    c_out[SHIndex(9, 3)] += c_in[SHIndex(9, 9)] * -0.53222566657035;
    c_out[SHIndex(9, 4)] += c_in[SHIndex(9, -9)] * -0.36157613954394;
    c_out[SHIndex(9, 4)] += c_in[SHIndex(9, -7)] * 0.67232906168594;
    c_out[SHIndex(9, 4)] += c_in[SHIndex(9, -5)] * 0.39218438743785;
    c_out[SHIndex(9, 4)] += c_in[SHIndex(9, -3)] * -0.13799626353637;
    c_out[SHIndex(9, 4)] += c_in[SHIndex(9, -1)] * -0.49435287561114;
    c_out[SHIndex(9, 5)] += c_in[SHIndex(9, 1)] * 0.29543235001858;
    c_out[SHIndex(9, 5)] += c_in[SHIndex(9, 3)] * 0.57727978755972;
    c_out[SHIndex(9, 5)] += c_in[SHIndex(9, 5)] * 0.140625;
    c_out[SHIndex(9, 5)] += c_in[SHIndex(9, 7)] * -0.7162405240429;
    c_out[SHIndex(9, 5)] += c_in[SHIndex(9, 9)] * 0.2160830732178;
    c_out[SHIndex(9, 6)] += c_in[SHIndex(9, -9)] * 0.11158481919598;
    c_out[SHIndex(9, 6)] += c_in[SHIndex(9, -7)] * -0.56832917123354;
    c_out[SHIndex(9, 6)] += c_in[SHIndex(9, -5)] * 0.60515364784491;
    c_out[SHIndex(9, 6)] += c_in[SHIndex(9, -3)] * 0.2981060004428;
    c_out[SHIndex(9, 6)] += c_in[SHIndex(9, -1)] * -0.45768182862115;
    c_out[SHIndex(9, 7)] += c_in[SHIndex(9, 1)] * 0.46242477217584;
    c_out[SHIndex(9, 7)] += c_in[SHIndex(9, 3)] * 0.38725105410605;
    c_out[SHIndex(9, 7)] += c_in[SHIndex(9, 5)] * -0.7162405240429;
    c_out[SHIndex(9, 7)] += c_in[SHIndex(9, 7)] * 0.34765625;
    c_out[SHIndex(9, 7)] += c_in[SHIndex(9, 9)] * -0.048317644050207;
    c_out[SHIndex(9, 8)] += c_in[SHIndex(9, -9)] * -0.01657281518406;
    c_out[SHIndex(9, 8)] += c_in[SHIndex(9, -7)] * 0.15944009087468;
    c_out[SHIndex(9, 8)] += c_in[SHIndex(9, -5)] * -0.50931268790646;
    c_out[SHIndex(9, 8)] += c_in[SHIndex(9, -3)] * 0.75268075590685;
    c_out[SHIndex(9, 8)] += c_in[SHIndex(9, -1)] * -0.38519665736316;
    c_out[SHIndex(9, 9)] += c_in[SHIndex(9, 1)] * 0.81712550553564;
    c_out[SHIndex(9, 9)] += c_in[SHIndex(9, 3)] * -0.53222566657035;
    c_out[SHIndex(9, 9)] += c_in[SHIndex(9, 5)] * 0.2160830732178;
    c_out[SHIndex(9, 9)] += c_in[SHIndex(9, 7)] * -0.048317644050207;
    c_out[SHIndex(9, 9)] += c_in[SHIndex(9, 9)] * 0.00390625;
    if (lmax == 9) return;
    
    
    c_out[SHIndex(10, -10)] += c_in[SHIndex(10, 1)] * -0.80044772017564;
    c_out[SHIndex(10, -10)] += c_in[SHIndex(10, 3)] * 0.54379714235296;
    c_out[SHIndex(10, -10)] += c_in[SHIndex(10, 5)] * -0.24319347525427;
    c_out[SHIndex(10, -10)] += c_in[SHIndex(10, 7)] * 0.065945089906774;
    c_out[SHIndex(10, -10)] += c_in[SHIndex(10, 9)] * -0.0087346405371086;
    c_out[SHIndex(10, -9)] += c_in[SHIndex(10, -9)] * 0.03515625;
    c_out[SHIndex(10, -9)] += c_in[SHIndex(10, -7)] * -0.20644078533943;
    c_out[SHIndex(10, -9)] += c_in[SHIndex(10, -5)] * 0.54379714235296;
    c_out[SHIndex(10, -9)] += c_in[SHIndex(10, -3)] * -0.72958042576281;
    c_out[SHIndex(10, -9)] += c_in[SHIndex(10, -1)] * 0.35797110294949;
    c_out[SHIndex(10, -8)] += c_in[SHIndex(10, 1)] * -0.4645646483537;
    c_out[SHIndex(10, -8)] += c_in[SHIndex(10, 3)] * -0.31560952932381;
    c_out[SHIndex(10, -8)] += c_in[SHIndex(10, 5)] * 0.70572436191476;
    c_out[SHIndex(10, -8)] += c_in[SHIndex(10, 7)] * -0.42100604954086;
    c_out[SHIndex(10, -8)] += c_in[SHIndex(10, 9)] * 0.09631896879639;
    c_out[SHIndex(10, -7)] += c_in[SHIndex(10, -9)] * -0.20644078533943;
    c_out[SHIndex(10, -7)] += c_in[SHIndex(10, -7)] * 0.62890625;
    c_out[SHIndex(10, -7)] += c_in[SHIndex(10, -5)] * -0.5041938375082;
    c_out[SHIndex(10, -7)] += c_in[SHIndex(10, -3)] * -0.35432938970152;
    c_out[SHIndex(10, -7)] += c_in[SHIndex(10, -1)] * 0.42672987787578;
    c_out[SHIndex(10, -6)] += c_in[SHIndex(10, 1)] * -0.31049159295684;
    c_out[SHIndex(10, -6)] += c_in[SHIndex(10, 3)] * -0.5390625;
    c_out[SHIndex(10, -6)] += c_in[SHIndex(10, 5)] * 0.017469281074217;
    c_out[SHIndex(10, -6)] += c_in[SHIndex(10, 7)] * 0.69255289805297;
    c_out[SHIndex(10, -6)] += c_in[SHIndex(10, 9)] * -0.36479021288141;
    c_out[SHIndex(10, -5)] += c_in[SHIndex(10, -9)] * 0.54379714235296;
    c_out[SHIndex(10, -5)] += c_in[SHIndex(10, -7)] * -0.5041938375082;
    c_out[SHIndex(10, -5)] += c_in[SHIndex(10, -5)] * -0.484375;
    c_out[SHIndex(10, -5)] += c_in[SHIndex(10, -3)] * 0.034938562148434;
    c_out[SHIndex(10, -5)] += c_in[SHIndex(10, -1)] * 0.46285353886246;
    c_out[SHIndex(10, -4)] += c_in[SHIndex(10, 1)] * -0.19515618744995;
    c_out[SHIndex(10, -4)] += c_in[SHIndex(10, 3)] * -0.48613591206575;
    c_out[SHIndex(10, -4)] += c_in[SHIndex(10, 5)] * -0.49410588440131;
    c_out[SHIndex(10, -4)] += c_in[SHIndex(10, 7)] * -0.091108623356958;
    c_out[SHIndex(10, -4)] += c_in[SHIndex(10, 9)] * 0.68785502197047;
    c_out[SHIndex(10, -3)] += c_in[SHIndex(10, -9)] * -0.72958042576281;
    c_out[SHIndex(10, -3)] += c_in[SHIndex(10, -7)] * -0.35432938970152;
    c_out[SHIndex(10, -3)] += c_in[SHIndex(10, -5)] * 0.034938562148434;
    c_out[SHIndex(10, -3)] += c_in[SHIndex(10, -3)] * 0.328125;
    c_out[SHIndex(10, -3)] += c_in[SHIndex(10, -1)] * 0.4829869223773;
    c_out[SHIndex(10, -2)] += c_in[SHIndex(10, 1)] * -0.094721528538923;
    c_out[SHIndex(10, -2)] += c_in[SHIndex(10, 3)] * -0.27885262964961;
    c_out[SHIndex(10, -2)] += c_in[SHIndex(10, 5)] * -0.44538102542935;
    c_out[SHIndex(10, -2)] += c_in[SHIndex(10, 7)] * -0.57486942301328;
    c_out[SHIndex(10, -2)] += c_in[SHIndex(10, 9)] * -0.62002413794999;
    c_out[SHIndex(10, -1)] += c_in[SHIndex(10, -9)] * 0.35797110294949;
    c_out[SHIndex(10, -1)] += c_in[SHIndex(10, -7)] * 0.42672987787578;
    c_out[SHIndex(10, -1)] += c_in[SHIndex(10, -5)] * 0.46285353886246;
    c_out[SHIndex(10, -1)] += c_in[SHIndex(10, -3)] * 0.4829869223773;
    c_out[SHIndex(10, -1)] += c_in[SHIndex(10, -1)] * 0.4921875;
    c_out[SHIndex(10, 0)] += c_in[SHIndex(10, 0)] * -0.24609375;
    c_out[SHIndex(10, 0)] += c_in[SHIndex(10, 2)] * -0.35123682832287;
    c_out[SHIndex(10, 0)] += c_in[SHIndex(10, 4)] * -0.36182925552842;
    c_out[SHIndex(10, 0)] += c_in[SHIndex(10, 6)] * -0.38377788032374;
    c_out[SHIndex(10, 0)] += c_in[SHIndex(10, 8)] * -0.43066295528486;
    c_out[SHIndex(10, 0)] += c_in[SHIndex(10, 10)] * -0.59362791713657;
    c_out[SHIndex(10, 1)] += c_in[SHIndex(10, -10)] * -0.80044772017564;
    c_out[SHIndex(10, 1)] += c_in[SHIndex(10, -8)] * -0.4645646483537;
    c_out[SHIndex(10, 1)] += c_in[SHIndex(10, -6)] * -0.31049159295684;
    c_out[SHIndex(10, 1)] += c_in[SHIndex(10, -4)] * -0.19515618744995;
    c_out[SHIndex(10, 1)] += c_in[SHIndex(10, -2)] * -0.094721528538923;
    c_out[SHIndex(10, 2)] += c_in[SHIndex(10, 0)] * -0.35123682832287;
    c_out[SHIndex(10, 2)] += c_in[SHIndex(10, 2)] * -0.46484375;
    c_out[SHIndex(10, 2)] += c_in[SHIndex(10, 4)] * -0.36618880141431;
    c_out[SHIndex(10, 2)] += c_in[SHIndex(10, 6)] * -0.18922142726223;
    c_out[SHIndex(10, 2)] += c_in[SHIndex(10, 8)] * 0.10058119679362;
    c_out[SHIndex(10, 2)] += c_in[SHIndex(10, 10)] * 0.69320806007344;
    c_out[SHIndex(10, 3)] += c_in[SHIndex(10, -10)] * 0.54379714235296;
    c_out[SHIndex(10, 3)] += c_in[SHIndex(10, -8)] * -0.31560952932381;
    c_out[SHIndex(10, 3)] += c_in[SHIndex(10, -6)] * -0.5390625;
    c_out[SHIndex(10, 3)] += c_in[SHIndex(10, -4)] * -0.48613591206575;
    c_out[SHIndex(10, 3)] += c_in[SHIndex(10, -2)] * -0.27885262964961;
    c_out[SHIndex(10, 4)] += c_in[SHIndex(10, 0)] * -0.36182925552842;
    c_out[SHIndex(10, 4)] += c_in[SHIndex(10, 2)] * -0.36618880141431;
    c_out[SHIndex(10, 4)] += c_in[SHIndex(10, 4)] * 0.015625;
    c_out[SHIndex(10, 4)] += c_in[SHIndex(10, 6)] * 0.45851455342565;
    c_out[SHIndex(10, 4)] += c_in[SHIndex(10, 8)] * 0.6137165055779;
    c_out[SHIndex(10, 4)] += c_in[SHIndex(10, 10)] * -0.38452264694765;
    c_out[SHIndex(10, 5)] += c_in[SHIndex(10, -10)] * -0.24319347525427;
    c_out[SHIndex(10, 5)] += c_in[SHIndex(10, -8)] * 0.70572436191476;
    c_out[SHIndex(10, 5)] += c_in[SHIndex(10, -6)] * 0.017469281074217;
    c_out[SHIndex(10, 5)] += c_in[SHIndex(10, -4)] * -0.49410588440131;
    c_out[SHIndex(10, 5)] += c_in[SHIndex(10, -2)] * -0.44538102542935;
    c_out[SHIndex(10, 6)] += c_in[SHIndex(10, 0)] * -0.38377788032374;
    c_out[SHIndex(10, 6)] += c_in[SHIndex(10, 2)] * -0.18922142726223;
    c_out[SHIndex(10, 6)] += c_in[SHIndex(10, 4)] * 0.45851455342565;
    c_out[SHIndex(10, 6)] += c_in[SHIndex(10, 6)] * 0.462890625;
    c_out[SHIndex(10, 6)] += c_in[SHIndex(10, 8)] * -0.61149346306489;
    c_out[SHIndex(10, 6)] += c_in[SHIndex(10, 10)] * 0.13594928558824;
    c_out[SHIndex(10, 7)] += c_in[SHIndex(10, -10)] * 0.065945089906774;
    c_out[SHIndex(10, 7)] += c_in[SHIndex(10, -8)] * -0.42100604954086;
    c_out[SHIndex(10, 7)] += c_in[SHIndex(10, -6)] * 0.69255289805297;
    c_out[SHIndex(10, 7)] += c_in[SHIndex(10, -4)] * -0.091108623356958;
    c_out[SHIndex(10, 7)] += c_in[SHIndex(10, -2)] * -0.57486942301328;
    c_out[SHIndex(10, 8)] += c_in[SHIndex(10, 0)] * -0.43066295528486;
    c_out[SHIndex(10, 8)] += c_in[SHIndex(10, 2)] * 0.10058119679362;
    c_out[SHIndex(10, 8)] += c_in[SHIndex(10, 4)] * 0.6137165055779;
    c_out[SHIndex(10, 8)] += c_in[SHIndex(10, 6)] * -0.61149346306489;
    c_out[SHIndex(10, 8)] += c_in[SHIndex(10, 8)] * 0.23046875;
    c_out[SHIndex(10, 8)] += c_in[SHIndex(10, 10)] * -0.026921970218926;
    c_out[SHIndex(10, 9)] += c_in[SHIndex(10, -10)] * -0.0087346405371086;
    c_out[SHIndex(10, 9)] += c_in[SHIndex(10, -8)] * 0.09631896879639;
    c_out[SHIndex(10, 9)] += c_in[SHIndex(10, -6)] * -0.36479021288141;
    c_out[SHIndex(10, 9)] += c_in[SHIndex(10, -4)] * 0.68785502197047;
    c_out[SHIndex(10, 9)] += c_in[SHIndex(10, -2)] * -0.62002413794999;
    c_out[SHIndex(10, 10)] += c_in[SHIndex(10, 0)] * -0.59362791713657;
    c_out[SHIndex(10, 10)] += c_in[SHIndex(10, 2)] * 0.69320806007344;
    c_out[SHIndex(10, 10)] += c_in[SHIndex(10, 4)] * -0.38452264694765;
    c_out[SHIndex(10, 10)] += c_in[SHIndex(10, 6)] * 0.13594928558824;
    c_out[SHIndex(10, 10)] += c_in[SHIndex(10, 8)] * -0.026921970218926;
    c_out[SHIndex(10, 10)] += c_in[SHIndex(10, 10)] * 0.001953125;
    if (lmax == 10) return;
    
    
    c_out[SHIndex(11, -11)] += c_in[SHIndex(11, 0)] * 0.57997947393468;
    c_out[SHIndex(11, -11)] += c_in[SHIndex(11, 2)] * -0.68875008418642;
    c_out[SHIndex(11, -11)] += c_in[SHIndex(11, 4)] * 0.4032907544405;
    c_out[SHIndex(11, -11)] += c_in[SHIndex(11, 6)] * -0.158474160191;
    c_out[SHIndex(11, -11)] += c_in[SHIndex(11, 8)] * 0.0383230798249;
    c_out[SHIndex(11, -11)] += c_in[SHIndex(11, 10)] * -0.0045804841404526;
    c_out[SHIndex(11, -10)] += c_in[SHIndex(11, -10)] * 0.01953125;
    c_out[SHIndex(11, -10)] += c_in[SHIndex(11, -8)] * -0.13072812914595;
    c_out[SHIndex(11, -10)] += c_in[SHIndex(11, -6)] * 0.4054416537189;
    c_out[SHIndex(11, -10)] += c_in[SHIndex(11, -4)] * -0.68785502197047;
    c_out[SHIndex(11, -10)] += c_in[SHIndex(11, -2)] * 0.58736804535412;
    c_out[SHIndex(11, -9)] += c_in[SHIndex(11, 0)] * 0.41975832570892;
    c_out[SHIndex(11, -9)] += c_in[SHIndex(11, 2)] * -0.13594928558824;
    c_out[SHIndex(11, -9)] += c_in[SHIndex(11, 4)] * -0.55722625443603;
    c_out[SHIndex(11, -9)] += c_in[SHIndex(11, 6)] * 0.63603688806046;
    c_out[SHIndex(11, -9)] += c_in[SHIndex(11, 8)] * -0.29501240332439;
    c_out[SHIndex(11, -9)] += c_in[SHIndex(11, 10)] * 0.0569596350446;
    c_out[SHIndex(11, -8)] += c_in[SHIndex(11, -10)] * -0.13072812914595;
    c_out[SHIndex(11, -8)] += c_in[SHIndex(11, -8)] * 0.5;
    c_out[SHIndex(11, -8)] += c_in[SHIndex(11, -6)] * -0.64612728525423;
    c_out[SHIndex(11, -8)] += c_in[SHIndex(11, -2)] * 0.56163127361286;
    c_out[SHIndex(11, -7)] += c_in[SHIndex(11, 0)] * 0.37296505974557;
    c_out[SHIndex(11, -7)] += c_in[SHIndex(11, 2)] * 0.13689999147559;
    c_out[SHIndex(11, -7)] += c_in[SHIndex(11, 4)] * -0.49510851970963;
    c_out[SHIndex(11, -7)] += c_in[SHIndex(11, 6)] * -0.32425698663836;
    c_out[SHIndex(11, -7)] += c_in[SHIndex(11, 8)] * 0.65553753643092;
    c_out[SHIndex(11, -7)] += c_in[SHIndex(11, 10)] * -0.24930093301108;
    c_out[SHIndex(11, -6)] += c_in[SHIndex(11, -10)] * 0.4054416537189;
    c_out[SHIndex(11, -6)] += c_in[SHIndex(11, -8)] * -0.64612728525423;
    c_out[SHIndex(11, -6)] += c_in[SHIndex(11, -6)] * -0.18359375;
    c_out[SHIndex(11, -6)] += c_in[SHIndex(11, -4)] * 0.41751216299648;
    c_out[SHIndex(11, -6)] += c_in[SHIndex(11, -2)] * 0.45838141911581;
    c_out[SHIndex(11, -5)] += c_in[SHIndex(11, 0)] * 0.35033967020805;
    c_out[SHIndex(11, -5)] += c_in[SHIndex(11, 2)] * 0.31014124452052;
    c_out[SHIndex(11, -5)] += c_in[SHIndex(11, 4)] * -0.134354558765;
    c_out[SHIndex(11, -5)] += c_in[SHIndex(11, 6)] * -0.52272828294257;
    c_out[SHIndex(11, -5)] += c_in[SHIndex(11, 8)] * -0.40784785676472;
    c_out[SHIndex(11, -5)] += c_in[SHIndex(11, 10)] * 0.56871666443773;
    c_out[SHIndex(11, -4)] += c_in[SHIndex(11, -10)] * -0.68785502197047;
    c_out[SHIndex(11, -4)] += c_in[SHIndex(11, -6)] * 0.41751216299648;
    c_out[SHIndex(11, -4)] += c_in[SHIndex(11, -4)] * 0.5;
    c_out[SHIndex(11, -4)] += c_in[SHIndex(11, -2)] * 0.32021721143624;
    c_out[SHIndex(11, -3)] += c_in[SHIndex(11, 0)] * 0.33846027667471;
    c_out[SHIndex(11, -3)] += c_in[SHIndex(11, 2)] * 0.41655170126194;
    c_out[SHIndex(11, -3)] += c_in[SHIndex(11, 4)] * 0.235349536428;
    c_out[SHIndex(11, -3)] += c_in[SHIndex(11, 6)] * -0.047641839522072;
    c_out[SHIndex(11, -3)] += c_in[SHIndex(11, 8)] * -0.3940184631669;
    c_out[SHIndex(11, -3)] += c_in[SHIndex(11, 10)] * -0.70641320967457;
    c_out[SHIndex(11, -2)] += c_in[SHIndex(11, -10)] * 0.58736804535412;
    c_out[SHIndex(11, -2)] += c_in[SHIndex(11, -8)] * 0.56163127361286;
    c_out[SHIndex(11, -2)] += c_in[SHIndex(11, -6)] * 0.45838141911581;
    c_out[SHIndex(11, -2)] += c_in[SHIndex(11, -4)] * 0.32021721143624;
    c_out[SHIndex(11, -2)] += c_in[SHIndex(11, -2)] * 0.1640625;
    c_out[SHIndex(11, -1)] += c_in[SHIndex(11, 0)] * 0.33321251269015;
    c_out[SHIndex(11, -1)] += c_in[SHIndex(11, 2)] * 0.46765007670082;
    c_out[SHIndex(11, -1)] += c_in[SHIndex(11, 4)] * 0.45637974396672;
    c_out[SHIndex(11, -1)] += c_in[SHIndex(11, 6)] * 0.4355293578316;
    c_out[SHIndex(11, -1)] += c_in[SHIndex(11, 8)] * 0.40022386008782;
    c_out[SHIndex(11, -1)] += c_in[SHIndex(11, 10)] * 0.33485130540064;
    c_out[SHIndex(11, 0)] += c_in[SHIndex(11, -11)] * 0.57997947393468;
    c_out[SHIndex(11, 0)] += c_in[SHIndex(11, -9)] * 0.41975832570892;
    c_out[SHIndex(11, 0)] += c_in[SHIndex(11, -7)] * 0.37296505974557;
    c_out[SHIndex(11, 0)] += c_in[SHIndex(11, -5)] * 0.35033967020805;
    c_out[SHIndex(11, 0)] += c_in[SHIndex(11, -3)] * 0.33846027667471;
    c_out[SHIndex(11, 0)] += c_in[SHIndex(11, -1)] * 0.33321251269015;
    c_out[SHIndex(11, 1)] += c_in[SHIndex(11, 1)] * -0.041015625;
    c_out[SHIndex(11, 1)] += c_in[SHIndex(11, 3)] * -0.1249847402795;
    c_out[SHIndex(11, 1)] += c_in[SHIndex(11, 5)] * -0.21561916177558;
    c_out[SHIndex(11, 1)] += c_in[SHIndex(11, 7)] * -0.32136177701095;
    c_out[SHIndex(11, 1)] += c_in[SHIndex(11, 9)] * -0.46501810346249;
    c_out[SHIndex(11, 1)] += c_in[SHIndex(11, 11)] * -0.78529592002432;
    c_out[SHIndex(11, 2)] += c_in[SHIndex(11, -11)] * -0.68875008418642;
    c_out[SHIndex(11, 2)] += c_in[SHIndex(11, -9)] * -0.13594928558824;
    c_out[SHIndex(11, 2)] += c_in[SHIndex(11, -7)] * 0.13689999147559;
    c_out[SHIndex(11, 2)] += c_in[SHIndex(11, -5)] * 0.31014124452052;
    c_out[SHIndex(11, 2)] += c_in[SHIndex(11, -3)] * 0.41655170126194;
    c_out[SHIndex(11, 2)] += c_in[SHIndex(11, -1)] * 0.46765007670082;
    c_out[SHIndex(11, 3)] += c_in[SHIndex(11, 1)] * -0.1249847402795;
    c_out[SHIndex(11, 3)] += c_in[SHIndex(11, 3)] * -0.349609375;
    c_out[SHIndex(11, 3)] += c_in[SHIndex(11, 5)] * -0.49531072862952;
    c_out[SHIndex(11, 3)] += c_in[SHIndex(11, 7)] * -0.49716719187292;
    c_out[SHIndex(11, 3)] += c_in[SHIndex(11, 9)] * -0.25433782432394;
    c_out[SHIndex(11, 3)] += c_in[SHIndex(11, 11)] * 0.55222860860085;
    c_out[SHIndex(11, 4)] += c_in[SHIndex(11, -11)] * 0.4032907544405;
    c_out[SHIndex(11, 4)] += c_in[SHIndex(11, -9)] * -0.55722625443603;
    c_out[SHIndex(11, 4)] += c_in[SHIndex(11, -7)] * -0.49510851970963;
    c_out[SHIndex(11, 4)] += c_in[SHIndex(11, -5)] * -0.134354558765;
    c_out[SHIndex(11, 4)] += c_in[SHIndex(11, -3)] * 0.235349536428;
    c_out[SHIndex(11, 4)] += c_in[SHIndex(11, -1)] * 0.45637974396672;
    c_out[SHIndex(11, 5)] += c_in[SHIndex(11, 1)] * -0.21561916177558;
    c_out[SHIndex(11, 5)] += c_in[SHIndex(11, 3)] * -0.49531072862952;
    c_out[SHIndex(11, 5)] += c_in[SHIndex(11, 5)] * -0.3857421875;
    c_out[SHIndex(11, 5)] += c_in[SHIndex(11, 7)] * 0.14035007305082;
    c_out[SHIndex(11, 5)] += c_in[SHIndex(11, 9)] * 0.68448811471567;
    c_out[SHIndex(11, 5)] += c_in[SHIndex(11, 11)] * -0.2667517605753;
    c_out[SHIndex(11, 6)] += c_in[SHIndex(11, -11)] * -0.158474160191;
    c_out[SHIndex(11, 6)] += c_in[SHIndex(11, -9)] * 0.63603688806046;
    c_out[SHIndex(11, 6)] += c_in[SHIndex(11, -7)] * -0.32425698663836;
    c_out[SHIndex(11, 6)] += c_in[SHIndex(11, -5)] * -0.52272828294257;
    c_out[SHIndex(11, 6)] += c_in[SHIndex(11, -3)] * -0.047641839522072;
    c_out[SHIndex(11, 6)] += c_in[SHIndex(11, -1)] * 0.4355293578316;
    c_out[SHIndex(11, 7)] += c_in[SHIndex(11, 1)] * -0.32136177701095;
    c_out[SHIndex(11, 7)] += c_in[SHIndex(11, 3)] * -0.49716719187292;
    c_out[SHIndex(11, 7)] += c_in[SHIndex(11, 5)] * 0.14035007305082;
    c_out[SHIndex(11, 7)] += c_in[SHIndex(11, 7)] * 0.6279296875;
    c_out[SHIndex(11, 7)] += c_in[SHIndex(11, 9)] * -0.47810190182411;
    c_out[SHIndex(11, 7)] += c_in[SHIndex(11, 11)] * 0.08352321608099;
    c_out[SHIndex(11, 8)] += c_in[SHIndex(11, -11)] * 0.0383230798249;
    c_out[SHIndex(11, 8)] += c_in[SHIndex(11, -9)] * -0.29501240332439;
    c_out[SHIndex(11, 8)] += c_in[SHIndex(11, -7)] * 0.65553753643092;
    c_out[SHIndex(11, 8)] += c_in[SHIndex(11, -5)] * -0.40784785676472;
    c_out[SHIndex(11, 8)] += c_in[SHIndex(11, -3)] * -0.3940184631669;
    c_out[SHIndex(11, 8)] += c_in[SHIndex(11, -1)] * 0.40022386008782;
    c_out[SHIndex(11, 9)] += c_in[SHIndex(11, 1)] * -0.46501810346249;
    c_out[SHIndex(11, 9)] += c_in[SHIndex(11, 3)] * -0.25433782432394;
    c_out[SHIndex(11, 9)] += c_in[SHIndex(11, 5)] * 0.68448811471567;
    c_out[SHIndex(11, 9)] += c_in[SHIndex(11, 7)] * -0.47810190182411;
    c_out[SHIndex(11, 9)] += c_in[SHIndex(11, 9)] * 0.1474609375;
    c_out[SHIndex(11, 9)] += c_in[SHIndex(11, 11)] * -0.014842464993721;
    c_out[SHIndex(11, 10)] += c_in[SHIndex(11, -11)] * -0.0045804841404526;
    c_out[SHIndex(11, 10)] += c_in[SHIndex(11, -9)] * 0.0569596350446;
    c_out[SHIndex(11, 10)] += c_in[SHIndex(11, -7)] * -0.24930093301108;
    c_out[SHIndex(11, 10)] += c_in[SHIndex(11, -5)] * 0.56871666443773;
    c_out[SHIndex(11, 10)] += c_in[SHIndex(11, -3)] * -0.70641320967457;
    c_out[SHIndex(11, 10)] += c_in[SHIndex(11, -1)] * 0.33485130540064;
    c_out[SHIndex(11, 11)] += c_in[SHIndex(11, 1)] * -0.78529592002432;
    c_out[SHIndex(11, 11)] += c_in[SHIndex(11, 3)] * 0.55222860860085;
    c_out[SHIndex(11, 11)] += c_in[SHIndex(11, 5)] * -0.2667517605753;
    c_out[SHIndex(11, 11)] += c_in[SHIndex(11, 7)] * 0.08352321608099;
    c_out[SHIndex(11, 11)] += c_in[SHIndex(11, 9)] * -0.014842464993721;
    c_out[SHIndex(11, 11)] += c_in[SHIndex(11, 11)] * 0.0009765625;
    if (lmax == 11) return;
    
    
    c_out[SHIndex(12, -12)] += c_in[SHIndex(12, 1)] * 0.7714448170238;
    c_out[SHIndex(12, -12)] += c_in[SHIndex(12, 3)] * -0.55833076742161;
    c_out[SHIndex(12, -12)] += c_in[SHIndex(12, 5)] * 0.28725880996301;
    c_out[SHIndex(12, -12)] += c_in[SHIndex(12, 7)] * -0.1006664953504;
    c_out[SHIndex(12, -12)] += c_in[SHIndex(12, 9)] * 0.021967230232816;
    c_out[SHIndex(12, -12)] += c_in[SHIndex(12, 11)] * -0.0023920798269367;
    c_out[SHIndex(12, -11)] += c_in[SHIndex(12, -11)] * 0.0107421875;
    c_out[SHIndex(12, -11)] += c_in[SHIndex(12, -9)] * -0.08071275769896;
    c_out[SHIndex(12, -11)] += c_in[SHIndex(12, -7)] * 0.28767847243653;
    c_out[SHIndex(12, -11)] += c_in[SHIndex(12, -5)] * -0.58636459044042;
    c_out[SHIndex(12, -11)] += c_in[SHIndex(12, -3)] * 0.68381274393975;
    c_out[SHIndex(12, -11)] += c_in[SHIndex(12, -1)] * -0.31494102773718;
    c_out[SHIndex(12, -10)] += c_in[SHIndex(12, 1)] * 0.46435521203003;
    c_out[SHIndex(12, -10)] += c_in[SHIndex(12, 3)] * 0.20164537722025;
    c_out[SHIndex(12, -10)] += c_in[SHIndex(12, 5)] * -0.65705604098221;
    c_out[SHIndex(12, -10)] += c_in[SHIndex(12, 7)] * 0.52110934255626;
    c_out[SHIndex(12, -10)] += c_in[SHIndex(12, 9)] * -0.19834078136318;
    c_out[SHIndex(12, -10)] += c_in[SHIndex(12, 11)] * 0.033116845620729;
    c_out[SHIndex(12, -9)] += c_in[SHIndex(12, -11)] * -0.08071275769896;
    c_out[SHIndex(12, -9)] += c_in[SHIndex(12, -9)] * 0.3720703125;
    c_out[SHIndex(12, -9)] += c_in[SHIndex(12, -7)] * -0.65785022183448;
    c_out[SHIndex(12, -9)] += c_in[SHIndex(12, -5)] * 0.31925529371636;
    c_out[SHIndex(12, -9)] += c_in[SHIndex(12, -3)] * 0.42195411222922;
    c_out[SHIndex(12, -9)] += c_in[SHIndex(12, -1)] * -0.37724395759248;
    c_out[SHIndex(12, -8)] += c_in[SHIndex(12, 1)] * 0.32928552212043;
    c_out[SHIndex(12, -8)] += c_in[SHIndex(12, 3)] * 0.45497333155019;
    c_out[SHIndex(12, -8)] += c_in[SHIndex(12, 5)] * -0.23408184791529;
    c_out[SHIndex(12, -8)] += c_in[SHIndex(12, 7)] * -0.54296875;
    c_out[SHIndex(12, -8)] += c_in[SHIndex(12, 9)] * 0.55492127556106;
    c_out[SHIndex(12, -8)] += c_in[SHIndex(12, 11)] * -0.16438769853516;
    c_out[SHIndex(12, -7)] += c_in[SHIndex(12, -11)] * 0.28767847243653;
    c_out[SHIndex(12, -7)] += c_in[SHIndex(12, -9)] * -0.65785022183448;
    c_out[SHIndex(12, -7)] += c_in[SHIndex(12, -7)] * 0.1572265625;
    c_out[SHIndex(12, -7)] += c_in[SHIndex(12, -5)] * 0.52668415780941;
    c_out[SHIndex(12, -7)] += c_in[SHIndex(12, -3)] * 0.11374333288755;
    c_out[SHIndex(12, -7)] += c_in[SHIndex(12, -1)] * -0.41160690265054;
    c_out[SHIndex(12, -6)] += c_in[SHIndex(12, 1)] * 0.23130311098295;
    c_out[SHIndex(12, -6)] += c_in[SHIndex(12, 3)] * 0.49003980202465;
    c_out[SHIndex(12, -6)] += c_in[SHIndex(12, 5)] * 0.27404717188286;
    c_out[SHIndex(12, -6)] += c_in[SHIndex(12, 7)] * -0.30237819268448;
    c_out[SHIndex(12, -6)] += c_in[SHIndex(12, 9)] * -0.58930833781358;
    c_out[SHIndex(12, -6)] += c_in[SHIndex(12, 11)] * 0.43879508023282;
    c_out[SHIndex(12, -5)] += c_in[SHIndex(12, -11)] * -0.58636459044042;
    c_out[SHIndex(12, -5)] += c_in[SHIndex(12, -9)] * 0.31925529371636;
    c_out[SHIndex(12, -5)] += c_in[SHIndex(12, -7)] * 0.52668415780941;
    c_out[SHIndex(12, -5)] += c_in[SHIndex(12, -5)] * 0.2568359375;
    c_out[SHIndex(12, -5)] += c_in[SHIndex(12, -3)] * -0.15374580191487;
    c_out[SHIndex(12, -5)] += c_in[SHIndex(12, -1)] * -0.43272849689657;
    c_out[SHIndex(12, -4)] += c_in[SHIndex(12, 1)] * 0.14842464993721;
    c_out[SHIndex(12, -4)] += c_in[SHIndex(12, 3)] * 0.392578125;
    c_out[SHIndex(12, -4)] += c_in[SHIndex(12, 5)] * 0.48401456158384;
    c_out[SHIndex(12, -4)] += c_in[SHIndex(12, 7)] * 0.34122999866264;
    c_out[SHIndex(12, -4)] += c_in[SHIndex(12, 9)] * -0.074462490393392;
    c_out[SHIndex(12, -4)] += c_in[SHIndex(12, 11)] * -0.68381274393975;
    c_out[SHIndex(12, -3)] += c_in[SHIndex(12, -11)] * 0.68381274393975;
    c_out[SHIndex(12, -3)] += c_in[SHIndex(12, -9)] * 0.42195411222922;
    c_out[SHIndex(12, -3)] += c_in[SHIndex(12, -7)] * 0.11374333288755;
    c_out[SHIndex(12, -3)] += c_in[SHIndex(12, -5)] * -0.15374580191487;
    c_out[SHIndex(12, -3)] += c_in[SHIndex(12, -3)] * -0.345703125;
    c_out[SHIndex(12, -3)] += c_in[SHIndex(12, -1)] * -0.44527394981164;
    c_out[SHIndex(12, -2)] += c_in[SHIndex(12, 1)] * 0.072712931519478;
    c_out[SHIndex(12, -2)] += c_in[SHIndex(12, 3)] * 0.2152871844243;
    c_out[SHIndex(12, -2)] += c_in[SHIndex(12, 5)] * 0.34870255998744;
    c_out[SHIndex(12, -2)] += c_in[SHIndex(12, 7)] * 0.46435521203003;
    c_out[SHIndex(12, -2)] += c_in[SHIndex(12, 9)] * 0.54718531932211;
    c_out[SHIndex(12, -2)] += c_in[SHIndex(12, 11)] * 0.55833076742161;
    c_out[SHIndex(12, -1)] += c_in[SHIndex(12, -11)] * -0.31494102773718;
    c_out[SHIndex(12, -1)] += c_in[SHIndex(12, -9)] * -0.37724395759248;
    c_out[SHIndex(12, -1)] += c_in[SHIndex(12, -7)] * -0.41160690265054;
    c_out[SHIndex(12, -1)] += c_in[SHIndex(12, -5)] * -0.43272849689657;
    c_out[SHIndex(12, -1)] += c_in[SHIndex(12, -3)] * -0.44527394981164;
    c_out[SHIndex(12, -1)] += c_in[SHIndex(12, -1)] * -0.451171875;
    c_out[SHIndex(12, 0)] += c_in[SHIndex(12, 0)] * 0.2255859375;
    c_out[SHIndex(12, 0)] += c_in[SHIndex(12, 2)] * 0.32109161153485;
    c_out[SHIndex(12, 0)] += c_in[SHIndex(12, 4)] * 0.32771275372847;
    c_out[SHIndex(12, 0)] += c_in[SHIndex(12, 6)] * 0.34046896063985;
    c_out[SHIndex(12, 0)] += c_in[SHIndex(12, 8)] * 0.36352137351394;
    c_out[SHIndex(12, 0)] += c_in[SHIndex(12, 10)] * 0.41010741896822;
    c_out[SHIndex(12, 0)] += c_in[SHIndex(12, 12)] * 0.56776801212686;
    c_out[SHIndex(12, 1)] += c_in[SHIndex(12, -12)] * 0.7714448170238;
    c_out[SHIndex(12, 1)] += c_in[SHIndex(12, -10)] * 0.46435521203003;
    c_out[SHIndex(12, 1)] += c_in[SHIndex(12, -8)] * 0.32928552212043;
    c_out[SHIndex(12, 1)] += c_in[SHIndex(12, -6)] * 0.23130311098295;
    c_out[SHIndex(12, 1)] += c_in[SHIndex(12, -4)] * 0.14842464993721;
    c_out[SHIndex(12, 1)] += c_in[SHIndex(12, -2)] * 0.072712931519478;
    c_out[SHIndex(12, 2)] += c_in[SHIndex(12, 0)] * 0.32109161153485;
    c_out[SHIndex(12, 2)] += c_in[SHIndex(12, 2)] * 0.43359375;
    c_out[SHIndex(12, 2)] += c_in[SHIndex(12, 4)] * 0.37077237317519;
    c_out[SHIndex(12, 2)] += c_in[SHIndex(12, 6)] * 0.2609451018728;
    c_out[SHIndex(12, 2)] += c_in[SHIndex(12, 8)] * 0.092871042406006;
    c_out[SHIndex(12, 2)] += c_in[SHIndex(12, 10)] * -0.16464276106022;
    c_out[SHIndex(12, 2)] += c_in[SHIndex(12, 12)] * -0.68381274393975;
    c_out[SHIndex(12, 3)] += c_in[SHIndex(12, -12)] * -0.55833076742161;
    c_out[SHIndex(12, 3)] += c_in[SHIndex(12, -10)] * 0.20164537722025;
    c_out[SHIndex(12, 3)] += c_in[SHIndex(12, -8)] * 0.45497333155019;
    c_out[SHIndex(12, 3)] += c_in[SHIndex(12, -6)] * 0.49003980202465;
    c_out[SHIndex(12, 3)] += c_in[SHIndex(12, -4)] * 0.392578125;
    c_out[SHIndex(12, 3)] += c_in[SHIndex(12, -2)] * 0.2152871844243;
    c_out[SHIndex(12, 4)] += c_in[SHIndex(12, 0)] * 0.32771275372847;
    c_out[SHIndex(12, 4)] += c_in[SHIndex(12, 2)] * 0.37077237317519;
    c_out[SHIndex(12, 4)] += c_in[SHIndex(12, 4)] * 0.11669921875;
    c_out[SHIndex(12, 4)] += c_in[SHIndex(12, 6)] * -0.22371382266343;
    c_out[SHIndex(12, 4)] += c_in[SHIndex(12, 8)] * -0.51184499799396;
    c_out[SHIndex(12, 4)] += c_in[SHIndex(12, 10)] * -0.50411344305062;
    c_out[SHIndex(12, 4)] += c_in[SHIndex(12, 12)] * 0.41874807556621;
    c_out[SHIndex(12, 5)] += c_in[SHIndex(12, -12)] * 0.28725880996301;
    c_out[SHIndex(12, 5)] += c_in[SHIndex(12, -10)] * -0.65705604098221;
    c_out[SHIndex(12, 5)] += c_in[SHIndex(12, -8)] * -0.23408184791529;
    c_out[SHIndex(12, 5)] += c_in[SHIndex(12, -6)] * 0.27404717188286;
    c_out[SHIndex(12, 5)] += c_in[SHIndex(12, -4)] * 0.48401456158384;
    c_out[SHIndex(12, 5)] += c_in[SHIndex(12, -2)] * 0.34870255998744;
    c_out[SHIndex(12, 6)] += c_in[SHIndex(12, 0)] * 0.34046896063985;
    c_out[SHIndex(12, 6)] += c_in[SHIndex(12, 2)] * 0.2609451018728;
    c_out[SHIndex(12, 6)] += c_in[SHIndex(12, 4)] * -0.22371382266343;
    c_out[SHIndex(12, 6)] += c_in[SHIndex(12, 6)] * -0.525390625;
    c_out[SHIndex(12, 6)] += c_in[SHIndex(12, 8)] * -0.19810985037949;
    c_out[SHIndex(12, 6)] += c_in[SHIndex(12, 10)] * 0.64696804980671;
    c_out[SHIndex(12, 6)] += c_in[SHIndex(12, 12)] * -0.179137341369;
    c_out[SHIndex(12, 7)] += c_in[SHIndex(12, -12)] * -0.1006664953504;
    c_out[SHIndex(12, 7)] += c_in[SHIndex(12, -10)] * 0.52110934255626;
    c_out[SHIndex(12, 7)] += c_in[SHIndex(12, -8)] * -0.54296875;
    c_out[SHIndex(12, 7)] += c_in[SHIndex(12, -6)] * -0.30237819268448;
    c_out[SHIndex(12, 7)] += c_in[SHIndex(12, -4)] * 0.34122999866264;
    c_out[SHIndex(12, 7)] += c_in[SHIndex(12, -2)] * 0.46435521203003;
    c_out[SHIndex(12, 8)] += c_in[SHIndex(12, 0)] * 0.36352137351394;
    c_out[SHIndex(12, 8)] += c_in[SHIndex(12, 2)] * 0.092871042406006;
    c_out[SHIndex(12, 8)] += c_in[SHIndex(12, 4)] * -0.51184499799396;
    c_out[SHIndex(12, 8)] += c_in[SHIndex(12, 6)] * -0.19810985037949;
    c_out[SHIndex(12, 8)] += c_in[SHIndex(12, 8)] * 0.6572265625;
    c_out[SHIndex(12, 8)] += c_in[SHIndex(12, 10)] * -0.35144583567748;
    c_out[SHIndex(12, 8)] += c_in[SHIndex(12, 12)] * 0.050333247675202;
    c_out[SHIndex(12, 9)] += c_in[SHIndex(12, -12)] * 0.021967230232816;
    c_out[SHIndex(12, 9)] += c_in[SHIndex(12, -10)] * -0.19834078136318;
    c_out[SHIndex(12, 9)] += c_in[SHIndex(12, -8)] * 0.55492127556106;
    c_out[SHIndex(12, 9)] += c_in[SHIndex(12, -6)] * -0.58930833781358;
    c_out[SHIndex(12, 9)] += c_in[SHIndex(12, -4)] * -0.074462490393392;
    c_out[SHIndex(12, 9)] += c_in[SHIndex(12, -2)] * 0.54718531932211;
    c_out[SHIndex(12, 10)] += c_in[SHIndex(12, 0)] * 0.41010741896822;
    c_out[SHIndex(12, 10)] += c_in[SHIndex(12, 2)] * -0.16464276106022;
    c_out[SHIndex(12, 10)] += c_in[SHIndex(12, 4)] * -0.50411344305062;
    c_out[SHIndex(12, 10)] += c_in[SHIndex(12, 6)] * 0.64696804980671;
    c_out[SHIndex(12, 10)] += c_in[SHIndex(12, 8)] * -0.35144583567748;
    c_out[SHIndex(12, 10)] += c_in[SHIndex(12, 10)] * 0.091796875;
    c_out[SHIndex(12, 10)] += c_in[SHIndex(12, 12)] * -0.0081119373661309;
    c_out[SHIndex(12, 11)] += c_in[SHIndex(12, -12)] * -0.0023920798269367;
    c_out[SHIndex(12, 11)] += c_in[SHIndex(12, -10)] * 0.033116845620729;
    c_out[SHIndex(12, 11)] += c_in[SHIndex(12, -8)] * -0.16438769853516;
    c_out[SHIndex(12, 11)] += c_in[SHIndex(12, -6)] * 0.43879508023282;
    c_out[SHIndex(12, 11)] += c_in[SHIndex(12, -4)] * -0.68381274393975;
    c_out[SHIndex(12, 11)] += c_in[SHIndex(12, -2)] * 0.55833076742161;
    c_out[SHIndex(12, 12)] += c_in[SHIndex(12, 0)] * 0.56776801212686;
    c_out[SHIndex(12, 12)] += c_in[SHIndex(12, 2)] * -0.68381274393975;
    c_out[SHIndex(12, 12)] += c_in[SHIndex(12, 4)] * 0.41874807556621;
    c_out[SHIndex(12, 12)] += c_in[SHIndex(12, 6)] * -0.179137341369;
    c_out[SHIndex(12, 12)] += c_in[SHIndex(12, 8)] * 0.050333247675202;
    c_out[SHIndex(12, 12)] += c_in[SHIndex(12, 10)] * -0.0081119373661309;
    c_out[SHIndex(12, 12)] += c_in[SHIndex(12, 12)] * 0.00048828125;
    if (lmax == 12) return;
    
    
    c_out[SHIndex(13, -13)] += c_in[SHIndex(13, 0)] * -0.55674234096704;
    c_out[SHIndex(13, -13)] += c_in[SHIndex(13, 2)] * 0.67861257155894;
    c_out[SHIndex(13, -13)] += c_in[SHIndex(13, 4)] * -0.43155265360434;
    c_out[SHIndex(13, -13)] += c_in[SHIndex(13, 6)] * 0.19800993746086;
    c_out[SHIndex(13, -13)] += c_in[SHIndex(13, 8)] * -0.06261624017238;
    c_out[SHIndex(13, -13)] += c_in[SHIndex(13, 10)] * 0.012448778109357;
    c_out[SHIndex(13, -13)] += c_in[SHIndex(13, 12)] * -0.0012448778109357;
    c_out[SHIndex(13, -12)] += c_in[SHIndex(13, -12)] * 0.005859375;
    c_out[SHIndex(13, -12)] += c_in[SHIndex(13, -10)] * -0.048828125;
    c_out[SHIndex(13, -12)] += c_in[SHIndex(13, -8)] * 0.19648088031186;
    c_out[SHIndex(13, -12)] += c_in[SHIndex(13, -6)] * -0.46599532384532;
    c_out[SHIndex(13, -12)] += c_in[SHIndex(13, -4)] * 0.67707550826808;
    c_out[SHIndex(13, -12)] += c_in[SHIndex(13, -2)] * -0.53234749916129;
    c_out[SHIndex(13, -11)] += c_in[SHIndex(13, 0)] * -0.40147261151571;
    c_out[SHIndex(13, -11)] += c_in[SHIndex(13, 2)] * 0.18821326330232;
    c_out[SHIndex(13, -11)] += c_in[SHIndex(13, 4)] * 0.45482644910811;
    c_out[SHIndex(13, -11)] += c_in[SHIndex(13, 6)] * -0.64803329186808;
    c_out[SHIndex(13, -11)] += c_in[SHIndex(13, 8)] * 0.39943226817081;
    c_out[SHIndex(13, -11)] += c_in[SHIndex(13, 10)] * -0.12912985164247;
    c_out[SHIndex(13, -11)] += c_in[SHIndex(13, 12)] * 0.018989684065068;
    c_out[SHIndex(13, -10)] += c_in[SHIndex(13, -12)] * -0.048828125;
    c_out[SHIndex(13, -10)] += c_in[SHIndex(13, -10)] * 0.263671875;
    c_out[SHIndex(13, -10)] += c_in[SHIndex(13, -8)] * -0.58944264093559;
    c_out[SHIndex(13, -10)] += c_in[SHIndex(13, -6)] * 0.52812803369136;
    c_out[SHIndex(13, -10)] += c_in[SHIndex(13, -4)] * 0.13541510165362;
    c_out[SHIndex(13, -10)] += c_in[SHIndex(13, -2)] * -0.53234749916129;
    c_out[SHIndex(13, -9)] += c_in[SHIndex(13, 0)] * -0.35516344310694;
    c_out[SHIndex(13, -9)] += c_in[SHIndex(13, 2)] * -0.055501063431183;
    c_out[SHIndex(13, -9)] += c_in[SHIndex(13, 4)] * 0.51530701050804;
    c_out[SHIndex(13, -9)] += c_in[SHIndex(13, 6)] * 0.087450067714449;
    c_out[SHIndex(13, -9)] += c_in[SHIndex(13, 8)] * -0.62989984533657;
    c_out[SHIndex(13, -9)] += c_in[SHIndex(13, 10)] * 0.43555891764461;
    c_out[SHIndex(13, -9)] += c_in[SHIndex(13, 12)] * -0.10537715749466;
    c_out[SHIndex(13, -8)] += c_in[SHIndex(13, -12)] * 0.19648088031186;
    c_out[SHIndex(13, -8)] += c_in[SHIndex(13, -10)] * -0.58944264093559;
    c_out[SHIndex(13, -8)] += c_in[SHIndex(13, -8)] * 0.421875;
    c_out[SHIndex(13, -8)] += c_in[SHIndex(13, -6)] * 0.38293206041101;
    c_out[SHIndex(13, -8)] += c_in[SHIndex(13, -4)] * -0.26921970218926;
    c_out[SHIndex(13, -8)] += c_in[SHIndex(13, -2)] * -0.46568005127582;
    c_out[SHIndex(13, -7)] += c_in[SHIndex(13, 0)] * -0.33184809401475;
    c_out[SHIndex(13, -7)] += c_in[SHIndex(13, 2)] * -0.21780190046618;
    c_out[SHIndex(13, -7)] += c_in[SHIndex(13, 4)] * 0.28960676761644;
    c_out[SHIndex(13, -7)] += c_in[SHIndex(13, 6)] * 0.49108084137057;
    c_out[SHIndex(13, -7)] += c_in[SHIndex(13, 8)] * 0.0127888680212;
    c_out[SHIndex(13, -7)] += c_in[SHIndex(13, 10)] * -0.64326862003139;
    c_out[SHIndex(13, -7)] += c_in[SHIndex(13, 12)] * 0.3216343100157;
    c_out[SHIndex(13, -6)] += c_in[SHIndex(13, -12)] * -0.46599532384532;
    c_out[SHIndex(13, -6)] += c_in[SHIndex(13, -10)] * 0.52812803369136;
    c_out[SHIndex(13, -6)] += c_in[SHIndex(13, -8)] * 0.38293206041101;
    c_out[SHIndex(13, -6)] += c_in[SHIndex(13, -6)] * -0.134765625;
    c_out[SHIndex(13, -6)] += c_in[SHIndex(13, -4)] * -0.45121414845245;
    c_out[SHIndex(13, -6)] += c_in[SHIndex(13, -2)] * -0.3681524057339;
    c_out[SHIndex(13, -5)] += c_in[SHIndex(13, 0)] * -0.31847955038634;
    c_out[SHIndex(13, -5)] += c_in[SHIndex(13, 2)] * -0.32847213547974;
    c_out[SHIndex(13, -5)] += c_in[SHIndex(13, 4)] * -0.014846480269053;
    c_out[SHIndex(13, -5)] += c_in[SHIndex(13, 6)] * 0.34012635856225;
    c_out[SHIndex(13, -5)] += c_in[SHIndex(13, 8)] * 0.50923193110086;
    c_out[SHIndex(13, -5)] += c_in[SHIndex(13, 10)] * 0.23938234163585;
    c_out[SHIndex(13, -5)] += c_in[SHIndex(13, 12)] * -0.59845585408961;
    c_out[SHIndex(13, -4)] += c_in[SHIndex(13, -12)] * 0.67707550826808;
    c_out[SHIndex(13, -4)] += c_in[SHIndex(13, -10)] * 0.13541510165362;
    c_out[SHIndex(13, -4)] += c_in[SHIndex(13, -8)] * -0.26921970218926;
    c_out[SHIndex(13, -4)] += c_in[SHIndex(13, -6)] * -0.45121414845245;
    c_out[SHIndex(13, -4)] += c_in[SHIndex(13, -4)] * -0.427734375;
    c_out[SHIndex(13, -4)] += c_in[SHIndex(13, -2)] * -0.25337986301296;
    c_out[SHIndex(13, -3)] += c_in[SHIndex(13, 0)] * -0.31089561602034;
    c_out[SHIndex(13, -3)] += c_in[SHIndex(13, 2)] * -0.39838364181027;
    c_out[SHIndex(13, -3)] += c_in[SHIndex(13, 4)] * -0.27693877404914;
    c_out[SHIndex(13, -3)] += c_in[SHIndex(13, 6)] * -0.083251595146774;
    c_out[SHIndex(13, -3)] += c_in[SHIndex(13, 8)] * 0.16673428435382;
    c_out[SHIndex(13, -3)] += c_in[SHIndex(13, 10)] * 0.44139922820051;
    c_out[SHIndex(13, -3)] += c_in[SHIndex(13, 12)] * 0.66209884230076;
    c_out[SHIndex(13, -2)] += c_in[SHIndex(13, -12)] * -0.53234749916129;
    c_out[SHIndex(13, -2)] += c_in[SHIndex(13, -10)] * -0.53234749916129;
    c_out[SHIndex(13, -2)] += c_in[SHIndex(13, -8)] * -0.46568005127582;
    c_out[SHIndex(13, -2)] += c_in[SHIndex(13, -6)] * -0.3681524057339;
    c_out[SHIndex(13, -2)] += c_in[SHIndex(13, -4)] * -0.25337986301296;
    c_out[SHIndex(13, -2)] += c_in[SHIndex(13, -2)] * -0.12890625;
    c_out[SHIndex(13, -1)] += c_in[SHIndex(13, 0)] * -0.30742181295663;
    c_out[SHIndex(13, -1)] += c_in[SHIndex(13, 2)] * -0.43236470658687;
    c_out[SHIndex(13, -1)] += c_in[SHIndex(13, 4)] * -0.42493094836993;
    c_out[SHIndex(13, -1)] += c_in[SHIndex(13, 6)] * -0.41160690265054;
    c_out[SHIndex(13, -1)] += c_in[SHIndex(13, 8)] * -0.39048459390687;
    c_out[SHIndex(13, -1)] += c_in[SHIndex(13, 10)] * -0.357109558733;
    c_out[SHIndex(13, -1)] += c_in[SHIndex(13, 12)] * -0.29759129894417;
    c_out[SHIndex(13, 0)] += c_in[SHIndex(13, -13)] * -0.55674234096704;
    c_out[SHIndex(13, 0)] += c_in[SHIndex(13, -11)] * -0.40147261151571;
    c_out[SHIndex(13, 0)] += c_in[SHIndex(13, -9)] * -0.35516344310694;
    c_out[SHIndex(13, 0)] += c_in[SHIndex(13, -7)] * -0.33184809401475;
    c_out[SHIndex(13, 0)] += c_in[SHIndex(13, -5)] * -0.31847955038634;
    c_out[SHIndex(13, 0)] += c_in[SHIndex(13, -3)] * -0.31089561602034;
    c_out[SHIndex(13, 0)] += c_in[SHIndex(13, -1)] * -0.30742181295663;
    c_out[SHIndex(13, 1)] += c_in[SHIndex(13, 1)] * 0.0322265625;
    c_out[SHIndex(13, 1)] += c_in[SHIndex(13, 3)] * 0.097772148023234;
    c_out[SHIndex(13, 1)] += c_in[SHIndex(13, 5)] * 0.166928641738;
    c_out[SHIndex(13, 1)] += c_in[SHIndex(13, 7)] * 0.24350992753551;
    c_out[SHIndex(13, 1)] += c_in[SHIndex(13, 9)] * 0.33508120677026;
    c_out[SHIndex(13, 1)] += c_in[SHIndex(13, 11)] * 0.46294341611217;
    c_out[SHIndex(13, 1)] += c_in[SHIndex(13, 13)] * 0.75871192019586;
    c_out[SHIndex(13, 2)] += c_in[SHIndex(13, -13)] * 0.67861257155894;
    c_out[SHIndex(13, 2)] += c_in[SHIndex(13, -11)] * 0.18821326330232;
    c_out[SHIndex(13, 2)] += c_in[SHIndex(13, -9)] * -0.055501063431183;
    c_out[SHIndex(13, 2)] += c_in[SHIndex(13, -7)] * -0.21780190046618;
    c_out[SHIndex(13, 2)] += c_in[SHIndex(13, -5)] * -0.32847213547974;
    c_out[SHIndex(13, 2)] += c_in[SHIndex(13, -3)] * -0.39838364181027;
    c_out[SHIndex(13, 2)] += c_in[SHIndex(13, -1)] * -0.43236470658687;
    c_out[SHIndex(13, 3)] += c_in[SHIndex(13, 1)] * 0.097772148023234;
    c_out[SHIndex(13, 3)] += c_in[SHIndex(13, 3)] * 0.279052734375;
    c_out[SHIndex(13, 3)] += c_in[SHIndex(13, 5)] * 0.41641008901347;
    c_out[SHIndex(13, 3)] += c_in[SHIndex(13, 7)] * 0.47610564299334;
    c_out[SHIndex(13, 3)] += c_in[SHIndex(13, 9)] * 0.41417145645063;
    c_out[SHIndex(13, 3)] += c_in[SHIndex(13, 11)] * 0.15605819373554;
    c_out[SHIndex(13, 3)] += c_in[SHIndex(13, 13)] * -0.5626758194698;
    c_out[SHIndex(13, 4)] += c_in[SHIndex(13, -13)] * -0.43155265360434;
    c_out[SHIndex(13, 4)] += c_in[SHIndex(13, -11)] * 0.45482644910811;
    c_out[SHIndex(13, 4)] += c_in[SHIndex(13, -9)] * 0.51530701050804;
    c_out[SHIndex(13, 4)] += c_in[SHIndex(13, -7)] * 0.28960676761644;
    c_out[SHIndex(13, 4)] += c_in[SHIndex(13, -5)] * -0.014846480269053;
    c_out[SHIndex(13, 4)] += c_in[SHIndex(13, -3)] * -0.27693877404914;
    c_out[SHIndex(13, 4)] += c_in[SHIndex(13, -1)] * -0.42493094836993;
    c_out[SHIndex(13, 5)] += c_in[SHIndex(13, 1)] * 0.166928641738;
    c_out[SHIndex(13, 5)] += c_in[SHIndex(13, 3)] * 0.41641008901347;
    c_out[SHIndex(13, 5)] += c_in[SHIndex(13, 5)] * 0.438232421875;
    c_out[SHIndex(13, 5)] += c_in[SHIndex(13, 7)] * 0.16916849025752;
    c_out[SHIndex(13, 5)] += c_in[SHIndex(13, 9)] * -0.30447947908103;
    c_out[SHIndex(13, 5)] += c_in[SHIndex(13, 11)] * -0.62629484514798;
    c_out[SHIndex(13, 5)] += c_in[SHIndex(13, 13)] * 0.30515380780268;
    c_out[SHIndex(13, 6)] += c_in[SHIndex(13, -13)] * 0.19800993746086;
    c_out[SHIndex(13, 6)] += c_in[SHIndex(13, -11)] * -0.64803329186808;
    c_out[SHIndex(13, 6)] += c_in[SHIndex(13, -9)] * 0.087450067714449;
    c_out[SHIndex(13, 6)] += c_in[SHIndex(13, -7)] * 0.49108084137057;
    c_out[SHIndex(13, 6)] += c_in[SHIndex(13, -5)] * 0.34012635856225;
    c_out[SHIndex(13, 6)] += c_in[SHIndex(13, -3)] * -0.083251595146774;
    c_out[SHIndex(13, 6)] += c_in[SHIndex(13, -1)] * -0.41160690265054;
    c_out[SHIndex(13, 7)] += c_in[SHIndex(13, 1)] * 0.24350992753551;
    c_out[SHIndex(13, 7)] += c_in[SHIndex(13, 3)] * 0.47610564299334;
    c_out[SHIndex(13, 7)] += c_in[SHIndex(13, 5)] * 0.16916849025752;
    c_out[SHIndex(13, 7)] += c_in[SHIndex(13, 7)] * -0.40478515625;
    c_out[SHIndex(13, 7)] += c_in[SHIndex(13, 9)] * -0.45029618794257;
    c_out[SHIndex(13, 7)] += c_in[SHIndex(13, 11)] * 0.552329518352;
    c_out[SHIndex(13, 7)] += c_in[SHIndex(13, 13)] * -0.1171442587865;
    c_out[SHIndex(13, 8)] += c_in[SHIndex(13, -13)] * -0.06261624017238;
    c_out[SHIndex(13, 8)] += c_in[SHIndex(13, -11)] * 0.39943226817081;
    c_out[SHIndex(13, 8)] += c_in[SHIndex(13, -9)] * -0.62989984533657;
    c_out[SHIndex(13, 8)] += c_in[SHIndex(13, -7)] * 0.0127888680212;
    c_out[SHIndex(13, 8)] += c_in[SHIndex(13, -5)] * 0.50923193110086;
    c_out[SHIndex(13, 8)] += c_in[SHIndex(13, -3)] * 0.16673428435382;
    c_out[SHIndex(13, 8)] += c_in[SHIndex(13, -1)] * -0.39048459390687;
    c_out[SHIndex(13, 9)] += c_in[SHIndex(13, 1)] * 0.33508120677026;
    c_out[SHIndex(13, 9)] += c_in[SHIndex(13, 3)] * 0.41417145645063;
    c_out[SHIndex(13, 9)] += c_in[SHIndex(13, 5)] * -0.30447947908103;
    c_out[SHIndex(13, 9)] += c_in[SHIndex(13, 7)] * -0.45029618794257;
    c_out[SHIndex(13, 9)] += c_in[SHIndex(13, 9)] * 0.59912109375;
    c_out[SHIndex(13, 9)] += c_in[SHIndex(13, 11)] * -0.24672049987443;
    c_out[SHIndex(13, 9)] += c_in[SHIndex(13, 13)] * 0.029851121241791;
    c_out[SHIndex(13, 10)] += c_in[SHIndex(13, -13)] * 0.012448778109357;
    c_out[SHIndex(13, 10)] += c_in[SHIndex(13, -11)] * -0.12912985164247;
    c_out[SHIndex(13, 10)] += c_in[SHIndex(13, -9)] * 0.43555891764461;
    c_out[SHIndex(13, 10)] += c_in[SHIndex(13, -7)] * -0.64326862003139;
    c_out[SHIndex(13, 10)] += c_in[SHIndex(13, -5)] * 0.23938234163585;
    c_out[SHIndex(13, 10)] += c_in[SHIndex(13, -3)] * 0.44139922820051;
    c_out[SHIndex(13, 10)] += c_in[SHIndex(13, -1)] * -0.357109558733;
    c_out[SHIndex(13, 11)] += c_in[SHIndex(13, 1)] * 0.46294341611217;
    c_out[SHIndex(13, 11)] += c_in[SHIndex(13, 3)] * 0.15605819373554;
    c_out[SHIndex(13, 11)] += c_in[SHIndex(13, 5)] * -0.62629484514798;
    c_out[SHIndex(13, 11)] += c_in[SHIndex(13, 7)] * 0.552329518352;
    c_out[SHIndex(13, 11)] += c_in[SHIndex(13, 9)] * -0.24672049987443;
    c_out[SHIndex(13, 11)] += c_in[SHIndex(13, 11)] * 0.055908203125;
    c_out[SHIndex(13, 11)] += c_in[SHIndex(13, 13)] * -0.0044013077093066;
    c_out[SHIndex(13, 12)] += c_in[SHIndex(13, -13)] * -0.0012448778109357;
    c_out[SHIndex(13, 12)] += c_in[SHIndex(13, -11)] * 0.018989684065068;
    c_out[SHIndex(13, 12)] += c_in[SHIndex(13, -9)] * -0.10537715749466;
    c_out[SHIndex(13, 12)] += c_in[SHIndex(13, -7)] * 0.3216343100157;
    c_out[SHIndex(13, 12)] += c_in[SHIndex(13, -5)] * -0.59845585408961;
    c_out[SHIndex(13, 12)] += c_in[SHIndex(13, -3)] * 0.66209884230076;
    c_out[SHIndex(13, 12)] += c_in[SHIndex(13, -1)] * -0.29759129894417;
    c_out[SHIndex(13, 13)] += c_in[SHIndex(13, 1)] * 0.75871192019586;
    c_out[SHIndex(13, 13)] += c_in[SHIndex(13, 3)] * -0.5626758194698;
    c_out[SHIndex(13, 13)] += c_in[SHIndex(13, 5)] * 0.30515380780268;
    c_out[SHIndex(13, 13)] += c_in[SHIndex(13, 7)] * -0.1171442587865;
    c_out[SHIndex(13, 13)] += c_in[SHIndex(13, 9)] * 0.029851121241791;
    c_out[SHIndex(13, 13)] += c_in[SHIndex(13, 11)] * -0.0044013077093066;
    c_out[SHIndex(13, 13)] += c_in[SHIndex(13, 13)] * 0.000244140625;
    if (lmax == 13) return;
    
    c_out[SHIndex(14, -14)] += c_in[SHIndex(14, 1)] * -0.74694823221281;
    c_out[SHIndex(14, -14)] += c_in[SHIndex(14, 3)] * 0.56567677845719;
    c_out[SHIndex(14, -14)] += c_in[SHIndex(14, 5)] * -0.32081276509794;
    c_out[SHIndex(14, -14)] += c_in[SHIndex(14, 7)] * 0.13282910411489;
    c_out[SHIndex(14, -14)] += c_in[SHIndex(14, 9)] * -0.038268604914882;
    c_out[SHIndex(14, -14)] += c_in[SHIndex(14, 11)] * 0.006986859385378;
    c_out[SHIndex(14, -14)] += c_in[SHIndex(14, 13)] * -0.00064593537867788;
    c_out[SHIndex(14, -13)] += c_in[SHIndex(14, -13)] * 0.003173828125;
    c_out[SHIndex(14, -13)] += c_in[SHIndex(14, -11)] * -0.029048630881424;
    c_out[SHIndex(14, -13)] += c_in[SHIndex(14, -9)] * 0.13017755780507;
    c_out[SHIndex(14, -13)] += c_in[SHIndex(14, -7)] * -0.3514327763595;
    c_out[SHIndex(14, -13)] += c_in[SHIndex(14, -5)] * 0.60627913847438;
    c_out[SHIndex(14, -13)] += c_in[SHIndex(14, -3)] * -0.64141717638938;
    c_out[SHIndex(14, -13)] += c_in[SHIndex(14, -1)] * 0.28231989495349;
    c_out[SHIndex(14, -12)] += c_in[SHIndex(14, 1)] * -0.46102645791479;
    c_out[SHIndex(14, -12)] += c_in[SHIndex(14, 3)] * -0.11638109588228;
    c_out[SHIndex(14, -12)] += c_in[SHIndex(14, 5)] * 0.59402981238257;
    c_out[SHIndex(14, -12)] += c_in[SHIndex(14, 7)] * -0.57388732064693;
    c_out[SHIndex(14, -12)] += c_in[SHIndex(14, 9)] * 0.29131204754594;
    c_out[SHIndex(14, -12)] += c_in[SHIndex(14, 11)] * -0.081935281474721;
    c_out[SHIndex(14, -12)] += c_in[SHIndex(14, 13)] * 0.010764359221215;
    c_out[SHIndex(14, -11)] += c_in[SHIndex(14, -13)] * -0.029048630881424;
    c_out[SHIndex(14, -11)] += c_in[SHIndex(14, -11)] * 0.179931640625;
    c_out[SHIndex(14, -11)] += c_in[SHIndex(14, -9)] * -0.48540841888275;
    c_out[SHIndex(14, -11)] += c_in[SHIndex(14, -7)] * 0.61730946168753;
    c_out[SHIndex(14, -11)] += c_in[SHIndex(14, -5)] * -0.1681515785395;
    c_out[SHIndex(14, -11)] += c_in[SHIndex(14, -3)] * -0.45462596516248;
    c_out[SHIndex(14, -11)] += c_in[SHIndex(14, -1)] * 0.33930628577947;
    c_out[SHIndex(14, -10)] += c_in[SHIndex(14, 1)] * -0.33930628577947;
    c_out[SHIndex(14, -10)] += c_in[SHIndex(14, 3)] * -0.37556057991684;
    c_out[SHIndex(14, -10)] += c_in[SHIndex(14, 5)] * 0.35648134650374;
    c_out[SHIndex(14, -10)] += c_in[SHIndex(14, 7)] * 0.35738968834541;
    c_out[SHIndex(14, -10)] += c_in[SHIndex(14, 9)] * -0.61645531984834;
    c_out[SHIndex(14, -10)] += c_in[SHIndex(14, 11)] * 0.323486328125;
    c_out[SHIndex(14, -10)] += c_in[SHIndex(14, 13)] * -0.066019615639599;
    c_out[SHIndex(14, -9)] += c_in[SHIndex(14, -13)] * 0.13017755780507;
    c_out[SHIndex(14, -9)] += c_in[SHIndex(14, -11)] * -0.48540841888275;
    c_out[SHIndex(14, -9)] += c_in[SHIndex(14, -9)] * 0.56982421875;
    c_out[SHIndex(14, -9)] += c_in[SHIndex(14, -7)] * 0.10677294225083;
    c_out[SHIndex(14, -9)] += c_in[SHIndex(14, -5)] * -0.47892214576017;
    c_out[SHIndex(14, -9)] += c_in[SHIndex(14, -3)] * -0.20931182591491;
    c_out[SHIndex(14, -9)] += c_in[SHIndex(14, -1)] * 0.37169141324942;
    c_out[SHIndex(14, -8)] += c_in[SHIndex(14, 1)] * -0.25312379436441;
    c_out[SHIndex(14, -8)] += c_in[SHIndex(14, 3)] * -0.45711888182303;
    c_out[SHIndex(14, -8)] += c_in[SHIndex(14, 5)] * -0.075265034640616;
    c_out[SHIndex(14, -8)] += c_in[SHIndex(14, 7)] * 0.46051523295669;
    c_out[SHIndex(14, -8)] += c_in[SHIndex(14, 9)] * 0.29827230784789;
    c_out[SHIndex(14, -8)] += c_in[SHIndex(14, 11)] * -0.60740303427372;
    c_out[SHIndex(14, -8)] += c_in[SHIndex(14, 13)] * 0.22655408120134;
    c_out[SHIndex(14, -7)] += c_in[SHIndex(14, -13)] * -0.3514327763595;
    c_out[SHIndex(14, -7)] += c_in[SHIndex(14, -11)] * 0.61730946168753;
    c_out[SHIndex(14, -7)] += c_in[SHIndex(14, -9)] * 0.10677294225083;
    c_out[SHIndex(14, -7)] += c_in[SHIndex(14, -7)] * -0.41845703125;
    c_out[SHIndex(14, -7)] += c_in[SHIndex(14, -5)] * -0.39271064912769;
    c_out[SHIndex(14, -7)] += c_in[SHIndex(14, -3)] * 0.022873774761468;
    c_out[SHIndex(14, -7)] += c_in[SHIndex(14, -1)] * 0.39264796001216;
    c_out[SHIndex(14, -6)] += c_in[SHIndex(14, 1)] * -0.18176068675697;
    c_out[SHIndex(14, -6)] += c_in[SHIndex(14, 3)] * -0.42706950870872;
    c_out[SHIndex(14, -6)] += c_in[SHIndex(14, 5)] * -0.3750436280621;
    c_out[SHIndex(14, -6)] += c_in[SHIndex(14, 7)] * -0.011075484592006;
    c_out[SHIndex(14, -6)] += c_in[SHIndex(14, 9)] * 0.43385279709812;
    c_out[SHIndex(14, -6)] += c_in[SHIndex(14, 11)] * 0.46623796657139;
    c_out[SHIndex(14, -6)] += c_in[SHIndex(14, 13)] * -0.48804529210874;
    c_out[SHIndex(14, -5)] += c_in[SHIndex(14, -13)] * 0.60627913847438;
    c_out[SHIndex(14, -5)] += c_in[SHIndex(14, -11)] * -0.1681515785395;
    c_out[SHIndex(14, -5)] += c_in[SHIndex(14, -9)] * -0.47892214576017;
    c_out[SHIndex(14, -5)] += c_in[SHIndex(14, -7)] * -0.39271064912769;
    c_out[SHIndex(14, -5)] += c_in[SHIndex(14, -5)] * -0.099853515625;
    c_out[SHIndex(14, -5)] += c_in[SHIndex(14, -3)] * 0.21308945636309;
    c_out[SHIndex(14, -5)] += c_in[SHIndex(14, -1)] * 0.40642925122564;
    c_out[SHIndex(14, -4)] += c_in[SHIndex(14, 1)] * -0.11794190764568;
    c_out[SHIndex(14, -4)] += c_in[SHIndex(14, 3)] * -0.32292413190191;
    c_out[SHIndex(14, -4)] += c_in[SHIndex(14, 5)] * -0.43748201605755;
    c_out[SHIndex(14, -4)] += c_in[SHIndex(14, 7)] * -0.40964323385176;
    c_out[SHIndex(14, -4)] += c_in[SHIndex(14, 9)] * -0.20312265248043;
    c_out[SHIndex(14, -4)] += c_in[SHIndex(14, 11)] * 0.18542476450635;
    c_out[SHIndex(14, -4)] += c_in[SHIndex(14, 13)] * 0.66855849616849;
    c_out[SHIndex(14, -3)] += c_in[SHIndex(14, -13)] * -0.64141717638938;
    c_out[SHIndex(14, -3)] += c_in[SHIndex(14, -11)] * -0.45462596516248;
    c_out[SHIndex(14, -3)] += c_in[SHIndex(14, -9)] * -0.20931182591491;
    c_out[SHIndex(14, -3)] += c_in[SHIndex(14, -7)] * 0.022873774761468;
    c_out[SHIndex(14, -3)] += c_in[SHIndex(14, -5)] * 0.21308945636309;
    c_out[SHIndex(14, -3)] += c_in[SHIndex(14, -3)] * 0.346435546875;
    c_out[SHIndex(14, -3)] += c_in[SHIndex(14, -1)] * 0.41489743677369;
    c_out[SHIndex(14, -2)] += c_in[SHIndex(14, 1)] * -0.058097261762847;
    c_out[SHIndex(14, -2)] += c_in[SHIndex(14, 3)] * -0.17260776719378;
    c_out[SHIndex(14, -2)] += c_in[SHIndex(14, 5)] * -0.28180798175817;
    c_out[SHIndex(14, -2)] += c_in[SHIndex(14, 7)] * -0.38115332581581;
    c_out[SHIndex(14, -2)] += c_in[SHIndex(14, 9)] * -0.46389892469554;
    c_out[SHIndex(14, -2)] += c_in[SHIndex(14, 11)] * -0.51758647408139;
    c_out[SHIndex(14, -2)] += c_in[SHIndex(14, 13)] * -0.5089594286692;
    c_out[SHIndex(14, -1)] += c_in[SHIndex(14, -13)] * 0.28231989495349;
    c_out[SHIndex(14, -1)] += c_in[SHIndex(14, -11)] * 0.33930628577947;
    c_out[SHIndex(14, -1)] += c_in[SHIndex(14, -9)] * 0.37169141324942;
    c_out[SHIndex(14, -1)] += c_in[SHIndex(14, -7)] * 0.39264796001216;
    c_out[SHIndex(14, -1)] += c_in[SHIndex(14, -5)] * 0.40642925122564;
    c_out[SHIndex(14, -1)] += c_in[SHIndex(14, -3)] * 0.41489743677369;
    c_out[SHIndex(14, -1)] += c_in[SHIndex(14, -1)] * 0.4189453125;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 0)] * -0.20947265625;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 2)] * -0.29765989046048;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 4)] * -0.30213623022215;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 6)] * -0.31041546806428;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 8)] * -0.32421838231813;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 10)] * -0.34768548049628;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 12)] * -0.39367628467147;
    c_out[SHIndex(14, 0)] += c_in[SHIndex(14, 14)] * -0.54671012572894;
    c_out[SHIndex(14, 1)] += c_in[SHIndex(14, -14)] * -0.74694823221281;
    c_out[SHIndex(14, 1)] += c_in[SHIndex(14, -12)] * -0.46102645791479;
    c_out[SHIndex(14, 1)] += c_in[SHIndex(14, -10)] * -0.33930628577947;
    c_out[SHIndex(14, 1)] += c_in[SHIndex(14, -8)] * -0.25312379436441;
    c_out[SHIndex(14, 1)] += c_in[SHIndex(14, -6)] * -0.18176068675697;
    c_out[SHIndex(14, 1)] += c_in[SHIndex(14, -4)] * -0.11794190764568;
    c_out[SHIndex(14, 1)] += c_in[SHIndex(14, -2)] * -0.058097261762847;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 0)] * -0.29765989046048;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 2)] * -0.4068603515625;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 4)] * -0.36391209618543;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 6)] * -0.28986522975411;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 8)] * -0.17989751732912;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 10)] * -0.023526657912791;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 12)] * 0.20778181666967;
    c_out[SHIndex(14, 2)] += c_in[SHIndex(14, 14)] * 0.67329003784011;
    c_out[SHIndex(14, 3)] += c_in[SHIndex(14, -14)] * 0.56567677845719;
    c_out[SHIndex(14, 3)] += c_in[SHIndex(14, -12)] * -0.11638109588228;
    c_out[SHIndex(14, 3)] += c_in[SHIndex(14, -10)] * -0.37556057991684;
    c_out[SHIndex(14, 3)] += c_in[SHIndex(14, -8)] * -0.45711888182303;
    c_out[SHIndex(14, 3)] += c_in[SHIndex(14, -6)] * -0.42706950870872;
    c_out[SHIndex(14, 3)] += c_in[SHIndex(14, -4)] * -0.32292413190191;
    c_out[SHIndex(14, 3)] += c_in[SHIndex(14, -2)] * -0.17260776719378;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 0)] * -0.30213623022215;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 2)] * -0.36391209618543;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 4)] * -0.185791015625;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 6)] * 0.069981750766127;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 8)] * 0.33717349471455;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 10)] * 0.50991810239247;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 12)] * 0.40940679470382;
    c_out[SHIndex(14, 4)] += c_in[SHIndex(14, 14)] * -0.44220987944029;
    c_out[SHIndex(14, 5)] += c_in[SHIndex(14, -14)] * -0.32081276509794;
    c_out[SHIndex(14, 5)] += c_in[SHIndex(14, -12)] * 0.59402981238257;
    c_out[SHIndex(14, 5)] += c_in[SHIndex(14, -10)] * 0.35648134650374;
    c_out[SHIndex(14, 5)] += c_in[SHIndex(14, -8)] * -0.075265034640616;
    c_out[SHIndex(14, 5)] += c_in[SHIndex(14, -6)] * -0.3750436280621;
    c_out[SHIndex(14, 5)] += c_in[SHIndex(14, -4)] * -0.43748201605755;
    c_out[SHIndex(14, 5)] += c_in[SHIndex(14, -2)] * -0.28180798175817;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 0)] * -0.31041546806428;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 2)] * -0.28986522975411;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 4)] * 0.069981750766127;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 6)] * 0.4073486328125;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 8)] * 0.43617162672689;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 10)] * -0.0075199672027644;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 12)] * -0.64200733655529;
    c_out[SHIndex(14, 6)] += c_in[SHIndex(14, 14)] * 0.2152077452426;
    c_out[SHIndex(14, 7)] += c_in[SHIndex(14, -14)] * 0.13282910411489;
    c_out[SHIndex(14, 7)] += c_in[SHIndex(14, -12)] * -0.57388732064693;
    c_out[SHIndex(14, 7)] += c_in[SHIndex(14, -10)] * 0.35738968834541;
    c_out[SHIndex(14, 7)] += c_in[SHIndex(14, -8)] * 0.46051523295669;
    c_out[SHIndex(14, 7)] += c_in[SHIndex(14, -6)] * -0.011075484592006;
    c_out[SHIndex(14, 7)] += c_in[SHIndex(14, -4)] * -0.40964323385176;
    c_out[SHIndex(14, 7)] += c_in[SHIndex(14, -2)] * -0.38115332581581;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 0)] * -0.32421838231813;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 2)] * -0.17989751732912;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 4)] * 0.33717349471455;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 6)] * 0.43617162672689;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 8)] * -0.13818359375;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 10)] * -0.58383998553034;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 12)] * 0.43932900265328;
    c_out[SHIndex(14, 8)] += c_in[SHIndex(14, 14)] * -0.074925719670686;
    c_out[SHIndex(14, 9)] += c_in[SHIndex(14, -14)] * -0.038268604914882;
    c_out[SHIndex(14, 9)] += c_in[SHIndex(14, -12)] * 0.29131204754594;
    c_out[SHIndex(14, 9)] += c_in[SHIndex(14, -10)] * -0.61645531984834;
    c_out[SHIndex(14, 9)] += c_in[SHIndex(14, -8)] * 0.29827230784789;
    c_out[SHIndex(14, 9)] += c_in[SHIndex(14, -6)] * 0.43385279709812;
    c_out[SHIndex(14, 9)] += c_in[SHIndex(14, -4)] * -0.20312265248043;
    c_out[SHIndex(14, 9)] += c_in[SHIndex(14, -2)] * -0.46389892469554;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 0)] * -0.34768548049628;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 2)] * -0.023526657912791;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 4)] * 0.50991810239247;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 6)] * -0.0075199672027644;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 8)] * -0.58383998553034;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 10)] * 0.4993896484375;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 12)] * -0.16710485037608;
    c_out[SHIndex(14, 10)] += c_in[SHIndex(14, 14)] * 0.017467148463445;
    c_out[SHIndex(14, 11)] += c_in[SHIndex(14, -14)] * 0.006986859385378;
    c_out[SHIndex(14, 11)] += c_in[SHIndex(14, -12)] * -0.081935281474721;
    c_out[SHIndex(14, 11)] += c_in[SHIndex(14, -10)] * 0.323486328125;
    c_out[SHIndex(14, 11)] += c_in[SHIndex(14, -8)] * -0.60740303427372;
    c_out[SHIndex(14, 11)] += c_in[SHIndex(14, -6)] * 0.46623796657139;
    c_out[SHIndex(14, 11)] += c_in[SHIndex(14, -4)] * 0.18542476450635;
    c_out[SHIndex(14, 11)] += c_in[SHIndex(14, -2)] * -0.51758647408139;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 0)] * -0.39367628467147;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 2)] * 0.20778181666967;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 4)] * 0.40940679470382;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 6)] * -0.64200733655529;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 8)] * 0.43932900265328;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 10)] * -0.16710485037608;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 12)] * 0.033447265625;
    c_out[SHIndex(14, 12)] += c_in[SHIndex(14, 14)] * -0.0023733181268583;
    c_out[SHIndex(14, 13)] += c_in[SHIndex(14, -14)] * -0.00064593537867788;
    c_out[SHIndex(14, 13)] += c_in[SHIndex(14, -12)] * 0.010764359221215;
    c_out[SHIndex(14, 13)] += c_in[SHIndex(14, -10)] * -0.066019615639599;
    c_out[SHIndex(14, 13)] += c_in[SHIndex(14, -8)] * 0.22655408120134;
    c_out[SHIndex(14, 13)] += c_in[SHIndex(14, -6)] * -0.48804529210874;
    c_out[SHIndex(14, 13)] += c_in[SHIndex(14, -4)] * 0.66855849616849;
    c_out[SHIndex(14, 13)] += c_in[SHIndex(14, -2)] * -0.5089594286692;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 0)] * -0.54671012572894;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 2)] * 0.67329003784011;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 4)] * -0.44220987944029;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 6)] * 0.2152077452426;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 8)] * -0.074925719670686;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 10)] * 0.017467148463445;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 12)] * -0.0023733181268583;
    c_out[SHIndex(14, 14)] += c_in[SHIndex(14, 14)] * 0.0001220703125;
    Assert(lmax == 14);
}


#endif
