/* Pango
 * pango-xsl-attributes.c: PangoAttribute types needed for XSL support
 *
 * Copyright (C) 1999 Red Hat Software
 * Copyright (C) 2000 Tor Lillqvist
 * Copyright (C) 2001, 2002 Sun Microsystems
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	 See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "pango-xsl-attributes.h"

typedef struct _PangoBaselineTable PangoBaselineTable;
typedef struct _PangoAttrBaseline  PangoAttrBaseline;

struct _PangoAttrBaseline
{
  PangoAttribute attr;
  PangoBaseline baseline;
  PangoFontDescription *desc;
};

typedef struct _PangoAttrPointer PangoAttrPointer;

struct _PangoAttrPointer
{
  PangoAttribute attr;
  gpointer pointer;
};

struct _PangoBaselineTable
{
  gint units_per_EM;

  gint alphabetic;
  gint ideographic;
  gint hanging;
  gint mathematical;
  gint central;
  gint middle;
  gint text_before_edge;
  gint text_after_edge;
};

static PangoAttribute *pango_attr_int_new    (const PangoAttrClass *klass,
                                              int                   value);
static PangoAttribute *pango_attr_float_new  (const PangoAttrClass *klass,
                                              double                value);
static PangoAttribute *
pango_attr_int_copy (const PangoAttribute *attr)
{
  const PangoAttrInt *int_attr = (PangoAttrInt *)attr;
  
  return pango_attr_int_new (attr->klass, int_attr->value);
}

static void
pango_attr_int_destroy (PangoAttribute *attr)
{
  g_free (attr);
}

static gboolean
pango_attr_int_equal (const PangoAttribute *attr1,
		      const PangoAttribute *attr2)
{
  const PangoAttrInt *int_attr1 = (const PangoAttrInt *)attr1;
  const PangoAttrInt *int_attr2 = (const PangoAttrInt *)attr2;
  
  return (int_attr1->value == int_attr2->value);
}

static PangoAttribute *
pango_attr_int_new (const PangoAttrClass *klass,
		    int                   value)
{
  PangoAttrInt *result = g_new (PangoAttrInt, 1);
  result->attr.klass = klass;
  result->value = value;

  return (PangoAttribute *)result;
}

static PangoAttribute *
pango_attr_float_copy (const PangoAttribute *attr)
{
  const PangoAttrFloat *float_attr = (PangoAttrFloat *)attr;
  
  return pango_attr_float_new (attr->klass, float_attr->value);
}

static void
pango_attr_float_destroy (PangoAttribute *attr)
{
  g_free (attr);
}

static gboolean
pango_attr_float_equal (const PangoAttribute *attr1,
			const PangoAttribute *attr2)
{
  const PangoAttrFloat *float_attr1 = (const PangoAttrFloat *)attr1;
  const PangoAttrFloat *float_attr2 = (const PangoAttrFloat *)attr2;
  
  return (float_attr1->value == float_attr2->value);
}

static PangoAttribute*
pango_attr_float_new  (const PangoAttrClass *klass,
                       double                value)
{
  PangoAttrFloat *result = g_new (PangoAttrFloat, 1);
  result->attr.klass = klass;
  result->value = value;

  return (PangoAttribute *)result;
}

/**
 * pango_attr_alignment_adjust_get_type:
 * 
 * Register the 'alignment-adjust' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_alignment_adjust_get_type (void)
{
  static PangoAttrType attr_alignment_adjust = PANGO_ATTR_INVALID;

  if (attr_alignment_adjust == PANGO_ATTR_INVALID)
    {
      attr_alignment_adjust =
	pango_attr_type_register ("alignment-adjust");
    }

  return attr_alignment_adjust;
}

/**
 * pango_attr_alignment_adjust_new:
 * @alignment_adjust: the alignment-adjust value
 * 
 * Create a new alignment-adjust attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_alignment_adjust_new (gint alignment_adjust)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_alignment_adjust_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, alignment_adjust);
}

/**
 * pango_attr_alignment_baseline_get_type:
 * 
 * Register the 'alignment-baseline' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_alignment_baseline_get_type (void)
{
  static PangoAttrType attr_alignment_baseline = PANGO_ATTR_INVALID;

  if (attr_alignment_baseline == PANGO_ATTR_INVALID)
    {
      attr_alignment_baseline =
	pango_attr_type_register ("alignment-baseline");
    }

  return attr_alignment_baseline;
}

/**
 * pango_attr_alignment_baseline_new:
 * @alignment_baseline: the alignment-baseline value
 * 
 * Create a new alignment-baseline attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_alignment_baseline_new (gint alignment_baseline)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_alignment_baseline_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, alignment_baseline);
}

/**
 * pango_attr_db_reset_size_get_type:
 * 
 * Register the 'dominant-baseline="reset-size"' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_db_reset_size_get_type (void)
{
  static PangoAttrType attr_db_reset_size = PANGO_ATTR_INVALID;

  if (attr_db_reset_size == PANGO_ATTR_INVALID)
    {
      attr_db_reset_size =
	pango_attr_type_register ("dominant-baseline=\"reset-size\"");
    }

  return attr_db_reset_size;
}

/**
 * pango_attr_db_reset_size_new:
 * @db_reset_size: the dominant baseline.
 * 
 * Create a new dominant-baseline attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_db_reset_size_new (gboolean db_reset_size)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_db_reset_size_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, (int) db_reset_size);
}

/**
 * pango_attr_dominant_baseline_get_type:
 * 
 * Register the '' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_dominant_baseline_get_type (void)
{
  static PangoAttrType attr_dominant_baseline = PANGO_ATTR_INVALID;

  if (attr_dominant_baseline == PANGO_ATTR_INVALID)
    {
      attr_dominant_baseline =
	pango_attr_type_register ("dominant-baseline");
    }

  return attr_dominant_baseline;
}

static PangoAttribute *
pango_attr_baseline_copy (const PangoAttribute *attr)
{
  const PangoAttrBaseline *baseline_attr = (PangoAttrBaseline *)attr;
  
  return pango_attr_dominant_baseline_new (baseline_attr->baseline, baseline_attr->desc);
}

static void
pango_attr_baseline_destroy (PangoAttribute *attr)
{
  PangoAttrBaseline *baseline = (PangoAttrBaseline *) attr;

  pango_font_description_free (baseline->desc);
  g_free (attr);
}

static gboolean
pango_attr_baseline_equal (const PangoAttribute *attr1,
			const PangoAttribute *attr2)
{
  const PangoAttrBaseline *baseline_attr1 = (const PangoAttrBaseline *)attr1;
  const PangoAttrBaseline *baseline_attr2 = (const PangoAttrBaseline *)attr2;
  
  return ((baseline_attr1->baseline == baseline_attr2->baseline) &&
	  pango_font_description_equal (baseline_attr1->desc,
					baseline_attr2->desc));
}

/**
 * pango_attr_dominant_baseline_new:
 * @dominant_baseline: the dominant baseline.
 * @font_desc:         the #PangoFontDescription.
 * 
 * Create a new dominant-baseline attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_dominant_baseline_new (PangoBaseline dominant_baseline,
				  PangoFontDescription *font_desc)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};
  PangoAttrBaseline *result;

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_dominant_baseline_get_type();
      klass.copy = pango_attr_baseline_copy;
      klass.destroy = pango_attr_baseline_destroy;
      klass.equal = pango_attr_baseline_equal;
    }

  result = g_new (PangoAttrBaseline, 1);
  result->attr.klass = &klass;
  result->baseline = dominant_baseline;
  result->desc = pango_font_description_copy (font_desc);

  return (PangoAttribute *) result;
}

/**
 * pango_attr_callback_get_type:
 * 
 * Register the 'callback' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_callback_get_type (void)
{
  static PangoAttrType attr_callback = PANGO_ATTR_INVALID;

  if (attr_callback == PANGO_ATTR_INVALID)
    {
      attr_callback =
	pango_attr_type_register ("callback");
    }

  return attr_callback;
}

static PangoAttribute *
pango_attr_pointer_new (const PangoAttrClass *klass,
			gpointer              pointer)
{
  PangoAttrPointer *result = g_new (PangoAttrPointer, 1);
  result->attr.klass = klass;
  result->pointer = pointer;

  return (PangoAttribute *)result;
}

static PangoAttribute *
pango_attr_pointer_copy (const PangoAttribute *attr)
{
  const PangoAttrPointer *pointer_attr = (PangoAttrPointer *)attr;

  return pango_attr_pointer_new (attr->klass,
				 pointer_attr->pointer);
}

static void
pango_attr_pointer_destroy (PangoAttribute *attr)
{
  g_free (attr);
}

static gpointer
pango_attr_pointer_get_pointer (const PangoAttribute *attr)
{
  const PangoAttrPointer *pointer_attr = (PangoAttrPointer *)attr;
  
  return pointer_attr->pointer;
}

static gboolean
pango_attr_pointer_equal (const PangoAttribute *attr1,
				  const PangoAttribute *attr2)
{
  const PangoAttrPointer *pointer_attr1 =
    (const PangoAttrPointer *)attr1;
  const PangoAttrPointer *pointer_attr2 =
    (const PangoAttrPointer *)attr2;
  
  return (pointer_attr1->pointer == pointer_attr2->pointer);
}

/**
 * pango_attr_callback_new:
 * @callback: the callback
 * 
 * Create a new callback attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_callback_new (gpointer callback)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_callback_get_type();
      klass.copy = pango_attr_pointer_copy;
      klass.destroy = pango_attr_pointer_destroy;
      klass.equal = pango_attr_pointer_equal;
    }

  return pango_attr_pointer_new (&klass, callback);
}

/**
 * pango_attr_callback_get_callback:
 * @attr: The 'callback' attribute.
 * 
 * Get the callback associated with @attr.
 * 
 * Return value: The callback.
 **/
gpointer
pango_attr_callback_get_callback (const PangoAttribute *attr)
{
  return pango_attr_pointer_get_pointer (attr);
}

/**
 * pango_attr_keep_together_within_line_get_type:
 * 
 * Register the 'keep-together.within-line' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_keep_together_within_line_get_type (void)
{
  static PangoAttrType attr_keep_together_within_line = PANGO_ATTR_INVALID;

  if (attr_keep_together_within_line == PANGO_ATTR_INVALID)
    {
      attr_keep_together_within_line =
	pango_attr_type_register ("keep-together.within-line");
    }

  return attr_keep_together_within_line;
}

/**
 * pango_attr_keep_together_within_line_new
 * @strength: the strength of the keep.
 * 
 * Create a new 'keep-together.within-line' attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_keep_together_within_line_new (gint strength)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_keep_together_within_line_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, strength);
}

/**
 * pango_attr_keep_with_next_within_line_get_type:
 * 
 * Register the 'keep-with-next.within-line' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_keep_with_next_within_line_get_type (void)
{
  static PangoAttrType attr_keep_with_next_within_line = PANGO_ATTR_INVALID;

  if (attr_keep_with_next_within_line == PANGO_ATTR_INVALID)
    {
      attr_keep_with_next_within_line =
	pango_attr_type_register ("keep-with-next.within-line");
    }

  return attr_keep_with_next_within_line;
}

/**
 * pango_attr_keep_with_next_within_line_new
 * @strength: the strength of the keep.
 * 
 * Create a new keep-with_next.within-line attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_keep_with_next_within_line_new (gint strength)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_keep_with_next_within_line_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, strength);
}

/**
 * pango_attr_keep_with_previous_within_line_get_type:
 * 
 * Register the 'keep-with-previous.within-line' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_keep_with_previous_within_line_get_type (void)
{
  static PangoAttrType attr_keep_with_previous_within_line = PANGO_ATTR_INVALID;

  if (attr_keep_with_previous_within_line == PANGO_ATTR_INVALID)
    {
      attr_keep_with_previous_within_line =
	pango_attr_type_register ("keep-with-previous.within-line");
    }

  return attr_keep_with_previous_within_line;
}

/**
 * pango_attr_keep_with_previous_within_line_new
 * @strength: the strength of the keep.
 * 
 * Create a new keep-with_previous.within-line attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_keep_with_previous_within_line_new (gint strength)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_keep_with_previous_within_line_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, strength);
}

/**
 * pango_attr_line_height_get_type:
 * 
 * Register the 'line_height' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_line_height_get_type (void)
{
  static PangoAttrType attr_line_height = PANGO_ATTR_INVALID;

  if (attr_line_height == PANGO_ATTR_INVALID)
    {
      attr_line_height =
	pango_attr_type_register ("line-height");
    }

  return attr_line_height;
}

/**
 * pango_attr_line_height_new:
 * @line_height: the line-height value
 * 
 * Create a new line-height attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_line_height_new (gint line_height)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_line_height_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, line_height);
}

/**
 * pango_attr_line_stacking_strategy_get_type:
 * 
 * Register the 'line-stacking-strategy' PangoAttribute type.
 * 
 * Return value: The new #PangoAttrType.
 **/
PangoAttrType
pango_attr_line_stacking_strategy_get_type (void)
{
  static PangoAttrType attr_line_stacking_strategy = PANGO_ATTR_INVALID;

  if (attr_line_stacking_strategy == PANGO_ATTR_INVALID)
    {
      attr_line_stacking_strategy =
	pango_attr_type_register ("line-stacking-strategy");
    }

  return attr_line_stacking_strategy;
}

/**
 * pango_attr_line_stacking_strategy_new:
 * @line_stacking_strategy: the line-stacking_strategy value
 * 
 * Create a new line-stacking_strategy attribute.
 * 
 * Return value: the new #PangoAttribute.
 **/
PangoAttribute *
pango_attr_line_stacking_strategy_new (PangoLineStackingStrategy line_stacking_strategy)
{
  static PangoAttrClass klass = {PANGO_ATTR_INVALID};

  if (klass.type == PANGO_ATTR_INVALID)
    {
      klass.type = pango_attr_line_stacking_strategy_get_type();
      klass.copy = pango_attr_int_copy;
      klass.destroy = pango_attr_int_destroy;
      klass.equal = pango_attr_int_equal;
    }

  return pango_attr_int_new (&klass, line_stacking_strategy);
}
