(**************************************************************************)
(*                    ocamllex/ocamlyac backend                           *)
(*                                                                        *)
(*                     Viktor Vafeiadis, MPI-SWS                          *)
(*                                                                        *)
(*  Copyright 2011                                                        *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*  notice, this list of conditions and the following disclaimer.         *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*  notice, this list of conditions and the following disclaimer in the   *)
(*  documentation and/or other materials provided with the distribution.  *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*  products derived from this software without specific prior written    *)
(*  permission.                                                           *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER  *)
(*  IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR       *)
(*  OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN   *)
(*  IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.                         *)
(**************************************************************************)

open Types;;

(**** ocamllex backend ************************************************************)

let get_terminals xd =
  let r_tm = Auxl.rule_of_ntr xd "terminals" in
  let i = ref 0 in
  let l = List.fold_left (fun l p ->
      match p.prod_es with
        [ Lang_terminal t ] when not p.prod_meta ->
          let t' =
            try
              String.concat ""
                (Auxl.option_map 
	           (function Hom_string s | Hom_terminal s -> Some s | _ -> None) 
                   (List.assoc "lex" p.prod_homs))
            with Not_found -> t in
          (t, (t', p.prod_name, p.prod_homs)) :: l
      | _ -> l) [] r_tm.rule_ps in
  List.fold_left (fun l r ->
    List.fold_left (fun l p -> if p.prod_meta then l else
      List.fold_left (fun l e -> match e with
        | Lang_terminal t when List.for_all (fun (x, _) -> x <> t) l ->
	  incr i; (t, (t, "Terminals" ^ string_of_int !i, [])) :: l
        | _ -> l) l p.prod_es
      ) l r.rule_ps
    ) l xd.xd_rs

let pp_lex_terminal fd (_, (t, s, _)) =
  if t <> "" then begin
    Printf.fprintf fd "\n| \"%s\"  { %s }" (String.escaped t) s
  end

let pp_lex_systemdefn m sd oi =
  match oi with
  | (o,is)::[] ->
      let _ = Auxl.filename_check m o in
      let fd = open_out o in
      let tl = get_terminals sd.syntax in
      Printf.fprintf fd "(* generated by Ott %s from: %s *)\n" Version.n sd.sources;
      output_string fd "{\n}\n\n";
      output_string fd "rule normal = parse";
      List.iter (pp_lex_terminal fd) tl;
      output_string fd "\n\n{\n}\n\n";
      close_out fd
  | _ -> Auxl.error "must specify only one output file in the lex backend.\n"

(**** ocamlyacc backend ***********************************************************)

let pp_yacc_terminal fd (_, (t, s, _)) =
  if t <> "" then begin
    Printf.fprintf fd "\n%%token %s" s
  end

let pp_yacc_precedences fd ts =
  let go_fun (_, (t, s, hs)) = 
    if t = "" then None else
    try match List.assoc "prec" hs with
      | [Hom_string x] ->
           Some (s, int_of_string x,
                 List.exists (fun (x,_) -> x="leftassoc") hs,
                 List.exists (fun (x,_) -> x="rightassoc") hs)
      | _ -> None
    with _ -> None
  in
  let rec group xs = match xs with
    | [] -> []
    | (s,c,l,r)::xs -> group1 [s] c l r xs
  and group1 b c l r xs = match xs with
   | (s',c',l',r')::xs when c'=c -> group1 (s'::b) c (l||l') (r||r') xs
   | _ -> (b,l,r) :: group xs in
  let ts = Auxl.option_map go_fun ts in
  let ts = List.sort (fun (_,c1,_,_) (_,c2,_,_) -> c1-c2) ts in
  let gs = group ts in
  List.iter (fun (b,l,r) -> 
    if l then output_string fd "\n%left"
    else if r then output_string fd "\n%right"
    else output_string fd "\n%nonassoc";
    List.iter (fun s -> output_char fd ' '; output_string fd s) b) gs

let pp_yacc_systemdefn m sd oi =
  match oi with
  | (o,is)::[] ->
      let _ = Auxl.filename_check m o in
      let fd = open_out o in
      let tl = get_terminals sd.syntax in
      Printf.fprintf fd "/* generated by Ott %s from: %s */\n" Version.n sd.sources;
      output_string fd "%{\n%}\n\n";
      List.iter (pp_yacc_terminal fd) tl;
      pp_yacc_precedences fd tl;
      output_string fd "\n%%\n";
      output_string fd "\n%%\n\n";
      close_out fd
  | _ -> Auxl.error "must specify only one output file in the yacc backend.\n"

