// Copyright 2008 Google Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

#import <Foundation/Foundation.h>
#import "KSAction.h"

@class KSFetcherFactory;
@class KSServer;
@class GTMHTTPFetcher;
@class KSFetcherFactory;

// The KSUpdateCheckAction class checks for updates.  Its input consists of:
// - a KSServer which encapsulates the server communication;
// - an array of tickets, represting local installs which may need
//   updating; and
// - an optional fetcher factory which encapsulates a
//   request/response (e.g. loading of a URL request).
// All tickets must point to the same server URL.
//
// If there are updates to perform, the KSUpdateCheckAction will take
// the KSActions generated by its KSServer, presumably
// KSUpdateActions, and place them on it's own action processor.  This
// class will finish (call the KSActionProcessor's
// finishedProcessing:successfully:) when done deciding what to do,
// which may be before its "child" KSUpdateActions have completed.
//
// This class is the only interface that KSUpdateEngine will use for
// server communication.  It is expected that KSUpdateEngine will create a
// KSUpdateCheckAction for each unique server URL seen in a ticket
// store.
@interface KSUpdateCheckAction : KSAction {
 @private
  KSFetcherFactory *fetcherFactory_;
  KSServer *server_;
  NSArray *tickets_;

  // Array of fetchers we have created.  Size of this array is the
  // number of outstanding requests.
  NSMutableArray *fetchers_;

  // If any request is unsucessful, this is set to NO.
  BOOL allSuccessful_;
  
  id delegate_;  // weak
}

+ (id)checkerWithServer:(KSServer *)server tickets:(NSArray *)tickets;

// Calls through to designated initializer with a new instance of
// GTMHTTPFetcher.  This is the standard initailizer that you will
// almost always want to use.  |tickets|, an array of KSTickets, are
// required to all point to the same server URL.
- (id)initWithServer:(KSServer *)server tickets:(NSArray *)tickets;

// Designated initializer. Can be useful because a mock fetcher can be
// "injected" into the instance when testing.
- (id)initWithFetcherFactory:(KSFetcherFactory *)fetcherFactory
                      server:(KSServer *)server
                     tickets:(NSArray *)tickets;

// Return the number of outstanding requests.  Servers may have
// more than one outstanding request each.
- (int)outstandingRequests;

// Getter and setter for this object's optional delegate. See
// KSUpdateCheckActionDelegateMethods below for details about the callbacks
// that the delegate can receive.
- (id)delegate;
- (void)setDelegate:(id)delegate;

@end
