/* MatMult test */

#include <oski/common.h>
#include <oski/matcreate.h>
#include <oski/matmult.h>
#include "util/array_util.h"

/* ------------------------------------------------- */

#define A0_M 4    /* # of rows */
#define A0_N 5    /* # of cols */
#define A0_NNZ 8  /* # of stored non-zeros */
static oski_index_t A0_ptr[A0_N+1] = {1, 2, 3, 4, 5, 9}; /* end row pointers for A0 */
static oski_index_t A0_ind[A0_NNZ] = {1, 1, 2, 2, 1, 2, 3, 4}; /* end column indices pointers for A0 */
static oski_value_t A0_val[A0_NNZ] = {MAKE_VAL_COMPLEX(4, -1), MAKE_VAL_COMPLEX(-10, -1), MAKE_VAL_COMPLEX(-9, 10), MAKE_VAL_COMPLEX(-1, -10), MAKE_VAL_COMPLEX(10, -2), MAKE_VAL_COMPLEX(8, -7), MAKE_VAL_COMPLEX(-3, -4), MAKE_VAL_COMPLEX(6, -2)}; /* end stored non-zero values for A0 */

#define x0_LEN 5   /* vector length */
#define x0_VW  3   /* vector width */
static oski_value_t x0_val[x0_LEN*x0_VW] = {MAKE_VAL_COMPLEX(9, -8), MAKE_VAL_COMPLEX(9, -7), MAKE_VAL_COMPLEX(-2, -8), MAKE_VAL_COMPLEX(-10, 10), MAKE_VAL_COMPLEX(3, 10), MAKE_VAL_COMPLEX(-2, 10), MAKE_VAL_COMPLEX(8, -1), MAKE_VAL_COMPLEX(0, -2), MAKE_VAL_COMPLEX(-4, -7), MAKE_VAL_COMPLEX(4, -4), MAKE_VAL_COMPLEX(10, -8), MAKE_VAL_COMPLEX(-6, -10), MAKE_VAL_COMPLEX(-5, -7), MAKE_VAL_COMPLEX(10, 8), MAKE_VAL_COMPLEX(6, 8)}; /* data */

#define y0_LEN 4   /* vector length */
#define y0_VW  3   /* vector width */
static oski_value_t y0_val[y0_LEN*y0_VW] = {MAKE_VAL_COMPLEX(-9, 6), MAKE_VAL_COMPLEX(1, 10), MAKE_VAL_COMPLEX(9, -5), MAKE_VAL_COMPLEX(1, -8), MAKE_VAL_COMPLEX(1, -10), MAKE_VAL_COMPLEX(10, 3), MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(-10, 3), MAKE_VAL_COMPLEX(-6, 0), MAKE_VAL_COMPLEX(-3, -3), MAKE_VAL_COMPLEX(-6, 9), MAKE_VAL_COMPLEX(-6, 8)}; /* data */

/*
 *  BEGIN TEST 0
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_0( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(182, -185), MAKE_VAL_COMPLEX(177, -166), MAKE_VAL_COMPLEX(-12, -74), MAKE_VAL_COMPLEX(-251, 90), MAKE_VAL_COMPLEX(-2, -6), MAKE_VAL_COMPLEX(54, 31), MAKE_VAL_COMPLEX(-15, 27), MAKE_VAL_COMPLEX(100, -28), MAKE_VAL_COMPLEX(62, -12), MAKE_VAL_COMPLEX(-38, 10), MAKE_VAL_COMPLEX(178, -8), MAKE_VAL_COMPLEX(148, 8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_0, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_0", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_0() */
/* END TEST 0 */

/*
 *  BEGIN TEST 1
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_1( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-128, 227), MAKE_VAL_COMPLEX(-119, 140), MAKE_VAL_COMPLEX(-36, 30), MAKE_VAL_COMPLEX(201, -80), MAKE_VAL_COMPLEX(-60, 20), MAKE_VAL_COMPLEX(-56, -97), MAKE_VAL_COMPLEX(19, -35), MAKE_VAL_COMPLEX(-62, 82), MAKE_VAL_COMPLEX(-50, 48), MAKE_VAL_COMPLEX(26, 14), MAKE_VAL_COMPLEX(-112, 26), MAKE_VAL_COMPLEX(-88, 12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_1, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_1", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_1() */
/* END TEST 1 */

/*
 *  BEGIN TEST 2
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_2( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-30, 111), MAKE_VAL_COMPLEX(78, 73), MAKE_VAL_COMPLEX(37, -103), MAKE_VAL_COMPLEX(-48, -71), MAKE_VAL_COMPLEX(-62, -87), MAKE_VAL_COMPLEX(101, -46), MAKE_VAL_COMPLEX(15, 1), MAKE_VAL_COMPLEX(-59, 94), MAKE_VAL_COMPLEX(-48, 42), MAKE_VAL_COMPLEX(-45, -3), MAKE_VAL_COMPLEX(15, 114), MAKE_VAL_COMPLEX(8, 106) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_2, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_2", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_2() */
/* END TEST 2 */

/*
 *  BEGIN TEST 3
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_3( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1138, -843), MAKE_VAL_COMPLEX(-241, -1052), MAKE_VAL_COMPLEX(-284, -808), MAKE_VAL_COMPLEX(-223, 1430), MAKE_VAL_COMPLEX(-499, -619), MAKE_VAL_COMPLEX(1048, -655), MAKE_VAL_COMPLEX(267, 165), MAKE_VAL_COMPLEX(-413, -75), MAKE_VAL_COMPLEX(-302, -152), MAKE_VAL_COMPLEX(-330, 268), MAKE_VAL_COMPLEX(417, -443), MAKE_VAL_COMPLEX(418, -258) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(2, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_3, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_3", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_3() */
/* END TEST 3 */

/*
 *  BEGIN TEST 4
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_4( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-104, -14), MAKE_VAL_COMPLEX(-53, -10), MAKE_VAL_COMPLEX(-21, 1), MAKE_VAL_COMPLEX(186, -1), MAKE_VAL_COMPLEX(64, -142), MAKE_VAL_COMPLEX(-21, 35), MAKE_VAL_COMPLEX(48, 28), MAKE_VAL_COMPLEX(77, 104), MAKE_VAL_COMPLEX(-161, 115), MAKE_VAL_COMPLEX(-117, -22), MAKE_VAL_COMPLEX(-177, -54), MAKE_VAL_COMPLEX(-38, -41), MAKE_VAL_COMPLEX(-188, 37), MAKE_VAL_COMPLEX(28, 22), MAKE_VAL_COMPLEX(223, -88) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_4, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_4", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_4() */
/* END TEST 4 */

/*
 *  BEGIN TEST 5
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_5( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(49, -66), MAKE_VAL_COMPLEX(6, -69), MAKE_VAL_COMPLEX(-45, 7), MAKE_VAL_COMPLEX(-116, 91), MAKE_VAL_COMPLEX(19, 128), MAKE_VAL_COMPLEX(99, -9), MAKE_VAL_COMPLEX(-48, -93), MAKE_VAL_COMPLEX(-93, -106), MAKE_VAL_COMPLEX(101, -90), MAKE_VAL_COMPLEX(89, -14), MAKE_VAL_COMPLEX(123, -34), MAKE_VAL_COMPLEX(-48, 79), MAKE_VAL_COMPLEX(127, -4), MAKE_VAL_COMPLEX(46, -94), MAKE_VAL_COMPLEX(-153, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_5, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_5", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_5() */
/* END TEST 5 */

/*
 *  BEGIN TEST 6
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_6( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-5, -60), MAKE_VAL_COMPLEX(-1, -57), MAKE_VAL_COMPLEX(-38, -16), MAKE_VAL_COMPLEX(10, 70), MAKE_VAL_COMPLEX(49, 18), MAKE_VAL_COMPLEX(34, 38), MAKE_VAL_COMPLEX(20, -35), MAKE_VAL_COMPLEX(-8, -6), MAKE_VAL_COMPLEX(-40, -5), MAKE_VAL_COMPLEX(-4, -28), MAKE_VAL_COMPLEX(-2, -64), MAKE_VAL_COMPLEX(-58, -6), MAKE_VAL_COMPLEX(-43, -1), MAKE_VAL_COMPLEX(62, -16), MAKE_VAL_COMPLEX(50, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_6, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_6", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_6() */
/* END TEST 6 */

/*
 *  BEGIN TEST 7
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_7( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-36, -124), MAKE_VAL_COMPLEX(-36, -75), MAKE_VAL_COMPLEX(-29, 3), MAKE_VAL_COMPLEX(61, 206), MAKE_VAL_COMPLEX(176, 32), MAKE_VAL_COMPLEX(9, -33), MAKE_VAL_COMPLEX(-48, 18), MAKE_VAL_COMPLEX(-112, 81), MAKE_VAL_COMPLEX(-135, -131), MAKE_VAL_COMPLEX(-2, -125), MAKE_VAL_COMPLEX(2, -201), MAKE_VAL_COMPLEX(13, 6), MAKE_VAL_COMPLEX(-55, -154), MAKE_VAL_COMPLEX(-10, -28), MAKE_VAL_COMPLEX(108, 183) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 1);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_7, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_7", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_7() */
/* END TEST 7 */

/*
 *  BEGIN TEST 8
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_8( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-135, 1), MAKE_VAL_COMPLEX(-93, 22), MAKE_VAL_COMPLEX(3, 41), MAKE_VAL_COMPLEX(194, -59), MAKE_VAL_COMPLEX(25, -167), MAKE_VAL_COMPLEX(-15, 21), MAKE_VAL_COMPLEX(-135, 19), MAKE_VAL_COMPLEX(-121, 90), MAKE_VAL_COMPLEX(-82, -68), MAKE_VAL_COMPLEX(35, 14), MAKE_VAL_COMPLEX(1, 0), MAKE_VAL_COMPLEX(-70, 183), MAKE_VAL_COMPLEX(-74, 27), MAKE_VAL_COMPLEX(30, -78), MAKE_VAL_COMPLEX(143, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_8, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_8", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_8() */
/* END TEST 8 */

/*
 *  BEGIN TEST 9
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_9( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(76, -13), MAKE_VAL_COMPLEX(38, -37), MAKE_VAL_COMPLEX(-29, -9), MAKE_VAL_COMPLEX(-124, 69), MAKE_VAL_COMPLEX(6, 125), MAKE_VAL_COMPLEX(61, -43), MAKE_VAL_COMPLEX(107, -48), MAKE_VAL_COMPLEX(113, -84), MAKE_VAL_COMPLEX(66, 105), MAKE_VAL_COMPLEX(-63, -18), MAKE_VAL_COMPLEX(-63, -16), MAKE_VAL_COMPLEX(48, -129), MAKE_VAL_COMPLEX(61, 14), MAKE_VAL_COMPLEX(-28, 14), MAKE_VAL_COMPLEX(-129, -46) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_9, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_9", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_9() */
/* END TEST 9 */

/*
 *  BEGIN TEST 10
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_10( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-56, -63), MAKE_VAL_COMPLEX(-49, -63), MAKE_VAL_COMPLEX(-56, 14), MAKE_VAL_COMPLEX(70, 70), MAKE_VAL_COMPLEX(70, -21), MAKE_VAL_COMPLEX(70, 14), MAKE_VAL_COMPLEX(-7, -56), MAKE_VAL_COMPLEX(-14, 0), MAKE_VAL_COMPLEX(-49, 28), MAKE_VAL_COMPLEX(-28, -28), MAKE_VAL_COMPLEX(-56, -70), MAKE_VAL_COMPLEX(-70, 42), MAKE_VAL_COMPLEX(-49, 35), MAKE_VAL_COMPLEX(56, -70), MAKE_VAL_COMPLEX(56, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_10, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_10", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_10() */
/* END TEST 10 */

/*
 *  BEGIN TEST 11
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_11( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(490, 38), MAKE_VAL_COMPLEX(272, -273), MAKE_VAL_COMPLEX(-307, -91), MAKE_VAL_COMPLEX(-1009, 270), MAKE_VAL_COMPLEX(-202, 926), MAKE_VAL_COMPLEX(767, -183), MAKE_VAL_COMPLEX(946, -170), MAKE_VAL_COMPLEX(1128, -323), MAKE_VAL_COMPLEX(135, 1129), MAKE_VAL_COMPLEX(-526, -369), MAKE_VAL_COMPLEX(-618, -417), MAKE_VAL_COMPLEX(661, -754), MAKE_VAL_COMPLEX(373, 400), MAKE_VAL_COMPLEX(-204, -152), MAKE_VAL_COMPLEX(-812, -879) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, -3);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_11, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_11", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_11() */
/* END TEST 11 */

/*
 *  BEGIN TEST 12
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_12( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(44, -275), MAKE_VAL_COMPLEX(-277, 20), MAKE_VAL_COMPLEX(9, 123), MAKE_VAL_COMPLEX(18, -8), MAKE_VAL_COMPLEX(203, -52), MAKE_VAL_COMPLEX(22, -6), MAKE_VAL_COMPLEX(-8, -58), MAKE_VAL_COMPLEX(72, -58), MAKE_VAL_COMPLEX(102, -108), MAKE_VAL_COMPLEX(146, 97), MAKE_VAL_COMPLEX(-46, -114), MAKE_VAL_COMPLEX(0, -28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_12, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_12", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_12() */
/* END TEST 12 */

/*
 *  BEGIN TEST 13
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_13( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-8, 212), MAKE_VAL_COMPLEX(241, -77), MAKE_VAL_COMPLEX(-48, -66), MAKE_VAL_COMPLEX(0, 59), MAKE_VAL_COMPLEX(-179, 115), MAKE_VAL_COMPLEX(-91, 18), MAKE_VAL_COMPLEX(-1, 55), MAKE_VAL_COMPLEX(-21, 10), MAKE_VAL_COMPLEX(-66, 90), MAKE_VAL_COMPLEX(-119, -88), MAKE_VAL_COMPLEX(55, 42), MAKE_VAL_COMPLEX(12, -38) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_13, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_13", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_13() */
/* END TEST 13 */

/*
 *  BEGIN TEST 14
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_14( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, -57), MAKE_VAL_COMPLEX(-52, -15), MAKE_VAL_COMPLEX(7, 55), MAKE_VAL_COMPLEX(38, 21), MAKE_VAL_COMPLEX(48, 25), MAKE_VAL_COMPLEX(-35, 44), MAKE_VAL_COMPLEX(-7, 3), MAKE_VAL_COMPLEX(5, -56), MAKE_VAL_COMPLEX(12, -30), MAKE_VAL_COMPLEX(21, -9), MAKE_VAL_COMPLEX(-33, -48), MAKE_VAL_COMPLEX(-28, -46) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_14, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_14", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_14() */
/* END TEST 14 */

/*
 *  BEGIN TEST 15
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_15( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-116, 1252), MAKE_VAL_COMPLEX(1048, -556), MAKE_VAL_COMPLEX(44, -228), MAKE_VAL_COMPLEX(378, 94), MAKE_VAL_COMPLEX(-528, 716), MAKE_VAL_COMPLEX(-234, 678), MAKE_VAL_COMPLEX(112, 396), MAKE_VAL_COMPLEX(-550, -70), MAKE_VAL_COMPLEX(-432, 504), MAKE_VAL_COMPLEX(-914, -482), MAKE_VAL_COMPLEX(-54, 298), MAKE_VAL_COMPLEX(-444, -132) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_15, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_15", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_15() */
/* END TEST 15 */

/*
 *  BEGIN TEST 16
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_16( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-3, 9), MAKE_VAL_COMPLEX(21, -62), MAKE_VAL_COMPLEX(-30, -18), MAKE_VAL_COMPLEX(66, -21), MAKE_VAL_COMPLEX(-11, 129), MAKE_VAL_COMPLEX(54, 36), MAKE_VAL_COMPLEX(-85, -83), MAKE_VAL_COMPLEX(-120, 67), MAKE_VAL_COMPLEX(45, -24), MAKE_VAL_COMPLEX(111, -32), MAKE_VAL_COMPLEX(50, -127), MAKE_VAL_COMPLEX(-45, 3), MAKE_VAL_COMPLEX(-72, 59), MAKE_VAL_COMPLEX(68, -93), MAKE_VAL_COMPLEX(-53, 90) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 0);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_16, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_16", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_16() */
/* END TEST 16 */

/*
 *  BEGIN TEST 17
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_17( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(108, -54), MAKE_VAL_COMPLEX(81, 26), MAKE_VAL_COMPLEX(36, -60), MAKE_VAL_COMPLEX(-186, 81), MAKE_VAL_COMPLEX(8, -30), MAKE_VAL_COMPLEX(-102, 48), MAKE_VAL_COMPLEX(160, 98), MAKE_VAL_COMPLEX(126, -85), MAKE_VAL_COMPLEX(-60, -51), MAKE_VAL_COMPLEX(-63, 8), MAKE_VAL_COMPLEX(64, 85), MAKE_VAL_COMPLEX(21, -111), MAKE_VAL_COMPLEX(48, -137), MAKE_VAL_COMPLEX(-2, 195), MAKE_VAL_COMPLEX(83, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_17, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_17", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_17() */
/* END TEST 17 */

/*
 *  BEGIN TEST 18
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_18( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-7, -26), MAKE_VAL_COMPLEX(-5, -25), MAKE_VAL_COMPLEX(-18, -4), MAKE_VAL_COMPLEX(10, 30), MAKE_VAL_COMPLEX(23, 4), MAKE_VAL_COMPLEX(18, 14), MAKE_VAL_COMPLEX(6, -17), MAKE_VAL_COMPLEX(-4, -2), MAKE_VAL_COMPLEX(-18, 1), MAKE_VAL_COMPLEX(-4, -12), MAKE_VAL_COMPLEX(-6, -28), MAKE_VAL_COMPLEX(-26, 2), MAKE_VAL_COMPLEX(-19, 3), MAKE_VAL_COMPLEX(26, -12), MAKE_VAL_COMPLEX(22, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_18, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_18", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_18() */
/* END TEST 18 */

/*
 *  BEGIN TEST 19
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_19( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-40, 380), MAKE_VAL_COMPLEX(-291, -117), MAKE_VAL_COMPLEX(-68, 204), MAKE_VAL_COMPLEX(285, -815), MAKE_VAL_COMPLEX(364, 528), MAKE_VAL_COMPLEX(334, -342), MAKE_VAL_COMPLEX(-1000, 160), MAKE_VAL_COMPLEX(-233, 979), MAKE_VAL_COMPLEX(305, -255), MAKE_VAL_COMPLEX(371, -563), MAKE_VAL_COMPLEX(-477, -549), MAKE_VAL_COMPLEX(82, 364), MAKE_VAL_COMPLEX(157, 729), MAKE_VAL_COMPLEX(-473, -821), MAKE_VAL_COMPLEX(-75, 635) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(5, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_19, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_19", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_19() */
/* END TEST 19 */

/*
 *  BEGIN TEST 20
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_20( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, 43), MAKE_VAL_COMPLEX(51, -10), MAKE_VAL_COMPLEX(6, -22), MAKE_VAL_COMPLEX(34, -99), MAKE_VAL_COMPLEX(-37, 123), MAKE_VAL_COMPLEX(18, -4), MAKE_VAL_COMPLEX(103, -69), MAKE_VAL_COMPLEX(-52, -129), MAKE_VAL_COMPLEX(21, 34), MAKE_VAL_COMPLEX(-81, 12), MAKE_VAL_COMPLEX(2, 129), MAKE_VAL_COMPLEX(67, -61), MAKE_VAL_COMPLEX(-126, 107), MAKE_VAL_COMPLEX(-6, 43), MAKE_VAL_COMPLEX(-159, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_20, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_20", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_20() */
/* END TEST 20 */

/*
 *  BEGIN TEST 21
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_21( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(28, -67), MAKE_VAL_COMPLEX(-24, -11), MAKE_VAL_COMPLEX(-12, -2), MAKE_VAL_COMPLEX(-64, 129), MAKE_VAL_COMPLEX(46, -93), MAKE_VAL_COMPLEX(-24, 34), MAKE_VAL_COMPLEX(-79, 66), MAKE_VAL_COMPLEX(52, 123), MAKE_VAL_COMPLEX(-33, -55), MAKE_VAL_COMPLEX(93, -24), MAKE_VAL_COMPLEX(28, -153), MAKE_VAL_COMPLEX(-85, 31), MAKE_VAL_COMPLEX(111, -128), MAKE_VAL_COMPLEX(36, -19), MAKE_VAL_COMPLEX(177, 64) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_21, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_21", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_21() */
/* END TEST 21 */

/*
 *  BEGIN TEST 22
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_22( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-11, -103), MAKE_VAL_COMPLEX(-4, -98), MAKE_VAL_COMPLEX(-66, -26), MAKE_VAL_COMPLEX(20, 120), MAKE_VAL_COMPLEX(85, 29), MAKE_VAL_COMPLEX(60, 64), MAKE_VAL_COMPLEX(33, -61), MAKE_VAL_COMPLEX(-14, -10), MAKE_VAL_COMPLEX(-69, -7), MAKE_VAL_COMPLEX(-8, -48), MAKE_VAL_COMPLEX(-6, -110), MAKE_VAL_COMPLEX(-100, -8), MAKE_VAL_COMPLEX(-74, 0), MAKE_VAL_COMPLEX(106, -30), MAKE_VAL_COMPLEX(86, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_22, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_22", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_22() */
/* END TEST 22 */

/*
 *  BEGIN TEST 23
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_ROWMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_23( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_ROWMAJ, x0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(77, 290), MAKE_VAL_COMPLEX(-459, -57), MAKE_VAL_COMPLEX(-50, 276), MAKE_VAL_COMPLEX(-570, -524), MAKE_VAL_COMPLEX(867, -296), MAKE_VAL_COMPLEX(-74, -522), MAKE_VAL_COMPLEX(-1046, -585), MAKE_VAL_COMPLEX(-912, 748), MAKE_VAL_COMPLEX(434, -9), MAKE_VAL_COMPLEX(142, 804), MAKE_VAL_COMPLEX(722, 102), MAKE_VAL_COMPLEX(-320, -92), MAKE_VAL_COMPLEX(1349, 1013), MAKE_VAL_COMPLEX(-78, -254), MAKE_VAL_COMPLEX(-288, 1104) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-2, -8);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_23, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_23", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_23() */
/* END TEST 23 */

/*
 *  BEGIN TEST 24
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_24( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(5, 33), MAKE_VAL_COMPLEX(-103, -59), MAKE_VAL_COMPLEX(129, 207), MAKE_VAL_COMPLEX(336, 254), MAKE_VAL_COMPLEX(2, 70), MAKE_VAL_COMPLEX(214, -41), MAKE_VAL_COMPLEX(20, -43), MAKE_VAL_COMPLEX(17, -72), MAKE_VAL_COMPLEX(50, -78), MAKE_VAL_COMPLEX(71, 57), MAKE_VAL_COMPLEX(7, -116), MAKE_VAL_COMPLEX(48, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_24, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_24", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_24() */
/* END TEST 24 */

/*
 *  BEGIN TEST 25
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_25( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(70, -96), MAKE_VAL_COMPLEX(84, -30), MAKE_VAL_COMPLEX(-205, -153), MAKE_VAL_COMPLEX(-337, -181), MAKE_VAL_COMPLEX(-1, 21), MAKE_VAL_COMPLEX(-307, 24), MAKE_VAL_COMPLEX(-30, 35), MAKE_VAL_COMPLEX(70, 35), MAKE_VAL_COMPLEX(4, 72), MAKE_VAL_COMPLEX(-41, -33), MAKE_VAL_COMPLEX(38, 29), MAKE_VAL_COMPLEX(-2, -36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_25, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_25", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_25() */
/* END TEST 25 */

/*
 *  BEGIN TEST 26
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_26( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-24, 81), MAKE_VAL_COMPLEX(56, 55), MAKE_VAL_COMPLEX(29, -75), MAKE_VAL_COMPLEX(-34, -53), MAKE_VAL_COMPLEX(-44, -65), MAKE_VAL_COMPLEX(75, -32), MAKE_VAL_COMPLEX(11, 1), MAKE_VAL_COMPLEX(-45, 68), MAKE_VAL_COMPLEX(-36, 30), MAKE_VAL_COMPLEX(-33, -3), MAKE_VAL_COMPLEX(9, 84), MAKE_VAL_COMPLEX(4, 78) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_26, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_26", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_26() */
/* END TEST 26 */

/*
 *  BEGIN TEST 27
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_27( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(193, -786), MAKE_VAL_COMPLEX(365, -16), MAKE_VAL_COMPLEX(-1162, -940), MAKE_VAL_COMPLEX(-2150, -1379), MAKE_VAL_COMPLEX(250, 1), MAKE_VAL_COMPLEX(-2051, 459), MAKE_VAL_COMPLEX(-217, 286), MAKE_VAL_COMPLEX(248, 56), MAKE_VAL_COMPLEX(-74, 360), MAKE_VAL_COMPLEX(-266, -354), MAKE_VAL_COMPLEX(-52, 212), MAKE_VAL_COMPLEX(-308, -260) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_27, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_27", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_27() */
/* END TEST 27 */

/*
 *  BEGIN TEST 28
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_28( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-122, 2), MAKE_VAL_COMPLEX(11, -86), MAKE_VAL_COMPLEX(23, 128), MAKE_VAL_COMPLEX(29, 28), MAKE_VAL_COMPLEX(-91, -73), MAKE_VAL_COMPLEX(92, 13), MAKE_VAL_COMPLEX(-39, -153), MAKE_VAL_COMPLEX(77, 108), MAKE_VAL_COMPLEX(-134, 56), MAKE_VAL_COMPLEX(-52, 96), MAKE_VAL_COMPLEX(-65, -67), MAKE_VAL_COMPLEX(-141, 123), MAKE_VAL_COMPLEX(-149, 136), MAKE_VAL_COMPLEX(36, -205), MAKE_VAL_COMPLEX(211, -104) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_28, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_28", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_28() */
/* END TEST 28 */

/*
 *  BEGIN TEST 29
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_29( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-18, -87), MAKE_VAL_COMPLEX(-138, -3), MAKE_VAL_COMPLEX(-119, -70), MAKE_VAL_COMPLEX(141, 62), MAKE_VAL_COMPLEX(209, -6), MAKE_VAL_COMPLEX(46, -27), MAKE_VAL_COMPLEX(-6, 53), MAKE_VAL_COMPLEX(-103, -100), MAKE_VAL_COMPLEX(59, 24), MAKE_VAL_COMPLEX(-16, -132), MAKE_VAL_COMPLEX(-79, -31), MAKE_VAL_COMPLEX(35, -5), MAKE_VAL_COMPLEX(78, -43), MAKE_VAL_COMPLEX(28, 43), MAKE_VAL_COMPLEX(-131, -6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_29, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_29", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_29() */
/* END TEST 29 */

/*
 *  BEGIN TEST 30
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_30( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(129, -18), MAKE_VAL_COMPLEX(123, -9), MAKE_VAL_COMPLEX(30, -84), MAKE_VAL_COMPLEX(-150, 30), MAKE_VAL_COMPLEX(-33, 108), MAKE_VAL_COMPLEX(-78, 78), MAKE_VAL_COMPLEX(78, 39), MAKE_VAL_COMPLEX(12, -18), MAKE_VAL_COMPLEX(6, -87), MAKE_VAL_COMPLEX(60, -12), MAKE_VAL_COMPLEX(138, -12), MAKE_VAL_COMPLEX(6, -126), MAKE_VAL_COMPLEX(-3, -93), MAKE_VAL_COMPLEX(42, 132), MAKE_VAL_COMPLEX(6, 108) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_30, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_30", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_30() */
/* END TEST 30 */

/*
 *  BEGIN TEST 31
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_31( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-376, 120), MAKE_VAL_COMPLEX(763, -93), MAKE_VAL_COMPLEX(217, 821), MAKE_VAL_COMPLEX(-481, -237), MAKE_VAL_COMPLEX(-966, -576), MAKE_VAL_COMPLEX(37, 295), MAKE_VAL_COMPLEX(274, -744), MAKE_VAL_COMPLEX(327, 979), MAKE_VAL_COMPLEX(-730, -262), MAKE_VAL_COMPLEX(-412, 724), MAKE_VAL_COMPLEX(234, -136), MAKE_VAL_COMPLEX(-820, 62), MAKE_VAL_COMPLEX(-1079, 197), MAKE_VAL_COMPLEX(459, -735), MAKE_VAL_COMPLEX(1365, 273) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(7, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_31, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_31", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_31() */
/* END TEST 31 */

/*
 *  BEGIN TEST 32
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_32( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-63, -63), MAKE_VAL_COMPLEX(69, -144), MAKE_VAL_COMPLEX(-143, 50), MAKE_VAL_COMPLEX(109, 108), MAKE_VAL_COMPLEX(12, -26), MAKE_VAL_COMPLEX(48, 83), MAKE_VAL_COMPLEX(-2, -150), MAKE_VAL_COMPLEX(-121, 74), MAKE_VAL_COMPLEX(45, 23), MAKE_VAL_COMPLEX(20, -14), MAKE_VAL_COMPLEX(23, -95), MAKE_VAL_COMPLEX(-163, 65), MAKE_VAL_COMPLEX(-117, -118), MAKE_VAL_COMPLEX(38, 61), MAKE_VAL_COMPLEX(191, 62) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_32, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_32", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_32() */
/* END TEST 32 */

/*
 *  BEGIN TEST 33
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_33( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(29, 61), MAKE_VAL_COMPLEX(-101, 140), MAKE_VAL_COMPLEX(131, -30), MAKE_VAL_COMPLEX(-69, -108), MAKE_VAL_COMPLEX(2, 0), MAKE_VAL_COMPLEX(-24, -99), MAKE_VAL_COMPLEX(-16, 136), MAKE_VAL_COMPLEX(117, -70), MAKE_VAL_COMPLEX(-51, -1), MAKE_VAL_COMPLEX(-36, 14), MAKE_VAL_COMPLEX(-59, 91), MAKE_VAL_COMPLEX(155, -33), MAKE_VAL_COMPLEX(113, 142), MAKE_VAL_COMPLEX(-42, -97), MAKE_VAL_COMPLEX(-187, -90) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_33, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_33", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_33() */
/* END TEST 33 */

/*
 *  BEGIN TEST 34
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_34( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-127, -16), MAKE_VAL_COMPLEX(-119, -23), MAKE_VAL_COMPLEX(-50, 72), MAKE_VAL_COMPLEX(150, 10), MAKE_VAL_COMPLEX(59, -94), MAKE_VAL_COMPLEX(94, -54), MAKE_VAL_COMPLEX(-64, -57), MAKE_VAL_COMPLEX(-16, 14), MAKE_VAL_COMPLEX(-28, 81), MAKE_VAL_COMPLEX(-60, -4), MAKE_VAL_COMPLEX(-134, -24), MAKE_VAL_COMPLEX(-38, 118), MAKE_VAL_COMPLEX(-21, 89), MAKE_VAL_COMPLEX(-6, -136), MAKE_VAL_COMPLEX(22, -104) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_34, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_34", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_34() */
/* END TEST 34 */

/*
 *  BEGIN TEST 35
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_35( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_ROWMAJ, y0_VW );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(347, 177), MAKE_VAL_COMPLEX(-1244, -39), MAKE_VAL_COMPLEX(1118, 185), MAKE_VAL_COMPLEX(-396, -707), MAKE_VAL_COMPLEX(222, 628), MAKE_VAL_COMPLEX(397, -336), MAKE_VAL_COMPLEX(-586, 898), MAKE_VAL_COMPLEX(1422, 135), MAKE_VAL_COMPLEX(-663, -791), MAKE_VAL_COMPLEX(-146, -396), MAKE_VAL_COMPLEX(-505, -201), MAKE_VAL_COMPLEX(1079, 213), MAKE_VAL_COMPLEX(-414, 1503), MAKE_VAL_COMPLEX(979, -594), MAKE_VAL_COMPLEX(-478, -1545) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_35, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_35", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_35() */
/* END TEST 35 */

/*
 *  BEGIN TEST 36
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_36( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-109, 96), MAKE_VAL_COMPLEX(248, 267), MAKE_VAL_COMPLEX(115, -18), MAKE_VAL_COMPLEX(92, 12), MAKE_VAL_COMPLEX(19, -58), MAKE_VAL_COMPLEX(0, 83), MAKE_VAL_COMPLEX(-28, 8), MAKE_VAL_COMPLEX(-62, -74), MAKE_VAL_COMPLEX(122, 96), MAKE_VAL_COMPLEX(289, -109), MAKE_VAL_COMPLEX(-76, -30), MAKE_VAL_COMPLEX(-32, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_36, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_36", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_36() */
/* END TEST 36 */

/*
 *  BEGIN TEST 37
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_37( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(67, -120), MAKE_VAL_COMPLEX(-286, -243), MAKE_VAL_COMPLEX(-77, 44), MAKE_VAL_COMPLEX(-58, -24), MAKE_VAL_COMPLEX(23, 42), MAKE_VAL_COMPLEX(8, -37), MAKE_VAL_COMPLEX(26, -2), MAKE_VAL_COMPLEX(30, 40), MAKE_VAL_COMPLEX(-134, -120), MAKE_VAL_COMPLEX(-283, 91), MAKE_VAL_COMPLEX(28, 24), MAKE_VAL_COMPLEX(-12, -56) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_37, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_37", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_37() */
/* END TEST 37 */

/*
 *  BEGIN TEST 38
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_38( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(30, 45), MAKE_VAL_COMPLEX(50, -5), MAKE_VAL_COMPLEX(-25, -45), MAKE_VAL_COMPLEX(-40, -5), MAKE_VAL_COMPLEX(-50, -5), MAKE_VAL_COMPLEX(15, -50), MAKE_VAL_COMPLEX(5, -5), MAKE_VAL_COMPLEX(15, 50), MAKE_VAL_COMPLEX(0, 30), MAKE_VAL_COMPLEX(-15, 15), MAKE_VAL_COMPLEX(45, 30), MAKE_VAL_COMPLEX(40, 30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_38, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_38", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_38() */
/* END TEST 38 */

/*
 *  BEGIN TEST 39
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_39( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	oski_value_t* y_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-357, 192), MAKE_VAL_COMPLEX(245, 1434), MAKE_VAL_COMPLEX(172, 52), MAKE_VAL_COMPLEX(-16, 345), MAKE_VAL_COMPLEX(-87, -68), MAKE_VAL_COMPLEX(-236, -85), MAKE_VAL_COMPLEX(-85, -99), MAKE_VAL_COMPLEX(209, -122), MAKE_VAL_COMPLEX(126, 840), MAKE_VAL_COMPLEX(1125, 657), MAKE_VAL_COMPLEX(189, -204), MAKE_VAL_COMPLEX(56, 170) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_39, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_39", y_true, y_val, y0_LEN*y0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_39() */
/* END TEST 39 */

/*
 *  BEGIN TEST 40
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_40( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-147, -8), MAKE_VAL_COMPLEX(-12, -97), MAKE_VAL_COMPLEX(-171, -22), MAKE_VAL_COMPLEX(239, 20), MAKE_VAL_COMPLEX(18, 3), MAKE_VAL_COMPLEX(94, -73), MAKE_VAL_COMPLEX(-69, 32), MAKE_VAL_COMPLEX(-138, 83), MAKE_VAL_COMPLEX(-23, -32), MAKE_VAL_COMPLEX(-18, -133), MAKE_VAL_COMPLEX(-146, -44), MAKE_VAL_COMPLEX(0, 110), MAKE_VAL_COMPLEX(19, 77), MAKE_VAL_COMPLEX(-5, -97), MAKE_VAL_COMPLEX(-29, -28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_40, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_40", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_40() */
/* END TEST 40 */

/*
 *  BEGIN TEST 41
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_41( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(53, -134), MAKE_VAL_COMPLEX(-68, -43), MAKE_VAL_COMPLEX(55, 34), MAKE_VAL_COMPLEX(-119, 140), MAKE_VAL_COMPLEX(128, -25), MAKE_VAL_COMPLEX(42, 121), MAKE_VAL_COMPLEX(71, -146), MAKE_VAL_COMPLEX(110, -87), MAKE_VAL_COMPLEX(-83, 74), MAKE_VAL_COMPLEX(-30, 69), MAKE_VAL_COMPLEX(54, -112), MAKE_VAL_COMPLEX(-152, -46), MAKE_VAL_COMPLEX(-127, -21), MAKE_VAL_COMPLEX(137, -27), MAKE_VAL_COMPLEX(153, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_41, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_41", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_41() */
/* END TEST 41 */

/*
 *  BEGIN TEST 42
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_42( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, -86), MAKE_VAL_COMPLEX(-6, -82), MAKE_VAL_COMPLEX(-56, -20), MAKE_VAL_COMPLEX(20, 100), MAKE_VAL_COMPLEX(72, 22), MAKE_VAL_COMPLEX(52, 52), MAKE_VAL_COMPLEX(26, -52), MAKE_VAL_COMPLEX(-12, -8), MAKE_VAL_COMPLEX(-58, -4), MAKE_VAL_COMPLEX(-8, -40), MAKE_VAL_COMPLEX(-8, -92), MAKE_VAL_COMPLEX(-84, -4), MAKE_VAL_COMPLEX(-62, 2), MAKE_VAL_COMPLEX(88, -28), MAKE_VAL_COMPLEX(72, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_42, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_42", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_42() */
/* END TEST 42 */

/*
 *  BEGIN TEST 43
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_43( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-28, -570), MAKE_VAL_COMPLEX(-440, 987), MAKE_VAL_COMPLEX(1362, -131), MAKE_VAL_COMPLEX(-602, 963), MAKE_VAL_COMPLEX(-15, -1033), MAKE_VAL_COMPLEX(409, 332), MAKE_VAL_COMPLEX(-517, -999), MAKE_VAL_COMPLEX(433, -1430), MAKE_VAL_COMPLEX(493, 975), MAKE_VAL_COMPLEX(495, 1158), MAKE_VAL_COMPLEX(116, -320), MAKE_VAL_COMPLEX(-620, 390), MAKE_VAL_COMPLEX(-506, 442), MAKE_VAL_COMPLEX(79, -509), MAKE_VAL_COMPLEX(188, -1049) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_43, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_43", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_43() */
/* END TEST 43 */

/*
 *  BEGIN TEST 44
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_44( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(31, 79), MAKE_VAL_COMPLEX(149, -4), MAKE_VAL_COMPLEX(153, -124), MAKE_VAL_COMPLEX(-191, -70), MAKE_VAL_COMPLEX(-226, 168), MAKE_VAL_COMPLEX(-68, -45), MAKE_VAL_COMPLEX(16, 164), MAKE_VAL_COMPLEX(-44, -129), MAKE_VAL_COMPLEX(12, 58), MAKE_VAL_COMPLEX(52, 23), MAKE_VAL_COMPLEX(50, 66), MAKE_VAL_COMPLEX(134, -64), MAKE_VAL_COMPLEX(48, 10), MAKE_VAL_COMPLEX(-21, 61), MAKE_VAL_COMPLEX(-191, -16) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_44, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_44", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_44() */
/* END TEST 44 */

/*
 *  BEGIN TEST 45
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_45( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(170, -16), MAKE_VAL_COMPLEX(37, 76), MAKE_VAL_COMPLEX(-51, 22), MAKE_VAL_COMPLEX(-49, 10), MAKE_VAL_COMPLEX(103, -33), MAKE_VAL_COMPLEX(-100, 105), MAKE_VAL_COMPLEX(71, -53), MAKE_VAL_COMPLEX(74, 111), MAKE_VAL_COMPLEX(57, -181), MAKE_VAL_COMPLEX(44, 1), MAKE_VAL_COMPLEX(160, 12), MAKE_VAL_COMPLEX(-38, -116), MAKE_VAL_COMPLEX(12, -148), MAKE_VAL_COMPLEX(-9, 161), MAKE_VAL_COMPLEX(125, 178) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_45, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_45", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_45() */
/* END TEST 45 */

/*
 *  BEGIN TEST 46
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_46( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-51, -3), MAKE_VAL_COMPLEX(-48, -6), MAKE_VAL_COMPLEX(-18, 30), MAKE_VAL_COMPLEX(60, 0), MAKE_VAL_COMPLEX(21, -39), MAKE_VAL_COMPLEX(36, -24), MAKE_VAL_COMPLEX(-27, -21), MAKE_VAL_COMPLEX(-6, 6), MAKE_VAL_COMPLEX(-9, 33), MAKE_VAL_COMPLEX(-24, 0), MAKE_VAL_COMPLEX(-54, -6), MAKE_VAL_COMPLEX(-12, 48), MAKE_VAL_COMPLEX(-6, 36), MAKE_VAL_COMPLEX(-6, -54), MAKE_VAL_COMPLEX(6, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_46, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_46", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_46() */
/* END TEST 46 */

/*
 *  BEGIN TEST 47
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x0', LAYOUT_COLMAJ>
 *  y : <'y0', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_47( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A0_ptr, A0_ind, A0_val,
		A0_M, A0_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y0_val,
		y0_LEN * y0_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y0_LEN, y0_VW, LAYOUT_COLMAJ, y0_LEN );
	oski_value_t* y_val = array_Duplicate( x0_val,
		x0_LEN * x0_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x0_LEN, x0_VW, LAYOUT_COLMAJ, x0_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(72, 149), MAKE_VAL_COMPLEX(-300, -683), MAKE_VAL_COMPLEX(-573, -698), MAKE_VAL_COMPLEX(-121, 746), MAKE_VAL_COMPLEX(739, 1090), MAKE_VAL_COMPLEX(-176, -31), MAKE_VAL_COMPLEX(664, 45), MAKE_VAL_COMPLEX(-834, 217), MAKE_VAL_COMPLEX(468, 305), MAKE_VAL_COMPLEX(-6, -157), MAKE_VAL_COMPLEX(-28, 14), MAKE_VAL_COMPLEX(-84, -410), MAKE_VAL_COMPLEX(257, 49), MAKE_VAL_COMPLEX(-1, -221), MAKE_VAL_COMPLEX(-469, 754) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_47, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_47", y_true, y_val, x0_LEN*x0_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_47() */
/* END TEST 47 */

#define A48_M 4    /* # of rows */
#define A48_N 5    /* # of cols */
#define A48_NNZ 6  /* # of stored non-zeros */
static oski_index_t A48_ptr[A48_N+1] = {0, 1, 3, 6, 6, 6}; /* end row pointers for A48 */
static oski_index_t A48_ind[A48_NNZ] = {2, 0, 3, 0, 2, 3}; /* end column indices pointers for A48 */
static oski_value_t A48_val[A48_NNZ] = {MAKE_VAL_COMPLEX(10, -1), MAKE_VAL_COMPLEX(3, -8), MAKE_VAL_COMPLEX(-10, -10), MAKE_VAL_COMPLEX(6, 10), MAKE_VAL_COMPLEX(7, 1), MAKE_VAL_COMPLEX(9, 8)}; /* end stored non-zero values for A48 */

#define x48_LEN 5   /* vector length */
#define x48_VW  3   /* vector width */
static oski_value_t x48_val[x48_LEN*x48_VW] = {MAKE_VAL_COMPLEX(-3, 4), MAKE_VAL_COMPLEX(-1, -10), MAKE_VAL_COMPLEX(10, 5), MAKE_VAL_COMPLEX(6, 2), MAKE_VAL_COMPLEX(6, -9), MAKE_VAL_COMPLEX(-5, 2), MAKE_VAL_COMPLEX(6, 4), MAKE_VAL_COMPLEX(-9, 0), MAKE_VAL_COMPLEX(4, 10), MAKE_VAL_COMPLEX(2, 5), MAKE_VAL_COMPLEX(9, 2), MAKE_VAL_COMPLEX(-4, 5), MAKE_VAL_COMPLEX(-10, -10), MAKE_VAL_COMPLEX(-8, -4), MAKE_VAL_COMPLEX(-1, -1)}; /* data */

#define y48_LEN 4   /* vector length */
#define y48_VW  3   /* vector width */
static oski_value_t y48_val[y48_LEN*y48_VW] = {MAKE_VAL_COMPLEX(10, 6), MAKE_VAL_COMPLEX(-2, -4), MAKE_VAL_COMPLEX(-8, -3), MAKE_VAL_COMPLEX(-6, -6), MAKE_VAL_COMPLEX(-3, -6), MAKE_VAL_COMPLEX(7, 9), MAKE_VAL_COMPLEX(2, -6), MAKE_VAL_COMPLEX(-7, -1), MAKE_VAL_COMPLEX(2, 2), MAKE_VAL_COMPLEX(7, -10), MAKE_VAL_COMPLEX(-10, -7), MAKE_VAL_COMPLEX(-1, 10)}; /* data */

/*
 *  BEGIN TEST 48
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_48( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(120, 96), MAKE_VAL_COMPLEX(-126, -201), MAKE_VAL_COMPLEX(-147, 119), MAKE_VAL_COMPLEX(-54, -54), MAKE_VAL_COMPLEX(-27, -54), MAKE_VAL_COMPLEX(63, 81), MAKE_VAL_COMPLEX(30, 23), MAKE_VAL_COMPLEX(-146, -117), MAKE_VAL_COMPLEX(141, 132), MAKE_VAL_COMPLEX(45, -86), MAKE_VAL_COMPLEX(-321, -105), MAKE_VAL_COMPLEX(17, 242) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_48, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_48", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_48() */
/* END TEST 48 */

/*
 *  BEGIN TEST 49
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_49( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(40, 0), MAKE_VAL_COMPLEX(94, 137), MAKE_VAL_COMPLEX(19, -167), MAKE_VAL_COMPLEX(-42, -42), MAKE_VAL_COMPLEX(-21, -42), MAKE_VAL_COMPLEX(49, 63), MAKE_VAL_COMPLEX(2, -119), MAKE_VAL_COMPLEX(34, 101), MAKE_VAL_COMPLEX(-109, -100), MAKE_VAL_COMPLEX(67, -74), MAKE_VAL_COMPLEX(161, -7), MAKE_VAL_COMPLEX(-33, -82) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_49, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_49", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_49() */
/* END TEST 49 */

/*
 *  BEGIN TEST 50
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_50( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-66, -94), MAKE_VAL_COMPLEX(2, 44), MAKE_VAL_COMPLEX(60, 59), MAKE_VAL_COMPLEX(30, 78), MAKE_VAL_COMPLEX(3, 66), MAKE_VAL_COMPLEX(-27, -109), MAKE_VAL_COMPLEX(-42, 46), MAKE_VAL_COMPLEX(59, 37), MAKE_VAL_COMPLEX(-10, -26), MAKE_VAL_COMPLEX(-103, 62), MAKE_VAL_COMPLEX(62, 103), MAKE_VAL_COMPLEX(49, -86) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_50, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_50", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_50() */
/* END TEST 50 */

/*
 *  BEGIN TEST 51
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_51( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-66, 454), MAKE_VAL_COMPLEX(169, -2189), MAKE_VAL_COMPLEX(-1649, 880), MAKE_VAL_COMPLEX(6, 102), MAKE_VAL_COMPLEX(-21, 78), MAKE_VAL_COMPLEX(9, -137), MAKE_VAL_COMPLEX(-497, 815), MAKE_VAL_COMPLEX(64, -1488), MAKE_VAL_COMPLEX(307, 1853), MAKE_VAL_COMPLEX(-333, -56), MAKE_VAL_COMPLEX(-1751, -1852), MAKE_VAL_COMPLEX(-741, 1468) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(9, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, -8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_51, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_51", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_51() */
/* END TEST 51 */

/*
 *  BEGIN TEST 52
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_52( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(10, -115), MAKE_VAL_COMPLEX(7, 70), MAKE_VAL_COMPLEX(-93, 48), MAKE_VAL_COMPLEX(-154, -6), MAKE_VAL_COMPLEX(7, 288), MAKE_VAL_COMPLEX(88, -86), MAKE_VAL_COMPLEX(87, 72), MAKE_VAL_COMPLEX(18, -264), MAKE_VAL_COMPLEX(-197, -52), MAKE_VAL_COMPLEX(-51, -26), MAKE_VAL_COMPLEX(-86, 47), MAKE_VAL_COMPLEX(-3, -68), MAKE_VAL_COMPLEX(150, 10), MAKE_VAL_COMPLEX(92, -24), MAKE_VAL_COMPLEX(15, 1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_52, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_52", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_52() */
/* END TEST 52 */

/*
 *  BEGIN TEST 53
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_53( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 35), MAKE_VAL_COMPLEX(87, 62), MAKE_VAL_COMPLEX(-87, -38), MAKE_VAL_COMPLEX(54, 26), MAKE_VAL_COMPLEX(-19, -114), MAKE_VAL_COMPLEX(-34, 18), MAKE_VAL_COMPLEX(-203, -80), MAKE_VAL_COMPLEX(108, 192), MAKE_VAL_COMPLEX(61, -56), MAKE_VAL_COMPLEX(-17, -28), MAKE_VAL_COMPLEX(-56, -3), MAKE_VAL_COMPLEX(19, -34), MAKE_VAL_COMPLEX(70, 50), MAKE_VAL_COMPLEX(52, 16), MAKE_VAL_COMPLEX(7, 5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_53, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_53", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_53() */
/* END TEST 53 */

/*
 *  BEGIN TEST 54
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_54( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, 41), MAKE_VAL_COMPLEX(-38, -77), MAKE_VAL_COMPLEX(95, 10), MAKE_VAL_COMPLEX(54, -2), MAKE_VAL_COMPLEX(21, -90), MAKE_VAL_COMPLEX(-34, 31), MAKE_VAL_COMPLEX(60, 14), MAKE_VAL_COMPLEX(-72, 27), MAKE_VAL_COMPLEX(62, 68), MAKE_VAL_COMPLEX(31, 34), MAKE_VAL_COMPLEX(78, -11), MAKE_VAL_COMPLEX(-17, 52), MAKE_VAL_COMPLEX(-110, -50), MAKE_VAL_COMPLEX(-76, -8), MAKE_VAL_COMPLEX(-11, -5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_54, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_54", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_54() */
/* END TEST 54 */

/*
 *  BEGIN TEST 55
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_55( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-191, 168), MAKE_VAL_COMPLEX(365, 216), MAKE_VAL_COMPLEX(-162, -71), MAKE_VAL_COMPLEX(246, 346), MAKE_VAL_COMPLEX(410, -569), MAKE_VAL_COMPLEX(-299, -38), MAKE_VAL_COMPLEX(-602, -554), MAKE_VAL_COMPLEX(-123, 840), MAKE_VAL_COMPLEX(272, 152), MAKE_VAL_COMPLEX(-54, -19), MAKE_VAL_COMPLEX(-79, 58), MAKE_VAL_COMPLEX(-12, -67), MAKE_VAL_COMPLEX(150, -10), MAKE_VAL_COMPLEX(88, -36), MAKE_VAL_COMPLEX(15, -1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-4, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_55, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_55", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_55() */
/* END TEST 55 */

/*
 *  BEGIN TEST 56
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_56( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, -71), MAKE_VAL_COMPLEX(-38, -10), MAKE_VAL_COMPLEX(-7, 47), MAKE_VAL_COMPLEX(0, 284), MAKE_VAL_COMPLEX(217, -31), MAKE_VAL_COMPLEX(-91, -200), MAKE_VAL_COMPLEX(93, -240), MAKE_VAL_COMPLEX(-239, -14), MAKE_VAL_COMPLEX(-25, 174), MAKE_VAL_COMPLEX(-17, 1), MAKE_VAL_COMPLEX(-15, 25), MAKE_VAL_COMPLEX(-11, -17), MAKE_VAL_COMPLEX(40, -20), MAKE_VAL_COMPLEX(20, -20), MAKE_VAL_COMPLEX(4, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_56, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_56", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_56() */
/* END TEST 56 */

/*
 *  BEGIN TEST 57
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_57( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-31, 48), MAKE_VAL_COMPLEX(90, 25), MAKE_VAL_COMPLEX(-38, -7), MAKE_VAL_COMPLEX(-22, -258), MAKE_VAL_COMPLEX(-184, 79), MAKE_VAL_COMPLEX(91, 171), MAKE_VAL_COMPLEX(-125, 262), MAKE_VAL_COMPLEX(257, -31), MAKE_VAL_COMPLEX(-33, -174), MAKE_VAL_COMPLEX(-12, -1), MAKE_VAL_COMPLEX(-13, 16), MAKE_VAL_COMPLEX(-6, -13), MAKE_VAL_COMPLEX(30, -10), MAKE_VAL_COMPLEX(16, -12), MAKE_VAL_COMPLEX(3, -1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_57, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_57", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_57() */
/* END TEST 57 */

/*
 *  BEGIN TEST 58
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_58( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-24, 32), MAKE_VAL_COMPLEX(-8, -80), MAKE_VAL_COMPLEX(80, 40), MAKE_VAL_COMPLEX(48, 16), MAKE_VAL_COMPLEX(48, -72), MAKE_VAL_COMPLEX(-40, 16), MAKE_VAL_COMPLEX(48, 32), MAKE_VAL_COMPLEX(-72, 0), MAKE_VAL_COMPLEX(32, 80), MAKE_VAL_COMPLEX(16, 40), MAKE_VAL_COMPLEX(72, 16), MAKE_VAL_COMPLEX(-32, 40), MAKE_VAL_COMPLEX(-80, -80), MAKE_VAL_COMPLEX(-64, -32), MAKE_VAL_COMPLEX(-8, -8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_58, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_58", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_58() */
/* END TEST 58 */

/*
 *  BEGIN TEST 59
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_59( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-129, -43), MAKE_VAL_COMPLEX(-27, 107), MAKE_VAL_COMPLEX(69, -11), MAKE_VAL_COMPLEX(552, -12), MAKE_VAL_COMPLEX(-89, -413), MAKE_VAL_COMPLEX(-383, 181), MAKE_VAL_COMPLEX(-494, -204), MAKE_VAL_COMPLEX(-1, 487), MAKE_VAL_COMPLEX(346, 16), MAKE_VAL_COMPLEX(1, 17), MAKE_VAL_COMPLEX(25, 15), MAKE_VAL_COMPLEX(-17, 11), MAKE_VAL_COMPLEX(-20, -40), MAKE_VAL_COMPLEX(-20, -20), MAKE_VAL_COMPLEX(-2, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_59, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_59", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_59() */
/* END TEST 59 */

/*
 *  BEGIN TEST 60
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_60( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(68, -44), MAKE_VAL_COMPLEX(-30, 20), MAKE_VAL_COMPLEX(-4, 144), MAKE_VAL_COMPLEX(-60, 58), MAKE_VAL_COMPLEX(-153, -135), MAKE_VAL_COMPLEX(65, -65), MAKE_VAL_COMPLEX(-133, -118), MAKE_VAL_COMPLEX(-232, 15), MAKE_VAL_COMPLEX(-61, 128), MAKE_VAL_COMPLEX(-87, -46), MAKE_VAL_COMPLEX(77, 201), MAKE_VAL_COMPLEX(107, 150) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_60, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_60", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_60() */
/* END TEST 60 */

/*
 *  BEGIN TEST 61
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_61( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(62, -82), MAKE_VAL_COMPLEX(-38, -6), MAKE_VAL_COMPLEX(-73, -36), MAKE_VAL_COMPLEX(-54, 8), MAKE_VAL_COMPLEX(51, 156), MAKE_VAL_COMPLEX(98, -4), MAKE_VAL_COMPLEX(51, 64), MAKE_VAL_COMPLEX(189, 86), MAKE_VAL_COMPLEX(99, -150), MAKE_VAL_COMPLEX(-35, -99), MAKE_VAL_COMPLEX(-222, -79), MAKE_VAL_COMPLEX(39, -95) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_61, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_61", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_61() */
/* END TEST 61 */

/*
 *  BEGIN TEST 62
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_62( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(74, 58), MAKE_VAL_COMPLEX(-12, -34), MAKE_VAL_COMPLEX(-61, -32), MAKE_VAL_COMPLEX(-42, -54), MAKE_VAL_COMPLEX(-18, -51), MAKE_VAL_COMPLEX(47, 79), MAKE_VAL_COMPLEX(22, -46), MAKE_VAL_COMPLEX(-55, -15), MAKE_VAL_COMPLEX(14, 18), MAKE_VAL_COMPLEX(66, -73), MAKE_VAL_COMPLEX(-73, -66), MAKE_VAL_COMPLEX(-18, 79) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_62, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_62", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_62() */
/* END TEST 62 */

/*
 *  BEGIN TEST 63
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_63( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-142, 234), MAKE_VAL_COMPLEX(38, 16), MAKE_VAL_COMPLEX(-121, 240), MAKE_VAL_COMPLEX(12, -36), MAKE_VAL_COMPLEX(228, -795), MAKE_VAL_COMPLEX(-103, -21), MAKE_VAL_COMPLEX(97, -519), MAKE_VAL_COMPLEX(-512, -854), MAKE_VAL_COMPLEX(-689, 211), MAKE_VAL_COMPLEX(11, 112), MAKE_VAL_COMPLEX(139, 700), MAKE_VAL_COMPLEX(-431, 458) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_63, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_63", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_63() */
/* END TEST 63 */

/*
 *  BEGIN TEST 64
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_64( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-67, -60), MAKE_VAL_COMPLEX(42, 16), MAKE_VAL_COMPLEX(-197, -80), MAKE_VAL_COMPLEX(26, 54), MAKE_VAL_COMPLEX(-27, 170), MAKE_VAL_COMPLEX(168, -126), MAKE_VAL_COMPLEX(-115, -15), MAKE_VAL_COMPLEX(79, -189), MAKE_VAL_COMPLEX(-212, -17), MAKE_VAL_COMPLEX(-26, -36), MAKE_VAL_COMPLEX(-76, 2), MAKE_VAL_COMPLEX(22, -48), MAKE_VAL_COMPLEX(100, 60), MAKE_VAL_COMPLEX(72, 16), MAKE_VAL_COMPLEX(10, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_64, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_64", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_64() */
/* END TEST 64 */

/*
 *  BEGIN TEST 65
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_65( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(107, 40), MAKE_VAL_COMPLEX(-74, 68), MAKE_VAL_COMPLEX(137, 0), MAKE_VAL_COMPLEX(-66, -94), MAKE_VAL_COMPLEX(-57, -122), MAKE_VAL_COMPLEX(-120, 130), MAKE_VAL_COMPLEX(83, -41), MAKE_VAL_COMPLEX(-7, 225), MAKE_VAL_COMPLEX(220, -79), MAKE_VAL_COMPLEX(30, -12), MAKE_VAL_COMPLEX(12, -54), MAKE_VAL_COMPLEX(30, 24), MAKE_VAL_COMPLEX(-60, 60), MAKE_VAL_COMPLEX(-24, 48), MAKE_VAL_COMPLEX(-6, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_65, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_65", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_65() */
/* END TEST 65 */

/*
 *  BEGIN TEST 66
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_66( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-10, -20), MAKE_VAL_COMPLEX(42, 16), MAKE_VAL_COMPLEX(-40, 30), MAKE_VAL_COMPLEX(-20, 20), MAKE_VAL_COMPLEX(24, 42), MAKE_VAL_COMPLEX(2, -24), MAKE_VAL_COMPLEX(-28, 16), MAKE_VAL_COMPLEX(18, -36), MAKE_VAL_COMPLEX(-48, -4), MAKE_VAL_COMPLEX(-24, -2), MAKE_VAL_COMPLEX(-26, 32), MAKE_VAL_COMPLEX(-12, -26), MAKE_VAL_COMPLEX(60, -20), MAKE_VAL_COMPLEX(32, -24), MAKE_VAL_COMPLEX(6, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_66, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_66", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_66() */
/* END TEST 66 */

/*
 *  BEGIN TEST 67
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_67( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-178, -355), MAKE_VAL_COMPLEX(275, -78), MAKE_VAL_COMPLEX(-231, -496), MAKE_VAL_COMPLEX(10, 418), MAKE_VAL_COMPLEX(-255, 354), MAKE_VAL_COMPLEX(723, 62), MAKE_VAL_COMPLEX(-250, -166), MAKE_VAL_COMPLEX(597, -528), MAKE_VAL_COMPLEX(-713, -369), MAKE_VAL_COMPLEX(-34, -27), MAKE_VAL_COMPLEX(-71, 22), MAKE_VAL_COMPLEX(8, -51), MAKE_VAL_COMPLEX(110, 30), MAKE_VAL_COMPLEX(72, -4), MAKE_VAL_COMPLEX(11, 3) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_67, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_67", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_67() */
/* END TEST 67 */

/*
 *  BEGIN TEST 68
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_68( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-115, -54), MAKE_VAL_COMPLEX(104, -86), MAKE_VAL_COMPLEX(-113, 10), MAKE_VAL_COMPLEX(98, 150), MAKE_VAL_COMPLEX(203, -72), MAKE_VAL_COMPLEX(-126, -124), MAKE_VAL_COMPLEX(-61, -27), MAKE_VAL_COMPLEX(-159, -75), MAKE_VAL_COMPLEX(-46, 103), MAKE_VAL_COMPLEX(-36, 26), MAKE_VAL_COMPLEX(2, 76), MAKE_VAL_COMPLEX(-48, -22), MAKE_VAL_COMPLEX(60, -100), MAKE_VAL_COMPLEX(16, -72), MAKE_VAL_COMPLEX(6, -10) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_68, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_68", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_68() */
/* END TEST 68 */

/*
 *  BEGIN TEST 69
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_69( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(124, 42), MAKE_VAL_COMPLEX(-101, 116), MAKE_VAL_COMPLEX(83, -25), MAKE_VAL_COMPLEX(-116, -156), MAKE_VAL_COMPLEX(-221, 99), MAKE_VAL_COMPLEX(141, 118), MAKE_VAL_COMPLEX(43, 15), MAKE_VAL_COMPLEX(186, 75), MAKE_VAL_COMPLEX(34, -133), MAKE_VAL_COMPLEX(30, -41), MAKE_VAL_COMPLEX(-29, -82), MAKE_VAL_COMPLEX(60, 7), MAKE_VAL_COMPLEX(-30, 130), MAKE_VAL_COMPLEX(8, 84), MAKE_VAL_COMPLEX(-3, 13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_69, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_69", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_69() */
/* END TEST 69 */

/*
 *  BEGIN TEST 70
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_70( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(5, 10), MAKE_VAL_COMPLEX(-21, -8), MAKE_VAL_COMPLEX(20, -15), MAKE_VAL_COMPLEX(10, -10), MAKE_VAL_COMPLEX(-12, -21), MAKE_VAL_COMPLEX(-1, 12), MAKE_VAL_COMPLEX(14, -8), MAKE_VAL_COMPLEX(-9, 18), MAKE_VAL_COMPLEX(24, 2), MAKE_VAL_COMPLEX(12, 1), MAKE_VAL_COMPLEX(13, -16), MAKE_VAL_COMPLEX(6, 13), MAKE_VAL_COMPLEX(-30, 10), MAKE_VAL_COMPLEX(-16, 12), MAKE_VAL_COMPLEX(-3, 1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_70, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_70", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_70() */
/* END TEST 70 */

/*
 *  BEGIN TEST 71
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_ROWMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_71( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_ROWMAJ, x48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-846, 194), MAKE_VAL_COMPLEX(-186, -676), MAKE_VAL_COMPLEX(-1144, -92), MAKE_VAL_COMPLEX(1448, 160), MAKE_VAL_COMPLEX(340, -1608), MAKE_VAL_COMPLEX(-1350, -72), MAKE_VAL_COMPLEX(-774, -418), MAKE_VAL_COMPLEX(-1200, 858), MAKE_VAL_COMPLEX(578, 296), MAKE_VAL_COMPLEX(32, 22), MAKE_VAL_COMPLEX(62, -24), MAKE_VAL_COMPLEX(-4, 46), MAKE_VAL_COMPLEX(-100, -20), MAKE_VAL_COMPLEX(-64, 8), MAKE_VAL_COMPLEX(-10, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(8, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_71, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_71", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_71() */
/* END TEST 71 */

/*
 *  BEGIN TEST 72
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_72( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-9, 160), MAKE_VAL_COMPLEX(-14, -156), MAKE_VAL_COMPLEX(15, -142), MAKE_VAL_COMPLEX(-36, -48), MAKE_VAL_COMPLEX(-15, -45), MAKE_VAL_COMPLEX(40, 70), MAKE_VAL_COMPLEX(59, 48), MAKE_VAL_COMPLEX(-159, 2), MAKE_VAL_COMPLEX(44, -53), MAKE_VAL_COMPLEX(19, 172), MAKE_VAL_COMPLEX(-164, -231), MAKE_VAL_COMPLEX(63, -111) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_72, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_72", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_72() */
/* END TEST 72 */

/*
 *  BEGIN TEST 73
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_73( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-7, -156), MAKE_VAL_COMPLEX(20, 158), MAKE_VAL_COMPLEX(-4, 137), MAKE_VAL_COMPLEX(48, 48), MAKE_VAL_COMPLEX(24, 48), MAKE_VAL_COMPLEX(-56, -72), MAKE_VAL_COMPLEX(-55, -40), MAKE_VAL_COMPLEX(167, -8), MAKE_VAL_COMPLEX(-48, 53), MAKE_VAL_COMPLEX(-16, -155), MAKE_VAL_COMPLEX(181, 228), MAKE_VAL_COMPLEX(-72, 100) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_73, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_73", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_73() */
/* END TEST 73 */

/*
 *  BEGIN TEST 74
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_74( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(66, 94), MAKE_VAL_COMPLEX(-2, -44), MAKE_VAL_COMPLEX(-60, -59), MAKE_VAL_COMPLEX(-30, -78), MAKE_VAL_COMPLEX(-3, -66), MAKE_VAL_COMPLEX(27, 109), MAKE_VAL_COMPLEX(42, -46), MAKE_VAL_COMPLEX(-59, -37), MAKE_VAL_COMPLEX(10, 26), MAKE_VAL_COMPLEX(103, -62), MAKE_VAL_COMPLEX(-62, -103), MAKE_VAL_COMPLEX(-49, 86) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_74, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_74", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_74() */
/* END TEST 74 */

/*
 *  BEGIN TEST 75
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_75( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-83, -1284), MAKE_VAL_COMPLEX(748, 876), MAKE_VAL_COMPLEX(175, 1271), MAKE_VAL_COMPLEX(-54, 54), MAKE_VAL_COMPLEX(-54, 27), MAKE_VAL_COMPLEX(81, -63), MAKE_VAL_COMPLEX(-855, -400), MAKE_VAL_COMPLEX(672, -715), MAKE_VAL_COMPLEX(208, 657), MAKE_VAL_COMPLEX(-1220, -1948), MAKE_VAL_COMPLEX(1676, 688), MAKE_VAL_COMPLEX(610, 1749) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, 6);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_75, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_75", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_75() */
/* END TEST 75 */

/*
 *  BEGIN TEST 76
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_76( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(61, -58), MAKE_VAL_COMPLEX(-167, 26), MAKE_VAL_COMPLEX(153, -43), MAKE_VAL_COMPLEX(-14, -58), MAKE_VAL_COMPLEX(-102, -3), MAKE_VAL_COMPLEX(-30, 27), MAKE_VAL_COMPLEX(-6, 106), MAKE_VAL_COMPLEX(-9, -129), MAKE_VAL_COMPLEX(60, -82), MAKE_VAL_COMPLEX(30, -41), MAKE_VAL_COMPLEX(-7, -64), MAKE_VAL_COMPLEX(122, -28), MAKE_VAL_COMPLEX(-125, 130), MAKE_VAL_COMPLEX(8, 84), MAKE_VAL_COMPLEX(-3, 13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_76, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_76", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_76() */
/* END TEST 76 */

/*
 *  BEGIN TEST 77
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_77( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(25, 85), MAKE_VAL_COMPLEX(3, 51), MAKE_VAL_COMPLEX(-128, -157), MAKE_VAL_COMPLEX(12, -56), MAKE_VAL_COMPLEX(-87, -45), MAKE_VAL_COMPLEX(94, 46), MAKE_VAL_COMPLEX(38, -232), MAKE_VAL_COMPLEX(63, 282), MAKE_VAL_COMPLEX(86, -46), MAKE_VAL_COMPLEX(43, -23), MAKE_VAL_COMPLEX(-13, -101), MAKE_VAL_COMPLEX(-13, 66), MAKE_VAL_COMPLEX(15, 100), MAKE_VAL_COMPLEX(-28, 76), MAKE_VAL_COMPLEX(-8, 10) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_77, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_77", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_77() */
/* END TEST 77 */

/*
 *  BEGIN TEST 78
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_78( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-15, -55), MAKE_VAL_COMPLEX(97, 61), MAKE_VAL_COMPLEX(-115, 55), MAKE_VAL_COMPLEX(-60, 40), MAKE_VAL_COMPLEX(39, 117), MAKE_VAL_COMPLEX(17, -59), MAKE_VAL_COMPLEX(-78, 26), MAKE_VAL_COMPLEX(63, -81), MAKE_VAL_COMPLEX(-118, -34), MAKE_VAL_COMPLEX(-59, -17), MAKE_VAL_COMPLEX(-81, 67), MAKE_VAL_COMPLEX(-17, -71), MAKE_VAL_COMPLEX(160, -20), MAKE_VAL_COMPLEX(92, -44), MAKE_VAL_COMPLEX(16, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_78, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_78", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_78() */
/* END TEST 78 */

/*
 *  BEGIN TEST 79
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_79( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(101, 127), MAKE_VAL_COMPLEX(119, -219), MAKE_VAL_COMPLEX(-202, 269), MAKE_VAL_COMPLEX(48, -4), MAKE_VAL_COMPLEX(15, -81), MAKE_VAL_COMPLEX(48, -110), MAKE_VAL_COMPLEX(-286, -180), MAKE_VAL_COMPLEX(393, 120), MAKE_VAL_COMPLEX(58, 58), MAKE_VAL_COMPLEX(29, 29), MAKE_VAL_COMPLEX(11, 13), MAKE_VAL_COMPLEX(-5, 206), MAKE_VAL_COMPLEX(-5, -230), MAKE_VAL_COMPLEX(-68, -4), MAKE_VAL_COMPLEX(-10, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_79, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_79", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_79() */
/* END TEST 79 */

/*
 *  BEGIN TEST 80
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_80( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(30, -105), MAKE_VAL_COMPLEX(70, 343), MAKE_VAL_COMPLEX(6, -244), MAKE_VAL_COMPLEX(-58, 14), MAKE_VAL_COMPLEX(-3, 102), MAKE_VAL_COMPLEX(-39, -58), MAKE_VAL_COMPLEX(128, -60), MAKE_VAL_COMPLEX(-176, -32), MAKE_VAL_COMPLEX(-82, -60), MAKE_VAL_COMPLEX(-41, -30), MAKE_VAL_COMPLEX(-64, 51), MAKE_VAL_COMPLEX(-83, -243), MAKE_VAL_COMPLEX(139, 202), MAKE_VAL_COMPLEX(84, -8), MAKE_VAL_COMPLEX(13, 3) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_80, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_80", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_80() */
/* END TEST 80 */

/*
 *  BEGIN TEST 81
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_81( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, 81), MAKE_VAL_COMPLEX(-64, -283), MAKE_VAL_COMPLEX(-66, 214), MAKE_VAL_COMPLEX(22, -26), MAKE_VAL_COMPLEX(-33, -48), MAKE_VAL_COMPLEX(69, 46), MAKE_VAL_COMPLEX(-164, 36), MAKE_VAL_COMPLEX(230, 32), MAKE_VAL_COMPLEX(58, 0), MAKE_VAL_COMPLEX(29, 0), MAKE_VAL_COMPLEX(10, -63), MAKE_VAL_COMPLEX(107, 213), MAKE_VAL_COMPLEX(-79, -142), MAKE_VAL_COMPLEX(-36, 32), MAKE_VAL_COMPLEX(-7, 3) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_81, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_81", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_81() */
/* END TEST 81 */

/*
 *  BEGIN TEST 82
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_82( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(22, 4), MAKE_VAL_COMPLEX(-38, 24), MAKE_VAL_COMPLEX(0, -50), MAKE_VAL_COMPLEX(-4, -28), MAKE_VAL_COMPLEX(-48, -6), MAKE_VAL_COMPLEX(18, 16), MAKE_VAL_COMPLEX(4, -32), MAKE_VAL_COMPLEX(18, 36), MAKE_VAL_COMPLEX(32, -36), MAKE_VAL_COMPLEX(16, -18), MAKE_VAL_COMPLEX(-10, -40), MAKE_VAL_COMPLEX(28, 6), MAKE_VAL_COMPLEX(-20, 60), MAKE_VAL_COMPLEX(0, 40), MAKE_VAL_COMPLEX(-2, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_82, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_82", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_82() */
/* END TEST 82 */

/*
 *  BEGIN TEST 83
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_83( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_ROWMAJ, y48_VW );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-322, -89), MAKE_VAL_COMPLEX(1360, -311), MAKE_VAL_COMPLEX(-1416, -281), MAKE_VAL_COMPLEX(-18, 14), MAKE_VAL_COMPLEX(15, 36), MAKE_VAL_COMPLEX(-12, 343), MAKE_VAL_COMPLEX(-510, -912), MAKE_VAL_COMPLEX(331, 1200), MAKE_VAL_COMPLEX(-38, -8), MAKE_VAL_COMPLEX(-19, -4), MAKE_VAL_COMPLEX(68, -89), MAKE_VAL_COMPLEX(-832, 611), MAKE_VAL_COMPLEX(901, -227), MAKE_VAL_COMPLEX(28, -16), MAKE_VAL_COMPLEX(5, -1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_83, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_83", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_83() */
/* END TEST 83 */

/*
 *  BEGIN TEST 84
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_84( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(5, 182), MAKE_VAL_COMPLEX(-10, -40), MAKE_VAL_COMPLEX(-27, 45), MAKE_VAL_COMPLEX(-82, 169), MAKE_VAL_COMPLEX(-19, -186), MAKE_VAL_COMPLEX(45, 95), MAKE_VAL_COMPLEX(-81, -34), MAKE_VAL_COMPLEX(-162, -195), MAKE_VAL_COMPLEX(82, -91), MAKE_VAL_COMPLEX(83, -76), MAKE_VAL_COMPLEX(-44, -152), MAKE_VAL_COMPLEX(51, -92) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_84, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_84", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_84() */
/* END TEST 84 */

/*
 *  BEGIN TEST 85
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_85( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(29, -148), MAKE_VAL_COMPLEX(6, 22), MAKE_VAL_COMPLEX(-2, -65), MAKE_VAL_COMPLEX(64, -199), MAKE_VAL_COMPLEX(13, 159), MAKE_VAL_COMPLEX(-26, -52), MAKE_VAL_COMPLEX(95, 12), MAKE_VAL_COMPLEX(135, 184), MAKE_VAL_COMPLEX(-76, 101), MAKE_VAL_COMPLEX(-45, 43), MAKE_VAL_COMPLEX(11, 114), MAKE_VAL_COMPLEX(-65, 131) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_85, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_85", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_85() */
/* END TEST 85 */

/*
 *  BEGIN TEST 86
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_86( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(60, -32), MAKE_VAL_COMPLEX(-26, -2), MAKE_VAL_COMPLEX(-39, 31), MAKE_VAL_COMPLEX(-48, 12), MAKE_VAL_COMPLEX(-39, -3), MAKE_VAL_COMPLEX(66, -8), MAKE_VAL_COMPLEX(-24, -28), MAKE_VAL_COMPLEX(-26, 32), MAKE_VAL_COMPLEX(16, -4), MAKE_VAL_COMPLEX(-29, -65), MAKE_VAL_COMPLEX(-65, 29), MAKE_VAL_COMPLEX(47, 35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_86, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_86", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_86() */
/* END TEST 86 */

/*
 *  BEGIN TEST 87
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_87( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	oski_value_t* y_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-644, -1100), MAKE_VAL_COMPLEX(6, 42), MAKE_VAL_COMPLEX(-797, 11), MAKE_VAL_COMPLEX(-1684, -1188), MAKE_VAL_COMPLEX(1033, 535), MAKE_VAL_COMPLEX(-36, -102), MAKE_VAL_COMPLEX(280, -904), MAKE_VAL_COMPLEX(1840, -90), MAKE_VAL_COMPLEX(620, 972), MAKE_VAL_COMPLEX(-93, 69), MAKE_VAL_COMPLEX(493, 625), MAKE_VAL_COMPLEX(1159, 1273) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-4, 8);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_87, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_87", y_true, y_val, y48_LEN*y48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_87() */
/* END TEST 87 */

/*
 *  BEGIN TEST 88
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_88( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-55, -1), MAKE_VAL_COMPLEX(8, 65), MAKE_VAL_COMPLEX(-24, -65), MAKE_VAL_COMPLEX(14, -42), MAKE_VAL_COMPLEX(-63, -42), MAKE_VAL_COMPLEX(28, -27), MAKE_VAL_COMPLEX(31, 44), MAKE_VAL_COMPLEX(7, -108), MAKE_VAL_COMPLEX(70, -28), MAKE_VAL_COMPLEX(35, -14), MAKE_VAL_COMPLEX(-93, -123), MAKE_VAL_COMPLEX(167, -72), MAKE_VAL_COMPLEX(-230, 125), MAKE_VAL_COMPLEX(-28, 56), MAKE_VAL_COMPLEX(-7, 7) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_88, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_88", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_88() */
/* END TEST 88 */

/*
 *  BEGIN TEST 89
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_89( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(94, 49), MAKE_VAL_COMPLEX(-131, -83), MAKE_VAL_COMPLEX(114, -40), MAKE_VAL_COMPLEX(28, -24), MAKE_VAL_COMPLEX(-27, -57), MAKE_VAL_COMPLEX(-19, 93), MAKE_VAL_COMPLEX(35, -104), MAKE_VAL_COMPLEX(-34, 216), MAKE_VAL_COMPLEX(62, 10), MAKE_VAL_COMPLEX(31, 5), MAKE_VAL_COMPLEX(144, 21), MAKE_VAL_COMPLEX(-119, 135), MAKE_VAL_COMPLEX(80, -35), MAKE_VAL_COMPLEX(-44, 28), MAKE_VAL_COMPLEX(-8, 2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_89, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_89", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_89() */
/* END TEST 89 */

/*
 *  BEGIN TEST 90
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_90( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(40, 5), MAKE_VAL_COMPLEX(-66, 47), MAKE_VAL_COMPLEX(-5, -90), MAKE_VAL_COMPLEX(-10, -50), MAKE_VAL_COMPLEX(-87, -6), MAKE_VAL_COMPLEX(34, 27), MAKE_VAL_COMPLEX(4, -58), MAKE_VAL_COMPLEX(36, 63), MAKE_VAL_COMPLEX(54, -68), MAKE_VAL_COMPLEX(27, -34), MAKE_VAL_COMPLEX(-22, -71), MAKE_VAL_COMPLEX(51, 8), MAKE_VAL_COMPLEX(-30, 110), MAKE_VAL_COMPLEX(4, 72), MAKE_VAL_COMPLEX(-3, 11) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_90, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_90", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_90() */
/* END TEST 90 */

/*
 *  BEGIN TEST 91
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_91( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-45, -267), MAKE_VAL_COMPLEX(-60, 248), MAKE_VAL_COMPLEX(-54, -112), MAKE_VAL_COMPLEX(16, 32), MAKE_VAL_COMPLEX(60, -12), MAKE_VAL_COMPLEX(172, -32), MAKE_VAL_COMPLEX(-247, 135), MAKE_VAL_COMPLEX(484, -186), MAKE_VAL_COMPLEX(-24, 56), MAKE_VAL_COMPLEX(-12, 28), MAKE_VAL_COMPLEX(101, -337), MAKE_VAL_COMPLEX(396, 300), MAKE_VAL_COMPLEX(-325, -505), MAKE_VAL_COMPLEX(-16, -48), MAKE_VAL_COMPLEX(0, -8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_91, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_91", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_91() */
/* END TEST 91 */

/*
 *  BEGIN TEST 92
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_92( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-76, -6), MAKE_VAL_COMPLEX(57, 52), MAKE_VAL_COMPLEX(29, -98), MAKE_VAL_COMPLEX(38, -14), MAKE_VAL_COMPLEX(-6, -69), MAKE_VAL_COMPLEX(9, -28), MAKE_VAL_COMPLEX(165, -106), MAKE_VAL_COMPLEX(-186, 33), MAKE_VAL_COMPLEX(60, 34), MAKE_VAL_COMPLEX(30, 17), MAKE_VAL_COMPLEX(-40, -106), MAKE_VAL_COMPLEX(-100, -47), MAKE_VAL_COMPLEX(-64, 41), MAKE_VAL_COMPLEX(-56, 12), MAKE_VAL_COMPLEX(-9, -1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_92, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_92", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_92() */
/* END TEST 92 */

/*
 *  BEGIN TEST 93
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_93( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(47, 3), MAKE_VAL_COMPLEX(-10, -87), MAKE_VAL_COMPLEX(-24, 163), MAKE_VAL_COMPLEX(-30, 50), MAKE_VAL_COMPLEX(69, 72), MAKE_VAL_COMPLEX(-34, 9), MAKE_VAL_COMPLEX(-167, 148), MAKE_VAL_COMPLEX(159, -78), MAKE_VAL_COMPLEX(-98, 16), MAKE_VAL_COMPLEX(-49, 8), MAKE_VAL_COMPLEX(57, 157), MAKE_VAL_COMPLEX(63, 42), MAKE_VAL_COMPLEX(84, -121), MAKE_VAL_COMPLEX(52, -64), MAKE_VAL_COMPLEX(11, -7) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_93, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_93", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_93() */
/* END TEST 93 */

/*
 *  BEGIN TEST 94
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_94( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(38, -9), MAKE_VAL_COMPLEX(-44, 65), MAKE_VAL_COMPLEX(-35, -80), MAKE_VAL_COMPLEX(-26, -42), MAKE_VAL_COMPLEX(-81, 24), MAKE_VAL_COMPLEX(40, 13), MAKE_VAL_COMPLEX(-16, -54), MAKE_VAL_COMPLEX(54, 45), MAKE_VAL_COMPLEX(26, -80), MAKE_VAL_COMPLEX(13, -40), MAKE_VAL_COMPLEX(-44, -57), MAKE_VAL_COMPLEX(49, -10), MAKE_VAL_COMPLEX(10, 110), MAKE_VAL_COMPLEX(28, 64), MAKE_VAL_COMPLEX(1, 11) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_94, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_94", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_94() */
/* END TEST 94 */

/*
 *  BEGIN TEST 95
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x48', LAYOUT_COLMAJ>
 *  y : <'y48', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_95( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A48_ptr, A48_ind, A48_val,
		A48_M, A48_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y48_val,
		y48_LEN * y48_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y48_LEN, y48_VW, LAYOUT_COLMAJ, y48_LEN );
	oski_value_t* y_val = array_Duplicate( x48_val,
		x48_LEN * x48_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x48_LEN, x48_VW, LAYOUT_COLMAJ, x48_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(325, 776), MAKE_VAL_COMPLEX(-212, -1350), MAKE_VAL_COMPLEX(-111, 777), MAKE_VAL_COMPLEX(52, -36), MAKE_VAL_COMPLEX(-36, -102), MAKE_VAL_COMPLEX(-544, 302), MAKE_VAL_COMPLEX(-1377, -24), MAKE_VAL_COMPLEX(915, 939), MAKE_VAL_COMPLEX(104, 28), MAKE_VAL_COMPLEX(52, 14), MAKE_VAL_COMPLEX(241, 1058), MAKE_VAL_COMPLEX(188, 1278), MAKE_VAL_COMPLEX(-16, -493), MAKE_VAL_COMPLEX(-80, 40), MAKE_VAL_COMPLEX(-14, 2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_95, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_95", y_true, y_val, x48_LEN*x48_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_95() */
/* END TEST 95 */

#define A96_M 4    /* # of rows */
#define A96_N 5    /* # of cols */
#define A96_NNZ 4  /* # of stored non-zeros */
static oski_index_t A96_ptr[A96_N+1] = {1, 2, 3, 3, 3, 5}; /* end row pointers for A96 */
static oski_index_t A96_ind[A96_NNZ] = {2, 4, 1, 2}; /* end column indices pointers for A96 */
static oski_value_t A96_val[A96_NNZ] = {MAKE_VAL_COMPLEX(9, -6), MAKE_VAL_COMPLEX(9, 10), MAKE_VAL_COMPLEX(-10, 6), MAKE_VAL_COMPLEX(10, 2)}; /* end stored non-zero values for A96 */

#define x96_LEN 5   /* vector length */
#define x96_VW  3   /* vector width */
static oski_value_t x96_val[x96_LEN*x96_VW] = {MAKE_VAL_COMPLEX(-6, 3), MAKE_VAL_COMPLEX(1, 8), MAKE_VAL_COMPLEX(7, 8), MAKE_VAL_COMPLEX(10, -5), MAKE_VAL_COMPLEX(10, 3), MAKE_VAL_COMPLEX(8, -10), MAKE_VAL_COMPLEX(-10, 1), MAKE_VAL_COMPLEX(10, 10), MAKE_VAL_COMPLEX(-0, -3), MAKE_VAL_COMPLEX(8, 9), MAKE_VAL_COMPLEX(-9, 1), MAKE_VAL_COMPLEX(-9, 10), MAKE_VAL_COMPLEX(3, -4), MAKE_VAL_COMPLEX(-9, 5), MAKE_VAL_COMPLEX(4, 2)}; /* data */

#define y96_LEN 4   /* vector length */
#define y96_VW  3   /* vector width */
static oski_value_t y96_val[y96_LEN*y96_VW] = {MAKE_VAL_COMPLEX(-1, 9), MAKE_VAL_COMPLEX(-7, 8), MAKE_VAL_COMPLEX(-7, -7), MAKE_VAL_COMPLEX(6, -10), MAKE_VAL_COMPLEX(1, -2), MAKE_VAL_COMPLEX(1, 10), MAKE_VAL_COMPLEX(-3, -2), MAKE_VAL_COMPLEX(-10, 7), MAKE_VAL_COMPLEX(0, 5), MAKE_VAL_COMPLEX(6, -3), MAKE_VAL_COMPLEX(10, -2), MAKE_VAL_COMPLEX(-0, -5)}; /* data */

/*
 *  BEGIN TEST 96
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_96( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-59, -8), MAKE_VAL_COMPLEX(62, -194), MAKE_VAL_COMPLEX(46, 19), MAKE_VAL_COMPLEX(30, 130), MAKE_VAL_COMPLEX(-28, 121), MAKE_VAL_COMPLEX(88, -16), MAKE_VAL_COMPLEX(23, -3), MAKE_VAL_COMPLEX(38, -99), MAKE_VAL_COMPLEX(-30, -38), MAKE_VAL_COMPLEX(124, 121), MAKE_VAL_COMPLEX(-7, 202), MAKE_VAL_COMPLEX(193, 35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_96, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_96", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_96() */
/* END TEST 96 */

/*
 *  BEGIN TEST 97
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_97( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(78, -81), MAKE_VAL_COMPLEX(16, 121), MAKE_VAL_COMPLEX(17, 58), MAKE_VAL_COMPLEX(-100, -36), MAKE_VAL_COMPLEX(16, -102), MAKE_VAL_COMPLEX(-88, -85), MAKE_VAL_COMPLEX(5, 26), MAKE_VAL_COMPLEX(69, 39), MAKE_VAL_COMPLEX(35, -12), MAKE_VAL_COMPLEX(-187, -97), MAKE_VAL_COMPLEX(-95, -192), MAKE_VAL_COMPLEX(-198, 15) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_97, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_97", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_97() */
/* END TEST 97 */

/*
 *  BEGIN TEST 98
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_98( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(38, 68), MAKE_VAL_COMPLEX(-9, 91), MAKE_VAL_COMPLEX(-84, -14), MAKE_VAL_COMPLEX(-8, -100), MAKE_VAL_COMPLEX(-3, -19), MAKE_VAL_COMPLEX(57, 65), MAKE_VAL_COMPLEX(-31, 1), MAKE_VAL_COMPLEX(-35, 99), MAKE_VAL_COMPLEX(25, 35), MAKE_VAL_COMPLEX(27, -51), MAKE_VAL_COMPLEX(60, -64), MAKE_VAL_COMPLEX(-25, -35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_98, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_98", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_98() */
/* END TEST 98 */

/*
 *  BEGIN TEST 99
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_99( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(350, 65), MAKE_VAL_COMPLEX(-440, -270), MAKE_VAL_COMPLEX(25, 260), MAKE_VAL_COMPLEX(70, -90), MAKE_VAL_COMPLEX(495, 160), MAKE_VAL_COMPLEX(290, -780), MAKE_VAL_COMPLEX(-5, 65), MAKE_VAL_COMPLEX(85, 0), MAKE_VAL_COMPLEX(10, 0), MAKE_VAL_COMPLEX(305, -770), MAKE_VAL_COMPLEX(630, -305), MAKE_VAL_COMPLEX(-25, -815) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_99, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_99", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_99() */
/* END TEST 99 */

/*
 *  BEGIN TEST 100
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_100( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(53, -87), MAKE_VAL_COMPLEX(48, -72), MAKE_VAL_COMPLEX(84, -27), MAKE_VAL_COMPLEX(-10, -27), MAKE_VAL_COMPLEX(75, -18), MAKE_VAL_COMPLEX(-77, -39), MAKE_VAL_COMPLEX(65, 72), MAKE_VAL_COMPLEX(10, -133), MAKE_VAL_COMPLEX(-24, 23), MAKE_VAL_COMPLEX(30, -121), MAKE_VAL_COMPLEX(72, 64), MAKE_VAL_COMPLEX(134, 7), MAKE_VAL_COMPLEX(-14, -184), MAKE_VAL_COMPLEX(130, -98), MAKE_VAL_COMPLEX(94, 86) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_100, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_100", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_100() */
/* END TEST 100 */

/*
 *  BEGIN TEST 101
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_101( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-20, 63), MAKE_VAL_COMPLEX(-62, 25), MAKE_VAL_COMPLEX(-134, -14), MAKE_VAL_COMPLEX(-45, 67), MAKE_VAL_COMPLEX(-138, 10), MAKE_VAL_COMPLEX(39, 107), MAKE_VAL_COMPLEX(-6, -88), MAKE_VAL_COMPLEX(-80, 83), MAKE_VAL_COMPLEX(27, -5), MAKE_VAL_COMPLEX(-87, 75), MAKE_VAL_COMPLEX(-19, -79), MAKE_VAL_COMPLEX(-90, -76), MAKE_VAL_COMPLEX(0, 211), MAKE_VAL_COMPLEX(-81, 59), MAKE_VAL_COMPLEX(-120, -94) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_101, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_101", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_101() */
/* END TEST 101 */

/*
 *  BEGIN TEST 102
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_102( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 0);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_102, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_102", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_102() */
/* END TEST 102 */

/*
 *  BEGIN TEST 103
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_103( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-183, 380), MAKE_VAL_COMPLEX(27, 40), MAKE_VAL_COMPLEX(-21, -407), MAKE_VAL_COMPLEX(-274, -274), MAKE_VAL_COMPLEX(-191, -511), MAKE_VAL_COMPLEX(-287, 1), MAKE_VAL_COMPLEX(39, 49), MAKE_VAL_COMPLEX(54, -71), MAKE_VAL_COMPLEX(-2, -6), MAKE_VAL_COMPLEX(-12, -62), MAKE_VAL_COMPLEX(-3, 19), MAKE_VAL_COMPLEX(57, 21), MAKE_VAL_COMPLEX(-501, 480), MAKE_VAL_COMPLEX(-341, 369), MAKE_VAL_COMPLEX(-50, -616) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-3, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_103, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_103", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_103() */
/* END TEST 103 */

/*
 *  BEGIN TEST 104
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_104( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(104, -33), MAKE_VAL_COMPLEX(24, 11), MAKE_VAL_COMPLEX(-36, 98), MAKE_VAL_COMPLEX(45, -117), MAKE_VAL_COMPLEX(94, -124), MAKE_VAL_COMPLEX(-43, -63), MAKE_VAL_COMPLEX(-22, 10), MAKE_VAL_COMPLEX(20, 17), MAKE_VAL_COMPLEX(-3, -1), MAKE_VAL_COMPLEX(31, 7), MAKE_VAL_COMPLEX(-7, 9), MAKE_VAL_COMPLEX(-8, 24), MAKE_VAL_COMPLEX(106, -207), MAKE_VAL_COMPLEX(111, -41), MAKE_VAL_COMPLEX(68, 210) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_104, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_104", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_104() */
/* END TEST 104 */

/*
 *  BEGIN TEST 105
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_105( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-167, 72), MAKE_VAL_COMPLEX(-5, 76), MAKE_VAL_COMPLEX(121, -17), MAKE_VAL_COMPLEX(60, 52), MAKE_VAL_COMPLEX(19, 147), MAKE_VAL_COMPLEX(121, -55), MAKE_VAL_COMPLEX(-87, 11), MAKE_VAL_COMPLEX(100, 83), MAKE_VAL_COMPLEX(0, -32), MAKE_VAL_COMPLEX(66, 84), MAKE_VAL_COMPLEX(-91, 11), MAKE_VAL_COMPLEX(-81, 95), MAKE_VAL_COMPLEX(-77, 160), MAKE_VAL_COMPLEX(-205, 105), MAKE_VAL_COMPLEX(-22, -192) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_105, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_105", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_105() */
/* END TEST 105 */

/*
 *  BEGIN TEST 106
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_106( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-66, -27), MAKE_VAL_COMPLEX(-57, 64), MAKE_VAL_COMPLEX(-15, 112), MAKE_VAL_COMPLEX(110, 45), MAKE_VAL_COMPLEX(46, 101), MAKE_VAL_COMPLEX(136, -6), MAKE_VAL_COMPLEX(-78, -73), MAKE_VAL_COMPLEX(-10, 150), MAKE_VAL_COMPLEX(24, -21), MAKE_VAL_COMPLEX(-16, 127), MAKE_VAL_COMPLEX(-71, -65), MAKE_VAL_COMPLEX(-143, -2), MAKE_VAL_COMPLEX(53, -4), MAKE_VAL_COMPLEX(-103, -37), MAKE_VAL_COMPLEX(12, 46) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_106, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_106", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_106() */
/* END TEST 106 */

/*
 *  BEGIN TEST 107
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_107( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(693, -383), MAKE_VAL_COMPLEX(104, -106), MAKE_VAL_COMPLEX(-283, 500), MAKE_VAL_COMPLEX(-17, -612), MAKE_VAL_COMPLEX(238, -855), MAKE_VAL_COMPLEX(-433, -113), MAKE_VAL_COMPLEX(64, 23), MAKE_VAL_COMPLEX(-93, -68), MAKE_VAL_COMPLEX(-7, 54), MAKE_VAL_COMPLEX(5, -128), MAKE_VAL_COMPLEX(134, 6), MAKE_VAL_COMPLEX(107, -74), MAKE_VAL_COMPLEX(388, -1260), MAKE_VAL_COMPLEX(776, -488), MAKE_VAL_COMPLEX(534, 1170) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_107, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_107", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_107() */
/* END TEST 107 */

/*
 *  BEGIN TEST 108
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_108( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(38, 103), MAKE_VAL_COMPLEX(32, 98), MAKE_VAL_COMPLEX(-80, 15), MAKE_VAL_COMPLEX(112, -12), MAKE_VAL_COMPLEX(53, -110), MAKE_VAL_COMPLEX(31, 135), MAKE_VAL_COMPLEX(-14, 20), MAKE_VAL_COMPLEX(53, 216), MAKE_VAL_COMPLEX(-15, 32), MAKE_VAL_COMPLEX(161, 0), MAKE_VAL_COMPLEX(28, -71), MAKE_VAL_COMPLEX(133, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_108, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_108", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_108() */
/* END TEST 108 */

/*
 *  BEGIN TEST 109
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_109( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-59, 4), MAKE_VAL_COMPLEX(-124, -9), MAKE_VAL_COMPLEX(3, -106), MAKE_VAL_COMPLEX(-30, -102), MAKE_VAL_COMPLEX(-39, 87), MAKE_VAL_COMPLEX(-29, -14), MAKE_VAL_COMPLEX(-20, -47), MAKE_VAL_COMPLEX(-180, -142), MAKE_VAL_COMPLEX(10, 28), MAKE_VAL_COMPLEX(-86, -30), MAKE_VAL_COMPLEX(94, 57), MAKE_VAL_COMPLEX(-128, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_109, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_109", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_109() */
/* END TEST 109 */

/*
 *  BEGIN TEST 110
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_110( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-74, 10), MAKE_VAL_COMPLEX(-78, -40), MAKE_VAL_COMPLEX(42, -70), MAKE_VAL_COMPLEX(92, 28), MAKE_VAL_COMPLEX(18, 4), MAKE_VAL_COMPLEX(-78, 28), MAKE_VAL_COMPLEX(10, -28), MAKE_VAL_COMPLEX(-76, -66), MAKE_VAL_COMPLEX(-40, 10), MAKE_VAL_COMPLEX(36, 42), MAKE_VAL_COMPLEX(36, 76), MAKE_VAL_COMPLEX(40, -10) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_110, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_110", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_110() */
/* END TEST 110 */

/*
 *  BEGIN TEST 111
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_111( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-496, 350), MAKE_VAL_COMPLEX(-112, 360), MAKE_VAL_COMPLEX(-180, -74), MAKE_VAL_COMPLEX(344, 1440), MAKE_VAL_COMPLEX(1126, -112), MAKE_VAL_COMPLEX(-918, 414), MAKE_VAL_COMPLEX(-60, 148), MAKE_VAL_COMPLEX(-742, 1312), MAKE_VAL_COMPLEX(-326, -248), MAKE_VAL_COMPLEX(570, 1456), MAKE_VAL_COMPLEX(88, -114), MAKE_VAL_COMPLEX(938, 1264) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, 8);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_111, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_111", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_111() */
/* END TEST 111 */

/*
 *  BEGIN TEST 112
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_112( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(14, 153), MAKE_VAL_COMPLEX(116, 60), MAKE_VAL_COMPLEX(70, -116), MAKE_VAL_COMPLEX(97, -72), MAKE_VAL_COMPLEX(-161, -93), MAKE_VAL_COMPLEX(-20, -76), MAKE_VAL_COMPLEX(19, 51), MAKE_VAL_COMPLEX(37, -82), MAKE_VAL_COMPLEX(-8, 4), MAKE_VAL_COMPLEX(44, -76), MAKE_VAL_COMPLEX(14, 59), MAKE_VAL_COMPLEX(78, 29), MAKE_VAL_COMPLEX(-160, -40), MAKE_VAL_COMPLEX(40, 172), MAKE_VAL_COMPLEX(40, -96) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_112, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_112", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_112() */
/* END TEST 112 */

/*
 *  BEGIN TEST 113
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_113( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(52, -111), MAKE_VAL_COMPLEX(-42, -118), MAKE_VAL_COMPLEX(-32, -2), MAKE_VAL_COMPLEX(-207, 2), MAKE_VAL_COMPLEX(131, -25), MAKE_VAL_COMPLEX(-128, 56), MAKE_VAL_COMPLEX(51, 43), MAKE_VAL_COMPLEX(3, -78), MAKE_VAL_COMPLEX(-22, 14), MAKE_VAL_COMPLEX(-2, -58), MAKE_VAL_COMPLEX(50, 25), MAKE_VAL_COMPLEX(76, 1), MAKE_VAL_COMPLEX(102, 34), MAKE_VAL_COMPLEX(64, -112), MAKE_VAL_COMPLEX(-44, 44) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_113, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_113", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_113() */
/* END TEST 113 */

/*
 *  BEGIN TEST 114
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_114( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(21, -18), MAKE_VAL_COMPLEX(-12, -31), MAKE_VAL_COMPLEX(-36, -25), MAKE_VAL_COMPLEX(-35, 30), MAKE_VAL_COMPLEX(-43, -2), MAKE_VAL_COMPLEX(-22, 48), MAKE_VAL_COMPLEX(39, -14), MAKE_VAL_COMPLEX(-50, -30), MAKE_VAL_COMPLEX(3, 12), MAKE_VAL_COMPLEX(-41, -28), MAKE_VAL_COMPLEX(35, -13), MAKE_VAL_COMPLEX(26, -49), MAKE_VAL_COMPLEX(-8, 19), MAKE_VAL_COMPLEX(31, -29), MAKE_VAL_COMPLEX(-18, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_114, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_114", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_114() */
/* END TEST 114 */

/*
 *  BEGIN TEST 115
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_115( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1247, -441), MAKE_VAL_COMPLEX(446, -1096), MAKE_VAL_COMPLEX(-680, -138), MAKE_VAL_COMPLEX(-1033, -1263), MAKE_VAL_COMPLEX(516, 1468), MAKE_VAL_COMPLEX(-840, -268), MAKE_VAL_COMPLEX(40, 172), MAKE_VAL_COMPLEX(17, -103), MAKE_VAL_COMPLEX(-92, -62), MAKE_VAL_COMPLEX(-96, -122), MAKE_VAL_COMPLEX(175, 121), MAKE_VAL_COMPLEX(89, 37), MAKE_VAL_COMPLEX(330, 1320), MAKE_VAL_COMPLEX(1286, -526), MAKE_VAL_COMPLEX(-800, -28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-5, -9);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_115, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_115", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_115() */
/* END TEST 115 */

/*
 *  BEGIN TEST 116
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_116( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-76, 6), MAKE_VAL_COMPLEX(-73, 40), MAKE_VAL_COMPLEX(7, -28), MAKE_VAL_COMPLEX(-113, -87), MAKE_VAL_COMPLEX(-95, 172), MAKE_VAL_COMPLEX(-80, 38), MAKE_VAL_COMPLEX(61, -34), MAKE_VAL_COMPLEX(-93, -52), MAKE_VAL_COMPLEX(16, 19), MAKE_VAL_COMPLEX(-68, -57), MAKE_VAL_COMPLEX(51, -18), MAKE_VAL_COMPLEX(43, -93), MAKE_VAL_COMPLEX(-3, 44), MAKE_VAL_COMPLEX(61, 59), MAKE_VAL_COMPLEX(52, -98) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_116, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_116", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_116() */
/* END TEST 116 */

/*
 *  BEGIN TEST 117
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_117( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(160, -18), MAKE_VAL_COMPLEX(93, -140), MAKE_VAL_COMPLEX(-59, -96), MAKE_VAL_COMPLEX(-27, 107), MAKE_VAL_COMPLEX(-13, -248), MAKE_VAL_COMPLEX(-56, 50), MAKE_VAL_COMPLEX(63, 62), MAKE_VAL_COMPLEX(13, -108), MAKE_VAL_COMPLEX(-28, 17), MAKE_VAL_COMPLEX(8, -83), MAKE_VAL_COMPLEX(61, 42), MAKE_VAL_COMPLEX(105, 9), MAKE_VAL_COMPLEX(-49, -8), MAKE_VAL_COMPLEX(67, -83), MAKE_VAL_COMPLEX(-92, 58) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_117, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_117", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_117() */
/* END TEST 117 */

/*
 *  BEGIN TEST 118
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_118( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(30, -30), MAKE_VAL_COMPLEX(-22, -46), MAKE_VAL_COMPLEX(-58, -34), MAKE_VAL_COMPLEX(-50, 50), MAKE_VAL_COMPLEX(-66, 2), MAKE_VAL_COMPLEX(-28, 76), MAKE_VAL_COMPLEX(58, -26), MAKE_VAL_COMPLEX(-80, -40), MAKE_VAL_COMPLEX(6, 18), MAKE_VAL_COMPLEX(-66, -38), MAKE_VAL_COMPLEX(52, -24), MAKE_VAL_COMPLEX(34, -78), MAKE_VAL_COMPLEX(-10, 30), MAKE_VAL_COMPLEX(44, -48), MAKE_VAL_COMPLEX(-28, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_118, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_118", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_118() */
/* END TEST 118 */

/*
 *  BEGIN TEST 119
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_ROWMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_119( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_ROWMAJ, x96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-43, 275), MAKE_VAL_COMPLEX(148, 209), MAKE_VAL_COMPLEX(122, -121), MAKE_VAL_COMPLEX(-322, -56), MAKE_VAL_COMPLEX(350, 207), MAKE_VAL_COMPLEX(-134, 12), MAKE_VAL_COMPLEX(-40, 19), MAKE_VAL_COMPLEX(23, 14), MAKE_VAL_COMPLEX(3, -28), MAKE_VAL_COMPLEX(11, -12), MAKE_VAL_COMPLEX(-13, 38), MAKE_VAL_COMPLEX(-18, 24), MAKE_VAL_COMPLEX(32, -21), MAKE_VAL_COMPLEX(219, 115), MAKE_VAL_COMPLEX(-90, -260) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_119, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_119", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_119() */
/* END TEST 119 */

/*
 *  BEGIN TEST 120
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_120( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-163, 56), MAKE_VAL_COMPLEX(-179, -56), MAKE_VAL_COMPLEX(-54, -44), MAKE_VAL_COMPLEX(117, 115), MAKE_VAL_COMPLEX(75, -33), MAKE_VAL_COMPLEX(-85, 135), MAKE_VAL_COMPLEX(6, -10), MAKE_VAL_COMPLEX(-48, -9), MAKE_VAL_COMPLEX(-17, 11), MAKE_VAL_COMPLEX(-31, 92), MAKE_VAL_COMPLEX(-62, -60), MAKE_VAL_COMPLEX(-170, -10) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_120, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_120", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_120() */
/* END TEST 120 */

/*
 *  BEGIN TEST 121
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_121( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(181, -54), MAKE_VAL_COMPLEX(195, 70), MAKE_VAL_COMPLEX(40, 58), MAKE_VAL_COMPLEX(-137, -127), MAKE_VAL_COMPLEX(-79, 31), MAKE_VAL_COMPLEX(105, -137), MAKE_VAL_COMPLEX(-10, 16), MAKE_VAL_COMPLEX(62, 29), MAKE_VAL_COMPLEX(27, -11), MAKE_VAL_COMPLEX(25, -104), MAKE_VAL_COMPLEX(58, 40), MAKE_VAL_COMPLEX(160, 10) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_121, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_121", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_121() */
/* END TEST 121 */

/*
 *  BEGIN TEST 122
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_122( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(56, 70), MAKE_VAL_COMPLEX(7, 105), MAKE_VAL_COMPLEX(-98, 0), MAKE_VAL_COMPLEX(-28, -112), MAKE_VAL_COMPLEX(-7, -21), MAKE_VAL_COMPLEX(77, 63), MAKE_VAL_COMPLEX(-35, 7), MAKE_VAL_COMPLEX(-21, 119), MAKE_VAL_COMPLEX(35, 35), MAKE_VAL_COMPLEX(21, -63), MAKE_VAL_COMPLEX(56, -84), MAKE_VAL_COMPLEX(-35, -35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_122, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_122", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_122() */
/* END TEST 122 */

/*
 *  BEGIN TEST 123
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_123( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-451, -901), MAKE_VAL_COMPLEX(206, -1047), MAKE_VAL_COMPLEX(-225, -443), MAKE_VAL_COMPLEX(-908, 652), MAKE_VAL_COMPLEX(364, 439), MAKE_VAL_COMPLEX(-832, -169), MAKE_VAL_COMPLEX(-70, 89), MAKE_VAL_COMPLEX(-31, 184), MAKE_VAL_COMPLEX(73, 26), MAKE_VAL_COMPLEX(-747, -382), MAKE_VAL_COMPLEX(558, -1062), MAKE_VAL_COMPLEX(-455, -1520) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(2, 8);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_123, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_123", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_123() */
/* END TEST 123 */

/*
 *  BEGIN TEST 124
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_124( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(2, -69), MAKE_VAL_COMPLEX(148, 32), MAKE_VAL_COMPLEX(67, -35), MAKE_VAL_COMPLEX(-9, -83), MAKE_VAL_COMPLEX(77, -248), MAKE_VAL_COMPLEX(-64, -92), MAKE_VAL_COMPLEX(98, 152), MAKE_VAL_COMPLEX(80, -43), MAKE_VAL_COMPLEX(-11, -8), MAKE_VAL_COMPLEX(115, -178), MAKE_VAL_COMPLEX(51, 142), MAKE_VAL_COMPLEX(103, 48), MAKE_VAL_COMPLEX(-22, -24), MAKE_VAL_COMPLEX(17, 68), MAKE_VAL_COMPLEX(124, 106) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_124, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_124", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_124() */
/* END TEST 124 */

/*
 *  BEGIN TEST 125
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_125( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-62, 99), MAKE_VAL_COMPLEX(-138, 48), MAKE_VAL_COMPLEX(3, 115), MAKE_VAL_COMPLEX(109, 33), MAKE_VAL_COMPLEX(23, 278), MAKE_VAL_COMPLEX(144, -8), MAKE_VAL_COMPLEX(-198, -142), MAKE_VAL_COMPLEX(20, 143), MAKE_VAL_COMPLEX(11, -22), MAKE_VAL_COMPLEX(-35, 268), MAKE_VAL_COMPLEX(-141, -132), MAKE_VAL_COMPLEX(-193, 52), MAKE_VAL_COMPLEX(52, -16), MAKE_VAL_COMPLEX(-107, -18), MAKE_VAL_COMPLEX(-84, -86) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_125, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_125", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_125() */
/* END TEST 125 */

/*
 *  BEGIN TEST 126
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_126( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(42, -36), MAKE_VAL_COMPLEX(-24, -62), MAKE_VAL_COMPLEX(-72, -50), MAKE_VAL_COMPLEX(-70, 60), MAKE_VAL_COMPLEX(-86, -4), MAKE_VAL_COMPLEX(-44, 96), MAKE_VAL_COMPLEX(78, -28), MAKE_VAL_COMPLEX(-100, -60), MAKE_VAL_COMPLEX(6, 24), MAKE_VAL_COMPLEX(-82, -56), MAKE_VAL_COMPLEX(70, -26), MAKE_VAL_COMPLEX(52, -98), MAKE_VAL_COMPLEX(-16, 38), MAKE_VAL_COMPLEX(62, -58), MAKE_VAL_COMPLEX(-36, -8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_126, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_126", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_126() */
/* END TEST 126 */

/*
 *  BEGIN TEST 127
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_127( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-954, -259), MAKE_VAL_COMPLEX(496, -640), MAKE_VAL_COMPLEX(29, -1), MAKE_VAL_COMPLEX(-11, -117), MAKE_VAL_COMPLEX(-1329, -884), MAKE_VAL_COMPLEX(-192, -28), MAKE_VAL_COMPLEX(958, -596), MAKE_VAL_COMPLEX(96, 71), MAKE_VAL_COMPLEX(-1, -92), MAKE_VAL_COMPLEX(-951, -730), MAKE_VAL_COMPLEX(789, -218), MAKE_VAL_COMPLEX(-95, -448), MAKE_VAL_COMPLEX(26, -8), MAKE_VAL_COMPLEX(-33, 40), MAKE_VAL_COMPLEX(1364, -442) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, -8);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_127, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_127", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_127() */
/* END TEST 127 */

/*
 *  BEGIN TEST 128
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_128( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(95, 9), MAKE_VAL_COMPLEX(84, -55), MAKE_VAL_COMPLEX(87, 4), MAKE_VAL_COMPLEX(36, -93), MAKE_VAL_COMPLEX(182, -238), MAKE_VAL_COMPLEX(2, -112), MAKE_VAL_COMPLEX(17, -54), MAKE_VAL_COMPLEX(110, 7), MAKE_VAL_COMPLEX(-8, -20), MAKE_VAL_COMPLEX(226, -54), MAKE_VAL_COMPLEX(-106, 149), MAKE_VAL_COMPLEX(-43, 79), MAKE_VAL_COMPLEX(-6, -37), MAKE_VAL_COMPLEX(-24, 79), MAKE_VAL_COMPLEX(94, 198) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_128, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_128", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_128() */
/* END TEST 128 */

/*
 *  BEGIN TEST 129
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_129( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-65, 81), MAKE_VAL_COMPLEX(30, 57), MAKE_VAL_COMPLEX(39, -86), MAKE_VAL_COMPLEX(-86, -57), MAKE_VAL_COMPLEX(-120, 104), MAKE_VAL_COMPLEX(-126, -20), MAKE_VAL_COMPLEX(-23, 196), MAKE_VAL_COMPLEX(50, -127), MAKE_VAL_COMPLEX(-34, 14), MAKE_VAL_COMPLEX(-84, -40), MAKE_VAL_COMPLEX(102, -21), MAKE_VAL_COMPLEX(165, 67), MAKE_VAL_COMPLEX(-44, -13), MAKE_VAL_COMPLEX(76, 57), MAKE_VAL_COMPLEX(-58, -250) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_129, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_129", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_129() */
/* END TEST 129 */

/*
 *  BEGIN TEST 130
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_130( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(36, -18), MAKE_VAL_COMPLEX(-6, -48), MAKE_VAL_COMPLEX(-42, -48), MAKE_VAL_COMPLEX(-60, 30), MAKE_VAL_COMPLEX(-60, -18), MAKE_VAL_COMPLEX(-48, 60), MAKE_VAL_COMPLEX(60, -6), MAKE_VAL_COMPLEX(-60, -60), MAKE_VAL_COMPLEX(0, 18), MAKE_VAL_COMPLEX(-48, -54), MAKE_VAL_COMPLEX(54, -6), MAKE_VAL_COMPLEX(54, -60), MAKE_VAL_COMPLEX(-18, 24), MAKE_VAL_COMPLEX(54, -30), MAKE_VAL_COMPLEX(-24, -12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_130, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_130", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_130() */
/* END TEST 130 */

/*
 *  BEGIN TEST 131
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_131( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_ROWMAJ, y96_VW );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(831, 105), MAKE_VAL_COMPLEX(519, -498), MAKE_VAL_COMPLEX(36, -63), MAKE_VAL_COMPLEX(15, -60), MAKE_VAL_COMPLEX(1230, -924), MAKE_VAL_COMPLEX(36, -30), MAKE_VAL_COMPLEX(792, -447), MAKE_VAL_COMPLEX(-21, -48), MAKE_VAL_COMPLEX(48, 18), MAKE_VAL_COMPLEX(978, -36), MAKE_VAL_COMPLEX(-609, 414), MAKE_VAL_COMPLEX(-129, -303), MAKE_VAL_COMPLEX(-39, 12), MAKE_VAL_COMPLEX(45, 24), MAKE_VAL_COMPLEX(-270, 1410) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_131, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_131", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_131() */
/* END TEST 131 */

/*
 *  BEGIN TEST 132
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_132( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-188, 35), MAKE_VAL_COMPLEX(-4, 80), MAKE_VAL_COMPLEX(49, -48), MAKE_VAL_COMPLEX(15, 109), MAKE_VAL_COMPLEX(-111, -47), MAKE_VAL_COMPLEX(-5, -14), MAKE_VAL_COMPLEX(21, -13), MAKE_VAL_COMPLEX(-159, -157), MAKE_VAL_COMPLEX(-96, 10), MAKE_VAL_COMPLEX(-21, 140), MAKE_VAL_COMPLEX(27, 64), MAKE_VAL_COMPLEX(-155, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_132, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_132", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_132() */
/* END TEST 132 */

/*
 *  BEGIN TEST 133
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_133( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(170, -37), MAKE_VAL_COMPLEX(-12, -94), MAKE_VAL_COMPLEX(-35, 34), MAKE_VAL_COMPLEX(5, -97), MAKE_VAL_COMPLEX(115, 49), MAKE_VAL_COMPLEX(-15, 16), MAKE_VAL_COMPLEX(-17, 7), MAKE_VAL_COMPLEX(145, 137), MAKE_VAL_COMPLEX(86, -10), MAKE_VAL_COMPLEX(27, -128), MAKE_VAL_COMPLEX(-23, -44), MAKE_VAL_COMPLEX(165, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_133, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_133", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_133() */
/* END TEST 133 */

/*
 *  BEGIN TEST 134
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_134( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(46, -4), MAKE_VAL_COMPLEX(47, 27), MAKE_VAL_COMPLEX(-28, 42), MAKE_VAL_COMPLEX(-56, -20), MAKE_VAL_COMPLEX(-11, -3), MAKE_VAL_COMPLEX(49, -15), MAKE_VAL_COMPLEX(-7, 17), MAKE_VAL_COMPLEX(45, 43), MAKE_VAL_COMPLEX(25, -5), MAKE_VAL_COMPLEX(-21, -27), MAKE_VAL_COMPLEX(-20, -48), MAKE_VAL_COMPLEX(-25, 5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_134, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_134", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_134() */
/* END TEST 134 */

/*
 *  BEGIN TEST 135
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_135( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	oski_value_t* y_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-602, 779), MAKE_VAL_COMPLEX(963, 378), MAKE_VAL_COMPLEX(131, 20), MAKE_VAL_COMPLEX(25, 825), MAKE_VAL_COMPLEX(-1014, 329), MAKE_VAL_COMPLEX(195, -543), MAKE_VAL_COMPLEX(128, 9), MAKE_VAL_COMPLEX(-1051, -122), MAKE_VAL_COMPLEX(-356, 315), MAKE_VAL_COMPLEX(202, 876), MAKE_VAL_COMPLEX(-36, -1), MAKE_VAL_COMPLEX(-1100, 1000) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_135, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_135", y_true, y_val, y96_LEN*y96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_135() */
/* END TEST 135 */

/*
 *  BEGIN TEST 136
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_136( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(29, 138), MAKE_VAL_COMPLEX(182, -67), MAKE_VAL_COMPLEX(-2, -82), MAKE_VAL_COMPLEX(-69, -35), MAKE_VAL_COMPLEX(-165, -95), MAKE_VAL_COMPLEX(-20, 92), MAKE_VAL_COMPLEX(-104, 18), MAKE_VAL_COMPLEX(-3, -102), MAKE_VAL_COMPLEX(-25, 22), MAKE_VAL_COMPLEX(-3, 43), MAKE_VAL_COMPLEX(86, -18), MAKE_VAL_COMPLEX(151, -53), MAKE_VAL_COMPLEX(-25, 3), MAKE_VAL_COMPLEX(70, 15), MAKE_VAL_COMPLEX(26, -98) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_136, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_136", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_136() */
/* END TEST 136 */

/*
 *  BEGIN TEST 137
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_137( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(55, -90), MAKE_VAL_COMPLEX(-94, -9), MAKE_VAL_COMPLEX(42, -66), MAKE_VAL_COMPLEX(-71, -45), MAKE_VAL_COMPLEX(121, -49), MAKE_VAL_COMPLEX(-164, -108), MAKE_VAL_COMPLEX(196, 94), MAKE_VAL_COMPLEX(43, -98), MAKE_VAL_COMPLEX(-11, 2), MAKE_VAL_COMPLEX(47, -211), MAKE_VAL_COMPLEX(-2, 118), MAKE_VAL_COMPLEX(41, 81), MAKE_VAL_COMPLEX(-47, -7), MAKE_VAL_COMPLEX(62, 53), MAKE_VAL_COMPLEX(-34, 34) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_137, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_137", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_137() */
/* END TEST 137 */

/*
 *  BEGIN TEST 138
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_138( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(39, 48), MAKE_VAL_COMPLEX(70, -25), MAKE_VAL_COMPLEX(58, -79), MAKE_VAL_COMPLEX(-65, -80), MAKE_VAL_COMPLEX(7, -96), MAKE_VAL_COMPLEX(-106, -52), MAKE_VAL_COMPLEX(29, 88), MAKE_VAL_COMPLEX(70, -110), MAKE_VAL_COMPLEX(-27, 6), MAKE_VAL_COMPLEX(65, -90), MAKE_VAL_COMPLEX(27, 79), MAKE_VAL_COMPLEX(108, 61), MAKE_VAL_COMPLEX(-42, -19), MAKE_VAL_COMPLEX(63, 71), MAKE_VAL_COMPLEX(10, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_138, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_138", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_138() */
/* END TEST 138 */

/*
 *  BEGIN TEST 139
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_139( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-952, 716), MAKE_VAL_COMPLEX(1154, 838), MAKE_VAL_COMPLEX(-59, -117), MAKE_VAL_COMPLEX(77, 27), MAKE_VAL_COMPLEX(-759, -1105), MAKE_VAL_COMPLEX(-94, 1138), MAKE_VAL_COMPLEX(-877, -1337), MAKE_VAL_COMPLEX(-87, -55), MAKE_VAL_COMPLEX(-110, -6), MAKE_VAL_COMPLEX(-1019, 819), MAKE_VAL_COMPLEX(700, -186), MAKE_VAL_COMPLEX(743, -21), MAKE_VAL_COMPLEX(81, 85), MAKE_VAL_COMPLEX(65, -87), MAKE_VAL_COMPLEX(702, -222) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(7, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_139, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_139", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_139() */
/* END TEST 139 */

/*
 *  BEGIN TEST 140
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_140( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-100, 63), MAKE_VAL_COMPLEX(-21, -146), MAKE_VAL_COMPLEX(25, -35), MAKE_VAL_COMPLEX(-14, -50), MAKE_VAL_COMPLEX(22, -30), MAKE_VAL_COMPLEX(-90, 62), MAKE_VAL_COMPLEX(-15, 213), MAKE_VAL_COMPLEX(37, -42), MAKE_VAL_COMPLEX(-22, 7), MAKE_VAL_COMPLEX(44, 80), MAKE_VAL_COMPLEX(76, 50), MAKE_VAL_COMPLEX(-4, -12), MAKE_VAL_COMPLEX(-6, -14), MAKE_VAL_COMPLEX(20, 31), MAKE_VAL_COMPLEX(92, -108) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_140, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_140", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_140() */
/* END TEST 140 */

/*
 *  BEGIN TEST 141
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_141( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(88, -87), MAKE_VAL_COMPLEX(-11, 150), MAKE_VAL_COMPLEX(-57, 63), MAKE_VAL_COMPLEX(34, 90), MAKE_VAL_COMPLEX(-34, 70), MAKE_VAL_COMPLEX(130, -30), MAKE_VAL_COMPLEX(11, -253), MAKE_VAL_COMPLEX(-77, 82), MAKE_VAL_COMPLEX(34, -7), MAKE_VAL_COMPLEX(-80, -48), MAKE_VAL_COMPLEX(-80, -86), MAKE_VAL_COMPLEX(-36, -24), MAKE_VAL_COMPLEX(22, 26), MAKE_VAL_COMPLEX(-40, -67), MAKE_VAL_COMPLEX(-100, 124) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_141, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_141", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_141() */
/* END TEST 141 */

/*
 *  BEGIN TEST 142
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_142( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(54, 3), MAKE_VAL_COMPLEX(25, -60), MAKE_VAL_COMPLEX(-17, -84), MAKE_VAL_COMPLEX(-90, -5), MAKE_VAL_COMPLEX(-58, -61), MAKE_VAL_COMPLEX(-96, 38), MAKE_VAL_COMPLEX(74, 33), MAKE_VAL_COMPLEX(-30, -110), MAKE_VAL_COMPLEX(-12, 21), MAKE_VAL_COMPLEX(-20, -95), MAKE_VAL_COMPLEX(67, 29), MAKE_VAL_COMPLEX(103, -34), MAKE_VAL_COMPLEX(-37, 16), MAKE_VAL_COMPLEX(83, 1), MAKE_VAL_COMPLEX(-20, -30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_142, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_142", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_142() */
/* END TEST 142 */

/*
 *  BEGIN TEST 143
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x96', LAYOUT_COLMAJ>
 *  y : <'y96', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_143( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A96_ptr, A96_ind, A96_val,
		A96_M, A96_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y96_val,
		y96_LEN * y96_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y96_LEN, y96_VW, LAYOUT_COLMAJ, y96_LEN );
	oski_value_t* y_val = array_Duplicate( x96_val,
		x96_LEN * x96_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x96_LEN, x96_VW, LAYOUT_COLMAJ, x96_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(104, 536), MAKE_VAL_COMPLEX(-826, 74), MAKE_VAL_COMPLEX(-147, 21), MAKE_VAL_COMPLEX(-79, 65), MAKE_VAL_COMPLEX(-31, 9), MAKE_VAL_COMPLEX(434, 470), MAKE_VAL_COMPLEX(909, 191), MAKE_VAL_COMPLEX(-153, 13), MAKE_VAL_COMPLEX(46, 78), MAKE_VAL_COMPLEX(449, 65), MAKE_VAL_COMPLEX(198, -416), MAKE_VAL_COMPLEX(-291, 39), MAKE_VAL_COMPLEX(7, -3), MAKE_VAL_COMPLEX(3, -103), MAKE_VAL_COMPLEX(-418, -498) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_143, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_143", y_true, y_val, x96_LEN*x96_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_143() */
/* END TEST 143 */

#define A144_M 4    /* # of rows */
#define A144_N 5    /* # of cols */
#define A144_NNZ 4  /* # of stored non-zeros */
static oski_index_t A144_ptr[A144_N+1] = {0, 0, 2, 2, 3, 4}; /* end row pointers for A144 */
static oski_index_t A144_ind[A144_NNZ] = {0, 3, 2, 3}; /* end column indices pointers for A144 */
static oski_value_t A144_val[A144_NNZ] = {MAKE_VAL_COMPLEX(4, -10), MAKE_VAL_COMPLEX(3, 7), MAKE_VAL_COMPLEX(6, 0), MAKE_VAL_COMPLEX(-10, 10)}; /* end stored non-zero values for A144 */

#define x144_LEN 5   /* vector length */
#define x144_VW  3   /* vector width */
static oski_value_t x144_val[x144_LEN*x144_VW] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(4, 8), MAKE_VAL_COMPLEX(2, 1), MAKE_VAL_COMPLEX(4, -3), MAKE_VAL_COMPLEX(-10, 6), MAKE_VAL_COMPLEX(-5, -1), MAKE_VAL_COMPLEX(-7, 10), MAKE_VAL_COMPLEX(-2, 2), MAKE_VAL_COMPLEX(10, 0), MAKE_VAL_COMPLEX(-6, 7), MAKE_VAL_COMPLEX(-8, 4), MAKE_VAL_COMPLEX(-2, -10), MAKE_VAL_COMPLEX(-7, 1), MAKE_VAL_COMPLEX(-8, 2), MAKE_VAL_COMPLEX(-3, 6)}; /* data */

#define y144_LEN 4   /* vector length */
#define y144_VW  3   /* vector width */
static oski_value_t y144_val[y144_LEN*y144_VW] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(-2, -6), MAKE_VAL_COMPLEX(10, -4), MAKE_VAL_COMPLEX(-8, 7), MAKE_VAL_COMPLEX(-10, 10), MAKE_VAL_COMPLEX(-1, -6), MAKE_VAL_COMPLEX(-3, -10), MAKE_VAL_COMPLEX(10, -7), MAKE_VAL_COMPLEX(10, -4), MAKE_VAL_COMPLEX(-3, 5), MAKE_VAL_COMPLEX(6, -3), MAKE_VAL_COMPLEX(2, 0)}; /* data */

/*
 *  BEGIN TEST 144
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_144( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-10, -38), MAKE_VAL_COMPLEX(36, 68), MAKE_VAL_COMPLEX(82, 61), MAKE_VAL_COMPLEX(-103, 20), MAKE_VAL_COMPLEX(-150, 46), MAKE_VAL_COMPLEX(16, -60), MAKE_VAL_COMPLEX(-20, -53), MAKE_VAL_COMPLEX(75, 13), MAKE_VAL_COMPLEX(108, -46), MAKE_VAL_COMPLEX(35, -24), MAKE_VAL_COMPLEX(49, -145), MAKE_VAL_COMPLEX(-22, -128) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_144, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_144", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_144() */
/* END TEST 144 */

/*
 *  BEGIN TEST 145
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_145( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(21, 43), MAKE_VAL_COMPLEX(-70, -110), MAKE_VAL_COMPLEX(-14, -123), MAKE_VAL_COMPLEX(60, 60), MAKE_VAL_COMPLEX(100, 64), MAKE_VAL_COMPLEX(-42, 15), MAKE_VAL_COMPLEX(-34, -18), MAKE_VAL_COMPLEX(-16, -99), MAKE_VAL_COMPLEX(-40, -16), MAKE_VAL_COMPLEX(-44, 73), MAKE_VAL_COMPLEX(-10, 103), MAKE_VAL_COMPLEX(38, 122) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_145, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_145", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_145() */
/* END TEST 145 */

/*
 *  BEGIN TEST 146
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_146( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 6), MAKE_VAL_COMPLEX(12, -24), MAKE_VAL_COMPLEX(42, 18), MAKE_VAL_COMPLEX(-45, -3), MAKE_VAL_COMPLEX(-60, 0), MAKE_VAL_COMPLEX(15, -21), MAKE_VAL_COMPLEX(21, -39), MAKE_VAL_COMPLEX(51, 9), MAKE_VAL_COMPLEX(42, 18), MAKE_VAL_COMPLEX(-24, 6), MAKE_VAL_COMPLEX(27, 9), MAKE_VAL_COMPLEX(6, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_146, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_146", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_146() */
/* END TEST 146 */

/*
 *  BEGIN TEST 147
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_147( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-290, 410), MAKE_VAL_COMPLEX(820, -940), MAKE_VAL_COMPLEX(467, -16), MAKE_VAL_COMPLEX(-40, -85), MAKE_VAL_COMPLEX(102, -66), MAKE_VAL_COMPLEX(58, 66), MAKE_VAL_COMPLEX(687, 24), MAKE_VAL_COMPLEX(467, 279), MAKE_VAL_COMPLEX(-438, 444), MAKE_VAL_COMPLEX(-910, -325), MAKE_VAL_COMPLEX(-931, 1073), MAKE_VAL_COMPLEX(-736, 1118) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_147, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_147", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_147() */
/* END TEST 147 */

/*
 *  BEGIN TEST 148
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_148( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(6, -10), MAKE_VAL_COMPLEX(11, -6), MAKE_VAL_COMPLEX(-41, -9), MAKE_VAL_COMPLEX(-33, 49), MAKE_VAL_COMPLEX(4, -103), MAKE_VAL_COMPLEX(7, -3), MAKE_VAL_COMPLEX(12, -5), MAKE_VAL_COMPLEX(10, -14), MAKE_VAL_COMPLEX(-14, -49), MAKE_VAL_COMPLEX(70, -37), MAKE_VAL_COMPLEX(52, -22), MAKE_VAL_COMPLEX(-19, -73), MAKE_VAL_COMPLEX(-28, 98), MAKE_VAL_COMPLEX(-14, 23) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_148, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_148", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_148() */
/* END TEST 148 */

/*
 *  BEGIN TEST 149
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_149( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(54, -10), MAKE_VAL_COMPLEX(-2, -7), MAKE_VAL_COMPLEX(24, -22), MAKE_VAL_COMPLEX(65, 27), MAKE_VAL_COMPLEX(-16, 137), MAKE_VAL_COMPLEX(56, 62), MAKE_VAL_COMPLEX(0, 21), MAKE_VAL_COMPLEX(0, -56), MAKE_VAL_COMPLEX(57, 98), MAKE_VAL_COMPLEX(-50, 97), MAKE_VAL_COMPLEX(-124, 26), MAKE_VAL_COMPLEX(19, 123), MAKE_VAL_COMPLEX(34, -40), MAKE_VAL_COMPLEX(53, 4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_149, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_149", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_149() */
/* END TEST 149 */

/*
 *  BEGIN TEST 150
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_150( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(-4, 32), MAKE_VAL_COMPLEX(4, 7), MAKE_VAL_COMPLEX(18, -1), MAKE_VAL_COMPLEX(-42, -2), MAKE_VAL_COMPLEX(-13, -13), MAKE_VAL_COMPLEX(-41, 16), MAKE_VAL_COMPLEX(-10, 2), MAKE_VAL_COMPLEX(30, 20), MAKE_VAL_COMPLEX(-32, 9), MAKE_VAL_COMPLEX(-32, -4), MAKE_VAL_COMPLEX(14, -34), MAKE_VAL_COMPLEX(-23, -11), MAKE_VAL_COMPLEX(-28, -10), MAKE_VAL_COMPLEX(-21, 12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_150, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_150", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_150() */
/* END TEST 150 */

/*
 *  BEGIN TEST 151
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_151( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, 0), MAKE_VAL_COMPLEX(60, 88), MAKE_VAL_COMPLEX(-24, 95), MAKE_VAL_COMPLEX(292, -211), MAKE_VAL_COMPLEX(-82, -446), MAKE_VAL_COMPLEX(585, 661), MAKE_VAL_COMPLEX(9, 98), MAKE_VAL_COMPLEX(-36, 112), MAKE_VAL_COMPLEX(34, 104), MAKE_VAL_COMPLEX(400, 241), MAKE_VAL_COMPLEX(-190, 678), MAKE_VAL_COMPLEX(-222, 442), MAKE_VAL_COMPLEX(549, 353), MAKE_VAL_COMPLEX(-420, -722), MAKE_VAL_COMPLEX(-33, -204) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 6);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_151, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_151", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_151() */
/* END TEST 151 */

/*
 *  BEGIN TEST 152
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_152( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(54, 6), MAKE_VAL_COMPLEX(23, -10), MAKE_VAL_COMPLEX(13, 25), MAKE_VAL_COMPLEX(29, -11), MAKE_VAL_COMPLEX(60, 85), MAKE_VAL_COMPLEX(19, 65), MAKE_VAL_COMPLEX(12, 11), MAKE_VAL_COMPLEX(50, -54), MAKE_VAL_COMPLEX(-10, 3), MAKE_VAL_COMPLEX(54, 11), MAKE_VAL_COMPLEX(4, -54), MAKE_VAL_COMPLEX(57, 19), MAKE_VAL_COMPLEX(-112, 18), MAKE_VAL_COMPLEX(-2, 19) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_152, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_152", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_152() */
/* END TEST 152 */

/*
 *  BEGIN TEST 153
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_153( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(-30, -58), MAKE_VAL_COMPLEX(-26, -4), MAKE_VAL_COMPLEX(-44, -33), MAKE_VAL_COMPLEX(41, 37), MAKE_VAL_COMPLEX(-45, -56), MAKE_VAL_COMPLEX(59, -70), MAKE_VAL_COMPLEX(6, -9), MAKE_VAL_COMPLEX(-90, 4), MAKE_VAL_COMPLEX(69, -1), MAKE_VAL_COMPLEX(-2, 13), MAKE_VAL_COMPLEX(-46, 104), MAKE_VAL_COMPLEX(-24, 12), MAKE_VAL_COMPLEX(154, 14), MAKE_VAL_COMPLEX(44, -28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_153, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_153", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_153() */
/* END TEST 153 */

/*
 *  BEGIN TEST 154
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_154( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(-40, 0), MAKE_VAL_COMPLEX(-8, 6), MAKE_VAL_COMPLEX(4, 22), MAKE_VAL_COMPLEX(-4, -52), MAKE_VAL_COMPLEX(14, -18), MAKE_VAL_COMPLEX(-26, -48), MAKE_VAL_COMPLEX(-4, -12), MAKE_VAL_COMPLEX(-20, 40), MAKE_VAL_COMPLEX(-16, -38), MAKE_VAL_COMPLEX(0, -40), MAKE_VAL_COMPLEX(44, 12), MAKE_VAL_COMPLEX(10, -30), MAKE_VAL_COMPLEX(8, -36), MAKE_VAL_COMPLEX(-18, -24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_154, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_154", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_154() */
/* END TEST 154 */

/*
 *  BEGIN TEST 155
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_155( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(8, 4), MAKE_VAL_COMPLEX(-96, -36), MAKE_VAL_COMPLEX(37, -34), MAKE_VAL_COMPLEX(191, 358), MAKE_VAL_COMPLEX(62, -682), MAKE_VAL_COMPLEX(665, 183), MAKE_VAL_COMPLEX(-80, -143), MAKE_VAL_COMPLEX(40, -84), MAKE_VAL_COMPLEX(12, 26), MAKE_VAL_COMPLEX(-261, -358), MAKE_VAL_COMPLEX(310, -474), MAKE_VAL_COMPLEX(402, -242), MAKE_VAL_COMPLEX(461, -333), MAKE_VAL_COMPLEX(-582, -64), MAKE_VAL_COMPLEX(-190, -125) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_155, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_155", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_155() */
/* END TEST 155 */

/*
 *  BEGIN TEST 156
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_156( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-13, -63), MAKE_VAL_COMPLEX(-22, 39), MAKE_VAL_COMPLEX(-117, 12), MAKE_VAL_COMPLEX(169, -41), MAKE_VAL_COMPLEX(134, 142), MAKE_VAL_COMPLEX(-41, 42), MAKE_VAL_COMPLEX(-95, 94), MAKE_VAL_COMPLEX(-112, -173), MAKE_VAL_COMPLEX(-102, 7), MAKE_VAL_COMPLEX(40, -8), MAKE_VAL_COMPLEX(-50, -81), MAKE_VAL_COMPLEX(-50, -150) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_156, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_156", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_156() */
/* END TEST 156 */

/*
 *  BEGIN TEST 157
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_157( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(11, 65), MAKE_VAL_COMPLEX(34, -43), MAKE_VAL_COMPLEX(125, 8), MAKE_VAL_COMPLEX(-183, 25), MAKE_VAL_COMPLEX(-154, -162), MAKE_VAL_COMPLEX(53, -44), MAKE_VAL_COMPLEX(115, -100), MAKE_VAL_COMPLEX(126, 193), MAKE_VAL_COMPLEX(110, 13), MAKE_VAL_COMPLEX(-50, 2), MAKE_VAL_COMPLEX(56, 93), MAKE_VAL_COMPLEX(50, 154) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_157, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_157", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_157() */
/* END TEST 157 */

/*
 *  BEGIN TEST 158
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_158( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(5, 7), MAKE_VAL_COMPLEX(-6, -38), MAKE_VAL_COMPLEX(64, -14), MAKE_VAL_COMPLEX(-55, 34), MAKE_VAL_COMPLEX(-70, 50), MAKE_VAL_COMPLEX(0, -37), MAKE_VAL_COMPLEX(-8, -63), MAKE_VAL_COMPLEX(67, -32), MAKE_VAL_COMPLEX(64, -14), MAKE_VAL_COMPLEX(-23, 27), MAKE_VAL_COMPLEX(39, -12), MAKE_VAL_COMPLEX(12, 2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_158, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_158", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_158() */
/* END TEST 158 */

/*
 *  BEGIN TEST 159
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_159( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-23, -59), MAKE_VAL_COMPLEX(26, 43), MAKE_VAL_COMPLEX(-119, 94), MAKE_VAL_COMPLEX(144, -118), MAKE_VAL_COMPLEX(94, 42), MAKE_VAL_COMPLEX(4, 53), MAKE_VAL_COMPLEX(-16, 103), MAKE_VAL_COMPLEX(-93, -82), MAKE_VAL_COMPLEX(-104, 89), MAKE_VAL_COMPLEX(14, -44), MAKE_VAL_COMPLEX(-47, -30), MAKE_VAL_COMPLEX(-56, -136) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_159, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_159", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_159() */
/* END TEST 159 */

/*
 *  BEGIN TEST 160
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_160( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(-6, 58), MAKE_VAL_COMPLEX(18, 5), MAKE_VAL_COMPLEX(-35, -54), MAKE_VAL_COMPLEX(76, 193), MAKE_VAL_COMPLEX(-20, -112), MAKE_VAL_COMPLEX(-45, 32), MAKE_VAL_COMPLEX(-17, -4), MAKE_VAL_COMPLEX(56, 17), MAKE_VAL_COMPLEX(8, 6), MAKE_VAL_COMPLEX(-56, -63), MAKE_VAL_COMPLEX(48, -72), MAKE_VAL_COMPLEX(-27, -159), MAKE_VAL_COMPLEX(-74, 164), MAKE_VAL_COMPLEX(-47, 44) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_160, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_160", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_160() */
/* END TEST 160 */

/*
 *  BEGIN TEST 161
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_161( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(86, 22), MAKE_VAL_COMPLEX(10, -1), MAKE_VAL_COMPLEX(71, 2), MAKE_VAL_COMPLEX(-172, -81), MAKE_VAL_COMPLEX(-44, 120), MAKE_VAL_COMPLEX(1, 116), MAKE_VAL_COMPLEX(1, 36), MAKE_VAL_COMPLEX(64, -57), MAKE_VAL_COMPLEX(-52, 102), MAKE_VAL_COMPLEX(-24, 143), MAKE_VAL_COMPLEX(-112, -40), MAKE_VAL_COMPLEX(-53, 199), MAKE_VAL_COMPLEX(-14, -108), MAKE_VAL_COMPLEX(35, 40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_161, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_161", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_161() */
/* END TEST 161 */

/*
 *  BEGIN TEST 162
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_162( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(-64, -48), MAKE_VAL_COMPLEX(-20, 0), MAKE_VAL_COMPLEX(-20, 40), MAKE_VAL_COMPLEX(56, -88), MAKE_VAL_COMPLEX(44, -12), MAKE_VAL_COMPLEX(16, -108), MAKE_VAL_COMPLEX(8, -24), MAKE_VAL_COMPLEX(-80, 40), MAKE_VAL_COMPLEX(20, -80), MAKE_VAL_COMPLEX(48, -64), MAKE_VAL_COMPLEX(56, 72), MAKE_VAL_COMPLEX(52, -36), MAKE_VAL_COMPLEX(56, -48), MAKE_VAL_COMPLEX(0, -60) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_162, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_162", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_162() */
/* END TEST 162 */

/*
 *  BEGIN TEST 163
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_163( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(2, -2), MAKE_VAL_COMPLEX(60, -80), MAKE_VAL_COMPLEX(-15, -46), MAKE_VAL_COMPLEX(-153, 110), MAKE_VAL_COMPLEX(500, -234), MAKE_VAL_COMPLEX(-163, 47), MAKE_VAL_COMPLEX(138, -37), MAKE_VAL_COMPLEX(14, 8), MAKE_VAL_COMPLEX(-86, -102), MAKE_VAL_COMPLEX(77, -106), MAKE_VAL_COMPLEX(-34, 56), MAKE_VAL_COMPLEX(-110, 22), MAKE_VAL_COMPLEX(-235, 35), MAKE_VAL_COMPLEX(422, 116), MAKE_VAL_COMPLEX(118, 19) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_163, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_163", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_163() */
/* END TEST 163 */

/*
 *  BEGIN TEST 164
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_164( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(22, -86), MAKE_VAL_COMPLEX(2, -28), MAKE_VAL_COMPLEX(-39, 77), MAKE_VAL_COMPLEX(-32, -125), MAKE_VAL_COMPLEX(115, 123), MAKE_VAL_COMPLEX(146, -28), MAKE_VAL_COMPLEX(29, -10), MAKE_VAL_COMPLEX(-74, -83), MAKE_VAL_COMPLEX(156, -25), MAKE_VAL_COMPLEX(88, -35), MAKE_VAL_COMPLEX(-26, 78), MAKE_VAL_COMPLEX(214, 58), MAKE_VAL_COMPLEX(-90, 18), MAKE_VAL_COMPLEX(52, -44) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_164, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_164", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_164() */
/* END TEST 164 */

/*
 *  BEGIN TEST 165
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_165( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(-90, -30), MAKE_VAL_COMPLEX(-33, 15), MAKE_VAL_COMPLEX(-18, -28), MAKE_VAL_COMPLEX(176, 25), MAKE_VAL_COMPLEX(-39, -113), MAKE_VAL_COMPLEX(-51, -129), MAKE_VAL_COMPLEX(-1, -22), MAKE_VAL_COMPLEX(-76, 93), MAKE_VAL_COMPLEX(-73, -86), MAKE_VAL_COMPLEX(28, -33), MAKE_VAL_COMPLEX(66, 70), MAKE_VAL_COMPLEX(-110, -80), MAKE_VAL_COMPLEX(208, -56), MAKE_VAL_COMPLEX(-13, -49) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_165, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_165", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_165() */
/* END TEST 165 */

/*
 *  BEGIN TEST 166
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_166( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(32, -36), MAKE_VAL_COMPLEX(1, -12), MAKE_VAL_COMPLEX(-23, -14), MAKE_VAL_COMPLEX(50, 38), MAKE_VAL_COMPLEX(5, 27), MAKE_VAL_COMPLEX(64, 15), MAKE_VAL_COMPLEX(14, 6), MAKE_VAL_COMPLEX(-20, -50), MAKE_VAL_COMPLEX(47, 16), MAKE_VAL_COMPLEX(36, 32), MAKE_VAL_COMPLEX(-46, 30), MAKE_VAL_COMPLEX(19, 33), MAKE_VAL_COMPLEX(26, 36), MAKE_VAL_COMPLEX(36, 3) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_166, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_166", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_166() */
/* END TEST 166 */

/*
 *  BEGIN TEST 167
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_ROWMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_167( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_ROWMAJ, x144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-7, -11), MAKE_VAL_COMPLEX(58, -54), MAKE_VAL_COMPLEX(-106, 27), MAKE_VAL_COMPLEX(31, -772), MAKE_VAL_COMPLEX(1100, 1667), MAKE_VAL_COMPLEX(-586, -870), MAKE_VAL_COMPLEX(-151, -36), MAKE_VAL_COMPLEX(-3, 82), MAKE_VAL_COMPLEX(-70, 75), MAKE_VAL_COMPLEX(-538, 6), MAKE_VAL_COMPLEX(-78, 567), MAKE_VAL_COMPLEX(-316, 84), MAKE_VAL_COMPLEX(-1329, -433), MAKE_VAL_COMPLEX(1466, 560), MAKE_VAL_COMPLEX(107, 196) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-9, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_167, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_167", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_167() */
/* END TEST 167 */

/*
 *  BEGIN TEST 168
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_168( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(107, -3), MAKE_VAL_COMPLEX(33, 67), MAKE_VAL_COMPLEX(-48, -78), MAKE_VAL_COMPLEX(-39, 88), MAKE_VAL_COMPLEX(-57, 120), MAKE_VAL_COMPLEX(-28, -55), MAKE_VAL_COMPLEX(-28, -88), MAKE_VAL_COMPLEX(117, -84), MAKE_VAL_COMPLEX(13, -49), MAKE_VAL_COMPLEX(-9, -62), MAKE_VAL_COMPLEX(-52, -191), MAKE_VAL_COMPLEX(42, -138) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_168, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_168", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_168() */
/* END TEST 168 */

/*
 *  BEGIN TEST 169
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_169( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-92, 6), MAKE_VAL_COMPLEX(-87, -109), MAKE_VAL_COMPLEX(114, -18), MAKE_VAL_COMPLEX(9, 23), MAKE_VAL_COMPLEX(27, 30), MAKE_VAL_COMPLEX(-17, 7), MAKE_VAL_COMPLEX(-59, 16), MAKE_VAL_COMPLEX(-69, -39), MAKE_VAL_COMPLEX(53, -47), MAKE_VAL_COMPLEX(12, 125), MAKE_VAL_COMPLEX(88, 128), MAKE_VAL_COMPLEX(-24, 126) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_169, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_169", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_169() */
/* END TEST 169 */

/*
 *  BEGIN TEST 170
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_170( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(4, 10), MAKE_VAL_COMPLEX(4, -48), MAKE_VAL_COMPLEX(82, 2), MAKE_VAL_COMPLEX(-77, 25), MAKE_VAL_COMPLEX(-100, 40), MAKE_VAL_COMPLEX(11, -45), MAKE_VAL_COMPLEX(9, -79), MAKE_VAL_COMPLEX(91, -19), MAKE_VAL_COMPLEX(82, 2), MAKE_VAL_COMPLEX(-36, 26), MAKE_VAL_COMPLEX(51, -3), MAKE_VAL_COMPLEX(14, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_170, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_170", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_170() */
/* END TEST 170 */

/*
 *  BEGIN TEST 171
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_171( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(81, -211), MAKE_VAL_COMPLEX(259, 17), MAKE_VAL_COMPLEX(-222, 176), MAKE_VAL_COMPLEX(102, 13), MAKE_VAL_COMPLEX(123, 34), MAKE_VAL_COMPLEX(-53, 30), MAKE_VAL_COMPLEX(-53, -1), MAKE_VAL_COMPLEX(-30, -139), MAKE_VAL_COMPLEX(-103, 83), MAKE_VAL_COMPLEX(-177, -118), MAKE_VAL_COMPLEX(-437, 12), MAKE_VAL_COMPLEX(-248, -196) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_171, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_171", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_171() */
/* END TEST 171 */

/*
 *  BEGIN TEST 172
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_172( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(54, 19), MAKE_VAL_COMPLEX(19, -19), MAKE_VAL_COMPLEX(-17, -108), MAKE_VAL_COMPLEX(-42, 42), MAKE_VAL_COMPLEX(-45, 27), MAKE_VAL_COMPLEX(-8, 165), MAKE_VAL_COMPLEX(12, 23), MAKE_VAL_COMPLEX(136, -125), MAKE_VAL_COMPLEX(-16, 187), MAKE_VAL_COMPLEX(-14, 88), MAKE_VAL_COMPLEX(-89, -162), MAKE_VAL_COMPLEX(-31, 59), MAKE_VAL_COMPLEX(22, 54), MAKE_VAL_COMPLEX(7, 86) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_172, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_172", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_172() */
/* END TEST 172 */

/*
 *  BEGIN TEST 173
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_173( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(2, -27), MAKE_VAL_COMPLEX(-9, 9), MAKE_VAL_COMPLEX(7, 78), MAKE_VAL_COMPLEX(58, 30), MAKE_VAL_COMPLEX(29, 1), MAKE_VAL_COMPLEX(54, -103), MAKE_VAL_COMPLEX(-4, -7), MAKE_VAL_COMPLEX(-116, 65), MAKE_VAL_COMPLEX(46, -137), MAKE_VAL_COMPLEX(22, -32), MAKE_VAL_COMPLEX(25, 154), MAKE_VAL_COMPLEX(23, -15), MAKE_VAL_COMPLEX(-26, -2), MAKE_VAL_COMPLEX(23, -56) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_173, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_173", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_173() */
/* END TEST 173 */

/*
 *  BEGIN TEST 174
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_174( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(-28, -36), MAKE_VAL_COMPLEX(-11, -3), MAKE_VAL_COMPLEX(-17, 19), MAKE_VAL_COMPLEX(44, -40), MAKE_VAL_COMPLEX(26, 0), MAKE_VAL_COMPLEX(25, -57), MAKE_VAL_COMPLEX(8, -12), MAKE_VAL_COMPLEX(-50, 10), MAKE_VAL_COMPLEX(23, -41), MAKE_VAL_COMPLEX(36, -28), MAKE_VAL_COMPLEX(20, 48), MAKE_VAL_COMPLEX(34, -12), MAKE_VAL_COMPLEX(38, -18), MAKE_VAL_COMPLEX(9, -33) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_174, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_174", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_174() */
/* END TEST 174 */

/*
 *  BEGIN TEST 175
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_175( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-5, -7), MAKE_VAL_COMPLEX(183, 28), MAKE_VAL_COMPLEX(-6, 61), MAKE_VAL_COMPLEX(53, 377), MAKE_VAL_COMPLEX(88, 444), MAKE_VAL_COMPLEX(38, 34), MAKE_VAL_COMPLEX(295, -269), MAKE_VAL_COMPLEX(-59, 16), MAKE_VAL_COMPLEX(-501, 224), MAKE_VAL_COMPLEX(292, -564), MAKE_VAL_COMPLEX(-24, -26), MAKE_VAL_COMPLEX(-106, 699), MAKE_VAL_COMPLEX(-24, -6), MAKE_VAL_COMPLEX(-352, 54), MAKE_VAL_COMPLEX(146, -142) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_175, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_175", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_175() */
/* END TEST 175 */

/*
 *  BEGIN TEST 176
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_176( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(0, 113), MAKE_VAL_COMPLEX(3, 3), MAKE_VAL_COMPLEX(11, -54), MAKE_VAL_COMPLEX(6, -30), MAKE_VAL_COMPLEX(-23, -31), MAKE_VAL_COMPLEX(-36, -63), MAKE_VAL_COMPLEX(-8, -5), MAKE_VAL_COMPLEX(116, -5), MAKE_VAL_COMPLEX(-148, -19), MAKE_VAL_COMPLEX(-46, -16), MAKE_VAL_COMPLEX(115, 6), MAKE_VAL_COMPLEX(-29, -27), MAKE_VAL_COMPLEX(14, -46), MAKE_VAL_COMPLEX(-59, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_176, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_176", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_176() */
/* END TEST 176 */

/*
 *  BEGIN TEST 177
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_177( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(-4, -141), MAKE_VAL_COMPLEX(-8, -8), MAKE_VAL_COMPLEX(-26, 59), MAKE_VAL_COMPLEX(30, 22), MAKE_VAL_COMPLEX(37, 39), MAKE_VAL_COMPLEX(67, 40), MAKE_VAL_COMPLEX(16, 1), MAKE_VAL_COMPLEX(-146, -5), MAKE_VAL_COMPLEX(173, 4), MAKE_VAL_COMPLEX(74, 12), MAKE_VAL_COMPLEX(-119, 26), MAKE_VAL_COMPLEX(51, 31), MAKE_VAL_COMPLEX(12, 48), MAKE_VAL_COMPLEX(74, -13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_177, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_177", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_177() */
/* END TEST 177 */

/*
 *  BEGIN TEST 178
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_178( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(-44, 32), MAKE_VAL_COMPLEX(-4, 13), MAKE_VAL_COMPLEX(22, 21), MAKE_VAL_COMPLEX(-46, -54), MAKE_VAL_COMPLEX(1, -31), MAKE_VAL_COMPLEX(-67, -32), MAKE_VAL_COMPLEX(-14, -10), MAKE_VAL_COMPLEX(10, 60), MAKE_VAL_COMPLEX(-48, -29), MAKE_VAL_COMPLEX(-32, -44), MAKE_VAL_COMPLEX(58, -22), MAKE_VAL_COMPLEX(-13, -41), MAKE_VAL_COMPLEX(-20, -46), MAKE_VAL_COMPLEX(-39, -12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_178, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_178", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_178() */
/* END TEST 178 */

/*
 *  BEGIN TEST 179
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_179( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_ROWMAJ, y144_VW );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-4, 2), MAKE_VAL_COMPLEX(-183, 59), MAKE_VAL_COMPLEX(45, 17), MAKE_VAL_COMPLEX(230, -16), MAKE_VAL_COMPLEX(-124, 268), MAKE_VAL_COMPLEX(-16, -28), MAKE_VAL_COMPLEX(120, 254), MAKE_VAL_COMPLEX(-13, 45), MAKE_VAL_COMPLEX(149, 283), MAKE_VAL_COMPLEX(104, -208), MAKE_VAL_COMPLEX(-78, 34), MAKE_VAL_COMPLEX(-253, 103), MAKE_VAL_COMPLEX(-58, 14), MAKE_VAL_COMPLEX(-62, 194), MAKE_VAL_COMPLEX(32, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_179, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_179", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_179() */
/* END TEST 179 */

/*
 *  BEGIN TEST 180
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_180( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(106, -4), MAKE_VAL_COMPLEX(-28, -28), MAKE_VAL_COMPLEX(84, -75), MAKE_VAL_COMPLEX(-35, -38), MAKE_VAL_COMPLEX(27, 209), MAKE_VAL_COMPLEX(-32, -29), MAKE_VAL_COMPLEX(7, -59), MAKE_VAL_COMPLEX(-42, -228), MAKE_VAL_COMPLEX(-58, -74), MAKE_VAL_COMPLEX(-8, 34), MAKE_VAL_COMPLEX(-22, -26), MAKE_VAL_COMPLEX(40, -138) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_180, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_180", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_180() */
/* END TEST 180 */

/*
 *  BEGIN TEST 181
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_181( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-103, 17), MAKE_VAL_COMPLEX(42, -30), MAKE_VAL_COMPLEX(16, 93), MAKE_VAL_COMPLEX(-64, 54), MAKE_VAL_COMPLEX(-157, -179), MAKE_VAL_COMPLEX(54, -24), MAKE_VAL_COMPLEX(19, -36), MAKE_VAL_COMPLEX(157, 222), MAKE_VAL_COMPLEX(158, 92), MAKE_VAL_COMPLEX(-41, -9), MAKE_VAL_COMPLEX(85, 32), MAKE_VAL_COMPLEX(-24, 148) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_181, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_181", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_181() */
/* END TEST 181 */

/*
 *  BEGIN TEST 182
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_182( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(6, 10), MAKE_VAL_COMPLEX(-4, -52), MAKE_VAL_COMPLEX(88, -12), MAKE_VAL_COMPLEX(-78, 40), MAKE_VAL_COMPLEX(-100, 60), MAKE_VAL_COMPLEX(4, -50), MAKE_VAL_COMPLEX(-4, -86), MAKE_VAL_COMPLEX(94, -36), MAKE_VAL_COMPLEX(88, -12), MAKE_VAL_COMPLEX(-34, 34), MAKE_VAL_COMPLEX(54, -12), MAKE_VAL_COMPLEX(16, 4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_182, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_182", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_182() */
/* END TEST 182 */

/*
 *  BEGIN TEST 183
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_183( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	oski_value_t* y_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(215, -887), MAKE_VAL_COMPLEX(90, -14), MAKE_VAL_COMPLEX(-71, -275), MAKE_VAL_COMPLEX(-1006, -341), MAKE_VAL_COMPLEX(1172, -286), MAKE_VAL_COMPLEX(75, 92), MAKE_VAL_COMPLEX(202, -519), MAKE_VAL_COMPLEX(-1607, 382), MAKE_VAL_COMPLEX(-488, 1006), MAKE_VAL_COMPLEX(-101, -15), MAKE_VAL_COMPLEX(-45, 540), MAKE_VAL_COMPLEX(-1110, -628) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, -9);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_183, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_183", y_true, y_val, y144_LEN*y144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_183() */
/* END TEST 183 */

/*
 *  BEGIN TEST 184
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_184( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(-13, -31), MAKE_VAL_COMPLEX(22, -8), MAKE_VAL_COMPLEX(56, -45), MAKE_VAL_COMPLEX(-6, -86), MAKE_VAL_COMPLEX(-34, 26), MAKE_VAL_COMPLEX(150, 251), MAKE_VAL_COMPLEX(-3, 6), MAKE_VAL_COMPLEX(32, -107), MAKE_VAL_COMPLEX(-26, 222), MAKE_VAL_COMPLEX(-6, 44), MAKE_VAL_COMPLEX(-45, -129), MAKE_VAL_COMPLEX(-18, 29), MAKE_VAL_COMPLEX(14, 22), MAKE_VAL_COMPLEX(-8, 56) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_184, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_184", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_184() */
/* END TEST 184 */

/*
 *  BEGIN TEST 185
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_185( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(53, 31), MAKE_VAL_COMPLEX(-14, 2), MAKE_VAL_COMPLEX(-60, 23), MAKE_VAL_COMPLEX(10, 138), MAKE_VAL_COMPLEX(20, -8), MAKE_VAL_COMPLEX(-124, -203), MAKE_VAL_COMPLEX(7, 6), MAKE_VAL_COMPLEX(-12, 67), MAKE_VAL_COMPLEX(42, -184), MAKE_VAL_COMPLEX(6, -4), MAKE_VAL_COMPLEX(1, 117), MAKE_VAL_COMPLEX(8, 1), MAKE_VAL_COMPLEX(-22, 14), MAKE_VAL_COMPLEX(26, -32) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 0);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_185, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_185", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_185() */
/* END TEST 185 */

/*
 *  BEGIN TEST 186
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_186( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(28, -24), MAKE_VAL_COMPLEX(2, -9), MAKE_VAL_COMPLEX(-16, -13), MAKE_VAL_COMPLEX(34, 34), MAKE_VAL_COMPLEX(1, 21), MAKE_VAL_COMPLEX(47, 18), MAKE_VAL_COMPLEX(10, 6), MAKE_VAL_COMPLEX(-10, -40), MAKE_VAL_COMPLEX(34, 17), MAKE_VAL_COMPLEX(24, 28), MAKE_VAL_COMPLEX(-38, 18), MAKE_VAL_COMPLEX(11, 27), MAKE_VAL_COMPLEX(16, 30), MAKE_VAL_COMPLEX(27, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_186, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_186", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_186() */
/* END TEST 186 */

/*
 *  BEGIN TEST 187
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_187( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, -1), MAKE_VAL_COMPLEX(-963, 169), MAKE_VAL_COMPLEX(33, -115), MAKE_VAL_COMPLEX(266, -575), MAKE_VAL_COMPLEX(-1270, -1358), MAKE_VAL_COMPLEX(30, 148), MAKE_VAL_COMPLEX(2722, 157), MAKE_VAL_COMPLEX(-118, -69), MAKE_VAL_COMPLEX(-697, -414), MAKE_VAL_COMPLEX(1273, 1579), MAKE_VAL_COMPLEX(48, -174), MAKE_VAL_COMPLEX(-793, -783), MAKE_VAL_COMPLEX(37, -116), MAKE_VAL_COMPLEX(156, -532), MAKE_VAL_COMPLEX(-1, 307) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(8, -9);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_187, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_187", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_187() */
/* END TEST 187 */

/*
 *  BEGIN TEST 188
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_188( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(61, 153), MAKE_VAL_COMPLEX(29, 3), MAKE_VAL_COMPLEX(85, -48), MAKE_VAL_COMPLEX(66, 74), MAKE_VAL_COMPLEX(-56, 6), MAKE_VAL_COMPLEX(-213, -60), MAKE_VAL_COMPLEX(-19, 10), MAKE_VAL_COMPLEX(82, -77), MAKE_VAL_COMPLEX(-217, 39), MAKE_VAL_COMPLEX(-58, 40), MAKE_VAL_COMPLEX(55, -13), MAKE_VAL_COMPLEX(-56, 13), MAKE_VAL_COMPLEX(-32, 8), MAKE_VAL_COMPLEX(-41, 37) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_188, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_188", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_188() */
/* END TEST 188 */

/*
 *  BEGIN TEST 189
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_189( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -1), MAKE_VAL_COMPLEX(27, -177), MAKE_VAL_COMPLEX(-15, -21), MAKE_VAL_COMPLEX(-107, 2), MAKE_VAL_COMPLEX(-26, 38), MAKE_VAL_COMPLEX(36, 42), MAKE_VAL_COMPLEX(299, 150), MAKE_VAL_COMPLEX(35, 14), MAKE_VAL_COMPLEX(-62, -23), MAKE_VAL_COMPLEX(275, 35), MAKE_VAL_COMPLEX(82, 48), MAKE_VAL_COMPLEX(-159, 13), MAKE_VAL_COMPLEX(52, 59), MAKE_VAL_COMPLEX(36, 76), MAKE_VAL_COMPLEX(95, 5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_189, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_189", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_189() */
/* END TEST 189 */

/*
 *  BEGIN TEST 190
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_190( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(36, 52), MAKE_VAL_COMPLEX(15, 5), MAKE_VAL_COMPLEX(25, -25), MAKE_VAL_COMPLEX(-64, 52), MAKE_VAL_COMPLEX(-36, -2), MAKE_VAL_COMPLEX(-39, 77), MAKE_VAL_COMPLEX(-12, 16), MAKE_VAL_COMPLEX(70, -10), MAKE_VAL_COMPLEX(-35, 55), MAKE_VAL_COMPLEX(-52, 36), MAKE_VAL_COMPLEX(-24, -68), MAKE_VAL_COMPLEX(-48, 14), MAKE_VAL_COMPLEX(-54, 22), MAKE_VAL_COMPLEX(-15, 45) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_190, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_190", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_190() */
/* END TEST 190 */

/*
 *  BEGIN TEST 191
 *  Value type: complex
 *  A : <MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x144', LAYOUT_COLMAJ>
 *  y : <'y144', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_191( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A144_ptr, A144_ind, A144_val,
		A144_M, A144_N, SHARE_INPUTMAT,
		5, MAT_GENERAL, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y144_val,
		y144_LEN * y144_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y144_LEN, y144_VW, LAYOUT_COLMAJ, y144_LEN );
	oski_value_t* y_val = array_Duplicate( x144_val,
		x144_LEN * x144_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x144_LEN, x144_VW, LAYOUT_COLMAJ, x144_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, -1), MAKE_VAL_COMPLEX(49, -29), MAKE_VAL_COMPLEX(-13, -7), MAKE_VAL_COMPLEX(-20, -31), MAKE_VAL_COMPLEX(22, -198), MAKE_VAL_COMPLEX(28, -2), MAKE_VAL_COMPLEX(-166, 109), MAKE_VAL_COMPLEX(-10, -9), MAKE_VAL_COMPLEX(-97, 38), MAKE_VAL_COMPLEX(-33, 131), MAKE_VAL_COMPLEX(8, -46), MAKE_VAL_COMPLEX(111, -59), MAKE_VAL_COMPLEX(15, -30), MAKE_VAL_COMPLEX(0, -68), MAKE_VAL_COMPLEX(-29, -7) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_191, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_191", y_true, y_val, x144_LEN*x144_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_191() */
/* END TEST 191 */

#define A192_M 4    /* # of rows */
#define A192_N 5    /* # of cols */
#define A192_NNZ 4  /* # of stored non-zeros */
static oski_index_t A192_ptr[A192_N+1] = {1, 1, 2, 3, 3, 5}; /* end row pointers for A192 */
static oski_index_t A192_ind[A192_NNZ] = {2, 1, 2, 3}; /* end column indices pointers for A192 */
static oski_value_t A192_val[A192_NNZ] = {MAKE_VAL_COMPLEX(0, 10), MAKE_VAL_COMPLEX(-10, 4), MAKE_VAL_COMPLEX(10, -10), MAKE_VAL_COMPLEX(2, 7)}; /* end stored non-zero values for A192 */

#define x192_LEN 5   /* vector length */
#define x192_VW  3   /* vector width */
static oski_value_t x192_val[x192_LEN*x192_VW] = {MAKE_VAL_COMPLEX(-8, 0), MAKE_VAL_COMPLEX(10, -10), MAKE_VAL_COMPLEX(3, -4), MAKE_VAL_COMPLEX(-4, 2), MAKE_VAL_COMPLEX(5, 6), MAKE_VAL_COMPLEX(-7, -3), MAKE_VAL_COMPLEX(0, 4), MAKE_VAL_COMPLEX(0, -10), MAKE_VAL_COMPLEX(-5, -2), MAKE_VAL_COMPLEX(-10, 10), MAKE_VAL_COMPLEX(0, 1), MAKE_VAL_COMPLEX(-8, 10), MAKE_VAL_COMPLEX(-3, 5), MAKE_VAL_COMPLEX(9, 4), MAKE_VAL_COMPLEX(-9, 5)}; /* data */

#define y192_LEN 4   /* vector length */
#define y192_VW  3   /* vector width */
static oski_value_t y192_val[y192_LEN*y192_VW] = {MAKE_VAL_COMPLEX(-10, 3), MAKE_VAL_COMPLEX(8, -10), MAKE_VAL_COMPLEX(-4, -4), MAKE_VAL_COMPLEX(10, -1), MAKE_VAL_COMPLEX(-7, -6), MAKE_VAL_COMPLEX(8, 10), MAKE_VAL_COMPLEX(-7, 10), MAKE_VAL_COMPLEX(1, 7), MAKE_VAL_COMPLEX(7, -1), MAKE_VAL_COMPLEX(2, 0), MAKE_VAL_COMPLEX(-10, -9), MAKE_VAL_COMPLEX(2, -9)}; /* data */

/*
 *  BEGIN TEST 192
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_192( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-60, -114), MAKE_VAL_COMPLEX(136, 144), MAKE_VAL_COMPLEX(82, -40), MAKE_VAL_COMPLEX(28, 118), MAKE_VAL_COMPLEX(104, -68), MAKE_VAL_COMPLEX(-74, 154), MAKE_VAL_COMPLEX(-135, -47), MAKE_VAL_COMPLEX(-64, 93), MAKE_VAL_COMPLEX(-31, 1), MAKE_VAL_COMPLEX(4, 16), MAKE_VAL_COMPLEX(52, -98), MAKE_VAL_COMPLEX(76, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_192, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_192", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_192() */
/* END TEST 192 */

/*
 *  BEGIN TEST 193
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_193( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(84, -24), MAKE_VAL_COMPLEX(-64, 12), MAKE_VAL_COMPLEX(-10, 16), MAKE_VAL_COMPLEX(-76, 8), MAKE_VAL_COMPLEX(10, 20), MAKE_VAL_COMPLEX(-94, -118), MAKE_VAL_COMPLEX(57, -97), MAKE_VAL_COMPLEX(-26, -123), MAKE_VAL_COMPLEX(1, 89), MAKE_VAL_COMPLEX(-16, 8), MAKE_VAL_COMPLEX(116, 32), MAKE_VAL_COMPLEX(20, 80) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_193, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_193", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_193() */
/* END TEST 193 */

/*
 *  BEGIN TEST 194
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_194( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(10, -3), MAKE_VAL_COMPLEX(-8, 10), MAKE_VAL_COMPLEX(4, 4), MAKE_VAL_COMPLEX(-10, 1), MAKE_VAL_COMPLEX(7, 6), MAKE_VAL_COMPLEX(-8, -10), MAKE_VAL_COMPLEX(7, -10), MAKE_VAL_COMPLEX(-1, -7), MAKE_VAL_COMPLEX(-7, 1), MAKE_VAL_COMPLEX(-2, 0), MAKE_VAL_COMPLEX(10, 9), MAKE_VAL_COMPLEX(-2, 9) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_194, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_194", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_194() */
/* END TEST 194 */

/*
 *  BEGIN TEST 195
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_195( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-242, 216), MAKE_VAL_COMPLEX(668, -496), MAKE_VAL_COMPLEX(-108, -440), MAKE_VAL_COMPLEX(278, -96), MAKE_VAL_COMPLEX(-124, -522), MAKE_VAL_COMPLEX(568, -116), MAKE_VAL_COMPLEX(42, 324), MAKE_VAL_COMPLEX(598, -92), MAKE_VAL_COMPLEX(-348, 520), MAKE_VAL_COMPLEX(-8, -4), MAKE_VAL_COMPLEX(22, 56), MAKE_VAL_COMPLEX(-26, 32) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-2, -8);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_195, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_195", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_195() */
/* END TEST 195 */

/*
 *  BEGIN TEST 196
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_196( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(40, 16), MAKE_VAL_COMPLEX(-70, 30), MAKE_VAL_COMPLEX(-23, 14), MAKE_VAL_COMPLEX(34, 98), MAKE_VAL_COMPLEX(47, -110), MAKE_VAL_COMPLEX(-71, 109), MAKE_VAL_COMPLEX(96, -90), MAKE_VAL_COMPLEX(-60, 182), MAKE_VAL_COMPLEX(77, 44), MAKE_VAL_COMPLEX(70, -30), MAKE_VAL_COMPLEX(2, -5), MAKE_VAL_COMPLEX(60, -34), MAKE_VAL_COMPLEX(31, -158), MAKE_VAL_COMPLEX(-214, -7), MAKE_VAL_COMPLEX(256, 60) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_196, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_196", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_196() */
/* END TEST 196 */

/*
 *  BEGIN TEST 197
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_197( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-56, 48), MAKE_VAL_COMPLEX(10, -130), MAKE_VAL_COMPLEX(-3, -46), MAKE_VAL_COMPLEX(-26, -62), MAKE_VAL_COMPLEX(11, 82), MAKE_VAL_COMPLEX(33, -59), MAKE_VAL_COMPLEX(-64, 98), MAKE_VAL_COMPLEX(-20, -202), MAKE_VAL_COMPLEX(-103, -8), MAKE_VAL_COMPLEX(-10, 130), MAKE_VAL_COMPLEX(6, 7), MAKE_VAL_COMPLEX(4, 118), MAKE_VAL_COMPLEX(3, 192), MAKE_VAL_COMPLEX(264, -57), MAKE_VAL_COMPLEX(-234, 22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_197, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_197", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_197() */
/* END TEST 197 */

/*
 *  BEGIN TEST 198
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_198( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 56), MAKE_VAL_COMPLEX(-70, -70), MAKE_VAL_COMPLEX(-28, -21), MAKE_VAL_COMPLEX(14, 28), MAKE_VAL_COMPLEX(42, -35), MAKE_VAL_COMPLEX(-21, 49), MAKE_VAL_COMPLEX(28, 0), MAKE_VAL_COMPLEX(-70, 0), MAKE_VAL_COMPLEX(-14, 35), MAKE_VAL_COMPLEX(70, 70), MAKE_VAL_COMPLEX(7, 0), MAKE_VAL_COMPLEX(70, 56), MAKE_VAL_COMPLEX(35, 21), MAKE_VAL_COMPLEX(28, -63), MAKE_VAL_COMPLEX(35, 63) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_198, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_198", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_198() */
/* END TEST 198 */

/*
 *  BEGIN TEST 199
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_199( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(32, 16), MAKE_VAL_COMPLEX(-60, 20), MAKE_VAL_COMPLEX(-20, 10), MAKE_VAL_COMPLEX(-860, -720), MAKE_VAL_COMPLEX(72, 1006), MAKE_VAL_COMPLEX(182, -1414), MAKE_VAL_COMPLEX(-136, 1248), MAKE_VAL_COMPLEX(-756, -1336), MAKE_VAL_COMPLEX(-624, 274), MAKE_VAL_COMPLEX(60, -20), MAKE_VAL_COMPLEX(2, -4), MAKE_VAL_COMPLEX(52, -24), MAKE_VAL_COMPLEX(1086, 1146), MAKE_VAL_COMPLEX(1140, -1698), MAKE_VAL_COMPLEX(-2098, 1070) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, 8);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_199, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_199", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_199() */
/* END TEST 199 */

/*
 *  BEGIN TEST 200
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_200( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(48, -16), MAKE_VAL_COMPLEX(-40, 80), MAKE_VAL_COMPLEX(-10, 30), MAKE_VAL_COMPLEX(10, -120), MAKE_VAL_COMPLEX(-102, 44), MAKE_VAL_COMPLEX(148, -76), MAKE_VAL_COMPLEX(104, -14), MAKE_VAL_COMPLEX(-100, 128), MAKE_VAL_COMPLEX(58, 58), MAKE_VAL_COMPLEX(40, -80), MAKE_VAL_COMPLEX(-2, -6), MAKE_VAL_COMPLEX(28, -76), MAKE_VAL_COMPLEX(174, 123), MAKE_VAL_COMPLEX(-21, -129), MAKE_VAL_COMPLEX(31, 81) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_200, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_200", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_200() */
/* END TEST 200 */

/*
 *  BEGIN TEST 201
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_201( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-72, 40), MAKE_VAL_COMPLEX(40, -140), MAKE_VAL_COMPLEX(7, -51), MAKE_VAL_COMPLEX(-16, 138), MAKE_VAL_COMPLEX(135, -41), MAKE_VAL_COMPLEX(-178, 88), MAKE_VAL_COMPLEX(-92, 26), MAKE_VAL_COMPLEX(70, -158), MAKE_VAL_COMPLEX(-79, -49), MAKE_VAL_COMPLEX(-40, 140), MAKE_VAL_COMPLEX(5, 9), MAKE_VAL_COMPLEX(-22, 130), MAKE_VAL_COMPLEX(-168, -99), MAKE_VAL_COMPLEX(60, 114), MAKE_VAL_COMPLEX(-43, -39) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_201, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_201", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_201() */
/* END TEST 201 */

/*
 *  BEGIN TEST 202
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_202( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-8, 8), MAKE_VAL_COMPLEX(0, -20), MAKE_VAL_COMPLEX(-1, -7), MAKE_VAL_COMPLEX(-2, 6), MAKE_VAL_COMPLEX(11, 1), MAKE_VAL_COMPLEX(-10, 4), MAKE_VAL_COMPLEX(4, 4), MAKE_VAL_COMPLEX(-10, -10), MAKE_VAL_COMPLEX(-7, 3), MAKE_VAL_COMPLEX(0, 20), MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(2, 18), MAKE_VAL_COMPLEX(2, 8), MAKE_VAL_COMPLEX(13, -5), MAKE_VAL_COMPLEX(-4, 14) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_202, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_202", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_202() */
/* END TEST 202 */

/*
 *  BEGIN TEST 203
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_203( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(72, -40), MAKE_VAL_COMPLEX(-40, 140), MAKE_VAL_COMPLEX(-7, 51), MAKE_VAL_COMPLEX(-114, 582), MAKE_VAL_COMPLEX(425, -329), MAKE_VAL_COMPLEX(-682, 272), MAKE_VAL_COMPLEX(-672, -320), MAKE_VAL_COMPLEX(906, -78), MAKE_VAL_COMPLEX(23, -391), MAKE_VAL_COMPLEX(40, -140), MAKE_VAL_COMPLEX(-5, -9), MAKE_VAL_COMPLEX(22, -130), MAKE_VAL_COMPLEX(-676, -1346), MAKE_VAL_COMPLEX(-593, 665), MAKE_VAL_COMPLEX(392, -838) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_203, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_203", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_203() */
/* END TEST 203 */

/*
 *  BEGIN TEST 204
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_204( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(35, 21), MAKE_VAL_COMPLEX(-94, 14), MAKE_VAL_COMPLEX(-57, 29), MAKE_VAL_COMPLEX(-37, -67), MAKE_VAL_COMPLEX(19, 167), MAKE_VAL_COMPLEX(116, -86), MAKE_VAL_COMPLEX(81, 90), MAKE_VAL_COMPLEX(46, -28), MAKE_VAL_COMPLEX(30, -46), MAKE_VAL_COMPLEX(-16, 56), MAKE_VAL_COMPLEX(-86, 44), MAKE_VAL_COMPLEX(-69, 13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_204, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_204", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_204() */
/* END TEST 204 */

/*
 *  BEGIN TEST 205
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_205( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, 7), MAKE_VAL_COMPLEX(22, -6), MAKE_VAL_COMPLEX(57, 3), MAKE_VAL_COMPLEX(-7, 31), MAKE_VAL_COMPLEX(-15, -115), MAKE_VAL_COMPLEX(-108, 14), MAKE_VAL_COMPLEX(-13, -102), MAKE_VAL_COMPLEX(-22, -4), MAKE_VAL_COMPLEX(-62, 22), MAKE_VAL_COMPLEX(8, -64), MAKE_VAL_COMPLEX(90, 32), MAKE_VAL_COMPLEX(25, 15) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_205, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_205", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_205() */
/* END TEST 205 */

/*
 *  BEGIN TEST 206
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_206( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-65, 74), MAKE_VAL_COMPLEX(14, -120), MAKE_VAL_COMPLEX(-52, -12), MAKE_VAL_COMPLEX(75, -58), MAKE_VAL_COMPLEX(-86, -13), MAKE_VAL_COMPLEX(114, 40), MAKE_VAL_COMPLEX(-6, 115), MAKE_VAL_COMPLEX(43, 51), MAKE_VAL_COMPLEX(51, -43), MAKE_VAL_COMPLEX(16, -10), MAKE_VAL_COMPLEX(-125, -22), MAKE_VAL_COMPLEX(-29, -82) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_206, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_206", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_206() */
/* END TEST 206 */

/*
 *  BEGIN TEST 207
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_207( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-376, -63), MAKE_VAL_COMPLEX(356, 98), MAKE_VAL_COMPLEX(-193, 161), MAKE_VAL_COMPLEX(76, 53), MAKE_VAL_COMPLEX(707, -24), MAKE_VAL_COMPLEX(206, -302), MAKE_VAL_COMPLEX(287, 301), MAKE_VAL_COMPLEX(-35, 55), MAKE_VAL_COMPLEX(229, -313), MAKE_VAL_COMPLEX(404, 282), MAKE_VAL_COMPLEX(-493, 36), MAKE_VAL_COMPLEX(68, -51) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_207, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_207", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_207() */
/* END TEST 207 */

/*
 *  BEGIN TEST 208
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_208( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-48, 40), MAKE_VAL_COMPLEX(10, -110), MAKE_VAL_COMPLEX(-2, -39), MAKE_VAL_COMPLEX(86, 112), MAKE_VAL_COMPLEX(-40, 91), MAKE_VAL_COMPLEX(-57, 37), MAKE_VAL_COMPLEX(108, -46), MAKE_VAL_COMPLEX(44, -28), MAKE_VAL_COMPLEX(-106, 51), MAKE_VAL_COMPLEX(-10, 110), MAKE_VAL_COMPLEX(5, 6), MAKE_VAL_COMPLEX(2, 100), MAKE_VAL_COMPLEX(7, -171), MAKE_VAL_COMPLEX(170, -30), MAKE_VAL_COMPLEX(34, -33) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_208, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_208", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_208() */
/* END TEST 208 */

/*
 *  BEGIN TEST 209
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_209( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(72, -8), MAKE_VAL_COMPLEX(-80, 100), MAKE_VAL_COMPLEX(-23, 39), MAKE_VAL_COMPLEX(-66, -102), MAKE_VAL_COMPLEX(49, -129), MAKE_VAL_COMPLEX(66, 0), MAKE_VAL_COMPLEX(-92, 34), MAKE_VAL_COMPLEX(-84, 58), MAKE_VAL_COMPLEX(113, -25), MAKE_VAL_COMPLEX(80, -100), MAKE_VAL_COMPLEX(-1, -9), MAKE_VAL_COMPLEX(62, -98), MAKE_VAL_COMPLEX(22, 168), MAKE_VAL_COMPLEX(-181, -18), MAKE_VAL_COMPLEX(13, 54) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_209, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_209", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_209() */
/* END TEST 209 */

/*
 *  BEGIN TEST 210
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_210( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(8, 32), MAKE_VAL_COMPLEX(-50, -30), MAKE_VAL_COMPLEX(-19, -8), MAKE_VAL_COMPLEX(12, 14), MAKE_VAL_COMPLEX(19, -26), MAKE_VAL_COMPLEX(-5, 31), MAKE_VAL_COMPLEX(16, -4), MAKE_VAL_COMPLEX(-40, 10), MAKE_VAL_COMPLEX(-3, 22), MAKE_VAL_COMPLEX(50, 30), MAKE_VAL_COMPLEX(4, -1), MAKE_VAL_COMPLEX(48, 22), MAKE_VAL_COMPLEX(23, 7), MAKE_VAL_COMPLEX(7, -40), MAKE_VAL_COMPLEX(29, 31) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_210, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_210", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_210() */
/* END TEST 210 */

/*
 *  BEGIN TEST 211
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_211( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-56, 64), MAKE_VAL_COMPLEX(-10, -150), MAKE_VAL_COMPLEX(-11, -52), MAKE_VAL_COMPLEX(68, -54), MAKE_VAL_COMPLEX(163, 102), MAKE_VAL_COMPLEX(-53, 35), MAKE_VAL_COMPLEX(-38, -60), MAKE_VAL_COMPLEX(-48, -164), MAKE_VAL_COMPLEX(-13, 92), MAKE_VAL_COMPLEX(10, 150), MAKE_VAL_COMPLEX(8, 7), MAKE_VAL_COMPLEX(24, 134), MAKE_VAL_COMPLEX(-197, 59), MAKE_VAL_COMPLEX(86, -140), MAKE_VAL_COMPLEX(-131, 44) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_211, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_211", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_211() */
/* END TEST 211 */

/*
 *  BEGIN TEST 212
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_212( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(16, -56), MAKE_VAL_COMPLEX(50, 90), MAKE_VAL_COMPLEX(22, 29), MAKE_VAL_COMPLEX(-106, -112), MAKE_VAL_COMPLEX(48, -57), MAKE_VAL_COMPLEX(35, -63), MAKE_VAL_COMPLEX(84, 2), MAKE_VAL_COMPLEX(116, 108), MAKE_VAL_COMPLEX(-50, -49), MAKE_VAL_COMPLEX(-50, -90), MAKE_VAL_COMPLEX(-7, -2), MAKE_VAL_COMPLEX(-54, -76), MAKE_VAL_COMPLEX(115, -31), MAKE_VAL_COMPLEX(-10, 304), MAKE_VAL_COMPLEX(-80, -1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_212, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_212", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_212() */
/* END TEST 212 */

/*
 *  BEGIN TEST 213
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_213( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, -16), MAKE_VAL_COMPLEX(20, 20), MAKE_VAL_COMPLEX(8, 6), MAKE_VAL_COMPLEX(96, 72), MAKE_VAL_COMPLEX(-112, 90), MAKE_VAL_COMPLEX(6, 6), MAKE_VAL_COMPLEX(-120, -10), MAKE_VAL_COMPLEX(-26, -88), MAKE_VAL_COMPLEX(78, 8), MAKE_VAL_COMPLEX(-20, -20), MAKE_VAL_COMPLEX(-2, 0), MAKE_VAL_COMPLEX(-20, -16), MAKE_VAL_COMPLEX(-154, -6), MAKE_VAL_COMPLEX(-44, -231), MAKE_VAL_COMPLEX(53, -90) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_213, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_213", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_213() */
/* END TEST 213 */

/*
 *  BEGIN TEST 214
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_214( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(64, -72), MAKE_VAL_COMPLEX(10, 170), MAKE_VAL_COMPLEX(12, 59), MAKE_VAL_COMPLEX(14, -52), MAKE_VAL_COMPLEX(-94, -3), MAKE_VAL_COMPLEX(83, -39), MAKE_VAL_COMPLEX(-36, -32), MAKE_VAL_COMPLEX(90, 80), MAKE_VAL_COMPLEX(58, -29), MAKE_VAL_COMPLEX(-10, -170), MAKE_VAL_COMPLEX(-9, -8), MAKE_VAL_COMPLEX(-26, -152), MAKE_VAL_COMPLEX(-21, -67), MAKE_VAL_COMPLEX(-108, 49), MAKE_VAL_COMPLEX(27, -121) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_214, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_214", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_214() */
/* END TEST 214 */

/*
 *  BEGIN TEST 215
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_ROWMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_215( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_ROWMAJ, x192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-8, -48), MAKE_VAL_COMPLEX(70, 50), MAKE_VAL_COMPLEX(27, 14), MAKE_VAL_COMPLEX(544, 918), MAKE_VAL_COMPLEX(-1071, 376), MAKE_VAL_COMPLEX(-49, 115), MAKE_VAL_COMPLEX(-890, -412), MAKE_VAL_COMPLEX(-44, -852), MAKE_VAL_COMPLEX(545, 334), MAKE_VAL_COMPLEX(-70, -50), MAKE_VAL_COMPLEX(-6, 1), MAKE_VAL_COMPLEX(-68, -38), MAKE_VAL_COMPLEX(-1185, -445), MAKE_VAL_COMPLEX(444, -2042), MAKE_VAL_COMPLEX(681, -436) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, -3);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_215, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_215", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_215() */
/* END TEST 215 */

/*
 *  BEGIN TEST 216
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_216( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-28, -31), MAKE_VAL_COMPLEX(112, 174), MAKE_VAL_COMPLEX(46, -90), MAKE_VAL_COMPLEX(208, 191), MAKE_VAL_COMPLEX(15, 150), MAKE_VAL_COMPLEX(-228, 114), MAKE_VAL_COMPLEX(-105, -19), MAKE_VAL_COMPLEX(-147, -49), MAKE_VAL_COMPLEX(-52, 4), MAKE_VAL_COMPLEX(-2, 16), MAKE_VAL_COMPLEX(82, -71), MAKE_VAL_COMPLEX(70, 25) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_216, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_216", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_216() */
/* END TEST 216 */

/*
 *  BEGIN TEST 217
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_217( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(6, -115), MAKE_VAL_COMPLEX(12, -42), MAKE_VAL_COMPLEX(18, 42), MAKE_VAL_COMPLEX(-214, -49), MAKE_VAL_COMPLEX(83, -236), MAKE_VAL_COMPLEX(72, -22), MAKE_VAL_COMPLEX(-21, -99), MAKE_VAL_COMPLEX(47, 49), MAKE_VAL_COMPLEX(52, 96), MAKE_VAL_COMPLEX(-2, 12), MAKE_VAL_COMPLEX(64, -51), MAKE_VAL_COMPLEX(52, 21) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_217, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_217", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_217() */
/* END TEST 217 */

/*
 *  BEGIN TEST 218
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_218( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(69, -97), MAKE_VAL_COMPLEX(-2, 146), MAKE_VAL_COMPLEX(64, 8), MAKE_VAL_COMPLEX(-83, 79), MAKE_VAL_COMPLEX(105, 5), MAKE_VAL_COMPLEX(-142, -34), MAKE_VAL_COMPLEX(-7, -139), MAKE_VAL_COMPLEX(-58, -56), MAKE_VAL_COMPLEX(-56, 58), MAKE_VAL_COMPLEX(-18, 14), MAKE_VAL_COMPLEX(153, 11), MAKE_VAL_COMPLEX(45, 95) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_218, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_218", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_218() */
/* END TEST 218 */

/*
 *  BEGIN TEST 219
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_219( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(463, 447), MAKE_VAL_COMPLEX(1054, 286), MAKE_VAL_COMPLEX(-482, -422), MAKE_VAL_COMPLEX(2177, -1253), MAKE_VAL_COMPLEX(1591, 1623), MAKE_VAL_COMPLEX(-326, 1526), MAKE_VAL_COMPLEX(310, 521), MAKE_VAL_COMPLEX(-976, 458), MAKE_VAL_COMPLEX(-847, 145), MAKE_VAL_COMPLEX(-14, -6), MAKE_VAL_COMPLEX(43, 93), MAKE_VAL_COMPLEX(-41, 57) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, -9);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_219, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_219", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_219() */
/* END TEST 219 */

/*
 *  BEGIN TEST 220
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_220( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(72, -16), MAKE_VAL_COMPLEX(-60, 210), MAKE_VAL_COMPLEX(69, -28), MAKE_VAL_COMPLEX(32, -26), MAKE_VAL_COMPLEX(-51, -183), MAKE_VAL_COMPLEX(69, 13), MAKE_VAL_COMPLEX(52, -106), MAKE_VAL_COMPLEX(-20, 222), MAKE_VAL_COMPLEX(49, 8), MAKE_VAL_COMPLEX(-107, -79), MAKE_VAL_COMPLEX(-2, -9), MAKE_VAL_COMPLEX(-48, -26), MAKE_VAL_COMPLEX(73, -27), MAKE_VAL_COMPLEX(-89, -18), MAKE_VAL_COMPLEX(272, 4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_220, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_220", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_220() */
/* END TEST 220 */

/*
 *  BEGIN TEST 221
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_221( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-24, -32), MAKE_VAL_COMPLEX(60, -90), MAKE_VAL_COMPLEX(-63, 70), MAKE_VAL_COMPLEX(-20, -10), MAKE_VAL_COMPLEX(-15, 177), MAKE_VAL_COMPLEX(-9, -37), MAKE_VAL_COMPLEX(-76, 82), MAKE_VAL_COMPLEX(80, -162), MAKE_VAL_COMPLEX(-7, -26), MAKE_VAL_COMPLEX(107, -41), MAKE_VAL_COMPLEX(-4, 3), MAKE_VAL_COMPLEX(36, -82), MAKE_VAL_COMPLEX(-85, -21), MAKE_VAL_COMPLEX(11, 48), MAKE_VAL_COMPLEX(-248, -88) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_221, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_221", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_221() */
/* END TEST 221 */

/*
 *  BEGIN TEST 222
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_222( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-24, 32), MAKE_VAL_COMPLEX(-10, -70), MAKE_VAL_COMPLEX(-7, -24), MAKE_VAL_COMPLEX(-4, 22), MAKE_VAL_COMPLEX(39, -2), MAKE_VAL_COMPLEX(-33, 19), MAKE_VAL_COMPLEX(16, 12), MAKE_VAL_COMPLEX(-40, -30), MAKE_VAL_COMPLEX(-23, 14), MAKE_VAL_COMPLEX(10, 70), MAKE_VAL_COMPLEX(4, 3), MAKE_VAL_COMPLEX(16, 62), MAKE_VAL_COMPLEX(11, 27), MAKE_VAL_COMPLEX(43, -24), MAKE_VAL_COMPLEX(-7, 51) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_222, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_222", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_222() */
/* END TEST 222 */

/*
 *  BEGIN TEST 223
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_223( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(64, 48), MAKE_VAL_COMPLEX(-120, 220), MAKE_VAL_COMPLEX(128, -126), MAKE_VAL_COMPLEX(44, 8), MAKE_VAL_COMPLEX(8, -356), MAKE_VAL_COMPLEX(38, 66), MAKE_VAL_COMPLEX(144, -172), MAKE_VAL_COMPLEX(-140, 344), MAKE_VAL_COMPLEX(28, 46), MAKE_VAL_COMPLEX(-214, 42), MAKE_VAL_COMPLEX(6, -8), MAKE_VAL_COMPLEX(-76, 128), MAKE_VAL_COMPLEX(166, 26), MAKE_VAL_COMPLEX(-48, -86), MAKE_VAL_COMPLEX(504, 148) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(2, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_223, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_223", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_223() */
/* END TEST 223 */

/*
 *  BEGIN TEST 224
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_224( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-40, 0), MAKE_VAL_COMPLEX(40, -150), MAKE_VAL_COMPLEX(127, -10), MAKE_VAL_COMPLEX(-20, 10), MAKE_VAL_COMPLEX(191, 189), MAKE_VAL_COMPLEX(-35, -15), MAKE_VAL_COMPLEX(-60, 90), MAKE_VAL_COMPLEX(-120, 18), MAKE_VAL_COMPLEX(-25, -10), MAKE_VAL_COMPLEX(-9, -73), MAKE_VAL_COMPLEX(0, 5), MAKE_VAL_COMPLEX(60, -30), MAKE_VAL_COMPLEX(9, 81), MAKE_VAL_COMPLEX(45, 20), MAKE_VAL_COMPLEX(-58, 154) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_224, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_224", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_224() */
/* END TEST 224 */

/*
 *  BEGIN TEST 225
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_225( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-56, 32), MAKE_VAL_COMPLEX(40, -10), MAKE_VAL_COMPLEX(-107, -50), MAKE_VAL_COMPLEX(-20, 30), MAKE_VAL_COMPLEX(-107, -137), MAKE_VAL_COMPLEX(-61, 7), MAKE_VAL_COMPLEX(76, -42), MAKE_VAL_COMPLEX(80, -138), MAKE_VAL_COMPLEX(-43, 6), MAKE_VAL_COMPLEX(-71, 233), MAKE_VAL_COMPLEX(4, 7), MAKE_VAL_COMPLEX(-116, 182), MAKE_VAL_COMPLEX(-25, -9), MAKE_VAL_COMPLEX(79, -8), MAKE_VAL_COMPLEX(-30, -58) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_225, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_225", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_225() */
/* END TEST 225 */

/*
 *  BEGIN TEST 226
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_226( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-56, -64), MAKE_VAL_COMPLEX(150, 10), MAKE_VAL_COMPLEX(53, -4), MAKE_VAL_COMPLEX(-44, -18), MAKE_VAL_COMPLEX(-13, 82), MAKE_VAL_COMPLEX(-25, -77), MAKE_VAL_COMPLEX(-32, 28), MAKE_VAL_COMPLEX(80, -70), MAKE_VAL_COMPLEX(-19, -54), MAKE_VAL_COMPLEX(-150, -10), MAKE_VAL_COMPLEX(-8, 7), MAKE_VAL_COMPLEX(-136, 6), MAKE_VAL_COMPLEX(-61, 11), MAKE_VAL_COMPLEX(31, 100), MAKE_VAL_COMPLEX(-103, -37) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_226, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_226", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_226() */
/* END TEST 226 */

/*
 *  BEGIN TEST 227
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_227( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_ROWMAJ, y192_VW );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(56, -32), MAKE_VAL_COMPLEX(140, 800), MAKE_VAL_COMPLEX(-799, 82), MAKE_VAL_COMPLEX(20, -30), MAKE_VAL_COMPLEX(-1380, -969), MAKE_VAL_COMPLEX(61, -7), MAKE_VAL_COMPLEX(334, -578), MAKE_VAL_COMPLEX(812, -526), MAKE_VAL_COMPLEX(43, -6), MAKE_VAL_COMPLEX(-134, 792), MAKE_VAL_COMPLEX(-4, -7), MAKE_VAL_COMPLEX(-604, 558), MAKE_VAL_COMPLEX(-223, -415), MAKE_VAL_COMPLEX(-79, 8), MAKE_VAL_COMPLEX(5, -987) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, 1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_227, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_227", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_227() */
/* END TEST 227 */

/*
 *  BEGIN TEST 228
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_228( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-19, -1), MAKE_VAL_COMPLEX(252, 160), MAKE_VAL_COMPLEX(-8, 31), MAKE_VAL_COMPLEX(-5, 51), MAKE_VAL_COMPLEX(77, 71), MAKE_VAL_COMPLEX(-98, 230), MAKE_VAL_COMPLEX(-133, -95), MAKE_VAL_COMPLEX(-36, -2), MAKE_VAL_COMPLEX(8, -26), MAKE_VAL_COMPLEX(-142, 70), MAKE_VAL_COMPLEX(2, -94), MAKE_VAL_COMPLEX(43, 19) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_228, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_228", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_228() */
/* END TEST 228 */

/*
 *  BEGIN TEST 229
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_229( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-68, -71), MAKE_VAL_COMPLEX(-114, -148), MAKE_VAL_COMPLEX(20, -91), MAKE_VAL_COMPLEX(74, 33), MAKE_VAL_COMPLEX(-65, -170), MAKE_VAL_COMPLEX(56, -98), MAKE_VAL_COMPLEX(1, 92), MAKE_VAL_COMPLEX(-21, 53), MAKE_VAL_COMPLEX(43, 83), MAKE_VAL_COMPLEX(154, -52), MAKE_VAL_COMPLEX(19, -50), MAKE_VAL_COMPLEX(50, -55) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_229, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_229", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_229() */
/* END TEST 229 */

/*
 *  BEGIN TEST 230
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_230( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-34, 32), MAKE_VAL_COMPLEX(12, -56), MAKE_VAL_COMPLEX(-24, -8), MAKE_VAL_COMPLEX(38, -24), MAKE_VAL_COMPLEX(-40, -10), MAKE_VAL_COMPLEX(52, 24), MAKE_VAL_COMPLEX(-8, 54), MAKE_VAL_COMPLEX(18, 26), MAKE_VAL_COMPLEX(26, -18), MAKE_VAL_COMPLEX(8, -4), MAKE_VAL_COMPLEX(-58, -16), MAKE_VAL_COMPLEX(-10, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_230, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_230", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_230() */
/* END TEST 230 */

/*
 *  BEGIN TEST 231
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_231( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	oski_value_t* y_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-456, 246), MAKE_VAL_COMPLEX(648, 1860), MAKE_VAL_COMPLEX(-498, 66), MAKE_VAL_COMPLEX(60, -6), MAKE_VAL_COMPLEX(-402, 804), MAKE_VAL_COMPLEX(-1392, 1020), MAKE_VAL_COMPLEX(-282, -780), MAKE_VAL_COMPLEX(6, 42), MAKE_VAL_COMPLEX(474, -318), MAKE_VAL_COMPLEX(-1188, -480), MAKE_VAL_COMPLEX(-60, -690), MAKE_VAL_COMPLEX(12, -54) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, 6);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_231, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_231", y_true, y_val, y192_LEN*y192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_231() */
/* END TEST 231 */

/*
 *  BEGIN TEST 232
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_232( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(16, -16), MAKE_VAL_COMPLEX(100, 120), MAKE_VAL_COMPLEX(90, -56), MAKE_VAL_COMPLEX(4, -12), MAKE_VAL_COMPLEX(-22, -218), MAKE_VAL_COMPLEX(20, -8), MAKE_VAL_COMPLEX(-108, 72), MAKE_VAL_COMPLEX(114, 52), MAKE_VAL_COMPLEX(14, -6), MAKE_VAL_COMPLEX(96, -49), MAKE_VAL_COMPLEX(-2, -2), MAKE_VAL_COMPLEX(-4, -16), MAKE_VAL_COMPLEX(-70, 22), MAKE_VAL_COMPLEX(-26, 10), MAKE_VAL_COMPLEX(71, -136) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_232, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_232", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_232() */
/* END TEST 232 */

/*
 *  BEGIN TEST 233
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_233( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(8, -32), MAKE_VAL_COMPLEX(-70, -30), MAKE_VAL_COMPLEX(-75, 86), MAKE_VAL_COMPLEX(-4, -18), MAKE_VAL_COMPLEX(-29, 230), MAKE_VAL_COMPLEX(19, -25), MAKE_VAL_COMPLEX(84, -84), MAKE_VAL_COMPLEX(-54, -22), MAKE_VAL_COMPLEX(13, -18), MAKE_VAL_COMPLEX(-126, -41), MAKE_VAL_COMPLEX(-4, -1), MAKE_VAL_COMPLEX(-32, -62), MAKE_VAL_COMPLEX(49, -55), MAKE_VAL_COMPLEX(-25, 32), MAKE_VAL_COMPLEX(-74, 67) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_233, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_233", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_233() */
/* END TEST 233 */

/*
 *  BEGIN TEST 234
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_234( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, 32), MAKE_VAL_COMPLEX(0, -80), MAKE_VAL_COMPLEX(-4, -28), MAKE_VAL_COMPLEX(-8, 24), MAKE_VAL_COMPLEX(44, 4), MAKE_VAL_COMPLEX(-40, 16), MAKE_VAL_COMPLEX(16, 16), MAKE_VAL_COMPLEX(-40, -40), MAKE_VAL_COMPLEX(-28, 12), MAKE_VAL_COMPLEX(0, 80), MAKE_VAL_COMPLEX(4, 4), MAKE_VAL_COMPLEX(8, 72), MAKE_VAL_COMPLEX(8, 32), MAKE_VAL_COMPLEX(52, -20), MAKE_VAL_COMPLEX(-16, 56) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_234, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_234", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_234() */
/* END TEST 234 */

/*
 *  BEGIN TEST 235
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_235( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-24, 8), MAKE_VAL_COMPLEX(500, 1000), MAKE_VAL_COMPLEX(989, -223), MAKE_VAL_COMPLEX(-10, 10), MAKE_VAL_COMPLEX(885, -1715), MAKE_VAL_COMPLEX(-24, -2), MAKE_VAL_COMPLEX(-1116, 252), MAKE_VAL_COMPLEX(614, 602), MAKE_VAL_COMPLEX(-17, -1), MAKE_VAL_COMPLEX(784, 352), MAKE_VAL_COMPLEX(1, 3), MAKE_VAL_COMPLEX(-94, 198), MAKE_VAL_COMPLEX(-684, 58), MAKE_VAL_COMPLEX(31, 3), MAKE_VAL_COMPLEX(914, -588) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(8, 4);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_235, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_235", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_235() */
/* END TEST 235 */

/*
 *  BEGIN TEST 236
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_236( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(16, -48), MAKE_VAL_COMPLEX(-60, 0), MAKE_VAL_COMPLEX(130, 36), MAKE_VAL_COMPLEX(-4, -28), MAKE_VAL_COMPLEX(98, 18), MAKE_VAL_COMPLEX(32, -36), MAKE_VAL_COMPLEX(76, -88), MAKE_VAL_COMPLEX(106, 108), MAKE_VAL_COMPLEX(22, -26), MAKE_VAL_COMPLEX(-4, 169), MAKE_VAL_COMPLEX(-6, -2), MAKE_VAL_COMPLEX(-44, -88), MAKE_VAL_COMPLEX(-98, -46), MAKE_VAL_COMPLEX(-42, 46), MAKE_VAL_COMPLEX(-75, 8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_236, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_236", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_236() */
/* END TEST 236 */

/*
 *  BEGIN TEST 237
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_237( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(64, 56), MAKE_VAL_COMPLEX(-50, 90), MAKE_VAL_COMPLEX(-164, 1), MAKE_VAL_COMPLEX(46, 12), MAKE_VAL_COMPLEX(-142, -83), MAKE_VAL_COMPLEX(35, 73), MAKE_VAL_COMPLEX(-72, 48), MAKE_VAL_COMPLEX(-116, -8), MAKE_VAL_COMPLEX(26, 51), MAKE_VAL_COMPLEX(114, -259), MAKE_VAL_COMPLEX(7, -8), MAKE_VAL_COMPLEX(134, -4), MAKE_VAL_COMPLEX(133, -1), MAKE_VAL_COMPLEX(-44, -95), MAKE_VAL_COMPLEX(170, -49) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_237, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_237", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_237() */
/* END TEST 237 */

/*
 *  BEGIN TEST 238
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_238( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-56, -24), MAKE_VAL_COMPLEX(100, -40), MAKE_VAL_COMPLEX(33, -19), MAKE_VAL_COMPLEX(-34, 2), MAKE_VAL_COMPLEX(17, 57), MAKE_VAL_COMPLEX(-40, -42), MAKE_VAL_COMPLEX(-12, 28), MAKE_VAL_COMPLEX(30, -70), MAKE_VAL_COMPLEX(-29, -29), MAKE_VAL_COMPLEX(-100, 40), MAKE_VAL_COMPLEX(-3, 7), MAKE_VAL_COMPLEX(-86, 46), MAKE_VAL_COMPLEX(-36, 26), MAKE_VAL_COMPLEX(51, 55), MAKE_VAL_COMPLEX(-78, 8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_238, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_238", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_238() */
/* END TEST 238 */

/*
 *  BEGIN TEST 239
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x192', LAYOUT_COLMAJ>
 *  y : <'y192', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_239( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A192_ptr, A192_ind, A192_val,
		A192_M, A192_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y192_val,
		y192_LEN * y192_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y192_LEN, y192_VW, LAYOUT_COLMAJ, y192_LEN );
	oski_value_t* y_val = array_Duplicate( x192_val,
		x192_LEN * x192_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x192_LEN, x192_VW, LAYOUT_COLMAJ, x192_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-56, 40), MAKE_VAL_COMPLEX(-100, 440), MAKE_VAL_COMPLEX(-183, -511), MAKE_VAL_COMPLEX(-18, 34), MAKE_VAL_COMPLEX(-223, -559), MAKE_VAL_COMPLEX(-64, 14), MAKE_VAL_COMPLEX(-500, -212), MAKE_VAL_COMPLEX(210, -430), MAKE_VAL_COMPLEX(-45, 11), MAKE_VAL_COMPLEX(904, -522), MAKE_VAL_COMPLEX(5, 7), MAKE_VAL_COMPLEX(-86, 150), MAKE_VAL_COMPLEX(80, 382), MAKE_VAL_COMPLEX(83, -17), MAKE_VAL_COMPLEX(376, 188) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-2, -4);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_239, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_239", y_true, y_val, x192_LEN*x192_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_239() */
/* END TEST 239 */

#define A240_M 4    /* # of rows */
#define A240_N 5    /* # of cols */
#define A240_NNZ 7  /* # of stored non-zeros */
static oski_index_t A240_ptr[A240_N+1] = {0, 1, 2, 5, 6, 7}; /* end row pointers for A240 */
static oski_index_t A240_ind[A240_NNZ] = {0, 1, 0, 1, 2, 2, 1}; /* end column indices pointers for A240 */
static oski_value_t A240_val[A240_NNZ] = {MAKE_VAL_COMPLEX(-10, -1), MAKE_VAL_COMPLEX(4, 10), MAKE_VAL_COMPLEX(0, -10), MAKE_VAL_COMPLEX(5, 1), MAKE_VAL_COMPLEX(0, 1), MAKE_VAL_COMPLEX(-1, 1), MAKE_VAL_COMPLEX(10, 3)}; /* end stored non-zero values for A240 */

#define x240_LEN 5   /* vector length */
#define x240_VW  3   /* vector width */
static oski_value_t x240_val[x240_LEN*x240_VW] = {MAKE_VAL_COMPLEX(-10, 7), MAKE_VAL_COMPLEX(-9, -10), MAKE_VAL_COMPLEX(-3, 6), MAKE_VAL_COMPLEX(-2, -9), MAKE_VAL_COMPLEX(-5, -8), MAKE_VAL_COMPLEX(3, -2), MAKE_VAL_COMPLEX(10, -6), MAKE_VAL_COMPLEX(-6, -9), MAKE_VAL_COMPLEX(-1, -4), MAKE_VAL_COMPLEX(-9, 6), MAKE_VAL_COMPLEX(3, 2), MAKE_VAL_COMPLEX(2, -6), MAKE_VAL_COMPLEX(8, 10), MAKE_VAL_COMPLEX(9, 1), MAKE_VAL_COMPLEX(7, 8)}; /* data */

#define y240_LEN 4   /* vector length */
#define y240_VW  3   /* vector width */
static oski_value_t y240_val[y240_LEN*y240_VW] = {MAKE_VAL_COMPLEX(1, -8), MAKE_VAL_COMPLEX(6, 6), MAKE_VAL_COMPLEX(3, 7), MAKE_VAL_COMPLEX(-7, -10), MAKE_VAL_COMPLEX(-7, 10), MAKE_VAL_COMPLEX(-2, 7), MAKE_VAL_COMPLEX(-0, -7), MAKE_VAL_COMPLEX(-10, 1), MAKE_VAL_COMPLEX(6, -8), MAKE_VAL_COMPLEX(-2, -9), MAKE_VAL_COMPLEX(10, 1), MAKE_VAL_COMPLEX(6, -5)}; /* data */

/*
 *  BEGIN TEST 240
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_240( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(49, -111), MAKE_VAL_COMPLEX(-52, 139), MAKE_VAL_COMPLEX(-29, -86), MAKE_VAL_COMPLEX(240, 101), MAKE_VAL_COMPLEX(158, -163), MAKE_VAL_COMPLEX(82, 58), MAKE_VAL_COMPLEX(16, 37), MAKE_VAL_COMPLEX(63, -21), MAKE_VAL_COMPLEX(-20, 61), MAKE_VAL_COMPLEX(21, 52), MAKE_VAL_COMPLEX(-61, 4), MAKE_VAL_COMPLEX(-31, 36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_240, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_240", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_240() */
/* END TEST 240 */

/*
 *  BEGIN TEST 241
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_241( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-14, 91), MAKE_VAL_COMPLEX(46, -97), MAKE_VAL_COMPLEX(10, 119), MAKE_VAL_COMPLEX(-221, -159), MAKE_VAL_COMPLEX(-219, 165), MAKE_VAL_COMPLEX(-116, -45), MAKE_VAL_COMPLEX(12, -58), MAKE_VAL_COMPLEX(-97, -16), MAKE_VAL_COMPLEX(70, -61), MAKE_VAL_COMPLEX(9, -87), MAKE_VAL_COMPLEX(87, 39), MAKE_VAL_COMPLEX(69, -27) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_241, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_241", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_241() */
/* END TEST 241 */

/*
 *  BEGIN TEST 242
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_242( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(48, 71), MAKE_VAL_COMPLEX(-90, -6), MAKE_VAL_COMPLEX(-73, -35), MAKE_VAL_COMPLEX(126, 31), MAKE_VAL_COMPLEX(-14, -129), MAKE_VAL_COMPLEX(-33, -70), MAKE_VAL_COMPLEX(49, 56), MAKE_VAL_COMPLEX(73, -78), MAKE_VAL_COMPLEX(8, 106), MAKE_VAL_COMPLEX(79, 58), MAKE_VAL_COMPLEX(-87, 62), MAKE_VAL_COMPLEX(-13, 82) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_242, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_242", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_242() */
/* END TEST 242 */

/*
 *  BEGIN TEST 243
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_243( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(525, 1139), MAKE_VAL_COMPLEX(-743, -1022), MAKE_VAL_COMPLEX(280, 311), MAKE_VAL_COMPLEX(-1417, 582), MAKE_VAL_COMPLEX(-325, 1276), MAKE_VAL_COMPLEX(-986, -178), MAKE_VAL_COMPLEX(-29, 26), MAKE_VAL_COMPLEX(-69, 57), MAKE_VAL_COMPLEX(-41, -58), MAKE_VAL_COMPLEX(-14, -63), MAKE_VAL_COMPLEX(70, 7), MAKE_VAL_COMPLEX(42, -35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_243, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_243", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_243() */
/* END TEST 243 */

/*
 *  BEGIN TEST 244
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_244( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-54, -15), MAKE_VAL_COMPLEX(44, -118), MAKE_VAL_COMPLEX(-65, -109), MAKE_VAL_COMPLEX(148, -108), MAKE_VAL_COMPLEX(-54, -54), MAKE_VAL_COMPLEX(-68, 36), MAKE_VAL_COMPLEX(-70, 25), MAKE_VAL_COMPLEX(98, -57), MAKE_VAL_COMPLEX(95, 9), MAKE_VAL_COMPLEX(-23, -77), MAKE_VAL_COMPLEX(-13, 9), MAKE_VAL_COMPLEX(46, 42), MAKE_VAL_COMPLEX(-136, -77), MAKE_VAL_COMPLEX(-126, 149), MAKE_VAL_COMPLEX(-119, 104) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_244, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_244", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_244() */
/* END TEST 244 */

/*
 *  BEGIN TEST 245
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_245( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(53, -178), MAKE_VAL_COMPLEX(167, 91), MAKE_VAL_COMPLEX(14, 16), MAKE_VAL_COMPLEX(-13, 163), MAKE_VAL_COMPLEX(203, 61), MAKE_VAL_COMPLEX(67, 21), MAKE_VAL_COMPLEX(58, 159), MAKE_VAL_COMPLEX(73, 60), MAKE_VAL_COMPLEX(-34, 14), MAKE_VAL_COMPLEX(26, -94), MAKE_VAL_COMPLEX(-40, 12), MAKE_VAL_COMPLEX(14, 38), MAKE_VAL_COMPLEX(-66, 91), MAKE_VAL_COMPLEX(32, -41), MAKE_VAL_COMPLEX(-48, -85) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_245, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_245", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_245() */
/* END TEST 245 */

/*
 *  BEGIN TEST 246
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_246( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-106, -45), MAKE_VAL_COMPLEX(35, -122), MAKE_VAL_COMPLEX(-63, 6), MAKE_VAL_COMPLEX(62, -61), MAKE_VAL_COMPLEX(39, -80), MAKE_VAL_COMPLEX(31, 14), MAKE_VAL_COMPLEX(98, 50), MAKE_VAL_COMPLEX(42, -93), MAKE_VAL_COMPLEX(27, -28), MAKE_VAL_COMPLEX(-93, -42), MAKE_VAL_COMPLEX(-1, 34), MAKE_VAL_COMPLEX(58, -14), MAKE_VAL_COMPLEX(-40, 114), MAKE_VAL_COMPLEX(37, 77), MAKE_VAL_COMPLEX(-29, 96) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_246, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_246", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_246() */
/* END TEST 246 */

/*
 *  BEGIN TEST 247
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_247( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, -590), MAKE_VAL_COMPLEX(496, 319), MAKE_VAL_COMPLEX(181, 370), MAKE_VAL_COMPLEX(-251, 754), MAKE_VAL_COMPLEX(874, 49), MAKE_VAL_COMPLEX(462, -97), MAKE_VAL_COMPLEX(657, 398), MAKE_VAL_COMPLEX(30, 170), MAKE_VAL_COMPLEX(-343, 16), MAKE_VAL_COMPLEX(-55, -122), MAKE_VAL_COMPLEX(-69, 88), MAKE_VAL_COMPLEX(8, -44), MAKE_VAL_COMPLEX(259, 702), MAKE_VAL_COMPLEX(478, -515), MAKE_VAL_COMPLEX(98, -408) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_247, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_247", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_247() */
/* END TEST 247 */

/*
 *  BEGIN TEST 248
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_248( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-38, 136), MAKE_VAL_COMPLEX(-131, -86), MAKE_VAL_COMPLEX(-40, -31), MAKE_VAL_COMPLEX(-156, -11), MAKE_VAL_COMPLEX(31, 80), MAKE_VAL_COMPLEX(73, 32), MAKE_VAL_COMPLEX(66, -83), MAKE_VAL_COMPLEX(-132, 94), MAKE_VAL_COMPLEX(-94, 43), MAKE_VAL_COMPLEX(-40, 55), MAKE_VAL_COMPLEX(30, 13), MAKE_VAL_COMPLEX(-16, -32), MAKE_VAL_COMPLEX(-40, -45), MAKE_VAL_COMPLEX(7, 108), MAKE_VAL_COMPLEX(52, 102) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_248, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_248", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_248() */
/* END TEST 248 */

/*
 *  BEGIN TEST 249
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_249( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(79, -120), MAKE_VAL_COMPLEX(119, 133), MAKE_VAL_COMPLEX(64, 28), MAKE_VAL_COMPLEX(133, 35), MAKE_VAL_COMPLEX(-45, -49), MAKE_VAL_COMPLEX(-85, -37), MAKE_VAL_COMPLEX(-104, 65), MAKE_VAL_COMPLEX(117, -58), MAKE_VAL_COMPLEX(84, -32), MAKE_VAL_COMPLEX(76, -40), MAKE_VAL_COMPLEX(-30, -26), MAKE_VAL_COMPLEX(-6, 38), MAKE_VAL_COMPLEX(54, 1), MAKE_VAL_COMPLEX(-22, -137), MAKE_VAL_COMPLEX(-42, -139) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_249, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_249", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_249() */
/* END TEST 249 */

/*
 *  BEGIN TEST 250
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_250( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-18, -47), MAKE_VAL_COMPLEX(49, -26), MAKE_VAL_COMPLEX(-21, -18), MAKE_VAL_COMPLEX(38, 1), MAKE_VAL_COMPLEX(37, -12), MAKE_VAL_COMPLEX(5, 14), MAKE_VAL_COMPLEX(14, 46), MAKE_VAL_COMPLEX(42, -15), MAKE_VAL_COMPLEX(17, 0), MAKE_VAL_COMPLEX(-15, -42), MAKE_VAL_COMPLEX(-11, 10), MAKE_VAL_COMPLEX(22, 14), MAKE_VAL_COMPLEX(-48, 22), MAKE_VAL_COMPLEX(-13, 35), MAKE_VAL_COMPLEX(-39, 20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_250, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_250", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_250() */
/* END TEST 250 */

/*
 *  BEGIN TEST 251
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_251( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-455, 234), MAKE_VAL_COMPLEX(65, -540), MAKE_VAL_COMPLEX(203, -374), MAKE_VAL_COMPLEX(-522, -581), MAKE_VAL_COMPLEX(-333, 656), MAKE_VAL_COMPLEX(-41, 454), MAKE_VAL_COMPLEX(291, 43), MAKE_VAL_COMPLEX(-887, -78), MAKE_VAL_COMPLEX(-543, -236), MAKE_VAL_COMPLEX(-95, -14), MAKE_VAL_COMPLEX(-7, 94), MAKE_VAL_COMPLEX(-34, -78), MAKE_VAL_COMPLEX(99, -691), MAKE_VAL_COMPLEX(-700, 184), MAKE_VAL_COMPLEX(-372, 271) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_251, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_251", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_251() */
/* END TEST 251 */

/*
 *  BEGIN TEST 252
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_252( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(51, -127), MAKE_VAL_COMPLEX(158, 30), MAKE_VAL_COMPLEX(-10, -30), MAKE_VAL_COMPLEX(38, 33), MAKE_VAL_COMPLEX(8, 122), MAKE_VAL_COMPLEX(127, -126), MAKE_VAL_COMPLEX(11, 23), MAKE_VAL_COMPLEX(39, -14), MAKE_VAL_COMPLEX(-20, -9), MAKE_VAL_COMPLEX(94, 136), MAKE_VAL_COMPLEX(-33, 13), MAKE_VAL_COMPLEX(-19, 26) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_252, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_252", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_252() */
/* END TEST 252 */

/*
 *  BEGIN TEST 253
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_253( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-14, 91), MAKE_VAL_COMPLEX(-152, 24), MAKE_VAL_COMPLEX(-3, 77), MAKE_VAL_COMPLEX(-33, -111), MAKE_VAL_COMPLEX(-83, -100), MAKE_VAL_COMPLEX(-165, 153), MAKE_VAL_COMPLEX(17, -58), MAKE_VAL_COMPLEX(-93, -21), MAKE_VAL_COMPLEX(82, -7), MAKE_VAL_COMPLEX(-68, -189), MAKE_VAL_COMPLEX(79, 32), MAKE_VAL_COMPLEX(69, -27) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_253, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_253", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_253() */
/* END TEST 253 */

/*
 *  BEGIN TEST 254
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_254( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(27, 44), MAKE_VAL_COMPLEX(-54, -6), MAKE_VAL_COMPLEX(-43, -23), MAKE_VAL_COMPLEX(75, 22), MAKE_VAL_COMPLEX(-5, -78), MAKE_VAL_COMPLEX(-18, -43), MAKE_VAL_COMPLEX(28, 35), MAKE_VAL_COMPLEX(46, -45), MAKE_VAL_COMPLEX(2, 64), MAKE_VAL_COMPLEX(46, 37), MAKE_VAL_COMPLEX(-54, 35), MAKE_VAL_COMPLEX(-10, 49) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_254, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_254", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_254() */
/* END TEST 254 */

/*
 *  BEGIN TEST 255
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_255( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-57, 1556), MAKE_VAL_COMPLEX(-1836, -32), MAKE_VAL_COMPLEX(-119, 67), MAKE_VAL_COMPLEX(74, -22), MAKE_VAL_COMPLEX(-294, -1603), MAKE_VAL_COMPLEX(-980, 1090), MAKE_VAL_COMPLEX(16, 67), MAKE_VAL_COMPLEX(14, -62), MAKE_VAL_COMPLEX(166, 467), MAKE_VAL_COMPLEX(-839, -758), MAKE_VAL_COMPLEX(-112, 11), MAKE_VAL_COMPLEX(18, 46) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-9, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_255, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_255", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_255() */
/* END TEST 255 */

/*
 *  BEGIN TEST 256
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_256( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(77, 87), MAKE_VAL_COMPLEX(84, 12), MAKE_VAL_COMPLEX(-20, 53), MAKE_VAL_COMPLEX(-69, 148), MAKE_VAL_COMPLEX(-88, 81), MAKE_VAL_COMPLEX(54, -59), MAKE_VAL_COMPLEX(-153, 15), MAKE_VAL_COMPLEX(81, 187), MAKE_VAL_COMPLEX(-96, -68), MAKE_VAL_COMPLEX(74, 5), MAKE_VAL_COMPLEX(-1, -20), MAKE_VAL_COMPLEX(-53, 35), MAKE_VAL_COMPLEX(44, -22), MAKE_VAL_COMPLEX(-90, 13), MAKE_VAL_COMPLEX(5, -179) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_256, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_256", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_256() */
/* END TEST 256 */

/*
 *  BEGIN TEST 257
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_257( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-61, -128), MAKE_VAL_COMPLEX(-37, 0), MAKE_VAL_COMPLEX(17, -77), MAKE_VAL_COMPLEX(93, -125), MAKE_VAL_COMPLEX(119, -67), MAKE_VAL_COMPLEX(-59, 71), MAKE_VAL_COMPLEX(135, 23), MAKE_VAL_COMPLEX(-45, -172), MAKE_VAL_COMPLEX(107, 78), MAKE_VAL_COMPLEX(-59, -41), MAKE_VAL_COMPLEX(-12, 20), MAKE_VAL_COMPLEX(59, -13), MAKE_VAL_COMPLEX(-88, 8), MAKE_VAL_COMPLEX(61, 2), MAKE_VAL_COMPLEX(-42, 169) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_257, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_257", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_257() */
/* END TEST 257 */

/*
 *  BEGIN TEST 258
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_258( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1, 44), MAKE_VAL_COMPLEX(-48, 7), MAKE_VAL_COMPLEX(12, 21), MAKE_VAL_COMPLEX(-31, -12), MAKE_VAL_COMPLEX(-34, -1), MAKE_VAL_COMPLEX(0, -13), MAKE_VAL_COMPLEX(2, -42), MAKE_VAL_COMPLEX(-39, 0), MAKE_VAL_COMPLEX(-14, -5), MAKE_VAL_COMPLEX(0, 39), MAKE_VAL_COMPLEX(12, -5), MAKE_VAL_COMPLEX(-14, -18), MAKE_VAL_COMPLEX(46, -4), MAKE_VAL_COMPLEX(21, -25), MAKE_VAL_COMPLEX(38, -5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_258, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_258", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_258() */
/* END TEST 258 */

/*
 *  BEGIN TEST 259
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_259( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(122, 415), MAKE_VAL_COMPLEX(-14, -653), MAKE_VAL_COMPLEX(-59, 533), MAKE_VAL_COMPLEX(89, 401), MAKE_VAL_COMPLEX(-321, 231), MAKE_VAL_COMPLEX(173, -483), MAKE_VAL_COMPLEX(-121, 265), MAKE_VAL_COMPLEX(630, 103), MAKE_VAL_COMPLEX(-628, -161), MAKE_VAL_COMPLEX(-91, 53), MAKE_VAL_COMPLEX(66, 14), MAKE_VAL_COMPLEX(-13, 37), MAKE_VAL_COMPLEX(452, 228), MAKE_VAL_COMPLEX(74, 375), MAKE_VAL_COMPLEX(-217, -372) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(4, -3);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_259, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_259", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_259() */
/* END TEST 259 */

/*
 *  BEGIN TEST 260
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_260( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(110, 92), MAKE_VAL_COMPLEX(63, 17), MAKE_VAL_COMPLEX(5, 62), MAKE_VAL_COMPLEX(44, 39), MAKE_VAL_COMPLEX(49, 134), MAKE_VAL_COMPLEX(-131, -20), MAKE_VAL_COMPLEX(17, 13), MAKE_VAL_COMPLEX(-122, 66), MAKE_VAL_COMPLEX(45, 41), MAKE_VAL_COMPLEX(103, 2), MAKE_VAL_COMPLEX(-16, -25), MAKE_VAL_COMPLEX(-59, 19), MAKE_VAL_COMPLEX(82, -76), MAKE_VAL_COMPLEX(-56, 15), MAKE_VAL_COMPLEX(-48, -182) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_260, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_260", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_260() */
/* END TEST 260 */

/*
 *  BEGIN TEST 261
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_261( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(10, -176), MAKE_VAL_COMPLEX(45, 103), MAKE_VAL_COMPLEX(31, -134), MAKE_VAL_COMPLEX(-20, 69), MAKE_VAL_COMPLEX(11, -38), MAKE_VAL_COMPLEX(95, 44), MAKE_VAL_COMPLEX(-137, 59), MAKE_VAL_COMPLEX(194, 42), MAKE_VAL_COMPLEX(-33, 7), MAKE_VAL_COMPLEX(5, -74), MAKE_VAL_COMPLEX(-20, 1), MAKE_VAL_COMPLEX(35, 53), MAKE_VAL_COMPLEX(-178, -44), MAKE_VAL_COMPLEX(-52, -27), MAKE_VAL_COMPLEX(-36, 86) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_261, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_261", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_261() */
/* END TEST 261 */

/*
 *  BEGIN TEST 262
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_262( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, 82), MAKE_VAL_COMPLEX(-94, -24), MAKE_VAL_COMPLEX(6, 48), MAKE_VAL_COMPLEX(-48, -46), MAKE_VAL_COMPLEX(-62, -28), MAKE_VAL_COMPLEX(10, -24), MAKE_VAL_COMPLEX(36, -76), MAKE_VAL_COMPLEX(-72, -30), MAKE_VAL_COMPLEX(-22, -20), MAKE_VAL_COMPLEX(-30, 72), MAKE_VAL_COMPLEX(26, 0), MAKE_VAL_COMPLEX(-12, -44), MAKE_VAL_COMPLEX(88, 28), MAKE_VAL_COMPLEX(58, -30), MAKE_VAL_COMPLEX(74, 20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_262, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_262", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_262() */
/* END TEST 262 */

/*
 *  BEGIN TEST 263
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_ROWMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_263( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_ROWMAJ, x240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-642, 362), MAKE_VAL_COMPLEX(1072, -190), MAKE_VAL_COMPLEX(-901, 69), MAKE_VAL_COMPLEX(711, 371), MAKE_VAL_COMPLEX(-1, 625), MAKE_VAL_COMPLEX(-359, -757), MAKE_VAL_COMPLEX(458, 1048), MAKE_VAL_COMPLEX(-292, -992), MAKE_VAL_COMPLEX(278, 452), MAKE_VAL_COMPLEX(33, -49), MAKE_VAL_COMPLEX(-85, 7), MAKE_VAL_COMPLEX(68, -126), MAKE_VAL_COMPLEX(70, 826), MAKE_VAL_COMPLEX(-505, 435), MAKE_VAL_COMPLEX(334, -690) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(5, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_263, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_263", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_263() */
/* END TEST 263 */

/*
 *  BEGIN TEST 264
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_264( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(131, -67), MAKE_VAL_COMPLEX(-68, 71), MAKE_VAL_COMPLEX(119, -90), MAKE_VAL_COMPLEX(-61, -203), MAKE_VAL_COMPLEX(-7, 133), MAKE_VAL_COMPLEX(171, 193), MAKE_VAL_COMPLEX(-30, -24), MAKE_VAL_COMPLEX(-21, 51), MAKE_VAL_COMPLEX(-36, -46), MAKE_VAL_COMPLEX(-53, -26), MAKE_VAL_COMPLEX(45, -46), MAKE_VAL_COMPLEX(-1, -50) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_264, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_264", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_264() */
/* END TEST 264 */

/*
 *  BEGIN TEST 265
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_265( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-154, -9), MAKE_VAL_COMPLEX(146, -41), MAKE_VAL_COMPLEX(-64, 141), MAKE_VAL_COMPLEX(-42, 141), MAKE_VAL_COMPLEX(-16, -15), MAKE_VAL_COMPLEX(-161, -122), MAKE_VAL_COMPLEX(2, -39), MAKE_VAL_COMPLEX(-65, -2), MAKE_VAL_COMPLEX(58, -50), MAKE_VAL_COMPLEX(-1, -47), MAKE_VAL_COMPLEX(49, 15), MAKE_VAL_COMPLEX(35, -19) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_265, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_265", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_265() */
/* END TEST 265 */

/*
 *  BEGIN TEST 266
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_266( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(69, -32), MAKE_VAL_COMPLEX(-18, 78), MAKE_VAL_COMPLEX(-41, 59), MAKE_VAL_COMPLEX(45, -106), MAKE_VAL_COMPLEX(-115, -6), MAKE_VAL_COMPLEX(-66, 19), MAKE_VAL_COMPLEX(56, -35), MAKE_VAL_COMPLEX(-58, -75), MAKE_VAL_COMPLEX(94, 8), MAKE_VAL_COMPLEX(62, -61), MAKE_VAL_COMPLEX(42, 85), MAKE_VAL_COMPLEX(70, 23) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_266, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_266", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_266() */
/* END TEST 266 */

/*
 *  BEGIN TEST 267
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_267( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1828, 1183), MAKE_VAL_COMPLEX(-1785, -327), MAKE_VAL_COMPLEX(1554, -212), MAKE_VAL_COMPLEX(1946, -1741), MAKE_VAL_COMPLEX(-892, 289), MAKE_VAL_COMPLEX(-155, 2728), MAKE_VAL_COMPLEX(51, 32), MAKE_VAL_COMPLEX(77, 46), MAKE_VAL_COMPLEX(-234, -56), MAKE_VAL_COMPLEX(40, -75), MAKE_VAL_COMPLEX(64, 67), MAKE_VAL_COMPLEX(72, 1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(9, 9);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_267, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_267", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_267() */
/* END TEST 267 */

/*
 *  BEGIN TEST 268
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_268( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-107, 37), MAKE_VAL_COMPLEX(106, -213), MAKE_VAL_COMPLEX(-152, -64), MAKE_VAL_COMPLEX(62, -43), MAKE_VAL_COMPLEX(-4, -188), MAKE_VAL_COMPLEX(-28, -53), MAKE_VAL_COMPLEX(-46, 16), MAKE_VAL_COMPLEX(53, -105), MAKE_VAL_COMPLEX(33, -34), MAKE_VAL_COMPLEX(-178, 40), MAKE_VAL_COMPLEX(-25, -44), MAKE_VAL_COMPLEX(-28, -2), MAKE_VAL_COMPLEX(23, 105), MAKE_VAL_COMPLEX(31, 81), MAKE_VAL_COMPLEX(-69, 145) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_268, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_268", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_268() */
/* END TEST 268 */

/*
 *  BEGIN TEST 269
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_269( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-21, -156), MAKE_VAL_COMPLEX(7, 57), MAKE_VAL_COMPLEX(59, 40), MAKE_VAL_COMPLEX(58, -12), MAKE_VAL_COMPLEX(101, 94), MAKE_VAL_COMPLEX(65, 89), MAKE_VAL_COMPLEX(160, 106), MAKE_VAL_COMPLEX(55, -6), MAKE_VAL_COMPLEX(20, 8), MAKE_VAL_COMPLEX(67, -148), MAKE_VAL_COMPLEX(6, 92), MAKE_VAL_COMPLEX(118, 12), MAKE_VAL_COMPLEX(-139, 37), MAKE_VAL_COMPLEX(-18, 48), MAKE_VAL_COMPLEX(-22, -23) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_269, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_269", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_269() */
/* END TEST 269 */

/*
 *  BEGIN TEST 270
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_270( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(66, -76), MAKE_VAL_COMPLEX(92, 62), MAKE_VAL_COMPLEX(12, -54), MAKE_VAL_COMPLEX(34, 68), MAKE_VAL_COMPLEX(56, 54), MAKE_VAL_COMPLEX(-20, 22), MAKE_VAL_COMPLEX(-68, 68), MAKE_VAL_COMPLEX(66, 60), MAKE_VAL_COMPLEX(16, 30), MAKE_VAL_COMPLEX(60, -66), MAKE_VAL_COMPLEX(-28, -10), MAKE_VAL_COMPLEX(-4, 52), MAKE_VAL_COMPLEX(-84, -64), MAKE_VAL_COMPLEX(-74, 10), MAKE_VAL_COMPLEX(-72, -50) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_270, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_270", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_270() */
/* END TEST 270 */

/*
 *  BEGIN TEST 271
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_271( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-441, -146), MAKE_VAL_COMPLEX(706, 463), MAKE_VAL_COMPLEX(408, -615), MAKE_VAL_COMPLEX(-25, 76), MAKE_VAL_COMPLEX(754, -213), MAKE_VAL_COMPLEX(386, -313), MAKE_VAL_COMPLEX(146, -734), MAKE_VAL_COMPLEX(195, 114), MAKE_VAL_COMPLEX(74, 69), MAKE_VAL_COMPLEX(-444, -633), MAKE_VAL_COMPLEX(424, -143), MAKE_VAL_COMPLEX(-50, -442), MAKE_VAL_COMPLEX(-96, 334), MAKE_VAL_COMPLEX(-121, 17), MAKE_VAL_COMPLEX(-420, -271) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 6);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_271, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_271", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_271() */
/* END TEST 271 */

/*
 *  BEGIN TEST 272
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_272( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-37, 31), MAKE_VAL_COMPLEX(-78, -15), MAKE_VAL_COMPLEX(-2, -48), MAKE_VAL_COMPLEX(38, -3), MAKE_VAL_COMPLEX(-60, -104), MAKE_VAL_COMPLEX(-56, -39), MAKE_VAL_COMPLEX(102, 160), MAKE_VAL_COMPLEX(-39, 97), MAKE_VAL_COMPLEX(31, 4), MAKE_VAL_COMPLEX(-70, 76), MAKE_VAL_COMPLEX(-47, -52), MAKE_VAL_COMPLEX(92, 58), MAKE_VAL_COMPLEX(-131, 101), MAKE_VAL_COMPLEX(-19, 47), MAKE_VAL_COMPLEX(-39, 111) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_272, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_272", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_272() */
/* END TEST 272 */

/*
 *  BEGIN TEST 273
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_273( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(92, -144), MAKE_VAL_COMPLEX(209, 60), MAKE_VAL_COMPLEX(-1, -21), MAKE_VAL_COMPLEX(25, 74), MAKE_VAL_COMPLEX(145, 151), MAKE_VAL_COMPLEX(39, 72), MAKE_VAL_COMPLEX(-162, -56), MAKE_VAL_COMPLEX(138, -46), MAKE_VAL_COMPLEX(-2, 27), MAKE_VAL_COMPLEX(121, -175), MAKE_VAL_COMPLEX(10, 49), MAKE_VAL_COMPLEX(-80, 6), MAKE_VAL_COMPLEX(9, -151), MAKE_VAL_COMPLEX(-67, -11), MAKE_VAL_COMPLEX(-64, -148) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_273, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_273", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_273() */
/* END TEST 273 */

/*
 *  BEGIN TEST 274
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_274( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(26, -48), MAKE_VAL_COMPLEX(56, 22), MAKE_VAL_COMPLEX(0, -30), MAKE_VAL_COMPLEX(26, 32), MAKE_VAL_COMPLEX(36, 22), MAKE_VAL_COMPLEX(-8, 14), MAKE_VAL_COMPLEX(-28, 44), MAKE_VAL_COMPLEX(42, 24), MAKE_VAL_COMPLEX(12, 14), MAKE_VAL_COMPLEX(24, -42), MAKE_VAL_COMPLEX(-16, -2), MAKE_VAL_COMPLEX(4, 28), MAKE_VAL_COMPLEX(-52, -24), MAKE_VAL_COMPLEX(-38, 14), MAKE_VAL_COMPLEX(-44, -18) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_274, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_274", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_274() */
/* END TEST 274 */

/*
 *  BEGIN TEST 275
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_275( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_ROWMAJ, y240_VW );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(590, 564), MAKE_VAL_COMPLEX(-602, 1162), MAKE_VAL_COMPLEX(-102, -380), MAKE_VAL_COMPLEX(-16, 48), MAKE_VAL_COMPLEX(-1278, 288), MAKE_VAL_COMPLEX(-814, 186), MAKE_VAL_COMPLEX(1360, 28), MAKE_VAL_COMPLEX(458, 1392), MAKE_VAL_COMPLEX(124, -56), MAKE_VAL_COMPLEX(686, 1100), MAKE_VAL_COMPLEX(-736, -100), MAKE_VAL_COMPLEX(756, -248), MAKE_VAL_COMPLEX(70, 1058), MAKE_VAL_COMPLEX(-12, 112), MAKE_VAL_COMPLEX(672, 482) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, -8);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_275, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_275", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_275() */
/* END TEST 275 */

/*
 *  BEGIN TEST 276
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_276( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(192, -35), MAKE_VAL_COMPLEX(5, -174), MAKE_VAL_COMPLEX(-13, 20), MAKE_VAL_COMPLEX(23, -31), MAKE_VAL_COMPLEX(-159, 66), MAKE_VAL_COMPLEX(-52, 59), MAKE_VAL_COMPLEX(35, -10), MAKE_VAL_COMPLEX(-13, -29), MAKE_VAL_COMPLEX(102, -93), MAKE_VAL_COMPLEX(169, 140), MAKE_VAL_COMPLEX(-13, 47), MAKE_VAL_COMPLEX(21, 13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_276, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_276", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_276() */
/* END TEST 276 */

/*
 *  BEGIN TEST 277
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_277( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-191, 27), MAKE_VAL_COMPLEX(1, 180), MAKE_VAL_COMPLEX(16, -13), MAKE_VAL_COMPLEX(-30, 21), MAKE_VAL_COMPLEX(152, -56), MAKE_VAL_COMPLEX(50, -52), MAKE_VAL_COMPLEX(-35, 3), MAKE_VAL_COMPLEX(3, 30), MAKE_VAL_COMPLEX(-96, 85), MAKE_VAL_COMPLEX(-171, -149), MAKE_VAL_COMPLEX(23, -46), MAKE_VAL_COMPLEX(-15, -18) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_277, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_277", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_277() */
/* END TEST 277 */

/*
 *  BEGIN TEST 278
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_278( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(8, -64), MAKE_VAL_COMPLEX(48, 48), MAKE_VAL_COMPLEX(24, 56), MAKE_VAL_COMPLEX(-56, -80), MAKE_VAL_COMPLEX(-56, 80), MAKE_VAL_COMPLEX(-16, 56), MAKE_VAL_COMPLEX(0, -56), MAKE_VAL_COMPLEX(-80, 8), MAKE_VAL_COMPLEX(48, -64), MAKE_VAL_COMPLEX(-16, -72), MAKE_VAL_COMPLEX(80, 8), MAKE_VAL_COMPLEX(48, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_278, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_278", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_278() */
/* END TEST 278 */

/*
 *  BEGIN TEST 279
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_279( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	oski_value_t* y_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-369, 368), MAKE_VAL_COMPLEX(345, 724), MAKE_VAL_COMPLEX(-55, 78), MAKE_VAL_COMPLEX(24, -136), MAKE_VAL_COMPLEX(76, -451), MAKE_VAL_COMPLEX(-101, -192), MAKE_VAL_COMPLEX(20, -19), MAKE_VAL_COMPLEX(-88, -72), MAKE_VAL_COMPLEX(102, 437), MAKE_VAL_COMPLEX(-686, -265), MAKE_VAL_COMPLEX(100, 0), MAKE_VAL_COMPLEX(88, 8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-3, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_279, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_279", y_true, y_val, y240_LEN*y240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_279() */
/* END TEST 279 */

/*
 *  BEGIN TEST 280
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_280( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-11, 89), MAKE_VAL_COMPLEX(-46, 93), MAKE_VAL_COMPLEX(-57, 32), MAKE_VAL_COMPLEX(-19, -2), MAKE_VAL_COMPLEX(34, 83), MAKE_VAL_COMPLEX(78, -96), MAKE_VAL_COMPLEX(-84, -2), MAKE_VAL_COMPLEX(81, 109), MAKE_VAL_COMPLEX(3, 8), MAKE_VAL_COMPLEX(-35, 73), MAKE_VAL_COMPLEX(-66, 71), MAKE_VAL_COMPLEX(76, -58), MAKE_VAL_COMPLEX(-72, -105), MAKE_VAL_COMPLEX(-10, 0), MAKE_VAL_COMPLEX(15, -103) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_280, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_280", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_280() */
/* END TEST 280 */

/*
 *  BEGIN TEST 281
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_281( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(64, 8), MAKE_VAL_COMPLEX(-53, -22), MAKE_VAL_COMPLEX(108, 1), MAKE_VAL_COMPLEX(-64, 11), MAKE_VAL_COMPLEX(-111, -46), MAKE_VAL_COMPLEX(-93, 67), MAKE_VAL_COMPLEX(40, -94), MAKE_VAL_COMPLEX(-168, -64), MAKE_VAL_COMPLEX(-40, -3), MAKE_VAL_COMPLEX(80, 14), MAKE_VAL_COMPLEX(87, -96), MAKE_VAL_COMPLEX(-128, 34), MAKE_VAL_COMPLEX(170, 43), MAKE_VAL_COMPLEX(28, -80), MAKE_VAL_COMPLEX(64, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_281, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_281", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_281() */
/* END TEST 281 */

/*
 *  BEGIN TEST 282
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_282( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(42, -89), MAKE_VAL_COMPLEX(103, 34), MAKE_VAL_COMPLEX(-3, -54), MAKE_VAL_COMPLEX(50, 55), MAKE_VAL_COMPLEX(67, 36), MAKE_VAL_COMPLEX(-13, 26), MAKE_VAL_COMPLEX(-46, 82), MAKE_VAL_COMPLEX(78, 39), MAKE_VAL_COMPLEX(23, 24), MAKE_VAL_COMPLEX(39, -78), MAKE_VAL_COMPLEX(-29, -2), MAKE_VAL_COMPLEX(10, 50), MAKE_VAL_COMPLEX(-96, -38), MAKE_VAL_COMPLEX(-67, 29), MAKE_VAL_COMPLEX(-81, -28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_282, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_282", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_282() */
/* END TEST 282 */

/*
 *  BEGIN TEST 283
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_283( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(540, -205), MAKE_VAL_COMPLEX(737, -179), MAKE_VAL_COMPLEX(410, 340), MAKE_VAL_COMPLEX(47, 31), MAKE_VAL_COMPLEX(393, -671), MAKE_VAL_COMPLEX(-946, -183), MAKE_VAL_COMPLEX(448, 534), MAKE_VAL_COMPLEX(376, -1117), MAKE_VAL_COMPLEX(36, -102), MAKE_VAL_COMPLEX(537, 16), MAKE_VAL_COMPLEX(795, 205), MAKE_VAL_COMPLEX(-680, -370), MAKE_VAL_COMPLEX(-361, 1052), MAKE_VAL_COMPLEX(147, 91), MAKE_VAL_COMPLEX(-705, 410) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-4, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_283, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_283", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_283() */
/* END TEST 283 */

/*
 *  BEGIN TEST 284
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_284( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-55, 133), MAKE_VAL_COMPLEX(20, -87), MAKE_VAL_COMPLEX(111, 70), MAKE_VAL_COMPLEX(-17, -62), MAKE_VAL_COMPLEX(40, -1), MAKE_VAL_COMPLEX(76, -122), MAKE_VAL_COMPLEX(116, 2), MAKE_VAL_COMPLEX(-155, -81), MAKE_VAL_COMPLEX(-17, -16), MAKE_VAL_COMPLEX(-47, 121), MAKE_VAL_COMPLEX(-32, 95), MAKE_VAL_COMPLEX(-92, -54), MAKE_VAL_COMPLEX(120, 59), MAKE_VAL_COMPLEX(46, -14), MAKE_VAL_COMPLEX(3, -43) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_284, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_284", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_284() */
/* END TEST 284 */

/*
 *  BEGIN TEST 285
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_285( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(7, -10), MAKE_VAL_COMPLEX(-161, 51), MAKE_VAL_COMPLEX(-102, 2), MAKE_VAL_COMPLEX(-55, -7), MAKE_VAL_COMPLEX(-133, -41), MAKE_VAL_COMPLEX(-61, 86), MAKE_VAL_COMPLEX(-62, -116), MAKE_VAL_COMPLEX(47, 36), MAKE_VAL_COMPLEX(-16, -14), MAKE_VAL_COMPLEX(2, -13), MAKE_VAL_COMPLEX(71, -95), MAKE_VAL_COMPLEX(74, -12), MAKE_VAL_COMPLEX(12, -17), MAKE_VAL_COMPLEX(41, -31), MAKE_VAL_COMPLEX(108, 73) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_285, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_285", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_285() */
/* END TEST 285 */

/*
 *  BEGIN TEST 286
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_286( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(37, -11), MAKE_VAL_COMPLEX(17, 39), MAKE_VAL_COMPLEX(15, -15), MAKE_VAL_COMPLEX(-3, 29), MAKE_VAL_COMPLEX(7, 29), MAKE_VAL_COMPLEX(-11, 3), MAKE_VAL_COMPLEX(-36, 8), MAKE_VAL_COMPLEX(9, 33), MAKE_VAL_COMPLEX(-1, 13), MAKE_VAL_COMPLEX(33, -9), MAKE_VAL_COMPLEX(-7, -9), MAKE_VAL_COMPLEX(-12, 16), MAKE_VAL_COMPLEX(-14, -38), MAKE_VAL_COMPLEX(-26, -12), MAKE_VAL_COMPLEX(-13, -31) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_286, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_286", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_286() */
/* END TEST 286 */

/*
 *  BEGIN TEST 287
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x240', LAYOUT_COLMAJ>
 *  y : <'y240', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_287( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A240_ptr, A240_ind, A240_val,
		A240_M, A240_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y240_val,
		y240_LEN * y240_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y240_LEN, y240_VW, LAYOUT_COLMAJ, y240_LEN );
	oski_value_t* y_val = array_Duplicate( x240_val,
		x240_LEN * x240_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x240_LEN, x240_VW, LAYOUT_COLMAJ, x240_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-445, 187), MAKE_VAL_COMPLEX(132, 358), MAKE_VAL_COMPLEX(-20, 712), MAKE_VAL_COMPLEX(-16, -50), MAKE_VAL_COMPLEX(-220, 398), MAKE_VAL_COMPLEX(607, 159), MAKE_VAL_COMPLEX(-134, 250), MAKE_VAL_COMPLEX(-47, -619), MAKE_VAL_COMPLEX(-74, -54), MAKE_VAL_COMPLEX(-415, 183), MAKE_VAL_COMPLEX(-446, -176), MAKE_VAL_COMPLEX(-38, -566), MAKE_VAL_COMPLEX(151, 349), MAKE_VAL_COMPLEX(124, -102), MAKE_VAL_COMPLEX(477, -297) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_287, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_287", y_true, y_val, x240_LEN*x240_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_287() */
/* END TEST 287 */

#define A288_M 4    /* # of rows */
#define A288_N 5    /* # of cols */
#define A288_NNZ 4  /* # of stored non-zeros */
static oski_index_t A288_ptr[A288_N+1] = {1, 1, 2, 3, 4, 5}; /* end row pointers for A288 */
static oski_index_t A288_ind[A288_NNZ] = {1, 2, 1, 4}; /* end column indices pointers for A288 */
static oski_value_t A288_val[A288_NNZ] = {MAKE_VAL_COMPLEX(-6, -2), MAKE_VAL_COMPLEX(-5, 10), MAKE_VAL_COMPLEX(0, -8), MAKE_VAL_COMPLEX(9, -9)}; /* end stored non-zero values for A288 */

#define x288_LEN 5   /* vector length */
#define x288_VW  3   /* vector width */
static oski_value_t x288_val[x288_LEN*x288_VW] = {MAKE_VAL_COMPLEX(-8, -4), MAKE_VAL_COMPLEX(-6, 8), MAKE_VAL_COMPLEX(10, 4), MAKE_VAL_COMPLEX(1, 4), MAKE_VAL_COMPLEX(6, -9), MAKE_VAL_COMPLEX(10, -6), MAKE_VAL_COMPLEX(-6, 0), MAKE_VAL_COMPLEX(10, 0), MAKE_VAL_COMPLEX(-9, 7), MAKE_VAL_COMPLEX(-7, -10), MAKE_VAL_COMPLEX(6, 2), MAKE_VAL_COMPLEX(8, 0), MAKE_VAL_COMPLEX(8, -3), MAKE_VAL_COMPLEX(8, 10), MAKE_VAL_COMPLEX(-10, -10)}; /* data */

#define y288_LEN 4   /* vector length */
#define y288_VW  3   /* vector width */
static oski_value_t y288_val[y288_LEN*y288_VW] = {MAKE_VAL_COMPLEX(4, 3), MAKE_VAL_COMPLEX(1, -10), MAKE_VAL_COMPLEX(7, 6), MAKE_VAL_COMPLEX(-3, -10), MAKE_VAL_COMPLEX(-7, -5), MAKE_VAL_COMPLEX(-2, 10), MAKE_VAL_COMPLEX(-2, 0), MAKE_VAL_COMPLEX(-3, -4), MAKE_VAL_COMPLEX(7, -10), MAKE_VAL_COMPLEX(-5, 9), MAKE_VAL_COMPLEX(-10, -4), MAKE_VAL_COMPLEX(10, 5)}; /* data */

/*
 *  BEGIN TEST 288
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_288( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-113, 12), MAKE_VAL_COMPLEX(-40, 63), MAKE_VAL_COMPLEX(-110, -73), MAKE_VAL_COMPLEX(59, 1), MAKE_VAL_COMPLEX(3, 114), MAKE_VAL_COMPLEX(-13, -193), MAKE_VAL_COMPLEX(6, -2), MAKE_VAL_COMPLEX(32, 21), MAKE_VAL_COMPLEX(-41, 74), MAKE_VAL_COMPLEX(59, -168), MAKE_VAL_COMPLEX(232, 34), MAKE_VAL_COMPLEX(-237, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_288, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_288", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_288() */
/* END TEST 288 */

/*
 *  BEGIN TEST 289
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_289( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(101, 29), MAKE_VAL_COMPLEX(123, -85), MAKE_VAL_COMPLEX(83, 147), MAKE_VAL_COMPLEX(12, -55), MAKE_VAL_COMPLEX(16, -185), MAKE_VAL_COMPLEX(-73, 207), MAKE_VAL_COMPLEX(-12, -14), MAKE_VAL_COMPLEX(-9, -57), MAKE_VAL_COMPLEX(142, -48), MAKE_VAL_COMPLEX(-146, 155), MAKE_VAL_COMPLEX(-230, -126), MAKE_VAL_COMPLEX(227, 115) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_289, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_289", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_289() */
/* END TEST 289 */

/*
 *  BEGIN TEST 290
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_290( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(50, 0), MAKE_VAL_COMPLEX(-52, -86), MAKE_VAL_COMPLEX(92, 6), MAKE_VAL_COMPLEX(-84, -62), MAKE_VAL_COMPLEX(-86, 2), MAKE_VAL_COMPLEX(44, 92), MAKE_VAL_COMPLEX(-16, 12), MAKE_VAL_COMPLEX(-48, -14), MAKE_VAL_COMPLEX(-4, -122), MAKE_VAL_COMPLEX(14, 102), MAKE_VAL_COMPLEX(-104, 28), MAKE_VAL_COMPLEX(110, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_290, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_290", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_290() */
/* END TEST 290 */

/*
 *  BEGIN TEST 291
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_291( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(341, -784), MAKE_VAL_COMPLEX(198, -305), MAKE_VAL_COMPLEX(696, -217), MAKE_VAL_COMPLEX(162, 588), MAKE_VAL_COMPLEX(-384, -809), MAKE_VAL_COMPLEX(1061, 671), MAKE_VAL_COMPLEX(40, -32), MAKE_VAL_COMPLEX(-74, 93), MAKE_VAL_COMPLEX(-59, -120), MAKE_VAL_COMPLEX(590, 927), MAKE_VAL_COMPLEX(-1148, 1114), MAKE_VAL_COMPLEX(1037, -1264) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_291, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_291", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_291() */
/* END TEST 291 */

/*
 *  BEGIN TEST 292
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_292( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, -85), MAKE_VAL_COMPLEX(-99, -10), MAKE_VAL_COMPLEX(35, 110), MAKE_VAL_COMPLEX(-47, -4), MAKE_VAL_COMPLEX(75, 47), MAKE_VAL_COMPLEX(76, 4), MAKE_VAL_COMPLEX(77, -28), MAKE_VAL_COMPLEX(142, 31), MAKE_VAL_COMPLEX(-193, -110), MAKE_VAL_COMPLEX(57, -139), MAKE_VAL_COMPLEX(-70, 48), MAKE_VAL_COMPLEX(106, 13), MAKE_VAL_COMPLEX(108, 172), MAKE_VAL_COMPLEX(-158, 178), MAKE_VAL_COMPLEX(155, -185) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_292, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_292", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_292() */
/* END TEST 292 */

/*
 *  BEGIN TEST 293
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_293( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(56, 17), MAKE_VAL_COMPLEX(33, -52), MAKE_VAL_COMPLEX(-81, -24), MAKE_VAL_COMPLEX(10, 9), MAKE_VAL_COMPLEX(0, 16), MAKE_VAL_COMPLEX(-32, 92), MAKE_VAL_COMPLEX(-71, -26), MAKE_VAL_COMPLEX(-152, 59), MAKE_VAL_COMPLEX(139, 22), MAKE_VAL_COMPLEX(40, 86), MAKE_VAL_COMPLEX(46, 4), MAKE_VAL_COMPLEX(-114, 59), MAKE_VAL_COMPLEX(-89, -97), MAKE_VAL_COMPLEX(60, -116), MAKE_VAL_COMPLEX(-55, 105) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_293, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_293", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_293() */
/* END TEST 293 */

/*
 *  BEGIN TEST 294
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_294( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(16, 48), MAKE_VAL_COMPLEX(56, -8), MAKE_VAL_COMPLEX(-24, -56), MAKE_VAL_COMPLEX(12, -20), MAKE_VAL_COMPLEX(-60, 12), MAKE_VAL_COMPLEX(-64, -16), MAKE_VAL_COMPLEX(24, 24), MAKE_VAL_COMPLEX(-40, -40), MAKE_VAL_COMPLEX(64, 8), MAKE_VAL_COMPLEX(-12, 68), MAKE_VAL_COMPLEX(-16, -32), MAKE_VAL_COMPLEX(-32, -32), MAKE_VAL_COMPLEX(-44, -20), MAKE_VAL_COMPLEX(8, -72), MAKE_VAL_COMPLEX(0, 80) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_294, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_294", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_294() */
/* END TEST 294 */

/*
 *  BEGIN TEST 295
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_295( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(29, 89), MAKE_VAL_COMPLEX(113, -73), MAKE_VAL_COMPLEX(-5, 21), MAKE_VAL_COMPLEX(45, -377), MAKE_VAL_COMPLEX(-556, 218), MAKE_VAL_COMPLEX(-88, -456), MAKE_VAL_COMPLEX(715, 729), MAKE_VAL_COMPLEX(779, 27), MAKE_VAL_COMPLEX(-117, -967), MAKE_VAL_COMPLEX(236, 2), MAKE_VAL_COMPLEX(-586, -566), MAKE_VAL_COMPLEX(629, -99), MAKE_VAL_COMPLEX(-422, 1042), MAKE_VAL_COMPLEX(-1144, -324), MAKE_VAL_COMPLEX(1170, 440) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(7, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_295, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_295", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_295() */
/* END TEST 295 */

/*
 *  BEGIN TEST 296
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_296( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(48, 75), MAKE_VAL_COMPLEX(89, -44), MAKE_VAL_COMPLEX(-53, -76), MAKE_VAL_COMPLEX(-21, -57), MAKE_VAL_COMPLEX(-86, 99), MAKE_VAL_COMPLEX(-166, -14), MAKE_VAL_COMPLEX(-39, 110), MAKE_VAL_COMPLEX(-98, 41), MAKE_VAL_COMPLEX(224, -51), MAKE_VAL_COMPLEX(-23, 156), MAKE_VAL_COMPLEX(32, -42), MAKE_VAL_COMPLEX(-102, 21), MAKE_VAL_COMPLEX(-205, 20), MAKE_VAL_COMPLEX(-68, -246), MAKE_VAL_COMPLEX(75, 265) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_296, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_296", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_296() */
/* END TEST 296 */

/*
 *  BEGIN TEST 297
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_297( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(44, -59), MAKE_VAL_COMPLEX(-53, -54), MAKE_VAL_COMPLEX(-59, 66), MAKE_VAL_COMPLEX(-1, 20), MAKE_VAL_COMPLEX(53, 9), MAKE_VAL_COMPLEX(84, 104), MAKE_VAL_COMPLEX(99, -128), MAKE_VAL_COMPLEX(-2, -11), MAKE_VAL_COMPLEX(-155, -46), MAKE_VAL_COMPLEX(123, -77), MAKE_VAL_COMPLEX(-98, 40), MAKE_VAL_COMPLEX(22, 3), MAKE_VAL_COMPLEX(134, 34), MAKE_VAL_COMPLEX(-42, 170), MAKE_VAL_COMPLEX(55, -195) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_297, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_297", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_297() */
/* END TEST 297 */

/*
 *  BEGIN TEST 298
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_298( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-72, 44), MAKE_VAL_COMPLEX(34, 88), MAKE_VAL_COMPLEX(82, -60), MAKE_VAL_COMPLEX(37, 12), MAKE_VAL_COMPLEX(-42, -93), MAKE_VAL_COMPLEX(2, -110), MAKE_VAL_COMPLEX(-30, 48), MAKE_VAL_COMPLEX(50, -80), MAKE_VAL_COMPLEX(11, 107), MAKE_VAL_COMPLEX(-115, 6), MAKE_VAL_COMPLEX(46, -38), MAKE_VAL_COMPLEX(40, -64), MAKE_VAL_COMPLEX(16, -79), MAKE_VAL_COMPLEX(120, -14), MAKE_VAL_COMPLEX(-130, 30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_298, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_298", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_298() */
/* END TEST 298 */

/*
 *  BEGIN TEST 299
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_299( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(9, -15), MAKE_VAL_COMPLEX(7, 15), MAKE_VAL_COMPLEX(-79, -15), MAKE_VAL_COMPLEX(174, 57), MAKE_VAL_COMPLEX(-92, -239), MAKE_VAL_COMPLEX(274, 42), MAKE_VAL_COMPLEX(373, -499), MAKE_VAL_COMPLEX(-31, -453), MAKE_VAL_COMPLEX(-532, 278), MAKE_VAL_COMPLEX(145, -218), MAKE_VAL_COMPLEX(-376, 56), MAKE_VAL_COMPLEX(105, -327), MAKE_VAL_COMPLEX(576, -281), MAKE_VAL_COMPLEX(352, 562), MAKE_VAL_COMPLEX(-310, -620) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-5, 1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_299, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_299", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_299() */
/* END TEST 299 */

/*
 *  BEGIN TEST 300
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_300( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-71, 31), MAKE_VAL_COMPLEX(24, -87), MAKE_VAL_COMPLEX(21, 11), MAKE_VAL_COMPLEX(19, -136), MAKE_VAL_COMPLEX(-70, -6), MAKE_VAL_COMPLEX(-40, 123), MAKE_VAL_COMPLEX(4, 2), MAKE_VAL_COMPLEX(155, 11), MAKE_VAL_COMPLEX(-51, -81), MAKE_VAL_COMPLEX(-21, -99), MAKE_VAL_COMPLEX(-43, 5), MAKE_VAL_COMPLEX(-137, 5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_300, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_300", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_300() */
/* END TEST 300 */

/*
 *  BEGIN TEST 301
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_301( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(83, -22), MAKE_VAL_COMPLEX(-21, 57), MAKE_VAL_COMPLEX(0, 7), MAKE_VAL_COMPLEX(-28, 106), MAKE_VAL_COMPLEX(49, -9), MAKE_VAL_COMPLEX(34, -93), MAKE_VAL_COMPLEX(-10, -2), MAKE_VAL_COMPLEX(-164, -23), MAKE_VAL_COMPLEX(72, 51), MAKE_VAL_COMPLEX(6, 126), MAKE_VAL_COMPLEX(13, -17), MAKE_VAL_COMPLEX(167, 10) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_301, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_301", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_301() */
/* END TEST 301 */

/*
 *  BEGIN TEST 302
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_302( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-35, 5), MAKE_VAL_COMPLEX(45, 55), MAKE_VAL_COMPLEX(-65, 5), MAKE_VAL_COMPLEX(65, 35), MAKE_VAL_COMPLEX(60, -10), MAKE_VAL_COMPLEX(-40, -60), MAKE_VAL_COMPLEX(10, -10), MAKE_VAL_COMPLEX(35, 5), MAKE_VAL_COMPLEX(15, 85), MAKE_VAL_COMPLEX(-20, -70), MAKE_VAL_COMPLEX(70, -30), MAKE_VAL_COMPLEX(-75, 25) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_302, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_302", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_302() */
/* END TEST 302 */

/*
 *  BEGIN TEST 303
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_303( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-121, 163), MAKE_VAL_COMPLEX(23, -212), MAKE_VAL_COMPLEX(31, 55), MAKE_VAL_COMPLEX(-44, -329), MAKE_VAL_COMPLEX(-130, 6), MAKE_VAL_COMPLEX(-21, 294), MAKE_VAL_COMPLEX(6, -12), MAKE_VAL_COMPLEX(339, -159), MAKE_VAL_COMPLEX(-109, -89), MAKE_VAL_COMPLEX(-205, -189), MAKE_VAL_COMPLEX(-77, -15), MAKE_VAL_COMPLEX(-279, 217) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(2, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_303, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_303", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_303() */
/* END TEST 303 */

/*
 *  BEGIN TEST 304
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_304( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, 51), MAKE_VAL_COMPLEX(49, -13), MAKE_VAL_COMPLEX(-17, -66), MAKE_VAL_COMPLEX(-5, -56), MAKE_VAL_COMPLEX(-30, 66), MAKE_VAL_COMPLEX(-131, 39), MAKE_VAL_COMPLEX(126, 90), MAKE_VAL_COMPLEX(-132, -110), MAKE_VAL_COMPLEX(-11, -91), MAKE_VAL_COMPLEX(9, 26), MAKE_VAL_COMPLEX(-59, 20), MAKE_VAL_COMPLEX(-102, -83), MAKE_VAL_COMPLEX(-161, -83), MAKE_VAL_COMPLEX(-55, -81), MAKE_VAL_COMPLEX(135, 35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_304, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_304", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_304() */
/* END TEST 304 */

/*
 *  BEGIN TEST 305
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_305( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-76, 41), MAKE_VAL_COMPLEX(41, 93), MAKE_VAL_COMPLEX(75, -50), MAKE_VAL_COMPLEX(54, 48), MAKE_VAL_COMPLEX(-72, -147), MAKE_VAL_COMPLEX(69, -165), MAKE_VAL_COMPLEX(-132, -18), MAKE_VAL_COMPLEX(142, -10), MAKE_VAL_COMPLEX(86, 206), MAKE_VAL_COMPLEX(-136, 48), MAKE_VAL_COMPLEX(89, -90), MAKE_VAL_COMPLEX(110, -13), MAKE_VAL_COMPLEX(133, -16), MAKE_VAL_COMPLEX(183, -5), MAKE_VAL_COMPLEX(-265, 75) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_305, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_305", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_305() */
/* END TEST 305 */

/*
 *  BEGIN TEST 306
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_306( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-36, -28), MAKE_VAL_COMPLEX(-38, 34), MAKE_VAL_COMPLEX(46, 30), MAKE_VAL_COMPLEX(1, 21), MAKE_VAL_COMPLEX(39, -39), MAKE_VAL_COMPLEX(56, -20), MAKE_VAL_COMPLEX(-30, -6), MAKE_VAL_COMPLEX(50, 10), MAKE_VAL_COMPLEX(-52, 26), MAKE_VAL_COMPLEX(-25, -57), MAKE_VAL_COMPLEX(28, 16), MAKE_VAL_COMPLEX(40, 8), MAKE_VAL_COMPLEX(43, -7), MAKE_VAL_COMPLEX(30, 58), MAKE_VAL_COMPLEX(-40, -60) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_306, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_306", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_306() */
/* END TEST 306 */

/*
 *  BEGIN TEST 307
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_307( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-78, -22), MAKE_VAL_COMPLEX(34, 22), MAKE_VAL_COMPLEX(64, 132), MAKE_VAL_COMPLEX(319, 135), MAKE_VAL_COMPLEX(-465, -183), MAKE_VAL_COMPLEX(128, -752), MAKE_VAL_COMPLEX(-1038, 210), MAKE_VAL_COMPLEX(1022, -122), MAKE_VAL_COMPLEX(992, 170), MAKE_VAL_COMPLEX(101, 321), MAKE_VAL_COMPLEX(-26, -554), MAKE_VAL_COMPLEX(766, -106), MAKE_VAL_COMPLEX(1123, -331), MAKE_VAL_COMPLEX(462, -266), MAKE_VAL_COMPLEX(-580, 1020) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 6);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_307, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_307", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_307() */
/* END TEST 307 */

/*
 *  BEGIN TEST 308
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_308( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-52, 15), MAKE_VAL_COMPLEX(-5, 59), MAKE_VAL_COMPLEX(73, -30), MAKE_VAL_COMPLEX(-8, -4), MAKE_VAL_COMPLEX(44, -43), MAKE_VAL_COMPLEX(-1, 13), MAKE_VAL_COMPLEX(-128, 70), MAKE_VAL_COMPLEX(158, -70), MAKE_VAL_COMPLEX(88, 72), MAKE_VAL_COMPLEX(-102, 0), MAKE_VAL_COMPLEX(75, -74), MAKE_VAL_COMPLEX(130, 29), MAKE_VAL_COMPLEX(91, -164), MAKE_VAL_COMPLEX(89, -45), MAKE_VAL_COMPLEX(-45, 125) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_308, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_308", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_308() */
/* END TEST 308 */

/*
 *  BEGIN TEST 309
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_309( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, -87), MAKE_VAL_COMPLEX(-91, 19), MAKE_VAL_COMPLEX(35, 108), MAKE_VAL_COMPLEX(8, 55), MAKE_VAL_COMPLEX(55, -47), MAKE_VAL_COMPLEX(139, -55), MAKE_VAL_COMPLEX(56, -88), MAKE_VAL_COMPLEX(-38, 100), MAKE_VAL_COMPLEX(-217, -15), MAKE_VAL_COMPLEX(48, -141), MAKE_VAL_COMPLEX(-9, 116), MAKE_VAL_COMPLEX(-34, -5), MAKE_VAL_COMPLEX(14, 152), MAKE_VAL_COMPLEX(-23, 189), MAKE_VAL_COMPLEX(-45, -275) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_309, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_309", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_309() */
/* END TEST 309 */

/*
 *  BEGIN TEST 310
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_310( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(56, -32), MAKE_VAL_COMPLEX(-24, -68), MAKE_VAL_COMPLEX(-64, 44), MAKE_VAL_COMPLEX(-28, -10), MAKE_VAL_COMPLEX(30, 72), MAKE_VAL_COMPLEX(-4, 84), MAKE_VAL_COMPLEX(24, -36), MAKE_VAL_COMPLEX(-40, 60), MAKE_VAL_COMPLEX(-6, -82), MAKE_VAL_COMPLEX(88, -2), MAKE_VAL_COMPLEX(-36, 28), MAKE_VAL_COMPLEX(-32, 48), MAKE_VAL_COMPLEX(-14, 60), MAKE_VAL_COMPLEX(-92, 8), MAKE_VAL_COMPLEX(100, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_310, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_310", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_310() */
/* END TEST 310 */

/*
 *  BEGIN TEST 311
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_ROWMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_311( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_ROWMAJ, x288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-38, -79), MAKE_VAL_COMPLEX(-81, 5), MAKE_VAL_COMPLEX(55, 122), MAKE_VAL_COMPLEX(48, -3), MAKE_VAL_COMPLEX(3, 53), MAKE_VAL_COMPLEX(-27, -109), MAKE_VAL_COMPLEX(-36, -284), MAKE_VAL_COMPLEX(22, 316), MAKE_VAL_COMPLEX(-219, 195), MAKE_VAL_COMPLEX(4, -195), MAKE_VAL_COMPLEX(111, 190), MAKE_VAL_COMPLEX(-156, 175), MAKE_VAL_COMPLEX(248, 278), MAKE_VAL_COMPLEX(103, 207), MAKE_VAL_COMPLEX(-315, -185) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_311, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_311", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_311() */
/* END TEST 311 */

/*
 *  BEGIN TEST 312
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_312( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(104, -27), MAKE_VAL_COMPLEX(109, 8), MAKE_VAL_COMPLEX(87, -36), MAKE_VAL_COMPLEX(-117, 18), MAKE_VAL_COMPLEX(-105, 65), MAKE_VAL_COMPLEX(-16, 165), MAKE_VAL_COMPLEX(-4, 4), MAKE_VAL_COMPLEX(-11, -28), MAKE_VAL_COMPLEX(57, -73), MAKE_VAL_COMPLEX(-61, -68), MAKE_VAL_COMPLEX(-232, -48), MAKE_VAL_COMPLEX(-102, 45) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_312, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_312", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_312() */
/* END TEST 312 */

/*
 *  BEGIN TEST 313
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_313( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-36, 53), MAKE_VAL_COMPLEX(-135, -152), MAKE_VAL_COMPLEX(35, 92), MAKE_VAL_COMPLEX(35, -146), MAKE_VAL_COMPLEX(-13, -107), MAKE_VAL_COMPLEX(28, -17), MAKE_VAL_COMPLEX(-24, 4), MAKE_VAL_COMPLEX(-47, -16), MAKE_VAL_COMPLEX(1, -95), MAKE_VAL_COMPLEX(27, 214), MAKE_VAL_COMPLEX(76, 32), MAKE_VAL_COMPLEX(262, -15) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_313, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_313", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_313() */
/* END TEST 313 */

/*
 *  BEGIN TEST 314
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_314( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(5, 35), MAKE_VAL_COMPLEX(55, -45), MAKE_VAL_COMPLEX(5, 65), MAKE_VAL_COMPLEX(35, -65), MAKE_VAL_COMPLEX(-10, -60), MAKE_VAL_COMPLEX(-60, 40), MAKE_VAL_COMPLEX(-10, -10), MAKE_VAL_COMPLEX(5, -35), MAKE_VAL_COMPLEX(85, -15), MAKE_VAL_COMPLEX(-70, 20), MAKE_VAL_COMPLEX(-30, -70), MAKE_VAL_COMPLEX(25, 75) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_314, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_314", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_314() */
/* END TEST 314 */

/*
 *  BEGIN TEST 315
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_315( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(133, -684), MAKE_VAL_COMPLEX(956, -305), MAKE_VAL_COMPLEX(-204, -591), MAKE_VAL_COMPLEX(-6, 945), MAKE_VAL_COMPLEX(249, 817), MAKE_VAL_COMPLEX(574, 573), MAKE_VAL_COMPLEX(58, -34), MAKE_VAL_COMPLEX(14, -83), MAKE_VAL_COMPLEX(48, -164), MAKE_VAL_COMPLEX(-929, -412), MAKE_VAL_COMPLEX(-1022, 870), MAKE_VAL_COMPLEX(-705, 1092) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(5, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_315, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_315", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_315() */
/* END TEST 315 */

/*
 *  BEGIN TEST 316
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_316( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(52, -13), MAKE_VAL_COMPLEX(-29, -92), MAKE_VAL_COMPLEX(57, 44), MAKE_VAL_COMPLEX(-1, -35), MAKE_VAL_COMPLEX(48, 186), MAKE_VAL_COMPLEX(-15, 54), MAKE_VAL_COMPLEX(-9, 29), MAKE_VAL_COMPLEX(42, -9), MAKE_VAL_COMPLEX(-82, -76), MAKE_VAL_COMPLEX(-58, 66), MAKE_VAL_COMPLEX(-25, 22), MAKE_VAL_COMPLEX(-64, -8), MAKE_VAL_COMPLEX(-103, -36), MAKE_VAL_COMPLEX(-14, -59), MAKE_VAL_COMPLEX(215, -45) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_316, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_316", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_316() */
/* END TEST 316 */

/*
 *  BEGIN TEST 317
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_317( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-8, -95), MAKE_VAL_COMPLEX(-81, 22), MAKE_VAL_COMPLEX(-99, 90), MAKE_VAL_COMPLEX(-50, 52), MAKE_VAL_COMPLEX(75, -117), MAKE_VAL_COMPLEX(103, 70), MAKE_VAL_COMPLEX(3, -107), MAKE_VAL_COMPLEX(-32, 139), MAKE_VAL_COMPLEX(-18, -34), MAKE_VAL_COMPLEX(181, -167), MAKE_VAL_COMPLEX(5, 58), MAKE_VAL_COMPLEX(72, 112), MAKE_VAL_COMPLEX(150, 137), MAKE_VAL_COMPLEX(-108, 173), MAKE_VAL_COMPLEX(-95, -95) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_317, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_317", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_317() */
/* END TEST 317 */

/*
 *  BEGIN TEST 318
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_318( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-40, -40), MAKE_VAL_COMPLEX(-52, 36), MAKE_VAL_COMPLEX(52, 44), MAKE_VAL_COMPLEX(-2, 26), MAKE_VAL_COMPLEX(54, -42), MAKE_VAL_COMPLEX(72, -16), MAKE_VAL_COMPLEX(-36, -12), MAKE_VAL_COMPLEX(60, 20), MAKE_VAL_COMPLEX(-68, 24), MAKE_VAL_COMPLEX(-22, -74), MAKE_VAL_COMPLEX(32, 24), MAKE_VAL_COMPLEX(48, 16), MAKE_VAL_COMPLEX(54, -2), MAKE_VAL_COMPLEX(28, 76), MAKE_VAL_COMPLEX(-40, -80) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_318, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_318", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_318() */
/* END TEST 318 */

/*
 *  BEGIN TEST 319
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_319( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, -9), MAKE_VAL_COMPLEX(-257, 123), MAKE_VAL_COMPLEX(261, -791), MAKE_VAL_COMPLEX(-134, -158), MAKE_VAL_COMPLEX(900, -138), MAKE_VAL_COMPLEX(-81, -37), MAKE_VAL_COMPLEX(338, 296), MAKE_VAL_COMPLEX(-261, -643), MAKE_VAL_COMPLEX(-160, 636), MAKE_VAL_COMPLEX(232, 950), MAKE_VAL_COMPLEX(53, -55), MAKE_VAL_COMPLEX(-312, 168), MAKE_VAL_COMPLEX(-649, 485), MAKE_VAL_COMPLEX(-279, -473), MAKE_VAL_COMPLEX(-200, -970) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_319, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_319", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_319() */
/* END TEST 319 */

/*
 *  BEGIN TEST 320
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_320( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 51), MAKE_VAL_COMPLEX(19, -6), MAKE_VAL_COMPLEX(-87, 22), MAKE_VAL_COMPLEX(-11, 28), MAKE_VAL_COMPLEX(-183, 24), MAKE_VAL_COMPLEX(-49, -48), MAKE_VAL_COMPLEX(19, 87), MAKE_VAL_COMPLEX(-38, 41), MAKE_VAL_COMPLEX(123, 35), MAKE_VAL_COMPLEX(-90, -71), MAKE_VAL_COMPLEX(5, -28), MAKE_VAL_COMPLEX(-72, -52), MAKE_VAL_COMPLEX(86, -74), MAKE_VAL_COMPLEX(-4, 1), MAKE_VAL_COMPLEX(15, 205) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_320, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_320", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_320() */
/* END TEST 320 */

/*
 *  BEGIN TEST 321
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_321( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(56, -63), MAKE_VAL_COMPLEX(-21, -58), MAKE_VAL_COMPLEX(21, -2), MAKE_VAL_COMPLEX(-10, -44), MAKE_VAL_COMPLEX(189, 45), MAKE_VAL_COMPLEX(23, 118), MAKE_VAL_COMPLEX(11, -111), MAKE_VAL_COMPLEX(-12, -1), MAKE_VAL_COMPLEX(-106, -106), MAKE_VAL_COMPLEX(165, 93), MAKE_VAL_COMPLEX(-43, 42), MAKE_VAL_COMPLEX(32, 84), MAKE_VAL_COMPLEX(-114, 121), MAKE_VAL_COMPLEX(-76, -19), MAKE_VAL_COMPLEX(75, -195) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_321, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_321", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_321() */
/* END TEST 321 */

/*
 *  BEGIN TEST 322
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_322( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-96, 12), MAKE_VAL_COMPLEX(-6, 108), MAKE_VAL_COMPLEX(114, -24), MAKE_VAL_COMPLEX(33, 30), MAKE_VAL_COMPLEX(0, -117), MAKE_VAL_COMPLEX(54, -114), MAKE_VAL_COMPLEX(-54, 36), MAKE_VAL_COMPLEX(90, -60), MAKE_VAL_COMPLEX(-39, 117), MAKE_VAL_COMPLEX(-123, -48), MAKE_VAL_COMPLEX(66, -18), MAKE_VAL_COMPLEX(72, -48), MAKE_VAL_COMPLEX(54, -75), MAKE_VAL_COMPLEX(132, 42), MAKE_VAL_COMPLEX(-150, -30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_322, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_322", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_322() */
/* END TEST 322 */

/*
 *  BEGIN TEST 323
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_323( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_ROWMAJ, y288_VW );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-6, -31), MAKE_VAL_COMPLEX(99, -206), MAKE_VAL_COMPLEX(-407, -538), MAKE_VAL_COMPLEX(-240, -205), MAKE_VAL_COMPLEX(-30, -765), MAKE_VAL_COMPLEX(105, 70), MAKE_VAL_COMPLEX(-355, -51), MAKE_VAL_COMPLEX(-518, -139), MAKE_VAL_COMPLEX(-145, 369), MAKE_VAL_COMPLEX(863, -250), MAKE_VAL_COMPLEX(-49, 74), MAKE_VAL_COMPLEX(136, -276), MAKE_VAL_COMPLEX(149, 787), MAKE_VAL_COMPLEX(-380, -231), MAKE_VAL_COMPLEX(-805, 65) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 6);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_323, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_323", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_323() */
/* END TEST 323 */

/*
 *  BEGIN TEST 324
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_324( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(87, 4), MAKE_VAL_COMPLEX(-18, 15), MAKE_VAL_COMPLEX(24, 101), MAKE_VAL_COMPLEX(20, -232), MAKE_VAL_COMPLEX(81, -11), MAKE_VAL_COMPLEX(-142, 166), MAKE_VAL_COMPLEX(-6, -14), MAKE_VAL_COMPLEX(-158, -73), MAKE_VAL_COMPLEX(164, -109), MAKE_VAL_COMPLEX(-105, 132), MAKE_VAL_COMPLEX(-44, -105), MAKE_VAL_COMPLEX(-127, 120) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_324, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_324", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_324() */
/* END TEST 324 */

/*
 *  BEGIN TEST 325
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_325( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-83, 24), MAKE_VAL_COMPLEX(62, -51), MAKE_VAL_COMPLEX(-20, -49), MAKE_VAL_COMPLEX(8, 180), MAKE_VAL_COMPLEX(-89, -37), MAKE_VAL_COMPLEX(94, -134), MAKE_VAL_COMPLEX(-2, 6), MAKE_VAL_COMPLEX(162, 45), MAKE_VAL_COMPLEX(-96, 97), MAKE_VAL_COMPLEX(49, -116), MAKE_VAL_COMPLEX(20, 49), MAKE_VAL_COMPLEX(147, -60) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_325, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_325", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_325() */
/* END TEST 325 */

/*
 *  BEGIN TEST 326
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_326( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(63, -9), MAKE_VAL_COMPLEX(-81, -99), MAKE_VAL_COMPLEX(117, -9), MAKE_VAL_COMPLEX(-117, -63), MAKE_VAL_COMPLEX(-108, 18), MAKE_VAL_COMPLEX(72, 108), MAKE_VAL_COMPLEX(-18, 18), MAKE_VAL_COMPLEX(-63, -9), MAKE_VAL_COMPLEX(-27, -153), MAKE_VAL_COMPLEX(36, 126), MAKE_VAL_COMPLEX(-126, 54), MAKE_VAL_COMPLEX(135, -45) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, -9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_326, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_326", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_326() */
/* END TEST 326 */

/*
 *  BEGIN TEST 327
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_327( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	oski_value_t* y_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-764, -128), MAKE_VAL_COMPLEX(1128, -128), MAKE_VAL_COMPLEX(58, -96), MAKE_VAL_COMPLEX(-708, 1017), MAKE_VAL_COMPLEX(-342, -1142), MAKE_VAL_COMPLEX(1056, -268), MAKE_VAL_COMPLEX(-86, -44), MAKE_VAL_COMPLEX(958, 1104), MAKE_VAL_COMPLEX(-758, 182), MAKE_VAL_COMPLEX(616, -541), MAKE_VAL_COMPLEX(-186, 21), MAKE_VAL_COMPLEX(1384, 922) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_327, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_327", y_true, y_val, y288_LEN*y288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_327() */
/* END TEST 327 */

/*
 *  BEGIN TEST 328
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_328( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-40, -49), MAKE_VAL_COMPLEX(-83, 2), MAKE_VAL_COMPLEX(160, 124), MAKE_VAL_COMPLEX(16, -11), MAKE_VAL_COMPLEX(-48, -108), MAKE_VAL_COMPLEX(81, -17), MAKE_VAL_COMPLEX(-12, 36), MAKE_VAL_COMPLEX(-22, -40), MAKE_VAL_COMPLEX(-127, 74), MAKE_VAL_COMPLEX(-82, -100), MAKE_VAL_COMPLEX(43, 22), MAKE_VAL_COMPLEX(-11, 79), MAKE_VAL_COMPLEX(-10, -96), MAKE_VAL_COMPLEX(-44, 43), MAKE_VAL_COMPLEX(95, -145) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_328, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_328", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_328() */
/* END TEST 328 */

/*
 *  BEGIN TEST 329
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_329( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-20, -51), MAKE_VAL_COMPLEX(-39, 44), MAKE_VAL_COMPLEX(-78, -10), MAKE_VAL_COMPLEX(-33, 62), MAKE_VAL_COMPLEX(177, 51), MAKE_VAL_COMPLEX(71, 21), MAKE_VAL_COMPLEX(-54, -78), MAKE_VAL_COMPLEX(132, 110), MAKE_VAL_COMPLEX(-21, -60), MAKE_VAL_COMPLEX(75, -59), MAKE_VAL_COMPLEX(9, 42), MAKE_VAL_COMPLEX(99, -23), MAKE_VAL_COMPLEX(119, 119), MAKE_VAL_COMPLEX(62, 123), MAKE_VAL_COMPLEX(-135, -35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_329, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_329", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_329() */
/* END TEST 329 */

/*
 *  BEGIN TEST 330
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_330( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(100, -40), MAKE_VAL_COMPLEX(-24, -118), MAKE_VAL_COMPLEX(-116, 58), MAKE_VAL_COMPLEX(-44, -23), MAKE_VAL_COMPLEX(33, 126), MAKE_VAL_COMPLEX(-26, 138), MAKE_VAL_COMPLEX(48, -54), MAKE_VAL_COMPLEX(-80, 90), MAKE_VAL_COMPLEX(9, -137), MAKE_VAL_COMPLEX(146, 17), MAKE_VAL_COMPLEX(-66, 38), MAKE_VAL_COMPLEX(-64, 72), MAKE_VAL_COMPLEX(-37, 96), MAKE_VAL_COMPLEX(-154, -8), MAKE_VAL_COMPLEX(170, -10) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_330, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_330", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_330() */
/* END TEST 330 */

/*
 *  BEGIN TEST 331
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_331( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(92, -47), MAKE_VAL_COMPLEX(-141, 6), MAKE_VAL_COMPLEX(64, -432), MAKE_VAL_COMPLEX(-224, -63), MAKE_VAL_COMPLEX(-114, 636), MAKE_VAL_COMPLEX(-41, 145), MAKE_VAL_COMPLEX(228, -216), MAKE_VAL_COMPLEX(-258, 508), MAKE_VAL_COMPLEX(265, 8), MAKE_VAL_COMPLEX(146, 282), MAKE_VAL_COMPLEX(-103, 10), MAKE_VAL_COMPLEX(231, 269), MAKE_VAL_COMPLEX(-356, 476), MAKE_VAL_COMPLEX(-260, 317), MAKE_VAL_COMPLEX(-175, -495) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, -4);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_331, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_331", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_331() */
/* END TEST 331 */

/*
 *  BEGIN TEST 332
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_332( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(32, 87), MAKE_VAL_COMPLEX(69, -34), MAKE_VAL_COMPLEX(-140, -52), MAKE_VAL_COMPLEX(-6, -13), MAKE_VAL_COMPLEX(-42, -96), MAKE_VAL_COMPLEX(-119, -33), MAKE_VAL_COMPLEX(92, 68), MAKE_VAL_COMPLEX(38, -100), MAKE_VAL_COMPLEX(149, -46), MAKE_VAL_COMPLEX(-12, 56), MAKE_VAL_COMPLEX(-21, -66), MAKE_VAL_COMPLEX(-83, 27), MAKE_VAL_COMPLEX(28, -34), MAKE_VAL_COMPLEX(104, -65), MAKE_VAL_COMPLEX(45, 275) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_332, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_332", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_332() */
/* END TEST 332 */

/*
 *  BEGIN TEST 333
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_333( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-64, -23), MAKE_VAL_COMPLEX(-5, 82), MAKE_VAL_COMPLEX(172, -28), MAKE_VAL_COMPLEX(38, 5), MAKE_VAL_COMPLEX(-30, 48), MAKE_VAL_COMPLEX(71, -47), MAKE_VAL_COMPLEX(-92, -20), MAKE_VAL_COMPLEX(-38, 20), MAKE_VAL_COMPLEX(-93, 118), MAKE_VAL_COMPLEX(-68, 0), MAKE_VAL_COMPLEX(37, 18), MAKE_VAL_COMPLEX(83, -91), MAKE_VAL_COMPLEX(-52, -30), MAKE_VAL_COMPLEX(-24, 1), MAKE_VAL_COMPLEX(-125, -195) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_333, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_333", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_333() */
/* END TEST 333 */

/*
 *  BEGIN TEST 334
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_334( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(68, -56), MAKE_VAL_COMPLEX(-48, -86), MAKE_VAL_COMPLEX(-76, 74), MAKE_VAL_COMPLEX(-40, -7), MAKE_VAL_COMPLEX(57, 90), MAKE_VAL_COMPLEX(14, 114), MAKE_VAL_COMPLEX(24, -54), MAKE_VAL_COMPLEX(-40, 90), MAKE_VAL_COMPLEX(-27, -109), MAKE_VAL_COMPLEX(118, -23), MAKE_VAL_COMPLEX(-42, 46), MAKE_VAL_COMPLEX(-32, 72), MAKE_VAL_COMPLEX(-5, 84), MAKE_VAL_COMPLEX(-122, 32), MAKE_VAL_COMPLEX(130, -50) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_334, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_334", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_334() */
/* END TEST 334 */

/*
 *  BEGIN TEST 335
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x288', LAYOUT_COLMAJ>
 *  y : <'y288', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_335( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A288_ptr, A288_ind, A288_val,
		A288_M, A288_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y288_val,
		y288_LEN * y288_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y288_LEN, y288_VW, LAYOUT_COLMAJ, y288_LEN );
	oski_value_t* y_val = array_Duplicate( x288_val,
		x288_LEN * x288_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x288_LEN, x288_VW, LAYOUT_COLMAJ, x288_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(8, -33), MAKE_VAL_COMPLEX(197, 98), MAKE_VAL_COMPLEX(644, -304), MAKE_VAL_COMPLEX(174, -163), MAKE_VAL_COMPLEX(-432, 864), MAKE_VAL_COMPLEX(37, 83), MAKE_VAL_COMPLEX(-332, -200), MAKE_VAL_COMPLEX(-786, 240), MAKE_VAL_COMPLEX(-253, 372), MAKE_VAL_COMPLEX(-12, 450), MAKE_VAL_COMPLEX(-73, 82), MAKE_VAL_COMPLEX(165, -557), MAKE_VAL_COMPLEX(-750, 26), MAKE_VAL_COMPLEX(-684, -433), MAKE_VAL_COMPLEX(-255, -945) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_335, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_335", y_true, y_val, x288_LEN*x288_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_335() */
/* END TEST 335 */

#define A336_M 4    /* # of rows */
#define A336_N 5    /* # of cols */
#define A336_NNZ 4  /* # of stored non-zeros */
static oski_index_t A336_ptr[A336_N+1] = {0, 0, 1, 1, 2, 4}; /* end row pointers for A336 */
static oski_index_t A336_ind[A336_NNZ] = {0, 1, 2, 3}; /* end column indices pointers for A336 */
static oski_value_t A336_val[A336_NNZ] = {MAKE_VAL_COMPLEX(7, -10), MAKE_VAL_COMPLEX(10, 5), MAKE_VAL_COMPLEX(8, 0), MAKE_VAL_COMPLEX(5, 10)}; /* end stored non-zero values for A336 */

#define x336_LEN 5   /* vector length */
#define x336_VW  3   /* vector width */
static oski_value_t x336_val[x336_LEN*x336_VW] = {MAKE_VAL_COMPLEX(-3, 10), MAKE_VAL_COMPLEX(1, -7), MAKE_VAL_COMPLEX(-10, 5), MAKE_VAL_COMPLEX(5, 10), MAKE_VAL_COMPLEX(1, -5), MAKE_VAL_COMPLEX(9, 1), MAKE_VAL_COMPLEX(10, 7), MAKE_VAL_COMPLEX(10, -7), MAKE_VAL_COMPLEX(0, -7), MAKE_VAL_COMPLEX(-3, 5), MAKE_VAL_COMPLEX(-9, 9), MAKE_VAL_COMPLEX(10, -10), MAKE_VAL_COMPLEX(-10, 9), MAKE_VAL_COMPLEX(1, 8), MAKE_VAL_COMPLEX(-5, -9)}; /* data */

#define y336_LEN 4   /* vector length */
#define y336_VW  3   /* vector width */
static oski_value_t y336_val[y336_LEN*y336_VW] = {MAKE_VAL_COMPLEX(-3, 10), MAKE_VAL_COMPLEX(4, 9), MAKE_VAL_COMPLEX(-10, -3), MAKE_VAL_COMPLEX(-3, 1), MAKE_VAL_COMPLEX(10, -2), MAKE_VAL_COMPLEX(6, 1), MAKE_VAL_COMPLEX(-1, 3), MAKE_VAL_COMPLEX(-9, -10), MAKE_VAL_COMPLEX(-10, 2), MAKE_VAL_COMPLEX(-2, 3), MAKE_VAL_COMPLEX(4, -6), MAKE_VAL_COMPLEX(10, 4)}; /* data */

/*
 *  BEGIN TEST 336
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_336( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(60, 52), MAKE_VAL_COMPLEX(-80, 8), MAKE_VAL_COMPLEX(41, -150), MAKE_VAL_COMPLEX(-68, 31), MAKE_VAL_COMPLEX(-82, 92), MAKE_VAL_COMPLEX(177, -9), MAKE_VAL_COMPLEX(-92, 85), MAKE_VAL_COMPLEX(42, -37), MAKE_VAL_COMPLEX(-92, -131), MAKE_VAL_COMPLEX(-169, -50), MAKE_VAL_COMPLEX(-32, 59), MAKE_VAL_COMPLEX(91, -29) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_336, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_336", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_336() */
/* END TEST 336 */

/*
 *  BEGIN TEST 337
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_337( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-163, -72), MAKE_VAL_COMPLEX(-6, 41), MAKE_VAL_COMPLEX(-21, 47), MAKE_VAL_COMPLEX(55, -60), MAKE_VAL_COMPLEX(112, 6), MAKE_VAL_COMPLEX(-181, 70), MAKE_VAL_COMPLEX(61, -92), MAKE_VAL_COMPLEX(49, -63), MAKE_VAL_COMPLEX(62, 33), MAKE_VAL_COMPLEX(137, 33), MAKE_VAL_COMPLEX(96, -25), MAKE_VAL_COMPLEX(-121, 133) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_337, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_337", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_337() */
/* END TEST 337 */

/*
 *  BEGIN TEST 338
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_338( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(52, -28), MAKE_VAL_COMPLEX(20, -52), MAKE_VAL_COMPLEX(28, 52), MAKE_VAL_COMPLEX(16, 8), MAKE_VAL_COMPLEX(-48, -32), MAKE_VAL_COMPLEX(-20, -28), MAKE_VAL_COMPLEX(16, -8), MAKE_VAL_COMPLEX(-4, 76), MAKE_VAL_COMPLEX(48, 32), MAKE_VAL_COMPLEX(20, -4), MAKE_VAL_COMPLEX(-40, 8), MAKE_VAL_COMPLEX(-24, -56) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_338, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_338", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_338() */
/* END TEST 338 */

/*
 *  BEGIN TEST 339
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_339( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(570, -259), MAKE_VAL_COMPLEX(-193, -48), MAKE_VAL_COMPLEX(-92, -339), MAKE_VAL_COMPLEX(-12, 314), MAKE_VAL_COMPLEX(-280, 428), MAKE_VAL_COMPLEX(351, -676), MAKE_VAL_COMPLEX(52, 462), MAKE_VAL_COMPLEX(117, 178), MAKE_VAL_COMPLEX(-359, -23), MAKE_VAL_COMPLEX(-556, 303), MAKE_VAL_COMPLEX(-121, 381), MAKE_VAL_COMPLEX(-34, -622) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, -3);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_339, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_339", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_339() */
/* END TEST 339 */

/*
 *  BEGIN TEST 340
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_340( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(18, -60), MAKE_VAL_COMPLEX(-3, 58), MAKE_VAL_COMPLEX(60, -38), MAKE_VAL_COMPLEX(41, 31), MAKE_VAL_COMPLEX(121, 56), MAKE_VAL_COMPLEX(-157, 73), MAKE_VAL_COMPLEX(-71, -46), MAKE_VAL_COMPLEX(-79, 39), MAKE_VAL_COMPLEX(-10, 51), MAKE_VAL_COMPLEX(-16, -37), MAKE_VAL_COMPLEX(177, -39), MAKE_VAL_COMPLEX(-5, 114), MAKE_VAL_COMPLEX(22, -44), MAKE_VAL_COMPLEX(1, -126), MAKE_VAL_COMPLEX(-35, 199) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 0);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_340, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_340", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_340() */
/* END TEST 340 */

/*
 *  BEGIN TEST 341
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_341( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(24, 29), MAKE_VAL_COMPLEX(-22, -33), MAKE_VAL_COMPLEX(-5, 48), MAKE_VAL_COMPLEX(-31, -86), MAKE_VAL_COMPLEX(-140, -39), MAKE_VAL_COMPLEX(124, -104), MAKE_VAL_COMPLEX(52, -12), MAKE_VAL_COMPLEX(18, -41), MAKE_VAL_COMPLEX(-11, -23), MAKE_VAL_COMPLEX(43, 26), MAKE_VAL_COMPLEX(-114, 30), MAKE_VAL_COMPLEX(-65, -104), MAKE_VAL_COMPLEX(45, 38), MAKE_VAL_COMPLEX(19, 91), MAKE_VAL_COMPLEX(28, -148) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_341, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_341", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_341() */
/* END TEST 341 */

/*
 *  BEGIN TEST 342
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_342( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(32, -34), MAKE_VAL_COMPLEX(-18, 26), MAKE_VAL_COMPLEX(50, 0), MAKE_VAL_COMPLEX(0, -50), MAKE_VAL_COMPLEX(-14, 18), MAKE_VAL_COMPLEX(-34, -22), MAKE_VAL_COMPLEX(-26, -48), MAKE_VAL_COMPLEX(-54, 8), MAKE_VAL_COMPLEX(-14, 28), MAKE_VAL_COMPLEX(22, -14), MAKE_VAL_COMPLEX(54, -18), MAKE_VAL_COMPLEX(-60, 20), MAKE_VAL_COMPLEX(58, -16), MAKE_VAL_COMPLEX(12, -34), MAKE_VAL_COMPLEX(2, 46) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_342, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_342", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_342() */
/* END TEST 342 */

/*
 *  BEGIN TEST 343
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_343( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-67, -31), MAKE_VAL_COMPLEX(14, -79), MAKE_VAL_COMPLEX(-17, -27), MAKE_VAL_COMPLEX(-592, -450), MAKE_VAL_COMPLEX(-754, -17), MAKE_VAL_COMPLEX(523, -515), MAKE_VAL_COMPLEX(11, 76), MAKE_VAL_COMPLEX(156, 76), MAKE_VAL_COMPLEX(100, -57), MAKE_VAL_COMPLEX(179, -18), MAKE_VAL_COMPLEX(-807, -39), MAKE_VAL_COMPLEX(-324, -189), MAKE_VAL_COMPLEX(165, -177), MAKE_VAL_COMPLEX(-21, 474), MAKE_VAL_COMPLEX(313, -961) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 1);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_343, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_343", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_343() */
/* END TEST 343 */

/*
 *  BEGIN TEST 344
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_344( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(11, 36), MAKE_VAL_COMPLEX(-8, -7), MAKE_VAL_COMPLEX(-20, 27), MAKE_VAL_COMPLEX(-94, 51), MAKE_VAL_COMPLEX(-60, 89), MAKE_VAL_COMPLEX(-14, -136), MAKE_VAL_COMPLEX(33, -3), MAKE_VAL_COMPLEX(-3, -44), MAKE_VAL_COMPLEX(-24, -12), MAKE_VAL_COMPLEX(-23, 44), MAKE_VAL_COMPLEX(94, -40), MAKE_VAL_COMPLEX(75, -56), MAKE_VAL_COMPLEX(10, 97), MAKE_VAL_COMPLEX(-94, -136), MAKE_VAL_COMPLEX(-18, -72) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_344, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_344", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_344() */
/* END TEST 344 */

/*
 *  BEGIN TEST 345
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_345( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(31, 42), MAKE_VAL_COMPLEX(-28, -41), MAKE_VAL_COMPLEX(-10, 63), MAKE_VAL_COMPLEX(184, -21), MAKE_VAL_COMPLEX(36, -125), MAKE_VAL_COMPLEX(74, 88), MAKE_VAL_COMPLEX(69, -15), MAKE_VAL_COMPLEX(21, -58), MAKE_VAL_COMPLEX(-18, -30), MAKE_VAL_COMPLEX(35, 4), MAKE_VAL_COMPLEX(-94, 148), MAKE_VAL_COMPLEX(-75, -64), MAKE_VAL_COMPLEX(-16, 17), MAKE_VAL_COMPLEX(148, 178), MAKE_VAL_COMPLEX(-66, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_345, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_345", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_345() */
/* END TEST 345 */

/*
 *  BEGIN TEST 346
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_346( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-27, -19), MAKE_VAL_COMPLEX(20, 10), MAKE_VAL_COMPLEX(-5, -35), MAKE_VAL_COMPLEX(-35, 5), MAKE_VAL_COMPLEX(14, 8), MAKE_VAL_COMPLEX(-12, 26), MAKE_VAL_COMPLEX(-31, 23), MAKE_VAL_COMPLEX(11, 37), MAKE_VAL_COMPLEX(21, 7), MAKE_VAL_COMPLEX(-12, -14), MAKE_VAL_COMPLEX(-18, -36), MAKE_VAL_COMPLEX(20, 40), MAKE_VAL_COMPLEX(-17, -39), MAKE_VAL_COMPLEX(-25, -5), MAKE_VAL_COMPLEX(32, -6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_346, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_346", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_346() */
/* END TEST 346 */

/*
 *  BEGIN TEST 347
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_347( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(62, 84), MAKE_VAL_COMPLEX(78, -74), MAKE_VAL_COMPLEX(-84, 110), MAKE_VAL_COMPLEX(358, 1052), MAKE_VAL_COMPLEX(814, 386), MAKE_VAL_COMPLEX(-906, 278), MAKE_VAL_COMPLEX(84, 50), MAKE_VAL_COMPLEX(-20, 30), MAKE_VAL_COMPLEX(16, 38), MAKE_VAL_COMPLEX(206, 246), MAKE_VAL_COMPLEX(-662, -698), MAKE_VAL_COMPLEX(-68, -660), MAKE_VAL_COMPLEX(412, -42), MAKE_VAL_COMPLEX(-1194, 944), MAKE_VAL_COMPLEX(-542, -134) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, -8);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_347, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_347", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_347() */
/* END TEST 347 */

/*
 *  BEGIN TEST 348
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_348( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(81, -18), MAKE_VAL_COMPLEX(-116, 42), MAKE_VAL_COMPLEX(-22, 28), MAKE_VAL_COMPLEX(-140, -71), MAKE_VAL_COMPLEX(-60, 14), MAKE_VAL_COMPLEX(-158, 73), MAKE_VAL_COMPLEX(3, 42), MAKE_VAL_COMPLEX(3, 35), MAKE_VAL_COMPLEX(81, -144), MAKE_VAL_COMPLEX(147, -70), MAKE_VAL_COMPLEX(-16, -37), MAKE_VAL_COMPLEX(21, -57) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_348, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_348", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_348() */
/* END TEST 348 */

/*
 *  BEGIN TEST 349
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_349( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-105, -11), MAKE_VAL_COMPLEX(81, -48), MAKE_VAL_COMPLEX(51, -52), MAKE_VAL_COMPLEX(143, 60), MAKE_VAL_COMPLEX(46, 20), MAKE_VAL_COMPLEX(143, -57), MAKE_VAL_COMPLEX(-10, -51), MAKE_VAL_COMPLEX(45, -42), MAKE_VAL_COMPLEX(-67, 110), MAKE_VAL_COMPLEX(-152, 58), MAKE_VAL_COMPLEX(26, 61), MAKE_VAL_COMPLEX(-53, 79) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_349, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_349", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_349() */
/* END TEST 349 */

/*
 *  BEGIN TEST 350
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_350( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(35, -44), MAKE_VAL_COMPLEX(-2, -53), MAKE_VAL_COMPLEX(44, 35), MAKE_VAL_COMPLEX(17, 1), MAKE_VAL_COMPLEX(-54, -10), MAKE_VAL_COMPLEX(-28, -17), MAKE_VAL_COMPLEX(11, -13), MAKE_VAL_COMPLEX(25, 68), MAKE_VAL_COMPLEX(54, 10), MAKE_VAL_COMPLEX(16, -11), MAKE_VAL_COMPLEX(-32, 22), MAKE_VAL_COMPLEX(-42, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_350, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_350", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_350() */
/* END TEST 350 */

/*
 *  BEGIN TEST 351
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_351( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1052, 533), MAKE_VAL_COMPLEX(-595, 351), MAKE_VAL_COMPLEX(-787, 495), MAKE_VAL_COMPLEX(-1200, -755), MAKE_VAL_COMPLEX(-240, -486), MAKE_VAL_COMPLEX(-1281, 136), MAKE_VAL_COMPLEX(25, 548), MAKE_VAL_COMPLEX(-822, 280), MAKE_VAL_COMPLEX(689, -642), MAKE_VAL_COMPLEX(1504, -131), MAKE_VAL_COMPLEX(-152, -775), MAKE_VAL_COMPLEX(877, -717) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(9, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_351, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_351", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_351() */
/* END TEST 351 */

/*
 *  BEGIN TEST 352
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_352( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(98, -36), MAKE_VAL_COMPLEX(-53, 39), MAKE_VAL_COMPLEX(100, 47), MAKE_VAL_COMPLEX(128, -1), MAKE_VAL_COMPLEX(9, -86), MAKE_VAL_COMPLEX(-107, 38), MAKE_VAL_COMPLEX(-24, -132), MAKE_VAL_COMPLEX(-127, -28), MAKE_VAL_COMPLEX(-52, 43), MAKE_VAL_COMPLEX(53, 100), MAKE_VAL_COMPLEX(181, 39), MAKE_VAL_COMPLEX(-175, 14), MAKE_VAL_COMPLEX(37, -32), MAKE_VAL_COMPLEX(104, -180), MAKE_VAL_COMPLEX(5, 175) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_352, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_352", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_352() */
/* END TEST 352 */

/*
 *  BEGIN TEST 353
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_353( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(51, 48), MAKE_VAL_COMPLEX(-48, -32), MAKE_VAL_COMPLEX(0, 78), MAKE_VAL_COMPLEX(-3, -99), MAKE_VAL_COMPLEX(-82, 87), MAKE_VAL_COMPLEX(94, -167), MAKE_VAL_COMPLEX(92, -29), MAKE_VAL_COMPLEX(-1, -91), MAKE_VAL_COMPLEX(-46, -22), MAKE_VAL_COMPLEX(26, -73), MAKE_VAL_COMPLEX(-28, 60), MAKE_VAL_COMPLEX(5, -124), MAKE_VAL_COMPLEX(119, 145), MAKE_VAL_COMPLEX(5, 142), MAKE_VAL_COMPLEX(-116, -78) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_353, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_353", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_353() */
/* END TEST 353 */

/*
 *  BEGIN TEST 354
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_354( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-44, -35), MAKE_VAL_COMPLEX(33, 19), MAKE_VAL_COMPLEX(-5, -60), MAKE_VAL_COMPLEX(-60, 5), MAKE_VAL_COMPLEX(23, 15), MAKE_VAL_COMPLEX(-23, 43), MAKE_VAL_COMPLEX(-55, 36), MAKE_VAL_COMPLEX(15, 64), MAKE_VAL_COMPLEX(35, 14), MAKE_VAL_COMPLEX(-19, -25), MAKE_VAL_COMPLEX(-27, -63), MAKE_VAL_COMPLEX(30, 70), MAKE_VAL_COMPLEX(-25, -68), MAKE_VAL_COMPLEX(-42, -11), MAKE_VAL_COMPLEX(55, -7) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_354, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_354", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_354() */
/* END TEST 354 */

/*
 *  BEGIN TEST 355
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_355( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(34, 32), MAKE_VAL_COMPLEX(49, -47), MAKE_VAL_COMPLEX(-18, 35), MAKE_VAL_COMPLEX(1034, 27), MAKE_VAL_COMPLEX(-391, -884), MAKE_VAL_COMPLEX(413, 912), MAKE_VAL_COMPLEX(-112, 44), MAKE_VAL_COMPLEX(-3, 62), MAKE_VAL_COMPLEX(-2, -33), MAKE_VAL_COMPLEX(625, 582), MAKE_VAL_COMPLEX(419, -171), MAKE_VAL_COMPLEX(9, 320), MAKE_VAL_COMPLEX(-807, 338), MAKE_VAL_COMPLEX(-480, -884), MAKE_VAL_COMPLEX(675, 125) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(5, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_355, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_355", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_355() */
/* END TEST 355 */

/*
 *  BEGIN TEST 356
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_356( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(46, 101), MAKE_VAL_COMPLEX(-32, -58), MAKE_VAL_COMPLEX(-45, 107), MAKE_VAL_COMPLEX(-12, 84), MAKE_VAL_COMPLEX(68, 45), MAKE_VAL_COMPLEX(-22, -139), MAKE_VAL_COMPLEX(109, -24), MAKE_VAL_COMPLEX(20, -116), MAKE_VAL_COMPLEX(-45, -55), MAKE_VAL_COMPLEX(96, 127), MAKE_VAL_COMPLEX(56, 96), MAKE_VAL_COMPLEX(5, -96), MAKE_VAL_COMPLEX(-92, 144), MAKE_VAL_COMPLEX(-90, 113), MAKE_VAL_COMPLEX(24, -156) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_356, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_356", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_356() */
/* END TEST 356 */

/*
 *  BEGIN TEST 357
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_357( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(80, 24), MAKE_VAL_COMPLEX(-65, -13), MAKE_VAL_COMPLEX(40, 83), MAKE_VAL_COMPLEX(202, -79), MAKE_VAL_COMPLEX(-135, -100), MAKE_VAL_COMPLEX(109, 12), MAKE_VAL_COMPLEX(76, -70), MAKE_VAL_COMPLEX(-45, -90), MAKE_VAL_COMPLEX(-60, -1), MAKE_VAL_COMPLEX(-45, -42), MAKE_VAL_COMPLEX(7, 111), MAKE_VAL_COMPLEX(-75, -134), MAKE_VAL_COMPLEX(147, 78), MAKE_VAL_COMPLEX(218, -64), MAKE_VAL_COMPLEX(-199, 159) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_357, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_357", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_357() */
/* END TEST 357 */

/*
 *  BEGIN TEST 358
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_358( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-70, -21), MAKE_VAL_COMPLEX(49, 7), MAKE_VAL_COMPLEX(-35, -70), MAKE_VAL_COMPLEX(-70, 35), MAKE_VAL_COMPLEX(35, 7), MAKE_VAL_COMPLEX(-7, 63), MAKE_VAL_COMPLEX(-49, 70), MAKE_VAL_COMPLEX(49, 70), MAKE_VAL_COMPLEX(49, 0), MAKE_VAL_COMPLEX(-35, -21), MAKE_VAL_COMPLEX(-63, -63), MAKE_VAL_COMPLEX(70, 70), MAKE_VAL_COMPLEX(-63, -70), MAKE_VAL_COMPLEX(-56, 7), MAKE_VAL_COMPLEX(63, -35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_358, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_358", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_358() */
/* END TEST 358 */

/*
 *  BEGIN TEST 359
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_ROWMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_359( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_ROWMAJ, x336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(18, 49), MAKE_VAL_COMPLEX(-1, 25), MAKE_VAL_COMPLEX(-13, 38), MAKE_VAL_COMPLEX(-594, -307), MAKE_VAL_COMPLEX(1, 725), MAKE_VAL_COMPLEX(-29, -763), MAKE_VAL_COMPLEX(21, -122), MAKE_VAL_COMPLEX(-65, -62), MAKE_VAL_COMPLEX(-9, -8), MAKE_VAL_COMPLEX(79, 633), MAKE_VAL_COMPLEX(407, 167), MAKE_VAL_COMPLEX(-226, 126), MAKE_VAL_COMPLEX(-321, -226), MAKE_VAL_COMPLEX(-812, -363), MAKE_VAL_COMPLEX(937, 11) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(4, 4);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_359, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_359", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_359() */
/* END TEST 359 */

/*
 *  BEGIN TEST 360
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_360( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-134, -4), MAKE_VAL_COMPLEX(128, 24), MAKE_VAL_COMPLEX(-84, -229), MAKE_VAL_COMPLEX(-22, 109), MAKE_VAL_COMPLEX(115, -25), MAKE_VAL_COMPLEX(11, 111), MAKE_VAL_COMPLEX(-23, -22), MAKE_VAL_COMPLEX(-18, -72), MAKE_VAL_COMPLEX(-120, -101), MAKE_VAL_COMPLEX(33, 3), MAKE_VAL_COMPLEX(-11, -28), MAKE_VAL_COMPLEX(106, -13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_360, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_360", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_360() */
/* END TEST 360 */

/*
 *  BEGIN TEST 361
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_361( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-11, 15), MAKE_VAL_COMPLEX(-225, 73), MAKE_VAL_COMPLEX(73, 84), MAKE_VAL_COMPLEX(-6, -143), MAKE_VAL_COMPLEX(-39, 145), MAKE_VAL_COMPLEX(6, -28), MAKE_VAL_COMPLEX(-21, 24), MAKE_VAL_COMPLEX(103, -95), MAKE_VAL_COMPLEX(44, -19), MAKE_VAL_COMPLEX(-82, -14), MAKE_VAL_COMPLEX(109, 50), MAKE_VAL_COMPLEX(-108, 163) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_361, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_361", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_361() */
/* END TEST 361 */

/*
 *  BEGIN TEST 362
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_362( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, 6), MAKE_VAL_COMPLEX(18, -8), MAKE_VAL_COMPLEX(-6, 20), MAKE_VAL_COMPLEX(2, 6), MAKE_VAL_COMPLEX(-4, -20), MAKE_VAL_COMPLEX(2, -12), MAKE_VAL_COMPLEX(6, 2), MAKE_VAL_COMPLEX(-20, 18), MAKE_VAL_COMPLEX(4, 20), MAKE_VAL_COMPLEX(6, 4), MAKE_VAL_COMPLEX(-12, -8), MAKE_VAL_COMPLEX(8, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_362, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_362", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_362() */
/* END TEST 362 */

/*
 *  BEGIN TEST 363
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_363( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(402, 482), MAKE_VAL_COMPLEX(-1056, 260), MAKE_VAL_COMPLEX(39, 1155), MAKE_VAL_COMPLEX(91, -817), MAKE_VAL_COMPLEX(-310, 372), MAKE_VAL_COMPLEX(258, -504), MAKE_VAL_COMPLEX(-25, 269), MAKE_VAL_COMPLEX(58, -278), MAKE_VAL_COMPLEX(219, 515), MAKE_VAL_COMPLEX(-436, -2), MAKE_VAL_COMPLEX(465, 103), MAKE_VAL_COMPLEX(-475, 561) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_363, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_363", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_363() */
/* END TEST 363 */

/*
 *  BEGIN TEST 364
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_364( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(53, 114), MAKE_VAL_COMPLEX(28, 37), MAKE_VAL_COMPLEX(-41, 123), MAKE_VAL_COMPLEX(83, 38), MAKE_VAL_COMPLEX(-80, -29), MAKE_VAL_COMPLEX(84, -55), MAKE_VAL_COMPLEX(264, -3), MAKE_VAL_COMPLEX(15, -146), MAKE_VAL_COMPLEX(58, -32), MAKE_VAL_COMPLEX(24, -6), MAKE_VAL_COMPLEX(-10, 141), MAKE_VAL_COMPLEX(-94, -80), MAKE_VAL_COMPLEX(-18, 154), MAKE_VAL_COMPLEX(137, 100), MAKE_VAL_COMPLEX(-182, 104) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_364, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_364", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_364() */
/* END TEST 364 */

/*
 *  BEGIN TEST 365
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_365( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(25, -47), MAKE_VAL_COMPLEX(-87, -74), MAKE_VAL_COMPLEX(46, -13), MAKE_VAL_COMPLEX(27, -43), MAKE_VAL_COMPLEX(39, 0), MAKE_VAL_COMPLEX(-39, -22), MAKE_VAL_COMPLEX(-161, -59), MAKE_VAL_COMPLEX(-38, 28), MAKE_VAL_COMPLEX(-121, 4), MAKE_VAL_COMPLEX(9, 53), MAKE_VAL_COMPLEX(55, -24), MAKE_VAL_COMPLEX(44, -50), MAKE_VAL_COMPLEX(59, -28), MAKE_VAL_COMPLEX(-61, -77), MAKE_VAL_COMPLEX(81, -95) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_365, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_365", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_365() */
/* END TEST 365 */

/*
 *  BEGIN TEST 366
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_366( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(91, -49), MAKE_VAL_COMPLEX(-56, 42), MAKE_VAL_COMPLEX(105, 35), MAKE_VAL_COMPLEX(35, -105), MAKE_VAL_COMPLEX(-42, 28), MAKE_VAL_COMPLEX(-56, -70), MAKE_VAL_COMPLEX(-21, -119), MAKE_VAL_COMPLEX(-119, -21), MAKE_VAL_COMPLEX(-49, 49), MAKE_VAL_COMPLEX(56, -14), MAKE_VAL_COMPLEX(126, 0), MAKE_VAL_COMPLEX(-140, 0), MAKE_VAL_COMPLEX(133, 7), MAKE_VAL_COMPLEX(49, -63), MAKE_VAL_COMPLEX(-28, 98) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_366, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_366", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_366() */
/* END TEST 366 */

/*
 *  BEGIN TEST 367
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_367( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-82, -90), MAKE_VAL_COMPLEX(-476, 427), MAKE_VAL_COMPLEX(20, -85), MAKE_VAL_COMPLEX(-70, -220), MAKE_VAL_COMPLEX(-76, -205), MAKE_VAL_COMPLEX(-104, 59), MAKE_VAL_COMPLEX(-200, 648), MAKE_VAL_COMPLEX(25, 47), MAKE_VAL_COMPLEX(-85, 612), MAKE_VAL_COMPLEX(343, -5), MAKE_VAL_COMPLEX(24, -149), MAKE_VAL_COMPLEX(-410, -360), MAKE_VAL_COMPLEX(5, -154), MAKE_VAL_COMPLEX(-266, 282), MAKE_VAL_COMPLEX(-605, -321) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_367, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_367", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_367() */
/* END TEST 367 */

/*
 *  BEGIN TEST 368
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_368( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-21, 70), MAKE_VAL_COMPLEX(-118, -1), MAKE_VAL_COMPLEX(-61, 33), MAKE_VAL_COMPLEX(3, 88), MAKE_VAL_COMPLEX(18, 29), MAKE_VAL_COMPLEX(58, 15), MAKE_VAL_COMPLEX(8, 143), MAKE_VAL_COMPLEX(51, -52), MAKE_VAL_COMPLEX(94, -118), MAKE_VAL_COMPLEX(-130, -120), MAKE_VAL_COMPLEX(-64, 51), MAKE_VAL_COMPLEX(26, -180), MAKE_VAL_COMPLEX(-70, 56), MAKE_VAL_COMPLEX(81, 32), MAKE_VAL_COMPLEX(-20, -118) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_368, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_368", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_368() */
/* END TEST 368 */

/*
 *  BEGIN TEST 369
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_369( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, -69), MAKE_VAL_COMPLEX(140, -3), MAKE_VAL_COMPLEX(36, -58), MAKE_VAL_COMPLEX(-28, -63), MAKE_VAL_COMPLEX(-2, -31), MAKE_VAL_COMPLEX(-52, 13), MAKE_VAL_COMPLEX(-19, -106), MAKE_VAL_COMPLEX(-20, 75), MAKE_VAL_COMPLEX(-73, 111), MAKE_VAL_COMPLEX(112, 116), MAKE_VAL_COMPLEX(28, -69), MAKE_VAL_COMPLEX(14, 200), MAKE_VAL_COMPLEX(33, -77), MAKE_VAL_COMPLEX(-104, -21), MAKE_VAL_COMPLEX(42, 94) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_369, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_369", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_369() */
/* END TEST 369 */

/*
 *  BEGIN TEST 370
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_370( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(5, -53), MAKE_VAL_COMPLEX(2, 36), MAKE_VAL_COMPLEX(45, -35), MAKE_VAL_COMPLEX(-35, -45), MAKE_VAL_COMPLEX(0, 26), MAKE_VAL_COMPLEX(-46, 4), MAKE_VAL_COMPLEX(-57, -25), MAKE_VAL_COMPLEX(-43, 45), MAKE_VAL_COMPLEX(7, 35), MAKE_VAL_COMPLEX(10, -28), MAKE_VAL_COMPLEX(36, -54), MAKE_VAL_COMPLEX(-40, 60), MAKE_VAL_COMPLEX(41, -55), MAKE_VAL_COMPLEX(-13, -39), MAKE_VAL_COMPLEX(34, 40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_370, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_370", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_370() */
/* END TEST 370 */

/*
 *  BEGIN TEST 371
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_371( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_ROWMAJ, y336_VW );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(67, 31), MAKE_VAL_COMPLEX(759, 967), MAKE_VAL_COMPLEX(-20, 0), MAKE_VAL_COMPLEX(333, 209), MAKE_VAL_COMPLEX(509, -303), MAKE_VAL_COMPLEX(103, -41), MAKE_VAL_COMPLEX(1091, 213), MAKE_VAL_COMPLEX(-9, 103), MAKE_VAL_COMPLEX(-952, -646), MAKE_VAL_COMPLEX(-1036, 1472), MAKE_VAL_COMPLEX(-51, 117), MAKE_VAL_COMPLEX(-918, 646), MAKE_VAL_COMPLEX(-10, 100), MAKE_VAL_COMPLEX(-381, -593), MAKE_VAL_COMPLEX(-608, 56) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-3, -9);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_371, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_371", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_371() */
/* END TEST 371 */

/*
 *  BEGIN TEST 372
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_372( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-136, -70), MAKE_VAL_COMPLEX(-62, 146), MAKE_VAL_COMPLEX(19, -105), MAKE_VAL_COMPLEX(53, -26), MAKE_VAL_COMPLEX(163, 20), MAKE_VAL_COMPLEX(38, -21), MAKE_VAL_COMPLEX(-35, 26), MAKE_VAL_COMPLEX(5, -75), MAKE_VAL_COMPLEX(-53, -231), MAKE_VAL_COMPLEX(-41, 61), MAKE_VAL_COMPLEX(-8, -35), MAKE_VAL_COMPLEX(38, -17) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_372, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_372", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_372() */
/* END TEST 372 */

/*
 *  BEGIN TEST 373
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_373( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-6, 71), MAKE_VAL_COMPLEX(-39, -58), MAKE_VAL_COMPLEX(-20, -37), MAKE_VAL_COMPLEX(-78, -9), MAKE_VAL_COMPLEX(-97, 102), MAKE_VAL_COMPLEX(-27, 103), MAKE_VAL_COMPLEX(-8, -27), MAKE_VAL_COMPLEX(89, -82), MAKE_VAL_COMPLEX(-13, 109), MAKE_VAL_COMPLEX(-6, -75), MAKE_VAL_COMPLEX(102, 63), MAKE_VAL_COMPLEX(-50, 163) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_373, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_373", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_373() */
/* END TEST 373 */

/*
 *  BEGIN TEST 374
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_374( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-6, 20), MAKE_VAL_COMPLEX(8, 18), MAKE_VAL_COMPLEX(-20, -6), MAKE_VAL_COMPLEX(-6, 2), MAKE_VAL_COMPLEX(20, -4), MAKE_VAL_COMPLEX(12, 2), MAKE_VAL_COMPLEX(-2, 6), MAKE_VAL_COMPLEX(-18, -20), MAKE_VAL_COMPLEX(-20, 4), MAKE_VAL_COMPLEX(-4, 6), MAKE_VAL_COMPLEX(8, -12), MAKE_VAL_COMPLEX(20, 8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_374, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_374", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_374() */
/* END TEST 374 */

/*
 *  BEGIN TEST 375
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_375( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	oski_value_t* y_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(775, 16), MAKE_VAL_COMPLEX(-434, -939), MAKE_VAL_COMPLEX(217, 354), MAKE_VAL_COMPLEX(-364, 293), MAKE_VAL_COMPLEX(-951, 874), MAKE_VAL_COMPLEX(-108, 558), MAKE_VAL_COMPLEX(2, -305), MAKE_VAL_COMPLEX(494, -5), MAKE_VAL_COMPLEX(1025, 1043), MAKE_VAL_COMPLEX(-115, -614), MAKE_VAL_COMPLEX(512, 259), MAKE_VAL_COMPLEX(-168, 747) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, 4);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, -9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_375, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_375", y_true, y_val, y336_LEN*y336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_375() */
/* END TEST 375 */

/*
 *  BEGIN TEST 376
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_376( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(85, -29), MAKE_VAL_COMPLEX(28, 144), MAKE_VAL_COMPLEX(85, 37), MAKE_VAL_COMPLEX(32, 16), MAKE_VAL_COMPLEX(-146, -26), MAKE_VAL_COMPLEX(-37, -71), MAKE_VAL_COMPLEX(45, -225), MAKE_VAL_COMPLEX(-110, -25), MAKE_VAL_COMPLEX(-3, 72), MAKE_VAL_COMPLEX(100, -125), MAKE_VAL_COMPLEX(107, 11), MAKE_VAL_COMPLEX(-182, 107), MAKE_VAL_COMPLEX(127, 10), MAKE_VAL_COMPLEX(25, -31), MAKE_VAL_COMPLEX(9, 161) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_376, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_376", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_376() */
/* END TEST 376 */

/*
 *  BEGIN TEST 377
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_377( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(53, 5), MAKE_VAL_COMPLEX(-118, -114), MAKE_VAL_COMPLEX(35, 53), MAKE_VAL_COMPLEX(58, -136), MAKE_VAL_COMPLEX(80, 44), MAKE_VAL_COMPLEX(-5, -43), MAKE_VAL_COMPLEX(-21, 63), MAKE_VAL_COMPLEX(-34, -53), MAKE_VAL_COMPLEX(-81, -30), MAKE_VAL_COMPLEX(-22, 131), MAKE_VAL_COMPLEX(55, 43), MAKE_VAL_COMPLEX(2, -167), MAKE_VAL_COMPLEX(41, 56), MAKE_VAL_COMPLEX(65, -29), MAKE_VAL_COMPLEX(-87, -47) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_377, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_377", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_377() */
/* END TEST 377 */

/*
 *  BEGIN TEST 378
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_378( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(79, -9), MAKE_VAL_COMPLEX(-52, 14), MAKE_VAL_COMPLEX(65, 55), MAKE_VAL_COMPLEX(55, -65), MAKE_VAL_COMPLEX(-38, 8), MAKE_VAL_COMPLEX(-20, -66), MAKE_VAL_COMPLEX(19, -91), MAKE_VAL_COMPLEX(-79, -49), MAKE_VAL_COMPLEX(-49, 21), MAKE_VAL_COMPLEX(44, 6), MAKE_VAL_COMPLEX(90, 36), MAKE_VAL_COMPLEX(-100, -40), MAKE_VAL_COMPLEX(93, 43), MAKE_VAL_COMPLEX(53, -31), MAKE_VAL_COMPLEX(-48, 62) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_378, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_378", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_378() */
/* END TEST 378 */

/*
 *  BEGIN TEST 379
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_379( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(116, 122), MAKE_VAL_COMPLEX(1494, 202), MAKE_VAL_COMPLEX(-74, 116), MAKE_VAL_COMPLEX(884, 922), MAKE_VAL_COMPLEX(-1262, 442), MAKE_VAL_COMPLEX(70, -150), MAKE_VAL_COMPLEX(-414, -1412), MAKE_VAL_COMPLEX(-26, -28), MAKE_VAL_COMPLEX(566, -128), MAKE_VAL_COMPLEX(-522, -1286), MAKE_VAL_COMPLEX(-10, 150), MAKE_VAL_COMPLEX(460, 1292), MAKE_VAL_COMPLEX(38, -20), MAKE_VAL_COMPLEX(40, 386), MAKE_VAL_COMPLEX(858, 270) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(8, -8);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_379, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_379", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_379() */
/* END TEST 379 */

/*
 *  BEGIN TEST 380
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_380( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(27, 19), MAKE_VAL_COMPLEX(-138, 46), MAKE_VAL_COMPLEX(5, 27), MAKE_VAL_COMPLEX(112, 56), MAKE_VAL_COMPLEX(-100, 8), MAKE_VAL_COMPLEX(13, -29), MAKE_VAL_COMPLEX(117, 57), MAKE_VAL_COMPLEX(-22, -27), MAKE_VAL_COMPLEX(35, -30), MAKE_VAL_COMPLEX(-138, 73), MAKE_VAL_COMPLEX(17, 29), MAKE_VAL_COMPLEX(-122, -113), MAKE_VAL_COMPLEX(31, 24), MAKE_VAL_COMPLEX(29, 41), MAKE_VAL_COMPLEX(95, -113) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_380, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_380", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_380() */
/* END TEST 380 */

/*
 *  BEGIN TEST 381
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_381( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(27, 19), MAKE_VAL_COMPLEX(98, -66), MAKE_VAL_COMPLEX(5, 43), MAKE_VAL_COMPLEX(-42, -66), MAKE_VAL_COMPLEX(72, -24), MAKE_VAL_COMPLEX(11, -23), MAKE_VAL_COMPLEX(-55, -103), MAKE_VAL_COMPLEX(0, -47), MAKE_VAL_COMPLEX(-77, 16), MAKE_VAL_COMPLEX(162, -45), MAKE_VAL_COMPLEX(19, 43), MAKE_VAL_COMPLEX(82, 33), MAKE_VAL_COMPLEX(3, 54), MAKE_VAL_COMPLEX(21, -31), MAKE_VAL_COMPLEX(-159, 125) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_381, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_381", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_381() */
/* END TEST 381 */

/*
 *  BEGIN TEST 382
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_382( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(60, 18), MAKE_VAL_COMPLEX(-42, -6), MAKE_VAL_COMPLEX(30, 60), MAKE_VAL_COMPLEX(60, -30), MAKE_VAL_COMPLEX(-30, -6), MAKE_VAL_COMPLEX(6, -54), MAKE_VAL_COMPLEX(42, -60), MAKE_VAL_COMPLEX(-42, -60), MAKE_VAL_COMPLEX(-42, 0), MAKE_VAL_COMPLEX(30, 18), MAKE_VAL_COMPLEX(54, 54), MAKE_VAL_COMPLEX(-60, -60), MAKE_VAL_COMPLEX(54, 60), MAKE_VAL_COMPLEX(48, -6), MAKE_VAL_COMPLEX(-54, 30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_382, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_382", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_382() */
/* END TEST 382 */

/*
 *  BEGIN TEST 383
 *  Value type: complex
 *  A : <MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x336', LAYOUT_COLMAJ>
 *  y : <'y336', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_383( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A336_ptr, A336_ind, A336_val,
		A336_M, A336_N, SHARE_INPUTMAT,
		5, MAT_TRI_UPPER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y336_val,
		y336_LEN * y336_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y336_LEN, y336_VW, LAYOUT_COLMAJ, y336_LEN );
	oski_value_t* y_val = array_Duplicate( x336_val,
		x336_LEN * x336_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x336_LEN, x336_VW, LAYOUT_COLMAJ, x336_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-44, -144), MAKE_VAL_COMPLEX(12, -1194), MAKE_VAL_COMPLEX(152, -108), MAKE_VAL_COMPLEX(-997, 369), MAKE_VAL_COMPLEX(228, -770), MAKE_VAL_COMPLEX(-93, 81), MAKE_VAL_COMPLEX(-1240, 450), MAKE_VAL_COMPLEX(-110, 25), MAKE_VAL_COMPLEX(39, 680), MAKE_VAL_COMPLEX(65, -1670), MAKE_VAL_COMPLEX(103, -161), MAKE_VAL_COMPLEX(1025, -426), MAKE_VAL_COMPLEX(127, -2), MAKE_VAL_COMPLEX(-416, -196), MAKE_VAL_COMPLEX(695, 1687) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-4, 9);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_383, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_383", y_true, y_val, x336_LEN*x336_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_383() */
/* END TEST 383 */

#define A384_M 4    /* # of rows */
#define A384_N 5    /* # of cols */
#define A384_NNZ 4  /* # of stored non-zeros */
static oski_index_t A384_ptr[A384_N+1] = {1, 1, 3, 4, 5, 5}; /* end row pointers for A384 */
static oski_index_t A384_ind[A384_NNZ] = {2, 4, 3, 4}; /* end column indices pointers for A384 */
static oski_value_t A384_val[A384_NNZ] = {MAKE_VAL_COMPLEX(7, -10), MAKE_VAL_COMPLEX(-9, 10), MAKE_VAL_COMPLEX(-10, 5), MAKE_VAL_COMPLEX(10, 8)}; /* end stored non-zero values for A384 */

#define x384_LEN 5   /* vector length */
#define x384_VW  3   /* vector width */
static oski_value_t x384_val[x384_LEN*x384_VW] = {MAKE_VAL_COMPLEX(1, 2), MAKE_VAL_COMPLEX(0, 10), MAKE_VAL_COMPLEX(-6, 5), MAKE_VAL_COMPLEX(-10, -7), MAKE_VAL_COMPLEX(0, 3), MAKE_VAL_COMPLEX(-4, 7), MAKE_VAL_COMPLEX(-8, 5), MAKE_VAL_COMPLEX(9, -1), MAKE_VAL_COMPLEX(7, -6), MAKE_VAL_COMPLEX(-6, 5), MAKE_VAL_COMPLEX(2, 9), MAKE_VAL_COMPLEX(-5, 1), MAKE_VAL_COMPLEX(5, -8), MAKE_VAL_COMPLEX(10, -10), MAKE_VAL_COMPLEX(6, 0)}; /* data */

#define y384_LEN 4   /* vector length */
#define y384_VW  3   /* vector width */
static oski_value_t y384_val[y384_LEN*y384_VW] = {MAKE_VAL_COMPLEX(1, 10), MAKE_VAL_COMPLEX(10, -4), MAKE_VAL_COMPLEX(-9, -1), MAKE_VAL_COMPLEX(3, -7), MAKE_VAL_COMPLEX(10, -4), MAKE_VAL_COMPLEX(-10, 7), MAKE_VAL_COMPLEX(-6, -10), MAKE_VAL_COMPLEX(7, -9), MAKE_VAL_COMPLEX(-6, 6), MAKE_VAL_COMPLEX(5, 4), MAKE_VAL_COMPLEX(-2, -1), MAKE_VAL_COMPLEX(0, 10)}; /* data */

/*
 *  BEGIN TEST 384
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_384( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-33, 74), MAKE_VAL_COMPLEX(86, 12), MAKE_VAL_COMPLEX(-59, -43), MAKE_VAL_COMPLEX(-91, 14), MAKE_VAL_COMPLEX(116, 33), MAKE_VAL_COMPLEX(-56, 98), MAKE_VAL_COMPLEX(53, -184), MAKE_VAL_COMPLEX(0, 20), MAKE_VAL_COMPLEX(-106, 113), MAKE_VAL_COMPLEX(79, 13), MAKE_VAL_COMPLEX(-92, 64), MAKE_VAL_COMPLEX(-132, -63) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_384, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_384", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_384() */
/* END TEST 384 */

/*
 *  BEGIN TEST 385
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_385( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-93, -21), MAKE_VAL_COMPLEX(6, 102), MAKE_VAL_COMPLEX(36, -78), MAKE_VAL_COMPLEX(194, -3), MAKE_VAL_COMPLEX(-24, 81), MAKE_VAL_COMPLEX(-75, -200), MAKE_VAL_COMPLEX(53, 66), MAKE_VAL_COMPLEX(145, 35), MAKE_VAL_COMPLEX(4, -167), MAKE_VAL_COMPLEX(-111, 68), MAKE_VAL_COMPLEX(97, -94), MAKE_VAL_COMPLEX(2, 103) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_385, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_385", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_385() */
/* END TEST 385 */

/*
 *  BEGIN TEST 386
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_386( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(34, -64), MAKE_VAL_COMPLEX(-76, -16), MAKE_VAL_COMPLEX(50, 42), MAKE_VAL_COMPLEX(-46, 30), MAKE_VAL_COMPLEX(-76, -16), MAKE_VAL_COMPLEX(88, -2), MAKE_VAL_COMPLEX(-4, 84), MAKE_VAL_COMPLEX(-78, 26), MAKE_VAL_COMPLEX(60, -12), MAKE_VAL_COMPLEX(-14, -44), MAKE_VAL_COMPLEX(8, 14), MAKE_VAL_COMPLEX(40, -60) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_386, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_386", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_386() */
/* END TEST 386 */

/*
 *  BEGIN TEST 387
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_387( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-63, 77), MAKE_VAL_COMPLEX(98, 42), MAKE_VAL_COMPLEX(-56, -70), MAKE_VAL_COMPLEX(-706, 889), MAKE_VAL_COMPLEX(392, 69), MAKE_VAL_COMPLEX(531, 434), MAKE_VAL_COMPLEX(53, -962), MAKE_VAL_COMPLEX(-263, 711), MAKE_VAL_COMPLEX(16, 825), MAKE_VAL_COMPLEX(287, -422), MAKE_VAL_COMPLEX(-265, 860), MAKE_VAL_COMPLEX(-1246, -493) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(7, -4);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_387, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_387", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_387() */
/* END TEST 387 */

/*
 *  BEGIN TEST 388
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_388( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(8, -19), MAKE_VAL_COMPLEX(70, -60), MAKE_VAL_COMPLEX(71, 12), MAKE_VAL_COMPLEX(-123, 47), MAKE_VAL_COMPLEX(79, -157), MAKE_VAL_COMPLEX(-27, 45), MAKE_VAL_COMPLEX(193, 96), MAKE_VAL_COMPLEX(-86, 68), MAKE_VAL_COMPLEX(-54, -103), MAKE_VAL_COMPLEX(89, 92), MAKE_VAL_COMPLEX(39, -94), MAKE_VAL_COMPLEX(-43, 129), MAKE_VAL_COMPLEX(-86, 13), MAKE_VAL_COMPLEX(-130, -10), MAKE_VAL_COMPLEX(-36, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_388, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_388", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_388() */
/* END TEST 388 */

/*
 *  BEGIN TEST 389
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_389( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-2, -14), MAKE_VAL_COMPLEX(20, -60), MAKE_VAL_COMPLEX(46, -18), MAKE_VAL_COMPLEX(180, 127), MAKE_VAL_COMPLEX(-52, 121), MAKE_VAL_COMPLEX(138, -93), MAKE_VAL_COMPLEX(-52, -84), MAKE_VAL_COMPLEX(-31, -137), MAKE_VAL_COMPLEX(-84, 112), MAKE_VAL_COMPLEX(28, -98), MAKE_VAL_COMPLEX(18, -32), MAKE_VAL_COMPLEX(112, -96), MAKE_VAL_COMPLEX(-46, 38), MAKE_VAL_COMPLEX(-80, 40), MAKE_VAL_COMPLEX(-36, -12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_389, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_389", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_389() */
/* END TEST 389 */

/*
 *  BEGIN TEST 390
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_390( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-14, 2), MAKE_VAL_COMPLEX(-60, -20), MAKE_VAL_COMPLEX(-18, -46), MAKE_VAL_COMPLEX(62, -46), MAKE_VAL_COMPLEX(-18, -6), MAKE_VAL_COMPLEX(-34, -38), MAKE_VAL_COMPLEX(-14, -58), MAKE_VAL_COMPLEX(-12, 56), MAKE_VAL_COMPLEX(22, 54), MAKE_VAL_COMPLEX(-18, -46), MAKE_VAL_COMPLEX(-58, -6), MAKE_VAL_COMPLEX(4, -32), MAKE_VAL_COMPLEX(38, 46), MAKE_VAL_COMPLEX(40, 80), MAKE_VAL_COMPLEX(-12, 36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_390, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_390", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_390() */
/* END TEST 390 */

/*
 *  BEGIN TEST 391
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_391( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-15, 10), MAKE_VAL_COMPLEX(-80, 10), MAKE_VAL_COMPLEX(-46, -43), MAKE_VAL_COMPLEX(1512, -38), MAKE_VAL_COMPLEX(10, 1486), MAKE_VAL_COMPLEX(604, -956), MAKE_VAL_COMPLEX(-1318, -249), MAKE_VAL_COMPLEX(-258, -1154), MAKE_VAL_COMPLEX(145, 980), MAKE_VAL_COMPLEX(-528, -691), MAKE_VAL_COMPLEX(142, 211), MAKE_VAL_COMPLEX(307, -1259), MAKE_VAL_COMPLEX(69, 32), MAKE_VAL_COMPLEX(90, 70), MAKE_VAL_COMPLEX(6, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-9, 4);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_391, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_391", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_391() */
/* END TEST 391 */

/*
 *  BEGIN TEST 392
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_392( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, 18), MAKE_VAL_COMPLEX(-40, 70), MAKE_VAL_COMPLEX(-62, 11), MAKE_VAL_COMPLEX(44, -194), MAKE_VAL_COMPLEX(106, 122), MAKE_VAL_COMPLEX(-96, -108), MAKE_VAL_COMPLEX(-66, 133), MAKE_VAL_COMPLEX(-48, 84), MAKE_VAL_COMPLEX(163, -44), MAKE_VAL_COMPLEX(20, 11), MAKE_VAL_COMPLEX(-50, 77), MAKE_VAL_COMPLEX(41, 87), MAKE_VAL_COMPLEX(67, -36), MAKE_VAL_COMPLEX(110, -30), MAKE_VAL_COMPLEX(42, 24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_392, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_392", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_392() */
/* END TEST 392 */

/*
 *  BEGIN TEST 393
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_393( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-6, -2), MAKE_VAL_COMPLEX(-20, -20), MAKE_VAL_COMPLEX(2, -22), MAKE_VAL_COMPLEX(-52, 99), MAKE_VAL_COMPLEX(-124, -107), MAKE_VAL_COMPLEX(34, 119), MAKE_VAL_COMPLEX(-4, -156), MAKE_VAL_COMPLEX(99, -35), MAKE_VAL_COMPLEX(-92, 56), MAKE_VAL_COMPLEX(-80, -22), MAKE_VAL_COMPLEX(6, -20), MAKE_VAL_COMPLEX(-72, -112), MAKE_VAL_COMPLEX(6, 26), MAKE_VAL_COMPLEX(0, 40), MAKE_VAL_COMPLEX(-12, 12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_393, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_393", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_393() */
/* END TEST 393 */

/*
 *  BEGIN TEST 394
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_394( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-21, -2), MAKE_VAL_COMPLEX(-80, -50), MAKE_VAL_COMPLEX(-10, -73), MAKE_VAL_COMPLEX(106, -45), MAKE_VAL_COMPLEX(-24, -15), MAKE_VAL_COMPLEX(-36, -67), MAKE_VAL_COMPLEX(0, -89), MAKE_VAL_COMPLEX(-37, 77), MAKE_VAL_COMPLEX(13, 86), MAKE_VAL_COMPLEX(-10, -73), MAKE_VAL_COMPLEX(-82, -29), MAKE_VAL_COMPLEX(17, -45), MAKE_VAL_COMPLEX(39, 80), MAKE_VAL_COMPLEX(30, 130), MAKE_VAL_COMPLEX(-30, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_394, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_394", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_394() */
/* END TEST 394 */

/*
 *  BEGIN TEST 395
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_395( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(19, 3), MAKE_VAL_COMPLEX(70, 50), MAKE_VAL_COMPLEX(5, 67), MAKE_VAL_COMPLEX(-89, -834), MAKE_VAL_COMPLEX(1015, 48), MAKE_VAL_COMPLEX(-735, -482), MAKE_VAL_COMPLEX(565, 691), MAKE_VAL_COMPLEX(-317, 667), MAKE_VAL_COMPLEX(323, -589), MAKE_VAL_COMPLEX(415, -261), MAKE_VAL_COMPLEX(-43, 173), MAKE_VAL_COMPLEX(782, 220), MAKE_VAL_COMPLEX(-31, -75), MAKE_VAL_COMPLEX(-20, -120), MAKE_VAL_COMPLEX(30, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(5, -4);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_395, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_395", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_395() */
/* END TEST 395 */

/*
 *  BEGIN TEST 396
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_396( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-12, -19), MAKE_VAL_COMPLEX(-156, 69), MAKE_VAL_COMPLEX(74, -97), MAKE_VAL_COMPLEX(61, -18), MAKE_VAL_COMPLEX(-16, 18), MAKE_VAL_COMPLEX(43, -3), MAKE_VAL_COMPLEX(-63, 69), MAKE_VAL_COMPLEX(-87, 104), MAKE_VAL_COMPLEX(6, -18), MAKE_VAL_COMPLEX(28, 86), MAKE_VAL_COMPLEX(-35, 95), MAKE_VAL_COMPLEX(-102, -153) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_396, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_396", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_396() */
/* END TEST 396 */

/*
 *  BEGIN TEST 397
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_397( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(38, -24), MAKE_VAL_COMPLEX(104, -83), MAKE_VAL_COMPLEX(-41, 128), MAKE_VAL_COMPLEX(-94, 37), MAKE_VAL_COMPLEX(-36, -32), MAKE_VAL_COMPLEX(18, 5), MAKE_VAL_COMPLEX(57, -11), MAKE_VAL_COMPLEX(32, -89), MAKE_VAL_COMPLEX(36, 12), MAKE_VAL_COMPLEX(-36, -117), MAKE_VAL_COMPLEX(40, -85), MAKE_VAL_COMPLEX(132, 113) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_397, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_397", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_397() */
/* END TEST 397 */

/*
 *  BEGIN TEST 398
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_398( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(62, -87), MAKE_VAL_COMPLEX(-108, -38), MAKE_VAL_COMPLEX(65, 71), MAKE_VAL_COMPLEX(-73, 35), MAKE_VAL_COMPLEX(-108, -38), MAKE_VAL_COMPLEX(129, 14), MAKE_VAL_COMPLEX(-22, 122), MAKE_VAL_COMPLEX(-119, 23), MAKE_VAL_COMPLEX(90, -6), MAKE_VAL_COMPLEX(-12, -67), MAKE_VAL_COMPLEX(9, 22), MAKE_VAL_COMPLEX(70, -80) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_398, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_398", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_398() */
/* END TEST 398 */

/*
 *  BEGIN TEST 399
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_399( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(58, -26), MAKE_VAL_COMPLEX(1318, -231), MAKE_VAL_COMPLEX(-663, 756), MAKE_VAL_COMPLEX(-658, 191), MAKE_VAL_COMPLEX(-44, -52), MAKE_VAL_COMPLEX(-166, -203), MAKE_VAL_COMPLEX(827, -269), MAKE_VAL_COMPLEX(828, -571), MAKE_VAL_COMPLEX(48, 24), MAKE_VAL_COMPLEX(-186, -923), MAKE_VAL_COMPLEX(548, -761), MAKE_VAL_COMPLEX(622, 1361) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-9, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_399, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_399", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_399() */
/* END TEST 399 */

/*
 *  BEGIN TEST 400
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_400( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-5, -15), MAKE_VAL_COMPLEX(10, -70), MAKE_VAL_COMPLEX(47, -29), MAKE_VAL_COMPLEX(136, 24), MAKE_VAL_COMPLEX(30, 279), MAKE_VAL_COMPLEX(10, -157), MAKE_VAL_COMPLEX(156, -62), MAKE_VAL_COMPLEX(46, 68), MAKE_VAL_COMPLEX(-30, 35), MAKE_VAL_COMPLEX(133, -75), MAKE_VAL_COMPLEX(137, -99), MAKE_VAL_COMPLEX(-44, 98), MAKE_VAL_COMPLEX(-43, 51), MAKE_VAL_COMPLEX(-80, 60), MAKE_VAL_COMPLEX(-42, -6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_400, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_400", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_400() */
/* END TEST 400 */

/*
 *  BEGIN TEST 401
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_401( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(13, 6), MAKE_VAL_COMPLEX(40, 50), MAKE_VAL_COMPLEX(-10, 49), MAKE_VAL_COMPLEX(-151, 40), MAKE_VAL_COMPLEX(-15, -285), MAKE_VAL_COMPLEX(33, 163), MAKE_VAL_COMPLEX(-115, 92), MAKE_VAL_COMPLEX(-69, -111), MAKE_VAL_COMPLEX(-14, -58), MAKE_VAL_COMPLEX(-96, 95), MAKE_VAL_COMPLEX(-96, 71), MAKE_VAL_COMPLEX(59, -75), MAKE_VAL_COMPLEX(-7, -60), MAKE_VAL_COMPLEX(10, -90), MAKE_VAL_COMPLEX(30, -24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_401, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_401", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_401() */
/* END TEST 401 */

/*
 *  BEGIN TEST 402
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_402( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(15, 0), MAKE_VAL_COMPLEX(60, 30), MAKE_VAL_COMPLEX(12, 51), MAKE_VAL_COMPLEX(-72, 39), MAKE_VAL_COMPLEX(18, 9), MAKE_VAL_COMPLEX(30, 45), MAKE_VAL_COMPLEX(6, 63), MAKE_VAL_COMPLEX(21, -57), MAKE_VAL_COMPLEX(-15, -60), MAKE_VAL_COMPLEX(12, 51), MAKE_VAL_COMPLEX(60, 15), MAKE_VAL_COMPLEX(-9, 33), MAKE_VAL_COMPLEX(-33, -54), MAKE_VAL_COMPLEX(-30, -90), MAKE_VAL_COMPLEX(18, -36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_402, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_402", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_402() */
/* END TEST 402 */

/*
 *  BEGIN TEST 403
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_403( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(9, -17), MAKE_VAL_COMPLEX(70, -50), MAKE_VAL_COMPLEX(65, 17), MAKE_VAL_COMPLEX(512, -140), MAKE_VAL_COMPLEX(210, 2085), MAKE_VAL_COMPLEX(-106, -791), MAKE_VAL_COMPLEX(740, -214), MAKE_VAL_COMPLEX(718, 432), MAKE_VAL_COMPLEX(98, -19), MAKE_VAL_COMPLEX(667, -305), MAKE_VAL_COMPLEX(1047, -297), MAKE_VAL_COMPLEX(-528, 730), MAKE_VAL_COMPLEX(-81, 5), MAKE_VAL_COMPLEX(-120, -20), MAKE_VAL_COMPLEX(-30, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(7, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_403, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_403", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_403() */
/* END TEST 403 */

/*
 *  BEGIN TEST 404
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_404( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(7, -11), MAKE_VAL_COMPLEX(50, -30), MAKE_VAL_COMPLEX(43, 15), MAKE_VAL_COMPLEX(8, 176), MAKE_VAL_COMPLEX(-278, -49), MAKE_VAL_COMPLEX(142, -13), MAKE_VAL_COMPLEX(134, 80), MAKE_VAL_COMPLEX(-22, 88), MAKE_VAL_COMPLEX(-36, 3), MAKE_VAL_COMPLEX(17, -79), MAKE_VAL_COMPLEX(37, -183), MAKE_VAL_COMPLEX(100, 122), MAKE_VAL_COMPLEX(-55, -1), MAKE_VAL_COMPLEX(-80, -20), MAKE_VAL_COMPLEX(-18, -30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_404, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_404", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_404() */
/* END TEST 404 */

/*
 *  BEGIN TEST 405
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_405( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(10, -20), MAKE_VAL_COMPLEX(80, -60), MAKE_VAL_COMPLEX(76, 18), MAKE_VAL_COMPLEX(-9, 17), MAKE_VAL_COMPLEX(317, 22), MAKE_VAL_COMPLEX(-15, 2), MAKE_VAL_COMPLEX(3, -21), MAKE_VAL_COMPLEX(-72, -196), MAKE_VAL_COMPLEX(-105, -40), MAKE_VAL_COMPLEX(102, 112), MAKE_VAL_COMPLEX(62, 76), MAKE_VAL_COMPLEX(-42, -66), MAKE_VAL_COMPLEX(-94, 8), MAKE_VAL_COMPLEX(-140, -20), MAKE_VAL_COMPLEX(-36, -48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_405, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_405", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_405() */
/* END TEST 405 */

/*
 *  BEGIN TEST 406
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_406( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-8, 4), MAKE_VAL_COMPLEX(-40, 0), MAKE_VAL_COMPLEX(-20, -24), MAKE_VAL_COMPLEX(28, -40), MAKE_VAL_COMPLEX(-12, 0), MAKE_VAL_COMPLEX(-28, -16), MAKE_VAL_COMPLEX(-20, -32), MAKE_VAL_COMPLEX(4, 36), MAKE_VAL_COMPLEX(24, 28), MAKE_VAL_COMPLEX(-20, -24), MAKE_VAL_COMPLEX(-36, 8), MAKE_VAL_COMPLEX(-4, -20), MAKE_VAL_COMPLEX(32, 20), MAKE_VAL_COMPLEX(40, 40), MAKE_VAL_COMPLEX(0, 24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_406, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_406", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_406() */
/* END TEST 406 */

/*
 *  BEGIN TEST 407
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_ROWMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_407( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_ROWMAJ, x384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-5, 15), MAKE_VAL_COMPLEX(-50, 50), MAKE_VAL_COMPLEX(-55, -5), MAKE_VAL_COMPLEX(524, -898), MAKE_VAL_COMPLEX(1796, 2053), MAKE_VAL_COMPLEX(-792, -471), MAKE_VAL_COMPLEX(-330, -910), MAKE_VAL_COMPLEX(760, -930), MAKE_VAL_COMPLEX(80, -225), MAKE_VAL_COMPLEX(-437, 809), MAKE_VAL_COMPLEX(-897, 1089), MAKE_VAL_COMPLEX(10, -1200), MAKE_VAL_COMPLEX(65, -15), MAKE_VAL_COMPLEX(100, 0), MAKE_VAL_COMPLEX(30, 30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-7, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_407, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_407", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_407() */
/* END TEST 407 */

/*
 *  BEGIN TEST 408
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_408( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-42, -16), MAKE_VAL_COMPLEX(-4, 48), MAKE_VAL_COMPLEX(22, -34), MAKE_VAL_COMPLEX(122, 96), MAKE_VAL_COMPLEX(-10, 163), MAKE_VAL_COMPLEX(-33, 3), MAKE_VAL_COMPLEX(87, -84), MAKE_VAL_COMPLEX(-63, 101), MAKE_VAL_COMPLEX(-22, 69), MAKE_VAL_COMPLEX(-170, -228), MAKE_VAL_COMPLEX(148, -135), MAKE_VAL_COMPLEX(175, -99) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_408, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_408", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_408() */
/* END TEST 408 */

/*
 *  BEGIN TEST 409
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_409( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(15, -52), MAKE_VAL_COMPLEX(-58, 0), MAKE_VAL_COMPLEX(43, 23), MAKE_VAL_COMPLEX(-129, -41), MAKE_VAL_COMPLEX(-52, -115), MAKE_VAL_COMPLEX(89, -72), MAKE_VAL_COMPLEX(-25, 142), MAKE_VAL_COMPLEX(32, -24), MAKE_VAL_COMPLEX(52, -123), MAKE_VAL_COMPLEX(127, 210), MAKE_VAL_COMPLEX(-132, 138), MAKE_VAL_COMPLEX(-195, 29) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_409, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_409", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_409() */
/* END TEST 409 */

/*
 *  BEGIN TEST 410
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_410( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(33, 27), MAKE_VAL_COMPLEX(18, -42), MAKE_VAL_COMPLEX(-30, 24), MAKE_VAL_COMPLEX(-12, -30), MAKE_VAL_COMPLEX(18, -42), MAKE_VAL_COMPLEX(-9, 51), MAKE_VAL_COMPLEX(-48, -12), MAKE_VAL_COMPLEX(-6, -48), MAKE_VAL_COMPLEX(0, 36), MAKE_VAL_COMPLEX(27, -3), MAKE_VAL_COMPLEX(-9, 3), MAKE_VAL_COMPLEX(30, 30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_410, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_410", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_410() */
/* END TEST 410 */

/*
 *  BEGIN TEST 411
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_411( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(66, -47), MAKE_VAL_COMPLEX(-68, -54), MAKE_VAL_COMPLEX(29, 67), MAKE_VAL_COMPLEX(-471, -1323), MAKE_VAL_COMPLEX(791, -1047), MAKE_VAL_COMPLEX(713, -296), MAKE_VAL_COMPLEX(-921, 557), MAKE_VAL_COMPLEX(1059, 87), MAKE_VAL_COMPLEX(891, -857), MAKE_VAL_COMPLEX(-376, 3117), MAKE_VAL_COMPLEX(-2162, 199), MAKE_VAL_COMPLEX(-2418, -834) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-9, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_411, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_411", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_411() */
/* END TEST 411 */

/*
 *  BEGIN TEST 412
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_412( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(4, -2), MAKE_VAL_COMPLEX(-114, -65), MAKE_VAL_COMPLEX(120, 82), MAKE_VAL_COMPLEX(4, 100), MAKE_VAL_COMPLEX(6, 0), MAKE_VAL_COMPLEX(14, 8), MAKE_VAL_COMPLEX(68, -123), MAKE_VAL_COMPLEX(-27, 107), MAKE_VAL_COMPLEX(-24, -40), MAKE_VAL_COMPLEX(10, 12), MAKE_VAL_COMPLEX(18, -4), MAKE_VAL_COMPLEX(-98, 69), MAKE_VAL_COMPLEX(14, -100), MAKE_VAL_COMPLEX(-100, 80), MAKE_VAL_COMPLEX(0, -12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_412, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_412", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_412() */
/* END TEST 412 */

/*
 *  BEGIN TEST 413
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_413( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(8, -9), MAKE_VAL_COMPLEX(184, 45), MAKE_VAL_COMPLEX(-73, -50), MAKE_VAL_COMPLEX(-33, -16), MAKE_VAL_COMPLEX(15, -6), MAKE_VAL_COMPLEX(43, 6), MAKE_VAL_COMPLEX(-17, 169), MAKE_VAL_COMPLEX(2, -168), MAKE_VAL_COMPLEX(-32, 3), MAKE_VAL_COMPLEX(37, 20), MAKE_VAL_COMPLEX(41, -28), MAKE_VAL_COMPLEX(115, -36), MAKE_VAL_COMPLEX(-80, 81), MAKE_VAL_COMPLEX(10, -130), MAKE_VAL_COMPLEX(-12, -30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_413, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_413", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_413() */
/* END TEST 413 */

/*
 *  BEGIN TEST 414
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_414( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, 14), MAKE_VAL_COMPLEX(40, 90), MAKE_VAL_COMPLEX(-34, 69), MAKE_VAL_COMPLEX(-118, -23), MAKE_VAL_COMPLEX(12, 27), MAKE_VAL_COMPLEX(-8, 79), MAKE_VAL_COMPLEX(-52, 77), MAKE_VAL_COMPLEX(77, -45), MAKE_VAL_COMPLEX(39, -82), MAKE_VAL_COMPLEX(-34, 69), MAKE_VAL_COMPLEX(54, 73), MAKE_VAL_COMPLEX(-41, 29), MAKE_VAL_COMPLEX(13, -92), MAKE_VAL_COMPLEX(50, -130), MAKE_VAL_COMPLEX(54, -24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, -4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_414, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_414", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_414() */
/* END TEST 414 */

/*
 *  BEGIN TEST 415
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_415( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(4, 8), MAKE_VAL_COMPLEX(-191, 775), MAKE_VAL_COMPLEX(216, -600), MAKE_VAL_COMPLEX(342, -198), MAKE_VAL_COMPLEX(0, 12), MAKE_VAL_COMPLEX(-16, 28), MAKE_VAL_COMPLEX(-785, -131), MAKE_VAL_COMPLEX(686, -4), MAKE_VAL_COMPLEX(-90, 62), MAKE_VAL_COMPLEX(-24, 20), MAKE_VAL_COMPLEX(8, 36), MAKE_VAL_COMPLEX(375, 445), MAKE_VAL_COMPLEX(-460, -92), MAKE_VAL_COMPLEX(620, 260), MAKE_VAL_COMPLEX(24, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 0);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_415, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_415", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_415() */
/* END TEST 415 */

/*
 *  BEGIN TEST 416
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_416( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(7, -26), MAKE_VAL_COMPLEX(166, -195), MAKE_VAL_COMPLEX(104, 133), MAKE_VAL_COMPLEX(116, 143), MAKE_VAL_COMPLEX(24, -27), MAKE_VAL_COMPLEX(92, -31), MAKE_VAL_COMPLEX(230, 120), MAKE_VAL_COMPLEX(-204, -8), MAKE_VAL_COMPLEX(-139, 4), MAKE_VAL_COMPLEX(94, 3), MAKE_VAL_COMPLEX(54, -97), MAKE_VAL_COMPLEX(13, -110), MAKE_VAL_COMPLEX(-19, 2), MAKE_VAL_COMPLEX(-90, 110), MAKE_VAL_COMPLEX(-54, -48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_416, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_416", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_416() */
/* END TEST 416 */

/*
 *  BEGIN TEST 417
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_417( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, 5), MAKE_VAL_COMPLEX(-96, 125), MAKE_VAL_COMPLEX(-27, -126), MAKE_VAL_COMPLEX(-95, -24), MAKE_VAL_COMPLEX(-3, 6), MAKE_VAL_COMPLEX(-15, 10), MAKE_VAL_COMPLEX(-139, -99), MAKE_VAL_COMPLEX(134, -48), MAKE_VAL_COMPLEX(48, -11), MAKE_VAL_COMPLEX(-17, 4), MAKE_VAL_COMPLEX(-5, 20), MAKE_VAL_COMPLEX(29, 138), MAKE_VAL_COMPLEX(-72, 19), MAKE_VAL_COMPLEX(-50, -110), MAKE_VAL_COMPLEX(12, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_417, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_417", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_417() */
/* END TEST 417 */

/*
 *  BEGIN TEST 418
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_418( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-17, 6), MAKE_VAL_COMPLEX(-80, -10), MAKE_VAL_COMPLEX(-34, -53), MAKE_VAL_COMPLEX(66, -73), MAKE_VAL_COMPLEX(-24, -3), MAKE_VAL_COMPLEX(-52, -39), MAKE_VAL_COMPLEX(-32, -69), MAKE_VAL_COMPLEX(-1, 73), MAKE_VAL_COMPLEX(41, 62), MAKE_VAL_COMPLEX(-34, -53), MAKE_VAL_COMPLEX(-74, 7), MAKE_VAL_COMPLEX(-3, -41), MAKE_VAL_COMPLEX(59, 48), MAKE_VAL_COMPLEX(70, 90), MAKE_VAL_COMPLEX(-6, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_418, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_418", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_418() */
/* END TEST 418 */

/*
 *  BEGIN TEST 419
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_419( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_ROWMAJ, y384_VW );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-13, 4), MAKE_VAL_COMPLEX(331, 1012), MAKE_VAL_COMPLEX(-974, -491), MAKE_VAL_COMPLEX(-276, 521), MAKE_VAL_COMPLEX(-18, -3), MAKE_VAL_COMPLEX(-38, -31), MAKE_VAL_COMPLEX(-1201, 369), MAKE_VAL_COMPLEX(72, -970), MAKE_VAL_COMPLEX(99, -172), MAKE_VAL_COMPLEX(-24, -41), MAKE_VAL_COMPLEX(-56, 3), MAKE_VAL_COMPLEX(1146, 253), MAKE_VAL_COMPLEX(-107, 788), MAKE_VAL_COMPLEX(-970, 230), MAKE_VAL_COMPLEX(-6, 36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-4, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_419, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_419", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_419() */
/* END TEST 419 */

/*
 *  BEGIN TEST 420
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_420( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(34, -64), MAKE_VAL_COMPLEX(24, 54), MAKE_VAL_COMPLEX(85, -38), MAKE_VAL_COMPLEX(-190, -210), MAKE_VAL_COMPLEX(-76, -16), MAKE_VAL_COMPLEX(82, 113), MAKE_VAL_COMPLEX(-89, 139), MAKE_VAL_COMPLEX(62, -103), MAKE_VAL_COMPLEX(60, -12), MAKE_VAL_COMPLEX(-39, 13), MAKE_VAL_COMPLEX(-2, 119), MAKE_VAL_COMPLEX(255, -139) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_420, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_420", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_420() */
/* END TEST 420 */

/*
 *  BEGIN TEST 421
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_421( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-54, 66), MAKE_VAL_COMPLEX(-16, -34), MAKE_VAL_COMPLEX(-83, 20), MAKE_VAL_COMPLEX(204, 216), MAKE_VAL_COMPLEX(84, 36), MAKE_VAL_COMPLEX(-96, -133), MAKE_VAL_COMPLEX(109, -151), MAKE_VAL_COMPLEX(-44, 117), MAKE_VAL_COMPLEX(-72, 0), MAKE_VAL_COMPLEX(31, -3), MAKE_VAL_COMPLEX(4, -123), MAKE_VAL_COMPLEX(-275, 139) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_421, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_421", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_421() */
/* END TEST 421 */

/*
 *  BEGIN TEST 422
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_422( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-25, -48), MAKE_VAL_COMPLEX(-42, 40), MAKE_VAL_COMPLEX(47, -13), MAKE_VAL_COMPLEX(-1, 41), MAKE_VAL_COMPLEX(-42, 40), MAKE_VAL_COMPLEX(36, -55), MAKE_VAL_COMPLEX(50, 38), MAKE_VAL_COMPLEX(-17, 59), MAKE_VAL_COMPLEX(18, -42), MAKE_VAL_COMPLEX(-33, -10), MAKE_VAL_COMPLEX(12, 1), MAKE_VAL_COMPLEX(-20, -50) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_422, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_422", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_422() */
/* END TEST 422 */

/*
 *  BEGIN TEST 423
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_423( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	oski_value_t* y_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(42, -85), MAKE_VAL_COMPLEX(-60, -288), MAKE_VAL_COMPLEX(-128, 63), MAKE_VAL_COMPLEX(-395, 569), MAKE_VAL_COMPLEX(-100, -18), MAKE_VAL_COMPLEX(351, -109), MAKE_VAL_COMPLEX(193, 225), MAKE_VAL_COMPLEX(-499, -114), MAKE_VAL_COMPLEX(78, -18), MAKE_VAL_COMPLEX(119, -64), MAKE_VAL_COMPLEX(231, -67), MAKE_VAL_COMPLEX(-323, -431) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_423, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_423", y_true, y_val, y384_LEN*y384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_423() */
/* END TEST 423 */

/*
 *  BEGIN TEST 424
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_424( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-11, -2), MAKE_VAL_COMPLEX(33, -65), MAKE_VAL_COMPLEX(93, -74), MAKE_VAL_COMPLEX(144, -65), MAKE_VAL_COMPLEX(-12, -9), MAKE_VAL_COMPLEX(-16, -37), MAKE_VAL_COMPLEX(31, 253), MAKE_VAL_COMPLEX(87, 109), MAKE_VAL_COMPLEX(145, 12), MAKE_VAL_COMPLEX(-2, -39), MAKE_VAL_COMPLEX(-42, -19), MAKE_VAL_COMPLEX(-14, -135), MAKE_VAL_COMPLEX(42, 44), MAKE_VAL_COMPLEX(-70, 170), MAKE_VAL_COMPLEX(-18, 24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_424, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_424", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_424() */
/* END TEST 424 */

/*
 *  BEGIN TEST 425
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_425( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-15, 0), MAKE_VAL_COMPLEX(-133, 5), MAKE_VAL_COMPLEX(-107, -16), MAKE_VAL_COMPLEX(-14, 7), MAKE_VAL_COMPLEX(-18, -9), MAKE_VAL_COMPLEX(-30, -45), MAKE_VAL_COMPLEX(-33, -363), MAKE_VAL_COMPLEX(-131, -13), MAKE_VAL_COMPLEX(-127, 94), MAKE_VAL_COMPLEX(-12, -51), MAKE_VAL_COMPLEX(-60, -15), MAKE_VAL_COMPLEX(34, 79), MAKE_VAL_COMPLEX(8, 54), MAKE_VAL_COMPLEX(110, -10), MAKE_VAL_COMPLEX(-18, 36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_425, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_425", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_425() */
/* END TEST 425 */

/*
 *  BEGIN TEST 426
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_426( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-14, -8), MAKE_VAL_COMPLEX(-40, -60), MAKE_VAL_COMPLEX(16, -54), MAKE_VAL_COMPLEX(88, 2), MAKE_VAL_COMPLEX(-12, -18), MAKE_VAL_COMPLEX(-4, -58), MAKE_VAL_COMPLEX(28, -62), MAKE_VAL_COMPLEX(-50, 42), MAKE_VAL_COMPLEX(-18, 64), MAKE_VAL_COMPLEX(16, -54), MAKE_VAL_COMPLEX(-48, -46), MAKE_VAL_COMPLEX(26, -26), MAKE_VAL_COMPLEX(2, 68), MAKE_VAL_COMPLEX(-20, 100), MAKE_VAL_COMPLEX(-36, 24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_426, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_426", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_426() */
/* END TEST 426 */

/*
 *  BEGIN TEST 427
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_427( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(18, 1), MAKE_VAL_COMPLEX(193, -827), MAKE_VAL_COMPLEX(266, -1003), MAKE_VAL_COMPLEX(13, -1008), MAKE_VAL_COMPLEX(21, 12), MAKE_VAL_COMPLEX(33, 56), MAKE_VAL_COMPLEX(2865, 1633), MAKE_VAL_COMPLEX(1319, -637), MAKE_VAL_COMPLEX(532, -1555), MAKE_VAL_COMPLEX(11, 62), MAKE_VAL_COMPLEX(71, 22), MAKE_VAL_COMPLEX(-1171, -408), MAKE_VAL_COMPLEX(114, -292), MAKE_VAL_COMPLEX(390, 1210), MAKE_VAL_COMPLEX(24, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, -9);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_427, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_427", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_427() */
/* END TEST 427 */

/*
 *  BEGIN TEST 428
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_428( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-7, 21), MAKE_VAL_COMPLEX(-57, 175), MAKE_VAL_COMPLEX(8, 48), MAKE_VAL_COMPLEX(-47, -213), MAKE_VAL_COMPLEX(-21, 21), MAKE_VAL_COMPLEX(-77, 21), MAKE_VAL_COMPLEX(-384, -61), MAKE_VAL_COMPLEX(80, 186), MAKE_VAL_COMPLEX(89, -139), MAKE_VAL_COMPLEX(-77, -7), MAKE_VAL_COMPLEX(-49, 77), MAKE_VAL_COMPLEX(53, -40), MAKE_VAL_COMPLEX(106, -1), MAKE_VAL_COMPLEX(220, 100), MAKE_VAL_COMPLEX(42, 42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_428, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_428", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_428() */
/* END TEST 428 */

/*
 *  BEGIN TEST 429
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_429( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-2, 6), MAKE_VAL_COMPLEX(-33, -85), MAKE_VAL_COMPLEX(-107, -57), MAKE_VAL_COMPLEX(20, 60), MAKE_VAL_COMPLEX(-6, 6), MAKE_VAL_COMPLEX(-22, 6), MAKE_VAL_COMPLEX(267, 34), MAKE_VAL_COMPLEX(10, -114), MAKE_VAL_COMPLEX(28, 148), MAKE_VAL_COMPLEX(-22, -2), MAKE_VAL_COMPLEX(-14, 22), MAKE_VAL_COMPLEX(-107, 4), MAKE_VAL_COMPLEX(11, -26), MAKE_VAL_COMPLEX(-40, -100), MAKE_VAL_COMPLEX(12, 12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_429, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_429", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_429() */
/* END TEST 429 */

/*
 *  BEGIN TEST 430
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_430( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, -11), MAKE_VAL_COMPLEX(90, -10), MAKE_VAL_COMPLEX(51, 49), MAKE_VAL_COMPLEX(-53, 97), MAKE_VAL_COMPLEX(27, -3), MAKE_VAL_COMPLEX(67, 29), MAKE_VAL_COMPLEX(53, 67), MAKE_VAL_COMPLEX(-18, -80), MAKE_VAL_COMPLEX(-61, -57), MAKE_VAL_COMPLEX(51, 49), MAKE_VAL_COMPLEX(79, -27), MAKE_VAL_COMPLEX(14, 44), MAKE_VAL_COMPLEX(-77, -37), MAKE_VAL_COMPLEX(-100, -80), MAKE_VAL_COMPLEX(-6, -54) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_430, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_430", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_430() */
/* END TEST 430 */

/*
 *  BEGIN TEST 431
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x384', LAYOUT_COLMAJ>
 *  y : <'y384', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_431( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A384_ptr, A384_ind, A384_val,
		A384_M, A384_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y384_val,
		y384_LEN * y384_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y384_LEN, y384_VW, LAYOUT_COLMAJ, y384_LEN );
	oski_value_t* y_val = array_Duplicate( x384_val,
		x384_LEN * x384_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x384_LEN, x384_VW, LAYOUT_COLMAJ, x384_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-4, -8), MAKE_VAL_COMPLEX(170, 472), MAKE_VAL_COMPLEX(504, 170), MAKE_VAL_COMPLEX(-184, -416), MAKE_VAL_COMPLEX(0, -12), MAKE_VAL_COMPLEX(16, -28), MAKE_VAL_COMPLEX(-1473, 73), MAKE_VAL_COMPLEX(144, 644), MAKE_VAL_COMPLEX(-184, -704), MAKE_VAL_COMPLEX(24, -20), MAKE_VAL_COMPLEX(-8, -36), MAKE_VAL_COMPLEX(483, -159), MAKE_VAL_COMPLEX(75, 117), MAKE_VAL_COMPLEX(460, 460), MAKE_VAL_COMPLEX(-24, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(5, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_431, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_431", y_true, y_val, x384_LEN*x384_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_431() */
/* END TEST 431 */

#define A432_M 4    /* # of rows */
#define A432_N 5    /* # of cols */
#define A432_NNZ 4  /* # of stored non-zeros */
static oski_index_t A432_ptr[A432_N+1] = {0, 2, 2, 3, 4, 4}; /* end row pointers for A432 */
static oski_index_t A432_ind[A432_NNZ] = {1, 2, 2, 3}; /* end column indices pointers for A432 */
static oski_value_t A432_val[A432_NNZ] = {MAKE_VAL_COMPLEX(-10, -10), MAKE_VAL_COMPLEX(10, -4), MAKE_VAL_COMPLEX(-5, -3), MAKE_VAL_COMPLEX(7, 10)}; /* end stored non-zero values for A432 */

#define x432_LEN 5   /* vector length */
#define x432_VW  3   /* vector width */
static oski_value_t x432_val[x432_LEN*x432_VW] = {MAKE_VAL_COMPLEX(-3, 1), MAKE_VAL_COMPLEX(1, -1), MAKE_VAL_COMPLEX(1, -5), MAKE_VAL_COMPLEX(9, 10), MAKE_VAL_COMPLEX(2, -3), MAKE_VAL_COMPLEX(10, 1), MAKE_VAL_COMPLEX(5, -5), MAKE_VAL_COMPLEX(-3, -10), MAKE_VAL_COMPLEX(6, -3), MAKE_VAL_COMPLEX(-10, 5), MAKE_VAL_COMPLEX(-8, 4), MAKE_VAL_COMPLEX(0, 5), MAKE_VAL_COMPLEX(0, 7), MAKE_VAL_COMPLEX(2, 2), MAKE_VAL_COMPLEX(4, 2)}; /* data */

#define y432_LEN 4   /* vector length */
#define y432_VW  3   /* vector width */
static oski_value_t y432_val[y432_LEN*y432_VW] = {MAKE_VAL_COMPLEX(5, -2), MAKE_VAL_COMPLEX(1, -5), MAKE_VAL_COMPLEX(-7, -9), MAKE_VAL_COMPLEX(-4, 0), MAKE_VAL_COMPLEX(-10, -10), MAKE_VAL_COMPLEX(10, 10), MAKE_VAL_COMPLEX(1, 6), MAKE_VAL_COMPLEX(-3, 1), MAKE_VAL_COMPLEX(-7, -8), MAKE_VAL_COMPLEX(6, 5), MAKE_VAL_COMPLEX(7, -8), MAKE_VAL_COMPLEX(-8, 0)}; /* data */

/*
 *  BEGIN TEST 432
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_432( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-20, -21), MAKE_VAL_COMPLEX(-27, 5), MAKE_VAL_COMPLEX(-31, 53), MAKE_VAL_COMPLEX(48, 40), MAKE_VAL_COMPLEX(-50, 70), MAKE_VAL_COMPLEX(-30, -30), MAKE_VAL_COMPLEX(-38, 15), MAKE_VAL_COMPLEX(2, 58), MAKE_VAL_COMPLEX(-75, -6), MAKE_VAL_COMPLEX(-107, -105), MAKE_VAL_COMPLEX(-150, -71), MAKE_VAL_COMPLEX(-34, 75) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_432, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_432", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_432() */
/* END TEST 432 */

/*
 *  BEGIN TEST 433
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_433( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(47, 16), MAKE_VAL_COMPLEX(37, -29), MAKE_VAL_COMPLEX(5, -105), MAKE_VAL_COMPLEX(-68, -44), MAKE_VAL_COMPLEX(10, -130), MAKE_VAL_COMPLEX(70, 90), MAKE_VAL_COMPLEX(37, 16), MAKE_VAL_COMPLEX(-18, -56), MAKE_VAL_COMPLEX(48, -41), MAKE_VAL_COMPLEX(132, 136), MAKE_VAL_COMPLEX(193, 38), MAKE_VAL_COMPLEX(-6, -83) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_433, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_433", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_433() */
/* END TEST 433 */

/*
 *  BEGIN TEST 434
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_434( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-14, -6), MAKE_VAL_COMPLEX(-12, 8), MAKE_VAL_COMPLEX(-4, 32), MAKE_VAL_COMPLEX(8, 8), MAKE_VAL_COMPLEX(0, 40), MAKE_VAL_COMPLEX(0, -40), MAKE_VAL_COMPLEX(10, -14), MAKE_VAL_COMPLEX(8, 4), MAKE_VAL_COMPLEX(-2, 30), MAKE_VAL_COMPLEX(-2, -22), MAKE_VAL_COMPLEX(-30, 2), MAKE_VAL_COMPLEX(16, 16) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_434, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_434", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_434() */
/* END TEST 434 */

/*
 *  BEGIN TEST 435
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_435( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(61, 22), MAKE_VAL_COMPLEX(49, -37), MAKE_VAL_COMPLEX(9, -137), MAKE_VAL_COMPLEX(-176, -352), MAKE_VAL_COMPLEX(110, -70), MAKE_VAL_COMPLEX(570, 230), MAKE_VAL_COMPLEX(517, 200), MAKE_VAL_COMPLEX(244, -240), MAKE_VAL_COMPLEX(10, 459), MAKE_VAL_COMPLEX(409, 1083), MAKE_VAL_COMPLEX(443, 776), MAKE_VAL_COMPLEX(403, -24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_435, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_435", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_435() */
/* END TEST 435 */

/*
 *  BEGIN TEST 436
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_436( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(53, 73), MAKE_VAL_COMPLEX(-13, 227), MAKE_VAL_COMPLEX(-53, -263), MAKE_VAL_COMPLEX(-54, 121), MAKE_VAL_COMPLEX(35, 6), MAKE_VAL_COMPLEX(31, 94), MAKE_VAL_COMPLEX(78, -8), MAKE_VAL_COMPLEX(96, -63), MAKE_VAL_COMPLEX(62, 103), MAKE_VAL_COMPLEX(-93, 25), MAKE_VAL_COMPLEX(61, -42), MAKE_VAL_COMPLEX(-101, -60), MAKE_VAL_COMPLEX(-63, 28), MAKE_VAL_COMPLEX(-10, 26), MAKE_VAL_COMPLEX(-2, 44) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 9);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_436, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_436", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_436() */
/* END TEST 436 */

/*
 *  BEGIN TEST 437
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_437( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-86, -92), MAKE_VAL_COMPLEX(30, -226), MAKE_VAL_COMPLEX(106, 232), MAKE_VAL_COMPLEX(36, 40), MAKE_VAL_COMPLEX(8, -12), MAKE_VAL_COMPLEX(40, 4), MAKE_VAL_COMPLEX(7, 13), MAKE_VAL_COMPLEX(-30, -44), MAKE_VAL_COMPLEX(13, -73), MAKE_VAL_COMPLEX(-32, -75), MAKE_VAL_COMPLEX(-161, 2), MAKE_VAL_COMPLEX(56, 100), MAKE_VAL_COMPLEX(0, 28), MAKE_VAL_COMPLEX(8, 8), MAKE_VAL_COMPLEX(16, 8) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 0);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_437, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_437", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_437() */
/* END TEST 437 */

/*
 *  BEGIN TEST 438
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_438( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-1, -3), MAKE_VAL_COMPLEX(1, 1), MAKE_VAL_COMPLEX(5, 1), MAKE_VAL_COMPLEX(-10, 9), MAKE_VAL_COMPLEX(3, 2), MAKE_VAL_COMPLEX(-1, 10), MAKE_VAL_COMPLEX(5, 5), MAKE_VAL_COMPLEX(10, -3), MAKE_VAL_COMPLEX(3, 6), MAKE_VAL_COMPLEX(-5, -10), MAKE_VAL_COMPLEX(-4, -8), MAKE_VAL_COMPLEX(-5, 0), MAKE_VAL_COMPLEX(-7, 0), MAKE_VAL_COMPLEX(-2, 2), MAKE_VAL_COMPLEX(-2, 4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_438, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_438", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_438() */
/* END TEST 438 */

/*
 *  BEGIN TEST 439
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_439( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-184, 328), MAKE_VAL_COMPLEX(-706, 146), MAKE_VAL_COMPLEX(616, -524), MAKE_VAL_COMPLEX(-12, -134), MAKE_VAL_COMPLEX(-34, 12), MAKE_VAL_COMPLEX(-74, -68), MAKE_VAL_COMPLEX(42, 16), MAKE_VAL_COMPLEX(-30, 156), MAKE_VAL_COMPLEX(-238, 82), MAKE_VAL_COMPLEX(-183, 91), MAKE_VAL_COMPLEX(175, 417), MAKE_VAL_COMPLEX(214, -288), MAKE_VAL_COMPLEX(42, -56), MAKE_VAL_COMPLEX(-4, -28), MAKE_VAL_COMPLEX(-20, -40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_439, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_439", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_439() */
/* END TEST 439 */

/*
 *  BEGIN TEST 440
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_440( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(40, -4), MAKE_VAL_COMPLEX(166, 12), MAKE_VAL_COMPLEX(-210, -66), MAKE_VAL_COMPLEX(-133, -7), MAKE_VAL_COMPLEX(7, 35), MAKE_VAL_COMPLEX(-77, 63), MAKE_VAL_COMPLEX(-23, 43), MAKE_VAL_COMPLEX(103, 35), MAKE_VAL_COMPLEX(38, 82), MAKE_VAL_COMPLEX(127, -130), MAKE_VAL_COMPLEX(-3, -210), MAKE_VAL_COMPLEX(-91, 45), MAKE_VAL_COMPLEX(-49, -49), MAKE_VAL_COMPLEX(-28, 0), MAKE_VAL_COMPLEX(-42, 14) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_440, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_440", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_440() */
/* END TEST 440 */

/*
 *  BEGIN TEST 441
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_441( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-53, -45), MAKE_VAL_COMPLEX(-151, 5), MAKE_VAL_COMPLEX(289, 87), MAKE_VAL_COMPLEX(-36, 141), MAKE_VAL_COMPLEX(39, 0), MAKE_VAL_COMPLEX(51, 96), MAKE_VAL_COMPLEX(98, 42), MAKE_VAL_COMPLEX(60, -73), MAKE_VAL_COMPLEX(4, 17), MAKE_VAL_COMPLEX(-197, -35), MAKE_VAL_COMPLEX(-53, 78), MAKE_VAL_COMPLEX(11, -50), MAKE_VAL_COMPLEX(-63, 42), MAKE_VAL_COMPLEX(-6, 30), MAKE_VAL_COMPLEX(6, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_441, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_441", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_441() */
/* END TEST 441 */

/*
 *  BEGIN TEST 442
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_442( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-9, 3), MAKE_VAL_COMPLEX(3, -3), MAKE_VAL_COMPLEX(3, -15), MAKE_VAL_COMPLEX(27, 30), MAKE_VAL_COMPLEX(6, -9), MAKE_VAL_COMPLEX(30, 3), MAKE_VAL_COMPLEX(15, -15), MAKE_VAL_COMPLEX(-9, -30), MAKE_VAL_COMPLEX(18, -9), MAKE_VAL_COMPLEX(-30, 15), MAKE_VAL_COMPLEX(-24, 12), MAKE_VAL_COMPLEX(0, 15), MAKE_VAL_COMPLEX(0, 21), MAKE_VAL_COMPLEX(6, 6), MAKE_VAL_COMPLEX(12, 6) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_442, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_442", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_442() */
/* END TEST 442 */

/*
 *  BEGIN TEST 443
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_443( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-177, 107), MAKE_VAL_COMPLEX(-325, 823), MAKE_VAL_COMPLEX(985, -1001), MAKE_VAL_COMPLEX(106, -23), MAKE_VAL_COMPLEX(-13, -26), MAKE_VAL_COMPLEX(47, -66), MAKE_VAL_COMPLEX(166, -116), MAKE_VAL_COMPLEX(-36, 69), MAKE_VAL_COMPLEX(-210, 203), MAKE_VAL_COMPLEX(-64, 600), MAKE_VAL_COMPLEX(688, 169), MAKE_VAL_COMPLEX(-253, -420), MAKE_VAL_COMPLEX(49, 28), MAKE_VAL_COMPLEX(22, -6), MAKE_VAL_COMPLEX(30, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-2, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_443, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_443", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_443() */
/* END TEST 443 */

/*
 *  BEGIN TEST 444
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_444( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(15, -35), MAKE_VAL_COMPLEX(20, -10), MAKE_VAL_COMPLEX(-146, 22), MAKE_VAL_COMPLEX(-140, -45), MAKE_VAL_COMPLEX(-100, 0), MAKE_VAL_COMPLEX(80, 0), MAKE_VAL_COMPLEX(26, 70), MAKE_VAL_COMPLEX(-106, -32), MAKE_VAL_COMPLEX(-75, -5), MAKE_VAL_COMPLEX(-5, 35), MAKE_VAL_COMPLEX(-54, -132), MAKE_VAL_COMPLEX(-90, 75) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_444, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_444", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_444() */
/* END TEST 444 */

/*
 *  BEGIN TEST 445
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_445( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, -37), MAKE_VAL_COMPLEX(-59, -55), MAKE_VAL_COMPLEX(-21, -51), MAKE_VAL_COMPLEX(96, 85), MAKE_VAL_COMPLEX(-110, -10), MAKE_VAL_COMPLEX(130, 10), MAKE_VAL_COMPLEX(45, -14), MAKE_VAL_COMPLEX(83, 73), MAKE_VAL_COMPLEX(-82, -13), MAKE_VAL_COMPLEX(121, -40), MAKE_VAL_COMPLEX(51, -26), MAKE_VAL_COMPLEX(2, 5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_445, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_445", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_445() */
/* END TEST 445 */

/*
 *  BEGIN TEST 446
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_446( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(28, -46), MAKE_VAL_COMPLEX(-22, -46), MAKE_VAL_COMPLEX(-110, -30), MAKE_VAL_COMPLEX(-32, 24), MAKE_VAL_COMPLEX(-140, -20), MAKE_VAL_COMPLEX(140, 20), MAKE_VAL_COMPLEX(44, 42), MAKE_VAL_COMPLEX(-18, 26), MAKE_VAL_COMPLEX(-104, -22), MAKE_VAL_COMPLEX(78, 4), MAKE_VAL_COMPLEX(8, -106), MAKE_VAL_COMPLEX(-64, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_446, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_446", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_446() */
/* END TEST 446 */

/*
 *  BEGIN TEST 447
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_447( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(21, 38), MAKE_VAL_COMPLEX(-479, 143), MAKE_VAL_COMPLEX(403, -881), MAKE_VAL_COMPLEX(1596, -407), MAKE_VAL_COMPLEX(70, -90), MAKE_VAL_COMPLEX(110, -70), MAKE_VAL_COMPLEX(-326, -463), MAKE_VAL_COMPLEX(1269, -323), MAKE_VAL_COMPLEX(57, -64), MAKE_VAL_COMPLEX(186, -787), MAKE_VAL_COMPLEX(968, 169), MAKE_VAL_COMPLEX(162, -779) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-9, 8);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_447, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_447", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_447() */
/* END TEST 447 */

/*
 *  BEGIN TEST 448
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_448( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-152, -40), MAKE_VAL_COMPLEX(32, -134), MAKE_VAL_COMPLEX(42, -184), MAKE_VAL_COMPLEX(-94, -24), MAKE_VAL_COMPLEX(0, 26), MAKE_VAL_COMPLEX(-64, 34), MAKE_VAL_COMPLEX(-2, 116), MAKE_VAL_COMPLEX(71, 15), MAKE_VAL_COMPLEX(-83, 61), MAKE_VAL_COMPLEX(12, -110), MAKE_VAL_COMPLEX(1, -79), MAKE_VAL_COMPLEX(-76, -110), MAKE_VAL_COMPLEX(-28, -42), MAKE_VAL_COMPLEX(-20, -4), MAKE_VAL_COMPLEX(-32, 4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_448, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_448", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_448() */
/* END TEST 448 */

/*
 *  BEGIN TEST 449
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_449( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(156, 52), MAKE_VAL_COMPLEX(-36, 130), MAKE_VAL_COMPLEX(-62, 180), MAKE_VAL_COMPLEX(134, -12), MAKE_VAL_COMPLEX(-12, -34), MAKE_VAL_COMPLEX(68, -74), MAKE_VAL_COMPLEX(-18, -136), MAKE_VAL_COMPLEX(-111, -3), MAKE_VAL_COMPLEX(71, -85), MAKE_VAL_COMPLEX(8, 150), MAKE_VAL_COMPLEX(15, 111), MAKE_VAL_COMPLEX(96, 110), MAKE_VAL_COMPLEX(56, 42), MAKE_VAL_COMPLEX(28, -4), MAKE_VAL_COMPLEX(40, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_449, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_449", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_449() */
/* END TEST 449 */

/*
 *  BEGIN TEST 450
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_450( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, -19), MAKE_VAL_COMPLEX(-3, 11), MAKE_VAL_COMPLEX(13, 39), MAKE_VAL_COMPLEX(-103, -34), MAKE_VAL_COMPLEX(-2, 29), MAKE_VAL_COMPLEX(-74, 33), MAKE_VAL_COMPLEX(-15, 55), MAKE_VAL_COMPLEX(61, 58), MAKE_VAL_COMPLEX(-30, 45), MAKE_VAL_COMPLEX(50, -75), MAKE_VAL_COMPLEX(40, -60), MAKE_VAL_COMPLEX(-20, -35), MAKE_VAL_COMPLEX(-28, -49), MAKE_VAL_COMPLEX(-22, -6), MAKE_VAL_COMPLEX(-36, 2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_450, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_450", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_450() */
/* END TEST 450 */

/*
 *  BEGIN TEST 451
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_451( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1005, 311), MAKE_VAL_COMPLEX(-363, 831), MAKE_VAL_COMPLEX(-429, 1313), MAKE_VAL_COMPLEX(-2, -143), MAKE_VAL_COMPLEX(-37, 10), MAKE_VAL_COMPLEX(-73, -78), MAKE_VAL_COMPLEX(-57, -399), MAKE_VAL_COMPLEX(-157, 286), MAKE_VAL_COMPLEX(304, -73), MAKE_VAL_COMPLEX(243, 298), MAKE_VAL_COMPLEX(255, 193), MAKE_VAL_COMPLEX(291, 496), MAKE_VAL_COMPLEX(49, -56), MAKE_VAL_COMPLEX(-2, -30), MAKE_VAL_COMPLEX(-18, -44) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_451, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_451", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_451() */
/* END TEST 451 */

/*
 *  BEGIN TEST 452
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_452( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-2, -82), MAKE_VAL_COMPLEX(-206, 72), MAKE_VAL_COMPLEX(32, -34), MAKE_VAL_COMPLEX(-80, 72), MAKE_VAL_COMPLEX(24, 16), MAKE_VAL_COMPLEX(-8, 80), MAKE_VAL_COMPLEX(102, 64), MAKE_VAL_COMPLEX(57, -51), MAKE_VAL_COMPLEX(13, 109), MAKE_VAL_COMPLEX(-68, -40), MAKE_VAL_COMPLEX(-43, -27), MAKE_VAL_COMPLEX(-96, 80), MAKE_VAL_COMPLEX(-56, 0), MAKE_VAL_COMPLEX(-16, 16), MAKE_VAL_COMPLEX(-16, 32) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_452, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_452", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_452() */
/* END TEST 452 */

/*
 *  BEGIN TEST 453
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_453( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-17, 45), MAKE_VAL_COMPLEX(221, -61), MAKE_VAL_COMPLEX(35, 37), MAKE_VAL_COMPLEX(-32, 65), MAKE_VAL_COMPLEX(19, 4), MAKE_VAL_COMPLEX(15, 52), MAKE_VAL_COMPLEX(-27, -9), MAKE_VAL_COMPLEX(67, -8), MAKE_VAL_COMPLEX(38, -37), MAKE_VAL_COMPLEX(-17, -80), MAKE_VAL_COMPLEX(-25, -69), MAKE_VAL_COMPLEX(31, -70), MAKE_VAL_COMPLEX(-35, 14), MAKE_VAL_COMPLEX(-6, 14), MAKE_VAL_COMPLEX(-2, 24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_453, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_453", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_453() */
/* END TEST 453 */

/*
 *  BEGIN TEST 454
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_454( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(12, -34), MAKE_VAL_COMPLEX(2, 16), MAKE_VAL_COMPLEX(38, 44), MAKE_VAL_COMPLEX(-153, 11), MAKE_VAL_COMPLEX(13, 39), MAKE_VAL_COMPLEX(-79, 83), MAKE_VAL_COMPLEX(10, 80), MAKE_VAL_COMPLEX(111, 43), MAKE_VAL_COMPLEX(-15, 75), MAKE_VAL_COMPLEX(25, -125), MAKE_VAL_COMPLEX(20, -100), MAKE_VAL_COMPLEX(-45, -35), MAKE_VAL_COMPLEX(-63, -49), MAKE_VAL_COMPLEX(-32, 4), MAKE_VAL_COMPLEX(-46, 22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_454, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_454", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_454() */
/* END TEST 454 */

/*
 *  BEGIN TEST 455
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_ROWMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_455( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_ROWMAJ, x432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-60, -558), MAKE_VAL_COMPLEX(-1794, 1014), MAKE_VAL_COMPLEX(-124, -340), MAKE_VAL_COMPLEX(-97, 33), MAKE_VAL_COMPLEX(15, 23), MAKE_VAL_COMPLEX(-37, 67), MAKE_VAL_COMPLEX(626, 142), MAKE_VAL_COMPLEX(-182, -188), MAKE_VAL_COMPLEX(26, 622), MAKE_VAL_COMPLEX(-177, 331), MAKE_VAL_COMPLEX(-29, 287), MAKE_VAL_COMPLEX(-379, 817), MAKE_VAL_COMPLEX(-49, -21), MAKE_VAL_COMPLEX(-20, 8), MAKE_VAL_COMPLEX(-26, 22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(9, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_455, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_455", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_455() */
/* END TEST 455 */

/*
 *  BEGIN TEST 456
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_456( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(43, 35), MAKE_VAL_COMPLEX(50, -16), MAKE_VAL_COMPLEX(46, -108), MAKE_VAL_COMPLEX(20, -16), MAKE_VAL_COMPLEX(-50, -250), MAKE_VAL_COMPLEX(80, 180), MAKE_VAL_COMPLEX(-95, 83), MAKE_VAL_COMPLEX(65, 7), MAKE_VAL_COMPLEX(-6, -66), MAKE_VAL_COMPLEX(-52, 239), MAKE_VAL_COMPLEX(179, 62), MAKE_VAL_COMPLEX(-46, -38) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_456, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_456", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_456() */
/* END TEST 456 */

/*
 *  BEGIN TEST 457
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_457( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-38, 21), MAKE_VAL_COMPLEX(-3, 41), MAKE_VAL_COMPLEX(65, 65), MAKE_VAL_COMPLEX(-8, -24), MAKE_VAL_COMPLEX(180, 180), MAKE_VAL_COMPLEX(-210, -110), MAKE_VAL_COMPLEX(32, -91), MAKE_VAL_COMPLEX(-66, -40), MAKE_VAL_COMPLEX(107, 20), MAKE_VAL_COMPLEX(-16, -194), MAKE_VAL_COMPLEX(-120, 32), MAKE_VAL_COMPLEX(70, -42) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_457, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_457", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_457() */
/* END TEST 457 */

/*
 *  BEGIN TEST 458
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_458( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(25, -39), MAKE_VAL_COMPLEX(-18, -40), MAKE_VAL_COMPLEX(-94, -28), MAKE_VAL_COMPLEX(-28, 20), MAKE_VAL_COMPLEX(-120, -20), MAKE_VAL_COMPLEX(120, 20), MAKE_VAL_COMPLEX(37, 37), MAKE_VAL_COMPLEX(-16, 22), MAKE_VAL_COMPLEX(-89, -21), MAKE_VAL_COMPLEX(67, 5), MAKE_VAL_COMPLEX(9, -91), MAKE_VAL_COMPLEX(-56, 40) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_458, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_458", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_458() */
/* END TEST 458 */

/*
 *  BEGIN TEST 459
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_459( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-2, 53), MAKE_VAL_COMPLEX(41, 29), MAKE_VAL_COMPLEX(109, -27), MAKE_VAL_COMPLEX(-184, -236), MAKE_VAL_COMPLEX(450, 790), MAKE_VAL_COMPLEX(-770, -430), MAKE_VAL_COMPLEX(306, -187), MAKE_VAL_COMPLEX(-473, -383), MAKE_VAL_COMPLEX(432, -167), MAKE_VAL_COMPLEX(473, -852), MAKE_VAL_COMPLEX(-310, -283), MAKE_VAL_COMPLEX(136, -264) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_459, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_459", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_459() */
/* END TEST 459 */

/*
 *  BEGIN TEST 460
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_460( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(85, 119), MAKE_VAL_COMPLEX(-9, -7), MAKE_VAL_COMPLEX(-28, -36), MAKE_VAL_COMPLEX(63, 13), MAKE_VAL_COMPLEX(-26, -13), MAKE_VAL_COMPLEX(-28, 141), MAKE_VAL_COMPLEX(-45, -35), MAKE_VAL_COMPLEX(-59, 38), MAKE_VAL_COMPLEX(99, -31), MAKE_VAL_COMPLEX(50, 75), MAKE_VAL_COMPLEX(-62, -192), MAKE_VAL_COMPLEX(40, -5), MAKE_VAL_COMPLEX(67, 54), MAKE_VAL_COMPLEX(-42, -98), MAKE_VAL_COMPLEX(12, -34) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_460, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_460", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_460() */
/* END TEST 460 */

/*
 *  BEGIN TEST 461
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_461( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-78, -78), MAKE_VAL_COMPLEX(-2, -8), MAKE_VAL_COMPLEX(-35, 13), MAKE_VAL_COMPLEX(85, -110), MAKE_VAL_COMPLEX(-9, -19), MAKE_VAL_COMPLEX(61, -269), MAKE_VAL_COMPLEX(-10, -40), MAKE_VAL_COMPLEX(-77, -19), MAKE_VAL_COMPLEX(-126, -53), MAKE_VAL_COMPLEX(-5, 65), MAKE_VAL_COMPLEX(98, 304), MAKE_VAL_COMPLEX(25, 15), MAKE_VAL_COMPLEX(24, -40), MAKE_VAL_COMPLEX(72, 76), MAKE_VAL_COMPLEX(22, -14) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_461, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_461", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_461() */
/* END TEST 461 */

/*
 *  BEGIN TEST 462
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_462( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(5, -25), MAKE_VAL_COMPLEX(3, 11), MAKE_VAL_COMPLEX(31, 27), MAKE_VAL_COMPLEX(-106, 23), MAKE_VAL_COMPLEX(13, 26), MAKE_VAL_COMPLEX(-47, 66), MAKE_VAL_COMPLEX(15, 55), MAKE_VAL_COMPLEX(82, 19), MAKE_VAL_COMPLEX(-3, 54), MAKE_VAL_COMPLEX(5, -90), MAKE_VAL_COMPLEX(4, -72), MAKE_VAL_COMPLEX(-35, -20), MAKE_VAL_COMPLEX(-49, -28), MAKE_VAL_COMPLEX(-22, 6), MAKE_VAL_COMPLEX(-30, 20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_462, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_462", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_462() */
/* END TEST 462 */

/*
 *  BEGIN TEST 463
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_463( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-906, -192), MAKE_VAL_COMPLEX(-10, -2), MAKE_VAL_COMPLEX(53, 277), MAKE_VAL_COMPLEX(-403, -704), MAKE_VAL_COMPLEX(-26, 0), MAKE_VAL_COMPLEX(-988, -1526), MAKE_VAL_COMPLEX(-50, -10), MAKE_VAL_COMPLEX(-176, 124), MAKE_VAL_COMPLEX(-886, 537), MAKE_VAL_COMPLEX(70, 40), MAKE_VAL_COMPLEX(1928, 1034), MAKE_VAL_COMPLEX(30, -20), MAKE_VAL_COMPLEX(-329, -339), MAKE_VAL_COMPLEX(740, 180), MAKE_VAL_COMPLEX(-4, -32) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_463, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_463", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_463() */
/* END TEST 463 */

/*
 *  BEGIN TEST 464
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_464( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(32, 22), MAKE_VAL_COMPLEX(-2, 2), MAKE_VAL_COMPLEX(-25, -17), MAKE_VAL_COMPLEX(74, -45), MAKE_VAL_COMPLEX(-4, 6), MAKE_VAL_COMPLEX(146, -4), MAKE_VAL_COMPLEX(-10, 10), MAKE_VAL_COMPLEX(18, 6), MAKE_VAL_COMPLEX(-43, -120), MAKE_VAL_COMPLEX(20, -10), MAKE_VAL_COMPLEX(-222, -116), MAKE_VAL_COMPLEX(0, -10), MAKE_VAL_COMPLEX(59, 5), MAKE_VAL_COMPLEX(-60, 76), MAKE_VAL_COMPLEX(-8, -4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_464, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_464", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_464() */
/* END TEST 464 */

/*
 *  BEGIN TEST 465
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_465( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-13, -55), MAKE_VAL_COMPLEX(1, 15), MAKE_VAL_COMPLEX(56, 70), MAKE_VAL_COMPLEX(-235, 27), MAKE_VAL_COMPLEX(10, 37), MAKE_VAL_COMPLEX(-244, 75), MAKE_VAL_COMPLEX(5, 75), MAKE_VAL_COMPLEX(89, 60), MAKE_VAL_COMPLEX(13, 195), MAKE_VAL_COMPLEX(30, -115), MAKE_VAL_COMPLEX(262, 16), MAKE_VAL_COMPLEX(-40, -35), MAKE_VAL_COMPLEX(-115, -68), MAKE_VAL_COMPLEX(26, -78), MAKE_VAL_COMPLEX(-44, 18) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_465, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_465", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_465() */
/* END TEST 465 */

/*
 *  BEGIN TEST 466
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_466( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(17, 1), MAKE_VAL_COMPLEX(-7, 3), MAKE_VAL_COMPLEX(-15, 23), MAKE_VAL_COMPLEX(-25, -68), MAKE_VAL_COMPLEX(-16, 11), MAKE_VAL_COMPLEX(-48, -25), MAKE_VAL_COMPLEX(-35, 15), MAKE_VAL_COMPLEX(-5, 56), MAKE_VAL_COMPLEX(-36, 3), MAKE_VAL_COMPLEX(60, -5), MAKE_VAL_COMPLEX(48, -4), MAKE_VAL_COMPLEX(10, -25), MAKE_VAL_COMPLEX(14, -35), MAKE_VAL_COMPLEX(-6, -14), MAKE_VAL_COMPLEX(-16, -18) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_466, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_466", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_466() */
/* END TEST 466 */

/*
 *  BEGIN TEST 467
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_467( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_ROWMAJ, y432_VW );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(10, -98), MAKE_VAL_COMPLEX(-6, 2), MAKE_VAL_COMPLEX(-22, 118), MAKE_VAL_COMPLEX(-250, -192), MAKE_VAL_COMPLEX(-14, 8), MAKE_VAL_COMPLEX(-374, -352), MAKE_VAL_COMPLEX(-30, 10), MAKE_VAL_COMPLEX(-60, 50), MAKE_VAL_COMPLEX(-220, 314), MAKE_VAL_COMPLEX(50, 0), MAKE_VAL_COMPLEX(300, 692), MAKE_VAL_COMPLEX(10, -20), MAKE_VAL_COMPLEX(-66, -184), MAKE_VAL_COMPLEX(268, -60), MAKE_VAL_COMPLEX(-12, -16) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-2, -2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_467, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_467", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_467() */
/* END TEST 467 */

/*
 *  BEGIN TEST 468
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_468( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, -8), MAKE_VAL_COMPLEX(44, 0), MAKE_VAL_COMPLEX(-74, 8), MAKE_VAL_COMPLEX(-53, 160), MAKE_VAL_COMPLEX(-40, -40), MAKE_VAL_COMPLEX(-50, -70), MAKE_VAL_COMPLEX(93, 53), MAKE_VAL_COMPLEX(60, 43), MAKE_VAL_COMPLEX(-28, -32), MAKE_VAL_COMPLEX(144, 60), MAKE_VAL_COMPLEX(-15, 5), MAKE_VAL_COMPLEX(-38, 34) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_468, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_468", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_468() */
/* END TEST 468 */

/*
 *  BEGIN TEST 469
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_469( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(29, 0), MAKE_VAL_COMPLEX(-25, -43), MAKE_VAL_COMPLEX(29, -103), MAKE_VAL_COMPLEX(17, -168), MAKE_VAL_COMPLEX(-30, -70), MAKE_VAL_COMPLEX(120, 180), MAKE_VAL_COMPLEX(-96, 3), MAKE_VAL_COMPLEX(-89, -40), MAKE_VAL_COMPLEX(-19, -54), MAKE_VAL_COMPLEX(-100, -3), MAKE_VAL_COMPLEX(94, -63), MAKE_VAL_COMPLEX(-34, -50) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_469, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_469", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_469() */
/* END TEST 469 */

/*
 *  BEGIN TEST 470
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_470( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, -22), MAKE_VAL_COMPLEX(-34, 14), MAKE_VAL_COMPLEX(-26, 78), MAKE_VAL_COMPLEX(16, 24), MAKE_VAL_COMPLEX(-20, 100), MAKE_VAL_COMPLEX(20, -100), MAKE_VAL_COMPLEX(32, -30), MAKE_VAL_COMPLEX(18, 14), MAKE_VAL_COMPLEX(-20, 74), MAKE_VAL_COMPLEX(6, -56), MAKE_VAL_COMPLEX(-76, -10), MAKE_VAL_COMPLEX(32, 48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_470, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_470", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_470() */
/* END TEST 470 */

/*
 *  BEGIN TEST 471
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_471( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	oski_value_t* y_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(10, 25), MAKE_VAL_COMPLEX(85, 285), MAKE_VAL_COMPLEX(-359, -89), MAKE_VAL_COMPLEX(-948, 435), MAKE_VAL_COMPLEX(50, -50), MAKE_VAL_COMPLEX(140, -840), MAKE_VAL_COMPLEX(181, 566), MAKE_VAL_COMPLEX(88, 501), MAKE_VAL_COMPLEX(40, -35), MAKE_VAL_COMPLEX(255, 790), MAKE_VAL_COMPLEX(-317, -32), MAKE_VAL_COMPLEX(-194, 66) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(4, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_471, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_471", y_true, y_val, y432_LEN*y432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_471() */
/* END TEST 471 */

/*
 *  BEGIN TEST 472
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_472( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-143, -23), MAKE_VAL_COMPLEX(-9, 5), MAKE_VAL_COMPLEX(-9, 99), MAKE_VAL_COMPLEX(-71, -128), MAKE_VAL_COMPLEX(-20, 17), MAKE_VAL_COMPLEX(-34, -171), MAKE_VAL_COMPLEX(-45, 25), MAKE_VAL_COMPLEX(14, 43), MAKE_VAL_COMPLEX(-79, -14), MAKE_VAL_COMPLEX(80, -15), MAKE_VAL_COMPLEX(92, -230), MAKE_VAL_COMPLEX(10, -35), MAKE_VAL_COMPLEX(-45, -30), MAKE_VAL_COMPLEX(-66, -98), MAKE_VAL_COMPLEX(-24, -22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_472, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_472", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_472() */
/* END TEST 472 */

/*
 *  BEGIN TEST 473
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_473( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(189, 21), MAKE_VAL_COMPLEX(-9, 5), MAKE_VAL_COMPLEX(-25, -33), MAKE_VAL_COMPLEX(-15, -48), MAKE_VAL_COMPLEX(-20, 17), MAKE_VAL_COMPLEX(-102, 117), MAKE_VAL_COMPLEX(-45, 25), MAKE_VAL_COMPLEX(-12, 109), MAKE_VAL_COMPLEX(-17, 32), MAKE_VAL_COMPLEX(80, -15), MAKE_VAL_COMPLEX(36, 206), MAKE_VAL_COMPLEX(10, -35), MAKE_VAL_COMPLEX(73, -68), MAKE_VAL_COMPLEX(46, 62), MAKE_VAL_COMPLEX(-24, -22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_473, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_473", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_473() */
/* END TEST 473 */

/*
 *  BEGIN TEST 474
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_474( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(24, 12), MAKE_VAL_COMPLEX(-12, 0), MAKE_VAL_COMPLEX(-36, 24), MAKE_VAL_COMPLEX(6, -114), MAKE_VAL_COMPLEX(-30, 6), MAKE_VAL_COMPLEX(-54, -66), MAKE_VAL_COMPLEX(-60, 0), MAKE_VAL_COMPLEX(-42, 78), MAKE_VAL_COMPLEX(-54, -18), MAKE_VAL_COMPLEX(90, 30), MAKE_VAL_COMPLEX(72, 24), MAKE_VAL_COMPLEX(30, -30), MAKE_VAL_COMPLEX(42, -42), MAKE_VAL_COMPLEX(0, -24), MAKE_VAL_COMPLEX(-12, -36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_474, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_474", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_474() */
/* END TEST 474 */

/*
 *  BEGIN TEST 475
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_475( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1152, -1196), MAKE_VAL_COMPLEX(8, -4), MAKE_VAL_COMPLEX(-560, -360), MAKE_VAL_COMPLEX(522, 94), MAKE_VAL_COMPLEX(18, -14), MAKE_VAL_COMPLEX(1006, 1162), MAKE_VAL_COMPLEX(40, -20), MAKE_VAL_COMPLEX(188, 236), MAKE_VAL_COMPLEX(412, -116), MAKE_VAL_COMPLEX(-70, 10), MAKE_VAL_COMPLEX(1520, 1540), MAKE_VAL_COMPLEX(-10, 30), MAKE_VAL_COMPLEX(188, -544), MAKE_VAL_COMPLEX(984, 48), MAKE_VAL_COMPLEX(20, 20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 8);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_475, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_475", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_475() */
/* END TEST 475 */

/*
 *  BEGIN TEST 476
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_476( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(4, -84), MAKE_VAL_COMPLEX(6, 10), MAKE_VAL_COMPLEX(100, 42), MAKE_VAL_COMPLEX(-126, 92), MAKE_VAL_COMPLEX(20, 22), MAKE_VAL_COMPLEX(-242, 142), MAKE_VAL_COMPLEX(30, 50), MAKE_VAL_COMPLEX(63, -31), MAKE_VAL_COMPLEX(1, 91), MAKE_VAL_COMPLEX(-20, -90), MAKE_VAL_COMPLEX(-24, -114), MAKE_VAL_COMPLEX(-40, -10), MAKE_VAL_COMPLEX(-67, 47), MAKE_VAL_COMPLEX(-76, 92), MAKE_VAL_COMPLEX(-24, 28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_476, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_476", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_476() */
/* END TEST 476 */

/*
 *  BEGIN TEST 477
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_477( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-14, 84), MAKE_VAL_COMPLEX(-2, -12), MAKE_VAL_COMPLEX(-92, -56), MAKE_VAL_COMPLEX(143, -53), MAKE_VAL_COMPLEX(-11, -29), MAKE_VAL_COMPLEX(271, -129), MAKE_VAL_COMPLEX(-10, -60), MAKE_VAL_COMPLEX(-62, -2), MAKE_VAL_COMPLEX(20, -94), MAKE_VAL_COMPLEX(-15, 95), MAKE_VAL_COMPLEX(-4, 118), MAKE_VAL_COMPLEX(35, 25), MAKE_VAL_COMPLEX(60, -26), MAKE_VAL_COMPLEX(80, -84), MAKE_VAL_COMPLEX(34, -18) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_477, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_477", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_477() */
/* END TEST 477 */

/*
 *  BEGIN TEST 478
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_478( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-8, -14), MAKE_VAL_COMPLEX(6, 4), MAKE_VAL_COMPLEX(26, 0), MAKE_VAL_COMPLEX(-41, 55), MAKE_VAL_COMPLEX(17, 7), MAKE_VAL_COMPLEX(5, 51), MAKE_VAL_COMPLEX(30, 20), MAKE_VAL_COMPLEX(47, -25), MAKE_VAL_COMPLEX(21, 27), MAKE_VAL_COMPLEX(-35, -45), MAKE_VAL_COMPLEX(-28, -36), MAKE_VAL_COMPLEX(-25, 5), MAKE_VAL_COMPLEX(-35, 7), MAKE_VAL_COMPLEX(-8, 12), MAKE_VAL_COMPLEX(-6, 22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_478, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_478", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_478() */
/* END TEST 478 */

/*
 *  BEGIN TEST 479
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED>
 *  x : <'x432', LAYOUT_COLMAJ>
 *  y : <'y432', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_479( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A432_ptr, A432_ind, A432_val,
		A432_M, A432_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_DIAG_EXPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y432_val,
		y432_LEN * y432_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y432_LEN, y432_VW, LAYOUT_COLMAJ, y432_LEN );
	oski_value_t* y_val = array_Duplicate( x432_val,
		x432_LEN * x432_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x432_LEN, x432_VW, LAYOUT_COLMAJ, x432_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(117, -21), MAKE_VAL_COMPLEX(-3, -11), MAKE_VAL_COMPLEX(-17, 121), MAKE_VAL_COMPLEX(-2, -39), MAKE_VAL_COMPLEX(-13, -26), MAKE_VAL_COMPLEX(-295, -430), MAKE_VAL_COMPLEX(-15, -55), MAKE_VAL_COMPLEX(-51, -92), MAKE_VAL_COMPLEX(-82, -39), MAKE_VAL_COMPLEX(-5, 90), MAKE_VAL_COMPLEX(72, 14), MAKE_VAL_COMPLEX(35, 20), MAKE_VAL_COMPLEX(-84, 67), MAKE_VAL_COMPLEX(-194, -38), MAKE_VAL_COMPLEX(30, -20) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_479, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_479", y_true, y_val, x432_LEN*x432_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_479() */
/* END TEST 479 */

#define A480_M 4    /* # of rows */
#define A480_N 5    /* # of cols */
#define A480_NNZ 0  /* # of stored non-zeros */
static oski_index_t A480_ptr[A480_N+1] = {1, 1, 1, 1, 1, 1}; /* end row pointers for A480 */
static oski_index_t* A480_ind = NULL;
static oski_value_t* A480_val = NULL;

#define x480_LEN 5   /* vector length */
#define x480_VW  3   /* vector width */
static oski_value_t x480_val[x480_LEN*x480_VW] = {MAKE_VAL_COMPLEX(6, 10), MAKE_VAL_COMPLEX(8, -8), MAKE_VAL_COMPLEX(0, 10), MAKE_VAL_COMPLEX(8, 7), MAKE_VAL_COMPLEX(-5, 9), MAKE_VAL_COMPLEX(10, 4), MAKE_VAL_COMPLEX(-9, 2), MAKE_VAL_COMPLEX(-1, -5), MAKE_VAL_COMPLEX(-8, 3), MAKE_VAL_COMPLEX(6, 3), MAKE_VAL_COMPLEX(-6, 1), MAKE_VAL_COMPLEX(-10, 4), MAKE_VAL_COMPLEX(2, -3), MAKE_VAL_COMPLEX(-5, -10), MAKE_VAL_COMPLEX(-1, -10)}; /* data */

#define y480_LEN 4   /* vector length */
#define y480_VW  3   /* vector width */
static oski_value_t y480_val[y480_LEN*y480_VW] = {MAKE_VAL_COMPLEX(-2, -4), MAKE_VAL_COMPLEX(3, -10), MAKE_VAL_COMPLEX(-8, -5), MAKE_VAL_COMPLEX(8, 4), MAKE_VAL_COMPLEX(-10, 10), MAKE_VAL_COMPLEX(-6, -3), MAKE_VAL_COMPLEX(-3, -8), MAKE_VAL_COMPLEX(5, -10), MAKE_VAL_COMPLEX(10, 2), MAKE_VAL_COMPLEX(-3, 10), MAKE_VAL_COMPLEX(7, -5), MAKE_VAL_COMPLEX(-9, -10)}; /* data */

/*
 *  BEGIN TEST 480
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_480( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-4, 30), MAKE_VAL_COMPLEX(-41, 10), MAKE_VAL_COMPLEX(4, 57), MAKE_VAL_COMPLEX(0, -37), MAKE_VAL_COMPLEX(65, 19), MAKE_VAL_COMPLEX(16, 37), MAKE_VAL_COMPLEX(-32, 38), MAKE_VAL_COMPLEX(-56, 5), MAKE_VAL_COMPLEX(-30, -43), MAKE_VAL_COMPLEX(55, -15), MAKE_VAL_COMPLEX(-47, -12), MAKE_VAL_COMPLEX(-23, 70) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_480, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_480", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_480() */
/* END TEST 480 */

/*
 *  BEGIN TEST 481
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_481( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-38, 6), MAKE_VAL_COMPLEX(-88, -16), MAKE_VAL_COMPLEX(-40, 54), MAKE_VAL_COMPLEX(24, -71), MAKE_VAL_COMPLEX(85, 71), MAKE_VAL_COMPLEX(-34, 44), MAKE_VAL_COMPLEX(-55, 22), MAKE_VAL_COMPLEX(-79, -35), MAKE_VAL_COMPLEX(24, -83), MAKE_VAL_COMPLEX(74, 21), MAKE_VAL_COMPLEX(-34, -57), MAKE_VAL_COMPLEX(-70, 68) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_481, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_481", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_481() */
/* END TEST 481 */

/*
 *  BEGIN TEST 482
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_482( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(52, 14), MAKE_VAL_COMPLEX(66, 107), MAKE_VAL_COMPLEX(109, -32), MAKE_VAL_COMPLEX(-100, 40), MAKE_VAL_COMPLEX(-10, -170), MAKE_VAL_COMPLEX(75, -30), MAKE_VAL_COMPLEX(96, 37), MAKE_VAL_COMPLEX(50, 125), MAKE_VAL_COMPLEX(-98, 74), MAKE_VAL_COMPLEX(-66, -107), MAKE_VAL_COMPLEX(-11, 103), MAKE_VAL_COMPLEX(162, -1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_482, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_482", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_482() */
/* END TEST 482 */

/*
 *  BEGIN TEST 483
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_483( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(14, 60), MAKE_VAL_COMPLEX(142, -111), MAKE_VAL_COMPLEX(-79, 16), MAKE_VAL_COMPLEX(95, 136), MAKE_VAL_COMPLEX(-177, 107), MAKE_VAL_COMPLEX(29, 20), MAKE_VAL_COMPLEX(-78, -84), MAKE_VAL_COMPLEX(77, -108), MAKE_VAL_COMPLEX(1, 46), MAKE_VAL_COMPLEX(-15, 113), MAKE_VAL_COMPLEX(20, -29), MAKE_VAL_COMPLEX(-134, -105) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(8, 3);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_483, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_483", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_483() */
/* END TEST 483 */

/*
 *  BEGIN TEST 484
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_484( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(72, 6), MAKE_VAL_COMPLEX(-5, -82), MAKE_VAL_COMPLEX(42, 35), MAKE_VAL_COMPLEX(75, -8), MAKE_VAL_COMPLEX(15, 71), MAKE_VAL_COMPLEX(54, -37), MAKE_VAL_COMPLEX(-29, 45), MAKE_VAL_COMPLEX(-24, -25), MAKE_VAL_COMPLEX(-7, 54), MAKE_VAL_COMPLEX(36, -8), MAKE_VAL_COMPLEX(-12, 29), MAKE_VAL_COMPLEX(-29, 56), MAKE_VAL_COMPLEX(-7, -22), MAKE_VAL_COMPLEX(-70, -15), MAKE_VAL_COMPLEX(-54, -35) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_484, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_484", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_484() */
/* END TEST 484 */

/*
 *  BEGIN TEST 485
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_485( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(12, -2), MAKE_VAL_COMPLEX(-11, 2), MAKE_VAL_COMPLEX(18, 5), MAKE_VAL_COMPLEX(-1, -12), MAKE_VAL_COMPLEX(19, -5), MAKE_VAL_COMPLEX(10, -7), MAKE_VAL_COMPLEX(5, 17), MAKE_VAL_COMPLEX(-10, 11), MAKE_VAL_COMPLEX(-7, 6), MAKE_VAL_COMPLEX(6, -16), MAKE_VAL_COMPLEX(-6, 11), MAKE_VAL_COMPLEX(13, 20), MAKE_VAL_COMPLEX(-3, -2), MAKE_VAL_COMPLEX(-10, 5), MAKE_VAL_COMPLEX(-10, 1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_485, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_485", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_485() */
/* END TEST 485 */

/*
 *  BEGIN TEST 486
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_486( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, 56), MAKE_VAL_COMPLEX(48, -32), MAKE_VAL_COMPLEX(-10, 50), MAKE_VAL_COMPLEX(33, 43), MAKE_VAL_COMPLEX(-34, 40), MAKE_VAL_COMPLEX(46, 30), MAKE_VAL_COMPLEX(-47, 1), MAKE_VAL_COMPLEX(0, -26), MAKE_VAL_COMPLEX(-43, 7), MAKE_VAL_COMPLEX(27, 21), MAKE_VAL_COMPLEX(-31, -1), MAKE_VAL_COMPLEX(-54, 10), MAKE_VAL_COMPLEX(13, -13), MAKE_VAL_COMPLEX(-15, -55), MAKE_VAL_COMPLEX(5, -51) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_486, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_486", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_486() */
/* END TEST 486 */

/*
 *  BEGIN TEST 487
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_487( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(2, -86), MAKE_VAL_COMPLEX(-18, -46), MAKE_VAL_COMPLEX(-8, -86), MAKE_VAL_COMPLEX(29, -12), MAKE_VAL_COMPLEX(-33, 19), MAKE_VAL_COMPLEX(-58, -76), MAKE_VAL_COMPLEX(40, -35), MAKE_VAL_COMPLEX(39, -27), MAKE_VAL_COMPLEX(97, 44), MAKE_VAL_COMPLEX(-53, 24), MAKE_VAL_COMPLEX(79, -2), MAKE_VAL_COMPLEX(18, -64), MAKE_VAL_COMPLEX(-17, 6), MAKE_VAL_COMPLEX(-10, 55), MAKE_VAL_COMPLEX(-26, 43) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, 2);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_487, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_487", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_487() */
/* END TEST 487 */

/*
 *  BEGIN TEST 488
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_488( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(18, -16), MAKE_VAL_COMPLEX(-13, -26), MAKE_VAL_COMPLEX(12, -5), MAKE_VAL_COMPLEX(22, -12), MAKE_VAL_COMPLEX(8, 20), MAKE_VAL_COMPLEX(2, -23), MAKE_VAL_COMPLEX(1, 10), MAKE_VAL_COMPLEX(-5, -8), MAKE_VAL_COMPLEX(16, 18), MAKE_VAL_COMPLEX(3, -2), MAKE_VAL_COMPLEX(9, 7), MAKE_VAL_COMPLEX(-1, 10), MAKE_VAL_COMPLEX(-6, -4), MAKE_VAL_COMPLEX(-20, 10), MAKE_VAL_COMPLEX(-20, 2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_488, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_488", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_488() */
/* END TEST 488 */

/*
 *  BEGIN TEST 489
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_489( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(22, -76), MAKE_VAL_COMPLEX(-83, 10), MAKE_VAL_COMPLEX(58, -45), MAKE_VAL_COMPLEX(-13, -79), MAKE_VAL_COMPLEX(80, -30), MAKE_VAL_COMPLEX(-24, -67), MAKE_VAL_COMPLEX(58, 43), MAKE_VAL_COMPLEX(-25, 40), MAKE_VAL_COMPLEX(45, 23), MAKE_VAL_COMPLEX(-12, -55), MAKE_VAL_COMPLEX(28, 30), MAKE_VAL_COMPLEX(79, 40), MAKE_VAL_COMPLEX(-25, 5), MAKE_VAL_COMPLEX(-25, 75), MAKE_VAL_COMPLEX(-45, 55) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_489, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_489", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_489() */
/* END TEST 489 */

/*
 *  BEGIN TEST 490
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_490( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-58, -6), MAKE_VAL_COMPLEX(8, 56), MAKE_VAL_COMPLEX(-40, -30), MAKE_VAL_COMPLEX(-52, 11), MAKE_VAL_COMPLEX(-21, -47), MAKE_VAL_COMPLEX(-46, 28), MAKE_VAL_COMPLEX(19, -42), MAKE_VAL_COMPLEX(23, 11), MAKE_VAL_COMPLEX(12, -41), MAKE_VAL_COMPLEX(-30, 15), MAKE_VAL_COMPLEX(14, -27), MAKE_VAL_COMPLEX(14, -52), MAKE_VAL_COMPLEX(6, 17), MAKE_VAL_COMPLEX(55, 10), MAKE_VAL_COMPLEX(43, 26) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_490, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_490", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_490() */
/* END TEST 490 */

/*
 *  BEGIN TEST 491
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_491( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-88, 10), MAKE_VAL_COMPLEX(-110, 139), MAKE_VAL_COMPLEX(-11, 36), MAKE_VAL_COMPLEX(-112, -98), MAKE_VAL_COMPLEX(144, -84), MAKE_VAL_COMPLEX(-49, 82), MAKE_VAL_COMPLEX(14, 37), MAKE_VAL_COMPLEX(-84, 71), MAKE_VAL_COMPLEX(-30, -136), MAKE_VAL_COMPLEX(46, -53), MAKE_VAL_COMPLEX(-59, -39), MAKE_VAL_COMPLEX(46, 79), MAKE_VAL_COMPLEX(0, 26), MAKE_VAL_COMPLEX(70, 40), MAKE_VAL_COMPLEX(46, 56) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_491, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_491", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_491() */
/* END TEST 491 */

/*
 *  BEGIN TEST 492
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_492( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(2, 12), MAKE_VAL_COMPLEX(-2, 4), MAKE_VAL_COMPLEX(-14, 10), MAKE_VAL_COMPLEX(10, -5), MAKE_VAL_COMPLEX(18, 2), MAKE_VAL_COMPLEX(-8, 15), MAKE_VAL_COMPLEX(-9, -2), MAKE_VAL_COMPLEX(-16, -4), MAKE_VAL_COMPLEX(2, 0), MAKE_VAL_COMPLEX(20, 7), MAKE_VAL_COMPLEX(-13, -4), MAKE_VAL_COMPLEX(-20, 13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_492, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_492", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_492() */
/* END TEST 492 */

/*
 *  BEGIN TEST 493
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_493( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(36, 4), MAKE_VAL_COMPLEX(41, 84), MAKE_VAL_COMPLEX(100, -23), MAKE_VAL_COMPLEX(-90, 25), MAKE_VAL_COMPLEX(-8, -132), MAKE_VAL_COMPLEX(68, -30), MAKE_VAL_COMPLEX(78, 40), MAKE_VAL_COMPLEX(41, 104), MAKE_VAL_COMPLEX(-84, 46), MAKE_VAL_COMPLEX(-59, -95), MAKE_VAL_COMPLEX(-6, 81), MAKE_VAL_COMPLEX(143, 3) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_493, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_493", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_493() */
/* END TEST 493 */

/*
 *  BEGIN TEST 494
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_494( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-36, 18), MAKE_VAL_COMPLEX(-90, -27), MAKE_VAL_COMPLEX(-45, 72), MAKE_VAL_COMPLEX(36, -72), MAKE_VAL_COMPLEX(90, 90), MAKE_VAL_COMPLEX(-27, 54), MAKE_VAL_COMPLEX(-72, 27), MAKE_VAL_COMPLEX(-90, -45), MAKE_VAL_COMPLEX(18, -90), MAKE_VAL_COMPLEX(90, 27), MAKE_VAL_COMPLEX(-45, -63), MAKE_VAL_COMPLEX(-90, 81) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_494, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_494", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_494() */
/* END TEST 494 */

/*
 *  BEGIN TEST 495
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_495( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-98, 32), MAKE_VAL_COMPLEX(-61, 8), MAKE_VAL_COMPLEX(-66, -81), MAKE_VAL_COMPLEX(24, 52), MAKE_VAL_COMPLEX(34, 134), MAKE_VAL_COMPLEX(-108, -43), MAKE_VAL_COMPLEX(9, -42), MAKE_VAL_COMPLEX(-3, -108), MAKE_VAL_COMPLEX(-26, -10), MAKE_VAL_COMPLEX(-27, 136), MAKE_VAL_COMPLEX(1, -103), MAKE_VAL_COMPLEX(-97, -112) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 8);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_495, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_495", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_495() */
/* END TEST 495 */

/*
 *  BEGIN TEST 496
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_496( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-42, -48), MAKE_VAL_COMPLEX(-42, 58), MAKE_VAL_COMPLEX(0, -48), MAKE_VAL_COMPLEX(-44, -37), MAKE_VAL_COMPLEX(10, -53), MAKE_VAL_COMPLEX(-57, 0), MAKE_VAL_COMPLEX(35, -24), MAKE_VAL_COMPLEX(7, 16), MAKE_VAL_COMPLEX(44, -28), MAKE_VAL_COMPLEX(-25, -5), MAKE_VAL_COMPLEX(34, -21), MAKE_VAL_COMPLEX(37, -40), MAKE_VAL_COMPLEX(-7, 17), MAKE_VAL_COMPLEX(35, 45), MAKE_VAL_COMPLEX(15, 49) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_496, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_496", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_496() */
/* END TEST 496 */

/*
 *  BEGIN TEST 497
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_497( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, -30), MAKE_VAL_COMPLEX(-14, 30), MAKE_VAL_COMPLEX(-20, -42), MAKE_VAL_COMPLEX(-42, -10), MAKE_VAL_COMPLEX(17, -38), MAKE_VAL_COMPLEX(-41, -16), MAKE_VAL_COMPLEX(42, -12), MAKE_VAL_COMPLEX(12, 27), MAKE_VAL_COMPLEX(22, -15), MAKE_VAL_COMPLEX(-35, -10), MAKE_VAL_COMPLEX(18, 0), MAKE_VAL_COMPLEX(45, -16), MAKE_VAL_COMPLEX(-5, 14), MAKE_VAL_COMPLEX(30, 35), MAKE_VAL_COMPLEX(14, 39) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_497, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_497", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_497() */
/* END TEST 497 */

/*
 *  BEGIN TEST 498
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_498( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(2, 26), MAKE_VAL_COMPLEX(24, -8), MAKE_VAL_COMPLEX(-10, 20), MAKE_VAL_COMPLEX(9, 22), MAKE_VAL_COMPLEX(-19, 13), MAKE_VAL_COMPLEX(16, 18), MAKE_VAL_COMPLEX(-20, -5), MAKE_VAL_COMPLEX(3, -11), MAKE_VAL_COMPLEX(-19, -2), MAKE_VAL_COMPLEX(9, 12), MAKE_VAL_COMPLEX(-13, -4), MAKE_VAL_COMPLEX(-24, -2), MAKE_VAL_COMPLEX(7, -4), MAKE_VAL_COMPLEX(0, -25), MAKE_VAL_COMPLEX(8, -21) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_498, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_498", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_498() */
/* END TEST 498 */

/*
 *  BEGIN TEST 499
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_499( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-46, -94), MAKE_VAL_COMPLEX(-160, 96), MAKE_VAL_COMPLEX(62, -4), MAKE_VAL_COMPLEX(9, -102), MAKE_VAL_COMPLEX(-15, -107), MAKE_VAL_COMPLEX(-128, 54), MAKE_VAL_COMPLEX(8, -93), MAKE_VAL_COMPLEX(19, -47), MAKE_VAL_COMPLEX(121, -38), MAKE_VAL_COMPLEX(9, 52), MAKE_VAL_COMPLEX(115, -72), MAKE_VAL_COMPLEX(24, -150), MAKE_VAL_COMPLEX(-9, 20), MAKE_VAL_COMPLEX(40, 55), MAKE_VAL_COMPLEX(16, 59) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(8, 4);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_499, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_499", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_499() */
/* END TEST 499 */

/*
 *  BEGIN TEST 500
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_500( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, 10), MAKE_VAL_COMPLEX(-2, -14), MAKE_VAL_COMPLEX(20, 22), MAKE_VAL_COMPLEX(26, -4), MAKE_VAL_COMPLEX(-7, 20), MAKE_VAL_COMPLEX(21, 8), MAKE_VAL_COMPLEX(-24, 8), MAKE_VAL_COMPLEX(-10, -17), MAKE_VAL_COMPLEX(-6, 9), MAKE_VAL_COMPLEX(23, 4), MAKE_VAL_COMPLEX(-6, -2), MAKE_VAL_COMPLEX(-25, 8), MAKE_VAL_COMPLEX(1, -8), MAKE_VAL_COMPLEX(-20, -15), MAKE_VAL_COMPLEX(-12, -19) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_500, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_500", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_500() */
/* END TEST 500 */

/*
 *  BEGIN TEST 501
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_501( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(90, 60), MAKE_VAL_COMPLEX(42, -106), MAKE_VAL_COMPLEX(30, 78), MAKE_VAL_COMPLEX(89, 34), MAKE_VAL_COMPLEX(2, 95), MAKE_VAL_COMPLEX(99, -18), MAKE_VAL_COMPLEX(-56, 57), MAKE_VAL_COMPLEX(-25, -28), MAKE_VAL_COMPLEX(-59, 61), MAKE_VAL_COMPLEX(52, -4), MAKE_VAL_COMPLEX(-49, 42), MAKE_VAL_COMPLEX(-55, 82), MAKE_VAL_COMPLEX(4, -32), MAKE_VAL_COMPLEX(-80, -60), MAKE_VAL_COMPLEX(-48, -76) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_501, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_501", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_501() */
/* END TEST 501 */

/*
 *  BEGIN TEST 502
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_502( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-52, 4), MAKE_VAL_COMPLEX(16, 48), MAKE_VAL_COMPLEX(-40, -20), MAKE_VAL_COMPLEX(-44, 18), MAKE_VAL_COMPLEX(-26, -38), MAKE_VAL_COMPLEX(-36, 32), MAKE_VAL_COMPLEX(10, -40), MAKE_VAL_COMPLEX(22, 6), MAKE_VAL_COMPLEX(4, -38), MAKE_VAL_COMPLEX(-24, 18), MAKE_VAL_COMPLEX(8, -26), MAKE_VAL_COMPLEX(4, -48), MAKE_VAL_COMPLEX(8, 14), MAKE_VAL_COMPLEX(50, 0), MAKE_VAL_COMPLEX(42, 16) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_502, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_502", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_502() */
/* END TEST 502 */

/*
 *  BEGIN TEST 503
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_ROWMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_503( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_ROWMAJ, x480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, 86), MAKE_VAL_COMPLEX(178, -14), MAKE_VAL_COMPLEX(-110, -46), MAKE_VAL_COMPLEX(-86, 111), MAKE_VAL_COMPLEX(-13, 52), MAKE_VAL_COMPLEX(78, -17), MAKE_VAL_COMPLEX(13, 48), MAKE_VAL_COMPLEX(2, 65), MAKE_VAL_COMPLEX(-109, -21), MAKE_VAL_COMPLEX(-44, -42), MAKE_VAL_COMPLEX(-106, 33), MAKE_VAL_COMPLEX(-14, 93), MAKE_VAL_COMPLEX(16, 2), MAKE_VAL_COMPLEX(30, -40), MAKE_VAL_COMPLEX(38, -24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_503, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_503", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_503() */
/* END TEST 503 */

/*
 *  BEGIN TEST 504
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_504( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(16, -30), MAKE_VAL_COMPLEX(91, -48), MAKE_VAL_COMPLEX(-32, -82), MAKE_VAL_COMPLEX(40, 68), MAKE_VAL_COMPLEX(-139, 12), MAKE_VAL_COMPLEX(-34, -53), MAKE_VAL_COMPLEX(27, -64), MAKE_VAL_COMPLEX(94, -45), MAKE_VAL_COMPLEX(60, 71), MAKE_VAL_COMPLEX(-73, 59), MAKE_VAL_COMPLEX(71, 10), MAKE_VAL_COMPLEX(-8, -134) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_504, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_504", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_504() */
/* END TEST 504 */

/*
 *  BEGIN TEST 505
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_505( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(10, -8), MAKE_VAL_COMPLEX(14, 25), MAKE_VAL_COMPLEX(37, -15), MAKE_VAL_COMPLEX(-36, 24), MAKE_VAL_COMPLEX(-1, -52), MAKE_VAL_COMPLEX(31, -16), MAKE_VAL_COMPLEX(30, -3), MAKE_VAL_COMPLEX(21, 40), MAKE_VAL_COMPLEX(-28, 29), MAKE_VAL_COMPLEX(-32, -36), MAKE_VAL_COMPLEX(9, 28), MAKE_VAL_COMPLEX(53, 3) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_505, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_505", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_505() */
/* END TEST 505 */

/*
 *  BEGIN TEST 506
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_506( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-40, -10), MAKE_VAL_COMPLEX(-52, -81), MAKE_VAL_COMPLEX(-83, 26), MAKE_VAL_COMPLEX(76, -32), MAKE_VAL_COMPLEX(10, 130), MAKE_VAL_COMPLEX(-57, 24), MAKE_VAL_COMPLEX(-74, -27), MAKE_VAL_COMPLEX(-40, -95), MAKE_VAL_COMPLEX(74, -58), MAKE_VAL_COMPLEX(52, 81), MAKE_VAL_COMPLEX(7, -79), MAKE_VAL_COMPLEX(-124, 3) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_506, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_506", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_506() */
/* END TEST 506 */

/*
 *  BEGIN TEST 507
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_507( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-20, 146), MAKE_VAL_COMPLEX(78, 155), MAKE_VAL_COMPLEX(-32, -59), MAKE_VAL_COMPLEX(116, 16), MAKE_VAL_COMPLEX(-109, -113), MAKE_VAL_COMPLEX(-105, -66), MAKE_VAL_COMPLEX(-60, 97), MAKE_VAL_COMPLEX(56, -19), MAKE_VAL_COMPLEX(19, 17), MAKE_VAL_COMPLEX(-15, 106), MAKE_VAL_COMPLEX(-98, -14), MAKE_VAL_COMPLEX(93, -142) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(9, 9);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_507, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_507", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_507() */
/* END TEST 507 */

/*
 *  BEGIN TEST 508
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_508( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(60, -82), MAKE_VAL_COMPLEX(-80, -36), MAKE_VAL_COMPLEX(77, -38), MAKE_VAL_COMPLEX(29, -75), MAKE_VAL_COMPLEX(87, 13), MAKE_VAL_COMPLEX(5, -102), MAKE_VAL_COMPLEX(33, 76), MAKE_VAL_COMPLEX(-32, 13), MAKE_VAL_COMPLEX(55, 50), MAKE_VAL_COMPLEX(6, -57), MAKE_VAL_COMPLEX(18, 40), MAKE_VAL_COMPLEX(56, 65), MAKE_VAL_COMPLEX(-20, -5), MAKE_VAL_COMPLEX(-74, 60), MAKE_VAL_COMPLEX(-77, 38) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_508, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_508", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_508() */
/* END TEST 508 */

/*
 *  BEGIN TEST 509
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_509( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-84, 42), MAKE_VAL_COMPLEX(48, 68), MAKE_VAL_COMPLEX(-77, -2), MAKE_VAL_COMPLEX(-61, 47), MAKE_VAL_COMPLEX(-67, -49), MAKE_VAL_COMPLEX(-45, 86), MAKE_VAL_COMPLEX(3, -84), MAKE_VAL_COMPLEX(36, 7), MAKE_VAL_COMPLEX(-23, -62), MAKE_VAL_COMPLEX(-30, 45), MAKE_VAL_COMPLEX(6, -44), MAKE_VAL_COMPLEX(-16, -81), MAKE_VAL_COMPLEX(12, 17), MAKE_VAL_COMPLEX(94, -20), MAKE_VAL_COMPLEX(81, 2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_509, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_509", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_509() */
/* END TEST 509 */

/*
 *  BEGIN TEST 510
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_510( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(54, -46), MAKE_VAL_COMPLEX(-56, -40), MAKE_VAL_COMPLEX(60, -10), MAKE_VAL_COMPLEX(34, -55), MAKE_VAL_COMPLEX(59, 21), MAKE_VAL_COMPLEX(14, -64), MAKE_VAL_COMPLEX(21, 52), MAKE_VAL_COMPLEX(-29, 11), MAKE_VAL_COMPLEX(26, 45), MAKE_VAL_COMPLEX(12, -39), MAKE_VAL_COMPLEX(12, 35), MAKE_VAL_COMPLEX(34, 56), MAKE_VAL_COMPLEX(-20, -9), MAKE_VAL_COMPLEX(-55, 40), MAKE_VAL_COMPLEX(-59, 16) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_510, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_510", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_510() */
/* END TEST 510 */

/*
 *  BEGIN TEST 511
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_511( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-48, -96), MAKE_VAL_COMPLEX(-44, 60), MAKE_VAL_COMPLEX(-7, -101), MAKE_VAL_COMPLEX(-56, -31), MAKE_VAL_COMPLEX(49, -67), MAKE_VAL_COMPLEX(-77, -75), MAKE_VAL_COMPLEX(54, 33), MAKE_VAL_COMPLEX(8, 6), MAKE_VAL_COMPLEX(83, -38), MAKE_VAL_COMPLEX(-45, -30), MAKE_VAL_COMPLEX(20, -9), MAKE_VAL_COMPLEX(63, -25), MAKE_VAL_COMPLEX(13, 18), MAKE_VAL_COMPLEX(-7, 64), MAKE_VAL_COMPLEX(-2, 81) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_511, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_511", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_511() */
/* END TEST 511 */

/*
 *  BEGIN TEST 512
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_512( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(46, 8), MAKE_VAL_COMPLEX(8, -44), MAKE_VAL_COMPLEX(27, 22), MAKE_VAL_COMPLEX(42, 7), MAKE_VAL_COMPLEX(12, 42), MAKE_VAL_COMPLEX(45, -28), MAKE_VAL_COMPLEX(-31, 43), MAKE_VAL_COMPLEX(-13, -22), MAKE_VAL_COMPLEX(-8, 28), MAKE_VAL_COMPLEX(27, -9), MAKE_VAL_COMPLEX(-23, 16), MAKE_VAL_COMPLEX(-24, 39), MAKE_VAL_COMPLEX(7, -13), MAKE_VAL_COMPLEX(-54, -25), MAKE_VAL_COMPLEX(-33, -27) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_512, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_512", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_512() */
/* END TEST 512 */

/*
 *  BEGIN TEST 513
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_513( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(4, -38), MAKE_VAL_COMPLEX(-48, 4), MAKE_VAL_COMPLEX(23, -22), MAKE_VAL_COMPLEX(-7, -47), MAKE_VAL_COMPLEX(33, -17), MAKE_VAL_COMPLEX(-25, -22), MAKE_VAL_COMPLEX(41, 2), MAKE_VAL_COMPLEX(-12, 27), MAKE_VAL_COMPLEX(23, 12), MAKE_VAL_COMPLEX(-12, -21), MAKE_VAL_COMPLEX(28, 14), MAKE_VAL_COMPLEX(44, 11), MAKE_VAL_COMPLEX(-22, 3), MAKE_VAL_COMPLEX(4, 50), MAKE_VAL_COMPLEX(-17, 32) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_513, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_513", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_513() */
/* END TEST 513 */

/*
 *  BEGIN TEST 514
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_514( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-42, 66), MAKE_VAL_COMPLEX(72, 24), MAKE_VAL_COMPLEX(-60, 30), MAKE_VAL_COMPLEX(-18, 69), MAKE_VAL_COMPLEX(-69, -3), MAKE_VAL_COMPLEX(6, 72), MAKE_VAL_COMPLEX(-39, -48), MAKE_VAL_COMPLEX(27, -21), MAKE_VAL_COMPLEX(-42, -39), MAKE_VAL_COMPLEX(0, 45), MAKE_VAL_COMPLEX(-24, -33), MAKE_VAL_COMPLEX(-54, -48), MAKE_VAL_COMPLEX(24, 3), MAKE_VAL_COMPLEX(45, -60), MAKE_VAL_COMPLEX(57, -36) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_514, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_514", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_514() */
/* END TEST 514 */

/*
 *  BEGIN TEST 515
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_515( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_ROWMAJ, y480_VW );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(36, 4), MAKE_VAL_COMPLEX(28, -20), MAKE_VAL_COMPLEX(19, 3), MAKE_VAL_COMPLEX(19, 32), MAKE_VAL_COMPLEX(3, 37), MAKE_VAL_COMPLEX(57, -35), MAKE_VAL_COMPLEX(-63, 44), MAKE_VAL_COMPLEX(12, -38), MAKE_VAL_COMPLEX(8, 17), MAKE_VAL_COMPLEX(24, -3), MAKE_VAL_COMPLEX(-35, -8), MAKE_VAL_COMPLEX(-37, 17), MAKE_VAL_COMPLEX(28, 3), MAKE_VAL_COMPLEX(-52, -59), MAKE_VAL_COMPLEX(-23, -28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(3, 1);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_515, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_515", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_515() */
/* END TEST 515 */

/*
 *  BEGIN TEST 516
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_516( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(14, 16), MAKE_VAL_COMPLEX(12, 15), MAKE_VAL_COMPLEX(21, 12), MAKE_VAL_COMPLEX(-12, 7), MAKE_VAL_COMPLEX(20, -26), MAKE_VAL_COMPLEX(6, 2), MAKE_VAL_COMPLEX(13, 8), MAKE_VAL_COMPLEX(-8, 28), MAKE_VAL_COMPLEX(-28, 7), MAKE_VAL_COMPLEX(-14, -19), MAKE_VAL_COMPLEX(-7, 14), MAKE_VAL_COMPLEX(23, 1) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_516, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_516", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_516() */
/* END TEST 516 */

/*
 *  BEGIN TEST 517
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_517( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-52, -22), MAKE_VAL_COMPLEX(-67, -86), MAKE_VAL_COMPLEX(-96, 19), MAKE_VAL_COMPLEX(80, -43), MAKE_VAL_COMPLEX(0, 146), MAKE_VAL_COMPLEX(-57, 25), MAKE_VAL_COMPLEX(-84, -27), MAKE_VAL_COMPLEX(-37, -113), MAKE_VAL_COMPLEX(92, -67), MAKE_VAL_COMPLEX(69, 90), MAKE_VAL_COMPLEX(7, -88), MAKE_VAL_COMPLEX(-138, 12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_517, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_517", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_517() */
/* END TEST 517 */

/*
 *  BEGIN TEST 518
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_518( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-24, -28), MAKE_VAL_COMPLEX(4, -86), MAKE_VAL_COMPLEX(-74, -24), MAKE_VAL_COMPLEX(72, 16), MAKE_VAL_COMPLEX(-60, 100), MAKE_VAL_COMPLEX(-54, -12), MAKE_VAL_COMPLEX(-40, -58), MAKE_VAL_COMPLEX(20, -90), MAKE_VAL_COMPLEX(84, -4), MAKE_VAL_COMPLEX(-4, 86), MAKE_VAL_COMPLEX(46, -54), MAKE_VAL_COMPLEX(-92, -62) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -2);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_518, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_518", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_518() */
/* END TEST 518 */

/*
 *  BEGIN TEST 519
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_519( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	oski_value_t* y_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-42, 116), MAKE_VAL_COMPLEX(68, 19), MAKE_VAL_COMPLEX(-69, 94), MAKE_VAL_COMPLEX(-5, 66), MAKE_VAL_COMPLEX(82, 104), MAKE_VAL_COMPLEX(-65, -27), MAKE_VAL_COMPLEX(11, -12), MAKE_VAL_COMPLEX(-109, -33), MAKE_VAL_COMPLEX(-57, -70), MAKE_VAL_COMPLEX(-52, -57), MAKE_VAL_COMPLEX(4, -15), MAKE_VAL_COMPLEX(28, -48) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(6, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_519, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_519", y_true, y_val, y480_LEN*y480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_519() */
/* END TEST 519 */

/*
 *  BEGIN TEST 520
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_520( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(114, 8), MAKE_VAL_COMPLEX(-13, -122), MAKE_VAL_COMPLEX(72, 55), MAKE_VAL_COMPLEX(112, -18), MAKE_VAL_COMPLEX(42, 94), MAKE_VAL_COMPLEX(82, -46), MAKE_VAL_COMPLEX(-44, 81), MAKE_VAL_COMPLEX(-49, -30), MAKE_VAL_COMPLEX(-19, 72), MAKE_VAL_COMPLEX(60, -30), MAKE_VAL_COMPLEX(-18, 56), MAKE_VAL_COMPLEX(-31, 114), MAKE_VAL_COMPLEX(-5, -39), MAKE_VAL_COMPLEX(-119, -30), MAKE_VAL_COMPLEX(-86, -52) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_520, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_520", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_520() */
/* END TEST 520 */

/*
 *  BEGIN TEST 521
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_521( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(118, 16), MAKE_VAL_COMPLEX(-19, -102), MAKE_VAL_COMPLEX(88, 65), MAKE_VAL_COMPLEX(96, -26), MAKE_VAL_COMPLEX(42, 94), MAKE_VAL_COMPLEX(102, -66), MAKE_VAL_COMPLEX(-32, 87), MAKE_VAL_COMPLEX(-43, -14), MAKE_VAL_COMPLEX(-29, 92), MAKE_VAL_COMPLEX(60, -30), MAKE_VAL_COMPLEX(-38, 52), MAKE_VAL_COMPLEX(-25, 94), MAKE_VAL_COMPLEX(-19, -29), MAKE_VAL_COMPLEX(-101, -10), MAKE_VAL_COMPLEX(-86, -52) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_521, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_521", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_521() */
/* END TEST 521 */

/*
 *  BEGIN TEST 522
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_522( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(62, 58), MAKE_VAL_COMPLEX(40, -72), MAKE_VAL_COMPLEX(20, 70), MAKE_VAL_COMPLEX(70, 33), MAKE_VAL_COMPLEX(-17, 73), MAKE_VAL_COMPLEX(78, 8), MAKE_VAL_COMPLEX(-59, 32), MAKE_VAL_COMPLEX(-17, -33), MAKE_VAL_COMPLEX(-50, 37), MAKE_VAL_COMPLEX(48, 9), MAKE_VAL_COMPLEX(-40, 19), MAKE_VAL_COMPLEX(-62, 48), MAKE_VAL_COMPLEX(8, -25), MAKE_VAL_COMPLEX(-55, -60), MAKE_VAL_COMPLEX(-27, -68) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -2);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_522, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_522", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_522() */
/* END TEST 522 */

/*
 *  BEGIN TEST 523
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_523( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(82, 74), MAKE_VAL_COMPLEX(83, 1), MAKE_VAL_COMPLEX(75, 55), MAKE_VAL_COMPLEX(3, 61), MAKE_VAL_COMPLEX(-26, 68), MAKE_VAL_COMPLEX(74, -82), MAKE_VAL_COMPLEX(-16, 8), MAKE_VAL_COMPLEX(43, -9), MAKE_VAL_COMPLEX(-28, 104), MAKE_VAL_COMPLEX(45, 15), MAKE_VAL_COMPLEX(-101, 53), MAKE_VAL_COMPLEX(-101, -27), MAKE_VAL_COMPLEX(1, 37), MAKE_VAL_COMPLEX(50, -60), MAKE_VAL_COMPLEX(-17, -69) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-5, 5);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_523, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_523", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_523() */
/* END TEST 523 */

/*
 *  BEGIN TEST 524
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_524( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-24, -86), MAKE_VAL_COMPLEX(-69, 30), MAKE_VAL_COMPLEX(12, -75), MAKE_VAL_COMPLEX(-34, -61), MAKE_VAL_COMPLEX(53, -53), MAKE_VAL_COMPLEX(-72, -38), MAKE_VAL_COMPLEX(61, 1), MAKE_VAL_COMPLEX(-6, 29), MAKE_VAL_COMPLEX(67, -15), MAKE_VAL_COMPLEX(-36, -33), MAKE_VAL_COMPLEX(54, 7), MAKE_VAL_COMPLEX(75, 2), MAKE_VAL_COMPLEX(-13, 12), MAKE_VAL_COMPLEX(6, 70), MAKE_VAL_COMPLEX(-13, 72) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_524, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_524", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_524() */
/* END TEST 524 */

/*
 *  BEGIN TEST 525
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_525( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-10, -84), MAKE_VAL_COMPLEX(-83, 42), MAKE_VAL_COMPLEX(38, -65), MAKE_VAL_COMPLEX(-43, -77), MAKE_VAL_COMPLEX(62, -48), MAKE_VAL_COMPLEX(-48, -68), MAKE_VAL_COMPLEX(75, 16), MAKE_VAL_COMPLEX(-5, 46), MAKE_VAL_COMPLEX(60, 13), MAKE_VAL_COMPLEX(-33, -39), MAKE_VAL_COMPLEX(35, 9), MAKE_VAL_COMPLEX(85, -8), MAKE_VAL_COMPLEX(-30, 20), MAKE_VAL_COMPLEX(14, 95), MAKE_VAL_COMPLEX(-23, 73) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_525, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_525", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_525() */
/* END TEST 525 */

/*
 *  BEGIN TEST 526
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_526( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(22, -54), MAKE_VAL_COMPLEX(-56, -8), MAKE_VAL_COMPLEX(40, -30), MAKE_VAL_COMPLEX(4, -53), MAKE_VAL_COMPLEX(51, -7), MAKE_VAL_COMPLEX(-14, -52), MAKE_VAL_COMPLEX(35, 30), MAKE_VAL_COMPLEX(-17, 19), MAKE_VAL_COMPLEX(36, 23), MAKE_VAL_COMPLEX(-6, -33), MAKE_VAL_COMPLEX(22, 21), MAKE_VAL_COMPLEX(46, 28), MAKE_VAL_COMPLEX(-18, 1), MAKE_VAL_COMPLEX(-25, 50), MAKE_VAL_COMPLEX(-37, 34) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_526, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_526", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_526() */
/* END TEST 526 */

/*
 *  BEGIN TEST 527
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED>
 *  x : <'x480', LAYOUT_COLMAJ>
 *  y : <'y480', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_527( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A480_ptr, A480_ind, A480_val,
		A480_M, A480_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y480_val,
		y480_LEN * y480_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y480_LEN, y480_VW, LAYOUT_COLMAJ, y480_LEN );
	oski_value_t* y_val = array_Duplicate( x480_val,
		x480_LEN * x480_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x480_LEN, x480_VW, LAYOUT_COLMAJ, x480_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(112, -46), MAKE_VAL_COMPLEX(1, -44), MAKE_VAL_COMPLEX(118, -33), MAKE_VAL_COMPLEX(32, -13), MAKE_VAL_COMPLEX(67, 49), MAKE_VAL_COMPLEX(-8, -146), MAKE_VAL_COMPLEX(31, 41), MAKE_VAL_COMPLEX(10, -7), MAKE_VAL_COMPLEX(71, 107), MAKE_VAL_COMPLEX(30, -45), MAKE_VAL_COMPLEX(-20, 107), MAKE_VAL_COMPLEX(-35, 56), MAKE_VAL_COMPLEX(1, 28), MAKE_VAL_COMPLEX(-16, -14), MAKE_VAL_COMPLEX(-81, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 6);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_527, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_527", y_true, y_val, x480_LEN*x480_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_527() */
/* END TEST 527 */

#define A528_M 4    /* # of rows */
#define A528_N 5    /* # of cols */
#define A528_NNZ 1  /* # of stored non-zeros */
static oski_index_t A528_ptr[A528_N+1] = {0, 1, 1, 1, 1, 1}; /* end row pointers for A528 */
static oski_index_t A528_ind[A528_NNZ] = {1}; /* end column indices pointers for A528 */
static oski_value_t A528_val[A528_NNZ] = {MAKE_VAL_COMPLEX(6, -10)}; /* end stored non-zero values for A528 */

#define x528_LEN 5   /* vector length */
#define x528_VW  3   /* vector width */
static oski_value_t x528_val[x528_LEN*x528_VW] = {MAKE_VAL_COMPLEX(2, 0), MAKE_VAL_COMPLEX(-8, 4), MAKE_VAL_COMPLEX(0, -10), MAKE_VAL_COMPLEX(10, -3), MAKE_VAL_COMPLEX(-2, 1), MAKE_VAL_COMPLEX(7, 10), MAKE_VAL_COMPLEX(-5, -7), MAKE_VAL_COMPLEX(10, 4), MAKE_VAL_COMPLEX(-10, 5), MAKE_VAL_COMPLEX(-9, 4), MAKE_VAL_COMPLEX(-3, -10), MAKE_VAL_COMPLEX(7, 3), MAKE_VAL_COMPLEX(6, 10), MAKE_VAL_COMPLEX(9, 1), MAKE_VAL_COMPLEX(-6, 5)}; /* data */

#define y528_LEN 4   /* vector length */
#define y528_VW  3   /* vector width */
static oski_value_t y528_val[y528_LEN*y528_VW] = {MAKE_VAL_COMPLEX(-10, 4), MAKE_VAL_COMPLEX(7, -5), MAKE_VAL_COMPLEX(-8, 2), MAKE_VAL_COMPLEX(-10, 7), MAKE_VAL_COMPLEX(-3, -10), MAKE_VAL_COMPLEX(5, -2), MAKE_VAL_COMPLEX(4, 5), MAKE_VAL_COMPLEX(-6, -9), MAKE_VAL_COMPLEX(-8, -9), MAKE_VAL_COMPLEX(-1, 9), MAKE_VAL_COMPLEX(10, 10), MAKE_VAL_COMPLEX(7, 8)}; /* data */

/*
 *  BEGIN TEST 528
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_528( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(2, 58), MAKE_VAL_COMPLEX(-19, -41), MAKE_VAL_COMPLEX(-6, 34), MAKE_VAL_COMPLEX(37, 41), MAKE_VAL_COMPLEX(-66, 100), MAKE_VAL_COMPLEX(-93, -79), MAKE_VAL_COMPLEX(28, -17), MAKE_VAL_COMPLEX(-47, 16), MAKE_VAL_COMPLEX(-71, 27), MAKE_VAL_COMPLEX(34, 27), MAKE_VAL_COMPLEX(67, -40), MAKE_VAL_COMPLEX(61, -16) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_528, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_528", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_528() */
/* END TEST 528 */

/*
 *  BEGIN TEST 529
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_529( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-22, 8), MAKE_VAL_COMPLEX(22, -14), MAKE_VAL_COMPLEX(-16, 14), MAKE_VAL_COMPLEX(-42, 37), MAKE_VAL_COMPLEX(4, -125), MAKE_VAL_COMPLEX(103, 46), MAKE_VAL_COMPLEX(13, 17), MAKE_VAL_COMPLEX(-22, -22), MAKE_VAL_COMPLEX(-6, -23), MAKE_VAL_COMPLEX(7, 14), MAKE_VAL_COMPLEX(23, 30), MAKE_VAL_COMPLEX(7, 13) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(2, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_529, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_529", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_529() */
/* END TEST 529 */

/*
 *  BEGIN TEST 530
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_530( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(66, -38), MAKE_VAL_COMPLEX(-44, 42), MAKE_VAL_COMPLEX(54, -22), MAKE_VAL_COMPLEX(63, -59), MAKE_VAL_COMPLEX(31, 67), MAKE_VAL_COMPLEX(-33, 19), MAKE_VAL_COMPLEX(-33, -31), MAKE_VAL_COMPLEX(51, 57), MAKE_VAL_COMPLEX(65, 55), MAKE_VAL_COMPLEX(-2, -64), MAKE_VAL_COMPLEX(-80, -60), MAKE_VAL_COMPLEX(-57, -49) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_530, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_530", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_530() */
/* END TEST 530 */

/*
 *  BEGIN TEST 531
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_531( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-86, -64), MAKE_VAL_COMPLEX(130, 64), MAKE_VAL_COMPLEX(-124, -6), MAKE_VAL_COMPLEX(-348, -97), MAKE_VAL_COMPLEX(830, -421), MAKE_VAL_COMPLEX(61, 876), MAKE_VAL_COMPLEX(-44, 116), MAKE_VAL_COMPLEX(21, -173), MAKE_VAL_COMPLEX(98, -51), MAKE_VAL_COMPLEX(-4, 85), MAKE_VAL_COMPLEX(-78, 181), MAKE_VAL_COMPLEX(-28, 28) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-4, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 7);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_531, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_531", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_531() */
/* END TEST 531 */

/*
 *  BEGIN TEST 532
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_532( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-16, 140), MAKE_VAL_COMPLEX(-35, -43), MAKE_VAL_COMPLEX(-28, 20), MAKE_VAL_COMPLEX(-99, 1), MAKE_VAL_COMPLEX(16, -12), MAKE_VAL_COMPLEX(-21, -103), MAKE_VAL_COMPLEX(23, 76), MAKE_VAL_COMPLEX(-74, -71), MAKE_VAL_COMPLEX(87, -19), MAKE_VAL_COMPLEX(83, 4), MAKE_VAL_COMPLEX(4, 99), MAKE_VAL_COMPLEX(-40, -37), MAKE_VAL_COMPLEX(-18, -98), MAKE_VAL_COMPLEX(-69, -35), MAKE_VAL_COMPLEX(63, -22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_532, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_532", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_532() */
/* END TEST 532 */

/*
 *  BEGIN TEST 533
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_533( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(0, -156), MAKE_VAL_COMPLEX(131, 75), MAKE_VAL_COMPLEX(-52, 60), MAKE_VAL_COMPLEX(-5, -57), MAKE_VAL_COMPLEX(8, 20), MAKE_VAL_COMPLEX(45, -33), MAKE_VAL_COMPLEX(-39, 20), MAKE_VAL_COMPLEX(26, -41), MAKE_VAL_COMPLEX(33, 59), MAKE_VAL_COMPLEX(21, 36), MAKE_VAL_COMPLEX(-60, 5), MAKE_VAL_COMPLEX(8, -43), MAKE_VAL_COMPLEX(50, -30), MAKE_VAL_COMPLEX(5, -45), MAKE_VAL_COMPLEX(25, 30) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_533, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_533", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_533() */
/* END TEST 533 */

/*
 *  BEGIN TEST 534
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_534( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-16, 6), MAKE_VAL_COMPLEX(52, -56), MAKE_VAL_COMPLEX(30, 80), MAKE_VAL_COMPLEX(-71, 54), MAKE_VAL_COMPLEX(13, -14), MAKE_VAL_COMPLEX(-86, -59), MAKE_VAL_COMPLEX(61, 41), MAKE_VAL_COMPLEX(-92, -2), MAKE_VAL_COMPLEX(65, -70), MAKE_VAL_COMPLEX(60, -59), MAKE_VAL_COMPLEX(54, 71), MAKE_VAL_COMPLEX(-65, -3), MAKE_VAL_COMPLEX(-78, -62), MAKE_VAL_COMPLEX(-75, 19), MAKE_VAL_COMPLEX(33, -58) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, 3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_534, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_534", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_534() */
/* END TEST 534 */

/*
 *  BEGIN TEST 535
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_535( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(578, -1156), MAKE_VAL_COMPLEX(748, 664), MAKE_VAL_COMPLEX(-196, 502), MAKE_VAL_COMPLEX(96, 53), MAKE_VAL_COMPLEX(-16, 77), MAKE_VAL_COMPLEX(-129, 8), MAKE_VAL_COMPLEX(45, -65), MAKE_VAL_COMPLEX(-42, 144), MAKE_VAL_COMPLEX(28, 29), MAKE_VAL_COMPLEX(47, -134), MAKE_VAL_COMPLEX(29, -108), MAKE_VAL_COMPLEX(-63, -59), MAKE_VAL_COMPLEX(-78, 6), MAKE_VAL_COMPLEX(-33, 51), MAKE_VAL_COMPLEX(-12, -51) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-8, -4);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 6);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_535, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_535", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_535() */
/* END TEST 535 */

/*
 *  BEGIN TEST 536
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_536( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-152, -44), MAKE_VAL_COMPLEX(117, -159), MAKE_VAL_COMPLEX(92, 100), MAKE_VAL_COMPLEX(-55, 75), MAKE_VAL_COMPLEX(4, -26), MAKE_VAL_COMPLEX(-87, -27), MAKE_VAL_COMPLEX(69, 22), MAKE_VAL_COMPLEX(-86, 17), MAKE_VAL_COMPLEX(27, -89), MAKE_VAL_COMPLEX(33, -60), MAKE_VAL_COMPLEX(78, 55), MAKE_VAL_COMPLEX(-50, 25), MAKE_VAL_COMPLEX(-86, -30), MAKE_VAL_COMPLEX(-59, 39), MAKE_VAL_COMPLEX(11, -60) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-6, 5);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_536, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_536", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_536() */
/* END TEST 536 */

/*
 *  BEGIN TEST 537
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_537( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(142, 50), MAKE_VAL_COMPLEX(-89, 115), MAKE_VAL_COMPLEX(-62, -50), MAKE_VAL_COMPLEX(14, -30), MAKE_VAL_COMPLEX(3, 15), MAKE_VAL_COMPLEX(22, -2), MAKE_VAL_COMPLEX(-23, -2), MAKE_VAL_COMPLEX(24, -7), MAKE_VAL_COMPLEX(8, 34), MAKE_VAL_COMPLEX(0, 13), MAKE_VAL_COMPLEX(-33, -14), MAKE_VAL_COMPLEX(6, -19), MAKE_VAL_COMPLEX(26, -2), MAKE_VAL_COMPLEX(11, -17), MAKE_VAL_COMPLEX(4, 17) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_537, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_537", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_537() */
/* END TEST 537 */

/*
 *  BEGIN TEST 538
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_538( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-6, 6), MAKE_VAL_COMPLEX(12, -36), MAKE_VAL_COMPLEX(30, 30), MAKE_VAL_COMPLEX(-21, 39), MAKE_VAL_COMPLEX(3, -9), MAKE_VAL_COMPLEX(-51, -9), MAKE_VAL_COMPLEX(36, 6), MAKE_VAL_COMPLEX(-42, 18), MAKE_VAL_COMPLEX(15, -45), MAKE_VAL_COMPLEX(15, -39), MAKE_VAL_COMPLEX(39, 21), MAKE_VAL_COMPLEX(-30, 12), MAKE_VAL_COMPLEX(-48, -12), MAKE_VAL_COMPLEX(-30, 24), MAKE_VAL_COMPLEX(3, -33) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 3);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_538, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_538", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_538() */
/* END TEST 538 */

/*
 *  BEGIN TEST 539
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_539( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(182, 1196), MAKE_VAL_COMPLEX(-1021, -243), MAKE_VAL_COMPLEX(112, -454), MAKE_VAL_COMPLEX(89, 42), MAKE_VAL_COMPLEX(-58, 61), MAKE_VAL_COMPLEX(-34, -27), MAKE_VAL_COMPLEX(19, -48), MAKE_VAL_COMPLEX(-39, 78), MAKE_VAL_COMPLEX(-31, 92), MAKE_VAL_COMPLEX(67, -29), MAKE_VAL_COMPLEX(30, -110), MAKE_VAL_COMPLEX(28, -81), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0), MAKE_VAL_COMPLEX(0, 0) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-4, -7);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 0);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_539, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_539", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_539() */
/* END TEST 539 */

/*
 *  BEGIN TEST 540
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_540( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-22, -60), MAKE_VAL_COMPLEX(52, 19), MAKE_VAL_COMPLEX(-17, -55), MAKE_VAL_COMPLEX(-51, -56), MAKE_VAL_COMPLEX(52, -14), MAKE_VAL_COMPLEX(2, 135), MAKE_VAL_COMPLEX(-20, 28), MAKE_VAL_COMPLEX(51, -46), MAKE_VAL_COMPLEX(54, -58), MAKE_VAL_COMPLEX(-147, -56), MAKE_VAL_COMPLEX(-70, 65), MAKE_VAL_COMPLEX(-41, 45) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_540, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_540", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_540() */
/* END TEST 540 */

/*
 *  BEGIN TEST 541
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_541( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-84, -60), MAKE_VAL_COMPLEX(53, 54), MAKE_VAL_COMPLEX(-51, -47), MAKE_VAL_COMPLEX(-97, -49), MAKE_VAL_COMPLEX(73, -78), MAKE_VAL_COMPLEX(51, -75), MAKE_VAL_COMPLEX(-30, 53), MAKE_VAL_COMPLEX(45, -83), MAKE_VAL_COMPLEX(32, -99), MAKE_VAL_COMPLEX(16, 87), MAKE_VAL_COMPLEX(-20, 125), MAKE_VAL_COMPLEX(-36, 93) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, 8);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_541, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_541", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_541() */
/* END TEST 541 */

/*
 *  BEGIN TEST 542
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_542( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-86, 46), MAKE_VAL_COMPLEX(58, -52), MAKE_VAL_COMPLEX(-70, 26), MAKE_VAL_COMPLEX(-83, 73), MAKE_VAL_COMPLEX(-37, -87), MAKE_VAL_COMPLEX(43, -23), MAKE_VAL_COMPLEX(41, 41), MAKE_VAL_COMPLEX(-63, -75), MAKE_VAL_COMPLEX(-81, -73), MAKE_VAL_COMPLEX(0, 82), MAKE_VAL_COMPLEX(100, 80), MAKE_VAL_COMPLEX(71, 65) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(9, -1);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_542, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_542", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_542() */
/* END TEST 542 */

/*
 *  BEGIN TEST 543
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_543( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, 102), MAKE_VAL_COMPLEX(-158, -297), MAKE_VAL_COMPLEX(-99, 119), MAKE_VAL_COMPLEX(16, 215), MAKE_VAL_COMPLEX(-87, 25), MAKE_VAL_COMPLEX(847, 126), MAKE_VAL_COMPLEX(115, -77), MAKE_VAL_COMPLEX(-206, 5), MAKE_VAL_COMPLEX(-217, -1), MAKE_VAL_COMPLEX(-419, 766), MAKE_VAL_COMPLEX(190, 65), MAKE_VAL_COMPLEX(152, -60) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, -8);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, -9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_543, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_543", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_543() */
/* END TEST 543 */

/*
 *  BEGIN TEST 544
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_544( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-28, -94), MAKE_VAL_COMPLEX(43, -100), MAKE_VAL_COMPLEX(86, 105), MAKE_VAL_COMPLEX(-40, 20), MAKE_VAL_COMPLEX(14, -9), MAKE_VAL_COMPLEX(-46, -34), MAKE_VAL_COMPLEX(24, 32), MAKE_VAL_COMPLEX(-50, -5), MAKE_VAL_COMPLEX(55, -25), MAKE_VAL_COMPLEX(31, -22), MAKE_VAL_COMPLEX(19, 38), MAKE_VAL_COMPLEX(-31, 0), MAKE_VAL_COMPLEX(-40, -44), MAKE_VAL_COMPLEX(-46, 4), MAKE_VAL_COMPLEX(25, -31) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_544, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_544", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_544() */
/* END TEST 544 */

/*
 *  BEGIN TEST 545
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_545( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(18, 104), MAKE_VAL_COMPLEX(-23, 40), MAKE_VAL_COMPLEX(-36, -55), MAKE_VAL_COMPLEX(5, 45), MAKE_VAL_COMPLEX(-9, -6), MAKE_VAL_COMPLEX(-39, 19), MAKE_VAL_COMPLEX(36, -22), MAKE_VAL_COMPLEX(-20, 35), MAKE_VAL_COMPLEX(-30, -50), MAKE_VAL_COMPLEX(-6, -43), MAKE_VAL_COMPLEX(46, -3), MAKE_VAL_COMPLEX(-19, 20), MAKE_VAL_COMPLEX(-40, 24), MAKE_VAL_COMPLEX(-4, 36), MAKE_VAL_COMPLEX(-20, -24) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(0, 4);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_545, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_545", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_545() */
/* END TEST 545 */

/*
 *  BEGIN TEST 546
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_546( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-16, -14), MAKE_VAL_COMPLEX(92, 24), MAKE_VAL_COMPLEX(-70, 80), MAKE_VAL_COMPLEX(-101, -46), MAKE_VAL_COMPLEX(23, 6), MAKE_VAL_COMPLEX(14, -129), MAKE_VAL_COMPLEX(-9, 91), MAKE_VAL_COMPLEX(-52, -102), MAKE_VAL_COMPLEX(115, 30), MAKE_VAL_COMPLEX(100, 31), MAKE_VAL_COMPLEX(-46, 101), MAKE_VAL_COMPLEX(-35, -73), MAKE_VAL_COMPLEX(22, -122), MAKE_VAL_COMPLEX(-65, -71), MAKE_VAL_COMPLEX(83, 2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-8, -7);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_546, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_546", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_546() */
/* END TEST 546 */

/*
 *  BEGIN TEST 547
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_547( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-50, 300), MAKE_VAL_COMPLEX(-49, 217), MAKE_VAL_COMPLEX(-203, -201), MAKE_VAL_COMPLEX(-75, -10), MAKE_VAL_COMPLEX(-6, 3), MAKE_VAL_COMPLEX(14, -87), MAKE_VAL_COMPLEX(25, 45), MAKE_VAL_COMPLEX(-35, -65), MAKE_VAL_COMPLEX(35, -30), MAKE_VAL_COMPLEX(85, 0), MAKE_VAL_COMPLEX(-9, 82), MAKE_VAL_COMPLEX(-32, -64), MAKE_VAL_COMPLEX(6, -58), MAKE_VAL_COMPLEX(-33, -31), MAKE_VAL_COMPLEX(39, -2) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-3, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(-4, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_547, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_547", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_547() */
/* END TEST 547 */

/*
 *  BEGIN TEST 548
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_548( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(80, 60), MAKE_VAL_COMPLEX(23, -40), MAKE_VAL_COMPLEX(-24, 45), MAKE_VAL_COMPLEX(21, 78), MAKE_VAL_COMPLEX(-1, -19), MAKE_VAL_COMPLEX(-88, 55), MAKE_VAL_COMPLEX(53, -31), MAKE_VAL_COMPLEX(-38, 81), MAKE_VAL_COMPLEX(-20, -75), MAKE_VAL_COMPLEX(-33, -69), MAKE_VAL_COMPLEX(77, -23), MAKE_VAL_COMPLEX(-24, 61), MAKE_VAL_COMPLEX(-86, 38), MAKE_VAL_COMPLEX(-17, 71), MAKE_VAL_COMPLEX(-34, -53) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_548, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_548", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_548() */
/* END TEST 548 */

/*
 *  BEGIN TEST 549
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_549( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-74, -62), MAKE_VAL_COMPLEX(-43, 60), MAKE_VAL_COMPLEX(14, -75), MAKE_VAL_COMPLEX(6, -97), MAKE_VAL_COMPLEX(-4, 24), MAKE_VAL_COMPLEX(119, -32), MAKE_VAL_COMPLEX(-75, 15), MAKE_VAL_COMPLEX(72, -79), MAKE_VAL_COMPLEX(-5, 100), MAKE_VAL_COMPLEX(10, 90), MAKE_VAL_COMPLEX(-96, -4), MAKE_VAL_COMPLEX(48, -59), MAKE_VAL_COMPLEX(114, -14), MAKE_VAL_COMPLEX(45, -77), MAKE_VAL_COMPLEX(21, 74) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, -9);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_549, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_549", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_549() */
/* END TEST 549 */

/*
 *  BEGIN TEST 550
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_550( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-2, 12), MAKE_VAL_COMPLEX(-16, -52), MAKE_VAL_COMPLEX(60, 10), MAKE_VAL_COMPLEX(8, 63), MAKE_VAL_COMPLEX(-4, -13), MAKE_VAL_COMPLEX(-67, 32), MAKE_VAL_COMPLEX(47, -23), MAKE_VAL_COMPLEX(-34, 56), MAKE_VAL_COMPLEX(-20, -65), MAKE_VAL_COMPLEX(-15, -58), MAKE_VAL_COMPLEX(63, -8), MAKE_VAL_COMPLEX(-25, 39), MAKE_VAL_COMPLEX(-66, 26), MAKE_VAL_COMPLEX(-15, 53), MAKE_VAL_COMPLEX(-24, -41) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_550, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_550", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_550() */
/* END TEST 550 */

/*
 *  BEGIN TEST 551
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_ROWMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_551( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_ROWMAJ, x528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(1150, -354), MAKE_VAL_COMPLEX(635, -91), MAKE_VAL_COMPLEX(-681, 1171), MAKE_VAL_COMPLEX(80, -192), MAKE_VAL_COMPLEX(17, -43), MAKE_VAL_COMPLEX(188, 128), MAKE_VAL_COMPLEX(-136, 64), MAKE_VAL_COMPLEX(185, -19), MAKE_VAL_COMPLEX(130, 100), MAKE_VAL_COMPLEX(-75, 239), MAKE_VAL_COMPLEX(-219, -89), MAKE_VAL_COMPLEX(209, -9), MAKE_VAL_COMPLEX(108, 44), MAKE_VAL_COMPLEX(78, -46), MAKE_VAL_COMPLEX(-18, 76) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(9, -9);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_551, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_551", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_551() */
/* END TEST 551 */

/*
 *  BEGIN TEST 552
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_552( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(46, 52), MAKE_VAL_COMPLEX(-37, -22), MAKE_VAL_COMPLEX(25, 34), MAKE_VAL_COMPLEX(66, 30), MAKE_VAL_COMPLEX(83, 21), MAKE_VAL_COMPLEX(-133, -53), MAKE_VAL_COMPLEX(22, -44), MAKE_VAL_COMPLEX(-32, 58), MAKE_VAL_COMPLEX(-32, 76), MAKE_VAL_COMPLEX(66, -15), MAKE_VAL_COMPLEX(30, -75), MAKE_VAL_COMPLEX(43, -57) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-2, -6);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_552, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_552", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_552() */
/* END TEST 552 */

/*
 *  BEGIN TEST 553
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_553( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-4, -34), MAKE_VAL_COMPLEX(1, 16), MAKE_VAL_COMPLEX(5, -16), MAKE_VAL_COMPLEX(-15, -21), MAKE_VAL_COMPLEX(-104, 18), MAKE_VAL_COMPLEX(112, 44), MAKE_VAL_COMPLEX(-19, 17), MAKE_VAL_COMPLEX(23, -13), MAKE_VAL_COMPLEX(29, -25), MAKE_VAL_COMPLEX(-36, -9), MAKE_VAL_COMPLEX(-30, 15), MAKE_VAL_COMPLEX(-40, 12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, 3);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_553, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_553", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_553() */
/* END TEST 553 */

/*
 *  BEGIN TEST 554
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_554( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(34, -60), MAKE_VAL_COMPLEX(-15, 53), MAKE_VAL_COMPLEX(32, -42), MAKE_VAL_COMPLEX(22, -75), MAKE_VAL_COMPLEX(55, 38), MAKE_VAL_COMPLEX(-17, 30), MAKE_VAL_COMPLEX(-40, -9), MAKE_VAL_COMPLEX(66, 21), MAKE_VAL_COMPLEX(76, 13), MAKE_VAL_COMPLEX(-31, -49), MAKE_VAL_COMPLEX(-90, -10), MAKE_VAL_COMPLEX(-67, -12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_554, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_554", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_554() */
/* END TEST 554 */

/*
 *  BEGIN TEST 555
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_555( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(4, -108), MAKE_VAL_COMPLEX(89, 107), MAKE_VAL_COMPLEX(-39, -119), MAKE_VAL_COMPLEX(-61, -203), MAKE_VAL_COMPLEX(733, -493), MAKE_VAL_COMPLEX(-633, 249), MAKE_VAL_COMPLEX(-87, -29), MAKE_VAL_COMPLEX(161, -37), MAKE_VAL_COMPLEX(165, -13), MAKE_VAL_COMPLEX(-37, -81), MAKE_VAL_COMPLEX(-155, 115), MAKE_VAL_COMPLEX(-45, 17) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(5, -3);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_555, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_555", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_555() */
/* END TEST 555 */

/*
 *  BEGIN TEST 556
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_556( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(6, 130), MAKE_VAL_COMPLEX(-2, 83), MAKE_VAL_COMPLEX(-76, -25), MAKE_VAL_COMPLEX(5, -80), MAKE_VAL_COMPLEX(2, 19), MAKE_VAL_COMPLEX(-10, -61), MAKE_VAL_COMPLEX(-74, 9), MAKE_VAL_COMPLEX(56, -77), MAKE_VAL_COMPLEX(20, 105), MAKE_VAL_COMPLEX(5, 84), MAKE_VAL_COMPLEX(-87, -66), MAKE_VAL_COMPLEX(50, -49), MAKE_VAL_COMPLEX(90, -27), MAKE_VAL_COMPLEX(42, -61), MAKE_VAL_COMPLEX(22, 63) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(3, -8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_556, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_556", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_556() */
/* END TEST 556 */

/*
 *  BEGIN TEST 557
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_557( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(16, -140), MAKE_VAL_COMPLEX(-66, 1), MAKE_VAL_COMPLEX(26, -85), MAKE_VAL_COMPLEX(90, -3), MAKE_VAL_COMPLEX(-19, 2), MAKE_VAL_COMPLEX(137, 136), MAKE_VAL_COMPLEX(-16, -61), MAKE_VAL_COMPLEX(74, 71), MAKE_VAL_COMPLEX(-105, 0), MAKE_VAL_COMPLEX(-84, 5), MAKE_VAL_COMPLEX(4, -29), MAKE_VAL_COMPLEX(42, 47), MAKE_VAL_COMPLEX(26, 107), MAKE_VAL_COMPLEX(62, 27), MAKE_VAL_COMPLEX(-63, 22) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_557, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_557", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_557() */
/* END TEST 557 */

/*
 *  BEGIN TEST 558
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_558( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-18, 2), MAKE_VAL_COMPLEX(68, -44), MAKE_VAL_COMPLEX(10, 90), MAKE_VAL_COMPLEX(-87, 37), MAKE_VAL_COMPLEX(17, -11), MAKE_VAL_COMPLEX(-73, -83), MAKE_VAL_COMPLEX(52, 58), MAKE_VAL_COMPLEX(-94, -26), MAKE_VAL_COMPLEX(85, -55), MAKE_VAL_COMPLEX(77, -45), MAKE_VAL_COMPLEX(37, 87), MAKE_VAL_COMPLEX(-66, -20), MAKE_VAL_COMPLEX(-64, -84), MAKE_VAL_COMPLEX(-82, 0), MAKE_VAL_COMPLEX(49, -51) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-9, 1);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_558, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_558", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_558() */
/* END TEST 558 */

/*
 *  BEGIN TEST 559
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_559( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(716, -422), MAKE_VAL_COMPLEX(89, -63), MAKE_VAL_COMPLEX(93, 35), MAKE_VAL_COMPLEX(2, 79), MAKE_VAL_COMPLEX(6, -23), MAKE_VAL_COMPLEX(29, 646), MAKE_VAL_COMPLEX(50, 54), MAKE_VAL_COMPLEX(-129, 109), MAKE_VAL_COMPLEX(50, -195), MAKE_VAL_COMPLEX(31, -100), MAKE_VAL_COMPLEX(-205, 216), MAKE_VAL_COMPLEX(-98, 16), MAKE_VAL_COMPLEX(-143, 45), MAKE_VAL_COMPLEX(-52, 6), MAKE_VAL_COMPLEX(2, -83) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-3, -5);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 8);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_559, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_559", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_559() */
/* END TEST 559 */

/*
 *  BEGIN TEST 560
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_560( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-138, -42), MAKE_VAL_COMPLEX(-42, -37), MAKE_VAL_COMPLEX(64, -5), MAKE_VAL_COMPLEX(27, 66), MAKE_VAL_COMPLEX(-8, -11), MAKE_VAL_COMPLEX(36, -43), MAKE_VAL_COMPLEX(34, -47), MAKE_VAL_COMPLEX(-20, 55), MAKE_VAL_COMPLEX(-30, -45), MAKE_VAL_COMPLEX(-33, -50), MAKE_VAL_COMPLEX(99, 12), MAKE_VAL_COMPLEX(-6, 43), MAKE_VAL_COMPLEX(-62, 37), MAKE_VAL_COMPLEX(10, 63), MAKE_VAL_COMPLEX(-36, -31) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_560, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_560", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_560() */
/* END TEST 560 */

/*
 *  BEGIN TEST 561
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_561( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(134, 52), MAKE_VAL_COMPLEX(38, -11), MAKE_VAL_COMPLEX(-14, 25), MAKE_VAL_COMPLEX(-32, -10), MAKE_VAL_COMPLEX(7, -1), MAKE_VAL_COMPLEX(-100, 58), MAKE_VAL_COMPLEX(11, 36), MAKE_VAL_COMPLEX(-20, -13), MAKE_VAL_COMPLEX(25, -15), MAKE_VAL_COMPLEX(31, -3), MAKE_VAL_COMPLEX(-43, -7), MAKE_VAL_COMPLEX(-23, -14), MAKE_VAL_COMPLEX(0, -27), MAKE_VAL_COMPLEX(-33, -20), MAKE_VAL_COMPLEX(23, -9) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, -1);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_561, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_561", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_561() */
/* END TEST 561 */

/*
 *  BEGIN TEST 562
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_562( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(12, 14), MAKE_VAL_COMPLEX(-76, -32), MAKE_VAL_COMPLEX(70, -60), MAKE_VAL_COMPLEX(81, 52), MAKE_VAL_COMPLEX(-19, -8), MAKE_VAL_COMPLEX(-28, 109), MAKE_VAL_COMPLEX(19, -77), MAKE_VAL_COMPLEX(32, 94), MAKE_VAL_COMPLEX(-95, -40), MAKE_VAL_COMPLEX(-82, -39), MAKE_VAL_COMPLEX(52, -81), MAKE_VAL_COMPLEX(21, 67), MAKE_VAL_COMPLEX(-34, 102), MAKE_VAL_COMPLEX(47, 69), MAKE_VAL_COMPLEX(-71, -12) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 7);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_562, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_562", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_562() */
/* END TEST 562 */

/*
 *  BEGIN TEST 563
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_563( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_ROWMAJ, y528_VW );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(662, 394), MAKE_VAL_COMPLEX(25, 71), MAKE_VAL_COMPLEX(-95, -109), MAKE_VAL_COMPLEX(70, -148), MAKE_VAL_COMPLEX(-8, 24), MAKE_VAL_COMPLEX(-404, 410), MAKE_VAL_COMPLEX(-91, 37), MAKE_VAL_COMPLEX(133, 3), MAKE_VAL_COMPLEX(-80, 60), MAKE_VAL_COMPLEX(-40, 104), MAKE_VAL_COMPLEX(-274, -298), MAKE_VAL_COMPLEX(53, -27), MAKE_VAL_COMPLEX(159, 85), MAKE_VAL_COMPLEX(53, -111), MAKE_VAL_COMPLEX(-8, 88) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-5, -1);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, -8);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_563, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_563", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_563() */
/* END TEST 563 */

/*
 *  BEGIN TEST 564
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_564( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-58, 24), MAKE_VAL_COMPLEX(46, -46), MAKE_VAL_COMPLEX(-48, 2), MAKE_VAL_COMPLEX(-50, 39), MAKE_VAL_COMPLEX(-11, -50), MAKE_VAL_COMPLEX(167, -29), MAKE_VAL_COMPLEX(34, 34), MAKE_VAL_COMPLEX(-46, -49), MAKE_VAL_COMPLEX(-51, -64), MAKE_VAL_COMPLEX(-117, 27), MAKE_VAL_COMPLEX(66, 70), MAKE_VAL_COMPLEX(51, 49) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(6, 0);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_564, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_564", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_564() */
/* END TEST 564 */

/*
 *  BEGIN TEST 565
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_565( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-62, -34), MAKE_VAL_COMPLEX(49, 31), MAKE_VAL_COMPLEX(-42, -22), MAKE_VAL_COMPLEX(-85, -19), MAKE_VAL_COMPLEX(31, -65), MAKE_VAL_COMPLEX(-107, 34), MAKE_VAL_COMPLEX(-19, 36), MAKE_VAL_COMPLEX(31, -71), MAKE_VAL_COMPLEX(16, -66), MAKE_VAL_COMPLEX(62, 58), MAKE_VAL_COMPLEX(-16, 80), MAKE_VAL_COMPLEX(-21, 66) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(4, 5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_565, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_565", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_565() */
/* END TEST 565 */

/*
 *  BEGIN TEST 566
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=0, beta=general>
 */
static void
test_MatMult_566( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-30, 70), MAKE_VAL_COMPLEX(10, -60), MAKE_VAL_COMPLEX(-30, 50), MAKE_VAL_COMPLEX(-15, 85), MAKE_VAL_COMPLEX(-65, -35), MAKE_VAL_COMPLEX(15, -35), MAKE_VAL_COMPLEX(45, 5), MAKE_VAL_COMPLEX(-75, -15), MAKE_VAL_COMPLEX(-85, -5), MAKE_VAL_COMPLEX(40, 50), MAKE_VAL_COMPLEX(100, 0), MAKE_VAL_COMPLEX(75, 5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(5, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_566, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_566", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_566() */
/* END TEST 566 */

/*
 *  BEGIN TEST 567
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=general, beta=general>
 */
static void
test_MatMult_567( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	oski_value_t* y_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-102, -74), MAKE_VAL_COMPLEX(6, -28), MAKE_VAL_COMPLEX(-134, -78), MAKE_VAL_COMPLEX(-131, -107), MAKE_VAL_COMPLEX(143, -119), MAKE_VAL_COMPLEX(225, -825), MAKE_VAL_COMPLEX(27, 19), MAKE_VAL_COMPLEX(49, -47), MAKE_VAL_COMPLEX(-41, -137), MAKE_VAL_COMPLEX(-472, 666), MAKE_VAL_COMPLEX(52, 144), MAKE_VAL_COMPLEX(1, 67) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(2, -6);
	oski_value_t beta = MAKE_VAL_COMPLEX(7, 9);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_567, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_567", y_true, y_val, y528_LEN*y528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_567() */
/* END TEST 567 */

/*
 *  BEGIN TEST 568
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_568( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-32, -106), MAKE_VAL_COMPLEX(83, 7), MAKE_VAL_COMPLEX(-58, 72), MAKE_VAL_COMPLEX(-95, -22), MAKE_VAL_COMPLEX(19, 3), MAKE_VAL_COMPLEX(8, -177), MAKE_VAL_COMPLEX(5, 72), MAKE_VAL_COMPLEX(-46, -73), MAKE_VAL_COMPLEX(89, 6), MAKE_VAL_COMPLEX(83, 17), MAKE_VAL_COMPLEX(47, 140), MAKE_VAL_COMPLEX(-35, -47), MAKE_VAL_COMPLEX(18, -90), MAKE_VAL_COMPLEX(-51, -44), MAKE_VAL_COMPLEX(67, -5) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, -5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_568, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_568", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_568() */
/* END TEST 568 */

/*
 *  BEGIN TEST 569
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_569( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(20, 90), MAKE_VAL_COMPLEX(-3, 33), MAKE_VAL_COMPLEX(-22, -12), MAKE_VAL_COMPLEX(11, -40), MAKE_VAL_COMPLEX(1, 7), MAKE_VAL_COMPLEX(30, 61), MAKE_VAL_COMPLEX(-31, 10), MAKE_VAL_COMPLEX(18, -31), MAKE_VAL_COMPLEX(11, 44), MAKE_VAL_COMPLEX(3, 31), MAKE_VAL_COMPLEX(-109, -56), MAKE_VAL_COMPLEX(17, -27), MAKE_VAL_COMPLEX(26, -18), MAKE_VAL_COMPLEX(5, -34), MAKE_VAL_COMPLEX(9, 23) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(1, -3);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_569, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_569", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_569() */
/* END TEST 569 */

/*
 *  BEGIN TEST 570
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_570( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-6, 0), MAKE_VAL_COMPLEX(24, -12), MAKE_VAL_COMPLEX(0, 30), MAKE_VAL_COMPLEX(-30, 9), MAKE_VAL_COMPLEX(6, -3), MAKE_VAL_COMPLEX(-21, -30), MAKE_VAL_COMPLEX(15, 21), MAKE_VAL_COMPLEX(-30, -12), MAKE_VAL_COMPLEX(30, -15), MAKE_VAL_COMPLEX(27, -12), MAKE_VAL_COMPLEX(9, 30), MAKE_VAL_COMPLEX(-21, -9), MAKE_VAL_COMPLEX(-18, -30), MAKE_VAL_COMPLEX(-27, -3), MAKE_VAL_COMPLEX(18, -15) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 0);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_570, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_570", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_570() */
/* END TEST 570 */

/*
 *  BEGIN TEST 571
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_571( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-78, -374), MAKE_VAL_COMPLEX(32, -72), MAKE_VAL_COMPLEX(18, 38), MAKE_VAL_COMPLEX(-55, 87), MAKE_VAL_COMPLEX(1, -13), MAKE_VAL_COMPLEX(-43, -283), MAKE_VAL_COMPLEX(70, -12), MAKE_VAL_COMPLEX(-34, 58), MAKE_VAL_COMPLEX(-19, -101), MAKE_VAL_COMPLEX(7, -57), MAKE_VAL_COMPLEX(363, 235), MAKE_VAL_COMPLEX(-40, 62), MAKE_VAL_COMPLEX(-28, 40), MAKE_VAL_COMPLEX(-4, 74), MAKE_VAL_COMPLEX(-7, -45) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(4, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-3, 5);
	int err;

	err = oski_MatMult( A, OP_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_571, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_571", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_571() */
/* END TEST 571 */

/*
 *  BEGIN TEST 572
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=1, beta=general>
 */
static void
test_MatMult_572( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(72, 32), MAKE_VAL_COMPLEX(71, 23), MAKE_VAL_COMPLEX(-68, 52), MAKE_VAL_COMPLEX(-78, -38), MAKE_VAL_COMPLEX(16, 7), MAKE_VAL_COMPLEX(72, -64), MAKE_VAL_COMPLEX(-12, 63), MAKE_VAL_COMPLEX(-22, -75), MAKE_VAL_COMPLEX(74, 26), MAKE_VAL_COMPLEX(69, 34), MAKE_VAL_COMPLEX(-149, 103), MAKE_VAL_COMPLEX(-18, -48), MAKE_VAL_COMPLEX(40, -76), MAKE_VAL_COMPLEX(-32, -51), MAKE_VAL_COMPLEX(60, 11) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, -6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_572, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_572", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_572() */
/* END TEST 572 */

/*
 *  BEGIN TEST 573
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=-1, beta=general>
 */
static void
test_MatMult_573( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-92, -40), MAKE_VAL_COMPLEX(25, -31), MAKE_VAL_COMPLEX(28, 48), MAKE_VAL_COMPLEX(-34, 28), MAKE_VAL_COMPLEX(8, -9), MAKE_VAL_COMPLEX(-102, -64), MAKE_VAL_COMPLEX(34, 27), MAKE_VAL_COMPLEX(-62, -5), MAKE_VAL_COMPLEX(46, -36), MAKE_VAL_COMPLEX(37, -38), MAKE_VAL_COMPLEX(139, 9), MAKE_VAL_COMPLEX(-40, -10), MAKE_VAL_COMPLEX(-60, -48), MAKE_VAL_COMPLEX(-54, 5), MAKE_VAL_COMPLEX(20, -37) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-5, 2);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_573, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_573", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_573() */
/* END TEST 573 */

/*
 *  BEGIN TEST 574
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=0, beta=general>
 */
static void
test_MatMult_574( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(16, 12), MAKE_VAL_COMPLEX(-88, -16), MAKE_VAL_COMPLEX(60, -80), MAKE_VAL_COMPLEX(98, 36), MAKE_VAL_COMPLEX(-22, -4), MAKE_VAL_COMPLEX(-4, 122), MAKE_VAL_COMPLEX(2, -86), MAKE_VAL_COMPLEX(56, 92), MAKE_VAL_COMPLEX(-110, -20), MAKE_VAL_COMPLEX(-96, -22), MAKE_VAL_COMPLEX(36, -98), MAKE_VAL_COMPLEX(38, 66), MAKE_VAL_COMPLEX(-12, 116), MAKE_VAL_COMPLEX(66, 62), MAKE_VAL_COMPLEX(-78, 4) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(0, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 6);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_574, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_574", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_574() */
/* END TEST 574 */

/*
 *  BEGIN TEST 575
 *  Value type: complex
 *  A : <MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED>
 *  x : <'x528', LAYOUT_COLMAJ>
 *  y : <'y528', LAYOUT_COLMAJ>
 *  Kernel : <op=OP_CONJ_TRANS, alpha=general, beta=general>
 */
static void
test_MatMult_575( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A528_ptr, A528_ind, A528_val,
		A528_M, A528_N, SHARE_INPUTMAT,
		5, MAT_TRI_LOWER, MAT_UNIT_DIAG_IMPLICIT, INDEX_ZERO_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( y528_val,
		y528_LEN * y528_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		y528_LEN, y528_VW, LAYOUT_COLMAJ, y528_LEN );
	oski_value_t* y_val = array_Duplicate( x528_val,
		x528_LEN * x528_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		x528_LEN, x528_VW, LAYOUT_COLMAJ, x528_LEN );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-784, 318), MAKE_VAL_COMPLEX(-87, 79), MAKE_VAL_COMPLEX(74, -148), MAKE_VAL_COMPLEX(103, -96), MAKE_VAL_COMPLEX(-20, 0), MAKE_VAL_COMPLEX(-462, 269), MAKE_VAL_COMPLEX(-28, -29), MAKE_VAL_COMPLEX(5, 70), MAKE_VAL_COMPLEX(-1, 12), MAKE_VAL_COMPLEX(-88, -4), MAKE_VAL_COMPLEX(395, -1030), MAKE_VAL_COMPLEX(-13, -9), MAKE_VAL_COMPLEX(-122, 114), MAKE_VAL_COMPLEX(-30, 45), MAKE_VAL_COMPLEX(-68, 16) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-6, 7);
	oski_value_t beta = MAKE_VAL_COMPLEX(8, 4);
	int err;

	err = oski_MatMult( A, OP_CONJ_TRANS, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_575, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_575", y_true, y_val, x528_LEN*x528_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_575() */
/* END TEST 575 */

#define A576_M 5    /* # of rows */
#define A576_N 5    /* # of cols */
#define A576_NNZ 12  /* # of stored non-zeros */
static oski_index_t A576_ptr[A576_N+1] = {1, 4, 5, 8, 11, 13}; /* end row pointers for A576 */
static oski_index_t A576_ind[A576_NNZ] = {2, 3, 4, 1, 1, 4, 5, 1, 3, 5, 3, 4}; /* end column indices pointers for A576 */
static oski_value_t A576_val[A576_NNZ] = {MAKE_VAL_COMPLEX(6, 10), MAKE_VAL_COMPLEX(-5, -6), MAKE_VAL_COMPLEX(10, 3), MAKE_VAL_COMPLEX(6, 10), MAKE_VAL_COMPLEX(-5, -6), MAKE_VAL_COMPLEX(6, -7), MAKE_VAL_COMPLEX(0, -6), MAKE_VAL_COMPLEX(10, 3), MAKE_VAL_COMPLEX(6, -7), MAKE_VAL_COMPLEX(-10, -10), MAKE_VAL_COMPLEX(0, -6), MAKE_VAL_COMPLEX(-10, -10)}; /* end stored non-zero values for A576 */

#define x576_LEN 5   /* vector length */
#define x576_VW  3   /* vector width */
static oski_value_t x576_val[x576_LEN*x576_VW] = {MAKE_VAL_COMPLEX(3, -9), MAKE_VAL_COMPLEX(-6, 3), MAKE_VAL_COMPLEX(-7, -6), MAKE_VAL_COMPLEX(-10, -6), MAKE_VAL_COMPLEX(8, -10), MAKE_VAL_COMPLEX(-3, 3), MAKE_VAL_COMPLEX(1, 7), MAKE_VAL_COMPLEX(9, -1), MAKE_VAL_COMPLEX(-7, 10), MAKE_VAL_COMPLEX(-1, 9), MAKE_VAL_COMPLEX(-3, 0), MAKE_VAL_COMPLEX(5, -4), MAKE_VAL_COMPLEX(10, 5), MAKE_VAL_COMPLEX(-2, 7), MAKE_VAL_COMPLEX(9, 1)}; /* data */

#define y576_LEN 5   /* vector length */
#define y576_VW  3   /* vector width */
static oski_value_t y576_val[y576_LEN*y576_VW] = {MAKE_VAL_COMPLEX(10, 7), MAKE_VAL_COMPLEX(3, 8), MAKE_VAL_COMPLEX(-2, -2), MAKE_VAL_COMPLEX(-7, -10), MAKE_VAL_COMPLEX(-6, 2), MAKE_VAL_COMPLEX(-2, 9), MAKE_VAL_COMPLEX(-2, 10), MAKE_VAL_COMPLEX(9, 10), MAKE_VAL_COMPLEX(2, 3), MAKE_VAL_COMPLEX(7, -2), MAKE_VAL_COMPLEX(2, -7), MAKE_VAL_COMPLEX(4, 9), MAKE_VAL_COMPLEX(0, 4), MAKE_VAL_COMPLEX(1, 5), MAKE_VAL_COMPLEX(-10, 6)}; /* data */

/*
 *  BEGIN TEST 576
 *  Value type: complex
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=1, beta=general>
 */
static void
test_MatMult_576( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(25, -147), MAKE_VAL_COMPLEX(104, -61), MAKE_VAL_COMPLEX(101, -33), MAKE_VAL_COMPLEX(65, 21), MAKE_VAL_COMPLEX(-50, -14), MAKE_VAL_COMPLEX(65, -105), MAKE_VAL_COMPLEX(70, 28), MAKE_VAL_COMPLEX(113, -1), MAKE_VAL_COMPLEX(20, -54), MAKE_VAL_COMPLEX(45, -229), MAKE_VAL_COMPLEX(31, -110), MAKE_VAL_COMPLEX(-63, -101), MAKE_VAL_COMPLEX(162, -90), MAKE_VAL_COMPLEX(48, -34), MAKE_VAL_COMPLEX(10, 76) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-1, -5);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_576, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_576", y_true, y_val, y576_LEN*y576_VW );

	oski_DestroyVecView( y );
	oski_DestroyVecView( x );
	oski_DestroyMat( A );
	array_Destroy( y_val );
	array_Destroy( x_val );
} /* end test_MatMult_576() */
/* END TEST 576 */

/*
 *  BEGIN TEST 577
 *  Value type: complex
 *  A : <MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED>
 *  x : <'x576', LAYOUT_ROWMAJ>
 *  y : <'y576', LAYOUT_ROWMAJ>
 *  Kernel : <op=OP_NORMAL, alpha=-1, beta=general>
 */
static void
test_MatMult_577( void )
{
	oski_matrix_t A = oski_CreateMatCSC( A576_ptr, A576_ind, A576_val,
		A576_M, A576_N, SHARE_INPUTMAT,
		5, MAT_SYMM_FULL, MAT_DIAG_EXPLICIT, INDEX_ONE_BASED,
		INDEX_SORTED, INDEX_UNIQUE );
	oski_value_t* x_val = array_Duplicate( x576_val,
		x576_LEN * x576_VW );
	oski_vecview_t x = oski_CreateMultiVecView( x_val,
		x576_LEN, x576_VW, LAYOUT_ROWMAJ, x576_VW );
	oski_value_t* y_val = array_Duplicate( y576_val,
		y576_LEN * y576_VW );
	oski_vecview_t y = oski_CreateMultiVecView( y_val,
		y576_LEN, y576_VW, LAYOUT_ROWMAJ, y576_VW );
	const oski_value_t y_true[] = {MAKE_VAL_COMPLEX(-98, 81), MAKE_VAL_COMPLEX(-120, -6), MAKE_VAL_COMPLEX(-87, 51), MAKE_VAL_COMPLEX(-19, 66), MAKE_VAL_COMPLEX(100, 4), MAKE_VAL_COMPLEX(-40, 35), MAKE_VAL_COMPLEX(-44, -106), MAKE_VAL_COMPLEX(-175, -88), MAKE_VAL_COMPLEX(-33, 28), MAKE_VAL_COMPLEX(-103, 238), MAKE_VAL_COMPLEX(-54, 164), MAKE_VAL_COMPLEX(40, 25), MAKE_VAL_COMPLEX(-158, 58), MAKE_VAL_COMPLEX(-51, -7), MAKE_VAL_COMPLEX(76, -114) };
	oski_value_t alpha = MAKE_VAL_COMPLEX(-1, 0);
	oski_value_t beta = MAKE_VAL_COMPLEX(-7, 4);
	int err;

	err = oski_MatMult( A, OP_NORMAL, alpha, x, beta, y );
	if( err != 0 ) { OSKI_ERR(test_MatMult_577, err); exit( 1 ); }

	array_CompareElems( "test_MatMult_577", y_true