/*****************************************************************************
 * Copyright (c) 2014-2020 OpenRCT2 developers
 *
 * For a complete list of all authors, please refer to contributors.md
 * Interested in contributing? Visit https://github.com/OpenRCT2/OpenRCT2
 *
 * OpenRCT2 is licensed under the GNU General Public License version 3.
 *****************************************************************************/

#include "../../common.h"
#include "../../interface/Viewport.h"
#include "../../paint/Paint.h"
#include "../../paint/Supports.h"
#include "../../world/Entity.h"
#include "../Track.h"
#include "../TrackPaint.h"
#include "../Vehicle.h"

/** rct2: 0x008A2ABC */
static void paint_enterprise_structure(
    paint_session* session, const Ride* ride, int8_t xOffset, int8_t yOffset, uint16_t height, const TrackElement& trackElement)
{
    height += 7;

    const TileElement* savedTileElement = static_cast<const TileElement*>(session->CurrentlyDrawnItem);
    rct_ride_entry* rideEntry = get_ride_entry(ride->subtype);
    Vehicle* vehicle = nullptr;

    if (rideEntry == nullptr)
    {
        return;
    }

    uint32_t baseImageId = rideEntry->vehicles[0].base_image_id;

    if (ride->lifecycle_flags & RIDE_LIFECYCLE_ON_TRACK && ride->vehicles[0] != SPRITE_INDEX_NULL)
    {
        session->InteractionType = ViewportInteractionItem::Entity;
        vehicle = GetEntity<Vehicle>(ride->vehicles[0]);
        session->CurrentlyDrawnItem = vehicle;
    }

    uint32_t imageOffset = trackElement.GetDirectionWithOffset(session->CurrentRotation);
    if (vehicle != nullptr)
    {
        imageOffset = (vehicle->Pitch << 2) + (((vehicle->sprite_direction >> 3) + session->CurrentRotation) % 4);
    }

    uint32_t imageColourFlags = session->TrackColours[SCHEME_MISC];
    if (imageColourFlags == IMAGE_TYPE_REMAP)
    {
        imageColourFlags = SPRITE_ID_PALETTE_COLOUR_2(ride->vehicle_colours[0].Body, ride->vehicle_colours[0].Trim);
    }

    uint32_t imageId = (baseImageId + imageOffset) | imageColourFlags;
    PaintAddImageAsParent(session, imageId, { xOffset, yOffset, height }, { 24, 24, 48 }, { 0, 0, height });

    rct_drawpixelinfo* dpi = &session->DPI;

    if (dpi->zoom_level <= 0 && imageOffset < 12 && ride->lifecycle_flags & RIDE_LIFECYCLE_ON_TRACK && vehicle != nullptr)
    {
        for (int32_t i = 0; i < 15; i++)
        {
            if (vehicle->num_peeps <= i)
            {
                break;
            }

            uint32_t peepFrameOffset = ((imageOffset % 4) * 4 + (i * 4) % 15) & 0x0F;
            uint32_t ax = (imageOffset & 0xFFFFFFFC) << 2;
            imageId = (baseImageId + 196 + peepFrameOffset + ax) | SPRITE_ID_PALETTE_COLOUR_1(vehicle->peep_tshirt_colours[i]);
            PaintAddImageAsChild(session, imageId, xOffset, yOffset, 24, 24, 48, height, 0, 0, height);
        }
    }

    session->CurrentlyDrawnItem = savedTileElement;
    session->InteractionType = ViewportInteractionItem::Ride;
}

/** rct2: 0x008A1584 */
static void paint_enterprise(
    paint_session* session, const Ride* ride, uint8_t trackSequence, uint8_t direction, int32_t height,
    const TrackElement& trackElement)
{
    if (ride == nullptr)
        return;

    trackSequence = track_map_4x4[direction][trackSequence];

    int32_t edges = edges_4x4[trackSequence];

    wooden_a_supports_paint_setup(session, direction & 1, 0, height, session->TrackColours[SCHEME_MISC]);

    StationObject* stationObject = ride_get_station_object(ride);
    track_paint_util_paint_floor(session, edges, session->TrackColours[SCHEME_TRACK], height, floorSpritesCork, stationObject);

    track_paint_util_paint_fences(
        session, edges, session->MapPosition, trackElement, ride, session->TrackColours[SCHEME_TRACK], height, fenceSpritesRope,
        session->CurrentRotation);

    switch (trackSequence)
    {
        case 5:
            paint_enterprise_structure(session, ride, 16, 16, height, trackElement);
            break;
        case 6:
            paint_enterprise_structure(session, ride, 16, -16, height, trackElement);
            break;
        case 10:
            paint_enterprise_structure(session, ride, -16, -16, height, trackElement);
            break;
        case 9:
            paint_enterprise_structure(session, ride, -16, 16, height, trackElement);
            break;

        case 0:
            paint_enterprise_structure(session, ride, 48, 48, height, trackElement);
            break;
        case 3:
            paint_enterprise_structure(session, ride, 48, -48, height, trackElement);
            break;
        case 15:
            paint_enterprise_structure(session, ride, -48, -48, height, trackElement);
            break;
        case 12:
            paint_enterprise_structure(session, ride, -48, 48, height, trackElement);
            break;

        case 7:
            paint_enterprise_structure(session, ride, 16, -48, height, trackElement);
            break;
        case 11:
            paint_enterprise_structure(session, ride, -16, -48, height, trackElement);
            break;
        case 14:
            paint_enterprise_structure(session, ride, -48, -16, height, trackElement);
            break;
        case 13:
            paint_enterprise_structure(session, ride, -48, 16, height, trackElement);
            break;
    }

    int32_t cornerSegments = 0;
    switch (trackSequence)
    {
        case 0:
            cornerSegments = SEGMENT_B4 | SEGMENT_C8 | SEGMENT_CC;
            break;
        case 3:
            cornerSegments = SEGMENT_CC | SEGMENT_BC | SEGMENT_D4;
            break;
        case 12:
            cornerSegments = SEGMENT_C8 | SEGMENT_B8 | SEGMENT_D0;
            break;
        case 15:
            cornerSegments = SEGMENT_D0 | SEGMENT_C0 | SEGMENT_D4;
            break;
    }
    paint_util_set_segment_support_height(session, cornerSegments, height + 2, 0x20);
    paint_util_set_segment_support_height(session, SEGMENTS_ALL & ~cornerSegments, 0xFFFF, 0);
    paint_util_set_general_support_height(session, height + 160, 0x20);
}

/**
 * rct2: 0x008A13B4
 */
TRACK_PAINT_FUNCTION get_track_paint_function_enterprise(int32_t trackType)
{
    if (trackType != TrackElemType::FlatTrack4x4)
    {
        return nullptr;
    }

    return paint_enterprise;
}
