/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.hosted.code;

import com.oracle.svm.core.annotate.Uninterruptible;
import com.oracle.svm.core.option.HostedOptionKey;
import com.oracle.svm.core.util.UserError;
import com.oracle.svm.hosted.meta.HostedMethod;
import java.util.Collection;
import java.util.Set;
import java.util.TreeSet;
import org.graalvm.compiler.graph.Node;
import org.graalvm.compiler.nodes.Invoke;
import org.graalvm.compiler.nodes.StructuredGraph;
import org.graalvm.compiler.nodes.java.AbstractNewObjectNode;
import org.graalvm.compiler.options.Option;
import org.graalvm.nativeimage.c.function.CFunction;
import org.graalvm.nativeimage.c.function.InvokeCFunctionPointer;

public final class UninterruptibleAnnotationChecker {
    private final Collection<HostedMethod> methodCollection;
    private final Set<String> violations;

    public UninterruptibleAnnotationChecker(Collection<HostedMethod> methodCollection) {
        this.methodCollection = methodCollection;
        this.violations = new TreeSet<String>();
    }

    public void check() {
        this.checkUninterruptibleOverrides();
        this.checkUninterruptibleCallees();
        this.checkUninterruptibleCallers();
        this.checkUninterruptibleAllocations();
        if (!this.violations.isEmpty()) {
            String message = "Found " + this.violations.size() + " violations of @Uninterruptible usage:";
            for (String violation : this.violations) {
                message = message + System.lineSeparator() + violation;
            }
            throw UserError.abort(message, new Object[0]);
        }
    }

    private void checkUninterruptibleOverrides() {
        for (HostedMethod method : this.methodCollection) {
            Uninterruptible methodAnnotation = method.getAnnotation(Uninterruptible.class);
            if (methodAnnotation == null) continue;
            for (HostedMethod impl : method.getImplementations()) {
                Uninterruptible implAnnotation = impl.getAnnotation(Uninterruptible.class);
                if (implAnnotation != null) {
                    if (methodAnnotation.callerMustBe() != implAnnotation.callerMustBe()) {
                        this.violations.add("callerMustBe: " + method.format("%H.%n(%p)") + " != " + impl.format("%H.%n(%p)"));
                    }
                    if (methodAnnotation.calleeMustBe() == implAnnotation.calleeMustBe()) continue;
                    this.violations.add("calleeMustBe: " + method.format("%H.%n(%p)") + " != " + impl.format("%H.%n(%p)"));
                    continue;
                }
                this.violations.add("method " + method.format("%H.%n(%p)") + " is annotated but " + impl.format("%H.%n(%p) is not"));
            }
        }
    }

    private void checkUninterruptibleCallees() {
        if (Options.PrintUninterruptibleCalleeDOTGraph.getValue().booleanValue()) {
            System.out.println("/* DOT */ digraph uninterruptible {");
        }
        for (HostedMethod caller : this.methodCollection) {
            HostedMethod callee;
            Uninterruptible callerAnnotation = caller.getAnnotation(Uninterruptible.class);
            StructuredGraph graph = caller.compilationInfo.getGraph();
            if (callerAnnotation == null) continue;
            if (callerAnnotation.calleeMustBe()) {
                if (graph == null) continue;
                for (Invoke invoke : graph.getInvokes()) {
                    callee = (HostedMethod)invoke.callTarget().targetMethod();
                    if (Options.PrintUninterruptibleCalleeDOTGraph.getValue().booleanValue()) {
                        UninterruptibleAnnotationChecker.printDotGraphEdge(caller, callee);
                    }
                    if (UninterruptibleAnnotationChecker.isNotInterruptible(callee)) continue;
                    this.violations.add("Unannotated callee: " + callee.format("%H.%n(%p)") + " called by annotated caller " + caller.format("%H.%n(%p)"));
                }
                continue;
            }
            if (graph == null) continue;
            for (Invoke invoke : graph.getInvokes()) {
                callee = (HostedMethod)invoke.callTarget().targetMethod();
                if (!Options.PrintUninterruptibleCalleeDOTGraph.getValue().booleanValue()) continue;
                UninterruptibleAnnotationChecker.printDotGraphEdge(caller, callee);
            }
        }
        if (Options.PrintUninterruptibleCalleeDOTGraph.getValue().booleanValue()) {
            System.out.println("/* DOT */ }");
        }
    }

    private void checkUninterruptibleCallers() {
        for (HostedMethod caller : this.methodCollection) {
            Uninterruptible callerAnnotation = caller.getAnnotation(Uninterruptible.class);
            StructuredGraph graph = caller.compilationInfo.getGraph();
            if (callerAnnotation != null || graph == null) continue;
            for (Invoke invoke : graph.getInvokes()) {
                HostedMethod callee = (HostedMethod)invoke.callTarget().targetMethod();
                if (!UninterruptibleAnnotationChecker.isCallerMustBe(callee)) continue;
                this.violations.add("Unannotated caller: " + caller.format("%H.%n(%p)") + " calls annotated callee " + callee.format("%H.%n(%p)"));
            }
        }
    }

    private void checkUninterruptibleAllocations() {
        for (HostedMethod method : this.methodCollection) {
            Uninterruptible methodAnnotation = method.getAnnotation(Uninterruptible.class);
            StructuredGraph graph = method.compilationInfo.getGraph();
            if (methodAnnotation == null || graph == null) continue;
            for (Node node : graph.getNodes()) {
                if (!(node instanceof AbstractNewObjectNode)) continue;
                this.violations.add("Annotated method: " + method.format("%H.%n(%p)") + " allocates.");
            }
        }
    }

    private static boolean isNotInterruptible(HostedMethod method) {
        return UninterruptibleAnnotationChecker.isUninterruptible(method) || UninterruptibleAnnotationChecker.isNoTransitionCFunction(method);
    }

    private static boolean isUninterruptible(HostedMethod method) {
        return method.getAnnotation(Uninterruptible.class) != null;
    }

    private static boolean isCallerMustBe(HostedMethod method) {
        Uninterruptible uninterruptibleAnnotation = method.getAnnotation(Uninterruptible.class);
        return uninterruptibleAnnotation != null && uninterruptibleAnnotation.callerMustBe();
    }

    private static boolean isCalleeMustBe(HostedMethod method) {
        Uninterruptible uninterruptibleAnnotation = method.getAnnotation(Uninterruptible.class);
        return uninterruptibleAnnotation != null && uninterruptibleAnnotation.calleeMustBe();
    }

    private static boolean isNoTransitionCFunction(HostedMethod method) {
        CFunction cfunctionAnnotation = method.getAnnotation(CFunction.class);
        InvokeCFunctionPointer invokeCFunctionPointerAnnotation = method.getAnnotation(InvokeCFunctionPointer.class);
        return cfunctionAnnotation != null && cfunctionAnnotation.transition() == CFunction.Transition.NO_TRANSITION || invokeCFunctionPointerAnnotation != null && invokeCFunctionPointerAnnotation.transition() == CFunction.Transition.NO_TRANSITION;
    }

    private static void printDotGraphEdge(HostedMethod caller, HostedMethod callee) {
        String callerColor = " [color=black]";
        String calleeColor = " [color=black]";
        if (UninterruptibleAnnotationChecker.isUninterruptible(caller)) {
            callerColor = " [color=blue]";
            if (!UninterruptibleAnnotationChecker.isCalleeMustBe(caller)) {
                callerColor = " [color=orange]";
            }
        }
        if (UninterruptibleAnnotationChecker.isUninterruptible(callee)) {
            calleeColor = " [color=blue]";
            if (!UninterruptibleAnnotationChecker.isCalleeMustBe(callee)) {
                calleeColor = " [color=purple]";
            }
        } else {
            calleeColor = " [color=red]";
        }
        if (UninterruptibleAnnotationChecker.isNoTransitionCFunction(callee)) {
            calleeColor = " [color=green]";
        }
        System.out.println("/* DOT */    " + caller.format("<%h.%n>") + callerColor);
        System.out.println("/* DOT */    " + callee.format("<%h.%n>") + calleeColor);
        System.out.println("/* DOT */    " + caller.format("<%h.%n>") + " -> " + callee.format("<%h.%n>") + calleeColor);
    }

    public static class Options {
        @Option(help={"Print (to stderr) a DOT graph of the @Uninterruptible annotations."})
        public static final HostedOptionKey<Boolean> PrintUninterruptibleCalleeDOTGraph = new HostedOptionKey<Boolean>(false);
    }
}

