/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.svm.core.genscavenge;

import com.oracle.svm.core.MemoryWalker;
import com.oracle.svm.core.annotate.AlwaysInline;
import com.oracle.svm.core.annotate.Uninterruptible;
import com.oracle.svm.core.genscavenge.AlignedHeapChunk;
import com.oracle.svm.core.genscavenge.Generation;
import com.oracle.svm.core.genscavenge.GreyObjectsWalker;
import com.oracle.svm.core.genscavenge.HeapChunk;
import com.oracle.svm.core.genscavenge.HeapImpl;
import com.oracle.svm.core.genscavenge.HeapPolicy;
import com.oracle.svm.core.genscavenge.HeapVerifier;
import com.oracle.svm.core.genscavenge.ObjectHeaderImpl;
import com.oracle.svm.core.genscavenge.Space;
import com.oracle.svm.core.genscavenge.SpaceVerifier;
import com.oracle.svm.core.genscavenge.ThreadLocalAllocation;
import com.oracle.svm.core.genscavenge.UnalignedHeapChunk;
import com.oracle.svm.core.heap.ObjectVisitor;
import com.oracle.svm.core.log.Log;
import org.graalvm.nativeimage.Platform;
import org.graalvm.nativeimage.Platforms;
import org.graalvm.word.Pointer;
import org.graalvm.word.UnsignedWord;
import org.graalvm.word.WordFactory;

final class YoungGeneration
extends Generation {
    private final Space eden = new Space("edenSpace", true, 0);
    private final Space[] survivorFromSpaces;
    private final Space[] survivorToSpaces;
    private final GreyObjectsWalker[] survivorGreyObjectsWalkers;
    private final int maxSurvivorSpaces = HeapPolicy.getMaxSurvivorSpaces();

    @Platforms(value={Platform.HOSTED_ONLY.class})
    YoungGeneration(String name) {
        super(name);
        this.survivorFromSpaces = new Space[this.maxSurvivorSpaces];
        this.survivorToSpaces = new Space[this.maxSurvivorSpaces];
        this.survivorGreyObjectsWalkers = new GreyObjectsWalker[this.maxSurvivorSpaces];
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            this.survivorFromSpaces[i] = new Space("Survivor-" + (i + 1) + " From", true, i + 1);
            this.survivorToSpaces[i] = new Space("Survivor-" + (i + 1) + " To", false, i + 1);
            this.survivorGreyObjectsWalkers[i] = new GreyObjectsWalker();
        }
    }

    @Uninterruptible(reason="Called from uninterruptible code.", mayBeInlined=true)
    void tearDown() {
        ThreadLocalAllocation.tearDown();
        this.eden.tearDown();
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            this.survivorFromSpaces[i].tearDown();
            this.survivorToSpaces[i].tearDown();
        }
    }

    @Override
    public boolean walkObjects(ObjectVisitor visitor) {
        ThreadLocalAllocation.disableAndFlushForAllThreads();
        if (!this.getEden().walkObjects(visitor)) {
            return false;
        }
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            if (!this.survivorFromSpaces[i].walkObjects(visitor)) {
                return false;
            }
            if (this.survivorToSpaces[i].walkObjects(visitor)) continue;
            return false;
        }
        return true;
    }

    @Override
    public Log report(Log log, boolean traceHeapChunks) {
        log.string("[Young generation: ").indent(true);
        log.string("[Eden: ").indent(true);
        this.getEden().report(log, traceHeapChunks);
        log.redent(false).string("]").newline();
        log.string("[Survivors: ").indent(true);
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            this.survivorFromSpaces[i].report(log, traceHeapChunks).newline();
            this.survivorToSpaces[i].report(log, traceHeapChunks);
            if (i >= this.maxSurvivorSpaces - 1) continue;
            log.newline();
        }
        log.redent(false).string("]").redent(false).string("]");
        return log;
    }

    Space getEden() {
        return this.eden;
    }

    Space getSurvivorToSpaceAt(int index) {
        assert (index >= 0 && index < this.maxSurvivorSpaces) : "Survivor index should be between 0 and NumberOfSurvivorSpaces";
        return this.survivorToSpaces[index];
    }

    Space getSurvivorFromSpaceAt(int index) {
        assert (index >= 0 && index < this.maxSurvivorSpaces) : "Survivor index should be between 0 and NumberOfSurvivorSpaces";
        return this.survivorFromSpaces[index];
    }

    private GreyObjectsWalker getSurvivorGreyObjectsWalker(int index) {
        return this.survivorGreyObjectsWalkers[index];
    }

    @Override
    @AlwaysInline(value="GC performance")
    protected Object promoteObject(Object original, UnsignedWord header) {
        if (ObjectHeaderImpl.isAlignedHeader(original, header)) {
            AlignedHeapChunk.AlignedHeader originalChunk = AlignedHeapChunk.getEnclosingChunk(original);
            Space originalSpace = HeapChunk.getSpace(originalChunk);
            if (originalSpace.isFromSpace()) {
                return this.promoteAlignedObject(original, originalSpace);
            }
        } else {
            assert (ObjectHeaderImpl.isUnalignedHeader(original, header));
            UnalignedHeapChunk.UnalignedHeader chunk = UnalignedHeapChunk.getEnclosingChunk(original);
            Space originalSpace = HeapChunk.getSpace(chunk);
            if (originalSpace.isFromSpace()) {
                this.promoteUnalignedObject(chunk, originalSpace);
            }
        }
        return original;
    }

    private void releaseSurvivorSpaces(boolean isFromSpace) {
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            if (isFromSpace) {
                this.getSurvivorFromSpaceAt(i).releaseChunks();
                continue;
            }
            this.getSurvivorToSpaceAt(i).releaseChunks();
        }
    }

    void releaseSpaces() {
        this.getEden().releaseChunks();
        this.releaseSurvivorSpaces(true);
        if (HeapImpl.getHeapImpl().getGCImpl().isCompleteCollection()) {
            this.releaseSurvivorSpaces(false);
        }
    }

    void swapSpaces() {
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            assert (this.getSurvivorFromSpaceAt(i).isEmpty()) : "Survivor fromSpace should be empty.";
            this.getSurvivorFromSpaceAt(i).absorb(this.getSurvivorToSpaceAt(i));
        }
    }

    @Override
    protected boolean verify(HeapVerifier.Occasion occasion) {
        boolean result = true;
        HeapImpl heap = HeapImpl.getHeapImpl();
        HeapVerifier heapVerifier = heap.getHeapVerifier();
        SpaceVerifier spaceVerifier = heapVerifier.getSpaceVerifier();
        spaceVerifier.initialize(this.getEden());
        if (occasion.equals((Object)HeapVerifier.Occasion.AFTER_COLLECTION)) {
            if (spaceVerifier.containsChunks()) {
                result = false;
                heapVerifier.getWitnessLog().string("[YoungGeneration.verify:").string("  eden space contains chunks after collection").string("]").newline();
            }
        } else if (!spaceVerifier.verify()) {
            result = false;
            heapVerifier.getWitnessLog().string("[YoungGeneration.verify:").string("  eden space fails to verify").string("]").newline();
        }
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            spaceVerifier.initialize(this.survivorFromSpaces[i]);
            if (!spaceVerifier.verify()) {
                result = false;
                heapVerifier.getWitnessLog().string("[YoungGeneration.verify:").string("  survivor to space fails to verify").string("]").newline();
            }
            spaceVerifier.initialize(this.survivorToSpaces[i]);
            if (!spaceVerifier.verify()) {
                result = false;
                heapVerifier.getWitnessLog().string("[YoungGeneration.verify:").string("  survivor to space fails to verify").string("]").newline();
            }
            if (occasion.equals((Object)HeapVerifier.Occasion.DURING_COLLECTION) || !spaceVerifier.containsChunks()) continue;
            result = false;
            heapVerifier.getWitnessLog().string("[YoungGeneration.verify:").string("  survivor to space contains chunks").string("]").newline();
        }
        return result;
    }

    boolean slowlyFindPointer(Pointer p) {
        if (HeapVerifier.slowlyFindPointerInSpace(this.getEden(), p)) {
            return true;
        }
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            if (HeapVerifier.slowlyFindPointerInSpace(this.getSurvivorFromSpaceAt(i), p)) {
                return true;
            }
            if (!HeapVerifier.slowlyFindPointerInSpace(this.getSurvivorToSpaceAt(i), p)) continue;
            return true;
        }
        return false;
    }

    boolean walkHeapChunks(MemoryWalker.Visitor visitor) {
        if (this.getEden().walkHeapChunks(visitor)) {
            for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
                if (this.getSurvivorFromSpaceAt(i).walkHeapChunks(visitor) && this.getSurvivorToSpaceAt(i).walkHeapChunks(visitor)) continue;
                return false;
            }
            return true;
        }
        return false;
    }

    void prepareForPromotion() {
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            assert (this.getSurvivorToSpaceAt(i).isEmpty()) : "SurvivorToSpace should be empty.";
            this.getSurvivorGreyObjectsWalker(i).setScanStart(this.getSurvivorToSpaceAt(i));
        }
    }

    boolean scanGreyObjects() {
        int i;
        Log trace = Log.noopLog().string("[YoungGeneration.scanGreyObjects:");
        boolean needScan = false;
        for (i = 0; i < this.maxSurvivorSpaces; ++i) {
            if (!this.getSurvivorGreyObjectsWalker(i).haveGreyObjects()) continue;
            needScan = true;
            break;
        }
        if (!needScan) {
            return false;
        }
        for (i = 0; i < this.maxSurvivorSpaces; ++i) {
            trace.string("[Scanning survivor-").signed(i).string("]").newline();
            this.getSurvivorGreyObjectsWalker(i).walkGreyObjects();
        }
        trace.string("]").newline();
        return true;
    }

    UnsignedWord getSurvivorChunkUsedBytes() {
        UnsignedWord usedBytes = (UnsignedWord)WordFactory.zero();
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            usedBytes = usedBytes.add(this.survivorFromSpaces[i].getChunkBytes());
            usedBytes = usedBytes.add(this.survivorToSpaces[i].getChunkBytes());
        }
        return usedBytes;
    }

    UnsignedWord getChunkUsedBytes() {
        return this.getEden().getChunkBytes().add(this.getSurvivorChunkUsedBytes());
    }

    UnsignedWord getSurvivorObjectBytes() {
        UnsignedWord usedObjectBytes = (UnsignedWord)WordFactory.zero();
        for (int i = 0; i < this.maxSurvivorSpaces; ++i) {
            usedObjectBytes = usedObjectBytes.add(this.survivorFromSpaces[i].getObjectBytes());
            usedObjectBytes = usedObjectBytes.add(this.survivorToSpaces[i].getObjectBytes());
        }
        return usedObjectBytes;
    }

    UnsignedWord getObjectBytes() {
        return this.getEden().getObjectBytes().add(this.getSurvivorObjectBytes());
    }

    boolean contains(Object object) {
        return HeapChunk.getSpace(HeapChunk.getEnclosingHeapChunk(object)).isYoungSpace();
    }

    @AlwaysInline(value="GC performance")
    private Object promoteAlignedObject(Object original, Space originalSpace) {
        assert (ObjectHeaderImpl.isAlignedObject(original));
        assert (originalSpace.isEdenSpace() || originalSpace.isSurvivorSpace()) : "Should be Eden or survivor.";
        assert (originalSpace.isFromSpace()) : "must not be called for other objects";
        if (originalSpace.getAge() < this.maxSurvivorSpaces) {
            int age = originalSpace.getNextAgeForPromotion();
            Space toSpace = this.getSurvivorToSpaceAt(age - 1);
            return toSpace.promoteAlignedObject(original, originalSpace);
        }
        return HeapImpl.getHeapImpl().getOldGeneration().promoteAlignedObject(original, originalSpace);
    }

    @AlwaysInline(value="GC performance")
    private void promoteUnalignedObject(UnalignedHeapChunk.UnalignedHeader originalChunk, Space originalSpace) {
        assert (originalSpace.isFromSpace()) : "must not be called for other objects";
        if (originalSpace.getAge() < this.maxSurvivorSpaces) {
            int age = originalSpace.getNextAgeForPromotion();
            Space toSpace = this.getSurvivorToSpaceAt(age - 1);
            toSpace.promoteUnalignedHeapChunk(originalChunk, originalSpace);
        } else {
            HeapImpl.getHeapImpl().getOldGeneration().promoteUnalignedChunk(originalChunk, originalSpace);
        }
    }
}

