#ifndef __POLYGLOT_ISOLATE_H
#define __POLYGLOT_ISOLATE_H

#include <stdbool.h>
#include <stdint.h>
#include <graal_isolate.h>
#include <polyglot_types.h>

#if defined(__cplusplus)
extern "C" {
#endif

/*
 * Create a new isolate, considering the passed parameters (which may be NULL).
 * Returns poly_ok on success, or a poly_generic_failure value on failure.
 * On success, the current thread is attached to the created isolate, and the
 * address of the isolate structure is written to the passed pointer.
 * Every thread starts with a default handle scope. This scope is released when
 * the thread is detached.
 */
poly_status poly_create_isolate(const poly_isolate_params* params, poly_isolate* isolate, poly_thread* thread);

/*
 * Attaches the current thread to the passed isolate.
 * On failure, returns poly_generic_failure. On success, writes the address of the
 * created isolate thread structure to the passed pointer and returns poly_ok.
 * If the thread has already been attached, the call succeeds and also provides
 * the thread's isolate thread structure.
 */
poly_status poly_attach_thread(poly_isolate isolate, poly_thread* thread);

/*
 * Given an isolate to which the current thread is attached, returns the address of
 * the thread's associated isolate thread structure.  If the current thread is not
 * attached to the passed isolate or if another error occurs, returns NULL.
 */
poly_thread poly_get_current_thread(poly_isolate isolate);

/*
 * Given an isolate thread structure, determines to which isolate it belongs and
 * returns the address of its isolate structure.  If an error occurs, returns NULL
 * instead.
 */
poly_isolate poly_get_isolate(poly_thread thread);

/*
 * Detaches the passed isolate thread from its isolate and discards any state or
 * context that is associated with it. At the time of the call, no code may still
 * be executing in the isolate thread's context.
 * Returns poly_ok on success, or poly_generic_failure on failure.
 */
poly_status poly_detach_thread(poly_thread thread);

/*
 * In the isolate of the passed isolate thread, detach all those threads that were
 * externally started (not within Java, which includes the "main thread") and were
 * attached to the isolate afterwards. Afterwards, all threads that were started
 * within Java undergo a regular shutdown process, followed by the tear-down of the
 * entire isolate, which detaches the current thread and discards the objects,
 * threads, and any other state or context associated with the isolate.
 * None of the manually attached threads targeted by this function may be executing
 * Java code at the time when this function is called or at any point in the future
 * or this will cause entirely undefined (and likely fatal) behavior.
 * Returns poly_ok on success, or poly_generic_failure on failure.
 */
poly_status poly_detach_all_threads_and_tear_down_isolate(poly_thread thread);

/*
 * Tears down the passed isolate, waiting for any attached threads to detach from
 * it, then discards the isolate's objects, threads, and any other state or context
 * that is associated with it.
 * Returns poly_ok on success, or poly_generic_failure on failure.
 */
poly_status poly_tear_down_isolate(poly_thread thread);

#if defined(__cplusplus)
}
#endif
#endif
