# -*- python -*-

# This software was produced by NIST, an agency of the U.S. government,
# and by statute is not subject to copyright in the United States.
# Recipients of this software assume all responsibilities associated
# with its operation, modification and maintenance. However, to
# facilitate maintenance we ask that before distributing modified
# versions of this software, you first contact the authors at
# oof_manager@nist.gov. 

## Usage:

#  In the top oof2 directory (the one containing this file) type
#  something like this:
#    python setup.py install     # installs in the default location
#    python setup.py install --prefix=/some/other/place
#    python setup.py [build [--debug]] install --prefix ...
#  The flags --3D, --enable-mpi, --enable-petsc, and --enable-devel
#  can occur anywhere after 'setup.py' in the command line.

# Required version numbers of required external libraries.  These
# aren't used explicitly in this file, but they are used in the DIR.py
# files that are execfile'd here.

GTK_VERSION = "3.22.0"
MAGICK_VERSION = "6.0"
CAIROMM_VERSION = "1.12" # Don't know what the earliest acceptable version is.
PANGO_VERSION = "1.40"
PANGOCAIRO_VERSION = "1.40"
PYGOBJECT_VERSION = "3.22"
OOFCANVAS_VERSION = "1.0"

# The make_dist script edits the following line when a distribution is
# built.  Don't change it by hand.  On the git master branch,
# "0.0.0" is replaced by the version number.
version_from_make_dist = "2.2.2"

###############################

import distutils.core
from distutils.command import build
from distutils.command import build_ext
from distutils.command import build_py
from distutils.command import clean
from distutils.command import build_scripts
from distutils import errors
from distutils import log
from distutils.dir_util import remove_tree, mkpath
from distutils.sysconfig import get_config_var

log.set_verbosity(2)

## oof2installlib redefines the distutils install_lib command so that
## it runs install_name_tool on Macs. 
import oof2installlib

import shlib # adds build_shlib and install_shlib to the distutils command set
from shlib import build_shlib

from oof2setuputils import run_swig, find_file, extend_path

import os
import shlex
import stat
import string
import sys
import subprocess
import tempfile
import time
from types import *

# Tell distutils that .C is a C++ file suffix.
from distutils.ccompiler import CCompiler
CCompiler.language_map['.C'] = 'c++'

DIRFILE = "DIR.py"                      # oof subdirectory manifest files
SWIGCFILEEXT = 'cmodule.C'              # suffix for swig output files
SWIGINSTALLDIR = "SWIG"


##############

# readDIRs() walks through the oof2 source directory looking for
# DIR.py files, reads the files, and fills in the CLibInfo objects.
# CLibInfo categorize all of the source files except for the python
# files for pure python modules.

# DIR.py files contain the following python variables.  All are optional.

# dirname: The name of the directory. It's actually not used.

# clib: the name of the library to be built from the C and C++ files
# in the directory.  The library will be called lib<name>.<suffix>,
# where suffix is system dependent.  More than one DIR.py file can use
# the same name.

# cfiles: a list of the names of all of the C and C++ files in the
# directory that need to be compiled to form lib<name>.

# hfiles: a list of the names of the header files that go with the C
# and C++ files.

# swigfiles: a list of the names of the swig input files in the
# directory.  Swig-generated C++ code will be compiled but *not*
# included in lib<name>.  Each swig input file will create a separate
# python-loadable module which will *link* to lib<name>.

# swigpyfiles: a list of the names of python files that are included
# in swig output files.

# clib_order: an integer specifying the order in which the libraries
# must be built.  Later libraries may link to earlier ones.  This
# linking is done by setting clib.externalLibs in the set_clib_flags
# function in a DIR.py file.

# set_clib_flags: a function which may be called to set compilation
# and linker flags for building the library.  Its argument is a
# CLibInfo object.  The includeDirs, externalLibDirs, and externalLibs
# members of the object may be modified by set_clib_flags.

# subdirs: a list of subdirectories that should be processed.  Each
# subdirectory must have its own DIR.py file.

allCLibs = {}
purepyfiles = []

def getCLibInfo(name):
    try:
        return allCLibs[name]
    except KeyError:
        clib = allCLibs[name] = CLibInfo(name)
        return clib
                
class CLibInfo:
    def __init__(self, name):
        allCLibs[name] = self
        self.libname = name
        self.dirdata = {'cfiles': [],   # *.[Cc] -- c and c++ source code
                        'hfiles': [],   # *.h    -- c and c++ header files
                        'swigfiles': [], # *.swg  -- swig source code
                        'swigpyfiles': [], # *.spy  -- python included in swig
                        }
        self.pkgs = set()          # packages to run pkg-config on
        self.externalLibs = []
        self.externalLibDirs = []
        self.includeDirs = []
        self.extra_link_args = []
        self.extra_compile_args = []
        self.extensionObjs = None
        self.ordering = None

    def add_pkg(self, pkg):
        self.pkgs.add(pkg)

    def run_pkg_config(self):
        # Running pkg-config on all of the packages at the same time
        # reduces redundancy in the resulting compiler argument
        # list. This is called after all of the DIR.py files have been
        # read, so that self.pkgs contains all of the third party
        # packages that will be used.

        if not self.pkgs:
            return
        
        # Add tcmalloc if desired and available.
        # TODO? Should we be modifying the compiler args as well as
        # the link args when using tcmalloc with gcc?  The github page
        # for an older version of gperftools said:
        ## NOTE: When compiling with programs with gcc, that you plan
        ## to link with libtcmalloc, it's safest to pass in the flags
        ## -fno-builtin-malloc -fno-builtin-calloc
        ## -fno-builtin-realloc -fno-builtin-free when compiling.  gcc
        ## makes some optimizations assuming it is using its own,
        ## built-in malloc; that assumption obviously isn't true with
        ## tcmalloc.  In practice, we haven't seen any problems with
        ## this, but the expected risk is highest for users who
        ## register their own malloc hooks with tcmalloc (using
        ## gperftools/malloc_hook.h). The risk is lowest for folks who
        ## use tcmalloc_minimal (or, of course, who pass in the above
        ## flags :-) ).
        # The gperftools README file (as of 01-11-2022) contains only
        # the second half of that paragraph ("gcc makes some
        # assumption..." to "the above flags") but never actually says
        # what the flags are or what the problems might be.

        pkgs = self.pkgs.copy()
        if (USE_TCMALLOC and
            subprocess.call(["pkg-config", "--exists", "libtcmalloc"])):
            pkgs.add("libtcmalloc")
        
        # Run pkg-config --cflags.
        cmd = "pkg-config --cflags %s" % string.join(pkgs)
        log.info("%s: %s", self.libname, cmd)
        flags = subprocess.check_output(shlex.split(cmd))
        for flag in flags.split():
            if flag[:2] == "-I":
                self.includeDirs.append(flag[2:])
            else:
                self.extra_compile_args.append(flag)

        # Run pkg-config --libs.
        cmd = "pkg-config --libs %s" % string.join(pkgs)
        log.info("%s: %s", self.libname, cmd)
        flags = subprocess.check_output(shlex.split(cmd))
        for flag in flags.split():
            if flag[:2] == "-l":
                self.externalLibs.append(flag[2:])
            elif flag[:2] == '-L':
                self.externalLibDirs.append(flag[2:])
            else:
                self.extra_link_args.append(flag)
        
    # Parse the file lists in a DIR.py file.  The file has been read
    # already, and its key,list pairs are in dirdict.  Only the data
    # relevant to CLibInfo is dealt with here.  The rest is handled
    # by readDIRs().
    def extractData(self, srcdir, dirdict):
        for key in self.dirdata.keys():
            try:
                value = dirdict[key]
                del dirdict[key]
            except KeyError:
                pass
            else:
                for filename in value:
                    self.dirdata[key].append(os.path.join(srcdir, filename))
        try:
            flagFunc = dirdict['set_clib_flags']
            del dirdict['set_clib_flags']
        except KeyError:
            pass
        else:
            flagFunc(self)

            
        try:
            self.ordering = dirdict['clib_order']
            del dirdict['clib_order']
        except KeyError:
            pass

    def get_extensions(self):
        if self.extensionObjs is None:
            self.extensionObjs = []
            for swigfile in self.dirdata['swigfiles']:
                # The file name is of the form "./SRC/dirs/something.swg"
                # Strip the "./SRC/" and the suffix.
                basename = os.path.splitext(
                    os.path.relpath(swigfile, './SRC'))[0]

                modulename = os.path.splitext(basename + SWIGCFILEEXT)[0]
                sourcename = os.path.join(swigroot, basename+SWIGCFILEEXT)
                
                extension = distutils.core.Extension(
                    name = os.path.join(OOFNAME,"ooflib", SWIGINSTALLDIR,
                                        modulename),
                    language = 'c++',
                    sources = [sourcename],
                    define_macros = platform['macros'],
                    extra_compile_args = self.extra_compile_args + \
                        platform['extra_compile_args'],
                    include_dirs = self.includeDirs + platform['incdirs'],
                    library_dirs = self.externalLibDirs + platform['libdirs'],
                    libraries = [fixLibName(self.libname)] + self.externalLibs,
                                                        # + platform['libs'],
                    extra_link_args = self.extra_link_args + \
                        platform['extra_link_args']
                    )

                self.extensionObjs.append(extension)
        return self.extensionObjs

    def get_shlib(self):
        if self.dirdata['cfiles']:
            return build_shlib.SharedLibrary(
                self.libname,
                sources=self.dirdata['cfiles'],
                extra_compile_args=platform['extra_compile_args']+self.extra_compile_args,
                include_dirs=self.includeDirs + platform['incdirs'],
                libraries=self.externalLibs,# + platform['libs'],
                library_dirs=self.externalLibDirs +
                platform['libdirs'],
                extra_link_args=platform['extra_link_args'])

    # Find all directories containing at least one swig input file.  These
    # are used to create the swigged python packages.  This is done by
    # traversing the DIR.py files, so that random leftover .swg files in
    # strange places don't create packages, and so that modules can be
    # included conditionally by HAVE_XXX tests in DIR.py files.
    def find_swig_pkgs(self):
        pkgs = set()
        for swigfile in self.dirdata['swigfiles']:
            pkgs.add(os.path.split(swigfile)[0])
        # pkgs is a set of dirs containing swig files, relative to
        # the main OOF2 dir, eg, "./SRC/common".
        # Convert it to a list of dirs relative to the swigroot
        swigpkgs = []
        for pkg in pkgs:
            relpath = os.path.relpath(pkg, './SRC')
            relocated = os.path.normpath(
                os.path.join(OOFNAME, SWIGDIR, relpath))
            pkgname = relocated.replace('/', '.')
            swigpkgs.append(pkgname)
        return swigpkgs

# end class CLibInfo

def moduleSort(moduleA, moduleB):
    if moduleA.ordering is not None:
        if moduleB.ordering is not None:
            return cmp(moduleA.ordering, moduleB.ordering)
        return -1
    else:                               # moduleA.ordering is None
        if moduleB.ordering is not None:
            return 1
        return cmp(moduleA.name, moduleB.name)

def allFiles(key):
    hierlist = [lib.dirdata[key] for lib in allCLibs.values()]
    flatlist = []
    for sublist in hierlist:
        flatlist.extend(sublist)
    return flatlist


def readDIRs(srcdir):
    dirfile = os.path.join(srcdir, DIRFILE)
    if os.path.exists(dirfile):
        log.info("loading %s", dirfile)
        # dirfile defines variables whose names are the same as the
        # ModuleInfo.dirdata keys.  The variables contain lists of
        # file names.
        localdict = {}
        execfile(dirfile, globals(), localdict)
        # Now the variables and functions defined in dirfile are in localdict.
        try:
            dirname = localdict['dirname']
            del localdict['dirname']
        except KeyError:
            pass
        
        try:
            clib = localdict['clib']
            del localdict['clib']
        except KeyError:
            pass
        else:
            clibinfo = getCLibInfo(clib)
            clibinfo.extractData(srcdir, localdict)
            
        try:
            pyfiles = localdict['pyfiles']
            del localdict['pyfiles']
        except KeyError:
            pass
        else:
            for filename in pyfiles:
                purepyfiles.append(os.path.join(srcdir, filename))
        
        # dirfile also contains a list of subdirectories to process.
        try:
            subdirs = localdict['subdirs']
            del localdict['subdirs']
        except KeyError:
            pass
        else:
            # At this point, all args in localdict should have been processed.
            if len(localdict) > 0:
                log.warn("WARNING: unrecognized values %s in %s",
                         localdict.keys(), dirfile)
            for subdir in subdirs:
                readDIRs(os.path.join(srcdir, subdir))

##########

# Find all python packages and subpackages in a directory by looking
# for __init__.py files.

def find_pkgs():
    pkglist = []
    os.path.walk('SRC', _find_pkgs, pkglist)
    return pkglist

def _find_pkgs(pkglist, dirname, subdirs):
    if os.path.exists(os.path.join(dirname, '__init__.py')):
        pkglist.append(dirname)

##########

def swig_clibs(dry_run, force, debug, build_temp, with_swig=None):
    # First make sure that swig has been built.
    if with_swig is None:
        ## TODO 3.1: swig is installed inside the distutils
        ## build/temp* directory to avoid conflicts if oof is being
        ## built for multiple architectures on a shared file system.
        ## However, swig's .o file and other intermediate files
        ## (parser.cxx, parser.h, config.log, Makefiles, etc) are
        ## still in OOFSWIG/SWIG.  They'll have to be removed manually
        ## before building on a different architecture.  It would be
        ## better if they were in build/temp* too, but that might
        ## require modifying the Makefile.
        swigsrcdir = os.path.abspath('OOFSWIG')
        swigbuilddir = os.path.join(os.path.abspath(build_temp), 'swig-build')
        if not os.path.exists(swigbuilddir):
            os.mkdir(swigbuilddir)
        swigexec = os.path.join(swigbuilddir, 'bin', 'swig')
        if not os.path.exists(swigexec):
            log.info("Building swig")
            cwd = os.getcwd()
            try:
                os.chdir(swigsrcdir)
                cmd = "./configure --prefix=%s" % swigbuilddir
                log.info(cmd)
                if subprocess.call(shlex.split(cmd)):
                    log.error("Failed to configure swig.")
                    sys.exit(1)
                cmd = ["make"]
                log.info(cmd)
                if subprocess.call(cmd):
                    log.error("Failed to build swig")
                    sys.exit(1)
                cmd = ["make", "install"]
                log.info(cmd)
                if subprocess.call(cmd):
                    log.error("Failed to install swig")
                    sys.exit(1)
            finally:
                os.chdir(cwd)
    else:
        swigexec = with_swig
    srcdir = os.path.abspath('SRC')
    extra_args = platform['extra_swig_args']
    if debug:
        extra_args.append('-DDEBUG')
    for clib in allCLibs.values():
        for swigfile in clib.dirdata['swigfiles']:
            # run_swig requires a src dir and an input file path
            # relative to it.  The '+1' in the following line strips
            # off a '/', so that sfile doesn't look like an absolute
            # path.
            sfile = os.path.abspath(swigfile)[len(srcdir)+1:]
            run_swig(srcdir='SRC', swigfile=sfile, destdir=swigroot,
                     cext=SWIGCFILEEXT,
                     include_dirs = ['SRC'],
                     dry_run=dry_run,
                     extra_args=extra_args,
                     force=force,
                     with_swig=swigexec,
                     DIM_3=DIM_3
                     )

##########

# Get a file's modification time.  The time is returned as an integer.
# All we care about is that the integers are properly ordered.

def modification_time(phile):
    return os.stat(phile)[stat.ST_MTIME]

#########

# If we're building with python-dbg, the shared libraries that it
# builds will have a "_d" added to their names, and we need to
# know that in order to link to them.  SHLIB_EXT is either ".so"
# or "_d.so".  Unfortunately, the quotation marks are included.

_sfx = get_config_var("SHLIB_EXT").split('.')[0]
if _sfx[0] == '"':
    _sfx = _sfx[1:]

if _sfx:
    log.debug("Library suffix is %s", _sfx)

def fixLibName(libname):
    return libname + _sfx

def addOOFlibs(clib, *libnames):
    for libname in libnames:
        clib.externalLibs.append(fixLibName(libname))

#########

# Define subclasses of the distutils build_ext and build_shlib class.
# We need subclasses so that oofconfig.h can be created before the
# files are compiled, and so that makedepend can be run.
# oof_build_xxxx contains the routines that are being added to both
# build_ext and build_shlib.

_dependencies_checked = 0
class oof_build_xxxx:
    def make_oofconfig(self):
        cfgfilename = os.path.normpath(
            os.path.join(self.build_temp, 'SRC', 'oofconfig.h'))
        includedir = os.path.join('include', OOFNAME)
        self.distribution.data_files.append((includedir, [cfgfilename]))
        # If oofconfig.h already exists, don't recreate it unless
        # forced to.  It would require everything that depends on it
        # to be recompiled unnecessarily.
        if self.force or not os.path.exists(cfgfilename):
            log.info("creating %s", cfgfilename)
            if not self.dry_run:
                cmd = 'mkdir -p %s' % os.path.join(self.build_temp, 'SRC')
                log.info(cmd)
                if subprocess.call(shlex.split(cmd)):
                    log.error("Failed to make directory for %s" % cfgfilename)
                    sys.exit(1)
                cfgfile = open(cfgfilename, "w")
                print >> cfgfile, """\
// This file was created automatically by the oof2 setup script.
// Do not edit it.
// Re-run setup.py to change the options.
#ifndef OOFCONFIG_H
#define OOFCONFIG_H
                """
                if HAVE_PETSC:
                    print >> cfgfile, '#define HAVE_PETSC 1'
                if HAVE_MPI:
                    print >> cfgfile, '#define HAVE_MPI 1'
                if HAVE_OPENMP:
                    # HAVE_OPENMP allows us to override OpenMP flags,
                    # which may have been set in the c++ options of
                    # our dependencies.
                    print >> cfgfile, '#define HAVE_OPENMP'
                if DEVEL:
                    print >> cfgfile, '#define DEVEL ', DEVEL
                if NO_GUI:
                    print >> cfgfile, '#define NO_GUI 1'
                if ENABLE_SEGMENTATION:
                    print >> cfgfile, '#define ENABLE_SEGMENTATION'
                if NANOHUB:
                    print >> cfgfile, '#define NANOHUB'
                if DIM_3:
                    print >> cfgfile, '#define DIM 3'
                    print >> cfgfile, '#define DIM_3'
                else: # for good measure
                    print >> cfgfile, '#define DIM 2'
                if self.check_header('<sstream>'):
                    print >> cfgfile, '#define HAVE_SSTREAM'
                else:
                    print >> cfgfile, '// #define HAVE_SSTREAM'
                # Python pre-2.5 compatibility
                print >> cfgfile, """\
#include <Python.h>
#if PY_VERSION_HEX < 0x02050000 && !defined(PY_SSIZE_T_MIN)
typedef int Py_ssize_t;
#define PY_SSIZE_T_MAX INT_MAX
#define PY_SSIZE_T_MIN INT_MIN
#endif /* PY_VERSION_HEX check */
"""
                print >> cfgfile, "#endif"
                cfgfile.close()

    def check_header(self, headername):
        # Check that a c++ header file exists on the system.
        log.info("Testing for %s", headername)
        tmpfiled, tmpfilename = tempfile.mkstemp(suffix='.C')
        tmpfile = os.fdopen(tmpfiled, 'w')
        print >> tmpfile, """\
        #include %s
        int main(int, char**) { return 1; }
        """ % headername
        tmpfile.flush()
        try:
            try:
                ofiles = self.compiler.compile(
                    [tmpfilename],
                    extra_postargs=platform['extra_compile_args']
                    )
            except errors.CompileError:
                return 0
            ofile = ofiles[0]
            dir = os.path.split(ofile)[0]
            os.remove(ofiles[0])
            if dir:
                try:
                    os.removedirs(dir)
                except:
                    pass
            return 1
        finally:
            os.remove(tmpfilename)

    def find_dependencies(self):
        # distutils doesn't provide a makedepend-like facility, so we
        # have to do it ourselves.  makedepend is deprecated, so we
        # use "gcc -MM" and hope that gcc is available.  This routine
        # runs "gcc -MM" and constructs a dictionary listing the
        # dependencies of each .o file and each swig-generated .C
        # file.

        # TODO: Check for the existence of gcc and makedepend and use
        # the one that's available.

        # depdict[file] is a list of sources that the file depends
        # upon.
        depdict = {}

        # Run "gcc -MM" on the C++ files to detect dependencies.  "gcc
        # -MM" only prints the file name of the target, not its path
        # relative to the build directory, so we have to use the -MT
        # flag to specify the target.  That means that we can't
        # process more than one C++ file at a time.
        log.info("Finding dependencies for C++ files.")
        for phile in allFiles('cfiles'):
            ## Hack Alert.  We don't know the full paths to some of
            ## the system header files at this point.  The -MM flag is
            ## supposed to tell gcc to ignore the system headers, but
            ## apparently some versions still want to be able to find
            ## them, and fail when they don't.  So we use -MG, which
            ## tells gcc to add missing headers to the dependency
            ## list, and then we weed them out later.  At least this
            ## way, the "missing" headers don't cause errors.

            cmd = 'g++ -std=c++11 -MM -MG -MT %(target)s -ISRC -I%(builddir)s -I%(buildsrc)s %(file)s' \
              % {'file' : phile,
                 'target': os.path.splitext(phile)[0] + ".o",
                 'builddir' : self.build_temp,
                 'buildsrc' : os.path.join(self.build_temp, 'SRC')
                 }
            proc = subprocess.Popen(shlex.split(cmd), stdout=subprocess.PIPE,
                                    bufsize=4096)
            stdoutdata, stderrdata = proc.communicate()
            if stderrdata:
                log.error("Command failed: %s", cmd)
                log.error("%s", stderrdata)
                sys.exit(1)
            if not stdoutdata:
                log.error("Command failed, no data received: %s", cmd)
                sys.exit(1)
            # stdoutdata is a multiline string.  The first substring
            # is the name of the target file, followed by a colon.
            # The remaining substrings are the source files that the
            # target depends on, but there can also be line
            # continuation characters (backslashes) which must be
            # ignored.  It's even possible that the *first* line is
            # blank except for a backslash.
            files = [f for f in stdoutdata.split() if f != "\\"]
            target = files[0][:-1] # omit the colon
            realtarget = os.path.normpath(os.path.join(self.build_temp, target))
            for source in files[1:]:
                ## See Hack Alert, above.  Missing header files will
                ## all be outside of our directory hierarchy, so we
                ## just ignore any dependency that doesn't begin with
                ## "SRC/".
                if (source.startswith('SRC/') or
                    source.startswith(self.build_temp)):
                    depdict.setdefault(realtarget, []).append(source)

        # .C and.py files in the SWIG directory depend on those in the
        # SRC directory.  Run gcc -MM on the swig source files.
        log.info("Finding dependencies for .swg files.")
        for phile in allFiles('swigfiles'):
            cmd = 'g++ -std=c++11 -MM -MG -MT %(target)s -x c++ -I. -ISRC -I%(builddir)s %(file)s'\
              % {'file' : phile,
                 'target': os.path.splitext(phile)[0] + '.o',
                 'builddir' : self.build_temp
              }
            proc = subprocess.Popen(shlex.split(cmd), 
                                    stdout=subprocess.PIPE, bufsize=4096)
            stdoutdata, stderrdata = proc.communicate()
            if stderrdata:
                log.error("Command failed: %s", cmd)
                log.error(stderrdata)
                sys.exit(1)
            files = [f for f in stdoutdata.split() if f != "\\"]
            target = files[0][:-1]
            targetbase = os.path.splitext(target)[0]
            # On some systems, target begins with "SRC/".  On
            # others, it begins with "./SRC/".  Arrgh.  This
            # strips off either one.
            targetbase = targetbase.split("SRC/", 1)[1]
            targetc = os.path.normpath(
                os.path.join(swigroot, targetbase + SWIGCFILEEXT))
            targetpy = os.path.normpath(
                os.path.join(swigroot, targetbase + '.py'))
            for source in files[1:]:
                if (source.startswith('SRC/') or
                    source.startswith(self.build_temp)):
                    depdict.setdefault(targetc, []).append(source)
                    depdict.setdefault(targetpy,[]).append(source)

        ## Debugging:
        # def dumpdepdict(filename, depdict):
        #     print >> sys.stderr, "Dumping dependency information to", filename
        #     f = file(filename, "w")
        #     keys = depdict.keys()
        #     keys.sort()
        #     for target in keys:
        #         print >> f, target
        #         sources = depdict[target]
        #         sources.sort()
        #         for source in sources:
        #             print >> f, "   ", source
        #     f.close()
        # dumpdepdict("depdict", depdict)

        # Add in the implicit dependencies on the .swg files.
        for phile in allFiles('swigfiles'):
            # file is ./SRC/dir/whatver.swg
            base = os.path.splitext(phile)[0][4:] # dir/whatever
            cfile = os.path.normpath(os.path.join(swigroot,
                                                  base+SWIGCFILEEXT))
            pyfile = os.path.normpath(os.path.join(swigroot, base+'.py'))
            depdict.setdefault(cfile, []).append(phile)
            depdict.setdefault(pyfile, []).append(phile)
        # Add in the implicit dependencies on the .spy files.
        for underpyfile in allFiles('swigpyfiles'):
            relpath = os.path.relpath(underpyfile, './SRC')
            relocated = os.path.normpath(os.path.join(swigroot, relpath))
            # Replace .spy with .py
            pyfile = os.path.splitext(relocated)[0] + ".py"
            depdict.setdefault(pyfile, []).append(underpyfile)

        return depdict
    

    # Remove out-of-date target files.  We have to do this because
    # distutils for Python 2.5 and earlier checks the dates of the .C
    # and .o files, but doesn't check for any included .h files, so it
    # doesn't rebuild enough.  For 2.6 and later, it doesn't check
    # anything, and it rebuilds far too much. To fix that, we
    # monkeypatch _setup_compile from Python 2.5 as well as remove the
    # out of date target files.

    def clean_targets(self, depdict):
        outofdate = False
        if not self.dry_run:
            for target, sources in depdict.items():
                if os.path.exists(target):
                    targettime = modification_time(target)
                    sourcetime = max([modification_time(x) for x in sources])
                    if sourcetime > targettime:
                        os.remove(target)
                        log.info(
                            "clean_targets: Removed out-of-date target %s",
                            target)
                        outofdate = True
                else:
                    outofdate = True
        if outofdate:
            ## TODO: Remove the .so file.  I can't figure out how to
            ## find its name at this point, though.
            pass

    def clean_dependencies(self):
        global _dependencies_checked
        if not _dependencies_checked:
            # Read dependencies from a file unless MAKEDEPEND has been
            # defined by providing the --makedepend command line
            # option, or if the file doesn't exist.
            depfilename = os.path.join(self.build_temp, 'depend')
            if not MAKEDEPEND and os.path.exists(depfilename):
                locals = {}
                log.info("Loading dependencies from %s", depfilename)
                execfile(depfilename, globals(), locals)
                depdict = locals['depdict']
            else:
                depdict = self.find_dependencies()
                log.info("Saving dependencies in %s", depfilename)
                mkpath(self.build_temp)
                depfile = open(depfilename, "w")
                print >> depfile, "depdict=", depdict
                depfile.close()
            self.clean_targets(depdict)
            _dependencies_checked = True


# This does the swigging.
class oof_build_ext(build_ext.build_ext, oof_build_xxxx):
    description = "build the python extension modules for OOF2"
    user_options = build_ext.build_ext.user_options + [
        ('with-swig=', None, "specify the swig executable")]
    def initialize_options(self):
        self.with_swig = None
        build_ext.build_ext.initialize_options(self)
    def finalize_options(self):
        self.set_undefined_options('build', ('with_swig', 'with_swig'))
        build_ext.build_ext.finalize_options(self)
    # build_extensions is called by build_ext.run().
    def build_extensions(self):
        self.compiler.add_include_dir(os.path.join(self.build_temp, 'SRC'))
        self.compiler.add_include_dir('SRC')
        self.compiler.add_library_dir(self.build_lib)

        if self.debug:
            self.compiler.define_macro('DEBUG')
            # self.compiler.define_macro('Py_DEBUG')
            self.compiler.undefine_macro('NDEBUG')
        # Make the automatically generated .h files.
        self.make_oofconfig()
        # Run makedepend
        self.clean_dependencies()
        # Generate swigged .C and .py files
        swig_clibs(self.dry_run, self.force, self.debug, self.build_temp,
                   self.with_swig)
                                          
        # Build the swig extensions by calling the distutils base
        # class function
        build_ext.build_ext.build_extensions(self)

class oof_build_shlib(build_shlib.build_shlib, oof_build_xxxx):
    user_options = build_shlib.build_shlib.user_options + [
        ('with-swig=', None, "non-standard swig executable"),
        ('blas-libraries=', None, "libraries for blas and lapack"),
        ('blas-link-args=', None, "link arguments required for blas and lapack")
        ]
    def initialize_options(self):
        self.with_swig = None
        self.blas_libraries = None
        self.blas_link_args = None
        build_shlib.build_shlib.initialize_options(self)
    def finalize_options(self):
        self.set_undefined_options('build',
                                   ('with_swig', 'with_swig'),
                                   ('blas_libraries', 'blas_libraries'),
                                   ('blas_link_args', 'blas_link_args'),
                                   ('libraries', 'libraries'),
                                   ('library_dirs', 'library_dirs'))
        build_shlib.build_shlib.finalize_options(self)
    def build_libraries(self, libraries):
        self.make_oofconfig()
        self.clean_dependencies()
        self.compiler.add_include_dir(os.path.join(self.build_temp, 'SRC'))
        self.compiler.add_include_dir('SRC')
        if self.debug:
            self.compiler.define_macro('DEBUG')

        # The blas libs and arguments weren't actually put into the
        # SharedLibrary objects when they were created, because we
        # didn't know until now whether or not the user had provided
        # alternates.  It's time either to use the predefined values
        # from "platform" or to use the command line arguments.

        if self.blas_libraries is not None:
            blaslibs = string.split(self.blas_libraries)
        else:
            blaslibs = platform['blas_libs']
        if self.blas_link_args is not None:
            blasargs = string.split(self.blas_link_args)
        else:
            blasargs = platform['blas_link_args']
        extrablaslibs = self.check_extra_blaslibs(blaslibs, blasargs)
        blaslibs.extend(extrablaslibs)

        for library in libraries:
            library.libraries.extend(blaslibs)
            library.extra_link_args.extend(blasargs)

        build_shlib.build_shlib.build_libraries(self, libraries)

    def check_extra_blaslibs(self, blaslibs, linkargs):
        # Check to see if blas requires extra libraries to link
        # properly.  If it does, return a list of extra libraries.  If
        # it links without extra args, return [].  If it doesn't
        # link at all, raise an exception.  (This test is required
        # because different Linux distributions seem to build their
        # blas libraries differently, and we can't tell which
        # distribution we're using.)
        log.info("Testing if blas links correctly")
        # First create a temp directory to play in.
        tmpdir = tempfile.mkdtemp(dir=os.getcwd())
        tmpdirname = os.path.split(tmpdir)[1]
        # Create a file with dummy blas code in it.
        tmpfilename = os.path.join(tmpdirname, "blastest.C")
        tmpfile = open(tmpfilename, "w")
        print >> tmpfile, """\
        extern "C" {void dgemv_(char*, int*, int*, double*, double*, int*,
        double*, double*, double*, double*, int*);}
        int main(int argc, char **argv) {
        char c;
        int i;
        double x;
        dgemv_(&c, &i, &i, &x, &x, &i, &x, &x, &x, &x, &i);
        return 0;
        }
        """
        tmpfile.close()
        try:
            # Compile the dummy code.
            try:
                ofiles=self.compiler.compile(
                    [tmpfilename],
                    extra_postargs=platform['extra_compile_args'])
            except errors.CompileError:
                raise errors.DistutilsExecError("can't compile blas test")
            # Try linking without extra args
            try:
                self.compiler.link(
                    target_desc=self.compiler.EXECUTABLE,
                    objects=ofiles,
                    output_filename=tmpfilename[:-2],
                    library_dirs=platform['libdirs'],
                    libraries=blaslibs,
                    extra_preargs=linkargs,
		    target_lang='c++')
            except errors.LinkError:
                pass
            else:
                return []               # Extra args not needed
            # Try linking with -lg2c and -lgfortran
            for libname in ('g2c', 'gfortran'):
                try:
                    self.compiler.link(
                        target_desc=self.compiler.EXECUTABLE,
                        objects=ofiles,
                        output_filename=tmpfilename[:-2],
                        library_dirs=platform['libdirs'],
                        libraries=blaslibs+[libname],
                        extra_preargs=linkargs,
			target_lang='c++')
                except errors.LinkError:
                    pass
                else:
                    return [libname]   

        finally:
            # Clean out the temp directory
            remove_tree(tmpdirname)
        raise errors.DistutilsExecError("can't link blas!")


class oof_build(build.build):
    sep_by = " (separated by '%s')" % os.pathsep
    user_options = build.build.user_options + [
        ('with-swig=', None, "non-standard swig executable"),
        ('libraries=', None, 'external libraries to link with'),
        ('library-dirs=', None,
         "directories to search for external libraries" + sep_by),
        ('blas-libraries=', None, "libraries for blas and lapack"),
        ('blas-link-args=', None, "link args for blas and lapack")]
    def initialize_options(self):
        self.libraries = None
        self.library_dirs = None
        self.blas_libraries = None
        self.blas_link_args = None
        self.with_swig = None
        build.build.initialize_options(self)

    # override finalize_options in build.py in order to include the
    # dimension in the build directory.
    def finalize_options(self):
        if not DIM_3:
            dim = "2d"
        else:
            dim = "3d"
            
        plat_specifier = ".%s-%s-%s" % (build.get_platform(), sys.version[0:3],dim) 

        if self.build_purelib is None:
            self.build_purelib = os.path.join(self.build_base, 'lib')
        if self.build_platlib is None:
            self.build_platlib = os.path.join(self.build_base,
                                              'lib' + plat_specifier)

        if self.build_lib is None:
            if self.distribution.ext_modules:
                self.build_lib = self.build_platlib
            else:
                self.build_lib = self.build_purelib

        if self.build_temp is None:
            self.build_temp = os.path.join(self.build_base,
                                           'temp' + plat_specifier)
        if self.build_scripts is None:
            self.build_scripts = os.path.join(
                self.build_base, 'scripts-' + sys.version[0:3] + "-" + dim)

        try: #only in newer version of distutils
            if self.executable is None:
                self.executable = os.path.normpath(sys.executable)
        except AttributeError:
            pass


###################################################

## Modify the build_py command so that it creates oof2config.py.  The
## file is created in the build_lib directory so that it gets
## installed at the top level beside the oof2 or oof3d package.  We
## also need an init script in the oof2 or oof3d directory.

class oof_build_py(build_py.build_py):
    def run(self):
        # Tell oof2installlib which shared libraries to process.
        install_shlib = self.get_finalized_command('install_shlib')
        shared_libs = [lib.name for lib in install_shlib.shlibs]
        oof2installlib.shared_libs = shared_libs
        self.make_oof2config()
        self.make_toplevel_init()
        build_py.build_py.run(self) #this is where swigroot is copied
    def make_toplevel_init(self):
        initname = os.path.join(self.build_lib,OOFNAME,'__init__.py')
        initfile = open(initname,'w')
        initfile.close()
    def make_oof2config(self):
        cfgscriptname = os.path.join(self.build_lib, OOFNAME+'config.py')
        cfgscript = open(cfgscriptname, 'w')

        install = self.get_finalized_command('install')
        build_shlib = self.get_finalized_command('build_shlib')
        install_shlib = self.get_finalized_command('install_shlib')
      
        print >> cfgscript, 'root = "%s"' % os.path.abspath('.')
        print >> cfgscript, 'version = "%s"' % self.distribution.get_version()
        print >> cfgscript, 'prefix = "%s"' % install.prefix
        idirs = build_shlib.include_dirs + [
            os.path.abspath('SRC'),
            os.path.join(install.prefix, 'include', OOFNAME)
            ] + platform['incdirs']
        print >> cfgscript, 'swig_include = ', [os.path.abspath('SRC')]
        print >> cfgscript, 'extra_compile_args =', \
              platform['extra_compile_args']
        print >> cfgscript, 'include_dirs =', idirs
        print >> cfgscript, 'library_dirs =', [install_shlib.install_dir]
        shared_libs = [lib.name for lib in install_shlib.shlibs]
        print >> cfgscript, 'libraries =', shared_libs
        print >> cfgscript, 'extra_link_args =', platform['extra_link_args']
        print >> cfgscript, "import sys; sys.path.append(root)"
        cfgscript.close()


    def build_module (self, module, module_file, package):
        if type(package) is StringType:
            package = string.split(package, '.')
        elif type(package) not in (ListType, TupleType):
            raise TypeError, \
                  "'package' must be a string (dot-separated), list, or tuple"

        # Now put the module source file into the "build" area -- this is
        # easy, we just copy it somewhere under self.build_lib (the build
        # directory for Python source).
        # [Comment from distutils, not oof2]
        outfile = self.get_module_outfile(self.build_lib, package, module)

        # The next line is the only one that is different from the
        # original distutils.
        outfile = os.path.normpath(outfile.replace(SWIGDIR, SWIGINSTALLDIR))
        
        dir = os.path.dirname(outfile)
        self.mkpath(dir)
        return self.copy_file(module_file, outfile, preserve_mode=0)



###################################################

class oof_clean(clean.clean):
    user_options = clean.clean.user_options + [
        ('most', 'm', 'remove libraries and scripts, but not binary dist.'),
        ('swig', None, 'remove swig output files')]
    boolean_options = clean.clean.boolean_options + ['most', 'swig']

    def initialize_options(self):
        clean.clean.initialize_options(self)
        self.most = None
        self.swig = None
        
    def run(self):
        if self.most and not self.all:
            for d in [self.build_lib, self.build_scripts]:
                if os.path.exists(d):
                    remove_tree(d,dry_run=self.dry_run)
                else:
                    log.warn("'%s' does not exist -- can't clean it.", d)
        if (self.swig or self.all) and os.path.exists(swigroot):
            remove_tree(swigroot, dry_run=self.dry_run)
            swigsrcdir = os.path.abspath('OOFSWIG')
            log.info("Cleaning swig")
            cwd = os.getcwd()
            try:
                os.chdir(swigsrcdir)
                if subprocess.call(["make", "clean"]):
                    log.error("Failed to 'make clean' in swig directory.")
                    sys.exit(1)
            finally:
                os.chdir(cwd)

        clean.clean.run(self)
    
###################################################

def set_dirs():
    global swigroot, datadir, docdir
    swigroot = os.path.join('SRC', SWIGDIR)
    # Splitting and reassembling paths makes them portable to systems
    # that don't use '/' as the path separator.
    datadir = os.path.join(*DATADIR.split('/'))
    docdir = os.path.join(*DOCDIR.split('/'))

def get_global_args():
    # The --enable-xxxx flags in the command line have to be obtained
    # *before* we parse the DIR.py files, and the DIR.py files must be
    # parsed before calling distutils.core.setup (because the list of
    # source files comes from DIR.py).  But distutils.core.setup handles
    # the command line arguments, so we have to look for and remove the
    # --enable-xxxx flags here.

    # TODO: the more elegant way to do this would be to add a separate
    # distutils command that reads the DIR.py files and is always run
    # before any other command.  Then the --enable-xxxx flags could be
    # global distutils options, since they'd be processed *before*
    # DIR.py was read.  NO, that's not true.  DIR.py is read before
    # any distutils calls can possibly be made, because distutils.core
    # hasn't been called yet.

    global HAVE_MPI, HAVE_OPENMP, HAVE_PETSC, DEVEL, NO_GUI, \
        ENABLE_SEGMENTATION, MAKEDEPEND, PORTDIR, \
        DIM_3, DATADIR, DOCDIR, OOFNAME, SWIGDIR, NANOHUB, USE_TCMALLOC
    HAVE_MPI = _get_oof_arg('--enable-mpi')
    HAVE_PETSC = _get_oof_arg('--enable-petsc')
    DEVEL = _get_oof_arg('--enable-devel')
    NO_GUI = _get_oof_arg('--disable-gui')
    ENABLE_SEGMENTATION = _get_oof_arg('--enable-segmentation')
    DIM_3 = _get_oof_arg('--3D')
    MAKEDEPEND = _get_oof_arg('--makedepend')
    NANOHUB = _get_oof_arg('--nanoHUB')
    HAVE_OPENMP = _get_oof_arg('--enable-openmp')
    PORTDIR = _get_oof_arg('--port-dir', '/opt/local')
    USE_TCMALLOC = _get_oof_arg('--enable-tcmalloc')

    # The following determine some secondary installation directories.
    # They will be created within the main installation directory
    # specified by --prefix. 

    if not DIM_3:
        DATADIR = "share/oof2"
        DOCDIR = "share/oof2/doc"
        OOFNAME = "oof2"
        SWIGDIR = "SWIG2D"           # root dir for swig output, inside SRC
    else:
        DATADIR = "share/oof3d"
        DOCDIR = "share/oof3d/doc"
        OOFNAME = "oof3d"
        SWIGDIR = "SWIG3D"           # root dir for swig output, inside SRC


def _get_oof_arg(arg, default=None):
    # Search for an argument which begins like "arg" -- if found,
    # return the trailing portion if any, or 1 if none, and remove the
    # argument from sys.argv.
    for s in sys.argv:
        splits = s.split('=')
        if splits[0] == arg:
            sys.argv.remove(s)
            if len(splits) > 1:         # found an =
                return splits[1]
            return 1                    # just a plain arg
    return default                      # didn't find arg
        
platform = {}

def set_platform_values():
    ## Set platform-specific flags that don't specifically depend on
    ## OOF2 stuff.  They're stored in a dictionary just to keep things
    ## tidy.
    platform['extra_compile_args'] = []
    platform['macros'] = []
    platform['blas_libs'] = []
    platform['blas_link_args'] = []
    platform['libdirs'] = []
    platform['incdirs'] = [get_config_var('INCLUDEPY')]
    platform['extra_link_args'] = []
    platform['extra_swig_args'] = []

    if os.path.exists('/usr/local/lib'):
        platform['libdirs'].append('/usr/local/lib')
    # if os.path.exists('/usr/site/lib'):
    #     platform['libdirs'].append('/usr/site/lib')
    if os.path.exists('/usr/site/include'):
        platform['incdirs'].append('/usr/site/include')

    if sys.platform == 'darwin':
        platform['blas_link_args'].extend(['-framework', 'Accelerate'])
        platform['extra_link_args'].append('-headerpad_max_install_names')

        # If MacPorts was used to install dependencies, but we're not
        # actually building with MacPorts now, then the pkg-config
        # files for python-installed dependencies may not be in the
        # pkg-config search path.
        global PORTDIR
        if os.path.exists(PORTDIR):
            ## TODO: Having to encode such a long path here seems
            ## wrong.  If and when pkgconfig acquires a more robust
            ## way of finding its files, use it.
            pkgpath = os.path.join(PORTDIR, "Library/Frameworks/Python.framework/Versions/%d.%d/lib/pkgconfig/" % (sys.version_info[0], sys.version_info[1]))
            log.info("Adding %s to PKG_CONFIG_PATH", pkgpath)
            extend_path("PKG_CONFIG_PATH", pkgpath)

        # Enable C++11
        platform['extra_compile_args'].append('-Wno-c++11-extensions')
        platform['extra_compile_args'].append('-std=c++11')
        # After upgrading to High Sierra, deprecated register warnings
        # started coming from the python header files.
        platform['extra_compile_args'].append('-Wno-deprecated-register')
        if 'clang' in get_config_var('CC'):
            # If we're using clang, we want to suppress some warnings
            # about oddities in swig-generated code:
            platform['extra_compile_args'].append('-Wno-self-assign')
            
    elif sys.platform.startswith('linux'):
        # g2c isn't included here, because it's not always required.
        # We don't want to check whether or not it's required, either,
        # because the user might have provided a different blas
        # library on the command line.  The check is done later,just
        # before platform['blas_libs'] is used.
        platform['blas_libs'].extend(['lapack', 'blas', 'm'])
        # add -std=c++11 option to use c++11 standard
        platform['extra_compile_args'].append('-std=c++11')

    # ## Irix and cygwin args haven't been tested in years and are
    # ## certainly horribly out of date.
    # elif sys.platform[:4] == 'irix':
    #     platform['extra_compile_args'].append('-LANG:std')
    #     platform['extra_link_args'].append('-LANG:std')
    #     platform['blas_libs'].extend(['lapack', 'blas', 'ftn', 'm'])
    # elif sys.platform == 'cygwin':
    #     platform['blas_libs'].extend(['blas', 'lapack', 'm'])
    #     platform['libdirs'].append('/bin')

    # ## TODO: netbsd options may be out of date.  C++11 should be
    # ## enabled.
    # elif sys.platform[:6] == 'netbsd':
    #     platform['blas_libs'].extend(['lapack', 'blas', 'm'])
    #     platform['libdirs'].append('/usr/pkg/lib')

    if HAVE_OPENMP:
        platform['extra_compile_args'].append('-fopenmp')
        platform['extra_link_args'].append('-fopenmp') # needed?

    if HAVE_MPI:
        # mpi.h
        basedirs = ['/usr/local', '/usr', '/usr/lib', '/usr/site', '/sw']
        mpisubdirs = ['.', 'mpich', 'mpi']
        mpidirs = [os.path.join(bdir, 'include', mdir) for bdir in basedirs
                   for mdir in mpisubdirs]
        incdir = find_file('mpi.h', mpidirs)
        if not incdir:
            log.warn("Warning! Can't locate mpi.h!")
        else:
            if incdir not in platform['incdirs']:
                platform['incdirs'].append(incdir)
        # mpi++.h
        basedirs = ['/usr/local', '/usr', '/usr/lib', '/usr/site', '/sw']
        mpisubdirs = ['.', 'mpich', 'mpi', 'mpich/mpi2c++', 'mpi/mpi2c++']
        mpidirs = [os.path.join(bdir, 'include', mdir) for bdir in basedirs
                   for mdir in mpisubdirs]
        incdir = find_file('mpi++.h', mpidirs)
        if not incdir:
            log.warn("Warning! Can't locate mpi++.h!")
        else:
            if incdir not in platform['incdirs']:
                platform['incdirs'].append(incdir)

        platform['extra_swig_args'].append('-DHAVE_MPI')
        if sys.platform == 'darwin':
            # TODO: Need to know what Mac needs for mpi++
            # This may have to be changed or removed just as in MPI below
            platform['libs'].extend(['mpich', 'pmpich'])
        elif sys.platform.startswith('linux'):
            # libpmpich++.a
            basedirs = ['/usr/local', '/usr', '/usr/lib', '/usr/site', '/sw']
            mpisubdirs = ['.', 'mpich', 'mpi', 'mpich/lib', 'mpi/lib']
            mpidirs = [os.path.join(bdir, mdir) for bdir in basedirs
                       for mdir in mpisubdirs]
            libdir = find_file('libpmpich++.a', mpidirs)
            if not libdir:
                log.warn("Warning! Can't locate libpmpich++.a!")
            else:
                if libdir not in platform['libdirs']:
                    platform['libdirs'].append(libdir)
            #platform['libs'].extend(['pmpich++', 'mpich'])
        elif sys.platform[:4] == 'irix':
            pass

#=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=#

# In Python 2.6 and 2.7, distutils doesn't check for an existing .o
# file, and recompiles everything whether it needs to or not.  Since
# we remove all of the out-of-date .o files, we don't want to
# recompile all of them.  Here we monkeypatch the relevant method from
# the Python 2.5 distutils.

from distutils.dep_util import newer_pairwise, newer_group
from distutils.ccompiler import gen_preprocess_options
def _setup_compile(self, outdir, macros, incdirs, sources, depends,
                   extra):
    """Process arguments and decide which source files to compile.

    Merges _fix_compile_args() and _prep_compile().
    """
    if outdir is None:
        outdir = self.output_dir
    elif type(outdir) is not StringType:
        raise TypeError, "'output_dir' must be a string or None"

    if macros is None:
        macros = self.macros
    elif type(macros) is ListType:
        macros = macros + (self.macros or [])
    else:
        raise TypeError, "'macros' (if supplied) must be a list of tuples"

    if incdirs is None:
        incdirs = self.include_dirs
    elif type(incdirs) in (ListType, TupleType):
        incdirs = list(incdirs) + (self.include_dirs or [])
    else:
        raise TypeError, \
              "'include_dirs' (if supplied) must be a list of strings"

    if extra is None:
        extra = []

    # Get the list of expected output (object) files
    objects = self.object_filenames(sources,
                                    strip_dir=0,
                                    output_dir=outdir)
    assert len(objects) == len(sources)

    # XXX should redo this code to eliminate skip_source entirely.
    # XXX instead create build and issue skip messages inline

    if self.force:
        skip_source = {}            # rebuild everything
        for source in sources:
            skip_source[source] = 0
    elif depends is None:
        # If depends is None, figure out which source files we
        # have to recompile according to a simplistic check. We
        # just compare the source and object file, no deep
        # dependency checking involving header files.
        skip_source = {}            # rebuild everything
        for source in sources:      # no wait, rebuild nothing
            skip_source[source] = 1

        n_sources, n_objects = newer_pairwise(sources, objects)
        for source in n_sources:    # no really, only rebuild what's
            skip_source[source] = 0 # out-of-date
    else:
        # If depends is a list of files, then do a different
        # simplistic check.  Assume that each object depends on
        # its source and all files in the depends list.
        skip_source = {}
        # L contains all the depends plus a spot at the end for a
        # particular source file
        L = depends[:] + [None]
        for i in range(len(objects)):
            source = sources[i]
            L[-1] = source
            if newer_group(L, objects[i]):
                skip_source[source] = 0
            else:
                skip_source[source] = 1

    pp_opts = gen_preprocess_options(macros, incdirs)

    build = {}
    for i in range(len(sources)):
        src = sources[i]
        obj = objects[i]
        ext = os.path.splitext(src)[1]
        self.mkpath(os.path.dirname(obj))
        if skip_source[src]:
            log.debug("skipping %s (%s up-to-date)", src, obj)
        else:
            build[obj] = src, ext

    return macros, objects, extra, pp_opts, build
CCompiler._setup_compile = _setup_compile

# End of monkeypatch 

#=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=##=--=#

if __name__ == '__main__':
    get_global_args()
    set_dirs()
    set_platform_values()

    readDIRs('.')                       # Gather data from the DIR.py files.

    # Get the data to build the C++ extension modules.
    clibraries = allCLibs.values()
    clibraries.sort(moduleSort)
    extensions = []
    shlibs = []
    for clib in clibraries:
        clib.run_pkg_config()
        extensions.extend(clib.get_extensions())
        shlib = clib.get_shlib()
        if shlib is not None:
            shlibs.append(shlib)

    # Construct the list of pure python packages. If a subdirectory has an
    # __init__.py file, then all the .py files in the subdirectory will
    # form a package.

    # find non-swigged files
    pkg_list = set()
    pkg_list.add(OOFNAME)
    pkgs = find_pkgs()                      # ['SRC', 'SRC/common', ...]
    for pkg in pkgs:
        if pkg != 'SRC':
            pkgname = OOFNAME + '.' + pkg[4:].replace('/', '.')
            pkg_list.add(pkgname)

    # Ask each CLibInfo object for the swigged python modules it
    # creates. 
    for clib in allCLibs.values():
        pkg_list.update(clib.find_swig_pkgs())

    # Make sure that intermediate directories are in the package list.
    allpkgs = set()
    for pkg in pkg_list:
        comps = pkg.split('.')  # components of the package path
        if comps[0] == OOFNAME:
            p = comps[0]
            # Add components one by one to the base component and add
            # the resulting path to the package list.
            for pp in comps[1:]:
                p = '.'.join([p, pp])
                allpkgs.add(p)
        else:
            allpkgs.add(pkg)

    # The top directory in the package hierarchy doesn't get picked up
    # by the above hackery. 
    allpkgs.add(OOFNAME)

    pkgs = [pkg.replace(OOFNAME, OOFNAME+'.ooflib') for pkg in allpkgs]

    # Find example files that have to be installed.
    datafiles = []
    for topdir  in ("examples",): #, "TEST"):
        for dirpath, dirnames, filenames in os.walk(topdir):
            if filenames:
                datafiles.append(
                    (os.path.join(datadir, dirpath), # installation dir
                     [os.path.join(dirpath, phile) for phile in filenames
                      if not phile.endswith('~') and
                      os.path.isfile(os.path.join(dirpath, phile))]))

    # Add the testing files. 
    pkgs.extend([OOFNAME+'.TEST',
                 OOFNAME+'.TEST.UTILS',
                 OOFNAME+'.TEST.GUI'])
    pkg_data = {}
    pkg_data[OOFNAME+".TEST"] = []
    pkg_data[OOFNAME+".TEST.GUI"] = []
    # Include all subdirs of TEST and TEST/GUI that *aren't* packages.
    for dirpath in ('TEST', 'TEST/GUI'):
        pkgpath = dirpath.replace("/", ".")
        for f in os.listdir(dirpath):
            fullpath = os.path.join(dirpath, f)
            if os.path.isdir(fullpath):
                if not os.path.exists(os.path.join(fullpath, '__init__.py')):
                    pkg_data[OOFNAME+"."+pkgpath].append(os.path.join(f, "*"))
                    # Include any subdirectories
                    for dpath, dirnames, filenames in os.walk(fullpath):
                        for d in dirnames:
                            pkg_data[OOFNAME+"."+pkgpath].append(
                                os.path.join(
                                    os.path.relpath(dpath, dirpath),
                                    d, "*"))

    setupargs = dict(
        name = OOFNAME,
        version = version_from_make_dist,
        description = "Analysis of material microstructures, from NIST.",
        author = 'The NIST OOF Team',
        author_email = 'oof_manager@nist.gov',
        url = "http://www.ctcms.nist.gov/oof/oof2/",
        scripts = ['oof2', 'oof2-test', 'oof2-guitest'],
        cmdclass = {"build" : oof_build,
                    "build_ext" : oof_build_ext,
                    "build_py" : oof_build_py,
                    "build_shlib": oof_build_shlib,
                    "install_lib": oof2installlib.oof_install_lib,
                    "clean" : oof_clean},
        packages = pkgs,
        package_dir = {OOFNAME+'.ooflib':'SRC',
                       OOFNAME+'.TEST':'TEST',
                       OOFNAME+'.TEST.GUI':'TEST/GUI'},
        package_data = pkg_data,
        shlibs = shlibs,
        ext_modules = extensions,
        data_files = datafiles
        )

    # This used to check for python version 2.6 or greater..
    ## TODO: OOFCanvas doesn't set plat_name.  Why is it needed here?
    options = dict(build = dict(plat_name = distutils.util.get_platform()))
    setupargs['options'] = options


    distutils.core.setup(**setupargs)


