!! Copyright (C) 2002-2006 M. Marques, A. Castro, A. Rubio, G. Bertsch
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!


  ! ----------------------------------------------------------------------
  !> 
  subroutine target_init_hhg(tg, td, w0)
    type(target_t),   intent(inout) :: tg
    type(td_t),       intent(in)    :: td
    FLOAT,            intent(in)    :: w0

    integer       :: jj
    type(block_t) :: blk
    PUSH_SUB(target_init_hhg)

    tg%move_ions = ion_dynamics_ions_move(td%ions)

    !%Variable OCTOptimizeHarmonicSpectrum
    !%Type block
    !%Default no
    !%Section Calculation Modes::Optimal Control
    !%Description
    !% (Experimental)
    !% If <tt>OCTTargetOperator = oct_tg_hhg</tt>, the target is the harmonic emission spectrum.
    !% In that case, you must supply an <tt>OCTOptimizeHarmonicSpectrum</tt> block in the <tt>inp</tt>
    !% file. The target is given, in general, by:
    !%
    !% <math>J_1 = \int_0^\infty d\omega \alpha(\omega) H(\omega)</math>,
    !%
    !% where <math>H(\omega)</math> is the harmonic spectrum generated by the system, and
    !% <math>\alpha(\omega)</math> is some function that determines what exactly we want
    !% to optimize. The role of the <tt>OCTOptimizeHarmonicSpectrum</tt> block is to determine
    !% this <math>\alpha(\omega)</math> function. Currently, this function is defined as:
    !%
    !% <math>\alpha(\omega) = \sum_{L=1}^{M} \frac{\alpha_L}{a_L} \sqcap( (\omega - L\omega_0)/a_L )</math>,
    !%
    !% where <math>\omega_0</math> is the carrier frequency. <math>M</math> is
    !% the number of columns in the <tt>OCTOptimizeHarmonicSpectrum</tt> block. The values of <i>L</i> will be listed
    !% in the first row of this block; <math>\alpha_L</math> in the second row, and <math>a_L</math> in
    !% the third.
    !% 
    !% Example:
    !%
    !% <tt>%OCTOptimizeHarmonicSpectrum
    !% <br>&nbsp;&nbsp;  7    |  9    | 11
    !% <br>&nbsp;&nbsp; -1    |  1    | -1 
    !% <br>&nbsp;&nbsp;  0.01 |  0.01 |  0.01
    !% <br>%</tt>
    !%
    !%End
    if(parse_is_defined('OCTOptimizeHarmonicSpectrum')) then
      if(parse_block('OCTOptimizeHarmonicSpectrum', blk) == 0) then
        tg%hhg_nks = parse_block_cols(blk, 0)
        SAFE_ALLOCATE(    tg%hhg_k(1:tg%hhg_nks))
        SAFE_ALLOCATE(tg%hhg_alpha(1:tg%hhg_nks))
        SAFE_ALLOCATE(    tg%hhg_a(1:tg%hhg_nks))
        do jj = 1, tg%hhg_nks
          call parse_block_integer(blk, 0, jj - 1, tg%hhg_k(jj))
          call parse_block_float(blk, 1, jj - 1, tg%hhg_alpha(jj))
          call parse_block_float(blk, 2, jj - 1, tg%hhg_a(jj))
        end do
        call parse_block_end(blk)
      else
        message(1) = '"OCTOptimizeHarmonicSpectrum" has to be specified as a block.'
        call messages_info(1)
        call messages_input_error('OCTOptimizeHarmonicSpectrum')
      end if
    else
      write(message(1), '(a)') 'If "OCTTargetMode = oct_targetmode_hhg", you must supply an'
      write(message(2), '(a)') '"OCTOptimizeHarmonicSpectrum" block.'
      call messages_fatal(2)
    end if

    tg%hhg_w0 = w0
    tg%dt     = td%dt
    SAFE_ALLOCATE(tg%td_fitness(0:td%max_iter))
    tg%td_fitness = M_ZERO

    POP_SUB(target_init_hhg)
  end subroutine target_init_hhg


  ! ----------------------------------------------------------------------
  !> 
  subroutine target_init_hhgnew(gr, tg, td, geo, ep)
    type(grid_t),     intent(in)    :: gr
    type(target_t),   intent(inout) :: tg
    type(td_t),       intent(in)    :: td
    type(geometry_t), intent(in)    :: geo
    type(epot_t),     intent(inout) :: ep

    integer :: ist, jst, iunit, jj, nn(3), optimize_parity(3)
    logical :: optimize(3)
    FLOAT :: dw, psi_re, psi_im, ww
    FLOAT, allocatable  :: vl(:), vl_grad(:,:)
    PUSH_SUB(target_init_hhgnew)

    tg%move_ions = ion_dynamics_ions_move(td%ions)

    ! We allocate many things that are perhaps not necessary if we use a direct optimization scheme.
    SAFE_ALLOCATE(tg%vel(1:td%max_iter+1, 1:gr%mesh%sb%dim))
    SAFE_ALLOCATE(tg%acc(1:td%max_iter+1, 1:gr%mesh%sb%dim))
    SAFE_ALLOCATE(tg%gvec(1:td%max_iter+1, 1:gr%mesh%sb%dim))
    SAFE_ALLOCATE(tg%alpha(1:td%max_iter))

    ! The following is a temporary hack, that assumes only one atom at the origin of coordinates.
    if(geo%natoms > 1) then
      message(1) = 'If "OCTTargetOperator = oct_tg_hhgnew", then you can only have one atom.'
      call messages_fatal(1)
    end if

    ! The following is a temporary hack, that assumes only one atom at the origin of coordinates.
    if(geo%natoms > 1) then
      message(1) = 'If "OCTTargetOperator = oct_tg_hhgnew", then you can only have one atom.'
      call messages_fatal(1)
    end if

    SAFE_ALLOCATE(tg%grad_local_pot(1:geo%natoms, 1:gr%mesh%np, 1:gr%sb%dim))
    SAFE_ALLOCATE(vl(1:gr%mesh%np_part))
    SAFE_ALLOCATE(vl_grad(1:gr%mesh%np, 1:gr%sb%dim))
    SAFE_ALLOCATE(tg%rho(1:gr%mesh%np))

    vl(:) = M_ZERO
    vl_grad(:,:) = M_ZERO
    call epot_local_potential(ep, gr%der, gr%dgrid, geo, 1, vl)
    call dderivatives_grad(gr%der, vl, vl_grad)
    forall(ist=1:gr%mesh%np, jst=1:gr%sb%dim)
      tg%grad_local_pot(1, ist, jst) = vl_grad(ist, jst)
    end forall
    ! Note that the calculation of the gradient of the potential
    ! is wrong at the borders of the box, since it assumes zero boundary
    ! conditions. The best way to solve this problems is to define the 
    ! target making use of the definition of the forces based on the gradient
    ! of the density, rather than on the gradient of the potential.

          
    !%Variable OCTHarmonicWeight
    !%Type string
    !%Default "1"
    !%Section Calculation Modes::Optimal Control
    !%Description
    !% (Experimental) If <tt>OCTTargetOperator = oct_tg_plateau</tt>, then the function to optimize is the integral of the
    !% harmonic spectrum <math>H(\omega)</math>, weighted with a function <math>f(\omega)</math>
    !% that is defined as a string here. For example, if 
    !% you set <tt>OCTHarmonicWeight  = "step(w-1)"</tt>, the function to optimize is
    !% the integral of <math>step(\omega-1)*H(\omega)</math>, <i>i.e.</i>
    !% <math>\int_1^{\infty} H \left( \omega \right) d\omega</math>.
    !% In practice, it is better if you also set an upper limit, <i>e.g.</i>
    !% <math>f(\omega) = step(\omega-1) step(2-\omega)</math>.
    !%End
    call parse_variable('OCTHarmonicWeight', '1', tg%plateau_string)
    tg%dt = td%dt
    SAFE_ALLOCATE(tg%td_fitness(0:td%max_iter))
    tg%td_fitness = M_ZERO

    iunit = io_open('.alpha', action = 'write')
    dw = (M_TWO * M_PI) / (td%max_iter * tg%dt)
    do jj = 0, td%max_iter - 1
      ww = jj * dw
      call parse_expression(psi_re, psi_im, "w", ww, tg%plateau_string)
      tg%alpha(jj+1) = psi_re
      write(iunit, *) ww, psi_re
    end do
    call io_close(iunit)

    nn(1:3) = (/ td%max_iter, 1, 1 /)
    optimize(1:3) = .false.
    optimize_parity(1:3) = -1
    call fft_init(tg%fft_handler, nn(1:3), 1, FFT_COMPLEX, FFTLIB_FFTW, optimize, optimize_parity)

    POP_SUB(target_init_hhgnew)
  end subroutine target_init_hhgnew


  ! ----------------------------------------------------------------------
  !> 
  subroutine target_end_hhg(tg)
    type(target_t),   intent(inout) :: tg
    PUSH_SUB(target_end_hhg)
    SAFE_DEALLOCATE_P(tg%hhg_k)
    SAFE_DEALLOCATE_P(tg%hhg_alpha)
    SAFE_DEALLOCATE_P(tg%hhg_a)
    SAFE_DEALLOCATE_P(tg%td_fitness)
    POP_SUB(target_end_hhg)
  end subroutine target_end_hhg


  ! ----------------------------------------------------------------------
  subroutine target_output_hhg(tg, gr, dir, geo, hm, outp)
    type(target_t), intent(inout) :: tg
    type(grid_t), intent(inout)   :: gr
    character(len=*), intent(in)  :: dir
    type(geometry_t),       intent(in)  :: geo
    type(hamiltonian_t),    intent(in)  :: hm
    type(output_t),         intent(in)  :: outp

    PUSH_SUB(target_output_hhg)
    
    call io_mkdir(trim(dir))
    call output_states(tg%st, gr, geo, hm, trim(dir), outp)

    POP_SUB(target_output_hhg)
  end subroutine target_output_hhg
  ! ----------------------------------------------------------------------


  ! ----------------------------------------------------------------------
  !> 
  subroutine target_end_hhgnew(tg, oct)
    type(target_t),   intent(inout) :: tg
    type(oct_t), intent(in)       :: oct
    PUSH_SUB(target_init_hhgnew)
    if((oct%algorithm  ==  OPTION__OCTSCHEME__OCT_CG) .or. (oct%algorithm == OPTION__OCTSCHEME__OCT_BFGS)) then
      SAFE_DEALLOCATE_P(tg%grad_local_pot)
      SAFE_DEALLOCATE_P(tg%rho)
      SAFE_DEALLOCATE_P(tg%vel)
      SAFE_DEALLOCATE_P(tg%acc)
      SAFE_DEALLOCATE_P(tg%gvec)
      SAFE_DEALLOCATE_P(tg%alpha)
      SAFE_DEALLOCATE_P(tg%td_fitness)
      call fft_end(tg%fft_handler)
    end if
    POP_SUB(target_end_hhgnew)
  end subroutine target_end_hhgnew


  ! ----------------------------------------------------------------------
  !> 
  FLOAT function target_j1_hhg(tg) result(j1)
    type(target_t),   intent(inout) :: tg

    integer :: maxiter, jj
    FLOAT :: aa, ww, maxhh, omega
    CMPLX, allocatable :: ddipole(:)
    PUSH_SUB(target_j1_hhg)

    maxiter = size(tg%td_fitness) - 1
    SAFE_ALLOCATE(ddipole(0:maxiter))
    ddipole = M_z0
    ddipole = tg%td_fitness

    call spectrum_hsfunction_init(tg%dt, 0, maxiter, maxiter, ddipole)
    do jj = 1, tg%hhg_nks
      aa = tg%hhg_a(jj) * tg%hhg_w0
      ww = tg%hhg_k(jj) * tg%hhg_w0
      call spectrum_hsfunction_min(ww - aa, ww + aa, omega, maxhh)
      j1 = j1 + tg%hhg_alpha(jj) * log(-maxhh)
    end do
    call spectrum_hsfunction_end()

    SAFE_DEALLOCATE_A(ddipole)
    POP_SUB(target_j1_hhg)
  end function target_j1_hhg


  ! ----------------------------------------------------------------------
  !> 
  FLOAT function target_j1_hhgnew(gr, tg) result(j1)
    type(grid_t),     intent(in)    :: gr
    type(target_t),   intent(inout) :: tg

    integer :: maxiter, i
    FLOAT :: dw, ww
    PUSH_SUB(target_j1_hhgnew)

    maxiter = size(tg%td_fitness) - 1
    dw = (M_TWO * M_PI) / (maxiter * tg%dt)
    j1 = M_ZERO
    do i = 0, maxiter - 1
      ww = i * dw
      j1 = j1 + dw * tg%alpha(i+1) * sum(abs(tg%vel(i+1, 1:gr%sb%dim))**2)
    end do

    POP_SUB(target_j1_hhgnew)
  end function target_j1_hhgnew


  ! ----------------------------------------------------------------------
  !> 
  subroutine target_chi_hhg(gr, chi_out)
    type(grid_t),      intent(inout) :: gr
    type(states_t),    intent(inout) :: chi_out

    integer :: ik, ib
    PUSH_SUB(target_chi_hhg)

    !we have a time-dependent target --> Chi(T)=0
    do ik = chi_out%d%kpt%start, chi_out%d%kpt%end
      do ib = chi_out%group%block_start, chi_out%group%block_end
        call batch_set_zero(chi_out%group%psib(ib, ik))
      end do
    end do
    
    POP_SUB(target_chi_hhg)
  end subroutine target_chi_hhg

 
  ! ---------------------------------------------------------
  !> 
  !!
  subroutine target_tdcalc_hhgnew(tg, gr, psi, time, max_time)
    type(target_t),      intent(inout) :: tg
    type(grid_t),        intent(inout) :: gr
    type(states_t),      intent(inout) :: psi
    integer,             intent(in)    :: time
    integer,             intent(in)    :: max_time

    CMPLX, allocatable :: opsi(:, :), zpsi(:, :)
    integer :: iw, ia, ist, idim, ik
    FLOAT :: acc(MAX_DIM), dt, dw

    PUSH_SUB(target_tdcalc_hhgnew)

    tg%td_fitness(time) = M_ZERO

    ! If the ions move, the tg is computed in the propagation routine.
    if(.not.target_move_ions(tg)) then

      SAFE_ALLOCATE(opsi(1:gr%mesh%np_part, 1:1))
      SAFE_ALLOCATE(zpsi(1:gr%mesh%np_part, 1:1))

      opsi = M_z0
      ! WARNING This does not work for spinors.
      ! The following is a temporary hack. It assumes only one atom at the origin.
      acc = M_ZERO
      do ik = 1, psi%d%nik
        do ist = 1, psi%nst
          call states_get_state(psi, gr%mesh, ist, ik, zpsi)
          do idim = 1, gr%sb%dim
            opsi(1:gr%mesh%np, 1) = tg%grad_local_pot(1, 1:gr%mesh%np, idim)*zpsi(1:gr%mesh%np, 1)
            acc(idim) = acc(idim) + real( psi%occ(ist, ik) * &
                zmf_dotp(gr%mesh, psi%d%dim, opsi, zpsi), REAL_PRECISION )
            tg%acc(time+1, idim) = tg%acc(time+1, idim) + psi%occ(ist, ik)*zmf_dotp(gr%mesh, psi%d%dim, opsi, zpsi)
          end do
        end do
      end do

      SAFE_DEALLOCATE_A(opsi)
      SAFE_DEALLOCATE_A(zpsi)
      
    end if

    dt = tg%dt
    dw = (M_TWO * M_PI/(max_time * tg%dt))
    if(time  ==  max_time) then
      tg%acc(1, 1:gr%sb%dim) = M_HALF * (tg%acc(1, 1:gr%sb%dim) + tg%acc(max_time+1, 1:gr%sb%dim))
      do ia = 1, gr%sb%dim
        call zfft_forward1(tg%fft_handler, tg%acc(1:max_time, ia), tg%vel(1:max_time, ia))
      end do
      tg%vel = tg%vel * tg%dt
      do iw = 1, max_time
        ! We add the one-half dt term because when doing the propagation we want the value at interpolated times.

        tg%acc(iw, 1:gr%sb%dim) = tg%vel(iw, 1:gr%sb%dim) * tg%alpha(iw) * exp(M_zI * (iw-1) * dw * M_HALF * dt)
      end do
      do ia = 1, gr%sb%dim
        call zfft_backward1(tg%fft_handler, tg%acc(1:max_time, ia), tg%gvec(1:max_time, ia))
      end do
      tg%gvec(max_time + 1, 1:gr%sb%dim) = tg%gvec(1, 1:gr%sb%dim)
      tg%gvec = tg%gvec * (M_TWO * M_PI/ tg%dt)
    end if

    POP_SUB(target_tdcalc_hhgnew)
  end subroutine target_tdcalc_hhgnew
  ! ----------------------------------------------------------------------


  ! ---------------------------------------------------------
  !> 
  !!
  subroutine target_tdcalc_hhg(tg, hm, gr, geo, psi, time)
    type(target_t),      intent(inout) :: tg
    type(hamiltonian_t), intent(inout) :: hm
    type(grid_t),        intent(inout) :: gr
    type(geometry_t),    intent(inout) :: geo
    type(states_t),      intent(inout) :: psi
    integer,             intent(in)    :: time

    FLOAT :: acc(MAX_DIM)
    PUSH_SUB(target_tdcalc_hhg)

    call td_calc_tacc(gr, geo, psi, hm, acc, time*tg%dt)
    tg%td_fitness(time) = acc(1)

    POP_SUB(target_tdcalc_hhg)
  end subroutine target_tdcalc_hhg
  ! ----------------------------------------------------------------------



!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:
