!! Copyright (C) 2014 Alain Delgado Gran, Carlo Andrea Rozzi, Stefano Corni
!!
!! This program is free software; you can redistribute it and/or modify
!! it under the terms of the GNU General Public License as published by
!! the Free Software Foundation; either version 2, or (at your option)
!! any later version.
!!
!! This program is distributed in the hope that it will be useful,
!! but WITHOUT ANY WARRANTY; without even the implied warranty of
!! MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!! GNU General Public License for more details.
!!
!! You should have received a copy of the GNU General Public License
!! along with this program; if not, write to the Free Software
!! Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
!! 02110-1301, USA.
!!
!! $Id: pcm.F90 15157 2016-03-01 22:01:17Z umberto $

#include "global.h"

module pcm_m
  use comm_m
  use global_m
  use geometry_m
  use grid_m
  use io_m
  use index_m
  use messages_m
  use mesh_m 
  use mesh_interpolation_m 
  use mpi_m
  use par_vec_m
  use parser_m
  use poisson_m
  use profiling_m
  use simul_box_m
  use species_m
  
  ! to output debug info
  use unit_m
  use unit_system_m
  use io_function_m
  use mesh_function_m
  

  implicit none

  private

  public ::                 &
    pcm_t,                  &
    pcm_sphere_t,           &
    pcm_tessera_t,          &
    pcm_init,               &
    pcm_end,                &
    pcm_charges,            &
    pcm_get_vdw_radius,     &
    pcm_pot_rs,             &
    pcm_elect_energy,       &
    pcm_v_nuclei_cav,       &
    pcm_v_electrons_cav_li, &
    pcm_update,             &
    pcm_calc_pot_rs


  !> The cavity hosting the solute molecule is built from a set of 
  !! interlocking spheres with optimized radii centered at the nuclear positions.  
  type :: pcm_sphere_t  
    FLOAT :: x !
    FLOAT :: y !< center of the sphere
    FLOAT :: z !
    FLOAT :: r !< radius of the sphere (different for each species)
  end type pcm_sphere_t

  !> The resulting cavity is discretized by a set of tesserae defined in 3d.  
  integer, parameter :: pcm_dim_space = 3

  type :: pcm_tessera_t
    FLOAT :: point(1:pcm_dim_space)  !< representative point of the tessera  
    FLOAT :: area                    !< area of the tessera
    FLOAT :: normal(1:pcm_dim_space) !< unitary outgoing vector normal to the tessera surface  
    FLOAT :: r_sphere                !< radius of the sphere to which the tessera belongs
  end type pcm_tessera_t

  type pcm_t
    logical                          :: run_pcm       !< If .true., PCM calculation is enabled
    integer                          :: n_spheres     !< Number of spheres used to build the VdW cavity
    integer                          :: n_tesserae    !< Total number of tesserae
    type(pcm_sphere_t),  allocatable :: spheres(:)    !< See type pcm_sphere_t
    type(pcm_tessera_t), allocatable :: tess(:)       !< See type pcm_tessera_t
    FLOAT                            :: scale_r       !< scaling factor for the radii of the spheres used in PCM
    FLOAT, allocatable               :: matrix(:,:)   !< PCM response matrix
    FLOAT, allocatable               :: q_e(:)        !< polarization charges due to the solute electrons        
    FLOAT, allocatable               :: q_n(:)        !< polarization charges due to the solute nuclei
    FLOAT, allocatable               :: rho_e(:)      !< polarization density due to the solute electrons        
    FLOAT, allocatable               :: rho_n(:)      !< polarization density due to the solute nuclei
    FLOAT                            :: qtot_e        !< total polarization charge due to electrons
    FLOAT                            :: qtot_n        !< total polarization charge due to nuclei
    FLOAT, allocatable               :: v_e(:)        !< Hartree potential at each tessera
    FLOAT, allocatable               :: v_n(:)        !< Nuclear potential at each tessera
    FLOAT, allocatable               :: v_e_rs(:)     !< PCM potential in real-space produced by q_e(:) 
    FLOAT, allocatable               :: v_n_rs(:)     !< PCM potential in real-space produced by q_n(:)
!     FLOAT, allocatable               :: arg_li(:,:)   !< used in the trilinear interpolation to estimate
                                                      !< the Hartree potential at the tesserae 
    FLOAT                            :: epsilon_0     !< Static dielectric constant of the solvent 
    FLOAT                            :: epsilon_infty !< Infinite-frequency dielectric constant of the solvent
    FLOAT                            :: gaussian_width!< Parameter to change the width of density of polarization charges  
!     integer                          :: n_vertices    !< Number of grid points used to interpolate the Hartree potential
                                                      !! at the tesserae 
!     integer, allocatable             :: ind_vh(:,:)   !< Grid points used during interpolation
    integer                          :: info_unit     !< unit for pcm info file
    integer                          :: counter       !< used to print the number of SCF or TD iterations in energy_calc
    character(len=80)                :: input_cavity  !< file name containing the geometry of the VdW cavity
    
    integer                          :: update_iter   !< how often the pcm potential is updated
    integer                          :: iter          !< update iteration counter
    
    integer                          :: calc_method   !< which method should be used to obtain the pcm potential 
    integer                          :: tess_nn       !< number of tessera center mesh-point nearest neighbors
  end type pcm_t

  FLOAT, allocatable :: s_mat_act(:,:) !< S_I matrix 
  FLOAT, allocatable :: d_mat_act(:,:) !< D_I matrix
  FLOAT, allocatable :: Sigma(:,:)     !< S_E matrix
  FLOAT, allocatable :: Delta(:,:)     !< D_E matrix in JCP 139, 024105 (2013).

  logical            :: gamess_benchmark !< Decide to output pcm_matrix in a GAMESS format 
  FLOAT, allocatable :: mat_gamess(:,:)  !< PCM matrix formatted to be inputed to GAMESS

  integer, parameter ::     &
    PCM_ELECTRONS = 0,       &
    PCM_NUCLEI    = 1

  integer, parameter ::     &
    PCM_CALC_DIRECT  = 1,    &
    PCM_CALC_POISSON = 2

  integer, parameter ::       &
    PCM_VDW_OPTIMIZED   = 1,   &
    PCM_VDW_SPECIES     = 2 
    

contains


  !-------------------------------------------------------------------------------------------------------
  !> Initializes the PCM calculation: reads the VdW molecular cavity and generates the PCM response matrix.
  subroutine pcm_init(pcm, geo, grid)
    type(geometry_t), intent(in) :: geo
    type(grid_t), intent(in)     :: grid
    type(pcm_t), intent(out)     :: pcm

    integer :: ia, itess, jtess, cav_unit_test, subdivider, ii
    integer :: pcm_vdw_type
    integer :: pcmmat_unit, pcmmat_gamess_unit, iunit, ip

    integer, parameter :: mxts = 10000

    FLOAT              :: default_value
    FLOAT              :: vdw_radius

    type(pcm_tessera_t) :: dum2(1)

    logical :: band
    logical :: add_spheres_h, changed_default_nn

    type(species_t), pointer :: spci 
    FLOAT :: z_ia
    
    integer :: default_nn
    FLOAT   :: max_area
    

    PUSH_SUB(pcm_init)
    
    pcm%iter = 0
    pcm%update_iter = 1

    !%Variable PCMCalculation
    !%Type logical
    !%Default no
    !%Section Hamiltonian::PCM
    !%Description
    !% (Experimental) If true, the calculation is performed accounting for solvation effects
    !% by using the Integral Equation Formalism Polarizable Continuum Model IEF-PCM
    !% formulated in real-space and real-time (arXiv:1507.05471, <i>Chem. Rev.</i> <b>105</b>, 2999 (2005),
    !% <i>J. Chem. Phys.</i> <b>139</b>, 024105 (2013)). At the moment, this option is available 
    !% only for <tt>TheoryLevel = DFT</tt>.
    !%End

    call parse_variable('PCMCalculation', .false., pcm%run_pcm)
    if (pcm%run_pcm) then
      call messages_print_stress(stdout, trim('PCM'))
      if ( (grid%sb%box_shape /= MINIMUM).OR.(grid%sb%dim /= pcm_dim_space) ) then
        message(1) = "PCM is only available for BoxShape = minimum and 3d calculations"
        call messages_fatal(1)
      else 
        call messages_experimental("Polarizable Continuum Model")
      end if
    else
      POP_SUB(pcm_init)
      return
    end if

    !%Variable PCMVdWRadii
    !%Type integer
    !%Default pcm_vdw_optimized
    !%Section Hamiltonian::PCM
    !%Description
    !% This variable selects which van der Waals radius will be used to generate the solvent cavity.
    !%Option pcm_vdw_optimized  1
    !% Use the van der Waals radius optimized by Stefan Grimme in J. Comput. Chem. 27: 1787-1799, 2006, 
    !% except for C, N and O, reported in J. Chem. Phys. 120, 3893 (2004).
    !%Option pcm_vdw_species  2
    !% The vdW radii are set from the <tt>share/pseudopotentials/elements</tt> file. These values are obtained from 
    !% Alvarez S., Dalton Trans., 2013, 42, 8617-8636. Values can be changed in the <tt>Species</tt> block.
    !%End
    call parse_variable('PCMVdWRadii', PCM_VDW_OPTIMIZED, pcm_vdw_type)
    call messages_print_var_option(stdout, "PCMVdWRadii", pcm_vdw_type)

    select case (pcm_vdw_type)
      case (PCM_VDW_OPTIMIZED) 
         default_value = 1.2d0
      case (PCM_VDW_SPECIES) 
         default_value = 1.d0
    end select 
   
    !%Variable PCMRadiusScaling
    !%Type float
    !%Section Hamiltonian::PCM
    !%Description
    !% Scales the radii of the spheres used to build the solute cavity surface.
    !% The default value depends on the choice of <tt>PCMVdWRadii</tt>:
    !% 1.2 for <tt>pcm_vdw_optimized</tt> and 1.0 for <tt>pcm_vdw_species</tt>.
    !%End
    call parse_variable('PCMRadiusScaling', default_value, pcm%scale_r)
    call messages_print_var_value(stdout, "PCMRadiusScaling", pcm%scale_r)

    !%Variable PCMStaticEpsilon
    !%Type float
    !%Default 1.0
    !%Section Hamiltonian::PCM
    !%Description
    !% Static dielectric constant of the solvent (<math>\varepsilon_0</math>). 1.0 indicates gas phase.
    !%End
    call parse_variable('PCMStaticEpsilon', M_ONE, pcm%epsilon_0)
    call messages_print_var_value(stdout, "PCMStaticEpsilon", pcm%epsilon_0)

    !%Variable PCMDynamicEpsilon
    !%Type float
    !%Default PCMStaticEpsilon
    !%Section Hamiltonian::PCM
    !%Description
    !% High-frequency dielectric constant of the solvent (<math>\varepsilon_d</math>). 1.0 indicates gas phase.
    !% At present, non-equilibrium effects within PCM calculations are not implemented. For td calculations
    !% take <tt>PCMDynamicEpsilon = PCMStaticEpsilon</tt> (default). 
    !%End
    call parse_variable('PCMDynamicEpsilon', pcm%epsilon_0, pcm%epsilon_infty)
    call messages_print_var_value(stdout, "PCMDynamicEpsilon", pcm%epsilon_infty)
    
    !%Variable PCMUpdateIter
    !%Type integer
    !%Default 1
    !%Section Hamiltonian::PCM
    !%Description
    !% Defines how often the PCM potential is updated during time propagation.
    !%End
    call parse_variable('PCMUpdateIter', 1, pcm%update_iter)
    call messages_print_var_value(stdout, "PCMUpdateIter", pcm%update_iter)

    !%Variable PCMGamessBenchmark
    !%Type logical
    !%Default .false.
    !%Section Hamiltonian::PCM
    !%Description
    !% If PCMGamessBenchmark is set to "yes", the pcm_matrix is also written in a Gamess format.
    !% for benchamarking purposes.
    !%End
    call parse_variable('PCMGamessBenchmark', .false., gamess_benchmark)

    !%Variable PCMSmearingFactor
    !%Type float
    !%Default 1.0
    !%Section Hamiltonian::PCM
    !%Description
    !% Parameter used to control the width (area of each tessera) of the Gaussians used to distribute
    !% the polarization charges on each tessera (arXiv:1507.05471). If set to zero, the solvent 
    !% reaction potential in real-space is defined by using point charges.
    !%End
    call parse_variable('PCMSmearingFactor', M_ONE, pcm%gaussian_width)
    call messages_print_var_value(stdout, "PCMSmearingFactor", pcm%gaussian_width)

    if (pcm%gaussian_width == M_ZERO) then
      message(1) = "Info: PCM potential will be defined in terms of polarization point charges"
      call messages_info(1)
    else
      message(1) = "Info: PCM potential is regularized to avoid Coulomb singularity"
      call messages_info(1)        
    end if

    call io_mkdir('pcm')

    !%Variable PCMCavity
    !%Type string
    !%Section Hamiltonian::PCM
    !%Description
    !% Name of the file containing the geometry of the cavity hosting the solute molecule.
    !% The data must be in atomic units and the file must contain the following information sequentially:
    !%  T               < Number of tesserae
    !%  s_x(1:T)        < coordinates x of the tesserae 
    !%  s_y(1:T)        < coordinates y of the tesserae        
    !%  s_z(1:T)        < coordinates z of the tesserae            
    !%  A(1:T)          < areas of the tesserae
    !%  R_sph(1:T)      < Radii of the spheres to which the tesserae belong
    !%  normal(1:T,1:3) < Outgoing unitary vectors at the tesserae surfaces 
    !%End
    call parse_variable('PCMCavity', '', pcm%input_cavity)

    if (pcm%input_cavity == '') then

      !%Variable PCMSpheresOnH
      !%Type logical
      !%Default no
      !%Section Hamiltonian::PCM
      !%Description
      !% If true, spheres centered at the Hydrogens atoms are included to build the solute cavity surface.
      !%End
      call parse_variable('PCMSpheresOnH', .false., add_spheres_h)

      pcm%n_spheres = 0
      band = .false.
      do ia = 1, geo%natoms
        if ( (.not.(add_spheres_h)).and.(geo%atom(ia)%label == 'H') ) cycle
        pcm%n_spheres = pcm%n_spheres + 1 !counting the number of species different from Hydrogen
      end do

      SAFE_ALLOCATE( pcm%spheres(1:pcm%n_spheres) )
      pcm%spheres(:)%x = M_ZERO
      pcm%spheres(:)%y = M_ZERO
      pcm%spheres(:)%z = M_ZERO        
      pcm%spheres(:)%r = M_ZERO

      pcm%n_spheres = 0
      do ia = 1, geo%natoms
        if ( (.not.(add_spheres_h)).and.(geo%atom(ia)%label == 'H') ) cycle
        pcm%n_spheres = pcm%n_spheres + 1

        !> These coordinates are already in atomic units (Bohr)
        pcm%spheres(pcm%n_spheres)%x = geo%atom(ia)%x(1)
        pcm%spheres(pcm%n_spheres)%y = geo%atom(ia)%x(2)
        pcm%spheres(pcm%n_spheres)%z = geo%atom(ia)%x(3)

        vdw_radius = pcm_get_vdw_radius(geo%atom(ia)%species, pcm_vdw_type)
        pcm%spheres(pcm%n_spheres)%r = vdw_radius*pcm%scale_r     
      end do

      if ( mpi_grp_is_root(mpi_world) ) then
        pcm%info_unit = io_open(PCM_DIR//'pcm_info.out', action='write')
      
        write(pcm%info_unit,'(A35)') '# Configuration: Molecule + Solvent'
        write(pcm%info_unit,'(A35)') '# ---------------------------------'
        write(pcm%info_unit,'(A21,F12.3)') '# Epsilon(Solvent) = ', pcm%epsilon_0
        write(pcm%info_unit,'(A1)')'#' 
        write(pcm%info_unit,'(A35,I4)') '# Number of interlocking spheres = ', pcm%n_spheres
        write(pcm%info_unit,'(A1)')'#'  
      
        write(pcm%info_unit,'(A8,3X,A7,8X,A26,20X,A10)') '# SPHERE', 'ELEMENT', 'CENTER  (X,Y,Z) (A)', 'RADIUS (A)'
        write(pcm%info_unit,'(A8,3X,A7,4X,A43,7X,A10)') '# ------', '-------', &
          '-------------------------------------------', '----------'  
      end if
      
        pcm%n_spheres = 0
        do ia = 1, geo%natoms
          if (geo%atom(ia)%label == 'H') cycle
          pcm%n_spheres = pcm%n_spheres + 1       
      
          if ( mpi_grp_is_root(mpi_world) ) & 
            write(pcm%info_unit,'(A1,2X,I3,7X,A2,3X,F14.8,2X,F14.8,2X,F14.8,4X,F14.8)')'#', pcm%n_spheres, &
              geo%atom(ia)%label,          &
              geo%atom(ia)%x(1)*P_a_B,     &
              geo%atom(ia)%x(2)*P_a_B,     &
              geo%atom(ia)%x(3)*P_a_B,     &
              pcm%spheres(pcm%n_spheres)%r*P_a_B
        end do

      !%Variable PCMTessSubdivider
      !%Type integer
      !%Default 1
      !%Section Hamiltonian::PCM
      !%Description
      !% Allows to subdivide further each tessera. Can take only two values, 1 or 4.
      !% 1 => 60 tesserae/sphere; 4 => 240 tesserae/sphere.
      !%End
      call parse_variable('PCMTessSubdivider', 1, subdivider)

      !> Counting the number of tesserae
      call cav_gen(0, subdivider, pcm%n_spheres, pcm%spheres, pcm%n_tesserae, dum2, pcm%info_unit)

      SAFE_ALLOCATE(pcm%tess(1:pcm%n_tesserae))

      do ia=1, pcm%n_tesserae
        pcm%tess(ia)%point    = M_ZERO
        pcm%tess(ia)%area     = M_ZERO
        pcm%tess(ia)%r_sphere = M_ZERO
        pcm%tess(ia)%normal   = M_ZERO
      end do

      !> Generating the Van der Waals discretized surface of the solute system
      call cav_gen(1, subdivider, pcm%n_spheres, pcm%spheres, pcm%n_tesserae, pcm%tess, pcm%info_unit)

      message(1) = "Info: van der Waals surface has been calculated"
      call messages_info(1)

    else

      !> The cavity surface will be read from a external file
      iunit = io_open(trim(pcm%input_cavity), status='old', action='read')
      read(iunit,*) pcm%n_tesserae

      if (pcm%n_tesserae.gt.mxts) then
        write(message(1),'(a,I5,a,I5)') "total number of tesserae", pcm%n_tesserae, ">",mxts
        call messages_warning(1)     
      end if

      SAFE_ALLOCATE(pcm%tess(1:pcm%n_tesserae))

      do ia=1, pcm%n_tesserae
        pcm%tess(ia)%point    = M_ZERO
        pcm%tess(ia)%area     = M_ZERO
        pcm%tess(ia)%r_sphere = M_ZERO
        pcm%tess(ia)%normal   = M_ZERO
      end do

      do ia=1, pcm%n_tesserae
        read(iunit,*) pcm%tess(ia)%point(1)
      end do

      do ia=1, pcm%n_tesserae
        read(iunit,*) pcm%tess(ia)%point(2)
      end do

      do ia=1, pcm%n_tesserae
        read(iunit,*) pcm%tess(ia)%point(3)
      end do

      do ia=1, pcm%n_tesserae
        read(iunit,*) pcm%tess(ia)%area
      end do

      do ia=1, pcm%n_tesserae
        read(iunit,*) pcm%tess(ia)%r_sphere
      end do

      do ia=1, pcm%n_tesserae
        read(iunit,*) pcm%tess(ia)%normal
      end do

      call io_close(iunit)
      message(1) = "Info: van der Waals surface has been read from " // trim(pcm%input_cavity)
      call messages_info(1)
    end if

    if ( mpi_grp_is_root(mpi_world) ) then
      cav_unit_test = io_open(PCM_DIR//'cavity_mol.xyz', action='write')
     
      write (cav_unit_test,'(2X,I4)') pcm%n_tesserae + geo%natoms
      write (cav_unit_test,'(2X)')
     
      do ia = 1, pcm%n_tesserae
        write(cav_unit_test,'(2X,A2,3X,4f15.8,3X,4f15.8,3X,4f15.8)') 'H', pcm%tess(ia)%point*P_a_B
      end do
     
      do ia = 1, geo%natoms
        write(cav_unit_test,'(2X,A2,3X,4f15.8,3X,4f15.8,3X,4f15.8)') geo%atom(ia)%label,      &
          geo%atom(ia)%x*P_a_B
      end do
     
      call io_close(cav_unit_test)
     
      write(pcm%info_unit,'(A1)')'#'  
      write(pcm%info_unit,'(A1,4X,A4,14X,A4,21X,A4,21X,A4,21X,A4,21X,A8,17X,A5,20X,A5)') &
        '#','iter', 'E_ee', 'E_en', 'E_nn', 'E_ne', 'E_M-solv', 'Q_M^e','Q_M^n'
    end if
    pcm%counter = 0

!     pcm%n_vertices = 8
!     cSAFE_ALLOCATE(pcm%ind_vh(1:pcm%n_tesserae, 1:pcm%n_vertices))
!     pcm%ind_vh = INT(M_ZERO)

!     cSAFE_ALLOCATE(pcm%arg_li(1:pcm%n_tesserae, 1:pcm_dim_space))
!     pcm%arg_li = M_ZERO

!     !> Creating the list of the nearest grid points to each tessera
!     !! to be used to interpolate the Hartree potential at the representative points
!     do ia = 1, pcm%n_tesserae
!       call nearest_cube_vertices(pcm%tess(ia)%point, grid%mesh, pcm%ind_vh(ia,:), pcm%arg_li(ia,:))
!     end do

    !>Generating the dynamical PCM matrix
    if ( gamess_benchmark ) then
      SAFE_ALLOCATE( mat_gamess(1:pcm%n_tesserae, 1:pcm%n_tesserae) )
      mat_gamess = M_ZERO
    end if

    SAFE_ALLOCATE( pcm%matrix(1:pcm%n_tesserae, 1:pcm%n_tesserae) )
    pcm%matrix = M_ZERO

    call pcm_matrix(pcm%epsilon_infty, pcm%tess, pcm%n_tesserae, pcm%matrix) 

    if ( gamess_benchmark .and. mpi_grp_is_root(mpi_world)) then 
      pcmmat_gamess_unit = io_open(PCM_DIR//'pcm_matrix_gamess_dyn.out', action='write')

      do jtess = 1, pcm%n_tesserae
        do itess = 1, pcm%n_tesserae
          write(pcmmat_gamess_unit,*) mat_gamess(itess,jtess) !< for benchmarking with GAMESS
        end do
      end do

      call io_close(pcmmat_gamess_unit)
      mat_gamess = M_ZERO
    end if

    pcm%matrix = M_ZERO

    call pcm_matrix(pcm%epsilon_0, pcm%tess, pcm%n_tesserae, pcm%matrix) 
    message(1) = "Info: PCM response matrix has been evaluated"
    call messages_info(1)

    if (mpi_grp_is_root(mpi_world)) then
      pcmmat_unit = io_open(PCM_DIR//'pcm_matrix.out', action='write')
      if ( gamess_benchmark ) pcmmat_gamess_unit = io_open(PCM_DIR//'pcm_matrix_gamess.out', action='write')

      do jtess = 1, pcm%n_tesserae
        do itess = 1, pcm%n_tesserae
          write(pcmmat_unit,*) pcm%matrix(itess,jtess)
          if ( gamess_benchmark ) write(pcmmat_gamess_unit,*) mat_gamess(itess,jtess) !< for benchmarking with GAMESS
        end do
      end do
      call io_close(pcmmat_unit)
      if ( gamess_benchmark ) call io_close(pcmmat_gamess_unit)
    end if
#ifdef HAVE_MPI
    call MPI_Barrier(mpi_world%comm, mpi_err)
#endif
    if ( gamess_benchmark ) then 
      SAFE_DEALLOCATE_A ( mat_gamess )
    end if
    
    
    !%Variable PCMCalcMethod
    !%Type integer
    !%Default pcm_direct
    !%Section Hamiltonian::PCM
    !%Description
    !% Defines the method to be used to obtain the PCM potential.
    !%Option pcm_direct 1
    !% Direct sum of the potential generated by the polarization charges regularized 
    !% with a Gaussian smearing [A. Delgado, et al., J Chem Phys 143, 144111 (2015)].
    !%Option pcm_poisson 2
    !% Solving the Poisson equation for the polarization charge density.
    !%End
    call parse_variable('PCMCalcMethod', PCM_CALC_DIRECT, pcm%calc_method)
    call messages_print_var_option(stdout, "PCMCalcMethod", pcm%calc_method)


    if (pcm%calc_method == PCM_CALC_POISSON) then

      max_area = M_EPSILON
      do ia = 1, pcm%n_tesserae
        if (pcm%tess(ia)%area > max_area) max_area = pcm%tess(ia)%area
      end do
    
      !default is as many neighbor to contain 2 gaussian width 
      default_nn=int(2*max_area*pcm%gaussian_width/minval(grid%mesh%spacing(1:grid%mesh%sb%dim)))
      
      changed_default_nn = .false.

      do ii=default_nn, 1, -1
        pcm%tess_nn = ii 
        if (pcm_nn_in_mesh(pcm,grid%mesh)) then 
          exit
        else 
          changed_default_nn = .true.
        end if
      end do
      if (changed_default_nn) then
        call messages_write('PCM nearest neighbors have been reduced from ')
        call messages_write(default_nn)
        call messages_write(' to ')
        call messages_write(pcm%tess_nn)
        call messages_new_line()
        call messages_write('in order to fit them into the mesh.')        
        call messages_new_line()
        call messages_write('This may produce unexpected results. ')        
        call messages_warning()
      end if
      

      !%Variable PCMChargeSmearNN
      !%Type integer
      !%Default 2 * max_area * PCMSmearingFactor
      !%Section Hamiltonian::PCM
      !%Description
      !% Defines the number of nearest neighbor mesh-points to be taken around each 
      !% cavity tessera in order to smear the charge when PCMCalcMethod = pcm_poisson.
      !% Setting PCMChargeSmearNN = 1 means first nearest neighbors, PCMChargeSmearNN = 2
      !% second nearest neighbors, and so on.
      !% The default value is such that the neighbor mesh contains 2 times the Gaussian 
      !% width used for the smearing.
      !%End
      
      call parse_variable('PCMChargeSmearNN', pcm%tess_nn, pcm%tess_nn)
      call messages_print_var_value(stdout, "PCMChargeSmearNN", pcm%tess_nn)
      
      call pcm_poisson_sanity_check(pcm, grid%mesh)
      
    end if
    
    

    if (pcm%run_pcm)  call messages_print_stress(stdout)

    if (pcm%calc_method == PCM_CALC_POISSON) then
      SAFE_ALLOCATE( pcm%rho_n(1:grid%mesh%np_part) )
      SAFE_ALLOCATE( pcm%rho_e(1:grid%mesh%np_part) )
    end if 


    SAFE_ALLOCATE( pcm%v_n(1:pcm%n_tesserae) )
    SAFE_ALLOCATE( pcm%q_n(1:pcm%n_tesserae) )
    SAFE_ALLOCATE( pcm%v_n_rs(1:grid%mesh%np) )
    pcm%v_n    = M_ZERO
    pcm%q_n    = M_ZERO
    pcm%v_n_rs = M_ZERO

    SAFE_ALLOCATE( pcm%v_e(1:pcm%n_tesserae) )
    SAFE_ALLOCATE( pcm%q_e(1:pcm%n_tesserae) )
    SAFE_ALLOCATE( pcm%v_e_rs(1:grid%mesh%np) )
    pcm%v_e    = M_ZERO
    pcm%q_e    = M_ZERO
    pcm%v_e_rs = M_ZERO

    POP_SUB(pcm_init)
  end subroutine pcm_init

  ! -----------------------------------------------------------------------------

  subroutine pcm_calc_pot_rs(pcm, mesh, geo, v_h)
    type(pcm_t),             intent(inout) :: pcm
    type(mesh_t),               intent(in) :: mesh  
    type(geometry_t), optional, intent(in) :: geo
    FLOAT,            optional, intent(in) :: v_h(:)
    
    integer :: calc

    PUSH_SUB(pcm_calc_pot_rs)  
    
    ASSERT(present(v_h) .or. present(geo))
    
    if (present(v_h)) calc = PCM_ELECTRONS
    if (present(geo)) calc = PCM_NUCLEI
    
    if (calc == PCM_NUCLEI) then
      call pcm_v_nuclei_cav(pcm%v_n, geo, pcm%tess, pcm%n_tesserae)
      call pcm_charges(pcm%q_n, pcm%qtot_n, pcm%v_n, pcm%matrix, pcm%n_tesserae)
      if (pcm%calc_method == PCM_CALC_POISSON) call pcm_charge_density(pcm, pcm%q_n, pcm%qtot_n, mesh, pcm%rho_n)
      call pcm_pot_rs(pcm, pcm%v_n_rs, pcm%q_n, pcm%rho_n, mesh)      
    end if

    if (calc == PCM_ELECTRONS) then
      call pcm_v_electrons_cav_li(pcm%v_e, v_h, pcm, mesh)
      call pcm_charges(pcm%q_e, pcm%qtot_e, pcm%v_e, pcm%matrix, pcm%n_tesserae) 
      if (pcm%calc_method == PCM_CALC_POISSON) call pcm_charge_density(pcm, pcm%q_e, pcm%qtot_e, mesh, pcm%rho_e)
      call pcm_pot_rs(pcm, pcm%v_e_rs, pcm%q_e, pcm%rho_e, mesh )
    end if
    
    
    POP_SUB(pcm_calc_pot_rs)  
  end subroutine pcm_calc_pot_rs


  ! -----------------------------------------------------------------------------

  !> Calculates the Hartree potential at the tessera representative points by doing 
  !! a 3D linear interpolation. 
  subroutine pcm_v_electrons_cav_li(v_e_cav, v_hartree, pcm, mesh)
    type(pcm_t), intent(in)  :: pcm
    type(mesh_t), intent(in) :: mesh
    FLOAT, intent(in)        :: v_hartree(:) !< (1:mesh%np)
    FLOAT, intent(out)       :: v_e_cav(:)   !< (1:n_tess)

    integer :: ia

    type(mesh_interpolation_t)  :: mesh_interpolation

    PUSH_SUB(pcm_v_electrons_cav_li)    

    v_e_cav = M_ZERO
    
    call mesh_interpolation_init(mesh_interpolation, mesh)

    do ia = 1, pcm%n_tesserae

      call mesh_interpolation_evaluate(mesh_interpolation, v_hartree, pcm%tess(ia)%point, v_e_cav(ia))

    end do

    call mesh_interpolation_end(mesh_interpolation)

    POP_SUB(pcm_v_electrons_cav_li)
  end subroutine pcm_v_electrons_cav_li

  !--------------------------------------------------------------------------------
  
  !> Calculates the classical electrostatic potential geneated by the nuclei at the tesserae.
  !! v_n_cav(ik) = \sum_{I=1}^{natoms} Z_val / |s_{ik} - R_I|
  subroutine pcm_v_nuclei_cav(v_n_cav, geo, tess, n_tess)
    FLOAT,               intent(out)   :: v_n_cav(:) !< (1:n_tess)
    type(geometry_t),    intent(in)    :: geo
    type(pcm_tessera_t),     intent(in)    :: tess(:)    !< (1:n_tess)
    integer,             intent(in)    :: n_tess

    FLOAT   :: diff(1:pcm_dim_space), dist, z_ia
    integer :: ik, ia
    type(species_t), pointer :: spci 

    PUSH_SUB(pcm_v_nuclei_cav)

    diff = M_ZERO
    v_n_cav = M_ZERO

    do ik = 1, n_tess
      do ia = 1, geo%natoms

        diff = geo%atom(ia)%x(1:pcm_dim_space) - tess(ik)%point
        dist = dot_product( diff, diff )
        dist = sqrt(dist)

        spci => geo%atom(ia)%species
        z_ia = species_zval(spci)

        v_n_cav(ik) = v_n_cav(ik) + z_ia/dist       
      end do
    end do

    v_n_cav = -v_n_cav

    POP_SUB(pcm_v_nuclei_cav)
  end subroutine pcm_v_nuclei_cav

  ! -----------------------------------------------------------------------------
  
  !> Calculates the solute-solvent electrostatic interaction energy
  !! E_M-solv = \sum{ik=1}^n_tess { [VHartree(ik) + Vnuclei(ik)]*[q_e(ik) + q_n(ik)] }   
  subroutine pcm_elect_energy(geo, pcm, E_int_ee, E_int_en, E_int_ne, E_int_nn)
    type(geometry_t), intent(in)    :: geo
    type(pcm_t),      intent(in)    :: pcm
    FLOAT,            intent(out)   :: E_int_ee 
    FLOAT,            intent(out)   :: E_int_en 
    FLOAT,            intent(out)   :: E_int_ne 
    FLOAT,            intent(out)   :: E_int_nn 

    FLOAT   :: diff(1:pcm_dim_space)
    FLOAT   :: dist, z_ia
    integer :: ik, ia
    type(species_t), pointer :: spci 

    PUSH_SUB(pcm_elect_energy)

    E_int_ee = M_ZERO
    E_int_en = M_ZERO
    E_int_ne = M_ZERO
    E_int_nn = M_ZERO

    diff = M_ZERO

    do ik = 1, pcm%n_tesserae

      E_int_ee = E_int_ee + pcm%v_e(ik)*pcm%q_e(ik)
      E_int_en = E_int_en + pcm%v_e(ik)*pcm%q_n(ik)

      do ia = 1, geo%natoms

        diff = geo%atom(ia)%x(1:pcm_dim_space) - pcm%tess(ik)%point
        dist = dot_product( diff, diff )
        dist = sqrt(dist)

        spci => geo%atom(ia)%species
        z_ia = -species_zval(spci)

        E_int_ne = E_int_ne + z_ia*pcm%q_e(ik) / dist
        E_int_nn = E_int_nn + z_ia*pcm%q_n(ik) / dist

      end do
    end do

    E_int_ee = M_HALF*E_int_ee
    E_int_en = M_HALF*E_int_en
    E_int_ne = M_HALF*E_int_ne
    E_int_nn = M_HALF*E_int_nn

! print results of the iteration in pcm_info file

    if ( mpi_grp_is_root(mpi_world) ) &
      write(pcm%info_unit,'(3X,I5,5X,F20.8,5X,F20.8,5X,F20.8,5X,F20.8,5X,F20.8,5X,F20.8,5X,F20.8)') &
                              pcm%counter, &
                              units_from_atomic(units_out%energy, E_int_ee ), & 
                              units_from_atomic(units_out%energy, E_int_en ), &
                              units_from_atomic(units_out%energy, E_int_nn ), &
                              units_from_atomic(units_out%energy, E_int_ne ), &
                              units_from_atomic(units_out%energy, E_int_ee +  &
                                                                  E_int_en +  &
                                                                  E_int_nn +  &
                                                                  E_int_ne ), &
                               (pcm%epsilon_0/(pcm%epsilon_0-M_ONE))*pcm%qtot_e, &
                               (pcm%epsilon_0/(pcm%epsilon_0-M_ONE))*pcm%qtot_n


    POP_SUB(pcm_elect_energy)
  end subroutine pcm_elect_energy

  ! -----------------------------------------------------------------------------
  
  !> Creating the list of the nearest 8 cube vertices in real-space 
  !! to calculate the Hartree potential at 'point'
  subroutine nearest_cube_vertices(point, mesh, vert_idx, weight_li)
    FLOAT, intent(in)        :: point(:)!< (1:pcm_dim_space)
    type(mesh_t), intent(in) :: mesh
    integer, intent(out)     :: vert_idx(:)
    FLOAT, intent(out)       :: weight_li(:)

    FLOAT   :: dmin
    integer :: rankmin

    FLOAT   :: coord_0(1:pcm_dim_space)
    integer :: sign_x
    integer :: sign_y
    integer :: sign_z
    integer :: point_0(1:pcm_dim_space)
    integer :: point_f(1:pcm_dim_space)

    PUSH_SUB(nearest_cube_vertices)    

    coord_0 = M_ZERO
    point_0 = 0
    point_f = 0

    vert_idx(1) = mesh_nearest_point(mesh, point, dmin, rankmin)

    coord_0 = mesh%x(vert_idx(1), 1:pcm_dim_space)

    point_0 = NINT(coord_0/mesh%spacing(1:pcm_dim_space))

    sign_x = INT( sign( CNST(1.0), point(1) - coord_0(1) ) )
    sign_y = INT( sign( CNST(1.0), point(2) - coord_0(2) ) )
    sign_z = INT( sign( CNST(1.0), point(3) - coord_0(3) ) )

    weight_li = ABS(point - coord_0)/mesh%spacing(1:pcm_dim_space)

    !FRONT CUBE PLANE
    point_f = point_0

    !POINT P2
    point_f(2) = point_f(2) + sign_y
    vert_idx(2) = index_from_coords(mesh%idx, point_f)

    !POINT P3
    point_f(3) = point_f(3) + sign_z
    vert_idx(3) = index_from_coords(mesh%idx, point_f)

    !POINT P4
    point_f(2) = point_f(2) - sign_y
    vert_idx(4) = index_from_coords(mesh%idx, point_f)

    !REAR CUBE PLANE
    point_f = point_0 

    !POINT P5
    point_f(1) = point_f(1) + sign_x
    vert_idx(5) = index_from_coords(mesh%idx, point_f)

    !POINT P6
    point_f(2) = point_f(2) + sign_y
    vert_idx(6) = index_from_coords(mesh%idx, point_f)

    !POINT P7
    point_f(3) = point_f(3) + sign_z
    vert_idx(7) = index_from_coords(mesh%idx, point_f)

    !POINT P8
    point_f(2) = point_f(2) - sign_y
    vert_idx(8) = index_from_coords(mesh%idx, point_f)

    POP_SUB(nearest_cube_vertices)    
  end subroutine nearest_cube_vertices

  ! -----------------------------------------------------------------------------
  
  !> Calculates the polarization charges at each tessera by using the response matrix 'pcm_mat',
  !! provided the value of the molecular electrostatic potential at 
  !! the tesserae: q_pcm(ia) = \sum_{ib}^{n_tess} pcm_mat(ia,ib)*v_cav(ib).
  subroutine pcm_charges(q_pcm, q_pcm_tot, v_cav, pcm_mat, n_tess)
    FLOAT,   intent(out)   :: q_pcm(:)     !< (1:n_tess)
    FLOAT,   intent(out)   :: q_pcm_tot
    FLOAT,   intent(in)    :: v_cav(:)     !< (1:n_tess)
    FLOAT,   intent(in)    :: pcm_mat(:,:) !< (1:n_tess, 1:n_tess)
    integer, intent(in)    :: n_tess

    integer :: ia, ib
    type(profile_t), save :: prof_init

    PUSH_SUB(pcm_charges)
    call profiling_in(prof_init, 'PCM_CHARGES') 
    
    q_pcm     = M_ZERO
    q_pcm_tot = M_ZERO

    do ia = 1, n_tess
      do ib = 1, n_tess
        q_pcm(ia) = q_pcm(ia) + pcm_mat(ia,ib)*v_cav(ib) !< transpose matrix might speed up
      end do
      q_pcm_tot = q_pcm_tot + q_pcm(ia)
    end do
    
    call profiling_out(prof_init)
    
    POP_SUB(pcm_charges)
  end subroutine pcm_charges

  ! -----------------------------------------------------------------------------    
  !> Check wether the nearest neighbor requested are in the mesh or not
  logical function pcm_nn_in_mesh(pcm, mesh) result(in_mesh)
      type(pcm_t),     intent(in) :: pcm 
      type(mesh_t),    intent(in) :: mesh

      integer :: ia, nm(1:MAX_DIM), ipt, npt, i1, i2, i3
      FLOAT :: posrel(1:MAX_DIM)
      integer :: pt
      
      PUSH_SUB(pcm_nn_in_mesh)
  
      in_mesh = .true.
      do ia = 1, pcm%n_tesserae
  
        posrel(1:mesh%sb%dim) = pcm%tess(ia)%point(1:mesh%sb%dim)/mesh%spacing(1:mesh%sb%dim)

        nm(1:mesh%sb%dim) = floor(posrel(1:mesh%sb%dim))
  
        ! Get the nearest neighboring points
        ipt = 0
        do i1 = -pcm%tess_nn +1 , pcm%tess_nn
          do i2 = -pcm%tess_nn +1 , pcm%tess_nn
            do i3 = -pcm%tess_nn +1 , pcm%tess_nn
              ipt = ipt+1
              pt = mesh%idx%lxyz_inv(i1 + nm(1), i2 + nm(2), i3 + nm(3))

              if (pt <= 0 .or. pt > mesh%np_part_global) then 
                in_mesh = .false.
                POP_SUB(pcm_nn_in_mesh)
                return 
              end if 
          
            end do
          end do
        end do
      end do 

      POP_SUB(pcm_nn_in_mesh)
    
    end function pcm_nn_in_mesh
  
  ! -----------------------------------------------------------------------------  
  !> Check that all the required nearest neighbors are prensent in the mesh
  subroutine pcm_poisson_sanity_check(pcm, mesh)
    type(pcm_t),     intent(in) :: pcm 
    type(mesh_t),    intent(in) :: mesh
    
    
    integer :: ia, nm(1:MAX_DIM), ipt, npt, i1, i2, i3
    FLOAT :: posrel(1:MAX_DIM)
    integer :: pt
    
    PUSH_SUB(pcm_poisson_sanity_check)

    if ( .not. pcm_nn_in_mesh(pcm, mesh) ) then 
      message(1) = 'The simulation box is too small to contain all the requested'
      message(2) = 'nearest neighbors for each tessera.'
      message(3) = 'Consider using a larger box or reduce PCMChargeSmearNN.'
      call messages_warning(3)
    end if 
    
    POP_SUB(pcm_poisson_sanity_check)
    
  end subroutine pcm_poisson_sanity_check
  
  ! -----------------------------------------------------------------------------  
  !> Generates the polarization charge density smearing the charge with a gaussian 
  !> distribution on the mesh nearest neighboring points   of each tessera.
  subroutine pcm_charge_density(pcm, q_pcm, q_pcm_tot, mesh, rho)
    type(pcm_t),     intent(inout) :: pcm 
    FLOAT,           intent(in)    :: q_pcm(:)     !< (1:n_tess)
    FLOAT,           intent(in)    :: q_pcm_tot
    type(mesh_t),    intent(in)    :: mesh
    FLOAT,           intent(out)   :: rho(:)
    
    
    integer :: ia
    FLOAT   :: Norm, qtot, RR, XX(1:MAX_DIM), PP(1:MAX_DIM)
    
    ! nearest neighbor variables 
    integer :: nm(1:MAX_DIM), ipoint
    FLOAT :: posrel(1:MAX_DIM)
    integer :: npt, ipt
    integer :: i1, i2, i3
    integer, ALLOCATABLE :: pt(:)  
    FLOAT,   ALLOCATABLE :: lrho(:) ! local charge density on a tessera NN 
    logical :: inner_point, boundary_point
    
    !profiling and debug 
    type(profile_t), save :: prof_init
    integer  :: ierr

    
    PUSH_SUB(pcm_charge_density)

    call profiling_in(prof_init, 'PCM_CHARGE_DENSITY') 
    

    npt = (2*pcm%tess_nn)**mesh%sb%dim
    SAFE_ALLOCATE(pt(1:npt))
    SAFE_ALLOCATE(lrho(1:npt))

    pt = 0 
    rho = M_ZERO
    
    do ia = 1, pcm%n_tesserae
      
      PP(1:mesh%sb%dim) = pcm%tess(ia)%point(1:mesh%sb%dim)
      posrel(1:mesh%sb%dim) = PP(1:mesh%sb%dim)/mesh%spacing(1:mesh%sb%dim)

      nm(1:mesh%sb%dim) = floor(posrel(1:mesh%sb%dim))
      
      ! Get the nearest neighboring points
      ipt = 0
      do i1 = -pcm%tess_nn +1 , pcm%tess_nn
        do i2 = -pcm%tess_nn +1 , pcm%tess_nn
          do i3 = -pcm%tess_nn +1 , pcm%tess_nn
            ipt = ipt+1
            pt(ipt) = mesh%idx%lxyz_inv(i1 + nm(1), i2 + nm(2), i3 + nm(3))
          end do
        end do
      end do
      
      
      ! Extrapolate the tessera point charge with a gaussian distritibution
      ! to the neighboring points 
      ! rho(r) = N exp(-|r-sk|^2/(alpha*Ak))
      Norm = 0 
      lrho = 0
      do ipt = 1, npt
        
        ! Check the point is inside the mesh skip otherwise
        if (pt(ipt) > 0 .and. pt(ipt) <= mesh%np_part_global) then
          
          if(mesh%parallel_in_domains) then
            pt(ipt) = vec_global2local(mesh%vp, pt(ipt), mesh%vp%partno)
            boundary_point = pt(ipt) > mesh%np + mesh%vp%np_ghost
            inner_point = pt(ipt) > 0 .and. pt(ipt) <= mesh%np

            if(boundary_point .or. inner_point) then
              XX(1:mesh%sb%dim) = mesh%x(pt(ipt),1:mesh%sb%dim)
            else 
              cycle
            end if
          
          else
            XX(1:mesh%sb%dim) = mesh%x(pt(ipt),1:mesh%sb%dim)
          end if
        
          RR = sum((XX(1:mesh%sb%dim) - PP(1:mesh%sb%dim))**2)
          Norm = Norm + exp(-RR/(pcm%tess(ia)%area*pcm%gaussian_width))
          lrho(ipt) = lrho(ipt) + exp(-RR/(pcm%tess(ia)%area*pcm%gaussian_width))
          
        end if 
      end do

      ! reduce the local density scattered among nodes
      call comm_allreduce(mesh%mpi_grp%comm, lrho, npt)
      
      ! Normalize the integral to the tessera point charge q_pcm(ia)
      Norm = sum(lrho(1:npt)) * mesh%volume_element
      Norm = q_pcm(ia)/Norm       
      lrho(:) = lrho(:) * Norm

      ! Add up the local density to the full charge density 
      do ipt = 1, npt
        
        if (pt(ipt) > 0 .and. pt(ipt) <= mesh%np_part_global) then
        
          if(mesh%parallel_in_domains) then
               boundary_point = pt(ipt) > mesh%np + mesh%vp%np_ghost
               inner_point = pt(ipt) > 0 .and. pt(ipt) <= mesh%np
               if(boundary_point .or. inner_point) rho(pt(ipt)) = rho(pt(ipt)) + lrho(ipt)
          else
               rho(pt(ipt)) = rho(pt(ipt)) + lrho(ipt)           
          end if
          
        end if
      end do
            
    end do 

    

    if (debug%info) then  
      qtot = dmf_integrate(mesh, rho)
      call messages_write(' PCM charge density integrates to q = ')
      call messages_write(qtot)
      call messages_write(' (qtot = ')
      call messages_write(q_pcm_tot)
      call messages_write(')')
      call messages_info()
      
      !   Keep this here for debug purposes.    
      call dio_function_output(io_function_fill_how("VTK"), &
                              ".", "rho_pcm",  mesh, rho, unit_one, ierr)
    end if  

    SAFE_DEALLOCATE_A(pt)
    SAFE_DEALLOCATE_A(lrho)
    
    call profiling_out(prof_init)
    
    POP_SUB(pcm_charge_density)
  end subroutine pcm_charge_density


  ! -----------------------------------------------------------------------------  
  !> Generates the potential 'v_pcm' in real-space.
  subroutine pcm_pot_rs(pcm, v_pcm, q_pcm, rho, mesh)
    type(pcm_t),     intent(inout) :: pcm 
    FLOAT,           intent(inout) :: v_pcm(:)!< (1:mesh%np) running serially np=np_global
    FLOAT,           intent(in)    :: q_pcm(:)!< (1:n_tess)
    FLOAT,           intent(inout) :: rho(:)
    type(mesh_t),    intent(in)    :: mesh

    type(profile_t), save :: prof_init
    integer  :: ierr

    PUSH_SUB(pcm_pot_rs)

    call profiling_in(prof_init, 'PCM_POT_RS') 
    
    v_pcm = M_ZERO


    select case(pcm%calc_method)
    case(PCM_CALC_DIRECT)
      call pcm_pot_rs_direct(v_pcm, q_pcm, pcm%tess, pcm%n_tesserae, mesh, pcm%gaussian_width)

    case(PCM_CALC_POISSON)
      call pcm_pot_rs_poisson(pcm, v_pcm, rho, mesh)
    case default

      message(1) = "BAD BAD BAD"
      call messages_fatal(1,only_root_writes = .true.)

    end select
    
    
    if (debug%info) then  
      !   Keep this here for debug purposes.    
      call dio_function_output(io_function_fill_how("VTK"), &
                              ".", "v_pcm",  mesh, v_pcm, unit_one, ierr)
    end if  
    
    call profiling_out(prof_init)

    POP_SUB(pcm_pot_rs)
  end subroutine pcm_pot_rs

  
  ! -----------------------------------------------------------------------------  
  !> Generates the potential 'v_pcm' in real-space solving the poisson equation for rho
  subroutine pcm_pot_rs_poisson(pcm, v_pcm, rho, mesh)
    type(pcm_t),   intent(inout) :: pcm 
    FLOAT,         intent(inout) :: v_pcm(:)
    FLOAT,         intent(inout) :: rho(:)
    type(mesh_t),     intent(in) :: mesh
      
    PUSH_SUB(pcm_pot_rs_poisson)
    
        call dpoisson_solve(psolver, v_pcm, rho)
    
    POP_SUB(pcm_pot_rs_poisson)
  end subroutine pcm_pot_rs_poisson
  

  ! -----------------------------------------------------------------------------  
  !> Generates the potential 'v_pcm' in real-space by direct sum.
  subroutine pcm_pot_rs_direct(v_pcm, q_pcm, tess, n_tess, mesh, width_factor)
    FLOAT,           intent(out) :: v_pcm(:)!< (1:mesh%np) running serially np=np_global
    FLOAT,           intent(in)  :: q_pcm(:)!< (1:n_tess)
    FLOAT,           intent(in)  :: width_factor
    integer,         intent(in)  :: n_tess  
    type(mesh_t),    intent(in)  :: mesh
    type(pcm_tessera_t), intent(in)  :: tess(:) !< (1:n_tess)

    FLOAT, parameter :: p_1 = CNST(0.119763)
    FLOAT, parameter :: p_2 = CNST(0.205117)
    FLOAT, parameter :: q_1 = CNST(0.137546)
    FLOAT, parameter :: q_2 = CNST(0.434344)
    FLOAT            :: arg
    FLOAT            :: term
    integer 	     :: ip
    integer          :: ia

    PUSH_SUB(pcm_pot_rs_direct)

    v_pcm = M_ZERO

    if (width_factor /= M_ZERO) then
     !< regularized PCM field
     do ia = 1, n_tess
      do ip = 1, mesh%np
        ! Computing the distances between tesserae and grid points.
        call mesh_r(mesh, ip, term, origin=tess(ia)%point)
        arg = term/sqrt( tess(ia)%area*width_factor )        
        term = ( 1 + p_1*arg + p_2*arg**2 )/( 1 + q_1*arg + q_2*arg**2 + p_2*arg**3 )
        v_pcm(ip) = v_pcm(ip) + q_pcm(ia)*term/sqrt( tess(ia)%area*width_factor )
      end do
     end do
 
     v_pcm = M_TWO*v_pcm/sqrt(M_Pi)

    else
     !< standard PCM field
     do ia = 1, n_tess
      do ip = 1, mesh%np
        ! Computing the distances between tesserae and grid points.
        call mesh_r(mesh, ip, term, origin=tess(ia)%point)
        v_pcm(ip) = v_pcm(ip) + q_pcm(ia)/term         
      end do
     end do     
    endif
    

    POP_SUB(pcm_pot_rs_direct)
  end subroutine pcm_pot_rs_direct




  ! -----------------------------------------------------------------------------
  
  !> Generates the PCM response matrix. J. Tomassi et al. Chem. Rev. 105, 2999 (2005). 
  subroutine pcm_matrix(eps, tess, n_tess, pcm_mat )
    FLOAT, intent(in)           :: eps
    type(pcm_tessera_t), intent(in) :: tess(:)      !< (1:n_tess)
    integer, intent(in)         :: n_tess
    FLOAT, intent(out)          :: pcm_mat(:,:) !< (1:n_tess, 1:n_tess)

    integer :: i, info
    integer, allocatable :: iwork(:)
    FLOAT, allocatable :: mat_tmp(:,:)

    PUSH_SUB(pcm_matrix)

    !> Conforming the S_I matrix
    SAFE_ALLOCATE( s_mat_act(1:n_tess, 1:n_tess) )
    call s_i_matrix(n_tess, tess)

    !> Defining the matrix S_E=S_I/eps
    SAFE_ALLOCATE( Sigma(1:n_tess, 1:n_tess) )
    Sigma = s_mat_act/eps

    !> Conforming the D_I matrix
    SAFE_ALLOCATE( d_mat_act(1:n_tess, 1:n_tess) )
    call d_i_matrix(n_tess, tess)

    !> Defining the matrix D_E=D_I 
    SAFE_ALLOCATE( Delta(1:n_tess, 1:n_tess) )
    Delta = d_mat_act

    !> Start conforming the PCM matrix
    pcm_mat = -d_mat_act

    do i=1, n_tess
      pcm_mat(i,i) = pcm_mat(i,i) + M_TWO*M_Pi  
    end do

    SAFE_DEALLOCATE_A(d_mat_act) 

    SAFE_ALLOCATE( iwork(1:n_tess) )

    !> Solving for X = S_I^-1*(2*Pi - D_I) 
    ! FIXME: use interface, or routine in lalg_adv_lapack_inc.F90
    call dgesv(n_tess, n_tess, s_mat_act, n_tess, iwork, pcm_mat, n_tess, info)        

    SAFE_DEALLOCATE_A(iwork)

    SAFE_DEALLOCATE_A(s_mat_act)

    !> Computing -S_E*S_I^-1*(2*Pi - D_I)
    pcm_mat = -matmul( Sigma, pcm_mat ) 

    do i=1, n_tess
      pcm_mat(i,i) = pcm_mat(i,i) + M_TWO*M_Pi
    end do

    pcm_mat = pcm_mat - Delta

    SAFE_ALLOCATE( mat_tmp(1:n_tess,1:n_tess) )
    mat_tmp = M_ZERO

    SAFE_ALLOCATE( d_mat_act(1:n_tess,1:n_tess) )  
    call d_i_matrix(n_tess, tess)

    mat_tmp = transpose(d_mat_act)        

    mat_tmp = matmul(Sigma, mat_tmp)

    mat_tmp = mat_tmp + M_TWO*M_Pi*Sigma

    SAFE_DEALLOCATE_A(d_mat_act)

    SAFE_ALLOCATE( s_mat_act(1:n_tess, 1:n_tess) )
    call s_i_matrix(n_tess, tess)

    mat_tmp = mat_tmp + M_TWO*M_Pi*s_mat_act - matmul(Delta, s_mat_act)

    SAFE_DEALLOCATE_A(s_mat_act)
    SAFE_DEALLOCATE_A(Sigma)
    SAFE_DEALLOCATE_A(Delta)

    SAFE_ALLOCATE( iwork(1:n_tess) )

    !> Solving for [(2*pi - D_E)*S_I + S_E*(2*Pi + D_I*)]*X = [(2*Pi - D_E) - S_E*S_I^-1*(2*Pi - D_I)]    
    call dgesv(n_tess, n_tess, mat_tmp, n_tess, iwork, pcm_mat, n_tess, info)		  		  

    SAFE_DEALLOCATE_A(iwork)
    SAFE_DEALLOCATE_A(mat_tmp)

    pcm_mat = -pcm_mat

    !   Testing
    if ( gamess_benchmark ) then
      do i=1, n_tess
        mat_gamess(i,:) = pcm_mat(i,:)/tess(i)%area
      end do
    end if

    POP_SUB(pcm_matrix)

  end subroutine pcm_matrix

  ! -----------------------------------------------------------------------------
  
  subroutine s_i_matrix(n_tess, tess)
    integer,         intent(in)    :: n_tess
    type(pcm_tessera_t), intent(in)    :: tess(:)

    integer :: ii, jj

    s_mat_act = M_ZERO 

    do ii = 1, n_tess
      do jj = ii, n_tess

        s_mat_act(ii,jj) = s_mat_elem_I(tess(ii), tess(jj))
        if (ii /= jj) s_mat_act(jj,ii) = s_mat_act(ii,jj) !symmetric matrix

      end do
    end do

  end subroutine s_i_matrix

  ! -----------------------------------------------------------------------------

  subroutine d_i_matrix(n_tess, tess)
    integer,         intent(in)    :: n_tess
    type(pcm_tessera_t), intent(in)    :: tess(:)

    integer :: ii, jj

    d_mat_act = M_ZERO 

    do ii = 1, n_tess
      do jj = 1, n_tess !< non-symmetric matrix

        d_mat_act(ii,jj) = d_mat_elem_I(tess(ii), tess(jj))

      end do
    end do

  end subroutine d_i_matrix

  ! -----------------------------------------------------------------------------

  !> electrostatic Green function in vacuo:
  !! G_I(r,r^\prime) = 1 / | r - r^\prime |
  FLOAT function s_mat_elem_I(tessi, tessj)
    type(pcm_tessera_t), intent(in) :: tessi
    type(pcm_tessera_t), intent(in) :: tessj

    FLOAT, parameter :: M_SD_DIAG    = CNST(1.0694)
    FLOAT, parameter :: M_DIST_MIN   = CNST(0.1)

    FLOAT :: diff(1:pcm_dim_space)
    FLOAT :: dist
    FLOAT :: s_diag
    FLOAT :: s_off_diag

    s_diag = M_ZERO
    s_off_diag = M_ZERO
    diff = M_ZERO

    diff = tessi%point - tessj%point

    dist = dot_product( diff, diff )
    dist = sqrt(dist)

    if  (dist == M_ZERO) then

      s_diag = M_SD_DIAG*sqrt( M_FOUR*M_Pi/tessi%area )
      s_mat_elem_I = s_diag

    else

      if ( dist > M_DIST_MIN ) s_off_diag = M_ONE/dist 
      s_mat_elem_I = s_off_diag

    end if

  end function s_mat_elem_I

  ! -----------------------------------------------------------------------------

  !> Gradient of the Green function in vacuo GRAD[G_I(r,r^\prime)]
  FLOAT function d_mat_elem_I(tessi, tessj)
    type(pcm_tessera_t), intent(in) :: tessi
    type(pcm_tessera_t), intent(in) :: tessj

    FLOAT, parameter :: M_SD_DIAG    = CNST(1.0694)
    FLOAT, parameter :: M_DIST_MIN   = CNST(0.04)

    FLOAT :: diff(1:pcm_dim_space)
    FLOAT :: dist
    FLOAT :: d_diag
    FLOAT :: d_off_diag

    d_diag = M_ZERO
    d_off_diag = M_ZERO
    diff = M_ZERO

    diff = tessi%point - tessj%point

    dist = dot_product( diff, diff )
    dist = sqrt(dist)

    if (dist == M_ZERO) then
      !> Diagonal matrix elements  
      d_diag = M_SD_DIAG*sqrt(M_FOUR*M_Pi*tessi%area)
      d_diag = -d_diag/(M_TWO*tessi%r_sphere)
      d_mat_elem_I = d_diag

    else
      !> off-diagonal matrix elements  
      if (dist > M_DIST_MIN) then
        d_off_diag = dot_product( diff, tessj%normal(:) )	
        d_off_diag = d_off_diag*tessj%area/dist**3
      end if

      d_mat_elem_I = d_off_diag

    end if

  end function d_mat_elem_I

  ! -----------------------------------------------------------------------------

  !> It builds the solute cavity surface and calculates the vertices,
  !! representative points and areas of the tesserae by using the 
  !! Gauss-Bonnet theorem.
  subroutine cav_gen(i_count, tess_sphere, nesf, sfe, nts, cts, unit_pcminfo)
    integer,              intent(in)    :: i_count
    integer,              intent(in)    :: tess_sphere
    type(pcm_sphere_t),   intent(inout) :: sfe(:) !< (1:pcm%n_spheres)
    integer,              intent(in)    :: nesf
    integer,              intent(out)   :: nts
    type(pcm_tessera_t),  intent(out)   :: cts(:) !< (1:pcm%n_tesserae)
    integer,              intent(in)    :: unit_pcminfo


    integer, parameter :: dim_angles = 24
    integer, parameter :: dim_ten = 10
    integer, parameter :: dim_vertices = 122
    integer, parameter :: n_tess_sphere = 60
    integer, parameter :: max_vertices = 6
    integer, parameter :: mxts = 10000

    FLOAT :: thev(1:dim_angles)
    FLOAT :: fiv(1:dim_angles)
    FLOAT :: fir
    FLOAT :: cv(1:dim_vertices, 1:pcm_dim_space)
    FLOAT :: th
    FLOAT :: fi
    FLOAT :: cth
    FLOAT :: sth

    FLOAT :: xctst(tess_sphere*n_tess_sphere)
    FLOAT :: yctst(tess_sphere*n_tess_sphere)
    FLOAT :: zctst(tess_sphere*n_tess_sphere)
    FLOAT :: ast(tess_sphere*n_tess_sphere)
    FLOAT :: nctst(pcm_dim_space, tess_sphere*n_tess_sphere)

    FLOAT :: pts(1:pcm_dim_space, 1:dim_ten)
    FLOAT :: pp(1:pcm_dim_space)
    FLOAT :: pp1(1:pcm_dim_space)
    FLOAT :: ccc(1:pcm_dim_space, 1:dim_ten)

    integer :: idum(1:n_tess_sphere*max_vertices)
    integer :: jvt1(1:max_vertices,1:n_tess_sphere)
    integer :: isfet(1:dim_ten*dim_angles)

    integer :: ii
    integer :: ia
    integer :: ja
    integer :: nn
    integer :: nsfe
    integer :: its
    integer :: n1
    integer :: n2
    integer :: n3
    integer :: nv
    integer :: i_tes

    FLOAT :: xen
    FLOAT :: yen
    FLOAT :: zen
    FLOAT :: ren
    FLOAT :: area
    FLOAT :: test
    FLOAT :: test2
    FLOAT :: rij
    FLOAT :: dnorm

    FLOAT :: xi
    FLOAT :: yi
    FLOAT :: zi
    FLOAT :: xj
    FLOAT :: yj
    FLOAT :: zj

    FLOAT :: vol
    FLOAT :: stot
    FLOAT :: prod
    FLOAT :: dr  

    logical :: band_iter

    PUSH_SUB(cav_gen)

    !> Angles corresponding to the vertices and centres of a polyhedron
    !! within a sphere of unitary radius and centered at the origin
    data thev/ CNST(0.6523581398) , CNST(1.107148718)  , CNST(1.382085796) , &
      CNST(1.759506858)  , CNST(2.034443936)  , CNST(2.489234514) , &
      CNST(0.3261790699) , CNST(0.5535743589), &
      CNST(0.8559571251) , CNST(0.8559571251) , CNST(1.017221968) , &
      CNST(1.229116717)  , CNST(1.229116717)  , CNST(1.433327788) , &
      CNST(1.570796327)  , CNST(1.570796327)  , CNST(1.708264866) , &
      CNST(1.912475937)  , CNST(1.912475937)  , CNST(2.124370686) , &
      CNST(2.285635528)  , CNST(2.285635528)  , CNST(2.588018295) , &
      CNST(2.815413584) /
    data fiv/                       CNST(0.6283185307) , M_ZERO            , &
      CNST(0.6283185307) , M_ZERO             , CNST(0.6283185307), &
      M_ZERO             , CNST(0.6283185307) , M_ZERO, 	     &
      CNST(0.2520539002) , CNST(1.004583161)  , CNST(0.6283185307), &
      CNST(0.3293628477) , CNST(0.9272742138) , M_ZERO, 	     &
      CNST(0.3141592654) , CNST(0.9424777961) , CNST(0.6283185307), &
      CNST(0.2989556830) , CNST(0.9576813784) , M_ZERO, 	     &
      CNST(0.3762646305) , CNST(0.8803724309) , CNST(0.6283188307), &
      M_ZERO /
    data fir / CNST(1.256637061)  /

    !> the vector idum, contained in the matrix jvt1, indicates the vertices 
    !! of the tesserae (using less than 19 continuations)
    data (idum(ii),ii = 1, 280) /                                   &
      1, 6, 2, 32, 36, 37, 1, 2, 3, 33, 32, 38, 1, 3, 4, 34,         &
      33, 39, 1, 4, 5, 35, 34, 40, 1, 5, 6, 36, 35, 41, 7, 2, 6, 51, &
      42, 37, 8, 3, 2, 47, 43, 38, 9, 4, 3, 48, 44, 39, 10, 5, 4,    &
      49, 45, 40, 11, 6, 5, 50, 46, 41, 8, 2, 12, 62, 47, 52, 9,     &
      3, 13, 63, 48, 53, 10, 4, 14, 64, 49, 54, 11, 5, 15, 65, 50,   &
      55, 7, 6, 16, 66, 51, 56, 7, 12, 2, 42, 57, 52, 8, 13, 3,      &
      43, 58, 53, 9, 14, 4, 44, 59, 54, 10, 15, 5, 45, 60, 55, 11,   &
      16, 6, 46, 61, 56, 8, 12, 18, 68, 62, 77, 9, 13, 19, 69, 63,   &
      78, 10, 14, 20, 70, 64, 79, 11, 15, 21, 71, 65, 80, 7, 16,     &
      17, 67, 66, 81, 7, 17, 12, 57, 67, 72, 8, 18, 13, 58, 68, 73,  &
      9, 19, 14, 59, 69, 74, 10, 20, 15, 60, 70, 75, 11, 21, 16,     &
      61, 71, 76, 22, 12, 17, 87, 82, 72, 23, 13, 18, 88, 83, 73,    &
      24, 14, 19, 89, 84, 74, 25, 15, 20, 90, 85, 75, 26, 16, 21,    &
      91, 86, 76, 22, 18, 12, 82, 92, 77, 23, 19, 13, 83, 93, 78,    &
      24, 20, 14, 84, 94, 79, 25, 21, 15, 85, 95, 80, 26, 17, 16,    &
      86, 96, 81, 22, 17, 27, 102, 87, 97, 23, 18, 28, 103, 88, 98,  &
      24, 19, 29, 104, 89, 99, 25, 20, 30, 105, 90, 100, 26, 21,     &
      31, 106, 91, 101, 22, 28, 18, 92, 107, 98, 23, 29, 19, 93 /
    data (idum(ii),ii = 281,360) / 				      &
      108, 99, 24, 30, 20, 94, 109, 100, 25, 31, 21, 95, 110, 101,   &
      26, 27, 17, 96, 111, 97, 22, 27, 28, 107, 102, 112, 23, 28,    &
      29, 108, 103, 113, 24, 29, 30, 109, 104, 114, 25, 30, 31,      &
      110, 105, 115, 26, 31, 27, 111, 106, 116, 122, 28, 27, 117,    &
      118, 112, 122, 29, 28, 118, 119, 113, 122, 30, 29, 119, 120,   &
      114, 122, 31, 30, 120, 121, 115, 122, 27, 31, 121, 117, 116 /

    if (i_count == 0 .and.  mpi_grp_is_root(mpi_world)) then
      if (tess_sphere == 1) then
        write(unit_pcminfo,'(A1)')  '#' 
        write(unit_pcminfo,'(A34)') '# Number of tesserae / sphere = 60'
        write(unit_pcminfo,'(A1)')  '#' 
      else
        write(unit_pcminfo,'(A1)')  '#' 
        write(unit_pcminfo,'(A35)') '# Number of tesserae / sphere = 240' 
        write(unit_pcminfo,'(A1)')  '#' 
      end if
    end if

    !> geometrical data are converted to Angstrom and back transformed
    !! to Bohr at the end of the subroutine.
    dr = CNST(0.01)
    dr = dr*P_a_B

    sfe(:)%x = sfe(:)%x*P_a_B
    sfe(:)%y = sfe(:)%y*P_a_B
    sfe(:)%z = sfe(:)%z*P_a_B
    sfe(:)%r = sfe(:)%r*P_a_B

    vol  = M_ZERO
    stot = M_ZERO
    jvt1 = reshape(idum,(/6,60/))

    !> Coordinates of vertices of tesserae in a sphere with unit radius.
    !! the matrix 'cv' and 'xc', 'yc', 'zc' conatin the vertices and 
    !! the centers of 240 tesserae. The matrix 'jvt1(i,j)' denotes the index
    !! of the i-th vertex of the j-th big tessera. On each big tessera
    !! the 6 vertices are ordered as follows:
    !!  
    !!                      1
    !!
    !!                   4     5
    !!
    !!                3     6     2

    cv(1,1)   =  M_ZERO
    cv(1,2)   =  M_ZERO
    cv(1,3)   =  M_ONE

    cv(122,1) =  M_ZERO
    cv(122,2) =  M_ZERO
    cv(122,3) = -M_ONE

    ii = 1
    do ia = 1, dim_angles
      th = thev(ia)
      fi = fiv(ia)
      cth = cos(th)
      sth = sin(th)
      do ja = 1, 5
        fi = fi + fir
        if (ja == 1) fi = fiv(ia)
        ii = ii + 1
        cv(ii,1) = sth*cos(fi)
        cv(ii,2) = sth*sin(fi)
        cv(ii,3) = cth
      end do
    end do

    !> Controls whether the tessera is covered or need to be reshaped it
    nn = 0
    do nsfe = 1, nesf
      xen = sfe(nsfe)%x
      yen = sfe(nsfe)%y
      zen = sfe(nsfe)%z
      ren = sfe(nsfe)%r

      xctst(:) = M_ZERO
      yctst(:) = M_ZERO
      zctst(:) = M_ZERO
      ast(:)   = M_ZERO

      do its = 1, n_tess_sphere 
        do i_tes = 1, tess_sphere
          if (tess_sphere == 1) then
            n1 = jvt1(1,its)
            n2 = jvt1(2,its)
            n3 = jvt1(3,its)
          else
            if (i_tes == 1)      then
              n1 = jvt1(1,its)
              n2 = jvt1(5,its)
              n3 = jvt1(4,its)
            elseif (i_tes == 2)  then 
              n1 = jvt1(4,its)
              n2 = jvt1(6,its)
              n3 = jvt1(3,its)
            elseif (i_tes == 3)  then
              n1 = jvt1(4,its)
              n2 = jvt1(5,its)
              n3 = jvt1(6,its)
            elseif (i_tes == 4)  then
              n1 = jvt1(2,its)
              n2 = jvt1(6,its)
              n3 = jvt1(5,its)
            end if
          end if

          pts(1,1) = cv(n1,1)*ren + xen
          pts(2,1) = cv(n1,3)*ren + yen
          pts(3,1) = cv(n1,2)*ren + zen

          pts(1,2) = cv(n2,1)*ren + xen
          pts(2,2) = cv(n2,3)*ren + yen
          pts(3,2) = cv(n2,2)*ren + zen

          pts(1,3) = cv(n3,1)*ren + xen
          pts(2,3) = cv(n3,3)*ren + yen
          pts(3,3) = cv(n3,2)*ren + zen

          pp(:)  = M_ZERO
          pp1(:) = M_ZERO
          nv = 3

          call subtessera(sfe, nsfe, nesf, nv, pts ,ccc, pp, pp1, area)

          if (area == M_ZERO) cycle

          xctst(tess_sphere*(its-1) + i_tes)   = pp(1)
          yctst(tess_sphere*(its-1) + i_tes)   = pp(2)
          zctst(tess_sphere*(its-1) + i_tes)   = pp(3)
          nctst(:,tess_sphere*(its-1) + i_tes) = pp1(:)
          ast(tess_sphere*(its-1) + i_tes)     = area
          isfet(tess_sphere*(its-1) + i_tes)   = nsfe

        end do
      end do !> loop through the tesseare on the sphere 'nsfe'

      do its = 1, n_tess_sphere*tess_sphere

        if (ast(its) == M_ZERO) cycle
        nn = nn + 1

        if (nn > mxts) then !> check the total number of tessera
          write(message(1),'(a,I5,a,I5)') "total number of tesserae", nn, ">",mxts
          call messages_warning(1)     
        end if

        if (i_count ==  1) then
          cts(nn)%point(1)  = xctst(its)
          cts(nn)%point(2)  = yctst(its)
          cts(nn)%point(3)  = zctst(its)
          cts(nn)%normal(:) = nctst(:,its)
          cts(nn)%area      = ast(its)
          cts(nn)%r_sphere  = sfe(isfet(its))%r
        end if

      end do
    end do !> loop through the spheres

    nts = nn

    if (i_count == 1) then

      !> checks if two tesseare are too close
      test = CNST(0.1)
      test2 = test*test

      band_iter = .false.
      do while (.not.(band_iter))
        band_iter = .true.

        loop_ia: do ia = 1, nts-1
          if (cts(ia)%area == M_ZERO) cycle
          xi = cts(ia)%point(1)
          yi = cts(ia)%point(2)
          zi = cts(ia)%point(3)

          loop_ja: do ja = ia+1, nts
            if (cts(ja)%area == M_ZERO) cycle
            xj = cts(ja)%point(1)
            yj = cts(ja)%point(2)
            zj = cts(ja)%point(3)

            rij = (xi-xj)**2 + (yi-yj)**2 + (zi-zj)**2

            if (rij > test2) cycle

            if ( mpi_grp_is_root(mpi_world) ) &
              write(unit_pcminfo,'(A40,I4,A5,I4,A4,F8.4,A13,F8.4,A3)' ) &
                '# Warning: The distance between tesserae', &
                ia,' and ', ja,' is ',sqrt(rij),' A, less than', test,' A.'

            !> calculating the coordinates of the new tessera weighted by the areas
            xi = (xi*cts(ia)%area + xj*cts(ja)%area) / (cts(ia)%area + cts(ja)%area)
            yi = (yi*cts(ia)%area + yj*cts(ja)%area) / (cts(ia)%area + cts(ja)%area)
            zi = (zi*cts(ia)%area + zj*cts(ja)%area) / (cts(ia)%area + cts(ja)%area)

            cts(ia)%point(1) = xi
            cts(ia)%point(2) = yi
            cts(ia)%point(3) = zi

            !> calculating the normal vector of the new tessera weighted by the areas
            cts(ia)%normal = (cts(ia)%normal*cts(ia)%area + cts(ja)%normal*cts(ja)%area)
            dnorm = sqrt( dot_product(cts(ia)%normal, cts(ia)%normal) )
            cts(ia)%normal = cts(ia)%normal/dnorm

            !> calculating the sphere radius of the new tessera weighted by the areas
            cts(ia)%r_sphere = ( cts(ia)%r_sphere*cts(ia)%area + cts(ja)%r_sphere*cts(ja)%area ) / &
              ( cts(ia)%area + cts(ja)%area )

            !> calculating the area of the new tessera
            cts(ia)%area = cts(ia)%area + cts(ja)%area

            !> deleting tessera ja
            do ii = ja+1, nts
              cts(ii-1) = cts(ii)
            end do
            nts = nts -1 
            band_iter = .false.
            exit loop_ia

          end do loop_ja
        end do loop_ia
      end do !> while loop

      !> Calculates the cavity volume: vol = \sum_{its=1}^nts A_{its} s*n/3.
      vol = M_ZERO
      do its = 1, nts
        prod = dot_product( cts(its)%point, cts(its)%normal ) 
        vol  = vol + cts(its)%area * prod / M_THREE
        stot = stot + cts(its)%area
      end do

      if ( mpi_grp_is_root(mpi_world) ) then
        write(unit_pcminfo, '(A2)')  '# '
        write(unit_pcminfo, '(A29,I4)')    '# Total number of tesserae = ', nts
        write(unit_pcminfo, '(A30,F12.6)') '# Cavity surface area (A^2) = ' , stot
        write(unit_pcminfo, '(A24,F12.6)') '# Cavity volume (A^3) = '       , vol
        write(unit_pcminfo, '(A2)')  '# '
      end if

      !> transforms results into Bohr.
      cts(:)%area     = cts(:)%area*P_Ang*P_Ang
      cts(:)%point(1) = cts(:)%point(1)*P_Ang
      cts(:)%point(2) = cts(:)%point(2)*P_Ang
      cts(:)%point(3) = cts(:)%point(3)*P_Ang
      cts(:)%r_sphere = cts(:)%r_sphere*P_Ang
    end if

    sfe(:)%x=sfe(:)%x*P_Ang
    sfe(:)%y=sfe(:)%y*P_Ang
    sfe(:)%z=sfe(:)%z*P_Ang
    sfe(:)%r=sfe(:)%r*P_Ang

    
    POP_SUB(cav_gen)
  end subroutine cav_gen

  
  ! -----------------------------------------------------------------------------


  !> find the uncovered region for each tessera and computes the area,
  !! the representative point (pp) and the unitary normal vector (pp1)
  subroutine subtessera(sfe, ns, nesf, nv, pts, ccc, pp, pp1, area)
    type(pcm_sphere_t), intent(in) :: sfe(:) !< (1:nesf)
    integer, intent(in)        :: ns 
    integer, intent(in)        :: nesf
    integer, intent(inout)     :: nv
    FLOAT, intent(inout)       :: pts(:,:) !< (1:pcm_dim_space, 1:dim_ten)
    FLOAT, intent(out)         :: ccc(:,:) !< (1:pcm_dim_space, 1:dim_ten)
    FLOAT, intent(out)         :: pp(:)    !< (1:pcm_dim_space)
    FLOAT, intent(out)         :: pp1(:)   !< (1:pcm_dim_space)
    FLOAT, intent(out)         :: area

    FLOAT, parameter   :: tol = -CNST(1.0e-10)
    integer, parameter :: dim_ten = 10

    integer :: intsph(1:dim_ten)
    integer :: nsfe1
    integer :: na
    integer :: icop
    integer :: ll
    integer :: iv1
    integer :: iv2
    integer :: ii
    integer :: icut
    integer :: jj

    FLOAT  :: p1(1:pcm_dim_space)
    FLOAT  :: p2(1:pcm_dim_space)
    FLOAT  :: p3(1:pcm_dim_space)
    FLOAT  :: p4(1:pcm_dim_space)
    FLOAT  :: point(1:pcm_dim_space)
    FLOAT  :: pscr(1:pcm_dim_space,1:dim_ten)
    FLOAT  :: cccp(1:pcm_dim_space,1:dim_ten)
    FLOAT  :: pointl(1:pcm_dim_space,1:dim_ten)
    FLOAT  :: diff(1:pcm_dim_space)

    integer :: ind(1:dim_ten)
    integer :: ltyp(1:dim_ten)
    integer :: intscr(1:dim_ten)

    FLOAT  :: delr
    FLOAT  :: delr2
    FLOAT  :: rc
    FLOAT  :: rc2
    FLOAT  :: dnorm
    FLOAT  :: dist
    FLOAT  :: de2

    PUSH_SUB(subtessera)

    p1     = M_ZERO
    p2     = M_ZERO
    p3     = M_ZERO
    p4     = M_ZERO
    point  = M_ZERO
    pscr   = M_ZERO
    cccp   = M_ZERO
    pointl = M_ZERO
    diff   = M_ZERO
    area   = M_ZERO

    do jj=1, 3
      ccc(1,jj) = sfe(ns)%x
      ccc(2,jj) = sfe(ns)%y
      ccc(3,jj) = sfe(ns)%z
    end do

    intsph = ns
    do nsfe1 = 1, nesf 
      if (nsfe1 == ns) cycle
      do jj =1, nv
        intscr(jj) = intsph(jj)
        pscr(:,jj) = pts(:,jj)
        cccp(:,jj) = ccc(:,jj)
      end do

      icop = 0
      ind = 0
      ltyp = 0

      do ii = 1, nv
        delr2 = ( pts(1,ii) - sfe(nsfe1)%x )**2 + ( pts(2,ii) - sfe(nsfe1)%y )**2 + &
          ( pts(3,ii) - sfe(nsfe1)%z )**2
        delr = sqrt(delr2)
        if (delr < sfe(nsfe1)%r) then
          ind(ii) = 1
          icop = icop + 1
        end if
      end do

      if (icop == nv) then 
        POP_SUB(subtessera)
        return
      end if

      do ll = 1, nv
        iv1 = ll
        iv2 = ll+1
        if (ll == nv) iv2 = 1
        IF ( (ind(iv1) == 1) .and. (ind(iv2) == 1) ) then
          ltyp(ll) = 0
        else if ( (ind(iv1) == 0) .and. (ind(iv2) == 1) ) then
          ltyp(ll) = 1
        else if ( (ind(iv1) == 1) .and. (ind(iv2) == 0) ) then
          ltyp(ll) = 2
        else if ( (ind(iv1) == 0) .and. (ind(iv2) == 0) ) then
          ltyp(ll) = 4
          diff = ccc(:,ll) - pts(:,ll)
          rc2 = dot_product(diff,diff)
          rc = sqrt(rc2)

          do ii = 1, 11
            point = pts(:,iv1) + ii * (pts(:,iv2) - pts(:,iv1)) / 11
            point = point - CCC(:,ll)
            dnorm = sqrt( dot_product(point, point) )
            point = point * rc / dnorm + CCC(:,ll)

            dist = sqrt(  (point(1) - sfe(nsfe1)%x)**2 + ( point(2) - sfe(nsfe1)%y)**2 &
              + ( point(3) - sfe(nsfe1)%z)**2  )

            if ( (dist - sfe(nsfe1)%r) < tol) then
              ltyp(ll) = 3
              pointl(:,ll) = point
              exit
            end if

          end do
        end if
      end do

      icut = 0
      do ll = 1, nv
        if ( (ltyp(ll) == 1) .or. (ltyp(ll) == 2) ) icut = icut + 1
        if (ltyp(ll) == 3) icut = icut + 2
      end do
      icut = icut / 2
      if (icut > 1) then 
        POP_SUB(subtessera)
        return
      end if

      na = 1
      do ll = 1, nv

        if (ltyp(ll) == 0) cycle
        iv1 = ll
        iv2 = ll + 1
        if (ll == nv) iv2 = 1

        if (ltyp(ll) == 1) then
          pts(:,na) = pscr(:,iv1)
          ccc(:,na) = cccp(:,iv1)
          intsph(na) = intscr(iv1)
          na = na + 1
          p1 = pscr(:,iv1)
          p2 = pscr(:,iv2)
          p3 = cccp(:,iv1)

          call inter(sfe, p1, p2, p3, p4, nsfe1, 0)
          pts(:,na) = p4

          de2 = ( sfe(nsfe1)%x - sfe(ns)%x )**2 + ( sfe(nsfe1)%y - sfe(ns)%y )**2 + &
            ( sfe(nsfe1)%z - sfe(ns)%z )**2

          ccc(1,na) = sfe(ns)%x + ( sfe(nsfe1)%x - sfe(ns)%x)* &
            ( sfe(ns)%r**2 - sfe(nsfe1)%r**2 + de2 ) / (M_TWO*de2)

          ccc(2,na) = sfe(ns)%y + ( sfe(nsfe1)%y - sfe(ns)%y)* &
            ( sfe(ns)%r**2 - sfe(nsfe1)%r**2 + de2 ) / (M_TWO*de2)

          ccc(3,na) = sfe(ns)%z + ( sfe(nsfe1)%z - sfe(ns)%z)* &
            ( sfe(ns)%r**2 - sfe(nsfe1)%r**2 + de2 ) / (M_TWO*de2)

          intsph(na) = nsfe1
          na = na + 1
        end if

        if (ltyp(ll) == 2) then
          p1 = pscr(:,iv1)
          p2 = pscr(:,iv2)
          p3 = cccp(:,iv1)

          call inter( sfe, p1, p2, p3, p4, nsfe1, 1 )
          pts(:,na) = p4
          ccc(:,na) = cccp(:,iv1)
          intsph(na) = intscr(iv1)
          na = na + 1
        end if

        if (ltyp(ll) == 3) then
          pts(:,na) = pscr(:,iv1)
          ccc(:,na) = cccp(:,iv1)
          intsph(na) = intscr(iv1)
          na = na + 1
          p1 = pscr(:,iv1)
          p2 = pointl(:,ll)
          p3 = cccp(:,iv1)

          call inter( sfe, p1, p2, p3, p4, nsfe1, 0 )
          pts(:,na) = p4

          de2 = ( sfe(nsfe1)%x - sfe(ns)%x )**2 + ( sfe(nsfe1)%y - sfe(ns)%y )**2 + &
            ( sfe(nsfe1)%z - sfe(ns)%z )**2

          ccc(1,na) = sfe(ns)%x + ( sfe(nsfe1)%x - sfe(ns)%x )* &
            ( sfe(ns)%r**2 - sfe(nsfe1)%r**2 + de2 ) / (M_TWO*de2)

          ccc(2,na) = sfe(ns)%y + ( sfe(nsfe1)%y - sfe(ns)%y )* &
            ( sfe(ns)%r**2 - sfe(nsfe1)%r**2 + de2 ) / (M_TWO*de2)

          ccc(3,na) = sfe(ns)%z + ( sfe(nsfe1)%z - sfe(ns)%z )* &
            ( sfe(ns)%r**2 - sfe(nsfe1)%r**2 + de2 ) / (M_TWO*de2)

          intsph(na) = nsfe1
          na = na + 1
          p1 = pointl(:,ll)
          p2 = pscr(:,iv2)
          p3 = cccp(:,iv1)

          call inter( sfe, p1, p2, p3, p4, nsfe1, 1 )
          pts(:,na) = p4
          ccc(:,na) = cccp(:,iv1)
          intsph(na) = intscr(iv1)
          na = na + 1
        end if

        if (ltyp(ll) == 4) then
          pts(:,na) = pscr(:,iv1)
          ccc(:,na) = cccp(:,iv1)
          intsph(na) = intscr(iv1)
          na = na + 1
        end if
      end do

      nv = na - 1
      if (nv > 10) then
        message(1) = "Too many vertices on the tessera"
        call messages_fatal(1)     
      end if
    end do

    call gaubon( sfe, nv, ns, pts, ccc, pp, pp1, area, intsph)
    
    POP_SUB(subtessera)
  end subroutine subtessera
  
  ! -----------------------------------------------------------------------------

  !    !> Finds the point 'p4', on the arc 'p1'-'p2' developed from 'p3',
  !    !! which is on the surface of sphere 'ns'. p4 is a linear combination 
  !! of p1 and p2 with the 'alpha' parameter optimized iteratively.
  subroutine inter( sfe, p1, p2, p3, p4, ns, ia)
    type(pcm_sphere_t), intent(in) :: sfe(:) !< (1:nesf)
    FLOAT, intent(in)          :: p1(:)  !< (1:pcm_dim_space)
    FLOAT, intent(in)          :: p2(:)  !< (1:pcm_dim_space)
    FLOAT, intent(in)          :: p3(:)  !< (1:pcm_dim_space)
    FLOAT, intent(out)         :: p4(:)  !< (1:pcm_dim_space)
    integer, intent(in) :: ns
    integer, intent(in) :: ia

    FLOAT, parameter :: tol = CNST(1.0e-08)

    integer :: m_iter
    FLOAT  :: r
    FLOAT  :: alpha
    FLOAT  :: delta
    FLOAT  :: dnorm
    FLOAT  :: diff
    FLOAT  :: diff_vec(1:pcm_dim_space)
    logical :: band_iter

    diff_vec = M_ZERO

    diff_vec = p1 - p3
    r = sqrt( dot_product(diff_vec, diff_vec) )

    alpha = M_HALF
    delta = M_ZERO
    m_iter = 1

    band_iter = .false.
    do while(.not.(band_iter))
      if (m_iter > 1000) then
        message(1) = "Too many iterations inside subrotuine inter"
        call messages_fatal(1)     
      end if

      band_iter = .true.

      alpha = alpha + delta
      dnorm = M_ZERO

      p4 = p1 + alpha*(p2-p1)-p3
      dnorm = sqrt( dot_product(p4,p4) )
      p4 = p4*r/dnorm + p3
      diff =( p4(1) - sfe(ns)%x )**2 + ( p4(2) - sfe(ns)%y )**2 + ( p4(3) - sfe(ns)%z )**2
      diff = sqrt(diff) - sfe(ns)%r

      if ( abs(diff) < tol ) return

      if (ia == 0) then
        if (diff > M_ZERO) delta =  M_ONE/(M_TWO**(m_iter+1))
        if (diff < M_ZERO) delta = -M_ONE/(M_TWO**(m_iter+1))
        m_iter = m_iter + 1
        band_iter = .false.
      end if

      if (ia == 1) then
        if (diff > M_ZERO) delta = -M_ONE/(M_TWO**(m_iter+1))
        if (diff < M_ZERO) delta =  M_ONE/(M_TWO**(m_iter+1))
        m_iter = m_iter + 1
        band_iter = .false.
      end if
    end do
    
  end subroutine inter

  ! -----------------------------------------------------------------------------

  !> Use the Gauss-Bonnet theorem to calculate the area of the 
  !! tessera with vertices 'pts(3,nv)'. 
  !! Area = R^2 [ 2pi + S(Phi(N)cosT(N)) - S(Beta(N)) ]
  !! Phi(n): length of the arc in radians of the side 'n'. 
  !! T(n): azimuthal angle for the side 'n'
  !! Beta(n): external angle respect to vertex 'n'.
  subroutine gaubon( sfe, nv, ns, pts, ccc, pp, pp1, area, intsph )
    type(pcm_sphere_t), intent(in) :: sfe(:)    !< (1:nesf)
    FLOAT, intent(in)          :: pts(:,:)  !< (1:pcm_dim_space,1:dim_ten) 
    FLOAT, intent(in)          :: ccc(:,:)  !< (1:pcm_dim_space,1:dim_ten)
    FLOAT, intent(inout)       :: pp(:)     !< (1:pcm_dim_space)
    FLOAT, intent(inout)       :: pp1(:)    !< (1:pcm_dim_space)
    integer, intent(in)        :: intsph(:) !< (1:dim_ten)
    FLOAT, intent(out)         :: area
    integer, intent(in)        :: nv
    integer, intent(in)        :: ns

    FLOAT :: p1(1:pcm_dim_space), p2(1:pcm_dim_space), p3(1:pcm_dim_space)
    FLOAT :: u1(1:pcm_dim_space), u2(1:pcm_dim_space)
    FLOAT :: point_1(1:pcm_dim_space), point_2(1:pcm_dim_space)
    FLOAT :: tpi, sum1, dnorm, dnorm1, dnorm2
    FLOAT :: cosphin, phin, costn, sum2, betan
    integer :: nsfe1, ia, nn, n0, n1, n2

    PUSH_SUB(gaubon)

    point_1 = M_ZERO
    point_2 = M_ZERO
    p1      = M_ZERO
    p2      = M_ZERO
    p3      = M_ZERO
    u1      = M_ZERO
    u2      = M_ZERO

    tpi = M_TWO*M_Pi
    sum1 = M_ZERO
    do nn = 1, nv
      point_1 = pts(:,nn) - ccc(:,nn)
      if (nn < nv) then
        point_2 = pts(:,nn+1) - ccc(:,nn)
      else
        point_2 = pts(:,1) - ccc(:,nn)
      end if

      dnorm1 = sqrt( dot_product(point_1, point_1) )
      dnorm2 = sqrt( dot_product(point_2, point_2) )
      cosphin = dot_product(point_1, point_2) / (dnorm1*dnorm2)

      if (cosphin >  M_ONE) cosphin =  M_ONE
      if (cosphin < -M_ONE) cosphin = -M_ONE

      phin = acos(cosphin)
      nsfe1 = intsph(nn)

      point_1(1) = sfe(nsfe1)%x - sfe(ns)%x
      point_1(2) = sfe(nsfe1)%y - sfe(ns)%y
      point_1(3) = sfe(nsfe1)%z - sfe(ns)%z

      dnorm1 = sqrt( dot_product(point_1, point_1) )

      if (dnorm1 == M_ZERO) dnorm1 = M_ONE

      point_2(1) = pts(1,nn) - sfe(ns)%x
      point_2(2) = pts(2,nn) - sfe(ns)%y
      point_2(3) = pts(3,nn) - sfe(ns)%z

      dnorm2 = sqrt( dot_product(point_2, point_2) )

      costn  = dot_product(point_1, point_2)/(dnorm1*dnorm2)
      sum1 = sum1 + phin * costn
    end do

    sum2 = M_ZERO
    !> Loop over the vertices
    do nn = 1, nv
      p1 = M_ZERO
      p2 = M_ZERO    
      p3 = M_ZERO  

      n1 = nn
      if (nn > 1)   n0 = nn - 1
      if (nn == 1)  n0 = nv
      if (nn < nv)  n2 = nn + 1
      if (nn == nv) n2 = 1

      p1 = pts(:,n1) - ccc(:,n0)
      p2 = pts(:,n0) - ccc(:,n0)
      call vecp(p1, p2, p3, dnorm)
      p2 = p3    

      call vecp(p1, p2, p3, dnorm)
      u1 = p3/dnorm

      p1 = pts(:,n1) - ccc(:,n1)
      p2 = pts(:,n2) - ccc(:,n1)
      call vecp(p1, p2, p3, dnorm)
      p2 = p3

      call vecp(p1, p2, p3, dnorm)
      u2 = p3/dnorm

      betan = acos( dot_product(u1, u2) )
      sum2 = sum2 + (M_Pi - betan)
    end do

    !> computes the area of the tessera
    area = sfe(ns)%r*sfe(ns)%r*(tpi + sum1 - sum2)

    !> computes the representative point
    pp = M_ZERO

    do ia = 1, nv
      pp(1) = pp(1) + ( pts(1,ia) - sfe(ns)%x )
      pp(2) = pp(2) + ( pts(2,ia) - sfe(ns)%y )
      pp(3) = pp(3) + ( pts(3,ia) - sfe(ns)%z )
    end do

    dnorm = M_ZERO
    dnorm = sqrt( dot_product(pp,pp) )

    pp(1) = sfe(ns)%x + pp(1) * sfe(ns)%r / dnorm
    pp(2) = sfe(ns)%y + pp(2) * sfe(ns)%r / dnorm
    pp(3) = sfe(ns)%z + pp(3) * sfe(ns)%r / dnorm

    !> finds the internal normal at the representative point
    pp1(1) = (pp(1) - sfe(ns)%x) / sfe(ns)%r
    pp1(2) = (pp(2) - sfe(ns)%y) / sfe(ns)%r
    pp1(3) = (pp(3) - sfe(ns)%z) / sfe(ns)%r

    !> If the area of the tessera is negative (0^-), due to numerical errors, is discarded
    if (area < M_ZERO) area = M_ZERO

    
    POP_SUB(gaubon)
  end subroutine gaubon

  ! -----------------------------------------------------------------------------

  !> calculates the vectorial product p3 = p1 x p2
  subroutine vecp(p1, p2, p3, dnorm)
    FLOAT, intent(in)  :: P1(:) !< (1:pcm_dim_space)
    FLOAT, intent(in)  :: P2(:) !< (1:pcm_dim_space)
    FLOAT, intent(out) :: P3(:) !< (1:pcm_dim_space)
    FLOAT, intent(out) :: dnorm

    p3 = M_ZERO
    p3(1) = p1(2)*p2(3) - p1(3)*p2(2)
    p3(2) = p1(3)*p2(1) - p1(1)*p2(3)
    P3(3) = p1(1)*p2(2) - p1(2)*p2(1)

    dnorm = M_ZERO
    dnorm = sqrt( dot_product(p3, p3) )
    
  end subroutine vecp

  subroutine pcm_end(pcm)
    type(pcm_t), intent(inout) :: pcm

    PUSH_SUB(pcm_end)

    SAFE_DEALLOCATE_A(pcm%spheres)
    SAFE_DEALLOCATE_A(pcm%tess)
    SAFE_DEALLOCATE_A(pcm%matrix)
    SAFE_DEALLOCATE_A(pcm%q_e)
    SAFE_DEALLOCATE_A(pcm%q_n) 
    SAFE_DEALLOCATE_A(pcm%v_e)
    SAFE_DEALLOCATE_A(pcm%v_n)
    SAFE_DEALLOCATE_A(pcm%v_e_rs)
    SAFE_DEALLOCATE_A(pcm%v_n_rs)
!     cSAFE_DEALLOCATE_A(pcm%ind_vh)
!     cSAFE_DEALLOCATE_A(pcm%arg_li)

    if (pcm%calc_method == PCM_CALC_POISSON) then
      SAFE_DEALLOCATE_A( pcm%rho_n)
      SAFE_DEALLOCATE_A( pcm%rho_e)
    end if 


    if ( mpi_grp_is_root(mpi_world) ) call io_close(pcm%info_unit)

    POP_SUB(pcm_end)
  end subroutine pcm_end

  ! -----------------------------------------------------------------------------
  !> Update pcm potential
  logical function pcm_update(this, time) result(update)
      type(pcm_t), intent(inout) :: this
      FLOAT,       intent(in)    :: time

      this%iter = this%iter + 1 
      update = (this%iter <= 6 .or. mod(this%iter, this%update_iter) == 0)
      
      if (debug%info .and. update) then
        call messages_write(' PCM potential updated')
        call messages_new_line()
        call messages_write(' PCM update iteration counter: ')
        call messages_write(this%iter)
        call messages_info()
      end if

    end function pcm_update

  ! -----------------------------------------------------------------------------
  !> get the vdw radius
  FLOAT function pcm_get_vdw_radius(species, pcm_vdw_type)  result(vdw_r)
      type(species_t), intent(in) :: species
      integer,         intent(in) :: pcm_vdw_type
  
      integer            :: ia
      integer, parameter :: upto_Xe = 54
      FLOAT              :: vdw_radii(1:upto_Xe) !< van der Waals radii in Angstrom for elements H-Xe reported 
         !  by Stefan Grimme in J. Comput. Chem. 27: 1787-1799, 2006 
         !  except for C, N and O, reported in J. Chem. Phys. 120, 3893 (2004). 
      data (vdw_radii(ia), ia=1, upto_Xe)                                                                                  / &
       !H                                                                                                       He 
        CNST(1.001),                                                                                            CNST(1.012), &
       !Li           Be                        B            C            N            O            F            Ne
        CNST(0.825), CNST(1.408),              CNST(1.485), CNST(2.000), CNST(1.583), CNST(1.500), CNST(1.287), CNST(1.243), & 
       !Na           Mg                        Al           Si           P            S            Cl           Ar 
        CNST(1.144), CNST(1.364),              CNST(1.639), CNST(1.716), CNST(1.705), CNST(1.683), CNST(1.639), CNST(1.595), & 
       !K            Ca 
        CNST(1.485), CNST(1.474),                                                                                            & 
       !>      Sc -- Zn       <!                                        
        CNST(1.562), CNST(1.562),                                                                    & 
        CNST(1.562), CNST(1.562),                                                                    & 
        CNST(1.562), CNST(1.562),                                                                    & 
        CNST(1.562), CNST(1.562),                                                                    & 
        CNST(1.562), CNST(1.562),                                                                  & 
       !Ga           Ge           As           Se           Br           Kr  
        CNST(1.650), CNST(1.727), CNST(1.760), CNST(1.771), CNST(1.749), CNST(1.727), & 
                                  !Rb           Sr           !>      Y -- Cd        <!                                        
        CNST(1.628), CNST(1.606), CNST(1.639), CNST(1.639),                                                                  & 
        CNST(1.639), CNST(1.639),                                                            & 
        CNST(1.639), CNST(1.639),                                                            & 
        CNST(1.639), CNST(1.639),                                                                  & 
        CNST(1.639), CNST(1.639),                                                                  & 
       !In                  Sn           Sb           Te           I            Xe 
        CNST(2.672), CNST(1.804), CNST(1.881), CNST(1.892), CNST(1.892), CNST(1.881)  / 

        select case (pcm_vdw_type)

        case (PCM_VDW_OPTIMIZED)
          if (species_z(species) > upto_Xe) then
             write(message(1),'(a,a)') "The van der Waals radius is missing for element ", trim(species_label(species))
             write(message(2),'(a)') "Use PCMVdWRadii = pcm_vdw_species, for other vdw radii values" 
             call messages_fatal(2)
          end if
          ia = species_z(species)
          vdw_r = vdw_radii(ia)*P_Ang

        case (PCM_VDW_SPECIES)
          vdw_r = species_vdw_radius(species)
          if(vdw_r< CNST(0.0)) then
            call messages_write('The default vdW radius for species '//trim(species_label(species))//':')
            call messages_write(' is not defined. ')
            call messages_write(' Add a positive vdW radius value in %Species block. ')
            call messages_fatal()
          end if
        end select

  end function pcm_get_vdw_radius

end module pcm_m

!! Local Variables:
!! mode: f90
!! coding: utf-8
!! End:

