/* spacegroup_data.c */
/* Copyright (C) 2008 Atsushi Togo */

/* This program is free software; you can redistribute it and/or */
/* modify it under the terms of the GNU General Public License */
/* as published by the Free Software Foundation; either version 2 */
/* of the License, or (at your option) any later version. */

/* This program is distributed in the hope that it will be useful, */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the */
/* GNU General Public License for more details. */

/* You should have received a copy of the GNU General Public License */
/* along with this program; if not, write to the Free Software */
/* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. */

#ifdef DEBUG
#include "debug.h"
#endif

#include <stdio.h>
#include <stdlib.h>
#include "spacegroup_data.h"
#include "bravais.h"
#include "mathfunc.h"
#include "spacegroup.h"
#include "symmetry.h"

static int get_spacegroup_data_special_case(const Symmetry * symmetry,
					    const Bravais * bravais, const int rot_class[],
					    const double symprec);
static int check_class_table(const int a[32], const int b[32]);
static void get_class_table(int class_talbe[32], const int rot_class[], const int size);
static SpacegroupData get_spacegroup_data_table(const int num);


int tbl_get_spacegroup_data(const Symmetry * symmetry, const Bravais * bravais,
			    const int rot_class[], const double symprec)
{
  int i, spacegroup = 0, class_table[32];
  SpacegroupData spacegroup_data;

  get_class_table(class_table, rot_class, symmetry->size);

  class_table[0] = get_spacegroup_data_special_case(symmetry, bravais,
						    rot_class, symprec);

#ifdef DEBUG
  debug_print("%d  ", class_table[0]);
  for (i = 1; i < 11; i++)
    debug_print("%d ", class_table[i]);
  debug_print(" ");
  for (i = 11; i < 21; i++)
    debug_print("%d ", class_table[i]);
  debug_print(" ");
  for (i = 21; i < 32; i++)
    debug_print("%d ", class_table[i]);
  debug_print("\n");
#endif


  for (i = 0; i < 238; i++) {
    spacegroup_data = get_spacegroup_data_table(i);
    if (check_class_table(class_table, spacegroup_data.class_table))
      spacegroup = spacegroup_data.spacegroup;
  }

  return spacegroup;
}

static int get_spacegroup_data_special_case(const Symmetry * symmetry,
					    const Bravais * bravais,
					    const int rot_class[],
					    const double symprec)
{
  int i, j, k, size, direction;
  Holohedry holohedry;
  Centering centering;
  int class_table[32], directions[3];
  double trans[3][3];

  int tables[9][32] = {
    {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 2, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 2, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 0, 0, 0, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 0, 0, 0, 0, 1, 1, 3, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 2, 0, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
    {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0,
     0, 0, 0, 0, 0, 0, 0, 0, 0, 0}
  };

  holohedry = bravais->holohedry;
  centering = bravais->centering;
  size = symmetry->size;
  get_class_table(class_table, rot_class, size);

  if (centering == NO_CENTER && holohedry != RHOMB) {

    /* 27 (Pcc2) or 32 (Pba2) */
    if (check_class_table(class_table, tables[0])) {

      for (i = 0; i < size; i++)
	if (rot_class[i] == PRIMARY_2_AXIS) {
	  if (symmetry->rot[i][0][0] == 1)
	    direction = 0;
	  if (symmetry->rot[i][1][1] == 1)
	    direction = 1;
	  if (symmetry->rot[i][2][2] == 1)
	    direction = 2;
	}

      for (i = 0; i < size; i++) {
	if (rot_class[i] == A_B_C_PLANE){
	  if (mat_Dabs(symmetry->trans[i][direction]) > symprec)
	    return 1;
	}
      }
      return 2;
    }

    /* 55 (Pbam) or 57 (Pbcm) */
    if (check_class_table(class_table, tables[1])) {

      for (i = 0; i < size; i++)
	if (rot_class[i] == M_PLANE) {
	  if (symmetry->rot[i][0][0] == -1)
	    direction = 0;
	  if (symmetry->rot[i][1][1] == -1)
	    direction = 1;
	  if (symmetry->rot[i][2][2] == -1)
	    direction = 2;
	}

      for (i = 0; i < size; i++) {
	if (rot_class[i] == A_B_C_PLANE){
	  if (mat_Dabs(symmetry->trans[i][direction]) > symprec)
	    return 2;
	}
      }	  
      return 1;
    }

    /* 56 (Pccn) or 60 (Pbcn) */
    if (check_class_table(class_table, tables[2])) {

      for (i = 0; i < size; i++)
	if (rot_class[i] == N_PLANE) {
	  if (symmetry->rot[i][0][0] == -1)
	    direction = 0;
	  if (symmetry->rot[i][1][1] == -1)
	    direction = 1;
	  if (symmetry->rot[i][2][2] == -1)
	    direction = 2;
	}

      for (i = 0; i < size; i++) {
	if (rot_class[i] == A_B_C_PLANE){
	  if (mat_Dabs(symmetry->trans[i][direction]) < symprec) {
	    return 2;
	  }
	}
      }
      return 1;
    }
  }

  if (centering == BODY) {

    /* 23 (I222) or 24 (I2_12_12_1), 197 (I23) or 199 (I2_13) */
    if (check_class_table(class_table, tables[3]) ||
	check_class_table(class_table, tables[4])) {

      directions[0] = 0;
      directions[1] = 0;
      directions[2] = 0;

      for (i = 0; i < size; i++)

	if (rot_class[i] == PRIMARY_2_1_AXIS)

	  for (j = 0; j < 3; j++)

	    if (symmetry->rot[i][j][j] == 1) {
	      directions[j] = 1;

	      if (mat_Dabs(symmetry->trans[i][j]) < symprec)
		for (k = 0; k < 3; k++)
		  trans[j][k] = symmetry->trans[i][k];
	      else
		for (k = 0; k < 3; k++)
		  trans[j][k] =
		    symmetry->trans[i][k] + 0.5;

	      for (k = 0; k < 3; k++)
		trans[j][k] =
		  trans[j][k] - mat_Nint(trans[j][k]);
	    }

      if ((!directions[0] || !directions[1]) && !directions[2]) {
	fprintf(stderr, "spglib: Invalid symmetry of  space group recognition\n");
	fprintf(stderr, "spglib: of 23 and 24, and 197 and 199.\n");
      }

      if (mat_Dabs(trans[1][0] - trans[2][0]) > symprec ||
	  mat_Dabs(trans[0][1] - trans[2][1]) > symprec ||
	  mat_Dabs(trans[0][2] - trans[1][2]) > symprec)
	return 2;
      else
	return 1;
    }

  }

  /* Differentiate same class table: primitive and rhombohedral */
  if (check_class_table(class_table, tables[5]) ||	/* 143 (0) or 146 (1) */
      check_class_table(class_table, tables[6]) ||	/* 147 (0) or 148 (1) */
      check_class_table(class_table, tables[7]) ||	/* 156 (0) or 160 (1) */
      check_class_table(class_table, tables[8])) {      /* 158 (0) or 161 (1) */
    if (holohedry == RHOMB)
      return 2;
    else
      return 1;
  }
  /* default */
  return 0;
}

static void get_class_table(int class_table[32], const int rot_class[],
			    const int size)
{
  int i;
  for (i = 0; i < 32; i++)
    class_table[i] = 0;

  for (i = 0; i < size; i++)
    class_table[rot_class[i] % 100]++;
}

static int check_class_table(const int a[32], const int b[32])
{
  int i;

  for (i = 0; i < 32; i++)
    if (a[i] != b[i])
      return 0;

  return 1;
}


static SpacegroupData get_spacegroup_data_table(const int num)
{
  SpacegroupData spacegroup_data[238] = {
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      1},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      2},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      3},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      4},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      6},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      7},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      7},
    {
      {1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      143},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       2, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      144},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 2, 0, 0, 0, 0, 0, 0, 0, 0},
      145},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      10},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      11},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      13},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      13},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      14},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      14},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      16},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      17},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      18},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      19},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      25},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      26},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      29},
    {
      {1, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      27},
    {
      {2, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      32},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      28},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      30},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      31},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      33},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      34},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      75},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       0, 0, 2, 0, 0, 0, 0, 0, 0, 0},
      76},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      77},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 2, 0, 0, 0, 0, 0},
      78},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      81},
    {
      {1, 0, 0, 2, 0, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      147},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      149},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      150},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3,
       2, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      151},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3,
       0, 2, 0, 0, 0, 0, 0, 0, 0, 0},
      153},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       2, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      152},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 2, 0, 0, 0, 0, 0, 0, 0, 0},
      154},
    {
      {1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      156},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      157},
    {
      {1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      158},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      159},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      168},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       2, 0, 0, 0, 0, 2, 0, 0, 0, 0},
      169},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       0, 2, 0, 0, 0, 0, 0, 0, 0, 2},
      170},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 2, 0, 0, 0, 0, 2, 0, 0, 0},
      171},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       2, 0, 0, 0, 0, 0, 0, 0, 2, 0},
      172},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 2, 0, 0},
      173},
    {
      {0, 2, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      174},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      47},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      48},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      49},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      50},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 2, 0, 0, 0, 0, 0, 2, 1, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      51},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      52},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 2, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      53},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      54},
    {
      {1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      55},
    {
      {2, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      57},
    {
      {1, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      56},
    {
      {2, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 1, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      60},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 2, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      58},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 1, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      59},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      61},
    {
      {0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 1, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      62},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      89},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 2, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      90},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2,
       0, 0, 2, 0, 0, 0, 0, 0, 0, 0},
      91},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 2,
       0, 0, 0, 0, 2, 0, 0, 0, 0, 0},
      95},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 2,
       0, 0, 2, 0, 0, 0, 0, 0, 0, 0},
      92},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 2,
       0, 0, 0, 0, 2, 0, 0, 0, 0, 0},
      96},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      93},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      94},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 2, 0, 0, 4, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      99},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 2, 0, 0, 2, 2, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      100},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      101},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 2, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      102},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 2, 0, 0, 0, 0, 2, 0, 2, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      103},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 2, 0, 0, 0, 0, 0, 2, 2, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      104},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 0, 2, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      105},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0, 2, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      106},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 2, 0, 0, 1, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      83},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      84},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 2, 0, 0, 0, 0, 0, 1, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      85},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      86},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      111},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      112},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      113},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      114},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      115},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      116},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      117},
    {
      {0, 0, 2, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      118},
    {
      {0, 0, 0, 2, 0, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      162},
    {
      {0, 0, 0, 2, 0, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      163},
    {
      {0, 0, 0, 2, 3, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      164},
    {
      {0, 0, 0, 2, 3, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      165},
    {
      {0, 2, 0, 2, 0, 1, 0, 0, 1, 1, 2, 0, 0, 0, 2, 1, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      175},
    {
      {0, 2, 0, 2, 0, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 0, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 2, 0, 0},
      176},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 1, 2, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      177},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3,
       2, 0, 0, 0, 0, 2, 0, 0, 0, 0},
      178},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3,
       0, 2, 0, 0, 0, 0, 0, 0, 0, 2},
      179},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3,
       0, 2, 0, 0, 0, 0, 2, 0, 0, 0},
      180},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3,
       2, 0, 0, 0, 0, 0, 0, 0, 2, 0},
      181},
    {
      {0, 0, 0, 0, 3, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 3,
       0, 0, 0, 0, 0, 0, 0, 2, 0, 0},
      182},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 2, 3, 0, 3, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      183},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 2, 0, 3, 0, 3, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      184},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 3, 3, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 2, 0, 0},
      185},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 3, 0, 0, 3, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 2, 0, 0},
      186},
    {
      {0, 2, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      187},
    {
      {0, 2, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 3, 0, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      188},
    {
      {0, 2, 0, 0, 3, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 0, 3, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      189},
    {
      {0, 2, 0, 0, 3, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 0, 0, 3, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      190},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      195},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      198},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 2, 0, 0, 5, 0, 0, 0, 0, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      123},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 2, 0, 0, 1, 0, 2, 0, 2, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      124},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 2, 0, 0, 2, 2, 0, 1, 0, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      125},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 2, 0, 0, 0, 0, 0, 3, 2, 0, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      126},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 2, 0, 0, 3, 2, 0, 0, 0, 2, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      127},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 2, 0, 0, 1, 0, 0, 2, 2, 2, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      128},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 2, 0, 0, 4, 0, 0, 1, 0, 2, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      129},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 2, 0, 0, 0, 0, 2, 1, 2, 2, 2,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      130},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 3, 0, 0, 0, 2, 0, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      131},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 3, 0, 2, 0, 0, 0, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      132},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 0, 2, 0, 1, 2, 0, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      133},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 3, 0, 0, 0, 0, 0, 2, 0, 0, 3, 0, 0, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      134},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 2, 0, 0, 2, 2, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      135},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 3, 0, 0, 2, 0, 2, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      136},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 1, 2, 2, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      137},
    {
      {0, 0, 2, 0, 0, 1, 0, 0, 1, 1, 0, 0, 0, 0, 0, 2, 0, 2, 1, 0, 2, 2,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      138},
    {
      {0, 2, 0, 2, 3, 1, 0, 0, 1, 1, 2, 0, 0, 0, 2, 4, 0, 3, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      191},
    {
      {0, 2, 0, 2, 3, 1, 0, 0, 1, 1, 2, 0, 0, 0, 2, 1, 3, 0, 3, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      192},
    {
      {0, 2, 0, 2, 3, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 1, 3, 3, 0, 0, 1, 3,
       0, 0, 0, 0, 0, 0, 0, 2, 0, 0},
      193},
    {
      {0, 2, 0, 2, 3, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 4, 0, 0, 3, 0, 1, 3,
       0, 0, 0, 0, 0, 0, 0, 2, 0, 0},
      194},
    {
      {0, 0, 0, 8, 0, 1, 0, 0, 1, 3, 8, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      200},
    {
      {0, 0, 0, 8, 0, 1, 0, 0, 1, 3, 8, 0, 0, 0, 0, 0, 0, 0, 3, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      201},
    {
      {0, 0, 0, 8, 0, 1, 0, 0, 1, 0, 8, 0, 0, 0, 0, 0, 0, 0, 3, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      205},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 8, 0, 6, 0, 0, 0, 0, 0, 0, 0, 0, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      207},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6,
       0, 0, 0, 6, 0, 0, 0, 0, 0, 0},
      208},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 6,
       0, 0, 0, 0, 6, 0, 0, 0, 0, 0},
      212},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 8, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 6,
       0, 0, 6, 0, 0, 0, 0, 0, 0, 0},
      213},
    {
      {0, 0, 6, 0, 0, 0, 0, 0, 1, 3, 8, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      215},
    {
      {0, 0, 6, 0, 0, 0, 0, 0, 1, 3, 8, 0, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      218},
    {
      {0, 0, 6, 8, 0, 1, 0, 0, 1, 3, 8, 0, 6, 0, 0, 9, 0, 0, 0, 0, 0, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      221},
    {
      {0, 0, 6, 8, 0, 1, 0, 0, 1, 3, 8, 0, 6, 0, 0, 0, 0, 0, 9, 0, 0, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      222},
    {
      {0, 0, 6, 8, 0, 1, 0, 0, 1, 3, 8, 0, 0, 0, 0, 3, 0, 0, 6, 0, 0, 6,
       0, 0, 0, 6, 0, 0, 0, 0, 0, 0},
      223},
    {
      {0, 0, 6, 8, 0, 1, 0, 0, 1, 3, 8, 0, 0, 0, 0, 6, 0, 0, 3, 0, 0, 6,
       0, 0, 0, 6, 0, 0, 0, 0, 0, 0},
      224},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      8},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      15},
    {
      {1, 0, 0, 0, 0, 0, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      23},
    {
      {2, 0, 0, 0, 0, 0, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      24},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 2, 0, 0, 2, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      44},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      45},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 2, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      46},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      79},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 4, 0, 0, 0, 0, 0, 0, 0},
      80},
    {
      {0, 0, 4, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      82},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 3, 0, 0, 0, 0, 0, 3, 0, 0, 3, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      71},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 3, 0, 0, 0, 0, 0, 1, 4, 0, 1, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      72},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 3, 0, 0, 0, 0, 0, 0, 6, 0, 0, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      73},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 3, 0, 0, 0, 0, 0, 2, 2, 0, 2, 0, 3, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      74},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 6, 0, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 4,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      97},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4,
       0, 0, 4, 0, 0, 0, 0, 0, 0, 0},
      98},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 2, 0, 0, 8, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      107},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 2, 0, 0, 4, 4, 0, 0, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      108},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 4, 0, 4, 0, 0, 0, 0,
       0, 0, 4, 0, 0, 0, 0, 0, 0, 0},
      109},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 0, 4, 4, 0, 0, 0, 0,
       0, 0, 4, 0, 0, 0, 0, 0, 0, 0},
      110},
    {
      {0, 0, 4, 0, 0, 2, 0, 1, 1, 2, 0, 0, 2, 0, 0, 2, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      87},
    {
      {0, 0, 4, 0, 0, 2, 0, 1, 1, 2, 0, 0, 0, 0, 0, 0, 2, 0, 0, 0, 0, 0,
       0, 0, 4, 0, 0, 0, 0, 0, 0, 0},
      88},
    {
      {0, 0, 4, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 4,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      119},
    {
      {0, 0, 4, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 4,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      120},
    {
      {0, 0, 4, 0, 0, 0, 0, 1, 1, 6, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      121},
    {
      {0, 0, 4, 0, 0, 0, 0, 1, 1, 6, 0, 0, 0, 0, 0, 0, 0, 4, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      122},
    {
      {1, 0, 0, 0, 0, 0, 0, 1, 1, 3, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      197},
    {
      {2, 0, 0, 0, 0, 0, 0, 1, 1, 3, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      199},
    {
      {0, 0, 4, 0, 0, 2, 0, 1, 1, 6, 0, 0, 2, 0, 0, 10, 0, 0, 0, 0, 0,
       4, 0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      139},
    {
      {0, 0, 4, 0, 0, 2, 0, 1, 1, 6, 0, 0, 2, 0, 0, 6, 4, 0, 0, 0, 0, 4,
       0, 0, 0, 2, 0, 0, 0, 0, 0, 0},
      140},
    {
      {0, 0, 4, 0, 0, 2, 0, 1, 1, 6, 0, 0, 0, 0, 0, 4, 2, 4, 0, 0, 0, 4,
       0, 0, 4, 0, 0, 0, 0, 0, 0, 0},
      141},
    {
      {0, 0, 4, 0, 0, 2, 0, 1, 1, 6, 0, 0, 0, 0, 0, 0, 6, 4, 0, 0, 0, 4,
       0, 0, 4, 0, 0, 0, 0, 0, 0, 0},
      142},
    {
      {0, 0, 0, 16, 0, 2, 0, 1, 1, 3, 16, 0, 0, 0, 0, 3, 0, 0, 3, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      204},
    {
      {0, 0, 0, 16, 0, 2, 0, 1, 1, 3, 16, 0, 0, 0, 0, 0, 6, 0, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      206},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 12, 16, 0, 6, 0, 0, 0, 0, 0, 0, 0, 6,
       0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0},
      211},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 12, 16, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6,
       0, 0, 0, 12, 0, 0, 0, 0, 0, 0, 0},
      214},
    {
      {0, 0, 12, 0, 0, 0, 0, 1, 1, 3, 16, 0, 0, 0, 0, 6, 3, 0, 3, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      217},
    {
      {0, 0, 12, 0, 0, 0, 0, 1, 1, 3, 16, 0, 0, 0, 0, 0, 0, 12, 0, 0, 3,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      220},
    {
      {0, 0, 12, 16, 0, 2, 0, 1, 1, 12, 16, 0, 6, 0, 0, 9, 3, 0, 6, 0,
       6, 0, 0, 0, 0, 6, 0, 0, 0, 0, 0, 0},
      229},
    {
      {0, 0, 12, 16, 0, 2, 0, 1, 1, 12, 16, 0, 0, 0, 0, 0, 6, 12, 0, 0,
       6, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0, 0},
      230},
    {
      {0, 0, 0, 0, 0, 0, 0, 3, 1, 6, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      22},
    {
      {0, 0, 0, 0, 0, 0, 0, 3, 1, 2, 0, 0, 0, 0, 0, 2, 4, 0, 2, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      42},
    {
      {0, 0, 0, 0, 0, 0, 0, 3, 1, 2, 0, 0, 0, 0, 0, 0, 0, 8, 0, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      43},
    {
      {0, 0, 0, 0, 0, 4, 0, 3, 1, 6, 0, 0, 0, 0, 0, 3, 6, 0, 3, 0, 6, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      69},
    {
      {0, 0, 0, 0, 0, 4, 0, 3, 1, 6, 0, 0, 0, 0, 0, 0, 0, 12, 0, 0, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      70},
    {
      {0, 0, 0, 0, 0, 0, 0, 3, 1, 6, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      196},
    {
      {0, 0, 0, 32, 0, 4, 0, 3, 1, 6, 32, 0, 0, 0, 0, 3, 6, 0, 3, 0, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      202},
    {
      {0, 0, 0, 32, 0, 4, 0, 3, 1, 6, 32, 0, 0, 0, 0, 0, 0, 12, 0, 0, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      203},
    {
      {0, 0, 0, 0, 0, 0, 0, 3, 1, 18, 32, 0, 12, 0, 0, 0, 0, 0, 0, 0,
       18, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0},
      209},
    {
      {0, 0, 0, 0, 0, 0, 0, 3, 1, 18, 32, 0, 0, 0, 0, 0, 0, 0, 0, 0, 18,
       0, 0, 0, 24, 0, 0, 0, 0, 0, 0, 0},
      210},
    {
      {0, 0, 24, 0, 0, 0, 0, 3, 1, 6, 32, 0, 0, 0, 0, 9, 0, 0, 0, 15, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      216},
    {
      {0, 0, 24, 0, 0, 0, 0, 3, 1, 6, 32, 0, 0, 0, 0, 0, 9, 0, 0, 15, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      219},
    {
      {0, 0, 24, 0, 0, 0, 0, 3, 1, 6, 32, 0, 0, 0, 0, 0, 9, 0, 3, 12, 6,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      219},
    {
      {0, 0, 24, 32, 0, 4, 0, 3, 1, 18, 32, 0, 12, 0, 0, 12, 6, 0, 3,
       15, 18, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0},
      225},
    {
      {0, 0, 24, 32, 0, 4, 0, 3, 1, 18, 32, 0, 12, 0, 0, 3, 15, 0, 6,
       12, 18, 0, 0, 0, 0, 12, 0, 0, 0, 0, 0, 0},
      226},
    {
      {0, 0, 24, 32, 0, 4, 0, 3, 1, 18, 32, 0, 0, 0, 0, 9, 0, 12, 0, 15,
       18, 0, 0, 0, 24, 0, 0, 0, 0, 0, 0, 0},
      227},
    {
      {0, 0, 24, 32, 0, 4, 0, 3, 1, 18, 32, 0, 0, 0, 0, 6, 0, 12, 0, 18,
       18, 0, 0, 0, 24, 0, 0, 0, 0, 0, 0, 0},
      227},
    {
      {0, 0, 24, 32, 0, 4, 0, 3, 1, 18, 32, 0, 0, 0, 0, 0, 6, 12, 6, 12,
       18, 0, 0, 0, 24, 0, 0, 0, 0, 0, 0, 0},
      228},
    {
      {0, 0, 24, 32, 0, 4, 0, 3, 1, 18, 32, 0, 0, 0, 0, 0, 9, 12, 3, 12,
       18, 0, 0, 0, 24, 0, 0, 0, 0, 0, 0, 0},
      228},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      5},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      8},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      9},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 2, 0, 1, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      36},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      12},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      15},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 2, 1, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      38},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 3, 0, 0, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      39},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 2, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      40},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 1, 0, 0, 0, 0, 0, 0, 3, 0, 1, 0, 1, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      41},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 4, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      20},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 2, 2, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      35},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 2, 0, 0, 0, 0, 0, 0, 2, 0, 2, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      37},
    {
      {0, 0, 0, 0, 0, 0, 0, 1, 1, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      21},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 2, 0, 0, 0, 0, 0, 2, 2, 0, 2, 0, 4, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      63},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 2, 0, 0, 0, 0, 0, 1, 4, 0, 1, 0, 4, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      64},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 4, 0, 0, 0, 0, 0, 3, 2, 0, 1, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      65},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 4, 0, 0, 0, 0, 0, 2, 4, 0, 0, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      67},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 4, 0, 0, 0, 0, 0, 0, 4, 0, 2, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      68},
    {
      {0, 0, 0, 0, 0, 2, 0, 1, 1, 4, 0, 0, 0, 0, 0, 1, 2, 0, 3, 0, 2, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      66},
    {
      {2, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      146},
    {
      {2, 0, 0, 2, 0, 1, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      148},
    {
      {0, 0, 0, 0, 0, 0, 0, 0, 1, 3, 2, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      155},
    {
      {2, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      160},
    {
      {2, 0, 0, 0, 0, 0, 0, 0, 1, 0, 2, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      161},
    {
      {0, 0, 0, 2, 0, 1, 0, 0, 1, 3, 2, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      166},
    {
      {0, 0, 0, 2, 0, 1, 0, 0, 1, 3, 2, 0, 0, 0, 0, 0, 3, 0, 0, 0, 0, 0,
       0, 0, 0, 0, 0, 0, 0, 0, 0, 0},
      167}
  };

  return spacegroup_data[num];
}
