/*
 * Interface to win32 APIs
 * 
 * Copyright (C) 2002-2018 Andy Adler <adler@ncf.ca>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; If not, see <http://www.gnu.org/licenses/>.
 *
 * $Id$
 */

#include <octave/oct.h>

#ifdef HAVE_CONFIG_H
#  include "config.h"
#endif

#ifdef USING_WINDOWS
#include <windows.h>

HKEY string_to_rootkey(const char *key)
{
  HKEY hprimkey = 0;

  if (0 == strcmp(key, "HKEY_CLASSES_ROOT") ||
      0 == strcmp(key, "HKCR"))
    {
      hprimkey= HKEY_CLASSES_ROOT;
    }
  else if (0== strcmp(key, "HKEY_CURRENT_USER") ||
           0== strcmp(key, "HKCU")) 
      {
        hprimkey= HKEY_CURRENT_USER;
      }
  else if (0== strcmp(key, "HKEY_LOCAL_MACHINE") ||
           0== strcmp(key, "HKLM")) 
      {
        hprimkey= HKEY_LOCAL_MACHINE;
      }
  else if ( 0== strcmp(key, "HKEY_USERS") ||
            0== strcmp(key, "HKU")) 
      {
        hprimkey= HKEY_USERS;
      }

  return hprimkey;
}

bool win32_IsValidRootKey (const char *key)
{
  return string_to_rootkey (key) != 0;
}

int
win32_MessageBox (const char * text,
                  const char * title,
                  int boxtype)
{
  return MessageBox (NULL, text, title, boxtype | MB_SETFOREGROUND );
}

int
win32_ScanRegistry (const char *key,
                    const char *subkey,
                    std::list<std::string> &fields)
{
  HKEY hprimkey, hsubkey;

  fields.clear ();

  hprimkey = string_to_rootkey (key);
  if (hprimkey == 0)
    {
      return -1; // We can't handle this key
    }

  int retval;

  retval= RegOpenKeyEx (hprimkey, subkey, 0, KEY_READ, &hsubkey);
  if (retval == NO_ERROR)
    {
      int count = 0;
      #define MAX_KEYNAME_SIZE 256
      DWORD keynamesize = MAX_KEYNAME_SIZE;
      char keyname[MAX_KEYNAME_SIZE+1];

      while ( (retval = RegEnumValue (hsubkey, count, keyname, 
                                      &keynamesize, NULL, NULL, NULL, NULL
            )) == ERROR_SUCCESS)
        {
          fields.push_back(keyname);
          keynamesize = MAX_KEYNAME_SIZE;
          count ++;
        }

      if (retval == ERROR_NO_MORE_ITEMS)
        retval = NO_ERROR;

      RegCloseKey (hsubkey);
    }

  return retval;
}

int
win32_ReadRegistry (const char *key,
                    const char *subkey,
                    const char *value,
                    char * buffer,
                    int  * buffer_sz)
{
  HKEY hprimkey, hsubkey;

  hprimkey = string_to_rootkey(key);

  if (hprimkey == 0)
    {
      return -1; // We can't handle this key
    }

  int retval;

  retval= RegOpenKeyEx (hprimkey, subkey, 0, KEY_READ, &hsubkey);
  if (retval == NO_ERROR)
    {
      DWORD dwBuffSz= *buffer_sz;
      retval= RegQueryValueEx (hsubkey, value, NULL, NULL, 
                               (BYTE *) buffer, & dwBuffSz);
      *buffer_sz = dwBuffSz;
    }

  RegCloseKey (hsubkey);
  return retval;
}
#endif
