function [h, g, a, info] = wfilt_db(N)
%-*- texinfo -*-
%@deftypefn {Function} wfilt_db
%@verbatim
%WFILT_DB    Daubechies FIR filterbank
%   Usage:  [h, g, a, info] = wfilt_db(N, compat = 1);
%
%   Input parameters:
%         N      : Order of Daubechies filters.
%         compat : 0 is precomputed LUT for N <= 38,
%                  1 is runtime calculation (default).
%   Output parameters:
%         H      : cell array of analysing filters impulse reponses
%         G      : cell array of synthetizing filters impulse reponses
%         a      : array of subsampling (or hop) factors accociated with
%                  corresponding filters
%         info   : info.istight=1 indicates the wavelet system is a can. tight frame
%
%   [H, G] = WFILT_DB(N) computes a two-channel Daubechies FIR filterbank
%   from prototype maximum-phase analysing lowpass filter obtained by
%   spectral factorization of the Lagrange interpolator filter.  N also
%   denotes the number of zeros at z = -1 of the lowpass filters of length
%   2N.  The prototype lowpass filter has the following form (all roots of
%   R(z) are inside of the unit circle):
%
%      H_l(z) = (1 + z^-1)^N * R(z),
%
%   where R(z) is a spectral factor of the Lagrange interpolator P(z) = 2 R(z)  R(z^{-1})
%   All subsequent filters of the two-channel filterbank are derived as
%   follows:
%
%      H_h(z) =  H_l((-z)^-1)
%      G_l(z) =  H_l(z^-1)
%      G_h(z) = -H_l(-z)
%
%   making them an orthogonal perfect-reconstruction QMF.
%
%   Examples:
%   ---------
%   :
%
%     wfiltinfo('db8');
%
%   References:
%     I. Daubechies. Ten Lectures on Wavelets. Society for Industrial and
%     Applied Mathematics, Philadelphia, PA, USA, 1992.
%     
%@end verbatim
%@strong{Url}: @url{http://ltfat.github.io/doc/wavelets/wfilt_db.html}
%@end deftypefn

% Copyright (C) 2005-2022 Peter L. Soendergaard <peter@sonderport.dk> and others.
% This file is part of LTFAT version 2.5.0
%
% This program is free software: you can redistribute it and/or modify
% it under the terms of the GNU General Public License as published by
% the Free Software Foundation, either version 3 of the License, or
% (at your option) any later version.
%
% This program is distributed in the hope that it will be useful,
% but WITHOUT ANY WARRANTY; without even the implied warranty of
% MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
% GNU General Public License for more details.
%
% You should have received a copy of the GNU General Public License
% along with this program.  If not, see <http://www.gnu.org/licenses/>.

% AUTHOR: KIMURA Masaru, Zdenek Prusa

if (nargin < 1);
    error('Too few input parameters.');
end;
if (N < 1);
    error('Minimum N is 1.');
end;
h = cell(2,1);

try
    h{1} = lut(N);
catch
    h{1} = cal(N);
end

h{2} = (-1).^[0:(2 * N - 1)] .* fliplr(h{1});
% The reverse is here, because we use different convention for
% filterbanks than in Ten Lectures on Wavelets
h{1} = fliplr(h{1});
h{2} = fliplr(h{2});
d = cellfun(@(hEl) - length(hEl) / 2, h);
if (N > 2);
    % Do a filter alignment according to "center of gravity"
    d(1) = -floor(sum([1:(2 * N)] .* abs(h{1}).^2) / sum(abs(h{1}).^2));
    d(2) = -floor(sum([1:(2 * N)] .* abs(h{2}).^2) / sum(abs(h{2}).^2));
    if (rem(d(1) - d(2), 2) == 1);
        % Shift d(2) just a bit
        d(2) = d(2) + 1;
    end;
end;
h{1} = struct('h', h{1}, 'offset', d(1));
h{2} = struct('h', h{2}, 'offset', d(2));
g = h; info.istight = 1;
a = [2;2];

function c = cal(N)
if (N > 20); % FIXME: N > 20 is too large for what?
    warning('Instability may occur when N is too large.');
end;
sup = [(-N + 1):N];
a = zeros(1, N);
for n = 1:N;
    non  = sup(sup ~= n);
    a(n) = prod(0.5 - non) / prod(n - non);
end;
p = zeros(1, 2 * N - 1);
p(1:2:end) = a;
p = [fliplr(p), 1, p];
r = roots(p);
r = r(real(r) > 0 & abs(r) < 1);
% FIXME: poly(r) produces errors due to sorting.
%        At least, consuming known multiple zeros at -1 first dramatically reduces mantissa error.
%r = [-ones(1, N), sort(r,  'ascend')(:)'];
rTemp = sort(r, 'descend');
r = [-ones(1, N), rTemp(:)'];
% r = [-ones(1, N), sort(r, 'descend')(:)']; % NOTE: It looks being suboptimal at N = 38, but keep being compat.
%r = [sort(r,  'ascend')(:)', -ones(1, N)];
%r = [sort(r, 'descend')(:)', -ones(1, N)];

% NOTE: Estimated bit accuracy from reconstruction identity; b2 is slightly analyzed.
%       N.B. b2 is not monotonically descrease WRT N.
%                        | r sorting                                   |
%                        +-----------+-----------+----------+----------+
%                        | -1 before | -1 before | -1 after | -1 after |
%           b2 at N = 38 | ascend    | descend   | ascend   | descend  |
%       -----------+-----+-----------+-----------+----------+----------+
%       use        | no  | 37.607    | 35.425    | 20.435   | 20.673   | ... c = real(poly(r));
%       conjugate  +-----+-----------+-----------+----------+----------+
%       identities | yes | 36.325    | 36.146    | 20.237   | 20.249   | ... c = conv(c, ...);
%       -----------+-----+-----------+-----------+----------+----------+
 c = real(poly(r));
%c = [1];
%r = r(imag(r) >= 0);
%for i = 1:length(r);
%	if (imag(r(i)) == 0);
%		c = conv(c, [1, -real(r(i))]);
%	else
%		c = conv(c, [1, -2 * real(r(i)), abs(r(i))^2]);
%	end;
%end;
 c = c/norm(c);

function c = lut(N)
if (N > 38);
    error('Maximum N is 38.');
end;
switch (N)
    case 1
        c = [
             7.07106781186547524e-01 ...
             7.07106781186547524e-01
        ];
    case 2
        c = [
             4.82962913144534143e-01 ...
             8.36516303737807906e-01 ...
             2.24143868042013381e-01 ...
            -1.29409522551260381e-01
        ];
    case 3
        c = [
             3.32670552950082616e-01 ...
             8.06891509311092576e-01 ...
             4.59877502118491570e-01 ...
            -1.35011020010254589e-01 ...
            -8.54412738820266617e-02 ...
             3.52262918857095366e-02
        ];
    case 4
        c = [
             2.30377813308896501e-01 ...
             7.14846570552915647e-01 ...
             6.30880767929858908e-01 ...
            -2.79837694168598542e-02 ...
            -1.87034811719093084e-01 ...
             3.08413818355607636e-02 ...
             3.28830116668851997e-02 ...
            -1.05974017850690321e-02
        ];
    case 5
        c = [
             1.60102397974192914e-01 ...
             6.03829269797189671e-01 ...
             7.24308528437772928e-01 ...
             1.38428145901320732e-01 ...
            -2.42294887066382032e-01 ...
            -3.22448695846383746e-02 ...
             7.75714938400457135e-02 ...
            -6.24149021279827427e-03 ...
            -1.25807519990819995e-02 ...
             3.33572528547377128e-03
        ];
    case 6
        c = [
             1.11540743350109464e-01 ...
             4.94623890398453086e-01 ...
             7.51133908021095351e-01 ...
             3.15250351709197629e-01 ...
            -2.26264693965439820e-01 ...
            -1.29766867567261936e-01 ...
             9.75016055873230491e-02 ...
             2.75228655303057286e-02 ...
            -3.15820393174860296e-02 ...
             5.53842201161496139e-04 ...
             4.77725751094551064e-03 ...
            -1.07730108530847956e-03
        ];
    case 7
        c = [
             7.78520540850091790e-02 ...
             3.96539319481917307e-01 ...
             7.29132090846235120e-01 ...
             4.69782287405193122e-01 ...
            -1.43906003928564975e-01 ...
            -2.24036184993874983e-01 ...
             7.13092192668302648e-02 ...
             8.06126091510830719e-02 ...
            -3.80299369350144136e-02 ...
            -1.65745416306668807e-02 ...
             1.25509985560998406e-02 ...
             4.29577972921366521e-04 ...
            -1.80164070404749092e-03 ...
             3.53713799974520248e-04
        ];
    case 8
        c = [
             5.44158422431040100e-02 ...
             3.12871590914299971e-01 ...
             6.75630736297289807e-01 ...
             5.85354683654206713e-01 ...
            -1.58291052563493057e-02 ...
            -2.84015542961546927e-01 ...
             4.72484573913282770e-04 ...
             1.28747426620478459e-01 ...
            -1.73693010018075462e-02 ...
            -4.40882539307947515e-02 ...
             1.39810279173982816e-02 ...
             8.74609404740577672e-03 ...
            -4.87035299345157431e-03 ...
            -3.91740373376947046e-04 ...
             6.75449406450569366e-04 ...
            -1.17476784124769534e-04
        ];
    case 9
        c = [
             3.80779473638783466e-02 ...
             2.43834674612590354e-01 ...
             6.04823123690111112e-01 ...
             6.57288078051300538e-01 ...
             1.33197385825007576e-01 ...
            -2.93273783279174909e-01 ...
            -9.68407832229764605e-02 ...
             1.48540749338106380e-01 ...
             3.07256814793333792e-02 ...
            -6.76328290613299737e-02 ...
             2.50947114831451958e-04 ...
             2.23616621236790972e-02 ...
            -4.72320475775139728e-03 ...
            -4.28150368246342983e-03 ...
             1.84764688305622648e-03 ...
             2.30385763523195967e-04 ...
            -2.51963188942710137e-04 ...
             3.93473203162715995e-05
        ];
    case 10
        c = [
             2.66700579005555536e-02 ...
             1.88176800077691489e-01 ...
             5.27201188931725586e-01 ...
             6.88459039453603566e-01 ...
             2.81172343660577461e-01 ...
            -2.49846424327315379e-01 ...
            -1.95946274377377044e-01 ...
             1.27369340335793260e-01 ...
             9.30573646035723512e-02 ...
            -7.13941471663970871e-02 ...
            -2.94575368218758129e-02 ...
             3.32126740593410017e-02 ...
             3.60655356695616966e-03 ...
            -1.07331754833305750e-02 ...
             1.39535174705290117e-03 ...
             1.99240529518505612e-03 ...
            -6.85856694959711627e-04 ...
            -1.16466855129285451e-04 ...
             9.35886703200695913e-05 ...
            -1.32642028945212448e-05
        ];
    case 11
        c = [
             1.86942977614710840e-02 ...
             1.44067021150624513e-01 ...
             4.49899764356045335e-01 ...
             6.85686774916200511e-01 ...
             4.11964368947907463e-01 ...
            -1.62275245027490362e-01 ...
            -2.74230846817946961e-01 ...
             6.60435881966831919e-02 ...
             1.49812012466378496e-01 ...
            -4.64799551166841873e-02 ...
            -6.64387856950252053e-02 ...
             3.13350902190460760e-02 ...
             2.08409043601810630e-02 ...
            -1.53648209062015994e-02 ...
            -3.34085887301444561e-03 ...
             4.92841765605904112e-03 ...
            -3.08592858815143165e-04 ...
            -8.93023250666264613e-04 ...
             2.49152523552823499e-04 ...
             5.44390746993684717e-05 ...
            -3.46349841869849955e-05 ...
             4.49427427723651010e-06
        ];
    case 12
        c = [
             1.31122579572295175e-02 ...
             1.09566272821185155e-01 ...
             3.77355135214212657e-01 ...
             6.57198722579307089e-01 ...
             5.15886478427815609e-01 ...
            -4.47638856537746267e-02 ...
            -3.16178453752785537e-01 ...
            -2.37792572560697277e-02 ...
             1.82478605927579680e-01 ...
             5.35956967435215033e-03 ...
            -9.64321200965070820e-02 ...
             1.08491302558221844e-02 ...
             4.15462774950844407e-02 ...
            -1.22186490697482807e-02 ...
            -1.28408251983006833e-02 ...
             6.71149900879550918e-03 ...
             2.24860724099523760e-03 ...
            -2.17950361862776047e-03 ...
             6.54512821250959557e-06 ...
             3.88653062820931444e-04 ...
            -8.85041092082043242e-05 ...
            -2.42415457570307840e-05 ...
             1.27769522193797666e-05 ...
            -1.52907175806851090e-06
        ];
    case 13
        c = [
             9.20213353896236797e-03 ...
             8.28612438729027796e-02 ...
             3.11996322160438063e-01 ...
             6.11055851158787653e-01 ...
             5.88889570431218908e-01 ...
             8.69857261796472373e-02 ...
            -3.14972907711388633e-01 ...
            -1.24576730750815259e-01 ...
             1.79476079429339843e-01 ...
             7.29489336567771638e-02 ...
            -1.05807618187934326e-01 ...
            -2.64884064753436946e-02 ...
             5.61394771002834289e-02 ...
             2.37997225405907881e-03 ...
            -2.38314207103236490e-02 ...
             3.92394144879741624e-03 ...
             7.25558940161756619e-03 ...
            -2.76191123465686218e-03 ...
            -1.31567391189229894e-03 ...
             9.32326130867263386e-04 ...
             4.92515251262894619e-05 ...
            -1.65128988556505489e-04 ...
             3.06785375793254935e-05 ...
             1.04419305714081371e-05 ...
            -4.70041647936086833e-06 ...
             5.22003509845486469e-07
        ];
    case 14
        c = [
             6.46115346008794782e-03 ...
             6.23647588493988983e-02 ...
             2.54850267792621354e-01 ...
             5.54305617940893836e-01 ...
             6.31187849104856780e-01 ...
             2.18670687758906521e-01 ...
            -2.71688552278748041e-01 ...
            -2.18033529993276045e-01 ...
             1.38395213864806591e-01 ...
             1.39989016584460701e-01 ...
            -8.67484115681696890e-02 ...
            -7.15489555040461307e-02 ...
             5.52371262592160441e-02 ...
             2.69814083079129170e-02 ...
            -3.01853515403906352e-02 ...
            -5.61504953035695913e-03 ...
             1.27894932663334090e-02 ...
            -7.46218989268384937e-04 ...
            -3.84963886802218745e-03 ...
             1.06169108560676184e-03 ...
             7.08021154235527859e-04 ...
            -3.86831947312954482e-04 ...
            -4.17772457703725974e-05 ...
             6.87550425269750960e-05 ...
            -1.03372091845707739e-05 ...
            -4.38970490178139412e-06 ...
             1.72499467536781277e-06 ...
            -1.78713996831135908e-07
        ];
    case 15
        c = [
             4.53853736157889888e-03 ...
             4.67433948927662719e-02 ...
             2.06023863986995732e-01 ...
             4.92631771708139624e-01 ...
             6.45813140357424358e-01 ...
             3.39002535454731528e-01 ...
            -1.93204139609145429e-01 ...
            -2.88882596566965646e-01 ...
             6.52829528487728169e-02 ...
             1.90146714007122982e-01 ...
            -3.96661765557909445e-02 ...
            -1.11120936037231693e-01 ...
             3.38771439235076862e-02 ...
             5.47805505845076127e-02 ...
            -2.57670073284399626e-02 ...
            -2.08100501696930817e-02 ...
             1.50839180278359024e-02 ...
             5.10100036040754317e-03 ...
            -6.48773456031574500e-03 ...
            -2.41756490761624281e-04 ...
             1.94332398038221154e-03 ...
            -3.73482354137616992e-04 ...
            -3.59565244362468812e-04 ...
             1.55896489920599748e-04 ...
             2.57926991553189368e-05 ...
            -2.81332962660478136e-05 ...
             3.36298718173757980e-06 ...
             1.81127040794057708e-06 ...
            -6.31688232588166442e-07 ...
             6.13335991330575203e-08
        ];
    case 16
        c = [
             3.18922092534773803e-03 ...
             3.49077143236733464e-02 ...
             1.65064283488853118e-01 ...
             4.30312722846003814e-01 ...
             6.37356332083788899e-01 ...
             4.40290256886356900e-01 ...
            -8.97510894024896429e-02 ...
            -3.27063310527917705e-01 ...
            -2.79182081330282767e-02 ...
             2.11190693947104289e-01 ...
             2.73402637527160414e-02 ...
            -1.32388305563810390e-01 ...
            -6.23972275247487177e-03 ...
             7.59242360442763158e-02 ...
            -7.58897436885773764e-03 ...
            -3.68883976917301423e-02 ...
             1.02976596409559694e-02 ...
             1.39937688598287310e-02 ...
            -6.99001456341391667e-03 ...
            -3.64427962149838993e-03 ...
             3.12802338120626883e-03 ...
             4.07896980849712836e-04 ...
            -9.41021749359567589e-04 ...
             1.14241520038722393e-04 ...
             1.74787245225338180e-04 ...
            -6.10359662141093584e-05 ...
            -1.39456689882088935e-05 ...
             1.13366086612762586e-05 ...
            -1.04357134231160650e-06 ...
            -7.36365678545120551e-07 ...
             2.30878408685754587e-07 ...
            -2.10933963010074310e-08
        ];
    case 17
        c = [
             2.24180700103731285e-03 ...
             2.59853937036060434e-02 ...
             1.31214903307824407e-01 ...
             3.70350724152641150e-01 ...
             6.10996615684622818e-01 ...
             5.18315764056937839e-01 ...
             2.73149704032936350e-02 ...
            -3.28320748363961736e-01 ...
            -1.26599752215882703e-01 ...
             1.97310589565010993e-01 ...
             1.01135489177470272e-01 ...
            -1.26815691778286311e-01 ...
            -5.70914196316769273e-02 ...
             8.11059866541608851e-02 ...
             2.23123361781037960e-02 ...
            -4.69224383892697373e-02 ...
            -3.27095553581929378e-03 ...
             2.27336765839462703e-02 ...
            -3.04298998135463707e-03 ...
            -8.60292152032285483e-03 ...
             2.96799669152609487e-03 ...
             2.30120524215354562e-03 ...
            -1.43684530480297613e-03 ...
            -3.28132519409837971e-04 ...
             4.39465427768643678e-04 ...
            -2.56101095665484588e-05 ...
            -8.20480320245339184e-05 ...
             2.31868137987459508e-05 ...
             6.99060098507675127e-06 ...
            -4.50594247722298819e-06 ...
             3.01654960999455742e-07 ...
             2.95770093331685675e-07 ...
            -8.42394844600268018e-08 ...
             7.26749296856160811e-09
        ];
    case 18
        c = [
             1.57631021844076043e-03 ...
             1.92885317241463771e-02 ...
             1.03588465822423596e-01 ...
             3.14678941337031699e-01 ...
             5.71826807766607223e-01 ...
             5.71801654888651335e-01 ...
             1.47223111969928142e-01 ...
            -2.93654040736558744e-01 ...
            -2.16480934005142971e-01 ...
             1.49533975565377789e-01 ...
             1.67081312763257405e-01 ...
            -9.23318841508462806e-02 ...
            -1.06752246659828486e-01 ...
             6.48872162119054428e-02 ...
             5.70512477385368841e-02 ...
            -4.45261419029823247e-02 ...
            -2.37332103958600010e-02 ...
             2.66707059264705903e-02 ...
             6.26216795430570749e-03 ...
            -1.30514809466120018e-02 ...
             1.18630033858117466e-04 ...
             4.94334360546673813e-03 ...
            -1.11873266699249707e-03 ...
            -1.34059629833610663e-03 ...
             6.28465682965145713e-04 ...
             2.13581561910340688e-04 ...
            -1.98648552311747949e-04 ...
            -1.53591712353472468e-07 ...
             3.74123788074003818e-05 ...
            -8.52060253744669520e-06 ...
            -3.33263447888582189e-06 ...
             1.76871298362761546e-06 ...
            -7.69163268988517615e-08 ...
            -1.17609876702823170e-07 ...
             3.06883586304517480e-08 ...
            -2.50793445494859827e-09
        ];
    case 19
        c = [
             1.10866976318171057e-03 ...
             1.42810984507643974e-02 ...
             8.12781132654595507e-02 ...
             2.64388431740896785e-01 ...
             5.24436377464654915e-01 ...
             6.01704549127537895e-01 ...
             2.60894952651038829e-01 ...
            -2.28091394215482646e-01 ...
            -2.85838631755826242e-01 ...
             7.46522697081032664e-02 ...
             2.12349743306278489e-01 ...
            -3.35185419023028787e-02 ...
            -1.42785695038736575e-01 ...
             2.75843506256286688e-02 ...
             8.69067555558122325e-02 ...
            -2.65012362501230409e-02 ...
            -4.56742262772309081e-02 ...
             2.16237674095850471e-02 ...
             1.93755498891761276e-02 ...
            -1.39883886785351416e-02 ...
            -5.86692228101217473e-03 ...
             7.04074736710524315e-03 ...
             7.68954359257548356e-04 ...
            -2.68755180070158200e-03 ...
             3.41808653458595777e-04 ...
             7.35802520505435207e-04 ...
            -2.60676135678628006e-04 ...
            -1.24600791734158775e-04 ...
             8.71127046721992297e-05 ...
             5.10595048707388605e-06 ...
            -1.66401762971549445e-05 ...
             3.01096431629652634e-06 ...
             1.53193147669119306e-06 ...
            -6.86275565776914270e-07 ...
             1.44708829879784454e-08 ...
             4.63693777578260422e-08 ...
            -1.11640206703582582e-08 ...
             8.66684883899761935e-10
        ];
    case 20
        c = [
             7.79953613666846322e-04 ...
             1.05493946249503983e-02 ...
             6.34237804590815150e-02 ...
             2.19942113551397045e-01 ...
             4.72696185310901696e-01 ...
             6.10493238938593820e-01 ...
             3.61502298739331063e-01 ...
            -1.39212088011483873e-01 ...
            -3.26786800434034967e-01 ...
            -1.67270883090770076e-02 ...
             2.28291050819916323e-01 ...
             3.98502464577712022e-02 ...
            -1.55458750707267956e-01 ...
            -2.47168273386135840e-02 ...
             1.02291719174442558e-01 ...
             5.63224685730743551e-03 ...
            -6.17228996246804597e-02 ...
             5.87468181181182649e-03 ...
             3.22942995307695818e-02 ...
            -8.78932492390156135e-03 ...
            -1.38105261371519201e-02 ...
             6.72162730225945684e-03 ...
             4.42054238704579096e-03 ...
            -3.58149425960962278e-03 ...
            -8.31562172822556919e-04 ...
             1.39255961932313632e-03 ...
            -5.34975984399769505e-05 ...
            -3.85104748699217606e-04 ...
             1.01532889736702905e-04 ...
             6.77428082837772956e-05 ...
            -3.71058618339471286e-05 ...
            -4.37614386218399681e-06 ...
             7.24124828767362010e-06 ...
            -1.01199401001888615e-06 ...
            -6.84707959700055689e-07 ...
             2.63392422627000108e-07 ...
             2.01432202355051269e-10 ...
            -1.81484324829969597e-08 ...
             4.05612705555183277e-09 ...
            -2.99883648961931957e-10
        ];
    case 21
        c = [
             5.48822509852683709e-04 ...
             7.77663905235478375e-03 ...
             4.92477715381772749e-02 ...
             1.81359625440381516e-01 ...
             4.19687944939362773e-01 ...
             6.01506094935003898e-01 ...
             4.44590451927600340e-01 ...
            -3.57229196172552905e-02 ...
            -3.35664089530529509e-01 ...
            -1.12397071568450981e-01 ...
             2.11564527680872392e-01 ...
             1.15233298439687104e-01 ...
            -1.39940424932547225e-01 ...
            -8.17759429808638289e-02 ...
             9.66003903237242207e-02 ...
             4.57234057492287924e-02 ...
            -6.49775048937323206e-02 ...
            -1.86538592021185153e-02 ...
             3.97268354278504418e-02 ...
             3.35775639033811084e-03 ...
            -2.08920536779790795e-02 ...
             2.40347092080543476e-03 ...
             8.98882438197191188e-03 ...
            -2.89133434858890125e-03 ...
            -2.95837403893283128e-03 ...
             1.71660704063062414e-03 ...
             6.39418500512030215e-04 ...
            -6.90671117082101651e-04 ...
            -3.19640627768043719e-05 ...
             1.93664650416508062e-04 ...
            -3.63552025008633831e-05 ...
            -3.49966598498744795e-05 ...
             1.53548250927604928e-05 ...
             2.79033053981448705e-06 ...
            -3.09001716454569920e-06 ...
             3.16609544236703056e-07 ...
             2.99213663046485279e-07 ...
            -1.00040087903059733e-07 ...
            -2.25401497467333013e-09 ...
             7.05803354123112186e-09 ...
            -1.47195419765036527e-09 ...
             1.03880557102370655e-10
        ];
    case 22
        c = [
             3.86263231491098216e-04 ...
             5.72185463133453912e-03 ...
             3.80699372364110849e-02 ...
             1.48367540890111429e-01 ...
             3.67728683446037479e-01 ...
             5.78432731009524427e-01 ...
             5.07901090622163902e-01 ...
             7.37245011836301517e-02 ...
            -3.12726580428296192e-01 ...
            -2.00568406104887094e-01 ...
             1.64093188106766482e-01 ...
             1.79973187992891304e-01 ...
            -9.71107984091147097e-02 ...
            -1.31768137686683411e-01 ...
             6.80763143927322156e-02 ...
             8.45573763668260750e-02 ...
            -5.13642542974441325e-02 ...
            -4.65308118275067135e-02 ...
             3.69708466206980206e-02 ...
             2.05867076275653604e-02 ...
            -2.34800013444931887e-02 ...
            -6.21378284936465850e-03 ...
             1.25647252183433741e-02 ...
             3.00137398507643595e-04 ...
            -5.45569198615671708e-03 ...
             1.04426073918602532e-03 ...
             1.82701049565727908e-03 ...
            -7.70690988123119623e-04 ...
            -4.23787399839180080e-04 ...
             3.28609414213678734e-04 ...
             4.34589990453200338e-05 ...
            -9.40522363481576042e-05 ...
             1.13743496621259317e-05 ...
             1.73737569575618936e-05 ...
            -6.16672931646757837e-06 ...
            -1.56517913199516016e-06 ...
             1.29518205731887757e-06 ...
            -8.77987987336128628e-08 ...
            -1.28333622875175442e-07 ...
             3.76122874933736237e-08 ...
             1.68017140492298889e-09 ...
            -2.72962314663297608e-09 ...
             5.33593882166748991e-10 ...
            -3.60211348433955470e-11
        ];
    case 23
        c = [
             2.71904194128288841e-04 ...
             4.20274889318383354e-03 ...
             2.93100036578841151e-02 ...
             1.20515531783971934e-01 ...
             3.18450813852865236e-01 ...
             5.44931147873520428e-01 ...
             5.51018517241919391e-01 ...
             1.81392625363840014e-01 ...
            -2.61392148030644112e-01 ...
            -2.71402098607843056e-01 ...
             9.21254070824180526e-02 ...
             2.23573658242040232e-01 ...
            -3.30374470942893788e-02 ...
            -1.64011321531875925e-01 ...
             2.02830745756492997e-02 ...
             1.12297043618107289e-01 ...
            -2.11262123562272410e-02 ...
            -7.02073915749011095e-02 ...
             2.17658568344999756e-02 ...
             3.84953325225691990e-02 ...
            -1.85235136501561598e-02 ...
            -1.75371010030358454e-02 ...
             1.27519439315282865e-02 ...
             6.03184065002416282e-03 ...
            -7.07531927370615281e-03 ...
            -1.13486547335625169e-03 ...
             3.12287644981814500e-03 ...
            -2.46501400516351203e-04 ...
            -1.06123122888665132e-03 ...
             3.19420492709901150e-04 ...
             2.56762452007873721e-04 ...
            -1.50021850349034097e-04 ...
            -3.37889483412090343e-05 ...
             4.42607120310924608e-05 ...
            -2.63520788924918624e-06 ...
            -8.34787556785462554e-06 ...
             2.39756954684024006e-06 ...
             8.14757483477944778e-07 ...
            -5.33900540520942115e-07 ...
             1.85309178563396502e-08 ...
             5.41754917953927874e-08 ...
            -1.39993549543799885e-08 ...
            -9.47288590181205054e-10 ...
             1.05044645369654340e-09 ...
            -1.93240511131341754e-10 ...
             1.25020330235104094e-11
        ];
    case 24
        c = [
             1.91435800947551370e-04 ...
             3.08208171490549444e-03 ...
             2.24823399497164107e-02 ...
             9.72622358336251966e-02 ...
             2.72908916067726327e-01 ...
             5.04371040839924992e-01 ...
             5.74939221095541997e-01 ...
             2.80985553233711883e-01 ...
            -1.87271406885156238e-01 ...
            -3.17943078999362738e-01 ...
             4.77661368434472819e-03 ...
             2.39237388780310855e-01 ...
             4.25287296414838326e-02 ...
            -1.71175351370346890e-01 ...
            -3.87771735779200162e-02 ...
             1.21016303469224236e-01 ...
             2.09801137091448153e-02 ...
            -8.21616542080016670e-02 ...
            -4.57843624181922164e-03 ...
             5.13016200399808792e-02 ...
            -4.94470942812562830e-03 ...
            -2.82131070949018910e-02 ...
             7.66172188164658590e-03 ...
             1.30499708710857358e-02 ...
            -6.29143537001818778e-03 ...
            -4.74656878632311380e-03 ...
             3.73604617828252335e-03 ...
             1.15376493683948150e-03 ...
            -1.69645681897482439e-03 ...
            -4.41618485614152006e-05 ...
             5.86127059318310993e-04 ...
            -1.18123323796955474e-04 ...
            -1.46007981776261684e-04 ...
             6.55938863930563409e-05 ...
             2.18324146046655836e-05 ...
            -2.02288829261269768e-05 ...
             1.34115775080911472e-08 ...
             3.90110033859770261e-06 ...
            -8.98025314393840772e-07 ...
            -4.03250775687997162e-07 ...
             2.16633965327857464e-07 ...
            -5.05764541979250031e-10 ...
            -2.25574038817608611e-08 ...
             5.15777678967199964e-09 ...
             4.74837582425623112e-10 ...
            -4.02465864458437977e-10 ...
             6.99180115763823097e-11 ...
            -4.34278250380371025e-12
        ];
    case 25
        c = [
             1.34802979347018899e-04 ...
             2.25695959185477952e-03 ...
             1.71867412540401553e-02 ...
             7.80358628721326756e-02 ...
             2.31693507886021820e-01 ...
             4.59683415146094594e-01 ...
             5.81636896746057783e-01 ...
             3.67885074802946698e-01 ...
            -9.71746409646381428e-02 ...
            -3.36473079641746131e-01 ...
            -8.75876145876546614e-02 ...
             2.24537819745101713e-01 ...
             1.18155286719959860e-01 ...
            -1.50560213750579631e-01 ...
            -9.85086152899602215e-02 ...
             1.06633805018477953e-01 ...
             6.67521644940186067e-02 ...
            -7.70841110565741936e-02 ...
            -3.71739628611225089e-02 ...
             5.36179093987794996e-02 ...
             1.55426059291022916e-02 ...
            -3.40423204606533410e-02 ...
            -3.07983679484703666e-03 ...
             1.89228044766276284e-02 ...
            -1.98942578220273649e-03 ...
            -8.86070261804636840e-03 ...
             2.72693625873849574e-03 ...
             3.32270777397319178e-03 ...
            -1.84248429020333128e-03 ...
            -8.99977423746295049e-04 ...
             8.77258193674827484e-04 ...
             1.15321244046630046e-04 ...
            -3.09880099098469799e-04 ...
             3.54371452327605901e-05 ...
             7.90464000396552826e-05 ...
            -2.73304811996004175e-05 ...
            -1.27719529319978380e-05 ...
             8.99066139306258891e-06 ...
             5.23282770815307642e-07 ...
            -1.77920133265363456e-06 ...
             3.21203751886251909e-07 ...
             1.92280679014237160e-07 ...
            -8.65694173227850716e-08 ...
            -2.61159855611177086e-09 ...
             9.27922448008137237e-09 ...
            -1.88041575506215554e-09 ...
            -2.22847491022816890e-10 ...
             1.53590157016265720e-10 ...
            -2.52762516346564481e-11 ...
             1.50969208282391087e-12
        ];
    case 26
        c = [
             9.49379575071059212e-05 ...
             1.65052023353298825e-03 ...
             1.30975542925585008e-02 ...
             6.22747440251496048e-02 ...
             1.95039438716770099e-01 ...
             4.13292962278356369e-01 ...
             5.73669043034222260e-01 ...
             4.39158311789166232e-01 ...
             1.77407678098668573e-03 ...
            -3.26384593691780022e-01 ...
            -1.74839961289392504e-01 ...
             1.81291832311122696e-01 ...
             1.82755409589672375e-01 ...
            -1.04323900285927044e-01 ...
            -1.47977193275254494e-01 ...
             6.98231861132923651e-02 ...
             1.06482405249808630e-01 ...
            -5.34485616814831915e-02 ...
            -6.86547596040359153e-02 ...
             4.22321857963720354e-02 ...
             3.85357159711118643e-02 ...
            -3.13781103630677548e-02 ...
            -1.77609035683581835e-02 ...
             2.07349201799638248e-02 ...
             5.82958055531888797e-03 ...
            -1.17854979061930289e-02 ...
            -5.28738399262681444e-04 ...
             5.60194723942380485e-03 ...
            -9.39058250473828965e-04 ...
            -2.14553028156762098e-03 ...
             8.38348805654361605e-04 ...
             6.16138220457434419e-04 ...
            -4.31955707426180747e-04 ...
            -1.06057474828380389e-04 ...
             1.57479523860749359e-04 ...
            -5.27779549303786898e-06 ...
            -4.10967399639147782e-05 ...
             1.07422154087219503e-05 ...
             7.00007868296498673e-06 ...
            -3.88740016185679519e-06 ...
            -4.65046322064026264e-07 ...
             7.93921063370995209e-07 ...
            -1.07900423757867141e-07 ...
            -8.90446637016859077e-08 ...
             3.40779562129073001e-08 ...
             2.16932825985032311e-09 ...
            -3.77601047853232433e-09 ...
             6.78004724582863667e-10 ...
             1.00230319104652691e-10 ...
            -5.84040818534117147e-11 ...
             9.13051001637179624e-12 ...
            -5.25187122424443504e-13
        ];
    case 27
        c = [
             6.68713138543193173e-05 ...
             1.20553123167321323e-03 ...
             9.95258878087661977e-03 ...
             4.94525999829048800e-02 ...
             1.62922027502393321e-01 ...
             3.67110214125389823e-01 ...
             5.53849860990480049e-01 ...
             4.93406122677998998e-01 ...
             1.02840855061822911e-01 ...
            -2.89716803314594846e-01 ...
            -2.48264581903260567e-01 ...
             1.14823019517785358e-01 ...
             2.27273288414170827e-01 ...
            -3.87864186318023106e-02 ...
            -1.78031740959008582e-01 ...
             1.57993974602404843e-02 ...
             1.31197971717155329e-01 ...
            -1.40627515558087654e-02 ...
            -9.10229065295659180e-02 ...
             1.73110182654937109e-02 ...
             5.79694057347179881e-02 ...
            -1.85124935619980771e-02 ...
            -3.27390666310208715e-02 ...
             1.61469669223956668e-02 ...
             1.56655956489245787e-02 ...
            -1.15771864589762814e-02 ...
            -5.86209634546292597e-03 ...
             6.85663560968488068e-03 ...
             1.34262687730367961e-03 ...
            -3.33285446952000616e-03 ...
             1.45752962593172859e-04 ...
             1.30117745024413514e-03 ...
            -3.41835122691542761e-04 ...
            -3.87901857410132760e-04 ...
             2.01971987969032686e-04 ...
             7.66005838706857688e-05 ...
            -7.71114551779758421e-05 ...
            -3.51748361490744539e-06 ...
             2.06344264773688532e-05 ...
            -3.90116407063842553e-06 ...
            -3.65750090818710500e-06 ...
             1.63436962472563784e-06 ...
             3.05088068625199909e-07 ...
            -3.47246814739438927e-07 ...
             3.28655896805515953e-08 ...
             4.02625505286690864e-08 ...
            -1.32133227399005656e-08 ...
            -1.30946560685695515e-09 ...
             1.52161498477852174e-09 ...
            -2.41552692801113066e-10 ...
            -4.37498622429365440e-11 ...
             2.21366208806766249e-11 ...
            -3.29579012247658581e-12 ...
             1.82818835288242493e-13
        ];
    case 28
        c = [
             4.71080777501405110e-05 ...
             8.79498515984387027e-04 ...
             7.54265037764685918e-03 ...
             3.90926081154053443e-02 ...
             1.35137914253641045e-01 ...
             3.22563361285522426e-01 ...
             5.24998231630335556e-01 ...
             5.30516293441485808e-01 ...
             2.00176144045984438e-01 ...
            -2.30498954047582526e-01 ...
            -3.01327809532641782e-01 ...
             3.28578791633871047e-02 ...
             2.45808151373759554e-01 ...
             3.69068853157112721e-02 ...
            -1.82877330732984917e-01 ...
            -4.68382337445516762e-02 ...
             1.34627567910226088e-01 ...
             3.44786312750997052e-02 ...
            -9.76853558056524417e-02 ...
            -1.73419228313058991e-02 ...
             6.77478955019093396e-02 ...
             3.44801895554095114e-03 ...
            -4.33333686160862839e-02 ...
             4.43173291006298832e-03 ...
             2.46880600101518659e-02 ...
            -6.81554976455230964e-03 ...
            -1.20635919682184901e-02 ...
             5.83881662774894486e-03 ...
             4.78486311245424172e-03 ...
            -3.72546124707425480e-03 ...
            -1.36037384563969244e-03 ...
             1.87599866820279563e-03 ...
             1.41567239314046426e-04 ...
            -7.48674955911462999e-04 ...
             1.15465606365892125e-04 ...
             2.29579098223345620e-04 ...
            -8.90390149004448810e-05 ...
            -4.90771341619025086e-05 ...
             3.64140121105080278e-05 ...
             4.63866498139429465e-06 ...
            -1.00432604133342260e-05 ...
             1.24790031757483415e-06 ...
             1.84036373451776919e-06 ...
            -6.67021547995489259e-07 ...
            -1.75746117320984278e-07 ...
             1.49066001353536217e-07 ...
            -8.26238731562655697e-09 ...
            -1.78413869087571008e-08 ...
             5.04404705638343644e-09 ...
             6.94454032894622695e-10 ...
            -6.07704124722901022e-10 ...
             8.49222001105638211e-11 ...
             1.86736726378339042e-11 ...
            -8.36549047125880080e-12 ...
             1.18885053340590152e-12 ...
            -6.36777235471485734e-14
        ];
    case 29
        c = [
             3.31896627984152476e-05 ...
             6.40951680304443454e-04 ...
             5.70212651777337543e-03 ...
             3.07735802214083768e-02 ...
             1.11370116951740530e-01 ...
             2.80653455970982938e-01 ...
             4.89758804762199314e-01 ...
             5.51374432758375195e-01 ...
             2.89105238335829163e-01 ...
            -1.54028734459900054e-01 ...
            -3.30040948917588052e-01 ...
            -5.57068000729408578e-02 ...
             2.36105236153025942e-01 ...
             1.12419174873188376e-01 ...
            -1.60877988594187736e-01 ...
            -1.07845949938721420e-01 ...
             1.14472295893818258e-01 ...
             8.32207471624497579e-02 ...
            -8.51254926156355023e-02 ...
            -5.50274895253257232e-02 ...
             6.34791645842118663e-02 ...
             3.05315432727041365e-02 ...
            -4.51879812777883452e-02 ...
            -1.29171425542667946e-02 ...
             2.94704318717476411e-02 ...
             2.64832730767816792e-03 ...
            -1.70412245736066897e-02 ...
             1.73788033272051116e-03 ...
             8.46972549356075229e-03 ...
            -2.55080712778947266e-03 ...
            -3.47379898968110063e-03 ...
             1.87712092572365013e-03 ...
             1.08705394222606297e-03 ...
            -1.00077832708568054e-03 ...
            -2.00071136307677981e-04 ...
             4.11128345474276703e-04 ...
            -2.29201804121449990e-05 ...
            -1.29304484008072061e-04 ...
             3.64502606856277497e-05 ...
             2.91334475016904122e-05 ...
            -1.65732839530661629e-05 ...
            -3.59364480402518764e-06 ...
             4.75060924645255285e-06 ...
            -3.02905459205281829e-07 ...
            -8.97570175063628073e-07 ...
             2.63389838699769655e-07 ...
             9.38719741109586303e-08 ...
            -6.28615692201078617e-08 ...
             1.07659190661919614e-09 ...
             7.76897885477006224e-09 ...
            -1.89399538617198415e-09 ...
            -3.42680086326308900e-10 ...
             2.40709945350934296e-10 ...
            -2.94058925076453258e-11 ...
            -7.83250973362781703e-12 ...
             3.15276241337031042e-12 ...
            -4.28565487006834410e-13 ...
             2.21919131158830296e-14
        ];
    case 30
        c = [
             2.33861617273142147e-05 ...
             4.66637950428550934e-04 ...
             4.30079716504806951e-03 ...
             2.41308326715883790e-02 ...
             9.12383040670157068e-02 ...
             2.42020670940214099e-01 ...
             4.50487821853317837e-01 ...
             5.57572232912836430e-01 ...
             3.66242683371627979e-01 ...
            -6.61836707759373150e-02 ...
            -3.32966975020855607e-01 ...
            -1.41968513330082931e-01 ...
             1.99462121580664303e-01 ...
             1.77829873244836736e-01 ...
            -1.14558219432707781e-01 ...
            -1.57236817959993813e-01 ...
             7.27786589703644270e-02 ...
             1.22747746045009378e-01 ...
            -5.38064654582570768e-02 ...
            -8.76586900363836605e-02 ...
             4.38016646714177325e-02 ...
             5.67123657447356949e-02 ...
            -3.56733974967596097e-02 ...
            -3.22637589193522082e-02 ...
             2.70786195952941827e-02 ...
             1.52879607698573955e-02 ...
            -1.83997438681173412e-02 ...
            -5.29685966613108663e-03 ...
             1.09156316583048893e-02 ...
             6.19671756497724438e-04 ...
            -5.53073014819200329e-03 ...
             8.43384586662093398e-04 ...
             2.32452009406009930e-03 ...
            -8.60927696811042388e-04 ...
            -7.67878250438091870e-04 ...
             5.05094823903346780e-04 ...
             1.72482584235170973e-04 ...
            -2.16171830116963380e-04 ...
            -8.54830546758407099e-06 ...
             6.98200837080832785e-05 ...
            -1.33971686329397163e-05 ...
            -1.63615247872542649e-05 ...
             7.25214553589046902e-06 ...
             2.32754909849368651e-06 ...
            -2.18726767699616642e-06 ...
             1.09947433852620330e-08 ...
             4.26166232601157245e-07 ...
            -1.00041468235450090e-07 ...
            -4.76437996513945336e-08 ...
             2.60544275497762543e-08 ...
             5.55339786139705398e-10 ...
            -3.33110568046757825e-09 ...
             6.98486269183218258e-10 ...
             1.61362297827090436e-10 ...
            -9.46138799727680212e-11 ...
             1.00010513139317119e-11 ...
             3.23942863853228611e-12 ...
            -1.18523759210158233e-12 ...
             1.54399757084762005e-13 ...
            -7.73794263095440571e-15
        ];
    case 31
        c = [
             1.64801338645614075e-05 ...
             3.39412203776995670e-04 ...
             3.23688406862772122e-03 ...
             1.88536916129859127e-02 ...
             7.43360930116478870e-02 ...
             2.07012874485235329e-01 ...
             4.09192200037427856e-01 ...
             5.51139840914275498e-01 ...
             4.29468808206137296e-01 ...
             2.71692124973694642e-02 ...
            -3.10955118319507519e-01 ...
            -2.17978485523563352e-01 ...
             1.40178288765273268e-01 ...
             2.24966711473737093e-01 ...
            -4.99263491604682398e-02 ...
            -1.86962360895715449e-01 ...
             1.54369884294889341e-02 ...
             1.45089500931993198e-01 ...
            -8.13983227346923686e-03 ...
            -1.07612773323495633e-01 ...
             1.09412974523649693e-02 ...
             7.53536117432814070e-02 ...
            -1.48800266181048220e-02 ...
            -4.86190754648543300e-02 ...
             1.61541715659859111e-02 ...
             2.80476193667561691e-02 ...
            -1.42762752777635194e-02 ...
            -1.39005529392665288e-02 ...
             1.05176394873718409e-02 ...
             5.51616357331099257e-03 ...
            -6.52085237587461255e-03 ...
            -1.42826422321890989e-03 ...
             3.39306677671593193e-03 ...
            -6.39790110601460049e-05 ...
            -1.45904174198516094e-03 ...
             3.43139829690473444e-04 ...
             4.99881617563722261e-04 ...
            -2.39658346940294962e-04 ...
            -1.24341161725022867e-04 ...
             1.08958435041676688e-04 ...
             1.50133572744453300e-05 ...
            -3.63125515786008616e-05 ...
             4.03452023518427884e-06 ...
             8.79530134269298777e-06 ...
            -3.03514236589150963e-06 ...
            -1.36906023094294078e-06 ...
             9.81001542204437157e-07 ...
             5.32725065697491543e-08 ...
            -1.97592512917020625e-07 ...
             3.61682651733100481e-08 ...
             2.32830971382140964e-08 ...
            -1.06152960215025231e-08 ...
            -6.47431168795986140e-10 ...
             1.40856815102517743e-09 ...
            -2.52404395415335331e-10 ...
            -7.34893003248626390e-11 ...
             3.69210880887112941e-11 ...
            -3.32700896712597993e-12 ...
            -1.32433491724396316e-12 ...
             4.44546709629193216e-13 ...
            -5.55944205057901434e-14 ...
             2.69938287976266565e-15
        ];
    case 32
        c = [
             1.16146330213501489e-05 ...
             2.46656690638090335e-04 ...
             2.43126191957226610e-03 ...
             1.46810463814191356e-02 ...
             6.02574991203353708e-02 ...
             1.75750783639438899e-01 ...
             3.67509628597349636e-01 ...
             5.34317919340953832e-01 ...
             4.77809163733948403e-01 ...
             1.20630538265617827e-01 ...
            -2.66698181476675554e-01 ...
            -2.77421581558427215e-01 ...
             6.47133548055162383e-02 ...
             2.48310642356880174e-01 ...
             2.46624448396974044e-02 ...
            -1.92102344708546898e-01 ...
            -4.89951171846717385e-02 ...
             1.45232079475286646e-01 ...
             4.44049081999397402e-02 ...
            -1.09456113116089383e-01 ...
            -2.96278725084477049e-02 ...
             8.08741406384839574e-02 ...
             1.41061515161066077e-02 ...
            -5.69263140624784355e-02 ...
            -2.38026446493257383e-03 ...
             3.70514579235446801e-02 ...
            -4.14590766082721878e-03 ...
            -2.16628228363911935e-02 ...
             6.16752731068567511e-03 ...
             1.10174007154068812e-02 ...
            -5.41156825727579121e-03 ...
            -4.64921675118441153e-03 ...
             3.62722464068786496e-03 ...
             1.46895510046846777e-03 ...
            -1.96474055582177825e-03 ...
            -2.21167872957909792e-04 ...
             8.67305851845055534e-04 ...
            -1.02453731060739619e-04 ...
            -3.05965442382691175e-04 ...
             1.05391546173982811e-04 ...
             8.10367832913483839e-05 ...
            -5.25980928268432278e-05 ...
            -1.29404577940551272e-05 ...
             1.82426840198069122e-05 ...
            -6.36178153226025495e-07 ...
            -4.55830957626442314e-06 ...
             1.20288903632162099e-06 ...
             7.56004762559594782e-07 ...
            -4.28597069315145726e-07 ...
            -5.00336186874823029e-08 ...
             8.96596631195772838e-08 ...
            -1.21992435948337309e-08 ...
            -1.10438302172264898e-08 ...
             4.25042231198059298e-09 ...
             4.38438779994047437e-10 ...
            -5.88109146263460563e-10 ...
             8.90472379622160549e-11 ...
             3.26327074133290788e-11 ...
            -1.43091876516920232e-11 ...
             1.07561065350106212e-12 ...
             5.36148222961180164e-13 ...
            -1.66380048943340237e-13 ...
             2.00071530381052495e-14 ...
            -9.42101913953507842e-16
        ];
    case 33
        c = [
             8.18635831417509194e-06 ...
             1.79101615370279148e-04 ...
             1.82270943516408421e-03 ...
             1.13959433745816093e-02 ...
             4.86146665317161951e-02 ...
             1.48186313180052808e-01 ...
             3.26718130117707578e-01 ...
             5.09376172514939655e-01 ...
             5.11254770583267466e-01 ...
             2.09582350713055422e-01 ...
            -2.04202622398542105e-01 ...
            -3.15997410766560256e-01 ...
            -1.92783394369527592e-02 ...
             2.45420612119279111e-01 ...
             9.98515586803381570e-02 ...
            -1.71428099051859328e-01 ...
            -1.10844133116710791e-01 ...
             1.21967856403734615e-01 ...
             9.47880880506159589e-02 ...
            -9.11469683513314891e-02 ...
            -7.03024850540561592e-02 ...
             7.01911439409965325e-02 ...
             4.57345618938966774e-02 ...
            -5.34712513358222892e-02 ...
            -2.52485829774764993e-02 ...
             3.86870607602449648e-02 ...
             1.07032658200195494e-02 ...
            -2.57287617547329734e-02 ...
            -2.16775861735360732e-03 ...
             1.53169541158576655e-02 ...
            -1.59428878241460477e-03 ...
            -7.95354038705793924e-03 ...
             2.38906240816590858e-03 ...
             3.48080095340571200e-03 ...
            -1.86071821445579591e-03 ...
            -1.20430925760465888e-03 ...
             1.07438069635129136e-03 ...
             2.72730584733693721e-04 ...
            -4.90832900759035147e-04 ...
             4.39316625176618576e-06 ...
             1.78043189825124535e-04 ...
            -4.16043851627370931e-05 ...
            -4.92956442341730183e-05 ...
             2.42333539881689037e-05 ...
             9.07080575782845380e-06 ...
            -8.86612136675773617e-06 ...
            -3.60751610287977163e-07 ...
             2.28837127614152731e-06 ...
            -4.42692340795287015e-07 ...
            -3.98579129198594408e-07 ...
             1.82244333257105344e-07 ...
             3.37797270373085438e-08 ...
            -3.98783819851888072e-08 ...
             3.67286357683818134e-09 ...
             5.11121185734745384e-09 ...
            -1.67139267725193250e-09 ...
            -2.49640210524619365e-10 ...
             2.42683310230568231e-10 ...
            -3.04957445394586343e-11 ...
            -1.42023685988993679e-11 ...
             5.50941472076552455e-12 ...
            -3.34348121895327877e-13 ...
            -2.15248838683330262e-13 ...
             6.21474024717439832e-14 ...
            -7.19651054536332241e-15 ...
             3.28937367841630637e-16
        ];
    case 34
        c = [
             5.77051063273028563e-06 ...
             1.29947620067953004e-04 ...
             1.36406139005905000e-03 ...
             8.81988940388497880e-03 ...
             3.90488413517859414e-02 ...
             1.24152482111376808e-01 ...
             2.87765059233714563e-01 ...
             4.78478746279371062e-01 ...
             5.30555099656463177e-01 ...
             2.90366329507274951e-01 ...
            -1.28246842174437167e-01 ...
            -3.31525301508386942e-01 ...
            -1.03891915515640472e-01 ...
             2.16907220187427595e-01 ...
             1.66601750412207444e-01 ...
            -1.27337358223801156e-01 ...
            -1.60924927177866806e-01 ...
             7.79918469379481074e-02 ...
             1.34125960271136128e-01 ...
            -5.44829680641390464e-02 ...
            -1.02947596992814085e-01 ...
             4.35760946496312973e-02 ...
             7.31852354367956056e-02 ...
            -3.70128384178624496e-02 ...
            -4.74385596452777625e-02 ...
             3.07397465739593446e-02 ...
             2.72283507563541961e-02 ...
            -2.36717379228263649e-02 ...
            -1.31439800166571609e-02 ...
             1.64093741998651925e-02 ...
             4.71364926099980991e-03 ...
            -1.00455067083615192e-02 ...
            -6.19474884515387284e-04 ...
             5.33495076875993603e-03 ...
            -7.69212797506783698e-04 ...
            -2.39945394353705586e-03 ...
             8.58995987436366196e-04 ...
             8.75199906407868873e-04 ...
            -5.52735576214419798e-04 ...
            -2.32673214023353164e-04 ...
             2.65077239755805782e-04 ...
             2.66005001845344190e-05 ...
            -9.91469777078013460e-05 ...
             1.35311722724964958e-05 ...
             2.84495141969780738e-05 ...
            -1.05765749425795062e-05 ...
            -5.71082651099830394e-06 ...
             4.16987175854702840e-06 ...
             4.97971810142130775e-07 ...
            -1.11630653481700843e-06 ...
             1.44819570833318513e-07 ...
             2.02599066666785922e-07 ...
            -7.52670174041258941e-08 ...
            -1.99034650153173692e-08 ...
             1.74042333293606808e-08 ...
            -8.66574426136872222e-10 ...
            -2.31650194699548275e-09 ...
             6.44637821032340231e-10 ...
             1.30041031860941525e-10 ...
            -9.90477453763240902e-11 ...
             1.00420873546176986e-11 ...
             6.08012535400016725e-12 ...
            -2.10787910891530155e-12 ...
             9.79945115821159773e-14 ...
             8.57919405179973318e-14 ...
            -2.31708370390640848e-14 ...
             2.58733838193569956e-15 ...
            -1.14894475448059013e-16
        ];
    case 35
        c = [
             4.06793406114855903e-06 ...
             9.42146947557674063e-05 ...
             1.01912268037509811e-03 ...
             6.80729288431913201e-03 ...
             3.12362885114907145e-02 ...
             1.03404455861478379e-01 ...
             2.51307378994493313e-01 ...
             4.43592739224035438e-01 ...
             5.37008427509166103e-01 ...
             3.60345640518047328e-01 ...
            -4.38838818739340411e-02 ...
            -3.23822864912116121e-01 ...
            -1.81786976766727833e-01 ...
             1.66041357490780920e-01 ...
             2.17299289321089298e-01 ...
            -6.52628713106775389e-02 ...
            -1.91919589298593953e-01 ...
             1.93095446660183509e-02 ...
             1.55292480396237114e-01 ...
            -4.75268083411135045e-03 ...
            -1.20585522643393555e-01 ...
             4.73422917264194876e-03 ...
             8.99135475707295442e-02 ...
            -9.31855894990392484e-03 ...
            -6.33560374404434661e-02 ...
             1.32285495850365552e-02 ...
             4.12546930647050921e-02 ...
            -1.43668397842200718e-02 ...
            -2.41694978016602674e-02 ...
             1.27664567156567442e-02 ...
             1.22894360081187109e-02 ...
            -9.57779789923571000e-03 ...
            -5.08599164923342988e-03 ...
             6.13775458674052109e-03 ...
             1.42808879407076211e-03 ...
            -3.35764438092238323e-03 ...
             7.61596943517273655e-06 ...
             1.54963746970236298e-03 ...
            -3.34669216425085496e-04 ...
            -5.86481031899181753e-04 ...
             2.64832881996128904e-04 ...
             1.70001228366124904e-04 ...
            -1.36588307226116160e-04 ...
            -2.97699596284850974e-05 ...
             5.30414312291331022e-05 ...
            -2.43700152682778986e-06 ...
            -1.57244207727028169e-05 ...
             4.30804786171673119e-06 ...
             3.35334586287130989e-06 ...
            -1.89592961769315329e-06 ...
            -3.90393173328730617e-07 ...
             5.30236861690476092e-07 ...
            -3.70030837820512454e-08 ...
            -9.99039694453490076e-08 ...
             3.00818865071906693e-08 ...
             1.08490273378993483e-08 ...
            -7.45811655289303763e-09 ...
             5.89795131038436158e-11 ...
             1.03082334548543338e-09 ...
            -2.43354557375167294e-10 ...
            -6.40793825650188902e-11 ...
             4.00053662725374451e-11 ...
            -3.12563935710855754e-12 ...
            -2.56706547615508145e-12 ...
             8.01508853368790092e-13 ...
            -2.59795432889384808e-14 ...
            -3.39772085679626743e-14 ...
             8.62403743472008920e-15 ...
            -9.29801252932418542e-16 ...
             4.01462871233348865e-17
        ];
    case 36
        c = [
             2.86792518275594633e-06 ...
             6.82602867854635869e-05 ...
             7.60215109966848829e-04 ...
             5.24029737740988437e-03 ...
             2.48905656448279648e-02 ...
             8.56520925952640908e-02 ...
             2.17756953097900815e-01 ...
             4.06433697708255347e-01 ...
             5.32266895260728691e-01 ...
             4.17875335600969786e-01 ...
             4.39751975293486299e-02 ...
            -2.94421039589114571e-01 ...
            -2.46807036978125527e-01 ...
             9.81142041631147705e-02 ...
             2.46537277608974211e-01 ...
             7.27851509579222901e-03 ...
            -1.99337205608649620e-01 ...
            -4.58614007463927164e-02 ...
             1.54106236627642884e-01 ...
             5.02761800735384286e-02 ...
            -1.18803754310135632e-01 ...
            -3.98808535755131758e-02 ...
             9.11567822580165441e-02 ...
             2.50387214495684899e-02 ...
            -6.82090166368175112e-02 ...
            -1.13191003168174279e-02 ...
             4.85130835478090854e-02 ...
             1.42497266176539160e-03 ...
            -3.19807206776396965e-02 ...
             3.98404019871700486e-03 ...
             1.90635947806253593e-02 ...
            -5.65781324505881838e-03 ...
            -9.99026347328137235e-03 ...
             5.02298910666582900e-03 ...
             4.41348483535057525e-03 ...
            -3.48454144540488331e-03 ...
            -1.50307406629664375e-03 ...
             1.99079377185173727e-03 ...
             2.77681279571202607e-04 ...
            -9.46340382326110196e-04 ...
             8.61456575899270203e-05 ...
             3.69350728496751050e-04 ...
            -1.15511889584352710e-04 ...
            -1.13189946808466567e-04 ...
             6.69474119693059026e-05 ...
             2.37510668366086078e-05 ...
            -2.73139082465433791e-05 ...
            -1.18347105998561594e-06 ...
             8.37221819816078843e-06 ...
            -1.58614578243457750e-06 ...
            -1.87081160285918071e-06 ...
             8.31142127970777853e-07 ...
             2.54842352255657783e-07 ...
            -2.45537765843423270e-07 ...
             2.75324907333951225e-09 ...
             4.79904346545099201e-08 ...
            -1.15609368881700841e-08 ...
            -5.61278434332779140e-09 ...
             3.13884169578242402e-09 ...
             1.09081555371375181e-10 ...
            -4.51254577856324963e-10 ...
             8.96241820385961199e-11 ...
             3.03742909811253522e-11 ...
            -1.59971668926135714e-11 ...
             8.87684628721737421e-13 ...
             1.07096935711401700e-12 ...
            -3.02928502697487727e-13 ...
             5.54226318263980424e-15 ...
             1.33807138629910590e-14 ...
            -3.20462854340174986e-15 ...
             3.33997198481869321e-16 ...
            -1.40327417537319062e-17
        ];
    case 37
        c = [
             2.02206086249839212e-06 ...
             4.94234375062813200e-05 ...
             5.66241837706672401e-04 ...
             4.02414036825728677e-03 ...
             1.97622861538795915e-02 ...
             7.05848259771816083e-02 ...
             1.87326331862064945e-01 ...
             3.68440972400306141e-01 ...
             5.18167040855622887e-01 ...
             4.62207553661605715e-01 ...
             1.30878963233020173e-01 ...
            -2.46180429761083413e-01 ...
            -2.94375915262661772e-01 ...
             1.96715004523593898e-02 ...
             2.51523254360268693e-01 ...
             8.18060283872186234e-02 ...
            -1.81962291778608001e-01 ...
            -1.08451713823301785e-01 ...
             1.29929646959853753e-01 ...
             1.01780296838814180e-01 ...
            -9.66075406166843903e-02 ...
            -8.23302119065574087e-02 ...
             7.50476199483601793e-02 ...
             5.95674108715299525e-02 ...
            -5.92568156326589710e-02 ...
            -3.82538294793842488e-02 ...
             4.58079441512683325e-02 ...
             2.09728005925975488e-02 ...
            -3.35235840641009699e-02 ...
            -8.83349389041023239e-03 ...
             2.26186515445994736e-02 ...
             1.69047238348442374e-03 ...
            -1.37639819628947843e-02 ...
             1.51930577883339922e-03 ...
             7.38775745285558364e-03 ...
            -2.24805318700382471e-03 ...
            -3.39452327640839860e-03 ...
             1.81687134380142353e-03 ...
             1.26393425811747718e-03 ...
            -1.11148486531863020e-03 ...
            -3.28078847088019842e-04 ...
             5.49053277337363123e-04 ...
             1.53443902319550321e-05 ...
            -2.20894403245549385e-04 ...
             4.33672612594569521e-05 ...
             7.05513878206546508e-05 ...
            -3.09866292761993005e-05 ...
            -1.63916249616058310e-05 ...
             1.35432771841678181e-05 ...
             1.84994500311559039e-06 ...
            -4.30994155659709239e-06 ...
             4.85473139699641168e-07 ...
             1.00212139929717763e-06 ...
            -3.49494860344572765e-07 ...
            -1.50988538867158355e-07 ...
             1.10903123221643939e-07 ...
             5.35065751546143429e-09 ...
            -2.25219383672480578e-08 ...
             4.22448570636241927e-09 ...
             2.79397446595398266e-09 ...
            -1.29720500146943514e-09 ...
            -1.03141112909697497e-10 ...
             1.94616489408231502e-10 ...
            -3.20339824412324137e-11 ...
            -1.39841571553764149e-11 ...
             6.33495544097391325e-12 ...
            -2.09636319423480054e-13 ...
            -4.42161240987210537e-13 ...
             1.13805283092143968e-13 ...
            -4.51888960746372639e-16 ...
            -5.24302569188420583e-15 ...
             1.18901238750825288e-15 ...
            -1.19928033585287955e-16 ...
             4.90661506493520369e-18
        ];
    case 38
        c = [
             1.42577664167413167e-06 ...
             3.57625199426402301e-05 ...
             4.21170266472711643e-04 ...
             3.08308811925375177e-03 ...
             1.56372493475721562e-02 ...
             5.78899436128592565e-02 ...
             1.60071993564110697e-01 ...
             3.30775781411014651e-01 ...
             4.96591175311718098e-01 ...
             4.93356078517100798e-01 ...
             2.13050571355578514e-01 ...
            -1.82867667708335891e-01 ...
            -3.21675637808997863e-01 ...
            -6.22665060478243223e-02 ...
             2.32125963835353109e-01 ...
             1.49985119618717020e-01 ...
            -1.41795685973059622e-01 ...
            -1.59912565158244362e-01 ...
             8.56381215561510574e-02 ...
             1.41414734073382680e-01 ...
            -5.65864586307273815e-02 ...
            -1.14731170710744375e-01 ...
             4.30958954330476429e-02 ...
             8.72043982620397501e-02 ...
            -3.66051034028742957e-02 ...
            -6.17662087084131599e-02 ...
             3.19898775315378063e-02 ...
             4.00549811051159482e-02 ...
            -2.68914938808945144e-02 ...
            -2.31141340205493168e-02 ...
             2.09046452556552434e-02 ...
             1.12904972786859648e-02 ...
            -1.47018820653986821e-02 ...
            -4.13130665603108927e-03 ...
             9.21478503219718051e-03 ...
             5.62571574840353201e-04 ...
            -5.07131450921834809e-03 ...
             7.16982182106401926e-04 ...
             2.40069778189097318e-03 ...
            -8.44862666553777501e-04 ...
            -9.42461407722737796e-04 ...
             5.81075975053286366e-04 ...
             2.81763925038067075e-04 ...
            -3.03102046072661199e-04 ...
            -4.55568269666842027e-05 ...
             1.26204335016617071e-04 ...
            -1.15540910383371719e-05 ...
            -4.17514164854039780e-05 ...
             1.33417614992135038e-05 ...
             1.03735918404559980e-05 ...
            -6.45673042846961916e-06 ...
            -1.55084435011860258e-06 ...
             2.14996026993966521e-06 ...
            -8.48708758607259307e-08 ...
            -5.18773373887414443e-07 ...
             1.39637754550835548e-07 ...
             8.40035104689596553e-08 ...
            -4.88475793745928676e-08 ...
            -5.42427480028729851e-09 ...
             1.03470453927485848e-08 ...
            -1.43632948779513571e-09 ...
            -1.34919775398344882e-09 ...
             5.26113255735759849e-10 ...
             6.73233649018930869e-11 ...
            -8.27825652253813473e-11 ...
             1.10169293459945455e-11 ...
             6.29153731703950858e-12 ...
            -2.48478923756364286e-12 ...
             2.62649650406525207e-14 ...
             1.80866123627453058e-13 ...
            -4.24981781957146301e-14 ...
            -4.56339716212737311e-16 ...
             2.04509967678898891e-15 ...
            -4.40530704248346134e-16 ...
             4.30459683955879002e-17 ...
            -1.71615245108874419e-18
        ];
end;

