#pragma once

typedef struct {
  size_t modBits;
  uint8_t *n;
  size_t eBits;
  uint8_t *e;
  size_t dBits;
  uint8_t *d;
  size_t msgLen;
  uint8_t *msg;
  size_t saltLen;
  uint8_t *salt;
  uint8_t *sgnt_expected;
} rsapss_test_vector;


static uint8_t test1_n[128] = {
  0xa5, 0x6e, 0x4a, 0x0e, 0x70, 0x10, 0x17, 0x58, 0x9a, 0x51, 0x87, 0xdc, 0x7e, 0xa8, 0x41, 0xd1,
  0x56, 0xf2, 0xec, 0x0e, 0x36, 0xad, 0x52, 0xa4, 0x4d, 0xfe, 0xb1, 0xe6, 0x1f, 0x7a, 0xd9, 0x91,
  0xd8, 0xc5, 0x10, 0x56, 0xff, 0xed, 0xb1, 0x62, 0xb4, 0xc0, 0xf2, 0x83, 0xa1, 0x2a, 0x88, 0xa3,
  0x94, 0xdf, 0xf5, 0x26, 0xab, 0x72, 0x91, 0xcb, 0xb3, 0x07, 0xce, 0xab, 0xfc, 0xe0, 0xb1, 0xdf,
  0xd5, 0xcd, 0x95, 0x08, 0x09, 0x6d, 0x5b, 0x2b, 0x8b, 0x6d, 0xf5, 0xd6, 0x71, 0xef, 0x63, 0x77,
  0xc0, 0x92, 0x1c, 0xb2, 0x3c, 0x27, 0x0a, 0x70, 0xe2, 0x59, 0x8e, 0x6f, 0xf8, 0x9d, 0x19, 0xf1,
  0x05, 0xac, 0xc2, 0xd3, 0xf0, 0xcb, 0x35, 0xf2, 0x92, 0x80, 0xe1, 0x38, 0x6b, 0x6f, 0x64, 0xc4,
  0xef, 0x22, 0xe1, 0xe1, 0xf2, 0x0d, 0x0c, 0xe8, 0xcf, 0xfb, 0x22, 0x49, 0xbd, 0x9a, 0x21, 0x37 };

static uint8_t test1_e[3] = { 0x01, 0x00, 0x01 };

static uint8_t test1_d[128] = {
  0x33, 0xa5, 0x04, 0x2a, 0x90, 0xb2, 0x7d, 0x4f, 0x54, 0x51, 0xca, 0x9b, 0xbb, 0xd0, 0xb4, 0x47,
  0x71, 0xa1, 0x01, 0xaf, 0x88, 0x43, 0x40, 0xae, 0xf9, 0x88, 0x5f, 0x2a, 0x4b, 0xbe, 0x92, 0xe8,
  0x94, 0xa7, 0x24, 0xac, 0x3c, 0x56, 0x8c, 0x8f, 0x97, 0x85, 0x3a, 0xd0, 0x7c, 0x02, 0x66, 0xc8,
  0xc6, 0xa3, 0xca, 0x09, 0x29, 0xf1, 0xe8, 0xf1, 0x12, 0x31, 0x88, 0x44, 0x29, 0xfc, 0x4d, 0x9a,
  0xe5, 0x5f, 0xee, 0x89, 0x6a, 0x10, 0xce, 0x70, 0x7c, 0x3e, 0xd7, 0xe7, 0x34, 0xe4, 0x47, 0x27,
  0xa3, 0x95, 0x74, 0x50, 0x1a, 0x53, 0x26, 0x83, 0x10, 0x9c, 0x2a, 0xba, 0xca, 0xba, 0x28, 0x3c,
  0x31, 0xb4, 0xbd, 0x2f, 0x53, 0xc3, 0xee, 0x37, 0xe3, 0x52, 0xce, 0xe3, 0x4f, 0x9e, 0x50, 0x3b,
  0xd8, 0x0c, 0x06, 0x22, 0xad, 0x79, 0xc6, 0xdc, 0xee, 0x88, 0x35, 0x47, 0xc6, 0xa3, 0xb3, 0x25 };

static uint8_t test1_msg[51] = {
  0x85, 0x13, 0x84, 0xcd, 0xfe, 0x81, 0x9c, 0x22, 0xed, 0x6c, 0x4c, 0xcb, 0x30, 0xda, 0xeb, 0x5c,
  0xf0, 0x59, 0xbc, 0x8e, 0x11, 0x66, 0xb7, 0xe3, 0x53, 0x0c, 0x4c, 0x23, 0x3e, 0x2b, 0x5f, 0x8f,
  0x71, 0xa1, 0xcc, 0xa5, 0x82, 0xd4, 0x3e, 0xcc, 0x72, 0xb1, 0xbc, 0xa1, 0x6d, 0xfc, 0x70, 0x13,
  0x22, 0x6b, 0x9e };

static uint8_t test1_salt[20] = {
  0xef, 0x28, 0x69, 0xfa, 0x40, 0xc3, 0x46, 0xcb, 0x18, 0x3d, 0xab, 0x3d, 0x7b, 0xff, 0xc9, 0x8f,
  0xd5, 0x6d, 0xf4, 0x2d };

static uint8_t test1_sgnt_expected[128] = {
  0x62, 0xeb, 0xb8, 0x03, 0x3a, 0x2d, 0x0b, 0x8b, 0xec, 0x42, 0x56, 0x52, 0x29, 0x9b, 0x3f, 0x02,
  0x8f, 0xa8, 0x0c, 0x28, 0x11, 0x0d, 0xf5, 0x37, 0x47, 0x2e, 0x5e, 0xd6, 0x28, 0x62, 0xb9, 0x98,
  0x36, 0xe5, 0x7a, 0xa9, 0x8d, 0x4b, 0x94, 0x9a, 0x21, 0xf0, 0x21, 0xee, 0x33, 0x89, 0xff, 0x52,
  0x66, 0xe0, 0x54, 0xd4, 0x4e, 0x8c, 0x92, 0x48, 0x0a, 0xc9, 0x10, 0x67, 0xde, 0xfb, 0xae, 0xd4,
  0xdc, 0x3c, 0xe2, 0x43, 0xe8, 0x17, 0x52, 0x66, 0xd3, 0xec, 0x69, 0xfd, 0xb0, 0xed, 0xea, 0xc1,
  0x1c, 0x8c, 0x9e, 0x3e, 0x99, 0x41, 0x54, 0xa9, 0x33, 0x95, 0xa5, 0x11, 0xb4, 0xa1, 0x72, 0xf6,
  0x64, 0x4f, 0x37, 0xf6, 0x80, 0x7b, 0x86, 0x71, 0x6f, 0xc9, 0x07, 0xe1, 0xd0, 0xfc, 0x75, 0xbd,
  0xa7, 0x7e, 0x41, 0x1b, 0xfc, 0x60, 0xfd, 0x2e, 0xd9, 0x27, 0x8e, 0x92, 0x1a, 0x33, 0x02, 0x1f };


static uint8_t test2_n[129] = {
  0x01, 0xd4, 0x0c, 0x1b, 0xcf, 0x97, 0xa6, 0x8a, 0xe7, 0xcd, 0xbd, 0x8a, 0x7b, 0xf3, 0xe3, 0x4f,
  0xa1, 0x9d, 0xcc, 0xa4, 0xef, 0x75, 0xa4, 0x74, 0x54, 0x37, 0x5f, 0x94, 0x51, 0x4d, 0x88, 0xfe,
  0xd0, 0x06, 0xfb, 0x82, 0x9f, 0x84, 0x19, 0xff, 0x87, 0xd6, 0x31, 0x5d, 0xa6, 0x8a, 0x1f, 0xf3,
  0xa0, 0x93, 0x8e, 0x9a, 0xbb, 0x34, 0x64, 0x01, 0x1c, 0x30, 0x3a, 0xd9, 0x91, 0x99, 0xcf, 0x0c,
  0x7c, 0x7a, 0x8b, 0x47, 0x7d, 0xce, 0x82, 0x9e, 0x88, 0x44, 0xf6, 0x25, 0xb1, 0x15, 0xe5, 0xe9,
  0xc4, 0xa5, 0x9c, 0xf8, 0xf8, 0x11, 0x3b, 0x68, 0x34, 0x33, 0x6a, 0x2f, 0xd2, 0x68, 0x9b, 0x47,
  0x2c, 0xbb, 0x5e, 0x5c, 0xab, 0xe6, 0x74, 0x35, 0x0c, 0x59, 0xb6, 0xc1, 0x7e, 0x17, 0x68, 0x74,
  0xfb, 0x42, 0xf8, 0xfc, 0x3d, 0x17, 0x6a, 0x01, 0x7e, 0xdc, 0x61, 0xfd, 0x32, 0x6c, 0x4b, 0x33,
  0xc9 };

static uint8_t test2_e[3] = { 0x01, 0x00, 0x01 };

static uint8_t test2_d[128] = {
  0x02, 0x7d, 0x14, 0x7e, 0x46, 0x73, 0x05, 0x73, 0x77, 0xfd, 0x1e, 0xa2, 0x01, 0x56, 0x57, 0x72,
  0x17, 0x6a, 0x7d, 0xc3, 0x83, 0x58, 0xd3, 0x76, 0x04, 0x56, 0x85, 0xa2, 0xe7, 0x87, 0xc2, 0x3c,
  0x15, 0x57, 0x6b, 0xc1, 0x6b, 0x9f, 0x44, 0x44, 0x02, 0xd6, 0xbf, 0xc5, 0xd9, 0x8a, 0x3e, 0x88,
  0xea, 0x13, 0xef, 0x67, 0xc3, 0x53, 0xec, 0xa0, 0xc0, 0xdd, 0xba, 0x92, 0x55, 0xbd, 0x7b, 0x8b,
  0xb5, 0x0a, 0x64, 0x4a, 0xfd, 0xfd, 0x1d, 0xd5, 0x16, 0x95, 0xb2, 0x52, 0xd2, 0x2e, 0x73, 0x18,
  0xd1, 0xb6, 0x68, 0x7a, 0x1c, 0x10, 0xff, 0x75, 0x54, 0x5f, 0x3d, 0xb0, 0xfe, 0x60, 0x2d, 0x5f,
  0x2b, 0x7f, 0x29, 0x4e, 0x36, 0x01, 0xea, 0xb7, 0xb9, 0xd1, 0xce, 0xcd, 0x76, 0x7f, 0x64, 0x69,
  0x2e, 0x3e, 0x53, 0x6c, 0xa2, 0x84, 0x6c, 0xb0, 0xc2, 0xdd, 0x48, 0x6a, 0x39, 0xfa, 0x75, 0xb1 };

static uint8_t test2_msg[234] = {
  0xe4, 0xf8, 0x60, 0x1a, 0x8a, 0x6d, 0xa1, 0xbe, 0x34, 0x44, 0x7c, 0x09, 0x59, 0xc0, 0x58, 0x57,
  0x0c, 0x36, 0x68, 0xcf, 0xd5, 0x1d, 0xd5, 0xf9, 0xcc, 0xd6, 0xad, 0x44, 0x11, 0xfe, 0x82, 0x13,
  0x48, 0x6d, 0x78, 0xa6, 0xc4, 0x9f, 0x93, 0xef, 0xc2, 0xca, 0x22, 0x88, 0xce, 0xbc, 0x2b, 0x9b,
  0x60, 0xbd, 0x04, 0xb1, 0xe2, 0x20, 0xd8, 0x6e, 0x3d, 0x48, 0x48, 0xd7, 0x09, 0xd0, 0x32, 0xd1,
  0xe8, 0xc6, 0xa0, 0x70, 0xc6, 0xaf, 0x9a, 0x49, 0x9f, 0xcf, 0x95, 0x35, 0x4b, 0x14, 0xba, 0x61,
  0x27, 0xc7, 0x39, 0xde, 0x1b, 0xb0, 0xfd, 0x16, 0x43, 0x1e, 0x46, 0x93, 0x8a, 0xec, 0x0c, 0xf8,
  0xad, 0x9e, 0xb7, 0x2e, 0x83, 0x2a, 0x70, 0x35, 0xde, 0x9b, 0x78, 0x07, 0xbd, 0xc0, 0xed, 0x8b,
  0x68, 0xeb, 0x0f, 0x5a, 0xc2, 0x21, 0x6b, 0xe4, 0x0c, 0xe9, 0x20, 0xc0, 0xdb, 0x0e, 0xdd, 0xd3,
  0x86, 0x0e, 0xd7, 0x88, 0xef, 0xac, 0xca, 0xca, 0x50, 0x2d, 0x8f, 0x2b, 0xd6, 0xd1, 0xa7, 0xc1,
  0xf4, 0x1f, 0xf4, 0x6f, 0x16, 0x81, 0xc8, 0xf1, 0xf8, 0x18, 0xe9, 0xc4, 0xf6, 0xd9, 0x1a, 0x0c,
  0x78, 0x03, 0xcc, 0xc6, 0x3d, 0x76, 0xa6, 0x54, 0x4d, 0x84, 0x3e, 0x08, 0x4e, 0x36, 0x3b, 0x8a,
  0xcc, 0x55, 0xaa, 0x53, 0x17, 0x33, 0xed, 0xb5, 0xde, 0xe5, 0xb5, 0x19, 0x6e, 0x9f, 0x03, 0xe8,
  0xb7, 0x31, 0xb3, 0x77, 0x64, 0x28, 0xd9, 0xe4, 0x57, 0xfe, 0x3f, 0xbc, 0xb3, 0xdb, 0x72, 0x74,
  0x44, 0x2d, 0x78, 0x58, 0x90, 0xe9, 0xcb, 0x08, 0x54, 0xb6, 0x44, 0x4d, 0xac, 0xe7, 0x91, 0xd7,
  0x27, 0x3d, 0xe1, 0x88, 0x97, 0x19, 0x33, 0x8a, 0x77, 0xfe };

static uint8_t test2_salt[20] = {
  0x7f, 0x6d, 0xd3, 0x59, 0xe6, 0x04, 0xe6, 0x08, 0x70, 0xe8, 0x98, 0xe4, 0x7b, 0x19, 0xbf, 0x2e,
  0x5a, 0x7b, 0x2a, 0x90 };

static uint8_t test2_sgnt_expected[129] = {
  0x01, 0x90, 0x44, 0x7a, 0x91, 0xa3, 0xef, 0x1e, 0x9a, 0x36, 0x44, 0xb2, 0x2d, 0xb0, 0x9d, 0xb3,
  0x7b, 0x45, 0xe1, 0xd5, 0xfa, 0x2e, 0xa0, 0x8a, 0xec, 0x35, 0xd9, 0x81, 0x54, 0xc5, 0x2f, 0x31,
  0x5d, 0x4a, 0x71, 0x26, 0x70, 0xa2, 0x7e, 0xc4, 0xe5, 0xe3, 0xa0, 0x96, 0xf2, 0xe1, 0x0a, 0xa6,
  0x23, 0x90, 0x66, 0x40, 0x42, 0xc7, 0xb6, 0xb8, 0x2f, 0x24, 0x79, 0x70, 0xc6, 0x74, 0xf0, 0xca,
  0x79, 0x57, 0xb9, 0xe0, 0xf3, 0x0b, 0x23, 0x39, 0x07, 0x71, 0xee, 0x4a, 0x67, 0xd9, 0x1b, 0x30,
  0x39, 0xc6, 0x45, 0xee, 0x63, 0x7f, 0x50, 0x84, 0x20, 0x2d, 0x5b, 0x03, 0x03, 0xd5, 0x46, 0x6d,
  0x92, 0x72, 0xc5, 0xd7, 0x73, 0x36, 0x8a, 0xbc, 0x06, 0x84, 0xd6, 0xbc, 0xc1, 0x9d, 0x30, 0x27,
  0x73, 0x24, 0x54, 0x3e, 0xcd, 0xaf, 0x56, 0xf7, 0x44, 0x6e, 0x20, 0x79, 0xb8, 0x9c, 0xc4, 0x8f,
  0x2d };


static uint8_t test3_n[192] = {
  0xe6, 0xbd, 0x69, 0x2a, 0xc9, 0x66, 0x45, 0x79, 0x04, 0x03, 0xfd, 0xd0, 0xf5, 0xbe, 0xb8, 0xb9,
  0xbf, 0x92, 0xed, 0x10, 0x00, 0x7f, 0xc3, 0x65, 0x04, 0x64, 0x19, 0xdd, 0x06, 0xc0, 0x5c, 0x5b,
  0x5b, 0x2f, 0x48, 0xec, 0xf9, 0x89, 0xe4, 0xce, 0x26, 0x91, 0x09, 0x97, 0x9c, 0xbb, 0x40, 0xb4,
  0xa0, 0xad, 0x24, 0xd2, 0x24, 0x83, 0xd1, 0xee, 0x31, 0x5a, 0xd4, 0xcc, 0xb1, 0x53, 0x42, 0x68,
  0x35, 0x26, 0x91, 0xc5, 0x24, 0xf6, 0xdd, 0x8e, 0x6c, 0x29, 0xd2, 0x24, 0xcf, 0x24, 0x69, 0x73,
  0xae, 0xc8, 0x6c, 0x5b, 0xf6, 0xb1, 0x40, 0x1a, 0x85, 0x0d, 0x1b, 0x9a, 0xd1, 0xbb, 0x8c, 0xbc,
  0xec, 0x47, 0xb0, 0x6f, 0x0f, 0x8c, 0x7f, 0x45, 0xd3, 0xfc, 0x8f, 0x31, 0x92, 0x99, 0xc5, 0x43,
  0x3d, 0xdb, 0xc2, 0xb3, 0x05, 0x3b, 0x47, 0xde, 0xd2, 0xec, 0xd4, 0xa4, 0xca, 0xef, 0xd6, 0x14,
  0x83, 0x3d, 0xc8, 0xbb, 0x62, 0x2f, 0x31, 0x7e, 0xd0, 0x76, 0xb8, 0x05, 0x7f, 0xe8, 0xde, 0x3f,
  0x84, 0x48, 0x0a, 0xd5, 0xe8, 0x3e, 0x4a, 0x61, 0x90, 0x4a, 0x4f, 0x24, 0x8f, 0xb3, 0x97, 0x02,
  0x73, 0x57, 0xe1, 0xd3, 0x0e, 0x46, 0x31, 0x39, 0x81, 0x5c, 0x6f, 0xd4, 0xfd, 0x5a, 0xc5, 0xb8,
  0x17, 0x2a, 0x45, 0x23, 0x0e, 0xcb, 0x63, 0x18, 0xa0, 0x4f, 0x14, 0x55, 0xd8, 0x4e, 0x5a, 0x8b };

static uint8_t test3_e[3] = { 0x01, 0x00, 0x01 };

static uint8_t test3_d[192] = {
  0x6a, 0x7f, 0xd8, 0x4f, 0xb8, 0x5f, 0xad, 0x07, 0x3b, 0x34, 0x40, 0x6d, 0xb7, 0x4f, 0x8d, 0x61,
  0xa6, 0xab, 0xc1, 0x21, 0x96, 0xa9, 0x61, 0xdd, 0x79, 0x56, 0x5e, 0x9d, 0xa6, 0xe5, 0x18, 0x7b,
  0xce, 0x2d, 0x98, 0x02, 0x50, 0xf7, 0x35, 0x95, 0x75, 0x35, 0x92, 0x70, 0xd9, 0x15, 0x90, 0xbb,
  0x0e, 0x42, 0x7c, 0x71, 0x46, 0x0b, 0x55, 0xd5, 0x14, 0x10, 0xb1, 0x91, 0xbc, 0xf3, 0x09, 0xfe,
  0xa1, 0x31, 0xa9, 0x2c, 0x8e, 0x70, 0x27, 0x38, 0xfa, 0x71, 0x9f, 0x1e, 0x00, 0x41, 0xf5, 0x2e,
  0x40, 0xe9, 0x1f, 0x22, 0x9f, 0x4d, 0x96, 0xa1, 0xe6, 0xf1, 0x72, 0xe1, 0x55, 0x96, 0xb4, 0x51,
  0x0a, 0x6d, 0xae, 0xc2, 0x61, 0x05, 0xf2, 0xbe, 0xbc, 0x53, 0x31, 0x6b, 0x87, 0xbd, 0xf2, 0x13,
  0x11, 0x66, 0x60, 0x70, 0xe8, 0xdf, 0xee, 0x69, 0xd5, 0x2c, 0x71, 0xa9, 0x76, 0xca, 0xae, 0x79,
  0xc7, 0x2b, 0x68, 0xd2, 0x85, 0x80, 0xdc, 0x68, 0x6d, 0x9f, 0x51, 0x29, 0xd2, 0x25, 0xf8, 0x2b,
  0x3d, 0x61, 0x55, 0x13, 0xa8, 0x82, 0xb3, 0xdb, 0x91, 0x41, 0x6b, 0x48, 0xce, 0x08, 0x88, 0x82,
  0x13, 0xe3, 0x7e, 0xeb, 0x9a, 0xf8, 0x00, 0xd8, 0x1c, 0xab, 0x32, 0x8c, 0xe4, 0x20, 0x68, 0x99,
  0x03, 0xc0, 0x0c, 0x7b, 0x5f, 0xd3, 0x1b, 0x75, 0x50, 0x3a, 0x6d, 0x41, 0x96, 0x84, 0xd6, 0x29 };

static uint8_t test3_msg[107] = {
  0xc8, 0xc9, 0xc6, 0xaf, 0x04, 0xac, 0xda, 0x41, 0x4d, 0x22, 0x7e, 0xf2, 0x3e, 0x08, 0x20, 0xc3,
  0x73, 0x2c, 0x50, 0x0d, 0xc8, 0x72, 0x75, 0xe9, 0x5b, 0x0d, 0x09, 0x54, 0x13, 0x99, 0x3c, 0x26,
  0x58, 0xbc, 0x1d, 0x98, 0x85, 0x81, 0xba, 0x87, 0x9c, 0x2d, 0x20, 0x1f, 0x14, 0xcb, 0x88, 0xce,
  0xd1, 0x53, 0xa0, 0x19, 0x69, 0xa7, 0xbf, 0x0a, 0x7b, 0xe7, 0x9c, 0x84, 0xc1, 0x48, 0x6b, 0xc1,
  0x2b, 0x3f, 0xa6, 0xc5, 0x98, 0x71, 0xb6, 0x82, 0x7c, 0x8c, 0xe2, 0x53, 0xca, 0x5f, 0xef, 0xa8,
  0xa8, 0xc6, 0x90, 0xbf, 0x32, 0x6e, 0x8e, 0x37, 0xcd, 0xb9, 0x6d, 0x90, 0xa8, 0x2e, 0xba, 0xb6,
  0x9f, 0x86, 0x35, 0x0e, 0x18, 0x22, 0xe8, 0xbd, 0x53, 0x6a, 0x2e };

static uint8_t test3_salt[20] = {
  0xb3, 0x07, 0xc4, 0x3b, 0x48, 0x50, 0xa8, 0xda, 0xc2, 0xf1, 0x5f, 0x32, 0xe3, 0x78, 0x39, 0xef,
  0x8c, 0x5c, 0x0e, 0x91 };

static uint8_t test3_sgnt_expected[192] = {
  0x0c, 0x58, 0xaa, 0x0a, 0x5d, 0xe6, 0xd8, 0xa0, 0x0b, 0xb6, 0xac, 0x2d, 0x5c, 0x04, 0xfb, 0x0f,
  0xa3, 0x01, 0x12, 0x49, 0x3b, 0xde, 0x42, 0x28, 0x8a, 0x5b, 0xad, 0x5c, 0x7b, 0x4b, 0x51, 0x8e,
  0x21, 0xf3, 0x1c, 0x18, 0x54, 0x71, 0xb5, 0x9f, 0x87, 0x33, 0xc1, 0x3f, 0xe4, 0xc7, 0xfe, 0xc4,
  0xa2, 0x4d, 0x0d, 0x0c, 0xd6, 0x62, 0xec, 0xd5, 0xe7, 0x21, 0xb0, 0x53, 0x62, 0xd9, 0xb6, 0x72,
  0xa3, 0xd8, 0x26, 0x82, 0x55, 0x2c, 0x58, 0x30, 0x0d, 0xa6, 0x14, 0x65, 0x66, 0x38, 0xe6, 0x61,
  0x83, 0x9d, 0x33, 0xb4, 0xd3, 0xd3, 0x7e, 0x0f, 0xce, 0x8b, 0xa0, 0xe4, 0x93, 0xd0, 0x2b, 0xc4,
  0x73, 0xf8, 0x53, 0x78, 0x71, 0xbb, 0x56, 0x55, 0xc6, 0x94, 0x07, 0xb3, 0x62, 0xe0, 0x73, 0x90,
  0x07, 0xe0, 0x36, 0x7a, 0x39, 0xc0, 0x38, 0xce, 0xd3, 0x7f, 0xf4, 0xfb, 0x9f, 0x16, 0x0d, 0x4d,
  0x06, 0x39, 0x62, 0x17, 0x31, 0x5e, 0xe8, 0xd7, 0x5d, 0x91, 0x0b, 0x51, 0x28, 0x45, 0xf9, 0x70,
  0xfe, 0x74, 0xe4, 0x12, 0x26, 0x84, 0x71, 0xc9, 0x51, 0x81, 0x62, 0x51, 0x6c, 0xd6, 0xf9, 0x66,
  0x89, 0x2a, 0x74, 0x0e, 0x1b, 0x8a, 0x88, 0x76, 0x6a, 0x30, 0xfc, 0xe9, 0xb6, 0x0e, 0x03, 0x32,
  0xd7, 0xa0, 0x1b, 0xa5, 0xfa, 0x13, 0x5f, 0xe7, 0xc4, 0x92, 0x72, 0xac, 0xbb, 0x1d, 0x30, 0xf1 };


static uint8_t test4_n[256] = {
  0xa5, 0xdd, 0x86, 0x7a, 0xc4, 0xcb, 0x02, 0xf9, 0x0b, 0x94, 0x57, 0xd4, 0x8c, 0x14, 0xa7, 0x70,
  0xef, 0x99, 0x1c, 0x56, 0xc3, 0x9c, 0x0e, 0xc6, 0x5f, 0xd1, 0x1a, 0xfa, 0x89, 0x37, 0xce, 0xa5,
  0x7b, 0x9b, 0xe7, 0xac, 0x73, 0xb4, 0x5c, 0x00, 0x17, 0x61, 0x5b, 0x82, 0xd6, 0x22, 0xe3, 0x18,
  0x75, 0x3b, 0x60, 0x27, 0xc0, 0xfd, 0x15, 0x7b, 0xe1, 0x2f, 0x80, 0x90, 0xfe, 0xe2, 0xa7, 0xad,
  0xcd, 0x0e, 0xef, 0x75, 0x9f, 0x88, 0xba, 0x49, 0x97, 0xc7, 0xa4, 0x2d, 0x58, 0xc9, 0xaa, 0x12,
  0xcb, 0x99, 0xae, 0x00, 0x1f, 0xe5, 0x21, 0xc1, 0x3b, 0xb5, 0x43, 0x14, 0x45, 0xa8, 0xd5, 0xae,
  0x4f, 0x5e, 0x4c, 0x7e, 0x94, 0x8a, 0xc2, 0x27, 0xd3, 0x60, 0x40, 0x71, 0xf2, 0x0e, 0x57, 0x7e,
  0x90, 0x5f, 0xbe, 0xb1, 0x5d, 0xfa, 0xf0, 0x6d, 0x1d, 0xe5, 0xae, 0x62, 0x53, 0xd6, 0x3a, 0x6a,
  0x21, 0x20, 0xb3, 0x1a, 0x5d, 0xa5, 0xda, 0xbc, 0x95, 0x50, 0x60, 0x0e, 0x20, 0xf2, 0x7d, 0x37,
  0x39, 0xe2, 0x62, 0x79, 0x25, 0xfe, 0xa3, 0xcc, 0x50, 0x9f, 0x21, 0xdf, 0xf0, 0x4e, 0x6e, 0xea,
  0x45, 0x49, 0xc5, 0x40, 0xd6, 0x80, 0x9f, 0xf9, 0x30, 0x7e, 0xed, 0xe9, 0x1f, 0xff, 0x58, 0x73,
  0x3d, 0x83, 0x85, 0xa2, 0x37, 0xd6, 0xd3, 0x70, 0x5a, 0x33, 0xe3, 0x91, 0x90, 0x09, 0x92, 0x07,
  0x0d, 0xf7, 0xad, 0xf1, 0x35, 0x7c, 0xf7, 0xe3, 0x70, 0x0c, 0xe3, 0x66, 0x7d, 0xe8, 0x3f, 0x17,
  0xb8, 0xdf, 0x17, 0x78, 0xdb, 0x38, 0x1d, 0xce, 0x09, 0xcb, 0x4a, 0xd0, 0x58, 0xa5, 0x11, 0x00,
  0x1a, 0x73, 0x81, 0x98, 0xee, 0x27, 0xcf, 0x55, 0xa1, 0x3b, 0x75, 0x45, 0x39, 0x90, 0x65, 0x82,
  0xec, 0x8b, 0x17, 0x4b, 0xd5, 0x8d, 0x5d, 0x1f, 0x3d, 0x76, 0x7c, 0x61, 0x37, 0x21, 0xae, 0x05 };

static uint8_t test4_e[3] = { 0x01, 0x00, 0x01 };

static uint8_t test4_d[256] = {
  0x2d, 0x2f, 0xf5, 0x67, 0xb3, 0xfe, 0x74, 0xe0, 0x61, 0x91, 0xb7, 0xfd, 0xed, 0x6d, 0xe1, 0x12,
  0x29, 0x0c, 0x67, 0x06, 0x92, 0x43, 0x0d, 0x59, 0x69, 0x18, 0x40, 0x47, 0xda, 0x23, 0x4c, 0x96,
  0x93, 0xde, 0xed, 0x16, 0x73, 0xed, 0x42, 0x95, 0x39, 0xc9, 0x69, 0xd3, 0x72, 0xc0, 0x4d, 0x6b,
  0x47, 0xe0, 0xf5, 0xb8, 0xce, 0xe0, 0x84, 0x3e, 0x5c, 0x22, 0x83, 0x5d, 0xbd, 0x3b, 0x05, 0xa0,
  0x99, 0x79, 0x84, 0xae, 0x60, 0x58, 0xb1, 0x1b, 0xc4, 0x90, 0x7c, 0xbf, 0x67, 0xed, 0x84, 0xfa,
  0x9a, 0xe2, 0x52, 0xdf, 0xb0, 0xd0, 0xcd, 0x49, 0xe6, 0x18, 0xe3, 0x5d, 0xfd, 0xfe, 0x59, 0xbc,
  0xa3, 0xdd, 0xd6, 0x6c, 0x33, 0xce, 0xbb, 0xc7, 0x7a, 0xd4, 0x41, 0xaa, 0x69, 0x5e, 0x13, 0xe3,
  0x24, 0xb5, 0x18, 0xf0, 0x1c, 0x60, 0xf5, 0xa8, 0x5c, 0x99, 0x4a, 0xd1, 0x79, 0xf2, 0xa6, 0xb5,
  0xfb, 0xe9, 0x34, 0x02, 0xb1, 0x17, 0x67, 0xbe, 0x01, 0xbf, 0x07, 0x34, 0x44, 0xd6, 0xba, 0x1d,
  0xd2, 0xbc, 0xa5, 0xbd, 0x07, 0x4d, 0x4a, 0x5f, 0xae, 0x35, 0x31, 0xad, 0x13, 0x03, 0xd8, 0x4b,
  0x30, 0xd8, 0x97, 0x31, 0x8c, 0xbb, 0xba, 0x04, 0xe0, 0x3c, 0x2e, 0x66, 0xde, 0x6d, 0x91, 0xf8,
  0x2f, 0x96, 0xea, 0x1d, 0x4b, 0xb5, 0x4a, 0x5a, 0xae, 0x10, 0x2d, 0x59, 0x46, 0x57, 0xf5, 0xc9,
  0x78, 0x95, 0x53, 0x51, 0x2b, 0x29, 0x6d, 0xea, 0x29, 0xd8, 0x02, 0x31, 0x96, 0x35, 0x7e, 0x3e,
  0x3a, 0x6e, 0x95, 0x8f, 0x39, 0xe3, 0xc2, 0x34, 0x40, 0x38, 0xea, 0x60, 0x4b, 0x31, 0xed, 0xc6,
  0xf0, 0xf7, 0xff, 0x6e, 0x71, 0x81, 0xa5, 0x7c, 0x92, 0x82, 0x6a, 0x26, 0x8f, 0x86, 0x76, 0x8e,
  0x96, 0xf8, 0x78, 0x56, 0x2f, 0xc7, 0x1d, 0x85, 0xd6, 0x9e, 0x44, 0x86, 0x12, 0xf7, 0x04, 0x8f };

static uint8_t test4_msg[128] = {
  0xdd, 0x67, 0x0a, 0x01, 0x46, 0x58, 0x68, 0xad, 0xc9, 0x3f, 0x26, 0x13, 0x19, 0x57, 0xa5, 0x0c,
  0x52, 0xfb, 0x77, 0x7c, 0xdb, 0xaa, 0x30, 0x89, 0x2c, 0x9e, 0x12, 0x36, 0x11, 0x64, 0xec, 0x13,
  0x97, 0x9d, 0x43, 0x04, 0x81, 0x18, 0xe4, 0x44, 0x5d, 0xb8, 0x7b, 0xee, 0x58, 0xdd, 0x98, 0x7b,
  0x34, 0x25, 0xd0, 0x20, 0x71, 0xd8, 0xdb, 0xae, 0x80, 0x70, 0x8b, 0x03, 0x9d, 0xbb, 0x64, 0xdb,
  0xd1, 0xde, 0x56, 0x57, 0xd9, 0xfe, 0xd0, 0xc1, 0x18, 0xa5, 0x41, 0x43, 0x74, 0x2e, 0x0f, 0xf3,
  0xc8, 0x7f, 0x74, 0xe4, 0x58, 0x57, 0x64, 0x7a, 0xf3, 0xf7, 0x9e, 0xb0, 0xa1, 0x4c, 0x9d, 0x75,
  0xea, 0x9a, 0x1a, 0x04, 0xb7, 0xcf, 0x47, 0x8a, 0x89, 0x7a, 0x70, 0x8f, 0xd9, 0x88, 0xf4, 0x8e,
  0x80, 0x1e, 0xdb, 0x0b, 0x70, 0x39, 0xdf, 0x8c, 0x23, 0xbb, 0x3c, 0x56, 0xf4, 0xe8, 0x21, 0xac };

static uint8_t test4_salt[20] = {
  0x8b, 0x2b, 0xdd, 0x4b, 0x40, 0xfa, 0xf5, 0x45, 0xc7, 0x78, 0xdd, 0xf9, 0xbc, 0x1a, 0x49, 0xcb,
  0x57, 0xf9, 0xb7, 0x1b };

static uint8_t test4_sgnt_expected[256] = {
  0xa4, 0x4e, 0x5c, 0x83, 0xc6, 0xfe, 0xdf, 0x7f, 0x44, 0x33, 0x78, 0x82, 0x54, 0x2a, 0x96, 0x10,
  0x72, 0x4a, 0xa6, 0xf5, 0xb8, 0xf1, 0x3b, 0x4f, 0x51, 0xeb, 0x9e, 0xf9, 0x84, 0xf5, 0x19, 0xaa,
  0xe9, 0xe3, 0x4b, 0x26, 0x4e, 0x8d, 0x06, 0xb6, 0x93, 0x66, 0x4d, 0xe1, 0xcc, 0xe1, 0x36, 0xd0,
  0x6d, 0x10, 0x7f, 0x64, 0x51, 0x99, 0x8a, 0xf9, 0x01, 0x21, 0x3f, 0xc8, 0x95, 0x83, 0xe6, 0xbe,
  0xfe, 0x1e, 0xd1, 0x12, 0x35, 0xf5, 0xb5, 0xce, 0x8b, 0xd4, 0x72, 0xb3, 0x84, 0xef, 0xf0, 0xcd,
  0x80, 0xd3, 0x75, 0xbd, 0x6a, 0x88, 0xae, 0x6f, 0x5b, 0x76, 0x75, 0xc2, 0x50, 0x8b, 0xa9, 0xb9,
  0xf0, 0x17, 0x1e, 0x10, 0xc9, 0x58, 0xd4, 0xc0, 0x4c, 0x10, 0x0e, 0xf9, 0x06, 0xcc, 0x97, 0x58,
  0x0d, 0xe7, 0x73, 0xad, 0x9d, 0xf4, 0xda, 0x13, 0xd5, 0x95, 0xbe, 0xe2, 0x4a, 0xf8, 0x12, 0x88,
  0x4e, 0xd4, 0xdc, 0xe8, 0x09, 0x51, 0xec, 0xd0, 0x4b, 0x1b, 0xa6, 0xd7, 0x8c, 0x29, 0x34, 0xe6,
  0xab, 0x0a, 0x77, 0x36, 0x83, 0x91, 0x1f, 0xcc, 0x68, 0x91, 0x35, 0x37, 0x67, 0x27, 0x78, 0x09,
  0xec, 0x74, 0x6f, 0x95, 0x98, 0xe4, 0xf8, 0xf0, 0xcb, 0x1d, 0x3d, 0x37, 0x84, 0x3f, 0xea, 0x2a,
  0x8c, 0xb0, 0x91, 0xf2, 0x91, 0x91, 0x22, 0x76, 0x9e, 0xe4, 0x17, 0xda, 0x18, 0xd6, 0x03, 0xf7,
  0x98, 0x37, 0x0c, 0xad, 0x7b, 0x76, 0x0a, 0x7f, 0x57, 0x3a, 0xea, 0xf5, 0x16, 0xa0, 0xf9, 0x0d,
  0x95, 0x25, 0x65, 0xb8, 0xa1, 0x9a, 0x8f, 0xc3, 0xf0, 0xee, 0x7d, 0x39, 0x1d, 0x9b, 0x8b, 0x3f,
  0x98, 0xbe, 0xbb, 0x0d, 0x5d, 0x01, 0x0e, 0x32, 0xe0, 0xb8, 0x00, 0xe9, 0x65, 0x6f, 0x64, 0x08,
  0x2b, 0xb1, 0xac, 0x95, 0xa2, 0x23, 0xf4, 0x31, 0xec, 0x40, 0x6a, 0x42, 0x95, 0x4b, 0x2d, 0x57 };


static rsapss_test_vector vectors[] = {
  {
    .modBits = 1024,
    .n = test1_n,
    .eBits = 24,
    .e = test1_e,
    .dBits = 1024,
    .d = test1_d,
    .msgLen = sizeof(test1_msg)/sizeof(uint8_t),
    .msg = test1_msg,
    .saltLen = 20,
    .salt = test1_salt,
    .sgnt_expected = test1_sgnt_expected
  },
  {
    .modBits = 1025,
    .n = test2_n,
    .eBits = 24,
    .e = test2_e,
    .dBits = 1024,
    .d = test2_d,
    .msgLen = sizeof(test2_msg)/sizeof(uint8_t),
    .msg = test2_msg,
    .saltLen = 20,
    .salt = test2_salt,
    .sgnt_expected = test2_sgnt_expected
  },
  {
    .modBits = 1536,
    .n = test3_n,
    .eBits = 24,
    .e = test3_e,
    .dBits = 1536,
    .d = test3_d,
    .msgLen = sizeof(test3_msg)/sizeof(uint8_t),
    .msg = test3_msg,
    .saltLen = 20,
    .salt = test3_salt,
    .sgnt_expected = test3_sgnt_expected
  },
  {
    .modBits = 2048,
    .n = test4_n,
    .eBits = 24,
    .e = test4_e,
    .dBits = 2048,
    .d = test4_d,
    .msgLen = sizeof(test4_msg)/sizeof(uint8_t),
    .msg = test4_msg,
    .saltLen = 20,
    .salt = test4_salt,
    .sgnt_expected = test4_sgnt_expected
  }
};
