module Spec.ECDSA.Test.Vectors

open Lib.Meta

#set-options "--fuel 0 --ifuel 0"

///
/// ECDSA test vectors from NIST CAVP
/// https://csrc.nist.gov/Projects/Cryptographic-Algorithm-Validation-Program/Digital-Signatures#ecdsa2vs
///

type vec_SigVer = {
  msg: hex_string;
  qx: hex_string;
  qy: hex_string;
  r: hex_string;
  s: hex_string;
  result: bool;
}

type vec_SigGen = {
  msg': hex_string;
  d: hex_string;
  qx': hex_string;
  qy': hex_string;
  k: hex_string;
  r': hex_string;
  s': hex_string;
}


 let sigver_vectors_sha2_256 : list vec_SigVer = [
  { msg = "e4796db5f785f207aa30d311693b3702821dff1168fd2e04c0836825aefd850d9aa60326d88cde1a23c7745351392ca2288d632c264f197d05cd424a30336c19fd09bb229654f0222fcb881a4b35c290a093ac159ce13409111ff0358411133c24f5b8e2090d6db6558afc36f06ca1f6ef779785adba68db27a409859fc4c4a0";
    qx  = "87f8f2b218f49845f6f10eec3877136269f5c1a54736dbdf69f89940cad41555";
    qy  = "e15f369036f49842fac7a86c8a2b0557609776814448b8f5e84aa9f4395205e9";
    r   = "d19ff48b324915576416097d2544f7cbdf8768b1454ad20e0baac50e211f23b0";
    s   = "a3e81e59311cdfff2d4784949f7a2cb50ba6c3a91fa54710568e61aca3e847c6";
    result = false;
  };
  { msg = "069a6e6b93dfee6df6ef6997cd80dd2182c36653cef10c655d524585655462d683877f95ecc6d6c81623d8fac4e900ed0019964094e7de91f1481989ae1873004565789cbf5dc56c62aedc63f62f3b894c9c6f7788c8ecaadc9bd0e81ad91b2b3569ea12260e93924fdddd3972af5273198f5efda0746219475017557616170e";
    qx  = "5cf02a00d205bdfee2016f7421807fc38ae69e6b7ccd064ee689fc1a94a9f7d2";
    qy  = "ec530ce3cc5c9d1af463f264d685afe2b4db4b5828d7e61b748930f3ce622a85";
    r   = "dc23d130c6117fb5751201455e99f36f59aba1a6a21cf2d0e7481a97451d6693";
    s   = "d6ce7708c18dbf35d4f8aa7240922dc6823f2e7058cbc1484fcad1599db5018c";
    result = false;
  }  ;
  { msg = "df04a346cf4d0e331a6db78cca2d456d31b0a000aa51441defdb97bbeb20b94d8d746429a393ba88840d661615e07def615a342abedfa4ce912e562af714959896858af817317a840dcff85a057bb91a3c2bf90105500362754a6dd321cdd86128cfc5f04667b57aa78c112411e42da304f1012d48cd6a7052d7de44ebcc01de";
    qx  = "2ddfd145767883ffbb0ac003ab4a44346d08fa2570b3120dcce94562422244cb";
    qy  = "5f70c7d11ac2b7a435ccfbbae02c3df1ea6b532cc0e9db74f93fffca7c6f9a64";
    r   = "9913111cff6f20c5bf453a99cd2c2019a4e749a49724a08774d14e4c113edda8";
    s   = "9467cd4cd21ecb56b0cab0a9a453b43386845459127a952421f5c6382866c5cc";
    result = false;
  };
  { msg = "e1130af6a38ccb412a9c8d13e15dbfc9e69a16385af3c3f1e5da954fd5e7c45fd75e2b8c36699228e92840c0562fbf3772f07e17f1add56588dd45f7450e1217ad239922dd9c32695dc71ff2424ca0dec1321aa47064a044b7fe3c2b97d03ce470a592304c5ef21eed9f93da56bb232d1eeb0035f9bf0dfafdcc4606272b20a3";
    qx  = "e424dc61d4bb3cb7ef4344a7f8957a0c5134e16f7a67c074f82e6e12f49abf3c";
    qy  = "970eed7aa2bc48651545949de1dddaf0127e5965ac85d1243d6f60e7dfaee927";
    r   = "bf96b99aa49c705c910be33142017c642ff540c76349b9dab72f981fd9347f4f";
    s   = "17c55095819089c2e03b9cd415abdf12444e323075d98f31920b9e0f57ec871c";
    result = true;
  };
  { msg = "73c5f6a67456ae48209b5f85d1e7de7758bf235300c6ae2bdceb1dcb27a7730fb68c950b7fcada0ecc4661d3578230f225a875e69aaa17f1e71c6be5c831f22663bac63d0c7a9635edb0043ff8c6f26470f02a7bc56556f1437f06dfa27b487a6c4290d8bad38d4879b334e341ba092dde4e4ae694a9c09302e2dbf443581c08";
    qx  = "e0fc6a6f50e1c57475673ee54e3a57f9a49f3328e743bf52f335e3eeaa3d2864";
    qy  = "7f59d689c91e463607d9194d99faf316e25432870816dde63f5d4b373f12f22a";
    r   = "1d75830cd36f4c9aa181b2c4221e87f176b7f05b7c87824e82e396c88315c407";
    s   = "cb2acb01dac96efc53a32d4a0d85d0c2e48955214783ecf50a4f0414a319c05a";
    result = true;
  };
  { msg = "666036d9b4a2426ed6585a4e0fd931a8761451d29ab04bd7dc6d0c5b9e38e6c2b263ff6cb837bd04399de3d757c6c7005f6d7a987063cf6d7e8cb38a4bf0d74a282572bd01d0f41e3fd066e3021575f0fa04f27b700d5b7ddddf50965993c3f9c7118ed78888da7cb221849b3260592b8e632d7c51e935a0ceae15207bedd548";
    qx  = "a849bef575cac3c6920fbce675c3b787136209f855de19ffe2e8d29b31a5ad86";
    qy  = "bf5fe4f7858f9b805bd8dcc05ad5e7fb889de2f822f3d8b41694e6c55c16b471";
    r   = "25acc3aa9d9e84c7abf08f73fa4195acc506491d6fc37cb9074528a7db87b9d6";
    s   = "9b21d5b5259ed3f2ef07dfec6cc90d3a37855d1ce122a85ba6a333f307d31537";
    result = false;
  };
  { msg = "7e80436bce57339ce8da1b5660149a20240b146d108deef3ec5da4ae256f8f894edcbbc57b34ce37089c0daa17f0c46cd82b5a1599314fd79d2fd2f446bd5a25b8e32fcf05b76d644573a6df4ad1dfea707b479d97237a346f1ec632ea5660efb57e8717a8628d7f82af50a4e84b11f21bdff6839196a880ae20b2a0918d58cd";
    qx  = "3dfb6f40f2471b29b77fdccba72d37c21bba019efa40c1c8f91ec405d7dcc5df";
    qy  = "f22f953f1e395a52ead7f3ae3fc47451b438117b1e04d613bc8555b7d6e6d1bb";
    r   = "548886278e5ec26bed811dbb72db1e154b6f17be70deb1b210107decb1ec2a5a";
    s   = "e93bfebd2f14f3d827ca32b464be6e69187f5edbd52def4f96599c37d58eee75";
    result = false;
  };
  { msg = "1669bfb657fdc62c3ddd63269787fc1c969f1850fb04c933dda063ef74a56ce13e3a649700820f0061efabf849a85d474326c8a541d99830eea8131eaea584f22d88c353965dabcdc4bf6b55949fd529507dfb803ab6b480cd73ca0ba00ca19c438849e2cea262a1c57d8f81cd257fb58e19dec7904da97d8386e87b84948169";
    qx  = "69b7667056e1e11d6caf6e45643f8b21e7a4bebda463c7fdbc13bc98efbd0214";
    qy  = "d3f9b12eb46c7c6fda0da3fc85bc1fd831557f9abc902a3be3cb3e8be7d1aa2f";
    r   = "288f7a1cd391842cce21f00e6f15471c04dc182fe4b14d92dc18910879799790";
    s   = "247b3c4e89a3bcadfea73c7bfd361def43715fa382b8c3edf4ae15d6e55e9979";
    result = false;
  };
  { msg = "3fe60dd9ad6caccf5a6f583b3ae65953563446c4510b70da115ffaa0ba04c076115c7043ab8733403cd69c7d14c212c655c07b43a7c71b9a4cffe22c2684788ec6870dc2013f269172c822256f9e7cc674791bf2d8486c0f5684283e1649576efc982ede17c7b74b214754d70402fb4bb45ad086cf2cf76b3d63f7fce39ac970";
    qx  = "bf02cbcf6d8cc26e91766d8af0b164fc5968535e84c158eb3bc4e2d79c3cc682";
    qy  = "069ba6cb06b49d60812066afa16ecf7b51352f2c03bd93ec220822b1f3dfba03";
    r   = "f5acb06c59c2b4927fb852faa07faf4b1852bbb5d06840935e849c4d293d1bad";
    s   = "049dab79c89cc02f1484c437f523e080a75f134917fda752f2d5ca397addfe5d";
    result = false;
  };
  { msg = "983a71b9994d95e876d84d28946a041f8f0a3f544cfcc055496580f1dfd4e312a2ad418fe69dbc61db230cc0c0ed97e360abab7d6ff4b81ee970a7e97466acfd9644f828ffec538abc383d0e92326d1c88c55e1f46a668a039beaa1be631a89129938c00a81a3ae46d4aecbf9707f764dbaccea3ef7665e4c4307fa0b0a3075c";
    qx  = "224a4d65b958f6d6afb2904863efd2a734b31798884801fcab5a590f4d6da9de";
    qy  = "178d51fddada62806f097aa615d33b8f2404e6b1479f5fd4859d595734d6d2b9";
    r   = "87b93ee2fecfda54deb8dff8e426f3c72c8864991f8ec2b3205bb3b416de93d2";
    s   = "4044a24df85be0cc76f21a4430b75b8e77b932a87f51e4eccbc45c263ebf8f66";
    result = false;
  };
  { msg = "4a8c071ac4fd0d52faa407b0fe5dab759f7394a5832127f2a3498f34aac287339e043b4ffa79528faf199dc917f7b066ad65505dab0e11e6948515052ce20cfdb892ffb8aa9bf3f1aa5be30a5bbe85823bddf70b39fd7ebd4a93a2f75472c1d4f606247a9821f1a8c45a6cb80545de2e0c6c0174e2392088c754e9c8443eb5af";
    qx  = "43691c7795a57ead8c5c68536fe934538d46f12889680a9cb6d055a066228369";
    qy  = "f8790110b3c3b281aa1eae037d4f1234aff587d903d93ba3af225c27ddc9ccac";
    r   = "8acd62e8c262fa50dd9840480969f4ef70f218ebf8ef9584f199031132c6b1ce";
    s   = "cfca7ed3d4347fb2a29e526b43c348ae1ce6c60d44f3191b6d8ea3a2d9c92154";
    result = false;
  };
  { msg = "0a3a12c3084c865daf1d302c78215d39bfe0b8bf28272b3c0b74beb4b7409db0718239de700785581514321c6440a4bbaea4c76fa47401e151e68cb6c29017f0bce4631290af5ea5e2bf3ed742ae110b04ade83a5dbd7358f29a85938e23d87ac8233072b79c94670ff0959f9c7f4517862ff829452096c78f5f2e9a7e4e9216";
    qx  = "9157dbfcf8cf385f5bb1568ad5c6e2a8652ba6dfc63bc1753edf5268cb7eb596";
    qy  = "972570f4313d47fc96f7c02d5594d77d46f91e949808825b3d31f029e8296405";
    r   = "dfaea6f297fa320b707866125c2a7d5d515b51a503bee817de9faa343cc48eeb";
    s   = "8f780ad713f9c3e5a4f7fa4c519833dfefc6a7432389b1e4af463961f09764f2";
    result = false;
  };
  { msg = "785d07a3c54f63dca11f5d1a5f496ee2c2f9288e55007e666c78b007d95cc28581dce51f490b30fa73dc9e2d45d075d7e3a95fb8a9e1465ad191904124160b7c60fa720ef4ef1c5d2998f40570ae2a870ef3e894c2bc617d8a1dc85c3c55774928c38789b4e661349d3f84d2441a3b856a76949b9f1f80bc161648a1cad5588e";
    qx  = "072b10c081a4c1713a294f248aef850e297991aca47fa96a7470abe3b8acfdda";
    qy  = "9581145cca04a0fb94cedce752c8f0370861916d2a94e7c647c5373ce6a4c8f5";
    r   = "09f5483eccec80f9d104815a1be9cc1a8e5b12b6eb482a65c6907b7480cf4f19";
    s   = "a4f90e560c5e4eb8696cb276e5165b6a9d486345dedfb094a76e8442d026378d";
    result = false;
  };
  { msg = "76f987ec5448dd72219bd30bf6b66b0775c80b394851a43ff1f537f140a6e7229ef8cd72ad58b1d2d20298539d6347dd5598812bc65323aceaf05228f738b5ad3e8d9fe4100fd767c2f098c77cb99c2992843ba3eed91d32444f3b6db6cd212dd4e5609548f4bb62812a920f6e2bf1581be1ebeebdd06ec4e971862cc42055ca";
    qx  = "09308ea5bfad6e5adf408634b3d5ce9240d35442f7fe116452aaec0d25be8c24";
    qy  = "f40c93e023ef494b1c3079b2d10ef67f3170740495ce2cc57f8ee4b0618b8ee5";
    r   = "5cc8aa7c35743ec0c23dde88dabd5e4fcd0192d2116f6926fef788cddb754e73";
    s   = "9c9c045ebaa1b828c32f82ace0d18daebf5e156eb7cbfdc1eff4399a8a900ae7";
    result = false;
  };
  { msg = "60cd64b2cd2be6c33859b94875120361a24085f3765cb8b2bf11e026fa9d8855dbe435acf7882e84f3c7857f96e2baab4d9afe4588e4a82e17a78827bfdb5ddbd1c211fbc2e6d884cddd7cb9d90d5bf4a7311b83f352508033812c776a0e00c003c7e0d628e50736c7512df0acfa9f2320bd102229f46495ae6d0857cc452a84";
    qx  = "2d98ea01f754d34bbc3003df5050200abf445ec728556d7ed7d5c54c55552b6d";
    qy  = "9b52672742d637a32add056dfd6d8792f2a33c2e69dafabea09b960bc61e230a";
    r   = "06108e525f845d0155bf60193222b3219c98e3d49424c2fb2a0987f825c17959";
    s   = "62b5cdd591e5b507e560167ba8f6f7cda74673eb315680cb89ccbc4eec477dce";
    result = true;
  };  
]


 let sigver_vectors_sha2_384 : list vec_SigVer = [
  { msg = "fe9838f007bdc6afcd626974fcc6833f06b6fd970427b962d75c2aeadbef386bec8d018106197fe2547d2af02e7a7949965d5fbc4c5db909a95b9858426a33c080b0b25dae8b56c5cbc6c4eec3dbd81635c79457eaef4fab39e662a1d05b2481eda8c1074ae2d1704c8a3f769686a1f965ef3c87602efc288c7f9ff8cd5e22a4";
    qx  = "40ded13dbbe72c629c38f07f7f95cf75a50e2a524897604c84fafde5e4cafb9f";
    qy  = "a17202e92d7d6a37c438779349fd79567d75a40ef22b7d09ca21ccf4aec9a66c";
    r   = "be34730c31730b4e412e6c52c23edbd36583ace2102b39afa11d24b6848cb77f";
    s   = "03655202d5fd8c9e3ae971b6f080640c406112fd95e7015874e9b6ee77752b10";
    result = false;
  };
  { msg = "b69043b9b331da392b5dd689142dfc72324265da08f14abcedf03ad8263e6bdccbc75098a2700bbba1979de84c8f12891aa0d000f8a1abad7dde4981533f21da59cc80d9cf94517f3b61d1a7d9eecb2fcf052e1fc9e7188c031b86305e4a436a37948071f046e306befb8511dc03a53dc8769a90a86e9b4fdbf05dcdfa35ab73";
    qx  = "1f80e19ffeb51dd74f1c397ac3dfd3415ab16ebd0847ed119e6c3b15a1a884b8";
    qy  = "9b395787371dbfb55d1347d7bed1c261d2908121fb78de1d1bf2d00666a62aed";
    r   = "249ca2c3eb6e04ac57334c2f75dc5e658bbb485bf187100774f5099dd13ef707";
    s   = "97363a05202b602d13166346694e38135bbce025be94950e9233f4c8013bf5bf";
    result = false;
  };
    { msg = "d2fcaaede8b879c064b0aa46e68efc278a469b80a7f7e1939ec2ebc96c76206f23395967279c181fea157ebb79dfadc68e31345f07f13305c80de0d85e4330d3a45f957c5c2526b945838ce5a9c2844b6b2a665c0f70b748b1213a8cf20ba5dbdf8cab231f433da522104a5cd027d3e36bb373c4ed404d9af0cbec6f85ec2193";
    qx  = "ce4dcfa7384c83443ace0fb82c4ac1adfa100a9b2c7bf09f093f8b6d084e50c2";
    qy  = "d98ae7b91abee648d0bfde192703741ac21daad7262af418b50e406d825eb0d6";
    r   = "597e1e04d93a6b444ccc447a48651f17657ff43fb65fe94461d2bf816b01af40";
    s   = "359fe3817963548e676d6da34c2d0866aa42499237b682002889eaf8893814d2";
    result = true;
  };
  { msg = "06cd86481865181cef7acdc3202824970ec2d97662b519c4b588dc9e51617c068282b1a11a15bf7efc4858a2f37a3d74b05fb5790eb68338c8009b4da9b4270514d387a2e016a99ee109841e884a7909504ef31a5454e214663f830f23a5a76f91402fca5f5d61699fa874597bdbfb1ecff8f07ddbd07ef61e97d0d5262ef314";
    qx  = "1b677f535ac69d1acd4592c0d12fac13c9131e5a6f8ab4f9d0afdcb3a3f327e0";
    qy  = "5dca2c73ec89e58ef8267cba2bb5eb0f551f412f9dc087c1a6944f0ce475277a";
    r   = "df0b0cd76d2555d4c38b3d70bfdf964884d0beeb9f74385f0893e87d20c9642d";
    s   = "128299aabf1f5496112be1fe04365f5f8215b08a040abdfeca4626f4d15c005b";
    result = false;
  };
  { msg = "59ad297397f3503604a4a2d098a4f00a368ad95c6101b3d38f9d49d908776c5a6c8654b006adb7939ffb6c30afa325b54185d82c3cc0d836850dce54d3408b257c3a961d11fafe2b74ba8bddfc1102fa656d1028baf94c38340c26a11e992aab71ce3732271b767358671b25225926f3a4b9ec5f82c059f0c7d1446d5d9e4251";
    qx  = "7ffc2853f3e17887dda13b0eb43f183ce50a5ac0f8bba75fb1921172484f9b94";
    qy  = "4cc523d14192f80bd5b27d30b3b41e064da87bfbae15572dd382b9a176c123a2";
    r   = "3156176d52eb26f9391229de4251993a41b8172f78970bb70e32a245be4bb653";
    s   = "62827a29e12d2f29b00fb2d02dd5f2d5412e17a4455f4431a5c996881fdfc0ee";
    result = false;
  };
  { msg = "8215daca87e689a20392646a6511bb7b5a82d2d995ca9de89bd9d9c0b11464b7cb1e4e9a31e3e01ad8c2cd613d5a2cb44a2a8df6899fce4c282dea1e41af0df6c36be1f320036567f8d0d32aaa79c95fe53b16668f7e1a9e5d7d039ea260fd03711b7d1c177355fc52244d49ca5b238556a5541349014683cb7da326f443b752";
    qx  = "5569f76dc94243cde819fb6fc85144ec67e2b5d49539f62e24d406d1b68f0058";
    qy  = "1208c38dbe25870deab53c486f793a1e250c9d1b8e7c147ea68b71196c440730";
    r   = "706f2ba4025e7c06b66d6369a3f93b2fec46c51eceff42a158f7431919506cfb";
    s   = "b4e75ac34a96393237fc4337789e37168d79382705b248051c9c72bcbac5f516";
    result = false;
  };
  { msg = "a996b1fb800f692517a2eb80e837233193dd3e82484d3f49bd19ee0db8f7b440876b07e384c90aa8b9f7b6603ca0b5a4e06c1da0edb974a2fb9b6e7c720ddf3e5c0e314c2d189402903c08c0836776c361a284db887ebcc33e615de9720b01dadade585eef687b3346468bdafb490e56d657a9e7d44d92014069005a36c1cf63";
    qx  = "e4b470c65b2c04db060d7105ec6911589863d3c7f7ce48726ba3f369ea3467e8";
    qy  = "44c38d3ae098de05f5915a5868c17fee296a6e150beb1f000df5f3bec8fc4532";
    r   = "c9c347ee5717e4c759ddaf09e86f4e1db2c8658593177cfda4e6514b5e3ecb87";
    s   = "baae01e9e44a7b04d69c8eaaed77c9e3a36ce8962f95cc50a0db146b4e49eb40";
    result = false;
  };
  { msg = "1a6e49a377a08e992353d6acc557b687b1b69a41d83d43a75fadb97b8c928cfebadebaaf99ea7fb13148807f56ea17384a7912e578e62b1b009fefb2aafca5ac85539433619b286f10643a56f8dfa47ba4d01c02510deaec18029ea6b9682022b139dcb70814164c4c90ec717ad9d925485398531cdd5992a2524498b337f97d";
    qx  = "96050c5fa2ddd1b2e5451d89ee74a0b7b54347364ddc0231715a6ef1146fe8dc";
    qy  = "e0888a9e78aeea87f6e1e9002b2651169f36c4ee53013cfc8c9912b7fd504858";
    r   = "2353d6cd3c21b8ea7dbc1cd940519812dbe365a3b15cd6aebba9d11cf269867a";
    s   = "85f560273cd9e82e6801e4cb1c8cd29cdac34a020da211d77453756b604b8fa7";
    result = true;
  };
  { msg = "3e14f737c913931bc82764ebc440b12e3ce1ffe0f858c7b8f1cbd30fbbb1644fa59be1d2cca5f64a6d7dc5ed5c4420f39227516ae8eb3019ef86274d0e4d06cde7bf5e5c413243dfc421d9f141762109810e6b6a451eeb4bd8d4be1ff111426d7e44d0a916b4fe3db3594d8dd01ae90feecf8f1e230b574180cd0b8d43a3d33b";
    qx  = "0c07bb79f44012299fbfd5a0f31397aaf7d757f8a38437407c1b09271c6551a0";
    qy  = "84fe7846d5d403dc92c0091fbd39f3c5cbca3f94c10b5cae44e2e96562131b13";
    r   = "49e9425f82d0a8c503009cead24e12adc9d48a08594094ca4f6d13ad1e3c571d";
    s   = "1f1b70aaa30a8ff639aa0935944e9b88326a213ab8fce5194c1a9dec070eb433";
    result = false;
  };
  { msg = "4000106127a72746db77957cbc6bfd84ae3d1d63b8190087637e93689841331e2adc1930d6df4302935f4520bbee513505cdcfca99ebc6f83af7b23b0f2e7f7defba614022ceeae9c6886e8b13f7ea253a307ac301f3536720cbe3de82ba3e98310361b61801a8304ffc91ff774948e33176ddcddf1b76437b3f02c910578d46";
    qx  = "71db1de1a1f38f356c91feaff5cfe395d1a5b9d23cf6aa19f38ae0bcc90a486d";
    qy  = "ecdd6ffb174a50f1cc792985c2f9608c399c98b8a64a69d2b5b7cdd9241f67e2";
    r   = "b0443b33a6f249470d2f943675009d21b9ccbead1525ae57815df86bb20470bf";
    s   = "316dbee27d998e09128539c269e297ac8f34b9ef8249a0619168c3495c5c1198";
    result = false;
  };
  { msg = "b42e547d0e7ddd5e1069bb2d158a5b4d5d9c4310942a1bfd09490311a6e684bd3c29b0dcef86a9788b4b26fed7863f3d5e5439796b5b5ffe7aa2545d0f518ad020689ca21230f3a59e7f8cca465fe21df511e78d215fa805f5f0f88938e9d198515e6b9c819930755c6c6aea5114cd2904607243051c09dd7a147756cbc204a5";
    qx  = "8219b225aa15472262c648cac8de9aad4173d17a231ba24352a5a1c4eea70fad";
    qy  = "0fee2b08ad39fbf0db0016ef2896ca99adc07efc8c415f640f3720498be26037";
    r   = "134fb689101aaad3954de2819d9fbd12072fe2bc36f496bbf0d13fa72114ab96";
    s   = "e65c232bd915b59e087e7fd5ec90bf636cfa80526345c79a0adfd75003045d6f";
    result = false;
  };
  { msg = "aa563223a7d5201febdf13cab80a03dce6077c26e751bc98a941196a28848abc495e0324013c9a2094fb15dc65d100c3e8a136a52c1780b395f42588900b641b6d4361432e2173195a2f60189f3fcc85f4e9659cae52576f20d1852d43c2b400deea3144c8e870e1906d677425d8c85037c7a42a9d249b2da4b516e04476bd45";
    qx  = "c934195de33b60cf00461fc3c45dad068e9f5f7af5c7fa78591e95aeb04e2617";
    qy  = "b588dd5f9965fdaa523b475c2812c251bc6973e2df21d9beaace976abf5728cb";
    r   = "71f302440eb4ed2a939b69e33e905e6fdc545c743458d38f7e1a1d456e35f389";
    s   = "54eaa0eb9cd7503b19a9658f0a04955d9f0ab20ebc8a0877e33c89ee88ad068f";
    result = false;
  };
  { msg = "98e4babf890f52e5a04bd2a7d79bf0ae9a71967847347d87f29fb3997454c73c7979d15b5c4f4205ec3de7835d1885fb7abcf8dcde94baf08b1d691a0c74845317286540e8c9d378fefaa4762c302492f51023c0d7adbb1cc90b7b0335f11203664e71fea621bc2f59d2dbd0ee76d6597ec75510de59b6d25fa6750a71c59435";
    qx  = "9e1adcd48e2e3f0e4c213501808228e587c40558f52bb54ddbb6102d4048ea92";
    qy  = "34eff98704790938e7e0bdf87ae39807a6b77dfdc9ecdfe6dd0f241abae1aeb2";
    r   = "ce4f0d7480522c8dd1b02dd0eb382f22406642f038c1ede9411883d72b3e7ed0";
    s   = "8546e1ee3b77f9927cdaccbc2f1cf19d6b5576b0f738bb1b86a0c66b39ca56fb";
    result = false;
  };
  { msg = "bb6b03ad60d6ddbf0c4d17246206e61c886f916d252bb4608149da49cef9033484080e861f91bb2400baa0cd6c5d90c2f275e2fabc12d83847f7a1c3ff0eb40c8a3dd83d07d194ba3797d27238415a2f358d7292a1991af687bcb977486980f9138b3140321485638ac7bd22ecda00ffe5009b83b90397eff24ecf22c5495d67";
    qx  = "93edbecb0b019c2cc03060f54cb4904b920fdb34eb83badd752be9443036ae13";
    qy  = "b494e9295e080a9080fe7e73249b3a5904aa84e1c028121eecd3e2cf1a55f598";
    r   = "eec2986d47b71995892b0915d3d5becc4dcb2ab55206d772e0189541b2184ddf";
    s   = "8a6c1edeb6452627ad27c8319599c54ac44cdd831ea66f13f49d90affe6ad45b";
    result = true;
  };
  { msg = "33a5d489f671f396c776bc1acf193bc9a74306f4692dd8e05bcdfe28fdefbd5c09b831c204a1dec81d8e3541f324f7b474d692789013bb1eca066f82fbf3f1cf3ba64e9d8963e9ecc180b9251919e2e8a1ab05847a0d76ff67a47c00e170e38e5b319a56f59cc51038f90961ea27a9a7eb292a0a1aa2f4972568669246907a35";
    qx  = "3205bae876f9bd50b0713959e72457165e826cbbe3895d67320909daa48b0ebc";
    qy  = "d1592562273e5e0f57bbfb92cedd9af7f133255684ee050af9b6f02019bbcafa";
    r   = "0124f3f1c61ec458561a4eaa6c155bd29e59703d14556324924683db3a4cf43b";
    s   = "688a5c5fc0c7ba92210c50cce5b512a468a880e05acc21ca56571d89f45f603a";
    result = false;
  };  
]


 let sigver_vectors_sha2_512 : list vec_SigVer = [
  { msg = "273b063224ab48a1bf6c7efc93429d1f89de48fc4a4fa3ffe7a49ebba1a58ff5d208a9e4bff27b418252526243ba042d1605b6df3c2ec916ceef027853a41137f7bfb6fc63844de95f58e82b9ad2565f1367d2c69bd29100f6db21a8ab7ab58affd1661add0322bd915721378df9fa233ef0b7e0a0a85be31689e21891ec8977";
    qx  = "484e31e69ef70bb8527853c22c6b6b4cd2a51311dde66c7b63f097dbb6ab27bf";
    qy  = "e1ff8177f4061d4fbbacbbc70519f0fc8c8b6053d72af0fe4f048d615004f74e";
    r   = "91a303d8fe3ab4176070f6406267f6b79bfe5eb5f62ae6aeb374d90667858518";
    s   = "e152119cefa26826ea07ec40a428869132d70812c5578c5a260e48d6800e046a";
    result = false;
  };
  { msg = "d64ea1a768b0de29ab018ae93baa645d078c70a2f7aa4acd4ae7526538ebd5f697a11927cfd0ddc9187c095f14ad30544cb63ede9353af8b23c18ce22843881fe2d7bde748fc69085921677858d87d2dc3e244f6c7e2c2b2bd791f450dfdd4ff0ddd35ab2ada4f1b90ab16ef2bf63b3fbe88ce8a5d5bb85430740d3744849c13";
    qx  = "8b75fc0129c9a78f8395c63ae9694b05cd6950665cf5da7d66118de451422624";
    qy  = "b394171981d4896d6e1b4ef2336d9befe7d27e1eb87f1c14b8ddda622af379dc";
    r   = "17e298e67ad2af76f6892fdcead00a88256573868f79dc74431b55103058f0b0";
    s   = "881328cd91e43d30133f6e471e0b9b04353b17893fb7614fd7333d812a3df6b4";
    result = false;
  };
    { msg = "1db85445c9d8d1478a97dd9d6ffbf11ebcd2114d2ed4e8b6811171d947e7d4daedea35af6177debe2ef6d93f94ff9d770b45d458e91deb4eef59856425d7b00291aff9b6c9fa02375ec1a06f71f7548721790023301cf6ac7fee1d451228106ef4472681e652c8cd59b15d6d16f1e13440d888e265817cb4a654f7246e0980df";
    qx  = "76e51086e078b2b116fd1e9c6fa3d53f675ae40252fb9f0cc62817bd9ce8831d";
    qy  = "ca7e609a0b1d14b7c9249b53da0b2050450e2a25cb6c8f81c5311974a7efb576";
    r   = "23b653faaa7d4552388771931803ce939dd5ee62d3fa72b019be1b2272c85592";
    s   = "a03c6f5c54a10861d6b8922821708e9306fd6d5d10d566845a106539cbf4fadd";
    result = false;
  };
  { msg = "918d9f420e927b3e0a55d276b8b40d8a2c5df748727ff72a438c7e6593f542274050dce727980d3ef90c8aa5c13d53f1e8d631ebb650dee11b94902bbd7c92b8186af9039c56c43f3110697792c8cd1614166f06d09cdb58dab168cc3680a8473b1a623bf85dba855eace579d9410d2c4ca5ede6dc1e3db81e233c34ae922f49";
    qx  = "bc7c8e09bd093468f706740a4130c544374fdc924a535ef02e9d3be6c6d3bbfa";
    qy  = "af3f813ae6646f5b6dbfb0f261fd42537705c800bb1647386343428a9f2e10fc";
    r   = "6bd7ce95af25abfbf14aef4b17392f1da877ab562eca38d785fe39682e9c9324";
    s   = "6688bea20c87bab34d420642da9bdd4c69456bdec50835887367bb4fb7cd8650";
    result = false;
  };
  { msg = "6e2932153301a4eef680e6428929adae988c108d668a31ff55d0489947d75ff81a46bf89e84d6401f023be6e87688fbcd784d785ca846735524acb52d00452c84040a479e7cc330936441d93bbe722a9432a6e1db112b5c9403b10272cb1347fd619d463f7a9d223ad76fde06d8a6883500fb843235abff98e241bdfb5538c3e";
    qx  = "9cb0cf69303dafc761d4e4687b4ecf039e6d34ab964af80810d8d558a4a8d6f7";
    qy  = "2d51233a1788920a86ee08a1962c79efa317fb7879e297dad2146db995fa1c78";
    r   = "4b9f91e4285287261a1d1c923cf619cd52c175cfe7f1be60a5258c610348ba3d";
    s   = "28c45f901d71c41b298638ec0d6a85d7fcb0c33bbfec5a9c810846b639289a84";
    result = true;
  };
  { msg = "2f48ec387f181035b350772e27f478ae6ec7487923692fae217e0f8636acd062a6ac39f7435f27a0ebcfd8187a91ef00fb68d106b8da4a1dedc5a40a4fae709e92b00fcc218de76417d75185e59dff76ec1543fb429d87c2ca8134ff5ae9b45456cad93fc67223c68293231395287dc0b756355660721a1f5df83bf5bcb8456e";
    qx  = "e31096c2d512fbf84f81e9bdb16f33121702897605b43a3db546f8fb695b5f6f";
    qy  = "6fbec6a04a8c59d61c900a851d8bf8522187d3ec2637b10fa8f377689e086bba";
    r   = "1b244c21c08c0c0a10477fb7a21382d405b95c755088292859ca0e71bab68361";
    s   = "852f4cbfd346e90f404e1dd5c4b2c1debca3ea1abefe8400685d703aea6c5c7f";
    result = false;
  };
  { msg = "fd2e5de421ee46c9fe6290a33f95b394bd5b7762f23178f7f6834f1f056fa9a8831446403c098ff4dd764173f974be4c89d376119613a4a1890f6fc2ddff862bda292dd49f5410d9b1cfe1d97ef4582b6152494372fc083885f540c01f86d780e6f3e75a954af2190fdae9604e3f8ab32ab0292dc0d790bd2627e37b4b4885df";
    qx  = "633c2ee5630b62c9ce839efd4d485a6d35e8b9430d264ffe501d28dbace79123";
    qy  = "4b668a1a6d1a25b089f75c2bd8d8c6a9a14fe7b729f45a82565da2e866e2c490";
    r   = "bf2111c93ec055a7eda90c106fce494fd866045634fd2aa28d6e018f9106994e";
    s   = "86b0341208a0aa55edecfd272f49cb34408ce54b7febc1d0a1c2ce77ab6988f8";
    result = false;
  };
  { msg = "4bc2d9a898395b12701635f1048fbfd263ec115e4150532b034d59e625238f4ed32619744c612e35ac5a23bee8d5f5651641a492217d305e5051321c273647f14bc7c4afab518554e01c82d6fc1694c8bdbeb326bb607bcaf5436303bc09f64c02c6ec50de409a484f5237f7d34e2651ada7ec429ca3b99dd87c6015d2f4b342";
    qx  = "f78dce40d1cb8c4af2749bf22c6f8a9a470b1e41112796215dd017e57df1b38a";
    qy  = "61b29b0bc03dff7fa00613b4de1e2317cfbf2badd50dee3376c032a887c5b865";
    r   = "4a96169a5dea36a2594011537ee0dc19e8f9f74e82c07434079447155a830152";
    s   = "a204eaa4e97d7553a1521d9f6baadc0b6d6183ba0f385d8593d6ca83607c4d82";
    result = false;
  };
  { msg = "d3356a683417508a9b913643e6ceac1281ef583f428968f9d2b6540a189d7041c477da8d207d0529720f70dab6b0da8c2168837476c1c6b63b517ed3cad48ae331cf716ecf47a0f7d00b57073ac6a4749716d49d80c4d46261d38e2e34b4f43e0f20b280842f6e3ea34fefdddfb9fa2a040ffe915e8784cfdb29b3364a34ca62";
    qx  = "3fcc3b3e1b103fe435ac214c756bdaad309389e1c803e6d84bbbc27039fcf900";
    qy  = "7f09edd1ec87a6d36dc81c1528d52a62776e666c274415a9f441d6a8df6b9237";
    r   = "1cac13f277354456ae67ab09b09e07eb1af2a2bf45108da70f5c8c6a4cbcd538";
    s   = "5d83752e540525602ba7e6fee4d4263f3eda59e67df20aac79ca67e8899fed0d";
    result = false;
  };
  { msg = "d7f5da9f4cf9299b7f86c52b88364ce28fe9ada55dd551a1018790f9e1205e2405ac62429d65093f74ec35a16d9f195c993cd4eb8dc0aa0dabb70a503321d8a9649160d6b3d0a0854bb68c4c39693f592ef5dd478aa2432d0865d87d48b3aea9c7d7d114165c9200e4e8d7bd02a7895ec4418e6f2fed6b244bf66209039e98a9";
    qx  = "5ec702d43a67ada86efbfc136cf16d96078906954a3f1f9e440674cd907e4676";
    qy  = "05a62044fed8470dd4fca38d89d583ce36d50d28b66ab0b51922b21da92c56d9";
    r   = "75f3037298f1457dba55743999976a1c2636b2b8ab2ed3df4736a6d2934acc83";
    s   = "19d43ad168dda1bb8ac423f8f08876515234b3d841e57faef1b5ab27359b27ef";
    result = false;
  };
  { msg = "68f4b444e1cc2025e8ff55e8046ead735e6e317082edf7ce65e83573501cb92c408c1c1c6c4fcca6b96ad34224f17b20be471cc9f4f97f0a5b7bfae9558bdb2ecb6e452bb743603724273d9e8d2ca22afdda35c8a371b28153d772303e4a25dc4f28e9a6dc9635331450f5af290dfa3431c3c08b91d5c97284361c03ec78f1bc";
    qx  = "f63afe99e1b5fc652782f86b59926af22e6072be93390fe41f541204f9c935d1";
    qy  = "f6e19ce5935e336183c21becf66596b8f559d2d02ee282aa87a7d6f936f7260c";
    r   = "cef4831e4515c77ca062282614b54a11b7dc4057e6997685c2fbfa95b392bf72";
    s   = "f20dc01bf38e1344ba675a22239d9893b3a3e33d9a403329a3d21650e9125b75";
    result = true;
  };
  { msg = "e75be05be0aaf70719b488b89aaae9008707ca528994461db7130c4368575a024bf0981c305d61265e8b97599ec35c03badd1256b80d6bf70547ad6089b983e3bcc3481828f3259e43e655e177fc423fd7e066bd3ed68d81df84f773c0f9e5f8bf4469960b8b4d7b2a372fd0edd3521f6be670908f2d90a343f416358ea70e7e";
    qx  = "6d11b09d2767cf8d275faee746c203486259f66dd2bfa3a65c39371a66b23385";
    qy  = "4eb05c73e05261e979182833f20311e5366f72f4b949665ff294f959375534c6";
    r   = "15a697cdb614e11c0810e1e764cd501fcabc70874c957587bc4883d9438e177f";
    s   = "7bf6244f92bc768063cecb5336c8eaacd23db930b28703560f241c7d93950dfd";
    result = false;
  };
  { msg = "0dc4a3eab66bd2e703a8fff566c34d466f9823ae42bd2104f61a6b051c0b017833fcef4d609d137ad97c209c80eebe252857aa7fafc35f16000a2bd4b4be0fa83b6e229eddfd180101f1f40d0453148053d8306833df64d59599b90194b55541d7f22dd589da9f7be519cbbb9db416c71bfe40ec090b5b7a600eec29bfd47306";
    qx  = "f3899caba038efb534c4cea0bd276814ffd80194473c903b81af11c8c05cb6e6";
    qy  = "6ea6b17402fcf2e8e737d11ffc7c2ed3b2d0bc3b8f271a381f4294cff62682c3";
    r   = "57b99380452e1d37b133c49b9ba493dee8630940477ca3351a43d90b99871e6a";
    s   = "df599c3a37105af3ecc159b3b685ccb3e151b7d5cf2d97147974ae71f466b615";
    result = false;
  };
  { msg = "d55e5e124a7217879ca986f285e22ac51940b35959bbf5543104b5547356fd1a0ec37c0a23209004a2ec5bcaf3335bc45e4dc990eacd29b2d9b5cf349c7ba67711356299bceab6f048df761c65f2988803133d6723a2820fefb2654cc7c5f032f833ba78a34d2878c6b0ba654ebe26b110c935abb56024bd5d0f09b367724c07";
    qx  = "1fd6f4b98d0755291e7a230e9f81ecf909e6350aadb08e42a3262ff19200fbd2";
    qy  = "5578fef79bc477acfb8ed0dc10c4f5809c14dc5492405b3792a7940650b305d7";
    r   = "97a99e96e407b3ada2c2dcf9ceeeb984d9a4d0aa66ddf0a74ca23cabfb1566cc";
    s   = "0ecac315dc199cfea3c15348c130924a1f787019fe4cd3ae47ca8b111268754a";
    result = false;
  };
  { msg = "7753c03b4202cb38bc0190a9f931eb31858d705d92d650320ff449fc99167fb3770b764c8988f6b34ac5a3d507a10e0aff7f88293f6a22c7ed8a24248a52dc125e416e158833fc38af29199f8ca4931068d4ccaa87e299e95642068f68c208cb782df13908f950564743ed1692502bafafaff169dc8fe674fb5e4f3ffd578c35";
    qx  = "2dcbd8790cee552e9f18f2b3149a2252dcd58b99ca7dc9680b92c8c43aa33874";
    qy  = "5dbc8bb8813c8e019d80e19acdb0792f537980fecde93db621aaf1f6d0e6ee34";
    r   = "2bdbd8b0d759595662cc10b10236136ef6ce429641f68cf6480f472fcc77bc9f";
    s   = "7e7df0c8b86f7db06caf1610166f7b9c4c75447f991d5aaf4dea720c25985c8c";
    result = true;
  };  
]


 let siggen_vectors_sha2_256 : list vec_SigGen = [
  { msg' = "5905238877c77421f73e43ee3da6f2d9e2ccad5fc942dcec0cbd25482935faaf416983fe165b1a045ee2bcd2e6dca3bdf46c4310a7461f9a37960ca672d3feb5473e253605fb1ddfd28065b53cb5858a8ad28175bf9bd386a5e471ea7a65c17cc934a9d791e91491eb3754d03799790fe2d308d16146d5c9b0d0debd97d79ce8";
    d    = "519b423d715f8b581f4fa8ee59f4771a5b44c8130b4e3eacca54a56dda72b464";
    qx'  = "1ccbe91c075fc7f4f033bfa248db8fccd3565de94bbfb12f3c59ff46c271bf83";
    qy'  = "ce4014c68811f9a21a1fdb2c0e6113e06db7ca93b7404e78dc7ccd5ca89a4ca9";
    k    = "94a1bbb14b906a61a280f245f9e93c7f3b4a6247824f5d33b9670787642a68de";
    r'   = "f3ac8061b514795b8843e3d6629527ed2afd6b1f6a555a7acabb5e6f79c8c2ac";
    s'   = "8bf77819ca05a6b2786c76262bf7371cef97b218e96f175a3ccdda2acc058903";
  };
  { msg' = "c35e2f092553c55772926bdbe87c9796827d17024dbb9233a545366e2e5987dd344deb72df987144b8c6c43bc41b654b94cc856e16b96d7a821c8ec039b503e3d86728c494a967d83011a0e090b5d54cd47f4e366c0912bc808fbb2ea96efac88fb3ebec9342738e225f7c7c2b011ce375b56621a20642b4d36e060db4524af1";
    d    = "0f56db78ca460b055c500064824bed999a25aaf48ebb519ac201537b85479813";
    qx'  = "e266ddfdc12668db30d4ca3e8f7749432c416044f2d2b8c10bf3d4012aeffa8a";
    qy'  = "bfa86404a2e9ffe67d47c587ef7a97a7f456b863b4d02cfc6928973ab5b1cb39";
    k    = "6d3e71882c3b83b156bb14e0ab184aa9fb728068d3ae9fac421187ae0b2f34c6";
    r'   = "976d3a4e9d23326dc0baa9fa560b7c4e53f42864f508483a6473b6a11079b2db";
    s'   = "1b766e9ceb71ba6c01dcd46e0af462cd4cfa652ae5017d4555b8eeefe36e1932";
  }  ;
  { msg' = "3c054e333a94259c36af09ab5b4ff9beb3492f8d5b4282d16801daccb29f70fe61a0b37ffef5c04cd1b70e85b1f549a1c4dc672985e50f43ea037efa9964f096b5f62f7ffdf8d6bfb2cc859558f5a393cb949dbd48f269343b5263dcdb9c556eca074f2e98e6d94c2c29a677afaf806edf79b15a3fcd46e7067b7669f83188ee";
    d    = "e283871239837e13b95f789e6e1af63bf61c918c992e62bca040d64cad1fc2ef";
    qx'  = "74ccd8a62fba0e667c50929a53f78c21b8ff0c3c737b0b40b1750b2302b0bde8";
    qy'  = "29074e21f3a0ef88b9efdf10d06aa4c295cc1671f758ca0e4cd108803d0f2614";
    k    = "ad5e887eb2b380b8d8280ad6e5ff8a60f4d26243e0124c2f31a297b5d0835de2";
    r'   = "35fb60f5ca0f3ca08542fb3cc641c8263a2cab7a90ee6a5e1583fac2bb6f6bd1";
    s'   = "ee59d81bc9db1055cc0ed97b159d8784af04e98511d0a9a407b99bb292572e96";
  };
  { msg' = "0989122410d522af64ceb07da2c865219046b4c3d9d99b01278c07ff63eaf1039cb787ae9e2dd46436cc0415f280c562bebb83a23e639e476a02ec8cff7ea06cd12c86dcc3adefbf1a9e9a9b6646c7599ec631b0da9a60debeb9b3e19324977f3b4f36892c8a38671c8e1cc8e50fcd50f9e51deaf98272f9266fc702e4e57c30";
    d    = "a3d2d3b7596f6592ce98b4bfe10d41837f10027a90d7bb75349490018cf72d07";
    qx'  = "322f80371bf6e044bc49391d97c1714ab87f990b949bc178cb7c43b7c22d89e1";
    qy'  = "3c15d54a5cc6b9f09de8457e873eb3deb1fceb54b0b295da6050294fae7fd999";
    k    = "24fc90e1da13f17ef9fe84cc96b9471ed1aaac17e3a4bae33a115df4e5834f18";
    r'   = "d7c562370af617b581c84a2468cc8bd50bb1cbf322de41b7887ce07c0e5884ca";
    s'   = "b46d9f2d8c4bf83546ff178f1d78937c008d64e8ecc5cbb825cb21d94d670d89";
  };
  { msg' = "dc66e39f9bbfd9865318531ffe9207f934fa615a5b285708a5e9c46b7775150e818d7f24d2a123df3672fff2094e3fd3df6fbe259e3989dd5edfcccbe7d45e26a775a5c4329a084f057c42c13f3248e3fd6f0c76678f890f513c32292dd306eaa84a59abe34b16cb5e38d0e885525d10336ca443e1682aa04a7af832b0eee4e7";
    d    = "53a0e8a8fe93db01e7ae94e1a9882a102ebd079b3a535827d583626c272d280d";
    qx'  = "1bcec4570e1ec2436596b8ded58f60c3b1ebc6a403bc5543040ba82963057244";
    qy'  = "8af62a4c683f096b28558320737bf83b9959a46ad2521004ef74cf85e67494e1";
    k    = "5d833e8d24cc7a402d7ee7ec852a3587cddeb48358cea71b0bedb8fabe84e0c4";
    r'   = "18caaf7b663507a8bcd992b836dec9dc5703c080af5e51dfa3a9a7c387182604";
    s'   = "77c68928ac3b88d985fb43fb615fb7ff45c18ba5c81af796c613dfa98352d29c";
  };
  { msg' = "600974e7d8c5508e2c1aab0783ad0d7c4494ab2b4da265c2fe496421c4df238b0be25f25659157c8a225fb03953607f7df996acfd402f147e37aee2f1693e3bf1c35eab3ae360a2bd91d04622ea47f83d863d2dfecb618e8b8bdc39e17d15d672eee03bb4ce2cc5cf6b217e5faf3f336fdd87d972d3a8b8a593ba85955cc9d71";
    d    = "4af107e8e2194c830ffb712a65511bc9186a133007855b49ab4b3833aefc4a1d";
    qx'  = "a32e50be3dae2c8ba3f5e4bdae14cf7645420d425ead94036c22dd6c4fc59e00";
    qy'  = "d623bf641160c289d6742c6257ae6ba574446dd1d0e74db3aaa80900b78d4ae9";
    k    = "e18f96f84dfa2fd3cdfaec9159d4c338cd54ad314134f0b31e20591fc238d0ab";
    r'   = "8524c5024e2d9a73bde8c72d9129f57873bbad0ed05215a372a84fdbc78f2e68";
    s'   = "d18c2caf3b1072f87064ec5e8953f51301cada03469c640244760328eb5a05cb";
  };
  { msg' = "dfa6cb9b39adda6c74cc8b2a8b53a12c499ab9dee01b4123642b4f11af336a91a5c9ce0520eb2395a6190ecbf6169c4cba81941de8e76c9c908eb843b98ce95e0da29c5d4388040264e05e07030a577cc5d176387154eabae2af52a83e85c61c7c61da930c9b19e45d7e34c8516dc3c238fddd6e450a77455d534c48a152010b";
    d    = "78dfaa09f1076850b3e206e477494cddcfb822aaa0128475053592c48ebaf4ab";
    qx'  = "8bcfe2a721ca6d753968f564ec4315be4857e28bef1908f61a366b1f03c97479";
    qy'  = "0f67576a30b8e20d4232d8530b52fb4c89cbc589ede291e499ddd15fe870ab96";
    k    = "295544dbb2da3da170741c9b2c6551d40af7ed4e891445f11a02b66a5c258a77";
    r'   = "c5a186d72df452015480f7f338970bfe825087f05c0088d95305f87aacc9b254";
    s'   = "84a58f9e9d9e735344b316b1aa1ab5185665b85147dc82d92e969d7bee31ca30";
  };
  { msg' = "51d2547cbff92431174aa7fc7302139519d98071c755ff1c92e4694b58587ea560f72f32fc6dd4dee7d22bb7387381d0256e2862d0644cdf2c277c5d740fa089830eb52bf79d1e75b8596ecf0ea58a0b9df61e0c9754bfcd62efab6ea1bd216bf181c5593da79f10135a9bc6e164f1854bc8859734341aad237ba29a81a3fc8b";
    d    = "80e692e3eb9fcd8c7d44e7de9f7a5952686407f90025a1d87e52c7096a62618a";
    qx'  = "a88bc8430279c8c0400a77d751f26c0abc93e5de4ad9a4166357952fe041e767";
    qy'  = "2d365a1eef25ead579cc9a069b6abc1b16b81c35f18785ce26a10ba6d1381185";
    k    = "7c80fd66d62cc076cef2d030c17c0a69c99611549cb32c4ff662475adbe84b22";
    r'   = "9d0c6afb6df3bced455b459cc21387e14929392664bb8741a3693a1795ca6902";
    s'   = "d7f9ddd191f1f412869429209ee3814c75c72fa46a9cccf804a2f5cc0b7e739f";
  };
  { msg' = "558c2ac13026402bad4a0a83ebc9468e50f7ffab06d6f981e5db1d082098065bcff6f21a7a74558b1e8612914b8b5a0aa28ed5b574c36ac4ea5868432a62bb8ef0695d27c1e3ceaf75c7b251c65ddb268696f07c16d2767973d85beb443f211e6445e7fe5d46f0dce70d58a4cd9fe70688c035688ea8c6baec65a5fc7e2c93e8";
    d    = "5e666c0db0214c3b627a8e48541cc84a8b6fd15f300da4dff5d18aec6c55b881";
    qx'  = "1bc487570f040dc94196c9befe8ab2b6de77208b1f38bdaae28f9645c4d2bc3a";
    qy'  = "ec81602abd8345e71867c8210313737865b8aa186851e1b48eaca140320f5d8f";
    k    = "2e7625a48874d86c9e467f890aaa7cd6ebdf71c0102bfdcfa24565d6af3fdce9";
    r'   = "2f9e2b4e9f747c657f705bffd124ee178bbc5391c86d056717b140c153570fd9";
    s'   = "f5413bfd85949da8d83de83ab0d19b2986613e224d1901d76919de23ccd03199";
  };
  { msg' = "4d55c99ef6bd54621662c3d110c3cb627c03d6311393b264ab97b90a4b15214a5593ba2510a53d63fb34be251facb697c973e11b665cb7920f1684b0031b4dd370cb927ca7168b0bf8ad285e05e9e31e34bc24024739fdc10b78586f29eff94412034e3b606ed850ec2c1900e8e68151fc4aee5adebb066eb6da4eaa5681378e";
    d    = "f73f455271c877c4d5334627e37c278f68d143014b0a05aa62f308b2101c5308";
    qx'  = "b8188bd68701fc396dab53125d4d28ea33a91daf6d21485f4770f6ea8c565dde";
    qy'  = "423f058810f277f8fe076f6db56e9285a1bf2c2a1dae145095edd9c04970bc4a";
    k    = "62f8665fd6e26b3fa069e85281777a9b1f0dfd2c0b9f54a086d0c109ff9fd615";
    r'   = "1cc628533d0004b2b20e7f4baad0b8bb5e0673db159bbccf92491aef61fc9620";
    s'   = "880e0bbf82a8cf818ed46ba03cf0fc6c898e36fca36cc7fdb1d2db7503634430";
  };
  { msg' = "f8248ad47d97c18c984f1f5c10950dc1404713c56b6ea397e01e6dd925e903b4fadfe2c9e877169e71ce3c7fe5ce70ee4255d9cdc26f6943bf48687874de64f6cf30a012512e787b88059bbf561162bdcc23a3742c835ac144cc14167b1bd6727e940540a9c99f3cbb41fb1dcb00d76dda04995847c657f4c19d303eb09eb48a";
    d    = "b20d705d9bd7c2b8dc60393a5357f632990e599a0975573ac67fd89b49187906";
    qx'  = "51f99d2d52d4a6e734484a018b7ca2f895c2929b6754a3a03224d07ae61166ce";
    qy'  = "4737da963c6ef7247fb88d19f9b0c667cac7fe12837fdab88c66f10d3c14cad1";
    k    = "72b656f6b35b9ccbc712c9f1f3b1a14cbbebaec41c4bca8da18f492a062d6f6f";
    r'   = "9886ae46c1415c3bc959e82b760ad760aab66885a84e620aa339fdf102465c42";
    s'   = "2bf3a80bc04faa35ebecc0f4864ac02d349f6f126e0f988501b8d3075409a26c";
  };
  { msg' = "3b6ee2425940b3d240d35b97b6dcd61ed3423d8e71a0ada35d47b322d17b35ea0472f35edd1d252f87b8b65ef4b716669fc9ac28b00d34a9d66ad118c9d94e7f46d0b4f6c2b2d339fd6bcd351241a387cc82609057048c12c4ec3d85c661975c45b300cb96930d89370a327c98b67defaa89497aa8ef994c77f1130f752f94a4";
    d    = "d4234bebfbc821050341a37e1240efe5e33763cbbb2ef76a1c79e24724e5a5e7";
    qx'  = "8fb287f0202ad57ae841aea35f29b2e1d53e196d0ddd9aec24813d64c0922fb7";
    qy'  = "1f6daff1aa2dd2d6d3741623eecb5e7b612997a1039aab2e5cf2de969cfea573";
    k    = "d926fe10f1bfd9855610f4f5a3d666b1a149344057e35537373372ead8b1a778";
    r'   = "490efd106be11fc365c7467eb89b8d39e15d65175356775deab211163c2504cb";
    s'   = "644300fc0da4d40fb8c6ead510d14f0bd4e1321a469e9c0a581464c7186b7aa7";
  };
  { msg' = "c5204b81ec0a4df5b7e9fda3dc245f98082ae7f4efe81998dcaa286bd4507ca840a53d21b01e904f55e38f78c3757d5a5a4a44b1d5d4e480be3afb5b394a5d2840af42b1b4083d40afbfe22d702f370d32dbfd392e128ea4724d66a3701da41ae2f03bb4d91bb946c7969404cb544f71eb7a49eb4c4ec55799bda1eb545143a7";
    d    = "b58f5211dff440626bb56d0ad483193d606cf21f36d9830543327292f4d25d8c";
    qx'  = "68229b48c2fe19d3db034e4c15077eb7471a66031f28a980821873915298ba76";
    qy'  = "303e8ee3742a893f78b810991da697083dd8f11128c47651c27a56740a80c24c";
    k    = "e158bf4a2d19a99149d9cdb879294ccb7aaeae03d75ddd616ef8ae51a6dc1071";
    r'   = "e67a9717ccf96841489d6541f4f6adb12d17b59a6bef847b6183b8fcf16a32eb";
    s'   = "9ae6ba6d637706849a6a9fc388cf0232d85c26ea0d1fe7437adb48de58364333";
  };
  { msg' = "72e81fe221fb402148d8b7ab03549f1180bcc03d41ca59d7653801f0ba853add1f6d29edd7f9abc621b2d548f8dbf8979bd16608d2d8fc3260b4ebc0dd42482481d548c7075711b5759649c41f439fad69954956c9326841ea6492956829f9e0dc789f73633b40f6ac77bcae6dfc7930cfe89e526d1684365c5b0be2437fdb01";
    d    = "54c066711cdb061eda07e5275f7e95a9962c6764b84f6f1f3ab5a588e0a2afb1";
    qx'  = "0a7dbb8bf50cb605eb2268b081f26d6b08e012f952c4b70a5a1e6e7d46af98bb";
    qy'  = "f26dd7d799930062480849962ccf5004edcfd307c044f4e8f667c9baa834eeae";
    k    = "646fe933e96c3b8f9f507498e907fdd201f08478d0202c752a7c2cfebf4d061a";
    r'   = "b53ce4da1aa7c0dc77a1896ab716b921499aed78df725b1504aba1597ba0c64b";
    s'   = "d7c246dc7ad0e67700c373edcfdd1c0a0495fc954549ad579df6ed1438840851";
  };
  { msg' = "21188c3edd5de088dacc1076b9e1bcecd79de1003c2414c3866173054dc82dde85169baa77993adb20c269f60a5226111828578bcc7c29e6e8d2dae81806152c8ba0c6ada1986a1983ebeec1473a73a04795b6319d48662d40881c1723a706f516fe75300f92408aa1dc6ae4288d2046f23c1aa2e54b7fb6448a0da922bd7f34";
    d    = "34fa4682bf6cb5b16783adcd18f0e6879b92185f76d7c920409f904f522db4b1";
    qx'  = "105d22d9c626520faca13e7ced382dcbe93498315f00cc0ac39c4821d0d73737";
    qy'  = "6c47f3cbbfa97dfcebe16270b8c7d5d3a5900b888c42520d751e8faf3b401ef4";
    k    = "a6f463ee72c9492bc792fe98163112837aebd07bab7a84aaed05be64db3086f4";
    r'   = "542c40a18140a6266d6f0286e24e9a7bad7650e72ef0e2131e629c076d962663";
    s'   = "4f7f65305e24a6bbb5cff714ba8f5a2cee5bdc89ba8d75dcbf21966ce38eb66f";
  };  
]


 let siggen_vectors_sha2_384 : list vec_SigGen = [
  { msg' = "e0b8596b375f3306bbc6e77a0b42f7469d7e83635990e74aa6d713594a3a24498feff5006790742d9c2e9b47d714bee932435db747c6e733e3d8de41f2f91311f2e9fd8e025651631ffd84f66732d3473fbd1627e63dc7194048ebec93c95c159b5039ab5e79e42c80b484a943f125de3da1e04e5bf9c16671ad55a1117d3306";
    d    = "b6faf2c8922235c589c27368a3b3e6e2f42eb6073bf9507f19eed0746c79dced";
    qx'  = "e0e7b99bc62d8dd67883e39ed9fa0657789c5ff556cc1fd8dd1e2a55e9e3f243";
    qy'  = "63fbfd0232b95578075c903a4dbf85ad58f8350516e1ec89b0ee1f5e1362da69";
    k    = "9980b9cdfcef3ab8e219b9827ed6afdd4dbf20bd927e9cd01f15762703487007";
    r'   = "f5087878e212b703578f5c66f434883f3ef414dc23e2e8d8ab6a8d159ed5ad83";
    s'   = "306b4c6c20213707982dffbb30fba99b96e792163dd59dbe606e734328dd7c8a";
  };
  { msg' = "099a0131179fff4c6928e49886d2fdb3a9f239b7dd5fa828a52cbbe3fcfabecfbba3e192159b887b5d13aa1e14e6a07ccbb21f6ad8b7e88fee6bea9b86dea40ffb962f38554056fb7c5bb486418915f7e7e9b9033fe3baaf9a069db98bc02fa8af3d3d1859a11375d6f98aa2ce632606d0800dff7f55b40f971a8586ed6b39e9";
    d    = "118958fd0ff0f0b0ed11d3cf8fa664bc17cdb5fed1f4a8fc52d0b1ae30412181";
    qx'  = "afda82260c9f42122a3f11c6058839488f6d7977f6f2a263c67d06e27ea2c355";
    qy'  = "0ae2bbdd2207c590332c5bfeb4c8b5b16622134bd4dc55382ae806435468058b";
    k    = "23129a99eeda3d99a44a5778a46e8e7568b91c31fb7a8628c5d9820d4bed4a6b";
    r'   = "e446600cab1286ebc3bb332012a2f5cc33b0a5ef7291d5a62a84de5969d77946";
    s'   = "cf89b12793ee1792eb26283b48fa0bdcb45ae6f6ad4b02564bf786bb97057d5a";
  };
    { msg' = "0fbc07ea947c946bea26afa10c51511039b94ddbc4e2e4184ca3559260da24a14522d1497ca5e77a5d1a8e86583aeea1f5d4ff9b04a6aa0de79cd88fdb85e01f171143535f2f7c23b050289d7e05cebccdd131888572534bae0061bdcc3015206b9270b0d5af9f1da2f9de91772d178a632c3261a1e7b3fb255608b3801962f9";
    d    = "3e647357cd5b754fad0fdb876eaf9b1abd7b60536f383c81ce5745ec80826431";
    qx'  = "702b2c94d039e590dd5c8f9736e753cf5824aacf33ee3de74fe1f5f7c858d5ed";
    qy'  = "0c28894e907af99fb0d18c9e98f19ac80dd77abfa4bebe45055c0857b82a0f4d";
    k    = "9beab7722f0bcb468e5f234e074170a60225255de494108459abdf603c6e8b35";
    r'   = "c4021fb7185a07096547af1fb06932e37cf8bd90cf593dea48d48614fa237e5e";
    s'   = "7fb45d09e2172bec8d3e330aa06c43fbb5f625525485234e7714b7f6e92ba8f1";
  };
  { msg' = "1e38d750d936d8522e9db1873fb4996bef97f8da3c6674a1223d29263f1234a90b751785316444e9ba698bc8ab6cd010638d182c9adad4e334b2bd7529f0ae8e9a52ad60f59804b2d780ed52bdd33b0bf5400147c28b4304e5e3434505ae7ce30d4b239e7e6f0ecf058badd5b388eddbad64d24d2430dd04b4ddee98f972988f";
    d    = "76c17c2efc99891f3697ba4d71850e5816a1b65562cc39a13da4b6da9051b0fd";
    qx'  = "d12512e934c367e4c4384dbd010e93416840288a0ba00b299b4e7c0d91578b57";
    qy'  = "ebf8835661d9b578f18d14ae4acf9c357c0dc8b7112fc32824a685ed72754e23";
    k    = "77cffa6f9a73904306f9fcd3f6bbb37f52d71e39931bb4aec28f9b076e436ccf";
    r'   = "4d5a9d95b0f09ce8704b0f457b39059ee606092310df65d3f8ae7a2a424cf232";
    s'   = "7d3c014ca470a73cef1d1da86f2a541148ad542fbccaf9149d1b0b030441a7eb";
  };
  { msg' = "abcf0e0f046b2e0672d1cc6c0a114905627cbbdefdf9752f0c31660aa95f2d0ede72d17919a9e9b1add3213164e0c9b5ae3c76f1a2f79d3eeb444e6741521019d8bd5ca391b28c1063347f07afcfbb705be4b52261c19ebaf1d6f054a74d86fb5d091fa7f229450996b76f0ada5f977b09b58488eebfb5f5e9539a8fd89662ab";
    d    = "67b9dea6a575b5103999efffce29cca688c781782a41129fdecbce76608174de";
    qx'  = "b4238b029fc0b7d9a5286d8c29b6f3d5a569e9108d44d889cd795c4a385905be";
    qy'  = "8cb3fff8f6cca7187c6a9ad0a2b1d9f40ae01b32a7e8f8c4ca75d71a1fffb309";
    k    = "d02617f26ede3584f0afcfc89554cdfb2ae188c192092fdde3436335fafe43f1";
    r'   = "26fd9147d0c86440689ff2d75569795650140506970791c90ace0924b44f1586";
    s'   = "00a34b00c20a8099df4b0a757cbef8fea1cb3ea7ced5fbf7e987f70b25ee6d4f";
  };
  { msg' = "dc3d4884c741a4a687593c79fb4e35c5c13c781dca16db561d7e393577f7b62ca41a6e259fc1fb8d0c4e1e062517a0fdf95558b7799f20c211796167953e6372c11829beec64869d67bf3ee1f1455dd87acfbdbcc597056e7fb347a17688ad32fda7ccc3572da7677d7255c261738f07763cd45973c728c6e9adbeecadc3d961";
    d    = "ecf644ea9b6c3a04fdfe2de4fdcb55fdcdfcf738c0b3176575fa91515194b566";
    qx'  = "c3bdc7c795ec94620a2cfff614c13a3390a5e86c892e53a24d3ed22228bc85bf";
    qy'  = "70480fc5cf4aacd73e24618b61b5c56c1ced8c4f1b869580ea538e68c7a61ca3";
    k    = "53291d51f68d9a12d1dcdc58892b2f786cc15f631f16997d2a49bace513557d4";
    r'   = "a860c8b286edf973ce4ce4cf6e70dc9bbf3818c36c023a845677a9963705df8b";
    s'   = "5630f986b1c45e36e127dd7932221c4272a8cc6e255e89f0f0ca4ec3a9f76494";
  };
  { msg' = "719bf1911ae5b5e08f1d97b92a5089c0ab9d6f1c175ac7199086aeeaa416a17e6d6f8486c711d386f284f096296689a54d330c8efb0f5fa1c5ba128d3234a3da856c2a94667ef7103616a64c913135f4e1dc50e38daa60610f732ad1bedfcc396f87169392520314a6b6b9af6793dbabad4599525228cc7c9c32c4d8e097ddf6";
    d    = "4961485cbc978f8456ec5ac7cfc9f7d9298f99415ecae69c8491b258c029bfee";
    qx'  = "8d40bf2299e05d758d421972e81cfb0cce68b949240dc30f315836acc70bef03";
    qy'  = "5674e6f77f8b46f46cca937d83b128dffbe9bd7e0d3d08aa2cbbfdfb16f72c9a";
    k    = "373a825b5a74b7b9e02f8d4d876b577b4c3984168d704ba9f95b19c05ed590af";
    r'   = "ef6fb386ad044b63feb7445fa16b10319018e9cea9ef42bca83bdad01992234a";
    s'   = "ac1f42f652eb1786e57be01d847c81f7efa072ba566d4583af4f1551a3f76c65";
  };
  { msg' = "7cf19f4c851e97c5bca11a39f0074c3b7bd3274e7dd75d0447b7b84995dfc9f716bf08c25347f56fcc5e5149cb3f9cfb39d408ace5a5c47e75f7a827fa0bb9921bb5b23a6053dbe1fa2bba341ac874d9b1333fc4dc224854949f5c8d8a5fedd02fb26fdfcd3be351aec0fcbef18972956c6ec0effaf057eb4420b6d28e0c008c";
    d    = "587907e7f215cf0d2cb2c9e6963d45b6e535ed426c828a6ea2fb637cca4c5cbd";
    qx'  = "660da45c413cc9c9526202c16b402af602d30daaa7c342f1e722f15199407f31";
    qy'  = "e6f8cbb06913cc718f2d69ba2fb3137f04a41c27c676d1a80fbf30ea3ca46439";
    k    = "6b8eb7c0d8af9456b95dd70561a0e902863e6dfa1c28d0fd4a0509f1c2a647b2";
    r'   = "08fabf9b57de81875bfa7a4118e3e44cfb38ec6a9b2014940207ba3b1c583038";
    s'   = "a58d199b1deba7350616230d867b2747a3459421811c291836abee715b8f67b4";
  };
  { msg' = "b892ffabb809e98a99b0a79895445fc734fa1b6159f9cddb6d21e510708bdab6076633ac30aaef43db566c0d21f4381db46711fe3812c5ce0fb4a40e3d5d8ab24e4e82d3560c6dc7c37794ee17d4a144065ef99c8d1c88bc22ad8c4c27d85ad518fa5747ae35276fc104829d3f5c72fc2a9ea55a1c3a87007cd133263f79e405";
    d    = "24b1e5676d1a9d6b645a984141a157c124531feeb92d915110aef474b1e27666";
    qx'  = "b4909a5bdf25f7659f4ef35e4b811429fb2c59126e3dad09100b46aea6ebe7a6";
    qy'  = "760ae015fa6af5c9749c4030fdb5de6e58c6b5b1944829105cf7edf7d3a22cfb";
    k    = "88794923d8943b5dbcc7a7a76503880ff7da632b0883aaa60a9fcc71bf880fd6";
    r'   = "6ec9a340b77fae3c7827fa96d997e92722ff2a928217b6dd3c628f3d49ae4ce6";
    s'   = "637b54bbcfb7e7d8a41ea317fcfca8ad74eb3bb6b778bc7ef9dec009281976f7";
  };
  { msg' = "8144e37014c95e13231cbd6fa64772771f93b44e37f7b02f592099cc146343edd4f4ec9fa1bc68d7f2e9ee78fc370443aa2803ff4ca52ee49a2f4daf2c8181ea7b8475b3a0f608fc3279d09e2d057fbe3f2ffbe5133796124781299c6da60cfe7ecea3abc30706ded2cdf18f9d788e59f2c31662df3abe01a9b12304fb8d5c8c";
    d    = "bce49c7b03dcdc72393b0a67cf5aa5df870f5aaa6137ada1edc7862e0981ec67";
    qx'  = "c786d9421d67b72b922cf3def2a25eeb5e73f34543eb50b152e738a98afb0ca5";
    qy'  = "6796271e79e2496f9e74b126b1123a3d067de56b5605d6f51c8f6e1d5bb93aba";
    k    = "89e690d78a5e0d2b8ce9f7fcbf34e2605fd9584760fa7729043397612dd21f94";
    r'   = "07e5054c384839584624e8d730454dc27e673c4a90cbf129d88b91250341854d";
    s'   = "f7e665b88614d0c5cbb3007cafe713763d81831525971f1747d92e4d1ca263a7";
  };
  { msg' = "a3683d120807f0a030feed679785326698c3702f1983eaba1b70ddfa7f0b3188060b845e2b67ed57ee68087746710450f7427cb34655d719c0acbc09ac696adb4b22aba1b9322b7111076e67053a55f62b501a4bca0ad9d50a868f51aeeb4ef27823236f5267e8da83e143047422ce140d66e05e44dc84fb3a4506b2a5d7caa8";
    d    = "73188a923bc0b289e81c3db48d826917910f1b957700f8925425c1fb27cabab9";
    qx'  = "86662c014ab666ee770723be8da38c5cd299efc6480fc6f8c3603438fa8397b9";
    qy'  = "f26b3307a650c3863faaa5f642f3ba1384c3d3a02edd3d48c657c269609cc3fc";
    k    = "ec90584ab3b383b590626f36ed4f5110e49888aec7ae7a9c5ea62dd2dc378666";
    r'   = "13e9ad59112fde3af4163eb5c2400b5e9a602576d5869ac1c569075f08c90ff6";
    s'   = "708ac65ff2b0baaccc6dd954e2a93df46016bd04457636de06798fcc17f02be5";
  };
  { msg' = "b1df8051b213fc5f636537e37e212eb20b2423e6467a9c7081336a870e6373fc835899d59e546c0ac668cc81ce4921e88f42e6da2a109a03b4f4e819a17c955b8d099ec6b282fb495258dca13ec779c459da909475519a3477223c06b99afbd77f9922e7cbef844b93f3ce5f50db816b2e0d8b1575d2e17a6b8db9111d6da578";
    d    = "f637d55763fe819541588e0c603f288a693cc66823c6bb7b8e003bd38580ebce";
    qx'  = "74a4620c578601475fc169a9b84be613b4a16cb6acab8fd98848a6ec9fbd133d";
    qy'  = "42b9e35d347c107e63bd55f525f915bcf1e3d2b81d002d3c39acf10fc30645a1";
    k    = "4d578f5099636234d9c1d566f1215d5d887ae5d47022be17dbf32a11a03f053b";
    r'   = "113a933ebc4d94ce1cef781e4829df0c493b0685d39fb2048ce01b21c398dbba";
    s'   = "3005bd4ec63dbd04ce9ff0c6246ad65d27fcf62edb2b7e461589f9f0e7446ffd";
  };
  { msg' = "0b918ede985b5c491797d0a81446b2933be312f419b212e3aae9ba5914c00af431747a9d287a7c7761e9bcbc8a12aaf9d4a76d13dad59fc742f8f218ef66eb67035220a07acc1a357c5b562ecb6b895cf725c4230412fefac72097f2c2b829ed58742d7c327cad0f1058df1bddd4ae9c6d2aba25480424308684cecd6517cdd8";
    d    = "2e357d51517ff93b821f895932fddded8347f32596b812308e6f1baf7dd8a47f";
    qx'  = "7e4078a1d50c669fb2996dd9bacb0c3ac7ede4f58fa0fa1222e78dbf5d1f4186";
    qy'  = "0014e46e90cc171fbb83ea34c6b78202ea8137a7d926f0169147ed5ae3d6596f";
    k    = "be522b0940b9a40d84bf790fe6abdc252877e671f2efa63a33a65a512fc2aa5c";
    r'   = "a26b9ad775ac37ff4c7f042cdc4872c5e4e5e800485f488ddfaaed379f468090";
    s'   = "f88eae2019bebbba62b453b8ee3472ca5c67c267964cffe0cf2d2933c1723dff";
  };
  { msg' = "0fab26fde1a4467ca930dbe513ccc3452b70313cccde2994eead2fde85c8da1db84d7d06a024c9e88629d5344224a4eae01b21a2665d5f7f36d5524bf5367d7f8b6a71ea05d413d4afde33777f0a3be49c9e6aa29ea447746a9e77ce27232a550b31dd4e7c9bc8913485f2dc83a56298051c92461fd46b14cc895c300a4fb874";
    d    = "77d60cacbbac86ab89009403c97289b5900466856887d3e6112af427f7f0f50b";
    qx'  = "a62032dfdb87e25ed0c70cad20d927c7effeb2638e6c88ddd670f74df16090e5";
    qy'  = "44c5ee2cf740ded468f5d2efe13daa7c5234645a37c073af35330d03a4fed976";
    k    = "06c1e692b045f425a21347ecf72833d0242906c7c1094f805566cdcb1256e394";
    r'   = "eb173b51fb0aec318950d097e7fda5c34e529519631c3e2c9b4550b903da417d";
    s'   = "ca2c13574bf1b7d56e9dc18315036a31b8bceddf3e2c2902dcb40f0cc9e31b45";
  };
  { msg' = "7843f157ef8566722a7d69da67de7599ee65cb3975508f70c612b3289190e364141781e0b832f2d9627122742f4b5871ceeafcd09ba5ec90cae6bcc01ae32b50f13f63918dfb5177df9797c6273b92d103c3f7a3fc2050d2b196cc872c57b77f9bdb1782d4195445fcc6236dd8bd14c8bcbc8223a6739f6a17c9a861e8c821a6";
    d    = "486854e77962117f49e09378de6c9e3b3522fa752b10b2c810bf48db584d7388";
    qx'  = "760b5624bd64d19c866e54ccd74ad7f98851afdbc3ddeae3ec2c52a135be9cfa";
    qy'  = "feca15ce9350877102eee0f5af18b2fed89dc86b7df0bf7bc2963c1638e36fe8";
    k    = "e4f77c6442eca239b01b0254e11a4182782d96f48ab521cc3d1d68df12b5a41a";
    r'   = "bdff14e4600309c2c77f79a25963a955b5b500a7b2d34cb172cd6acd52905c7b";
    s'   = "b0479cdb3df79923ec36a104a129534c5d59f622be7d613aa04530ad2507d3a2";
  };  
]


 let siggen_vectors_sha2_512 : list vec_SigGen = [
  { msg' = "6c8572b6a3a4a9e8e03dbeed99334d41661b8a8417074f335ab1845f6cc852adb8c01d9820fcf8e10699cc827a8fbdca2cbd46cc66e4e6b7ba41ec3efa733587e4a30ec552cd8ddab8163e148e50f4d090782897f3ddac84a41e1fcfe8c56b6152c0097b0d634b41011471ffd004f43eb4aafc038197ec6bae2b4470e869bded";
    d    = "9dd0d3a3d514c2a8adb162b81e3adfba3299309f7d2018f607bdb15b1a25f499";
    qx'  = "6b738de3398b6ac57b9591f9d7985dd4f32137ad3460dcf8970c1390cb9eaf8d";
    qy'  = "83bc61e26d2bbbd3cf2d2ab445a2bc4ab5dde41f4a13078fd1d3cc36ab596d57";
    k    = "9106192170ccb3c64684d48287bb81bbed51b40d503462c900e5c7aae43e380a";
    r'   = "275fa760878b4dc05e9d157fedfd8e9b1c9c861222a712748cb4b7754c043fb1";
    s'   = "699d906bb8435a05345af3b37e3b357786939e94caae257852f0503adb1e0f7e";
  };
  { msg' = "7e3c8fe162d48cc8c5b11b5e5ebc05ebc45c439bdbc0b0902145921b8383037cb0812222031598cd1a56fa71694fbd304cc62938233465ec39c6e49f57dfe823983b6923c4e865633949183e6b90e9e06d8275f3907d97967d47b6239fe2847b7d49cf16ba69d2862083cf1bccf7afe34fdc90e21998964107b64abe6b89d126";
    d    = "f9bf909b7973bf0e3dad0e43dcb2d7fa8bda49dbe6e5357f8f0e2bd119be30e6";
    qx'  = "f2a6674d4e86152a527199bed293fa63acde1b4d8a92b62e552210ba45c38792";
    qy'  = "c72565c24f0eee6a094af341ddd8579747b865f91c8ed5b44cda8a19cc93776f";
    k    = "e547791f7185850f03d0c58419648f65b9d29cdc22ed1de2a64280220cfcafba";
    r'   = "4782903d2aaf8b190dab5cae2223388d2d8bd845b3875d37485c54e1ded1d3d8";
    s'   = "dfb40e406bfa074f0bf832771b2b9f186e2211f0bca279644a0ca8559acf39da";
  };
  { msg' = "d5aa8ac9218ca661cd177756af6fbb5a40a3fecfd4eea6d5872fbb9a2884784aa9b5f0c023a6e0da5cf6364754ee6465b4ee2d0ddc745b02994c98427a213c849537da5a4477b3abfe02648be67f26e80b56a33150490d062aaac137aa47f11cfeddba855bab9e4e028532a563326d927f9e6e3292b1fb248ee90b6f429798db";
    d    = "724567d21ef682dfc6dc4d46853880cfa86fe6fea0efd51fac456f03c3d36ead";
    qx'  = "70b877b5e365fcf08140b1eca119baba662879f38e059d074a2cb60b03ea5d39";
    qy'  = "5f56f94d591df40b9f3b8763ac4b3dbe622c956d5bd0c55658b6f46fa3deb201";
    k    = "79d6c967ed23c763ece9ca4b026218004c84dc2d4ccc86cf05c5d0f791f6279b";
    r'   = "2ba2ea2d316f8937f184ad3028e364574d20a202e4e7513d7af57ac2456804d1";
    s'   = "64fe94968d18c5967c799e0349041b9e40e6c6c92ebb475e80dd82f51cf07320";
  };
  { msg' = "790b06054afc9c3fc4dfe72df19dd5d68d108cfcfca6212804f6d534fd2fbe489bd8f64bf205ce04bcb50124a12ce5238fc3fe7dd76e6fa640206af52549f133d593a1bfd423ab737f3326fa79433cde293236f90d4238f0dd38ed69492ddbd9c3eae583b6325a95dec3166fe52b21658293d8c137830ef45297d67813b7a508";
    d    = "29c5d54d7d1f099d50f949bfce8d6073dae059c5a19cc70834722f18a7199edd";
    qx'  = "3088d4f45d274cc5f418c8ecc4cbcf96be87491f420250f8cbc01cdf2503ec47";
    qy'  = "634db48198129237ed068c88ff5809f6211921a6258f548f4b64dd125921b78b";
    k    = "0508ad7774908b5705895fda5c3b7a3032bf85dab7232bf981177019f3d76460";
    r'   = "acd9f3b63626c5f32103e90e1dd1695907b1904aa9b14f2132caef331321971b";
    s'   = "15c04a8bd6c13ed5e9961814b2f406f064670153e4d5465dcef63c1d9dd52a87";
  };
  { msg' = "6d549aa87afdb8bfa60d22a68e2783b27e8db46041e4df04be0c261c4734b608a96f198d1cdb8d082ae48579ec9defcf21fbc72803764a58c31e5323d5452b9fb57c8991d31749140da7ef067b18bf0d7dfbae6eefd0d8064f334bf7e9ec1e028daed4e86e17635ec2e409a3ed1238048a45882c5c57501b314e636b9bc81cbe";
    d    = "0d8095da1abba06b0d349c226511f642dabbf1043ad41baa4e14297afe8a3117";
    qx'  = "75a45758ced45ecf55f755cb56ca2601d794ebeaeb2e6107fe2fc443f580e23c";
    qy'  = "5303d47d5a75ec821d51a2ee7548448208c699eca0cd89810ffc1aa4faf81ead";
    k    = "5165c54def4026ab648f7768c4f1488bcb183f6db7ffe02c7022a529a116482a";
    r'   = "ebc85fc4176b446b3384ccc62fc2526b45665561a0e7e9404ac376c90e450b59";
    s'   = "8b2c09428e62c5109d17ed0cf8f9fd7c370d018a2a73f701effc9b17d04852c6";
  };
  { msg' = "1906e48b7f889ee3ff7ab0807a7aa88f53f4018808870bfed6372a77330c737647961324c2b4d46f6ee8b01190474951a701b048ae86579ff8e3fc889fecf926b17f98958ac7534e6e781ca2db2baa380dec766cfb2a3eca2a9d5818967d64dfab84f768d24ec122eebacaab0a4dc3a75f37331bb1c43dd8966cc09ec4945bbd";
    d    = "52fe57da3427b1a75cb816f61c4e8e0e0551b94c01382b1a80837940ed579e61";
    qx'  = "2177e20a2092a46667debdcc21e7e45d6da72f124adecbc5ada6a7bcc7b401d5";
    qy'  = "550e468f2626070a080afeeb98edd75a721eb773c8e62149f3e903cf9c4d7b61";
    k    = "0464fe9674b01ff5bd8be21af3399fad66f90ad30f4e8ee6e2eb9bcccfd5185c";
    r'   = "f8250f073f34034c1cde58f69a85e2f5a030703ebdd4dbfb98d3b3690db7d114";
    s'   = "a9e83e05f1d6e0fef782f186bedf43684c825ac480174d48b0e4d31505e27498";
  };
  { msg' = "7b59fef13daf01afec35dea3276541be681c4916767f34d4e874464d20979863ee77ad0fd1635bcdf93e9f62ed69ae52ec90aab5bbf87f8951213747ccec9f38c775c1df1e9d7f735c2ce39b42edb3b0c5086247556cfea539995c5d9689765288ec600848ecf085c01ca738bbef11f5d12d4457db988b4add90be00781024ad";
    d    = "003d91611445919f59bfe3ca71fe0bfdeb0e39a7195e83ac03a37c7eceef0df2";
    qx'  = "7b9c592f61aae0555855d0b9ebb6fd00fb6746e8842e2523565c858630b9ba00";
    qy'  = "d35b2e168b1875bbc563bea5e8d63c4e38957c774a65e762959a349eaf263ba0";
    k    = "ef9df291ea27a4b45708f7608723c27d7d56b7df0599a54bc2c2fabbff373b40";
    r'   = "66d057fd39958b0e4932bacd70a1769bbadcb62e4470937b45497a3d4500fabb";
    s'   = "6c853b889e18b5a49ee54b54dd1aaedfdd642e30eba171c5cab677f0df9e7318";
  };
  { msg' = "041a6767a935dc3d8985eb4e608b0cbfebe7f93789d4200bcfe595277ac2b0f402889b580b72def5da778a680fd380c955421f626d52dd9a83ea180187b850e1b72a4ec6dd63235e598fd15a9b19f8ce9aec1d23f0bd6ea4d92360d50f951152bc9a01354732ba0cf90aaed33c307c1de8fa3d14f9489151b8377b57c7215f0b";
    d    = "48f13d393899cd835c4193670ec62f28e4c4903e0bbe5817bf0996831a720bb7";
    qx'  = "82a1a96f4648393c5e42633ecdeb1d8245c78c5ea236b5bab460dedcc8924bc0";
    qy'  = "e8cbf03c34b5154f876de19f3bb6fd43cd2eabf6e7c95467bcfa8c8fc42d76fd";
    k    = "efed736e627899fea944007eea39a4a63c0c2e26491cd12adb546be3e5c68f7d";
    r'   = "cf7fc24bdaa09ac0cca8497e13298b961380668613c7493954048c06385a7044";
    s'   = "f38b1c8306cf82ab76ee3a772b14416b49993fe11f986e9b0f0593c52ec91525";
  };
  { msg' = "7905a9036e022c78b2c9efd40b77b0a194fbc1d45462779b0b76ad30dc52c564e48a493d8249a061e62f26f453ba566538a4d43c64fb9fdbd1f36409316433c6f074e1b47b544a847de25fc67d81ac801ed9f7371a43da39001c90766f943e629d74d0436ba1240c3d7fab990d586a6d6ef1771786722df56448815f2feda48f";
    d    = "95c99cf9ec26480275f23de419e41bb779590f0eab5cf9095d37dd70cb75e870";
    qx'  = "42c292b0fbcc9f457ae361d940a9d45ad9427431a105a6e5cd90a345fe3507f7";
    qy'  = "313b08fd2fa351908b3178051ee782cc62b9954ad95d4119aa564900f8ade70c";
    k    = "4c08dd0f8b72ae9c674e1e448d4e2afe3a1ee69927fa23bbff3716f0b99553b7";
    r'   = "f2bc35eb1b8488b9e8d4a1dbb200e1abcb855458e1557dc1bf988278a174eb3b";
    s'   = "ed9a2ec043a1d578e8eba6f57217976310e8674385ad2da08d6146c629de1cd9";
  };
  { msg' = "cf25e4642d4f39d15afb7aec79469d82fc9aedb8f89964e79b749a852d931d37436502804e39555f5a3c75dd958fd5291ada647c1a5e38fe7b1048f16f2b711fdd5d39acc0812ca65bd50d7f8119f2fd195ab16633503a78ee9102c1f9c4c22568e0b54bd4fa3f5ff7b49160bf23e7e2231b1ebebbdaf0e4a7d4484158a87e07";
    d    = "e15e835d0e2217bc7c6f05a498f20af1cd56f2f165c23d225eb3360aa2c5cbcf";
    qx'  = "89dd22052ec3ab4840206a62f2270c21e7836d1a9109a3407dd0974c7802b9ae";
    qy'  = "e91609ba35c7008b080c77a9068d97a14ca77b97299e74945217672b2fd5faf0";
    k    = "c9f621441c235fc47ec34eef4c08625df1ec74918e1f86075b753f2589f4c60b";
    r'   = "a70d1a2d555d599bfb8c9b1f0d43725341151d17a8d0845fa56f3563703528a7";
    s'   = "4e05c45adf41783e394a5312f86e66871c4be4896948c85966879d5c66d54b37";
  };
  { msg' = "7562c445b35883cc937be6349b4cefc3556a80255d70f09e28c3f393daac19442a7eecedcdfbe8f7628e30cd8939537ec56d5c9645d43340eb4e78fc5dd4322de8a07966b262770d7ff13a071ff3dce560718e60ed3086b7e0003a6abafe91af90af86733ce8689440bf73d2aa0acfe9776036e877599acbabfcb03bb3b50faa";
    d    = "808c08c0d77423a6feaaffc8f98a2948f17726e67c15eeae4e672edbe388f98c";
    qx'  = "b0c0ad5e1f6001d8e9018ec611b2e3b91923e69fa6c98690ab644d650f640c42";
    qy'  = "610539c0b9ed21ac0a2f27527c1a61d9b47cbf033187b1a6ada006eb5b2662ed";
    k    = "1f6d4a905c761a53d54c362976717d0d7fc94d222bb5489e4830080a1a67535d";
    r'   = "83404dcf8320baf206381800071e6a75160342d19743b4f176960d669dd03d07";
    s'   = "3f75dcf102008b2989f81683ae45e9f1d4b67a6ef6fd5c8af44828af80e1cfb5";
  };
  { msg' = "051c2db8e71e44653ea1cb0afc9e0abdf12658e9e761bfb767c20c7ab4adfcb18ed9b5c372a3ac11d8a43c55f7f99b33355437891686d42362abd71db8b6d84dd694d6982f0612178a937aa934b9ac3c0794c39027bdd767841c4370666c80dbc0f8132ca27474f553d266deefd7c9dbad6d734f9006bb557567701bb7e6a7c9";
    d    = "f7c6315f0081acd8f09c7a2c3ec1b7ece20180b0a6365a27dcd8f71b729558f9";
    qx'  = "250f7112d381c1751860045d9bcaf20dbeb25a001431f96ac6f19109362ffebb";
    qy'  = "49fba9efe73546135a5a31ab3753e247034741ce839d3d94bd73936c4a17e4aa";
    k    = "68c299be2c0c6d52d208d5d1a9e0ffa2af19b4833271404e5876e0aa93987866";
    r'   = "7b195e92d2ba95911cda7570607e112d02a1c847ddaa33924734b51f5d81adab";
    s'   = "10d9f206755cef70ab5143ac43f3f8d38aea2644f31d52eaf3b472ee816e11e5";
  };
  { msg' = "4dcb7b62ba31b866fce7c1feedf0be1f67bf611dbc2e2e86f004422f67b3bc1839c6958eb1dc3ead137c3d7f88aa97244577a775c8021b1642a8647bba82871e3c15d0749ed343ea6cad38f123835d8ef66b0719273105e924e8685b65fd5dc430efbc35b05a6097f17ebc5943cdcd9abcba752b7f8f37027409bd6e11cd158f";
    d    = "f547735a9409386dbff719ce2dae03c50cb437d6b30cc7fa3ea20d9aec17e5a5";
    qx'  = "4ca87c5845fb04c2f76ae3273073b0523e356a445e4e95737260eba9e2d021db";
    qy'  = "0f86475d07f82655320fdf2cd8db23b21905b1b1f2f9c48e2df87e24119c4880";
    k    = "91bd7d97f7ed3253cedefc144771bb8acbbda6eb24f9d752bbe1dd018e1384c7";
    r'   = "008c1755d3df81e64e25270dbaa9396641556df7ffc7ac9add6739c382705397";
    s'   = "77df443c729b039aded5b516b1077fecdd9986402d2c4b01734ba91e055e87fc";
  };
  { msg' = "efe55737771070d5ac79236b04e3fbaf4f2e9bed187d1930680fcf1aba769674bf426310f21245006f528779347d28b8aeacd2b1d5e3456dcbf188b2be8c07f19219e4067c1e7c9714784285d8bac79a76b56f2e2676ea93994f11eb573af1d03fc8ed1118eafc7f07a82f3263c33eb85e497e18f435d4076a774f42d276c323";
    d    = "26a1aa4b927a516b661986895aff58f40b78cc5d0c767eda7eaa3dbb835b5628";
    qx'  = "28afa3b0f81a0e95ad302f487a9b679fcdef8d3f40236ec4d4dbf4bb0cbba8b2";
    qy'  = "bb4ac1be8405cbae8a553fbc28e29e2e689fabe7def26d653a1dafc023f3cecf";
    k    = "f98e1933c7fad4acbe94d95c1b013e1d6931fa8f67e6dbb677b564ef7c3e56ce";
    r'   = "15a9a5412d6a03edd71b84c121ce9a94cdd166e40da9ce4d79f1afff6a395a53";
    s'   = "86bbc2b6c63bad706ec0b093578e3f064736ec69c0dba59b9e3e7f73762a4dc3";
  };
  { msg' = "ea95859cc13cccb37198d919803be89c2ee10befdcaf5d5afa09dcc529d333ae1e4ffd3bd8ba8642203badd7a80a3f77eeee9402eed365d53f05c1a995c536f8236ba6b6ff8897393506660cc8ea82b2163aa6a1855251c87d935e23857fe35b889427b449de7274d7754bdeace960b4303c5dd5f745a5cfd580293d6548c832";
    d    = "6a5ca39aae2d45aa331f18a8598a3f2db32781f7c92efd4f64ee3bbe0c4c4e49";
    qx'  = "c62cc4a39ace01006ad48cf49a3e71466955bbeeca5d318d672695df926b3aa4";
    qy'  = "c85ccf517bf2ebd9ad6a9e99254def0d74d1d2fd611e328b4a3988d4f045fe6f";
    k    = "dac00c462bc85bf39c31b5e01df33e2ec1569e6efcb334bf18f0951992ac6160";
    r'   = "6e7ff8ec7a5c48e0877224a9fa8481283de45fcbee23b4c252b0c622442c26ad";
    s'   = "3dfac320b9c873318117da6bd856000a392b815659e5aa2a6a1852ccb2501df3";
  };
]
