#!/bin/sh
set -e
# Shell script to update c-ares in the source tree to a specific version

BASE_DIR=$(cd "$(dirname "$0")/../.." && pwd)
DEPS_DIR="$BASE_DIR/deps"

[ -z "$NODE" ] && NODE="$BASE_DIR/out/Release/node"
[ -x "$NODE" ] || NODE=$(command -v node)

# shellcheck disable=SC1091
. "$BASE_DIR/tools/dep_updaters/utils.sh"

NEW_VERSION="$("$NODE" --input-type=module <<'EOF'
const res = await fetch('https://api.github.com/repos/c-ares/c-ares/releases/latest',
  process.env.GITHUB_TOKEN && {
    headers: {
      "Authorization": `Bearer ${process.env.GITHUB_TOKEN}`
    },
  });
if (!res.ok) throw new Error(`FetchError: ${res.status} ${res.statusText}`, { cause: res });
const { tag_name } = await res.json();
console.log(tag_name.replace('cares-', '').replaceAll('_', '.'));
EOF
)"

CURRENT_VERSION=$(grep "#define ARES_VERSION_STR" ./deps/cares/include/ares_version.h |  sed -n "s/^.*VERSION_STR \"\(.*\)\"/\1/p")

# This function exit with 0 if new version and current version are the same
compare_dependency_version "c-ares" "$NEW_VERSION" "$CURRENT_VERSION"

echo "Making temporary workspace"

WORKSPACE=$(mktemp -d 2> /dev/null || mktemp -d -t 'tmp')

cleanup () {
  EXIT_CODE=$?
  [ -d "$WORKSPACE" ] && rm -rf "$WORKSPACE"
  exit $EXIT_CODE
}

trap cleanup INT TERM EXIT

ARES_REF="cares-$(echo "$NEW_VERSION" | tr . _)"
ARES_TARBALL="c-ares-$NEW_VERSION.tar.gz"

cd "$WORKSPACE"

echo "Fetching c-ares source archive"
curl -sL -o "$ARES_TARBALL" "https://github.com/c-ares/c-ares/releases/download/$ARES_REF/$ARES_TARBALL"
log_and_verify_sha256sum "c-ares" "$ARES_TARBALL"
gzip -dc "$ARES_TARBALL" | tar xf -
rm "$ARES_TARBALL"
mv "c-ares-$NEW_VERSION" cares

echo "Removing tests"
rm -rf "$WORKSPACE/cares/test"

echo "Copying existing .gitignore, config, gyp and gn files"
cp -R "$DEPS_DIR/cares/config" "$WORKSPACE/cares"
cp "$DEPS_DIR/cares/.gitignore" "$WORKSPACE/cares"
cp "$DEPS_DIR/cares/cares.gyp" "$WORKSPACE/cares"
cp "$DEPS_DIR/cares/"*.gn "$DEPS_DIR/cares/"*.gni "$WORKSPACE/cares"

echo "Replacing existing c-ares"
rm -rf "$DEPS_DIR/cares"
mv "$WORKSPACE/cares" "$DEPS_DIR/"

# Update the version number on maintaining-dependencies.md
# and print the new version as the last line of the script as we need
# to add it to $GITHUB_ENV variable
finalize_version_update "c-ares" "$NEW_VERSION"
