#include <string.h>
#include "update_feed/parse_xml/parse_xml_feed.h"

// References:
// https://www.rssboard.org/media-rss
//
// Notes to the future.
//
// There is no need to parse thumbnail elements, because storing information
// about decorating pictures defeats the whole purpose of project being a
// console application with as few distractions as possible.
//
// Purpose of media:group element is to group content elements that are
// effectively the same content, but from the user's perspective, all
// attachments are just links with some metadata. Therefore, such grouping does
// not make sense and is ignored by the parser. Also, this grouping is difficult
// to reflect in the database and requires various sophistications.

static int8_t
mediarss_content_start(struct stream_callback_data *data, const XML_Char **attrs)
{
	const char *attr = get_value_of_attribute_key(attrs, "url");
	if (attr == NULL) {
		return PARSE_OKAY; // Ignore empty content entries.
	}
	const size_t attr_len = strlen(attr);
	if (attr_len == 0) {
		return PARSE_OKAY; // Ignore empty content entries.
	}
	struct string **dest = data->in_item ? &data->feed.item->attachments : &data->feed.attachments;
	if (serialize_caret(dest) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_array(dest, "url", 3, attr, attr_len) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_attribute(dest, attrs, "type", "type", 4) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_attribute(dest, attrs, "fileSize", "size", 4) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_attribute(dest, attrs, "duration", "duration", 8) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_attribute(dest, attrs, "width", "width", 5) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_attribute(dest, attrs, "height", "height", 6) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	return PARSE_OKAY;
}

static int8_t
embed_or_player_start(struct stream_callback_data *data, const XML_Char **attrs)
{
	const char *attr = get_value_of_attribute_key(attrs, "url");
	if (attr == NULL) {
		return PARSE_OKAY; // Ignore empty entries.
	}
	const size_t attr_len = strlen(attr);
	if (attr_len == 0) {
		return PARSE_OKAY; // Ignore empty entries.
	}
	struct string **dest = data->in_item ? &data->feed.item->attachments : &data->feed.attachments;
	if (serialize_caret(dest) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_array(dest, "url", 3, attr, attr_len) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	return PARSE_OKAY;
}

static int8_t
peerlink_start(struct stream_callback_data *data, const XML_Char **attrs)
{
	const char *attr = get_value_of_attribute_key(attrs, "href");
	if (attr == NULL) {
		return PARSE_OKAY; // Ignore empty entries.
	}
	const size_t attr_len = strlen(attr);
	if (attr_len == 0) {
		return PARSE_OKAY; // Ignore empty entries.
	}
	struct string **dest = data->in_item ? &data->feed.item->attachments : &data->feed.attachments;
	if (serialize_caret(dest) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_array(dest, "url", 3, attr, attr_len) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	if (serialize_attribute(dest, attrs, "type", "type", 4) == false) {
		return PARSE_FAIL_NOT_ENOUGH_MEMORY;
	}
	return PARSE_OKAY;
}

static int8_t
description_start(struct stream_callback_data *data, const XML_Char **attrs)
{
	if (data->in_item == true) {
		if (data->path[data->depth] == MEDIARSS_CONTENT) {
			if (serialize_attribute(&data->feed.item->attachments, attrs, "type", "description_type", 16) == false) {
				return PARSE_FAIL_NOT_ENOUGH_MEMORY;
			}
		} else {
			if (serialize_caret(&data->feed.item->content) == false) {
				return PARSE_FAIL_NOT_ENOUGH_MEMORY;
			}
			if (serialize_attribute(&data->feed.item->content, attrs, "type", "type", 4) == false) {
				return PARSE_FAIL_NOT_ENOUGH_MEMORY;
			}
		}
	}
	return PARSE_OKAY;
}

static int8_t
description_end(struct stream_callback_data *data)
{
	if (data->in_item == true) {
		if (data->path[data->depth] == MEDIARSS_CONTENT) {
			if (serialize_string(&data->feed.item->attachments, "description_text", 16, data->text) == false) {
				return PARSE_FAIL_NOT_ENOUGH_MEMORY;
			}
		} else {
			if (serialize_string(&data->feed.item->content, "text", 4, data->text) == false) {
				return PARSE_FAIL_NOT_ENOUGH_MEMORY;
			}
		}
	}
	return PARSE_OKAY;
}

const struct xml_element_handler xml_mediarss_handlers[] = {
	{"content",     MEDIARSS_CONTENT, &mediarss_content_start, NULL},
	{"embed",       XML_UNKNOWN_POS,  &embed_or_player_start,  NULL},
	{"player",      XML_UNKNOWN_POS,  &embed_or_player_start,  NULL},
	{"peerLink",    XML_UNKNOWN_POS,  &peerlink_start,         NULL},
	{"description", XML_UNKNOWN_POS,  &description_start,      &description_end},
	{NULL,          XML_UNKNOWN_POS,  NULL,                    NULL},
};
