/*
                                  NETWIB
                             Network library
                Copyright(c) 1999-2005 Laurent Constantin
                                  -----

  Main server    : http://www.laurentconstantin.com/
  Backup servers : http://go.to/laurentconstantin/
                   http://laurentconstantin.est-la.com/
                   http://laurentconstantin.free.fr/
                   http://membres.lycos.fr/lauconstantin/
  [my current email address is on the web servers]

                                  -----
  This file is part of Netwib.

  Netwib is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  version 2 as published by the Free Software Foundation.

  Netwib is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details (http://www.gnu.org/).

------------------------------------------------------------------------
*/

#include <netwib/inc/maininc.h>

/*-------------------------------------------------------------*/
struct netwib_dir {
 #if defined NETWIBDEF_SYSNAME_Unix
  DIR *pdirstream;
 #elif defined NETWIBDEF_SYSNAME_Windows
  netwib_buf dirname;
  netwib_bool dirhandlevalid;
  long dirhandle;
 #else
  #error "Unknown value for NETWIBDEF_SYSNAME"
 #endif
};

/*-------------------------------------------------------------*/
netwib_err netwib_dir_init(netwib_constbuf *pdirname,
                           netwib_dir **ppdir)
{
  netwib_string dirname;
  netwib_dir *pdir;

  /* parameter verification */
  if (ppdir == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }

  /* allocate needed memory to store pdir */
  netwib_er(netwib_ptr_malloc(sizeof(netwib_dir), (netwib_ptr*)&pdir));
  *ppdir = pdir;

  netwib__buf_ref_string(pdirname, dirname, bufstorage,
                         netwib_dir_init(&bufstorage, ppdir));

  /* set parameters */
#if defined NETWIBDEF_SYSNAME_Unix
  pdir->pdirstream = opendir(dirname);
  if (pdir->pdirstream == NULL) {
    return(NETWIB_ERR_FUOPENDIR);
  }
#elif defined NETWIBDEF_SYSNAME_Windows
  netwib_er(netwib_buf_init_mallocdefault(&pdir->dirname));
  netwib_er(netwib_buf_append_buf(pdirname, &pdir->dirname));
  if (dirname[netwib_c_strlen(dirname) - 1] == '/') {
    netwib_er(netwib_buf_append_byte('*', &pdir->dirname));
  } else {
    netwib_er(netwib_buf_append_text("/*", &pdir->dirname));
  }
  pdir->dirhandlevalid = NETWIB_FALSE;
  pdir->dirhandle = 0;
#else
 #error "Unknown value for NETWIBDEF_SYSNAME"
#endif

#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_add_dir(pdir));
#endif

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_dir_close(netwib_dir **ppdir)
{
  netwib_dir *pdir;
  int reti;

  /* parameter verification */
  if (ppdir == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
  pdir = *ppdir;

  /* close */
#if defined NETWIBDEF_SYSNAME_Unix
  reti = closedir(pdir->pdirstream);
  if (reti == -1) {
    return(NETWIB_ERR_FUCLOSEDIR);
  }
#elif defined NETWIBDEF_SYSNAME_Windows
  netwib_er(netwib_buf_close(&pdir->dirname));
  if (pdir->dirhandlevalid) {
    reti = _findclose(pdir->dirhandle);
    if (reti == -1) {
      return(NETWIB_ERR_FUFINDCLOSE);
    }
    pdir->dirhandlevalid = NETWIB_FALSE;
  }
#else
 #error "Unknown value for NETWIBDEF_SYSNAME"
#endif

  /* free memory */
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_del_dir(pdir));
#endif
  netwib_er(netwib_ptr_free((netwib_ptr*)&pdir));

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_dir_ctl_set(netwib_dir *pdir,
                              netwib_dir_ctltype type,
                              netwib_ptr p,
                              netwib_uint32 ui)
{

  /* simple cases */
  if (pdir == NULL) {
    /* we have no sufficient info, but we suppose it will be ok if
       a correct parameter is given*/
    return(NETWIB_ERR_OK);
  }
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_dir(pdir));
#endif

  switch(type) {
    case NETWIB_DIR_CTLTYPE_REWIND :
#if defined NETWIBDEF_SYSNAME_Unix
      rewinddir(pdir->pdirstream);
#elif defined NETWIBDEF_SYSNAME_Windows
      {
        int reti;

        if (pdir->dirhandlevalid) {
          reti = _findclose(pdir->dirhandle);
          if (reti == -1) {
            return(NETWIB_ERR_FUFINDCLOSE);
          }
          pdir->dirhandlevalid = NETWIB_FALSE;
        }
      }
#else
 #error "Unknown value for NETWIBDEF_SYSNAME"
#endif
      return(NETWIB_ERR_OK);
      break;
    default :
      return(NETWIB_ERR_PAINVALIDTYPE);
      /* perhaps used in those cases */
      p = p;
      ui = ui;
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
netwib_err netwib_dir_ctl_get(netwib_dir *pdir,
                              netwib_dir_ctltype type,
                              netwib_ptr p,
                              netwib_uint32 *pui)
{

  /* simple cases */
  if (pdir == NULL) {
    /* it's not possible to obtain info if this pointer is NULL */
    return(NETWIB_ERR_PANULLPTR);
  }
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_dir(pdir));
#endif

  switch(type) {
    case NETWIB_DIR_CTLTYPE_REWIND :
      return(NETWIB_ERR_PAINVALIDTYPE);
      break;
    default :
      return(NETWIB_ERR_PAINVALIDTYPE);
      /* perhaps used in those cases */
      p = p;
      pui = pui;
  }

  return(NETWIB_ERR_OK);
}

/*-------------------------------------------------------------*/
/* those defines are here because I have a script checking if
 * netwib contains MT-unsafe functions. Like this, they are ignored. */
#define netwib_readdir readdir

/*-------------------------------------------------------------*/
#if defined NETWIBDEF_SYSNAME_Unix
static netwib_err netwib_priv_dir_next(DIR *dir,
                                       netwib_ptr ptr, /* sufficient big */
                                       netwib_string *pstartname)
{

#if NETWIBDEF_HAVEFUNC_READDIR_R == 1
 #if NETWIBDEF_LIBPTHREADFROMDCE == 1
  {
    struct dirent *pentry;
    int reti;
    pentry = (struct dirent *)ptr;
    reti = readdir_r(dir, pentry);
    if (reti) {
      return(NETWIB_ERR_DATAEND);
    }
    *pstartname = pentry->d_name;
  }
 #else
   {
     struct dirent *pentry, *presult;
     int reti;
     pentry = (struct dirent *)ptr;
     reti = readdir_r(dir, pentry, &presult);
     if (reti) {
       if (errno == ENOENT) {
         /* Solaris 2.5.1 set errno instead of presult */
         return(NETWIB_ERR_DATAEND);
       } else {
         return(NETWIB_ERR_FUREADDIRR);
       }
     }
     if (presult == NULL) {
       return(NETWIB_ERR_DATAEND);
     }
     *pstartname = pentry->d_name;
   }
 #endif
#else
  {
    struct dirent *presult;
    netwib_string ptrstr = (netwib_string)ptr;
    netwib_er(netwib_priv_glovars_other_wrlock());
    presult = netwib_readdir(dir);
    if (presult == NULL) {
      netwib_er(netwib_priv_glovars_other_wrunlock());
      return(NETWIB_ERR_DATAEND);
    }
    netwib_c_strcpy(ptrstr, presult->d_name);
    netwib_er(netwib_priv_glovars_other_wrunlock());
    *pstartname = ptrstr;
  }
#endif

  return(NETWIB_ERR_OK);
}
#elif defined NETWIBDEF_SYSNAME_Windows
#else
 #error "Unknown value for NETWIBDEF_SYSNAME"
#endif

/*-------------------------------------------------------------*/
netwib_err netwib_dir_next(netwib_dir *pdir,
                           netwib_buf *pbuffilename)
{
  netwib_err ret=NETWIB_ERR_OK;

  /* parameter verification */
  if (pdir == NULL) {
    return(NETWIB_ERR_PANULLPTR);
  }
#if NETWIB_DEBUG_LEAK==1
  netwib_er(netwib_debug_leak_valid_dir(pdir));
#endif

  /* obtain next */
#if defined NETWIBDEF_SYSNAME_Unix
  {
    netwib_uint32 namemax;
    netwib_ptr ptr;
    netwib_string startname;

#if NETWIBDEF_HAVEFUNC_PATHCONF==1
    namemax = pathconf("/", _PC_NAME_MAX);
#elif NETWIBDEF_HAVEVAR_NAME_MAX==1
    namemax = NAME_MAX;
#else
    namemax = 255;
#endif

    netwib_er(netwib_ptr_malloc(sizeof(struct dirent) + namemax + 1, &ptr));
    while (NETWIB_TRUE) {
      ret = netwib_priv_dir_next(pdir->pdirstream, ptr, &startname);
      if (ret != NETWIB_ERR_OK) {
        break;
      }
      if (netwib_c_strcmp(startname, ".") &&
          netwib_c_strcmp(startname, "..")) {
        ret = netwib_buf_append_text(startname, pbuffilename);
        break;
      }
    }
    netwib_er(netwib_ptr_free(&ptr));
  }
#elif defined NETWIBDEF_SYSNAME_Windows
  {
    netwib_string dirname;
     struct _finddata_t finddata;
     int reti;

     while (NETWIB_TRUE) {
       if (pdir->dirhandlevalid) {
         reti = _findnext(pdir->dirhandle, &finddata);
         if (reti == -1) {
           return(NETWIB_ERR_DATAEND);
         }
       } else {
         netwib_er(netwib_buf_ref_string(&pdir->dirname, &dirname));
         pdir->dirhandle = _findfirst(dirname, &finddata);
         if (pdir->dirhandle == (long)-1) {
           return(NETWIB_ERR_DATAEND);
         }
         pdir->dirhandlevalid = NETWIB_TRUE;
       }
       if (netwib_c_strcmp(finddata.name, ".") &&
           netwib_c_strcmp(finddata.name, "..")) {
         netwib_er(netwib_buf_append_text(finddata.name, pbuffilename));
         break;
       }
     }
  }
#else
 #error "Unknown value for NETWIBDEF_SYSNAME"
#endif

  return(ret);
}
