/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/param.h>
#include <errno.h>

#include <gtk/gtk.h>

#include <nd.h>
#include <nd_prefs.h>
#include <nd_macros.h>
#include <nd_trace_registry.h>
#include <nd_gui.h>
#include "interface.h"
#include "support.h"


struct nd_prefs_domain
{
  /* The widget from which the various gui items can be requested */
  GtkWidget              *container;

  /* The widget that actually displays the config items */
  GtkWidget              *gui;
};


static LND_PrefsEntry prefs_entries_netdude[] = {
  { "show_timestamps",           LND_PREFS_INT, TRUE,       LND_UNUSED, LND_UNUSED  },
  { "show_timestamps_absolute",  LND_PREFS_INT, FALSE,      LND_UNUSED, LND_UNUSED  }, 
  { "timestamps_delay",          LND_PREFS_FLT, LND_UNUSED, 0.5,        LND_UNUSED  },
  { "show_splash",               LND_PREFS_INT, TRUE,       LND_UNUSED, LND_UNUSED  },
  { "show_full_path",            LND_PREFS_INT, TRUE,       LND_UNUSED, LND_UNUSED  },
  { "font_mono",                 LND_PREFS_STR, LND_UNUSED, LND_UNUSED, "-*-courier-medium-r-*-*-*-100-*-*-*-*-*-*" },
  { "recent_0",                  LND_PREFS_STR, LND_UNUSED, LND_UNUSED, "" },
  { "recent_1",                  LND_PREFS_STR, LND_UNUSED, LND_UNUSED, "" },
  { "recent_2",                  LND_PREFS_STR, LND_UNUSED, LND_UNUSED, "" },
  { "recent_3",                  LND_PREFS_STR, LND_UNUSED, LND_UNUSED, "" },
  { "recent_4",                  LND_PREFS_STR, LND_UNUSED, LND_UNUSED, "" },
};

static GtkWidget   *prefs_dialog = NULL;


const char    *
nd_prefs_get_plugin_dir_global(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/%s/plugins",
	     PACKAGE_DATA_DIR, VERSION_MAJOR);

  return dir;
}


const char    *
nd_prefs_get_plugin_dir_user(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/%s/plugins",
	     libnd_prefs_get_netdude_dir(), VERSION_MAJOR);

  return dir;
}


const char    *
nd_prefs_get_proto_dir_global(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/%s/protocols",
	     PACKAGE_DATA_DIR, VERSION_MAJOR);

  return dir;
}


const char    *
nd_prefs_get_proto_dir_user(void)
{
  static char dir[MAXPATHLEN] = "\0";

  if (dir[0] != 0)
    return dir;

  g_snprintf(dir, MAXPATHLEN, "%s/%s/protocols",
	     libnd_prefs_get_netdude_dir(), VERSION_MAJOR);

  return dir;
}


static ND_PrefsDomain *
prefs_domain_new(GtkWidget *container,
		 GtkWidget *gui)
{
  ND_PrefsDomain *domain;

  if (!container || !gui)
    return NULL;

  domain = g_new0(ND_PrefsDomain, 1);
  
  if (!domain)
    return NULL;

  domain->container = container;
  domain->gui = gui;

  return domain;
}


static void  
prefs_sync_settings_to_dialog_cb(LND_PrefsDomain *domain, void *user_data)
{
  ND_PrefsDomain *domain_gui;
  int        i;
  GtkWidget *w;
  
  if (! (domain_gui = nd_prefs_get_domain(domain)))
    return;
  
  D(("Syncing settings for domain %s\n", domain->name));

  for (i = 0; i < domain->num_entries; i++)
    {
      if (!domain_gui->container)
	continue;
      
      if ( (w = gtk_object_get_data(GTK_OBJECT(domain_gui->container),
				    domain->entries[i].key)) == NULL)
	{
	  D(("Widget lookup failed for %s/%s\n",
	     domain->name, domain->entries[i].key));
	  continue;
	}
      
      if (GTK_IS_TOGGLE_BUTTON(w))
	{
	  libnd_prefs_set_int_item(domain->name, domain->entries[i].key,
				   gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(w)));
	}
      else if (GTK_IS_SPIN_BUTTON(w))
	{
	  switch (domain->entries[i].type)
	    {
	    case LND_PREFS_FLT:
	      libnd_prefs_set_flt_item(domain->name, domain->entries[i].key,
				       gtk_spin_button_get_value_as_float(GTK_SPIN_BUTTON(w)));
	      break;
	      
	    case LND_PREFS_INT:
	      libnd_prefs_set_int_item(domain->name, domain->entries[i].key,
				       gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(w)));
	      break;
	      
	    default:
	      D(("WARNING -- unexpected preference data type.\n"));					
	    }
	}
      else if (GTK_IS_ENTRY(w))
	{
	  libnd_prefs_set_str_item(domain->name, domain->entries[i].key,
				   gtk_entry_get_text(GTK_ENTRY(w)));
	}
      else if (GTK_IS_LABEL(w))
	{
	  char *value;
	  
	  gtk_label_get(GTK_LABEL(w), &value);
	  libnd_prefs_set_str_item(domain->name, domain->entries[i].key, value);
	}
      else
	{
	  D(("Unhandled widget type.\n"));
	}
    }

  return;
  TOUCH(user_data);
}

static void  
prefs_sync_settings_to_dialog(void)
{
  libnd_prefs_foreach_domain(prefs_sync_settings_to_dialog_cb, NULL);
}


static void
prefs_dialog_sync_cb(LND_PrefsDomain *domain, void *user_data)
{
  int          i;
  int          int_val;
  float        flt_val;
  char        *str_val;
  GtkWidget   *w;
  ND_PrefsDomain *domain_gui;

  if (! (domain_gui = nd_prefs_get_domain(domain)))
    return;

  D(("Syncing gui for prefs domain %s with %i entries\n",
     domain->name, domain->num_entries));

  for (i = 0; i < domain->num_entries; i++)
    {
      if (!domain_gui->container)
	{
	  D(("  No container\n"));
	  continue;
	}
      
      if ( (w = gtk_object_get_data(GTK_OBJECT(domain_gui->container),
				    domain->entries[i].key)) == NULL)
	{
	  D(("  Widget lookup failed for %s/%s\n",
	     domain->name, domain->entries[i].key));
	  continue;
	}
      
      if (GTK_IS_TOGGLE_BUTTON(w))
	{
	  libnd_prefs_get_int_item(domain->name, domain->entries[i].key, &int_val);
	  gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(w), int_val);
	  D(("  Setting toggle button %s to %i\n", domain->entries[i].key, int_val));
	  continue;
	}
      else if (GTK_IS_SPIN_BUTTON(w))
	{
	  switch (domain->entries[i].type)
	    {
	    case LND_PREFS_FLT:
	      libnd_prefs_get_flt_item(domain->name, domain->entries[i].key, &flt_val);
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), flt_val); 
	      D(("  Setting spin button %s to %f\n", domain->entries[i].key, flt_val));
	      break;
	      
	    case LND_PREFS_INT:
	      libnd_prefs_get_int_item(domain->name, domain->entries[i].key, &int_val);
	      gtk_spin_button_set_value(GTK_SPIN_BUTTON(w), (float) int_val); 
	      D(("  Setting spin button %s to %f\n", domain->entries[i].key, (float) int_val));
	      break;
	      
	    default:
	      D(("  WARNING -- unexpected preference data type.\n"));					
	    }

	  continue;
	}
      else if (GTK_IS_ENTRY(w))
	{
	  libnd_prefs_get_str_item(domain->name, domain->entries[i].key, &str_val);
	  gtk_entry_set_text(GTK_ENTRY(w), str_val);
	  D(("  Setting entry field %s to %s\n", domain->entries[i].key, str_val));
	  continue;
	}
      else if (GTK_IS_LABEL(w))
	{
	  libnd_prefs_get_str_item(domain->name, domain->entries[i].key, &str_val);
	  nd_gui_add_monowidth_widget(w);
	  gtk_label_set_text(GTK_LABEL(w), str_val);	  
	  continue;
	}

      D(("  Unhandled widget type.\n"));
    }

  return;
  TOUCH(user_data);
}


void
nd_prefs_dialog_sync(void)
{
  D_ASSERT_PTR(prefs_dialog);

  if (!prefs_dialog)
    return;

  libnd_prefs_foreach_domain(prefs_dialog_sync_cb, NULL);
}

static void
prefs_update_packets_cb(LND_Trace *trace, void *user_data)
{
  libnd_tcpdump_close(trace);
  libnd_tcpdump_open(trace);
  
  nd_gui_list_update(trace);
}

static void
prefs_update_cb(LND_PrefsDomain *domain, void *user_data)
{
  LND_Trace      *trace;
  
  if ( (trace = nd_trace_registry_get_current()))
    nd_gui_set_windowtitle(libnd_trace_get_name(trace));
  
  nd_gui_update_monowidth_widgets();
  nd_trace_registry_foreach(prefs_update_packets_cb, NULL);
  
  return;
  TOUCH(domain);
  TOUCH(user_data);
}

/* This is our local callback for updating anything in the local
 * config domain to new settings. Gets called on startup and whenever
 * settings change.
 */
static void
prefs_apply_cb(LND_PrefsDomain *domain, void *user_data)
{
  prefs_sync_settings_to_dialog();

  return;
  TOUCH(domain);
  TOUCH(user_data);
}

	       
void
nd_prefs_init(void)
{
  LND_PrefsDomain *domain;
  GtkWidget       *general_tab;

  D_ENTER;

  if (!prefs_dialog)
    prefs_dialog = create_pref_dialog();

  libnd_prefs_load();

  domain = libnd_prefs_add_domain(ND_DOM_NETDUDE,
				  prefs_entries_netdude,
				  sizeof (prefs_entries_netdude) / sizeof(LND_PrefsEntry));

  libnd_prefs_domain_add_apply_cb(domain, prefs_apply_cb);
  libnd_prefs_domain_add_update_cb(domain, prefs_update_cb);

  libnd_prefs_save();

  ND_GTK_GET(general_tab, prefs_dialog, "general_tab");

  /* The preferences dialog already has a tab for libnetdude's and
   * Netdude's preference settings, so we just want to hook that tab
   * into the existing domains. Calling nd_prefs_add_domain_gui()
   * would also add a new tab to the preferences notebook, which
   * isn't what we want.
   */

  domain = libnd_prefs_get_domain(LND_DOM_NETDUDE);
  D_ASSERT_PTR(domain);
  domain->user_data = prefs_domain_new(prefs_dialog, general_tab);

  domain = libnd_prefs_get_domain(ND_DOM_NETDUDE);
  D_ASSERT_PTR(domain);
  D(("Num entries: %i\n", domain->num_entries));
  domain->user_data = prefs_domain_new(prefs_dialog, general_tab);

  D_RETURN;
}
  


void           
nd_prefs_add_domain_gui(LND_PrefsDomain *domain,
			GtkWidget *container,
			GtkWidget *gui)
{
  GtkNotebook    *notebook;
  ND_PrefsDomain *domain_gui;


  if (!domain || domain->user_data || !gui || !container)
    return;

  domain_gui = prefs_domain_new(container, gui);
  domain->user_data = domain_gui;

  ND_GTK_GET(notebook, prefs_dialog, "prefs_notebook");
  D_ASSERT_PTR(notebook);

  gtk_widget_show(gui);
  gtk_notebook_append_page(notebook, gui, gtk_label_new(domain->name));
}


ND_PrefsDomain *
nd_prefs_get_domain(LND_PrefsDomain *domain)
{
  if (!domain)
    return NULL;
  
  D(("Obtaining GUI for domain %s\n", domain->name));
  D_ASSERT(domain->user_data, "Domain doesn't have GUI info\n");
  
  return (ND_PrefsDomain *) domain->user_data;
}


void     
nd_prefs_dialog_show(void)
{
  if (!prefs_dialog)
    prefs_dialog = create_pref_dialog();
  
  nd_prefs_dialog_sync();
  gtk_widget_show(prefs_dialog);
}


void     
nd_prefs_dialog_ok(void)
{
  nd_prefs_dialog_apply();

  if (!prefs_dialog)
    return;

  gtk_widget_hide(prefs_dialog);
}


void     
nd_prefs_dialog_apply(void)
{
  libnd_prefs_apply();
  libnd_prefs_save();
}


void     
nd_prefs_dialog_cancel(void)
{
  if (!prefs_dialog)
    return;

  gtk_widget_hide(prefs_dialog);
}


static void
prefs_select_tcpdump_cb(const char *filename,
			void *user_data)			
{
  if (libnd_misc_can_exec(filename))
    {
      nd_dialog_filesel_close();
      libnd_prefs_set_str_item(LND_DOM_NETDUDE, "tcpdump_path", filename);
      nd_prefs_dialog_sync();
    }

  return;
  TOUCH(user_data);
}


void           
nd_prefs_select_tcpdump(void)
{
  char *tcpdump_path;
  
  libnd_prefs_get_str_item(LND_DOM_NETDUDE, "tcpdump_path", &tcpdump_path);
  nd_dialog_filesel(_("Select Tcpdump Executable"),
		    tcpdump_path,
		    prefs_select_tcpdump_cb,
		    NULL);
}


static void
prefs_fontsel_cb(const char *font_name,
		 void *user_data)
{
  GtkStyle     *style;
  GtkWidget    *w;

  if (!prefs_dialog)
    return;

  ND_GTK_GET(w, prefs_dialog, "font_mono");
  gtk_label_set_text(GTK_LABEL(w), font_name);

  style = gtk_style_copy(gtk_widget_get_style(w));
  gdk_font_unref(style->font);
  style->font = gdk_font_load (font_name);
  gtk_widget_set_style(w, style);

  return;
  TOUCH(user_data);
}


void           
nd_prefs_fontsel_show(void)
{
  GtkWidget *w;
  char *font_name;

  if (!prefs_dialog)
    return;

  ND_GTK_GET(w, prefs_dialog, "font_mono");
  gtk_label_get(GTK_LABEL(w), &font_name);
  nd_dialog_fontsel(_("Select Monowidth Font"), font_name,
		    prefs_fontsel_cb, NULL); 
}
