/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <signal.h>
#include <gtk/gtk.h>

#include <interface.h>
#include <support.h>

#include <nd.h>
#include <nd_timestamp.h>
#include <nd_clipboard.h>
#include <nd_globals.h>
#include <nd_dialog.h>
#include <nd_fam.h>
#include <nd_filter_gui.h>
#include <nd_gui.h>
#include <nd_main.h>
#include <nd_recent.h>
#include <nd_macros.h>
#include <nd_plugin.h>
#include <nd_protocol_plugin.h>
#include <nd_prefs.h>
#include <nd_trace_registry.h>
#include <nd_tp.h>
#include <nd_raw_protocol.h>

#ifdef RELEASE
#  define WELCOMESTRING   "Welcome to Netdude, Version " VERSION "."
#else
#  define WELCOMESTRING   "Welcome to Netdude, DEVELOPMENT Version " VERSION ", " LAST_UPDATE "."
#endif

static GList         *filenames = NULL;

static void
main_help_exit(char *progname)
{
  char *dbug = "", *fam = "";

  /* We can't just use the ifdef within the string statement as it's inside
   * the _() macro, so we just set the string only when debugging is enabled.
   */

#ifdef ND_DEBUG
  dbug = _("  --debug, -d              Print debugging output.\n"
	   "  --ludicrous-debug, -dd   More debugging output.\n"
	   "  --ridiculous-debug, -ddd Guess :)\n");
#endif

#ifdef HAVE_LIBFAM
  fam =  _("  --no-fam                 Don't use FAM to monitor installed plugins.\n");
#endif

  printf(_("Netdude -- The Network Dump data Displayer and Editor.\n"
	   "USAGE: %s [OPTIONS] [tracefiles]\n"
	   "\n"
	   "  --help, -h, -?           This message.\n"
	   "%s"
	   "%s"
	   "  --prefix                 Installation prefix.\n"
	   "  --include-dir            Prefix directory of header files.\n"
	   "  --cflags                 Preprocessor flags needed to build plugins.\n"
	   "  --plugin-dir             Plugin installation directory.\n"
	   "  --proto-dir              Protocol installation directory.\n"
	   "  --plugins                Lists all successfully registered plugins.\n"
	   "  --version                Prints out version info.\n"
	   "  --version-maj            Prints out major version number.\n"
	   "\n"),
	 progname, dbug, fam);

  exit(0);
}


static void
main_runtime_debugoptions_read(int argc, char** argv)
{
  int i;

  for (i = 1; i < argc; i++)
    {
      if (!strcmp(argv[i], "--debug") ||
	  !strcmp(argv[i], "-d"))
	{
	  nd_runtime_options.debug = TRUE;
	}
      else if (!strcmp(argv[i], "--ludicrous-debug") ||
	       !strcmp(argv[i], "-dd"))
	{
	  nd_runtime_options.debug    = TRUE;
	  libnd_runtime_options.debug = TRUE;
	}
      else if (!strcmp(argv[i], "--ridiculous-debug") ||
	       !strcmp(argv[i], "-ddd"))
	{
	  nd_runtime_options.debug    = TRUE;
	  libnd_runtime_options.debug = TRUE;
	  pcapnav_runtime_options.debug = 1;
	}
    }
}


static void
main_runtime_options_read(int argc, char** argv)
{
  int i;

  for (i = 1; i < argc; i++)
    {
      if (!strcmp(argv[i], "-h")     ||
	  !strcmp(argv[i], "--help") ||
	  !strcmp(argv[i], "-?"))
	{
	  main_help_exit(argv[0]);
	}
      else if (!strcmp(argv[i], "--prefix"))
	{
	  printf("%s\n", PACKAGE_PREFIX);
	  exit(0);
	}
      else if (!strcmp(argv[i], "--plugin-dir"))
	{	  
	  printf("%s\n", nd_prefs_get_plugin_dir_global());
	  exit(0);
	}
      else if (!strcmp(argv[i], "--include-dir"))
	{
	  printf("%s\n", PACKAGE_INCLUDE_DIR"/netdude/"VERSION_MAJOR);
	  exit(0);
	}
      else if (!strcmp(argv[i], "--cflags"))
	{
	  printf("%s %s -I%s\n",
		 LIBNETDUDE_CFLAGS, GTK_CFLAGS,
		 PACKAGE_INCLUDE_DIR"/netdude/"VERSION_MAJOR);
	  exit(0);
	}
      else if (!strcmp(argv[i], "--proto-dir"))
	{
	  printf("%s\n", nd_prefs_get_proto_dir_global());
	  exit(0);
	}
      else if (!strcmp(argv[i], "--plugins"))
	{
	  nd_runtime_options.show_plugins = TRUE;
	}
      else if (!strcmp(argv[i], "--version"))
	{
	  printf("%s\n", VERSION);
	  exit(0);
	}
      else if (!strcmp(argv[i], "--version-maj"))
	{
	  printf("%s\n", VERSION_MAJOR);
	  exit(0);
	}
#ifdef HAVE_LIBFAM
      else if (!strcmp(argv[i], "--no-fam"))
	{
	  nd_runtime_options.use_fam = FALSE;
	}
#endif
      else if (argv[i][0] != '-')
	filenames = g_list_append(filenames, argv[i]);
    }
}


static void
main_dump_lnd_proto(LND_Protocol *proto, void *user_data)
{
  if (proto->id != 1)
    printf("%-32s%s\n", proto->name,
	   (proto->plugin ? proto->plugin->version() : _("Unknown")));

  return;
  TOUCH(user_data);
}

static void
main_dump_lnd_plugin(LND_Plugin *plugin, void *user_data)
{
  printf("%-32s%s\n",
	 libnd_plugin_get_name(plugin),
	 libnd_plugin_get_version(plugin));
  
  return;
  TOUCH(user_data);
}

static void
main_dump_nd_proto(ND_ProtoPlugin *plugin, void *user_data)
{
  printf("%-32s%s\n",
	 nd_proto_plugin_get_name(plugin),
	 nd_proto_plugin_get_version(plugin));
  
  return;
  TOUCH(user_data);
}

static void
main_dump_nd_plugin(ND_Plugin *plugin, void *user_data)
{
  printf("%-32s%s\n",
	 nd_plugin_get_name(plugin),
	 nd_plugin_get_version(plugin));
  
  return;
  TOUCH(user_data);
}

static void
main_runtime_options_apply(void)
{
  if (nd_runtime_options.show_plugins)
    {
      printf(_("libnetdude protocol plugins:\n"));
      printf("--------------------------------------------------\n");
      libnd_proto_registry_foreach_proto(main_dump_lnd_proto, NULL);

      printf(_("\nlibnetdude feature plugins:\n"));
      printf("--------------------------------------------------\n");
      libnd_plugin_foreach(main_dump_lnd_plugin, NULL);

      printf(_("\nNetdude protocol plugins:\n"));
      printf("--------------------------------------------------\n");
      nd_proto_plugin_foreach(main_dump_nd_proto, NULL);

      printf(_("\nNetdude feature plugins:\n"));
      printf("--------------------------------------------------\n");
      nd_plugin_foreach(main_dump_nd_plugin, NULL);

      exit(0);
    }
}


static gint
main_startup_idle_handler(gpointer data)
{
  gboolean libnd_init_success = GPOINTER_TO_INT(data);

  if (!libnd_init_success)
    {
      nd_dialog_message(_("Initialization problem."),
			_("Netdude could not initialize its components\n"
			  "properly. Make sure to specify the full path\n"
			  " to your tcpdump binary in the preferences dialog."),
			TRUE);
      return 0;
    }
  
  if (filenames)
    {
      LND_Trace *trace;
      GList     *l;
      char       canonical[MAXPATHLEN];
      char      *final;

      for (l = filenames; l; l = g_list_next(l))
	{
	  char *filename = (char *) l->data;

	  if (filename)
	    {
	      if (filename[0] != '/')
		{
		  char *file;
		  int   len;

		  if (!getcwd(canonical, MAXPATHLEN))
		    continue;
		  
		  len = strlen(canonical);
		  
		  file = canonical + len;
		  snprintf(canonical + len, MAXPATHLEN - len, "/%s",
			   filename);

		  final = canonical;
		}
	      else
		{
		  final = filename;
		}

	      trace = nd_trace_new(final);
	      nd_trace_registry_add(trace);
	    }
	}
    }
  
  return 0;
}


static void
main_shutdown_trace_cb(LND_Trace *trace, void *user_data)
{
  libnd_tcpdump_close(trace);
  libnd_tpm_free(trace->tpm);
  trace->tpm = NULL;

  return;
  TOUCH(user_data);
}


static void
main_exit_cleanup(void)
{
  nd_trace_registry_foreach(main_shutdown_trace_cb, NULL);
  nd_fam_shutdown();
}


static void
main_cleanup_sighandler(int sig)
{
  D(("Received sig %i -- cleanup.\n", sig));
  exit(0);

  TOUCH(sig);
}


static void
main_safe_exit_cb(LND_Trace *trace, guint *dirty_count)
{
  if (trace && trace->dirty)
    (*dirty_count)++;
}

void
nd_main_safe_exit(void)
{
  guint dirty_count = 0;

  nd_trace_registry_foreach((ND_TraceFunc)main_safe_exit_cb, &dirty_count);

  if (dirty_count == 0)
    nd_main_unsafe_exit();
  
  nd_dialog_exit();
}


void
nd_main_unsafe_exit(void)
{
  libnd_prefs_save();
  exit(0);
}


/* Paints a string with Netdude version number etc onto the
 * splash screen image.
 */
static void
main_setup_splash(GtkWidget *splash)
{
  char          label[MAXPATHLEN];
  GdkFont      *font;
  GtkWidget    *pixmap;
  GdkColor      fg;
  GdkGC        *gc;
  GdkColormap  *cmap;
  int           label_w, pix_w, pix_h;
  
  ND_GTK_GET(pixmap, splash, "splash_pixmap");
  font = gdk_font_load("-*-helvetica-medium-r-normal-*-8-*-*-*-p-*-*");
  cmap = gtk_widget_get_colormap(splash);
  memset(&fg, 0, sizeof(GdkColor));
  fg.red = 16384;
  fg.green = 16384;
  fg.blue = 16384;
  gdk_color_alloc(cmap, &fg);
  gc = gdk_gc_new(GTK_PIXMAP(pixmap)->pixmap);
  gdk_gc_set_foreground(gc, &fg);
  
  g_snprintf(label, MAXPATHLEN,
	     _("The NETwork DUmp data Displayer and Editor, %s %s"),
	     _("Version"), VERSION);
  
  label_w = gdk_string_width(font, label);
  gdk_window_get_size(GTK_PIXMAP(pixmap)->pixmap, &pix_w, &pix_h);

  gdk_draw_string(GTK_PIXMAP(pixmap)->pixmap, font, gc,
		  (pix_w - label_w) / 2, pix_h - 1, label);
}


static void
main_setup_sighandlers()
{
  signal(SIGALRM,   SIG_IGN);
  signal(SIGBUS,    main_cleanup_sighandler);
#ifdef SIGEMT
  signal(SIGEMT,    main_cleanup_sighandler);
#endif
  signal(SIGHUP,    main_cleanup_sighandler);
  signal(SIGTERM,   main_cleanup_sighandler);
  signal(SIGTRAP,   main_cleanup_sighandler);
  signal(SIGUSR1,   SIG_IGN);
  signal(SIGUSR2,   SIG_IGN);
#ifndef __EMX__
  signal(SIGIO,     main_cleanup_sighandler);
  signal(SIGIOT,    main_cleanup_sighandler);
#ifdef SIGSTKFLT
  signal(SIGSTKFLT, main_cleanup_sighandler);
#endif
  signal(SIGVTALRM, main_cleanup_sighandler);
  signal(SIGXCPU,   main_cleanup_sighandler);
  signal(SIGXFSZ,   main_cleanup_sighandler);
#endif
  signal(SIGPIPE,   SIG_IGN);
  signal(SIGQUIT,   main_cleanup_sighandler);
  signal(SIGFPE,    main_cleanup_sighandler);
  signal(SIGILL,    main_cleanup_sighandler);
#ifdef SIGSYS
  signal(SIGSYS,    main_cleanup_sighandler);
#endif
  signal(SIGINT,    main_cleanup_sighandler);

  /*  signal(SIGSEGV,   main_cleanup_sighandler); */

  atexit(main_exit_cleanup);
}


int
main (int argc, char *argv[])
{
  int show_splash = TRUE;
  gboolean libnd_init_success;

  main_setup_sighandlers();

#ifdef ENABLE_NLS
  bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
  textdomain (PACKAGE);
#endif

  memset(&nd_runtime_options, 0, sizeof(ND_RuntimeOptions));
  nd_runtime_options.use_fam = TRUE;

#ifdef ND_DEBUG
  main_runtime_debugoptions_read(argc, argv);
#endif  

  libnd_init_success = libnd_init();
  D(("libnetdude initialized\n"));

  main_runtime_options_read(argc, argv);

  gtk_set_locale ();
  gtk_init (&argc, &argv);

  add_pixmap_directory (PACKAGE_DATA_DIR "/" VERSION_MAJOR "/pixmaps");
  add_pixmap_directory (PACKAGE_SOURCE_DIR "/pixmaps");

  /*
   * The following code was added by Glade to create one of each component
   * (except popup menus), just so that you see something after building
   * the project. Delete any components that you don't want shown initially.
   */
  nd_toplevel_window = create_toplevel();

  nd_packet_init();
  nd_trace_init();
  nd_tp_init();
  nd_gui_init();
  nd_filter_dialog_init();
  nd_clipboard_init();
  nd_prefs_init();
  nd_fam_init();
  nd_raw_proto_init();
  nd_recent_init();
  nd_plugin_init();
  nd_proto_plugin_init();
  nd_trace_registry_init();
  nd_gui_statusbar_set(WELCOMESTRING);

  main_runtime_options_apply();

  gtk_widget_show(nd_toplevel_window);

  libnd_prefs_get_int_item(ND_DOM_NETDUDE, "show_splash", &show_splash);
  
  if (show_splash)
    {
      GtkWidget *splash;
      splash = create_splash_window();
      
      main_setup_splash(splash);
      gtk_widget_show (splash);
      gtk_timeout_add(4000, (GtkFunction) gtk_widget_hide, splash);
    }

  gtk_idle_add(main_startup_idle_handler, GINT_TO_POINTER(libnd_init_success));

  gtk_main ();

  return 0;
}

