/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <string.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>

#include <pcap.h>
#include <nd_info_dialog.h>
#include <nd_tp.h>
#include <nd_trace.h>
#include <callbacks.h>
#include <interface.h>
#include <support.h>


static GtkWidget *info_dialog = NULL;

static struct linktype_map {
	int      	dlt;
	char           *linktype;
} map[] = {
	{ DLT_NULL,		"DLT_NULL" }, 
	{ DLT_EN10MB,		"DLT_EN10MB" },
	{ DLT_EN3MB,		"DLT_EN3MB" }, 
	{ DLT_AX25,		"DLT_AX25" }, 
	{ DLT_PRONET,		"DLT_PRONET" },
	{ DLT_CHAOS,		"DLT_CHAOS" },
	{ DLT_IEEE802,		"DLT_TOKEN_RING" },
	{ DLT_ARCNET,		"DLT_ARCNET" },
	{ DLT_SLIP,		"DLT_SLIP" },
	{ DLT_PPP,		"DLT_PPP" },
	{ DLT_FDDI,	 	"DLT_FDDI" },
#ifdef DLT_ATM_RFC1483
	{ DLT_ATM_RFC1483, 	"DLT_ATM_RFC1483" },
#endif
#ifdef DLR_RAW
	{ DLT_RAW,		"DLT_RAW" },
#endif
#ifdef DLT_SLIP_BSDOS
	{ DLT_SLIP_BSDOS,	"DLT_SLIP_BSDOS" },
#endif
#ifdef DLT_PPP_BSDOS
	{ DLT_PPP_BSDOS,	"DLT_PPP_BSDOS" },
#endif
#ifdef DLT_C_HDLC
	{ DLT_C_HDLC,		"DLT_C_HDLC" },
#endif
#ifdef DLT_ATM_CLIP
	{ DLT_ATM_CLIP,		"DLT_ATM_CLIP" },
#endif
#ifdef DLT_PPP_SERIAL
	{ DLT_PPP_SERIAL,	"DLT_PPP_HDLC" },
#endif
#ifdef DLT_PPP_ETHER
	{ DLT_PPP_ETHER,	"DLT_PPP_ETHER" },
#endif
#ifdef DLT_IEEE802_11
	{ DLT_IEEE802_11,	"DLT_IEEE802_11" },
#endif
#ifdef DLT_LOOP
	{ DLT_LOOP,		"DLT_LOOP" },
#endif
#ifdef DLT_LINUX_SLL
	{ DLT_LINUX_SLL,	"DLT_LINUX_SLL" },
#endif
#ifdef DLT_TALK
	{ DLT_LTALK,		"DLT_LTALK" },
#endif
#ifdef DLT_ECONET
	{ DLT_ECONET,		"DLT_ECONET" },
#endif
#ifdef DLT_CISCO_IOS
	{ DLT_CISCO_IOS,	"DLT_CISCO_IOS" },
#endif
#ifdef DLT_PRISM_HEADER
	{ DLT_PRISM_HEADER,	"DLT_PRISM_HEADER" },
#endif
#ifdef DLT_AIRONET_HEADER
	{ DLT_AIRONET_HEADER,	"DLT_AIRONET_HEADER" },
#endif
	{ -1,			NULL }
};


void      
nd_info_dialog_show(LND_Trace *trace)
{
  struct bpf_timeval  delta, delta2;
  GtkWidget *w;
  char       s[MAXPATHLEN];
  char       s2[MAXPATHLEN];
  char      *sptr, *sptr2;
  int        i;

  if (!info_dialog)
    info_dialog = create_file_info_dialog();

  ND_GTK_GET(w, info_dialog, "file_info_filename");
  gtk_label_set_text(GTK_LABEL(w), trace->filename);

  ND_GTK_GET(w, info_dialog, "file_info_size_label");
  g_snprintf(s, MAXPATHLEN, "%lu", (long unsigned)
	     (trace->tpm->size + sizeof (struct pcap_file_header)));

  /* Separate every three digits by a ",". First
   * reverse the string, inserting a comma every
   * three digits... */
  for (i = strlen(s) - 1, sptr = s2; i >= 0; i -= 3)
    {
      *sptr++ = s[i];
      
      if (i-1 >= 0) *sptr++ = s[i-1];	
      if (i-2 >= 0) *sptr++ = s[i-2];
      if (i-2 > 0)  *sptr++ = ',';	
    }

  /* ... and then simply reverse that string.
   */
  sptr--;
  sptr2 = s;

  do *sptr2++ = *sptr--;
  while (sptr >= s2);

  *sptr2 = '\0';
  g_snprintf(s2, MAXPATHLEN, "%s Bytes", s);
  gtk_label_set_text(GTK_LABEL(w), s2);

  ND_GTK_GET(w, info_dialog, "file_info_tcpdump");
  g_snprintf(s, MAXPATHLEN, "%i.%i",
	     trace->tcpdump.pfh.version_major,
	     trace->tcpdump.pfh.version_minor);
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, info_dialog, "file_info_magic");
  g_snprintf(s, MAXPATHLEN, "%8X", trace->tcpdump.pfh.magic);
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, info_dialog, "file_info_dlt");
  for (i = 0; map[i].dlt >= 0; i++)
    {
      if (map[i].dlt == (int) trace->tcpdump.pfh.linktype)
	break;
    }

  if (map[i].dlt >= 0)
    gtk_label_set_text(GTK_LABEL(w), map[i].linktype);
  else
    {
      g_snprintf(s, MAXPATHLEN, _("unknown (%u)"),
		 trace->tcpdump.pfh.linktype);
      gtk_label_set_text(GTK_LABEL(w), s);
    }

  ND_GTK_GET(w, info_dialog, "file_info_snaplen");
  g_snprintf(s, MAXPATHLEN, _("%u bytes"), trace->tcpdump.pfh.snaplen);
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, info_dialog, "file_info_start_epoch");
  g_snprintf(s, MAXPATHLEN, "%lu.%lu",
	     (long unsigned) trace->tpm->base->start_ts.tv_sec,
	     (long unsigned) trace->tpm->base->start_ts.tv_usec);
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, info_dialog, "file_info_end_epoch");
  g_snprintf(s, MAXPATHLEN, "%lu.%lu",
	     (long unsigned) trace->tpm->base->end_ts.tv_sec,
	     (long unsigned) trace->tpm->base->end_ts.tv_usec);
  gtk_label_set_text(GTK_LABEL(w), s);

  pcapnav_timeval_sub(&trace->tpm->base->end_ts,
		      &trace->tpm->base->start_ts,
		      &delta);

  pcapnav_timeval_sub(&trace->tpm->current->pl->ph.ts,
		      &trace->tpm->base->start_ts,
		      &delta2);

  ND_GTK_GET(w, info_dialog, "file_info_span_epoch");
  g_snprintf(s, MAXPATHLEN, "%lu.%lu",
	     (long unsigned) delta.tv_sec,
	     (long unsigned) delta.tv_usec);
  gtk_label_set_text(GTK_LABEL(w), s);


  ND_GTK_GET(w, info_dialog, "file_info_start_time");
  libnd_misc_ctime(&trace->tpm->base->start_ts, s, MAXPATHLEN, TRUE, FALSE);
  s[strlen(s)-1] = '\0';
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, info_dialog, "file_info_end_time");
  libnd_misc_ctime(&trace->tpm->base->end_ts, s, MAXPATHLEN, TRUE, FALSE);
  s[strlen(s)-1] = '\0';
  gtk_label_set_text(GTK_LABEL(w), s);

  ND_GTK_GET(w, info_dialog, "file_info_span_time");
  g_snprintf(s, MAXPATHLEN, "%s", libnd_misc_timeval_to_string(&delta));
  gtk_label_set_text(GTK_LABEL(w), s);

  gtk_widget_show(info_dialog);
}


void      
nd_info_dialog_hide(void)
{
  if (info_dialog)
    gtk_widget_hide(info_dialog);
}

