/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#ifdef LINUX
#define __FAVOR_BSD
#define __USE_BSD
#endif

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <errno.h>

#include <callbacks.h>
#include <interface.h>
#include <support.h>

#include <nd.h>
#include <nd_dialog.h>
#include <nd_clipboard.h>
#include <nd_filter_gui.h>
#include <nd_globals.h>
#include <nd_gui.h>
#include <nd_main.h>
#include <nd_trace.h>
#include <nd_trace_registry.h>
#include <nd_tpm_gui.h>
#include <nd_tp.h>
#include <nd_plugin.h>
#include <nd_prefs.h>
#include <nd_recent.h>
#include <nd_trace_area_gui.h>
#include <nd_protocol_plugin.h>
#include <nd_protocol_inst.h>
#include <nd_info_dialog.h>

void
on_filesel_ok_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_filesel_ok(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


void
on_filesel_cancel_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_filesel_close();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


gboolean
on_toplevel_delete_event               (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
  nd_main_safe_exit();
  
  return TRUE;
  TOUCH(user_data);
  TOUCH(widget);
  TOUCH(event);
}



void
on_confirm_yes_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *w;
  
  w = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_widget_destroy(w);
  nd_trace_save(nd_trace_registry_get_current());
  nd_main_unsafe_exit();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_confirm_no_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *w;
  
  w = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_widget_destroy(w);
  nd_main_unsafe_exit();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_confirm_cancel_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *w;
  
  w = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_widget_destroy(w);

  return;
  TOUCH(user_data);
}


void
on_trace_list_row_move               (GtkCList        *clist,
				      gint             arg1,
				      gint             arg2,
				      gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);

  libnd_tp_move_packet(trace->tpm->current, arg1, arg2);

  return;
  TOUCH(clist);
  TOUCH(user_data);
}


void
on_packet_cut_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_clipboard_cut();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_packet_copy_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_clipboard_copy();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_packet_paste_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_clipboard_paste();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}



gboolean
on_trace_list_button_press_event     (GtkWidget       *widget,
				      GdkEventButton  *event,
				      gpointer         user_data)
{
  LND_Trace     *trace;
  int           row, col;

  D_ENTER;

  return_val_if_no_current_trace(trace, FALSE);
  nd_gui_timestamp_hide();

  if ((event->button == 1) &&
      ((event->state & GDK_MODIFIER_MASK) == 0))
    {
      libnd_tp_clear_selection(trace->tpm->current);
    }
  else if (event->button == 3)
    {
      gtk_clist_get_selection_info(GTK_CLIST(widget), event->x, event->y, &row, &col);

      if (libnd_tp_get_sel_size(trace->tpm->current) == 0)
	gtk_clist_select_row(GTK_CLIST(widget), row, 0);

      /* nd_packet_set(nd_trace_packet_get_nth(row)); */

      nd_gui_show_packet_menu(event);
    }

  D_RETURN_(TRUE);
  TOUCH(user_data);
}


void
on_trace_list_unselect_row           (GtkCList        *clist,
				      gint             row,
				      gint             column,
				      GdkEvent        *event,
				      gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  

  D_ENTER;
  libnd_tp_unselect_packet(trace->tpm->current, row);
  D_RETURN;

  TOUCH(user_data);
  TOUCH(clist);
  TOUCH(event);
  TOUCH(column);
}


void
on_trace_list_select_row             (GtkCList        *clist,
				      gint             row,
				      gint             column,
				      GdkEvent        *event,
				      gpointer         user_data)
{
  LND_Trace *trace;

  D_ENTER;
  return_if_no_current_trace(trace);

  nd_tp_select_packet(trace->tpm->current, row);
  
  D_RETURN;
  TOUCH(user_data);
  TOUCH(clist);
  TOUCH(event);
  TOUCH(column);
}

void
on_packet_unselect_all_activate        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace   *trace;

  D_ENTER;
  return_if_no_current_trace(trace);

  /* This called OUR callback -- no signals are sent,
   * so we need to do this manually
   */
  libnd_tp_clear_selection(trace->tpm->current);

  D_RETURN;
  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_packet_select_all_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace   *trace;

  D_ENTER;
  return_if_no_current_trace(trace);

  libnd_tp_full_selection(trace->tpm->current);

  D_RETURN;
  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_trace_list_end_selection          (GtkCList        *clist,
				      gpointer         user_data)
{
  return;
  TOUCH(clist);
  TOUCH(user_data);
}

void
on_packet_delete_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);

  D_ENTER;
  libnd_tp_sel_delete(trace->tpm->current);
  D_RETURN;

  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_trace_list_select_all             (GtkCList        *clist,
				      gpointer         user_data)
{
  LND_Trace *trace;

  D_ENTER;
  return_if_no_current_trace(trace);

  /* This one is called whenever the GTK clist fires a select-all
   * event, therefore we only need to inform libnetdude:
   */
  libnd_tp_full_selection(trace->tpm->current);

  D_RETURN;
  TOUCH(clist);
  TOUCH(user_data);
}


void
on_trace_list_unselect_all           (GtkCList        *clist,
				      gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);

  D_ENTER;
  libnd_tp_clear_selection(trace->tpm->current);
  D_RETURN;

  TOUCH(clist);
  TOUCH(user_data);
}


void
on_trace_list_size_allocate          (GtkWidget       *widget,
				      GtkAllocation   *allocation,
				      gpointer         user_data)
{
  gtk_clist_set_column_width(GTK_CLIST(widget), 0,
			     allocation->width - 60);

  return;
  TOUCH(user_data);
}


void
on_about_okay_button_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *dialog;

  dialog = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_widget_hide(dialog);

  return;
  TOUCH(user_data);
}


void
on_preferences_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_prefs_dialog_show();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_pref_ok_clicked                     (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_prefs_dialog_ok();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_pref_cancel_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_prefs_dialog_cancel();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_pref_apply_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_prefs_dialog_apply();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


gboolean
on_trace_win_bar_button_release_event  (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  return FALSE;
  TOUCH(widget);
  TOUCH(event);
  TOUCH(user_data);
}


void
on_plugin_activate                     (gpointer         user_data,
					gint             action,
					GtkWidget       *widget)
{
  ND_Plugin *plugin = (ND_Plugin *) user_data;

  D_ASSERT_PTR(plugin);
  nd_plugin_run(plugin);

  return;
  TOUCH(user_data);
  TOUCH(action);
  TOUCH(widget);
}


void
on_pref_tcpdump_loc_browse_button_clicked
                                        (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_prefs_select_tcpdump();
  
  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_tcpdump_error_okay_button_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *w;

  w = gtk_widget_get_toplevel(GTK_WIDGET(button));
  gtk_widget_destroy(w);

  return;
  TOUCH(user_data);
}


gboolean
on_trace_list_motion_notify_event    (GtkWidget       *widget,
				      GdkEventMotion  *event,
				      gpointer         user_data)
{
  nd_gui_timestamp_schedule(GTK_CLIST(widget), event);

  return FALSE;
  TOUCH(user_data);
}


gboolean
on_trace_list_leave_notify_event     (GtkWidget       *widget,
				      GdkEventCrossing *event,
				      gpointer         user_data)
{
  nd_gui_timestamp_hide();

  return FALSE;
  TOUCH(widget);
  TOUCH(event);
  TOUCH(user_data);
}


gboolean
on_trace_list_key_press_event        (GtkWidget       *widget,
				      GdkEventKey     *event,
				      gpointer         user_data)
{
  nd_gui_timestamp_hide();

  return FALSE;
  TOUCH(widget);
  TOUCH(event);
  TOUCH(user_data);
}


gboolean
on_splash_window_button_press_event    (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  gtk_widget_hide(widget);

  return FALSE;
  TOUCH(event);
  TOUCH(user_data);
}


void
on_recent_item0_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_recent_load(0);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_recent_item1_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_recent_load(1);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_recent_item2_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_recent_load(2);

  return;
  TOUCH(menuitem);
  TOUCH(user_data)
}


void
on_recent_item3_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_recent_load(3);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_recent_item4_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_recent_load(4);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_about_plugin_activate               (gpointer         user_data,
					gint             action,
					GtkWidget       *widget)
{
  ND_Plugin *plugin = (ND_Plugin *) user_data;

  D_ASSERT_PTR(plugin);
  nd_plugin_show_about(plugin);

  return;
  TOUCH(user_data);
  TOUCH(action);
  TOUCH(widget);
}


void
on_about_proto_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  ND_ProtoPlugin *pp;

  pp = (ND_ProtoPlugin*) gtk_object_get_data(GTK_OBJECT(menuitem), "plugin");
  nd_proto_plugin_show_about(pp);

  return;
  TOUCH(user_data);
}


void
on_trace_close_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  LND_Trace *trace = (LND_Trace *) user_data;

  nd_trace_close(trace);

  return;
  TOUCH(button);
}



gboolean
on_trace_menu_press_event              (GtkWidget       *widget,
					GdkEventButton  *event,
					gpointer         user_data)
{
  nd_gui_show_iterator_mode_menu(event);

  return TRUE;
  TOUCH(user_data);
  TOUCH(widget);
}


void
on_traces_notebook_switch_page         (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        gint             page_num,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  D_ENTER;
  trace = gtk_object_get_data(GTK_OBJECT(page->child), "trace");
  D_ASSERT_PTR(trace);

  nd_trace_registry_set_current(trace);

  D_RETURN;
  TOUCH(notebook);
  TOUCH(page_num);
  TOUCH(user_data);
}


void
on_protos_notebook_switch_page         (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        gint             page_num,
                                        gpointer         user_data)
{
  LND_Trace        *trace;
  ND_Trace         *trace_gui;
  LND_ProtoInfo    *pinf;
  LND_ProtoData    *pd;
  LND_ProtoInst     pi, *pi_ptr;
  LND_Protocol     *pcap_proto;

  trace = gtk_object_get_data(GTK_OBJECT(notebook), "trace");
  trace_gui = nd_trace_get(trace);

  if (!trace_gui)
    return;

  if (!trace_gui->cur_packet)
    return;

  pinf = gtk_object_get_data(GTK_OBJECT(page->child), "pinf");
  D_ASSERT_PTR(pinf);

  if (!pinf)
    return;

  /* Special case for pcap tab: libnd_packet_get_proto_data()
   * won't find information on this tab, so we check if that's
   * the desired protocol and bypass the "real" lookup in the
   * packet.
   */
  pcap_proto = libnd_proto_registry_find(LND_PROTO_LAYER_LINK, 0xa1b2c3d4);

  if (pcap_proto && pinf->inst.proto == pcap_proto)
    {
      libnd_proto_inst_init(&pi, pcap_proto, 0);
      pi_ptr = &pi;
    }
  else
    {
      pd = libnd_packet_get_proto_data(trace_gui->cur_packet, pinf->inst.proto, pinf->inst.nesting);
      pi_ptr = &pinf->inst;
    }

  nd_trace_set_current_proto_selection(trace, pi_ptr);
  nd_trace_set_current_proto(trace, pi_ptr, TRUE);
  
  return;
  TOUCH(page_num);
  TOUCH(user_data);
}

void
on_file_save_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_trace_save(nd_trace_registry_get_current());  

  return;
  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_file_save_as_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_save_as_dialog(trace, NULL, NULL);

  return;
  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_file_new_activate                   (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_trace_registry_add(nd_trace_new(NULL));

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_file_open_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_trace_load_dialog();
		    
  return;
  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_file_exit_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_main_safe_exit();

  return;
  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_about_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_dialog_about();

  return;
  TOUCH(user_data);
  TOUCH(menuitem);
}


void
on_ip_entry_ok_clicked                 (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_ip_ok();

  return;
  TOUCH(button); TOUCH(user_data);
}


void
on_ip_entry_cancel_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_ip_cancel();

  return;
  TOUCH(button); TOUCH(user_data);
}



void
on_change_mono_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_prefs_fontsel_show();

  return;
  TOUCH(button); TOUCH(user_data);
}


void
on_font_ok_button_clicked              (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *nd_toplevel_window;

  nd_toplevel_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
  nd_dialog_fontsel_ok(GTK_FONT_SELECTION_DIALOG(nd_toplevel_window));

  return;
  TOUCH(user_data);
}


void
on_font_cancel_button_clicked          (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *nd_toplevel_window;

  nd_toplevel_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
  nd_dialog_fontsel_cancel(GTK_FONT_SELECTION_DIALOG(nd_toplevel_window));

  return;
  TOUCH(user_data);
}


void
on_font_apply_button_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  GtkWidget *nd_toplevel_window;

  nd_toplevel_window = gtk_widget_get_toplevel(GTK_WIDGET(button));
  nd_dialog_fontsel_apply(GTK_FONT_SELECTION_DIALOG(nd_toplevel_window));

  return;
  TOUCH(user_data);
}


gboolean
on_number_spin_key_press_event         (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
  GtkWidget *w;
  GtkWidget *w2;

  D_ENTER;

  w = gtk_widget_get_toplevel(GTK_WIDGET(widget));
  w2 = gtk_object_get_data(GTK_OBJECT(w), "number_ok_button");
  D_ASSERT_PTR(w2);

  if (event->keyval == GDK_Return)
    on_number_ok_clicked(GTK_BUTTON(w2), user_data);
  
  D_RETURN_(FALSE);
}


void
on_number_decimal_toggled              (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  GtkWidget *w, *entry;
  guint      value;
  char      *errptr;

  D_ENTER;

  if (gtk_toggle_button_get_active(togglebutton))
    {
      w = gtk_widget_get_toplevel(GTK_WIDGET(togglebutton));
      entry = gtk_object_get_data(GTK_OBJECT(w), "generic_entry");
      D_ASSERT_PTR(entry);
      
      value = strtoul(gtk_entry_get_text(GTK_ENTRY(entry)), &errptr, 16);
      
      if ((*errptr != '\0') || (errno == ERANGE))
	{
	  D(("Warning -- conversion error when converting %s\n",
	     gtk_entry_get_text(GTK_ENTRY(entry))));
	}
      else
	gtk_object_set_data(GTK_OBJECT(w), "value", GINT_TO_POINTER(value));
      
      nd_dialog_number_update();
      nd_dialog_number_set_base(ND_BASE_DEC);
    }
  
  D_RETURN;
  TOUCH(user_data);
}


void
on_number_hex_toggled                  (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  GtkWidget *w, *entry;
  guint      value;
  char      *errptr;
  
  D_ENTER;
  
  if (gtk_toggle_button_get_active(togglebutton))
    {
      w = gtk_widget_get_toplevel(GTK_WIDGET(togglebutton));
      entry = gtk_object_get_data(GTK_OBJECT(w), "generic_spin");
      D_ASSERT_PTR(entry);
      
      value = strtoul(gtk_entry_get_text(GTK_ENTRY(entry)), &errptr, 10);
      
      if ((*errptr != '\0') || (errno == ERANGE))
	{
	  D(("Warning -- conversion error when converting %s\n", 
	     gtk_entry_get_text(GTK_ENTRY(entry))));
	}
      else
	gtk_object_set_data(GTK_OBJECT(w), "value", GINT_TO_POINTER(value));
      
      nd_dialog_number_update();
      nd_dialog_number_set_base(ND_BASE_HEX);
    }

  D_RETURN;
  TOUCH(user_data);
}


void
on_number_ok_clicked                   (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_number_ok(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


void
on_number_cancel_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_number_cancel(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


gboolean
on_largenumber_entry_key_press_event    (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
  GtkWidget *w;
  GtkWidget *w2;

  w = gtk_widget_get_toplevel(GTK_WIDGET(widget));
  w2 = gtk_object_get_data(GTK_OBJECT(w), "largenumber_ok");
  D_ASSERT_PTR(w2);

  if (event->keyval == GDK_Return)
    on_largenumber_ok_clicked(GTK_BUTTON(w2), user_data);
  
  return TRUE;
}


void
on_largenumber_ok_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_largenumber_ok(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


void
on_largenumber_cancel_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  /* Same handling as for normal dialog :) */
  nd_dialog_number_cancel(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


void
on_largenumber_up_clicked               (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_largenumber_up(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


void
on_largenumber_down_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_largenumber_down(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


void
on_string_ok_button_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_string_ok(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


void
on_string_cancel_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_dialog_string_cancel(gtk_widget_get_toplevel(GTK_WIDGET(button)));

  return;
  TOUCH(user_data);
}


gboolean
on_string_entry_key_press_event        (GtkWidget       *widget,
                                        GdkEventKey     *event,
                                        gpointer         user_data)
{
  if (event->keyval == GDK_Return)
    nd_dialog_string_ok(gtk_widget_get_toplevel(widget));

  return FALSE;
  TOUCH(user_data);
}


void
on_pref_tmpdir_browse_button_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_reload_plugins_activate             (gpointer         user_data,
					gint             action,
					GtkWidget       *widget)
{
  nd_plugin_init();

  return;
  TOUCH(user_data);
  TOUCH(action);
  TOUCH(widget);
}


void
on_apply_to_selection_activate         (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);

  if (GTK_CHECK_MENU_ITEM(menuitem)->active)
    {
      trace->iterator_mode = LND_PACKET_IT_SEL_RW;
      nd_gui_iterator_mode_set(trace);
      D(("Switching to selection-only mode.\n"));
    }
  
  return;
  TOUCH(user_data);
}


void
on_apply_to_trace_area_activate        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);

  if (GTK_CHECK_MENU_ITEM(menuitem)->active)
    {
      trace->iterator_mode = LND_PACKET_IT_AREA_RW;
      nd_gui_iterator_mode_set(trace);
      D(("Switching to area-rw mode.\n"));
    }

  return;
  TOUCH(user_data);
}


void
on_other_trace_area_activate           (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_tpm_dialog_show();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_nav_load_button_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_tpm_dialog_apply();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_nav_cancel_button_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_tpm_dialog_hide();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_nav_type_radiobutton_toggled        (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
  LND_TPM_NavMode mode;

  mode = nd_tpm_dialog_get_mode();
  nd_tpm_dialog_set_mode(mode);

  return;
  TOUCH(togglebutton);
  TOUCH(user_data);
}


void
on_nav_time_entries_changed            (GtkEditable     *editable,
                                        gpointer         user_data)
{
  nd_tpm_dialog_sync_time_scale();

  return;
  TOUCH(editable);
  TOUCH(user_data);
}


void
on_nav_prev_button_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_tpm_dialog_prev_part();
  
  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_nav_next_button_clicked             (GtkButton       *button,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_tpm_dialog_next_part();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_file_info_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_info_dialog_show(trace);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_file_info_close_button_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_info_dialog_hide();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_filters_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_filter_dialog_show();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_filter_create_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_filter_dialog_create();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_filter_ok_button_clicked            (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_filter_dialog_hide();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_filter_modify_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_filter_dialog_modify();
 
  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_filter_delete_button_clicked        (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_filter_dialog_delete();
 
  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_filter_update_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_filter_dialog_apply_filters();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_filter_cancel_button_clicked  (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_filter_dialog_cancel();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_filtering_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_trace_filter_dialog_show();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_next_packet_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_NEXT);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_previous_packet_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_PREV);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_next_with_same_protocol_activate    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_NEXT_PROT);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_previous_with_same_protocol_activate
                                        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_PREV_PROT);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_next_selected_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_NEXT_SEL);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_previous_selected_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_PREV_SEL);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_file_menu_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_gui_sync_file_menu();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_edit_menu_activate                  (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_gui_sync_edit_menu();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}



void
on_go_menu_activate                    (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_gui_sync_go_menu();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_protocol_menu_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_gui_sync_protocol_menu();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}

void
on_view_fixed_button_clicked           (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_tpm_dialog_show();

  return;
  TOUCH(button); TOUCH(user_data);
}


gboolean
on_view_fixed_button_press_event       (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  nd_tpm_dialog_show();

  return FALSE;
  TOUCH(widget); TOUCH(event); TOUCH(user_data);
}


void
on_packet_filter_activate              (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  libnd_tp_sel_filter(trace->tpm->current);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_packet_unfilter_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  libnd_tp_sel_unfilter(trace->tpm->current);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_filtered_delete_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  /* FIXME libnd_trace_delete_hidden_packets(trace); */

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_next_unfiltered_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_NEXT_UNFILTERED);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_previous_unfiltered_activate        (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_goto(trace, ND_TRACE_JUMP_PREV_UNFILTERED);

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}

gboolean
on_ip_button_release_event             (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{
  if (GTK_SPIN_BUTTON(widget)->in_child)
    gtk_entry_select_region(GTK_ENTRY(widget), 0, -1);

  return FALSE;
  TOUCH(event); TOUCH(user_data);
}


void
on_pit_time_entries_changed            (GtkEditable     *editable,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_sync_time_scale((gboolean) GPOINTER_TO_INT(user_data));

  return;
  TOUCH(editable);
  TOUCH(user_data);
}


void
on_view_fixed_area_button_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_show();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_new_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_new_area();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_del_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_del_area();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_set_button_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_set_area();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_part_button_clicked      (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_set_current_part();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_current_button_clicked   (GtkButton       *button,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_area_dialog_sync(&trace->area);

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_trace_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_sync(NULL);

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_apply_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_apply();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_trace_area_close_button_clicked     (GtkButton       *button,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_hide();

  return;
  TOUCH(button);
  TOUCH(user_data);
}


void
on_area_list_select                    (GtkList         *list,
                                        GtkWidget       *widget,
                                        gpointer         user_data)
{
  LND_TraceArea *area;

  ND_GTK_GET(area, widget, "trace_area");
  if (area)
    nd_trace_area_dialog_select_area(area);

  return;
  TOUCH(list);
  TOUCH(user_data);
}



void
on_trace_areas_activate                (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_trace_area_dialog_show();

  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}


void
on_netdude_output_activate             (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  nd_runtime_options.debug = GTK_CHECK_MENU_ITEM(menuitem)->active;
  return;
  TOUCH(user_data);
}


void
on_libnetdude_output_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  libnd_runtime_options.debug = GTK_CHECK_MENU_ITEM(menuitem)->active;
  return;
  TOUCH(user_data);  
}


void
on_pcapnav_output_activate             (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  pcapnav_runtime_options.debug = GTK_CHECK_MENU_ITEM(menuitem)->active;  
  return;
  TOUCH(user_data);
}

void
on_file_close_activate                 (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
  LND_Trace *trace;

  return_if_no_current_trace(trace);
  nd_trace_close(trace);
  
  return;
  TOUCH(menuitem);
  TOUCH(user_data);
}

