/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <nd.h>
#include <nd_gui.h>

#include <nd_vlan.h>
#include <nd_vlan_callbacks.h>

static ND_ProtoField vlan_fields[] = {
  { ND_VAL_FIELD, N_("TCI (%d)"),   N_("Tag Control Information"), 16, nd_vlan_tci_cb  },
  { ND_VAL_FIELD, N_("Proto (%s)"), N_("Protocol"),                16, nd_vlan_proto_cb  },
  { 0, NULL, NULL, 0, NULL }
};

ND_MenuData vlan_menu_type_data[] = {
  { "IP",         N_("IP (0x0800)"),              ETHERTYPE_IP,     nd_vlan_proto_value_cb },
  { "ARP",        N_("ARP (0x0806)"),             ETHERTYPE_ARP,    nd_vlan_proto_value_cb },
  { "RARP",       N_("RARP (0x8035)"),            ETHERTYPE_REVARP, nd_vlan_proto_value_cb },
  { "Custom",     N_("Custom type value"),        -1,               nd_vlan_proto_custom_cb },
  { NULL, NULL, 0, NULL}
};

static LND_Protocol *vlan;
static ND_Protocol  *vlan_gui;


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return ("VLAN Plugin");
}


const char *
description(void)
{
  return ("A small plugin providing support for VLAN-tagged Ethernet.");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION_MAJOR;
}


LND_Protocol *
init(void)
{
  if (! (vlan = libnd_proto_registry_find(LND_PROTO_LAYER_NET, LND_PROTO_VLAN)))
    return NULL;
  
  vlan_gui = nd_proto_new(vlan);
  
  vlan_gui->create_gui      = nd_vlan_create_gui;
  vlan_gui->set_gui         = nd_vlan_set_gui;

  /* We're using a button table to display the protocol content,
     so we need to hook it in here: */
  vlan_gui->fields = vlan_fields;
  vlan_gui->header_width = 32;

  return vlan;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_vlan_create_gui(LND_Trace *trace, LND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);
  return table;
}


void       
nd_vlan_set_gui(const LND_Packet *packet, LND_ProtoInfo *pinf)
{
  struct lnd_vlan_header *vh;
  
  vh = (struct lnd_vlan_header*) libnd_packet_get_data(packet, vlan, pinf->inst.nesting);

  nd_vlan_set_gui_tci(pinf, vh);
  nd_vlan_set_gui_proto(pinf, vh);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_vlan_set_gui_tci(LND_ProtoInfo *pinf, struct lnd_vlan_header *vh)
{
  nd_proto_field_set(pinf, &vlan_fields[0], DATA_TO_PTR(ntohs(vh->vlan_tci)));
}


void
nd_vlan_set_gui_proto(LND_ProtoInfo *pinf, struct lnd_vlan_header *vh)
{
  GtkWidget *button;
  
  nd_proto_field_set_for_menu(pinf, &vlan_fields[1], DATA_TO_PTR(ntohs(vh->vlan_proto)),
			      vlan_menu_type_data, "0x%.4x");
}


LND_Protocol *
nd_vlan_get(void)
{
  return vlan;
}


ND_Protocol *
nd_vlan_get_gui(void)
{
  return vlan_gui;
}

