/*

Copyright (C) 2000 - 2010 Christian Kreibich <christian@whoop.org>.

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to
deal in the Software without restriction, including without limitation the
rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
sell copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies of the Software and its documentation and acknowledgment shall be
given in the documentation and software packages that this Software was
used.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
THE AUTHORS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

*/

#include <nd.h>
#include <nd_gui.h>
#include <nd_protocol_inst.h>

#include <nd_arp.h>
#include <nd_arp_callbacks.h>


static ND_ProtoField arp_fields[] = {
  { ND_VAL_FIELD, N_("HW type\n(%s)"),            N_("Hardware address type"),    16, nd_arp_hrd_cb },
  { ND_VAL_FIELD, N_("Prot. type\n(0x%.4x)"),     N_("Protocol address type"),    16, nd_arp_pro_cb },
  { ND_VAL_FIELD, N_("HL\n(%u)"),                 N_("Hardware address length"),  8,  nd_arp_hln_cb },
  { ND_VAL_FIELD, N_("PL\n(%u)"),                 N_("Protocol address length"),  8,  nd_arp_pln_cb },
  { ND_VAL_FIELD, N_("Op\n(%s)"),                 N_("ARP operation"),            16, nd_arp_op_cb  },
  { ND_VAL_FIELD, N_("HW sender addr.\n(%s)"),    N_("Hardware sender address"),  48, nd_arp_sha_cb },
  { ND_VAL_FIELD, N_("Proto sender addr.\n(%s)"), N_("Protocol sender address"),  32, nd_arp_spr_cb },
  { ND_VAL_FIELD, N_("HW target addr.\n(%s)"),    N_("Hardware target address"),  48, nd_arp_dha_cb },
  { ND_VAL_FIELD, N_("Proto target addr.\n(%s)"), N_("Protocol target address"),  32, nd_arp_dpr_cb },
  { 0, NULL, NULL, 0, NULL }
};

ND_MenuData arp_menu_hrd_data[] = {
#ifdef ARPHRD_NETROM
  { N_("Net/Rom"),     NULL,                   ARPHRD_NETROM,   nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_ETHER
  { N_("Ether"),       NULL,                   ARPHRD_ETHER,    nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_EETHER
  { N_("Exp. Ether"),  NULL,                   ARPHRD_EETHER,   nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_AX25
  { N_("AX.25"),       NULL,                   ARPHRD_AX25,     nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_PRONET
  { N_("PROnet"),      NULL,                   ARPHRD_PRONET,   nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_CHAOS
  { N_("Chaos"),       NULL,                   ARPHRD_CHAOS,    nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_IEEE802
  { N_("IEEE 802.2"),  NULL,                   ARPHRD_IEEE802,  nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_ARCNET
  { N_("ARCnet"),      NULL,                   ARPHRD_ARCNET,   nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_APPLETLK
  { N_("Appletalk"),   NULL,                   ARPHRD_APPLETLK, nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_DLCI
  { N_("DLCI"),        N_("Frame Relay DLCI"), ARPHRD_DLCI,     nd_arp_hrd_value_cb },
#endif
#ifdef ARPHRD_ATM
  { N_("ATM"),         NULL,                   ARPHRD_ATM,      nd_arp_hrd_value_cb },
#endif
  { N_("Custom"),      N_("Custom hardware value"), -1,         nd_arp_hrd_custom_cb },
  { NULL, NULL, 0, NULL }
};


ND_MenuData arp_menu_op_data[] = {
#ifdef ARPOP_REQUEST
  { N_("ARP req"),   N_("ARP request"),   ARPOP_REQUEST,   nd_arp_op_value_cb },
#endif
#ifdef ARPOP_REPLY
  { N_("ARP rep"),   N_("ARP reply"),     ARPOP_REPLY,     nd_arp_op_value_cb },
#endif
#ifdef ARPOP_RREQUEST
  { N_("RARP req"),  N_("RARP request"),  ARPOP_RREQUEST,  nd_arp_op_value_cb },
#endif
#ifdef ARPOP_RREPLY
  { N_("RARP rep"),  N_("RARP reply"),    ARPOP_RREPLY,    nd_arp_op_value_cb },
#endif
#ifdef ARPOP_InREQUEST
  { N_("InARP req"), N_("InARP request"), ARPOP_InREQUEST, nd_arp_op_value_cb },
#endif
#ifdef ARPOP_InREPLY
  { N_("InARP rep"), N_("InARP reply"),   ARPOP_InREPLY,   nd_arp_op_value_cb },
#endif
#ifdef ARPOP_NAK
  { N_("ARM ARP NAK"),   NULL, ARPOP_NAK,       nd_arp_op_value_cb },
#endif
  { N_("Custom"),        N_("Custom protocol value"), -1, nd_arp_op_custom_cb },
  { NULL, NULL, 0, NULL }
};


static LND_Protocol *arp;
static ND_Protocol  *arp_gui;


/* Plugin hook implementations: ---------------------------------------- */

const char *
name(void)
{
  return _("ARP Plugin");
}


const char *
description(void)
{
  return _("A plugin providing support for the ARP protocol.\n");
}


const char *
author(void)
{
  return ("Christian Kreibich, <christian@whoop.org>");
}


const char *
version(void)
{
  return VERSION_MAJOR;
}


LND_Protocol *
init(void)
{
  if (! (arp = libnd_proto_registry_find(LND_PROTO_LAYER_NET, ETHERTYPE_ARP)))
    return NULL;

  arp_gui = nd_proto_new(arp);
			     		
  arp_gui->create_gui      = nd_arp_create_gui;
  arp_gui->set_gui         = nd_arp_set_gui;

  /* We're using a button table to display the protocol content,
     so we need to hook them in here: */
  arp_gui->fields = arp_fields;
  arp_gui->header_width = 28 * 8; /* only valid for Ethernet/IP, but corrected later */
	     
  return arp;
}


/* Protocol method implementations: ------------------------------------ */

GtkWidget *
nd_arp_create_gui(LND_Trace *trace, LND_ProtoInfo *pinf)
{
  GtkWidget *table;

  table = nd_gui_proto_table_create(trace, pinf);

  return table;
}


void       
nd_arp_set_gui(const LND_Packet *packet, LND_ProtoInfo *pinf)
{
  GtkLabel *label;
  struct arphdr   *arphdr;

  arphdr = (struct arphdr*) libnd_packet_get_data(packet, arp, pinf->inst.nesting);
  if (!arphdr)
    return;

  label = GTK_LABEL(nd_proto_info_get(pinf)->proto_label);

  switch (ntohs(arphdr->ar_op))
    {
#ifdef ARPOP_RREQUEST
    case ARPOP_RREQUEST:
      gtk_label_set_text(label, _("RARP"));
      break;
#endif
#ifdef ARPOP_RREPLY
    case ARPOP_RREPLY:
      gtk_label_set_text(label, _("RARP"));
      break;
#endif
#ifdef ARPOP_InREQUEST
    case ARPOP_InREQUEST:
      gtk_label_set_text(label, _("InARP"));
      break;
#endif
#ifdef ARPOP_InREPLY
    case ARPOP_InREPLY:
      gtk_label_set_text(label, _("InARP"));
      break;
#endif
    default:
      gtk_label_set_text(label, _("ARP"));
    }

  nd_arp_set_gui_hrd(pinf, arphdr);
  nd_arp_set_gui_pro(pinf, arphdr);
  nd_arp_set_gui_hln(pinf, arphdr);
  nd_arp_set_gui_pln(pinf, arphdr);
  nd_arp_set_gui_op(pinf, arphdr);
  nd_arp_set_gui_addresses(pinf, arphdr, packet);
}


/* Misc helper stuff below --------------------------------------------- */

void
nd_arp_set_gui_hrd(LND_ProtoInfo *pinf, struct arphdr *arphdr)
{
  nd_proto_field_set_for_menu(pinf, &arp_fields[0], DATA_TO_PTR(ntohs(arphdr->ar_hrd)),
			      arp_menu_hrd_data, "%u");
}


void
nd_arp_set_gui_pro(LND_ProtoInfo *pinf, struct arphdr *arphdr)
{
  nd_proto_field_set(pinf, &arp_fields[1], DATA_TO_PTR(ntohs(arphdr->ar_pro)));
}


void
nd_arp_set_gui_hln(LND_ProtoInfo *pinf, struct arphdr *arphdr)
{
  nd_proto_field_set(pinf, &arp_fields[2], DATA_TO_PTR(arphdr->ar_hln));
}


void
nd_arp_set_gui_pln(LND_ProtoInfo *pinf, struct arphdr *arphdr)
{
  nd_proto_field_set(pinf, &arp_fields[3], DATA_TO_PTR(arphdr->ar_pln));
}


void
nd_arp_set_gui_op(LND_ProtoInfo *pinf, struct arphdr *arphdr)
{
  nd_proto_field_set_for_menu(pinf, &arp_fields[4], DATA_TO_PTR(ntohs(arphdr->ar_op)),
			      arp_menu_op_data, "%u");
}


void
nd_arp_set_gui_addresses(LND_ProtoInfo *pinf,
			 struct arphdr *arphdr,
			 const LND_Packet *packet)
     
{
  GtkWidget      *button;
  GtkTable       *table;
  int             i, real_len;
  char            label[MAXPATHLEN];
  gboolean        button_visible[4] = { FALSE, FALSE, FALSE, FALSE };

  /* Check if the address lengths are actually usable, i.e. we can
     display the header. If not, hide the addresses.
  */

  if ((guchar *) arphdr + 8 + 2 * (arphdr->ar_hln + arphdr->ar_pln) > libnd_packet_get_end(packet))
    {
      button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[5]));
      gtk_widget_hide(button);
      button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[6]));
      gtk_widget_hide(button);
      button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[7]));
      gtk_widget_hide(button);
      button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[8]));
      gtk_widget_hide(button);

      return;
    }

  real_len = (libnd_packet_get_end(packet) - ((guchar*) arphdr)) * 8;

  table = GTK_TABLE(nd_proto_info_get(pinf)->proto_gui);
  D_ASSERT_PTR(table);
  i = 64;

  button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[5]));
  D_ASSERT_PTR(button);
  gtk_widget_show(button);

  gtk_container_remove(GTK_CONTAINER(table), button);

  if (i + arphdr->ar_hln * 8 <= real_len)
    {
      gtk_table_attach (GTK_TABLE (table), button,
			i, i + arphdr->ar_hln * 8, 0, 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) 0, 0, 0);
      button_visible[0] = TRUE;
    }
  arp_gui->fields[5].bits = arphdr->ar_hln;
  i += arphdr->ar_hln * 8;


  button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[6]));
  D_ASSERT_PTR(button);
  gtk_widget_show(button);

  gtk_container_remove(GTK_CONTAINER(table), button);

  if (i + arphdr->ar_pln * 8 <= real_len)
    {
      gtk_table_attach (GTK_TABLE (table), button,
			i, i + arphdr->ar_pln * 8, 0, 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) 0, 0, 0);
      button_visible[1] = TRUE;
    }
  arp_gui->fields[6].bits = arphdr->ar_pln;
  i += arphdr->ar_pln * 8;
  

  button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[7]));
  D_ASSERT_PTR(button);
  gtk_widget_show(button);

  gtk_container_remove(GTK_CONTAINER(table), button);

  if (i + arphdr->ar_hln * 8 <= real_len)
    {
      gtk_table_attach (GTK_TABLE (table), button,
			i, i + arphdr->ar_hln * 8, 0, 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) 0, 0, 0);
      button_visible[2] = TRUE;
    }
  arp_gui->fields[7].bits = arphdr->ar_hln;
  i += arphdr->ar_hln * 8;
			     
  
  button = libnd_reg_get_data(pinf->registry, nd_proto_field_to_string(&arp_gui->fields[8]));
  D_ASSERT_PTR(button);
  gtk_widget_show(button);
  
  gtk_container_remove(GTK_CONTAINER(table), button);

  if (i + arphdr->ar_pln * 8 <= real_len)
    {
      gtk_table_attach (GTK_TABLE (table), button,
			i, i + arphdr->ar_pln * 8, 0, 1,
			(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
			(GtkAttachOptions) 0, 0, 0);
      button_visible[3] = TRUE;
    }
  arp_gui->fields[8].bits = arphdr->ar_pln;

  
  if (button_visible[0])
    {
      libnd_misc_get_hardware_string(label, MAXPATHLEN,
				     ((guchar *) arphdr) + 8,
				     arphdr->ar_hln);
      nd_proto_field_set(pinf, &arp_fields[5], label);
    }

  if (button_visible[2])
    {
      libnd_misc_get_hardware_string(label, MAXPATHLEN,
				     ((guchar *) arphdr) + 8 + arphdr->ar_hln + arphdr->ar_pln,
				     arphdr->ar_hln);
      nd_proto_field_set(pinf, &arp_fields[7], label);
    }

  
  if (ntohs(arphdr->ar_pro) == ETHERTYPE_IP)
    {
      struct in_addr in;

      if (button_visible[1])
	{
	  memcpy(&in, ((guchar *) arphdr) + 8 + arphdr->ar_hln, sizeof(struct in_addr));
	  nd_proto_field_set(pinf, &arp_fields[6], inet_ntoa(in));
	}
      
      if (button_visible[3])
	{
	  memcpy(&in, ((guchar *) arphdr) + 8 + 2 * arphdr->ar_hln + arphdr->ar_pln, sizeof(struct in_addr));
	  nd_proto_field_set(pinf, &arp_fields[8], inet_ntoa(in));
	}
    }
  else
    {
      if (button_visible[1])
	{
	  libnd_misc_get_hardware_string(label, MAXPATHLEN,
					 ((guchar *) arphdr) + 8 + arphdr->ar_hln,
					 arphdr->ar_pln);
	  nd_proto_field_set(pinf, &arp_fields[6], label);
	}
      
      if (button_visible[3])
	{
	  libnd_misc_get_hardware_string(label, MAXPATHLEN,
					 ((guchar *) arphdr) + 8 + 2 * arphdr->ar_hln + arphdr->ar_pln,
					 arphdr->ar_pln);
	  nd_proto_field_set(pinf, &arp_fields[8], label);
	}
    }
}


ND_Protocol *
nd_arp_get_gui(void)
{
  return arp_gui;
}


LND_Protocol *
nd_arp_get(void)
{
  return arp;
}

