local helpers = require('test.functional.helpers')(after_each)
local Screen = require('test.functional.ui.screen')
local clear = helpers.clear
local feed, command, insert = helpers.feed, helpers.command, helpers.insert
local eq = helpers.eq
local funcs = helpers.funcs
local meths = helpers.meths
local curwin = helpers.curwin
local poke_eventloop = helpers.poke_eventloop


describe('ext_multigrid', function()
  local screen

  before_each(function()
    clear{args_rm={'--headless'}, args={'--cmd', 'set laststatus=2'}}
    screen = Screen.new(53,14)
    screen:attach({ext_multigrid=true})
    screen:set_default_attr_ids({
      [1] = {bold = true, foreground = Screen.colors.Blue1},
      [2] = {foreground = Screen.colors.Magenta},
      [3] = {foreground = Screen.colors.Brown, bold = true},
      [4] = {foreground = Screen.colors.SlateBlue},
      [5] = {bold = true, foreground = Screen.colors.SlateBlue},
      [6] = {foreground = Screen.colors.Cyan4},
      [7] = {bold = true},
      [8] = {underline = true, bold = true, foreground = Screen.colors.SlateBlue},
      [9] = {foreground = Screen.colors.SlateBlue, underline = true},
      [10] = {foreground = Screen.colors.Red},
      [11] = {bold = true, reverse = true},
      [12] = {reverse = true},
      [13] = {foreground = Screen.colors.DarkBlue, background = Screen.colors.LightGrey},
      [14] = {foreground = Screen.colors.Grey100, background = Screen.colors.Red},
      [15] = {bold = true, foreground = Screen.colors.SeaGreen4},
      [16] = {background = Screen.colors.LightGrey, underline = true},
      [17] = {background = Screen.colors.LightGrey, underline = true, bold = true, foreground = Screen.colors.Magenta},
      [18] = {bold = true, foreground = Screen.colors.Magenta},
      [19] = {foreground = Screen.colors.Brown},
      [20] = {background = Screen.colors.LightGrey},
      [21] = {background = Screen.colors.LightMagenta},
      [22] = {background = Screen.colors.LightMagenta, bold = true, foreground = Screen.colors.Blue},
      [23] = {background = Screen.colors.Grey90},
      [24] = {background = Screen.colors.Grey},
    })
  end)

  it('default initial screen', function()
    screen:expect{grid=[[
    ## grid 1
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ]]}
  end)

  it('positions windows correctly', function()
    command('vsplit')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]], condition=function()
      eq({
        [2] = { win = {id=1000}, startrow = 0, startcol = 27, width = 26, height = 12 },
        [4] = { win = {id=1001}, startrow = 0, startcol =  0, width = 26, height = 12 }
      }, screen.win_position)
    end}
    command('wincmd l')
    command('split')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│{11:[No Name]                 }|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      {12:[No Name]                  [No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]], condition=function()
      eq({
        [2] = { win = {id=1000}, startrow = 7, startcol = 27, width = 26, height =  5 },
        [4] = { win = {id=1001}, startrow = 0, startcol =  0, width = 26, height = 12 },
        [5] = { win = {id=1002}, startrow = 0, startcol = 27, width = 26, height =  6 }
      }, screen.win_position)
    end}
    command('wincmd h')
    command('q')
    screen:expect{grid=[[
    ## grid 1
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 5
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]], condition=function()
      eq({
        [2] = { win = {id=1000}, startrow = 7, startcol = 0, width = 53, height =  5 },
        [5] = { win = {id=1002}, startrow = 0, startcol = 0, width = 53, height =  6 }
      }, screen.win_position)
    end}
  end)

  describe('split', function ()
    describe('horizontally', function ()
      it('allocates grids', function ()
        command('sp')
        screen:expect([[
        ## grid 1
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ]])
      end)

      it('resizes grids', function ()
        command('sp')
        command('resize 8')
        screen:expect([[
        ## grid 1
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ]])
      end)

      it('splits vertically', function()
        command('sp')
        command('vsp')
        command('vsp')
        screen:expect{grid=[[
        ## grid 1
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          {11:[No Name]            }{12:[No Name]        [No Name]      }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
                         |
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
        ## grid 5
                          |
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
        ## grid 6
          ^                    |
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
        ]]}
        insert('hello')
        screen:expect{grid=[[
        ## grid 1
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          [6:--------------------]│[5:----------------]│[4:---------------]|
          {11:[No Name] [+]        }{12:[No Name] [+]    [No Name] [+]  }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name] [+]                                        }|
          [3:-----------------------------------------------------]|
        ## grid 2
          hello                                                |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          hello          |
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
          {1:~              }|
        ## grid 5
          hello           |
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
          {1:~               }|
        ## grid 6
          hell^o               |
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
          {1:~                   }|
        ]]}
      end)
      it('closes splits', function ()
        command('sp')
        screen:expect{grid=[[
        ## grid 1
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          [4:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {12:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                               |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ## grid 4
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ]]}
        command('q')
        screen:expect{grid=[[
        ## grid 1
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ]]}
      end)
    end)

    describe('vertically', function ()
      it('allocates grids', function ()
        command('vsp')
        screen:expect{grid=[[
        ## grid 1
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          {11:[No Name]                  }{12:[No Name]                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
          ^                          |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
      end)
      it('resizes grids', function ()
        command('vsp')
        command('vertical resize 10')
        screen:expect{grid=[[
        ## grid 1
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          [4:----------]│[2:------------------------------------------]|
          {11:<No Name]  }{12:[No Name]                                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                                    |
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
          {1:~                                         }|
        ## grid 3
                                                               |
        ## grid 4
          ^          |
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
          {1:~         }|
        ]]}
      end)
      it('splits horizontally', function ()
        command('vsp')
        command('sp')
        screen:expect{grid=[[
        ## grid 1
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          {11:[No Name]                 }│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          {12:[No Name]                  [No Name]                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 5
          ^                          |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
        insert('hello')
        screen:expect{grid=[[
        ## grid 1
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          [5:--------------------------]│[2:--------------------------]|
          {11:[No Name] [+]             }│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          {12:[No Name] [+]              [No Name] [+]             }|
          [3:-----------------------------------------------------]|
        ## grid 2
          hello                     |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
          hello                     |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 5
          hell^o                     |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
      end)
      it('closes splits', function ()
        command('vsp')
        screen:expect{grid=[[
        ## grid 1
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          [4:--------------------------]│[2:--------------------------]|
          {11:[No Name]                  }{12:[No Name]                 }|
          [3:-----------------------------------------------------]|
        ## grid 2
                                    |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ## grid 3
                                                               |
        ## grid 4
          ^                          |
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
          {1:~                         }|
        ]]}
        command('q')
        screen:expect{grid=[[
        ## grid 1
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          [2:-----------------------------------------------------]|
          {11:[No Name]                                            }|
          [3:-----------------------------------------------------]|
        ## grid 2
          ^                                                     |
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
          {1:~                                                    }|
        ## grid 3
                                                               |
        ]]}
      end)
    end)
  end)

  describe('on resize', function ()
    it('rebuilds all grids', function ()
      screen:try_resize(25, 6)
      screen:expect{grid=[[
      ## grid 1
        [2:-------------------------]|
        [2:-------------------------]|
        [2:-------------------------]|
        [2:-------------------------]|
        {11:[No Name]                }|
        [3:-------------------------]|
      ## grid 2
        ^                         |
        {1:~                        }|
        {1:~                        }|
        {1:~                        }|
      ## grid 3
                                 |
      ]]}
    end)

    it('has minimum width/height values', function()
      screen:try_resize(1, 1)
      screen:expect{grid=[[
      ## grid 1
        [2:------------]|
        {11:[No Name]   }|
        [3:------------]|
      ## grid 2
        ^            |
      ## grid 3
                    |
      ]]}

      feed('<esc>:ls')
      screen:expect{grid=[[
      ## grid 1
        [2:------------]|
        {11:[No Name]   }|
        [3:------------]|
      ## grid 2
                    |
      ## grid 3
        :ls^         |
      ]]}
    end)
  end)

  describe('grid of smaller inner size', function()
    it('is rendered correctly', function()
      screen:try_resize_grid(2, 8, 5)

      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name]                                            }|
        [3:-----------------------------------------------------]|
      ## grid 2
        ^        |
        {1:~       }|
        {1:~       }|
        {1:~       }|
        {1:~       }|
      ## grid 3
                                                             |
      ]]}
    end)
  end)

  describe('grid of bigger inner size', function()
    it('is rendered correctly', function()
      screen:try_resize_grid(2, 80, 20)

      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name]                                            }|
        [3:-----------------------------------------------------]|
      ## grid 2
        ^                                                                                |
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
        {1:~                                                                               }|
      ## grid 3
                                                             |
      ]]}
    end)
  end)


  describe('with resized grid', function()
    before_each(function()
      screen:try_resize_grid(2, 60, 20)
    end)

    it('winwidth() winheight() getwininfo() return inner width and height #19743', function()
      eq(60, funcs.winwidth(0))
      eq(20, funcs.winheight(0))
      local win_info = funcs.getwininfo(curwin().id)[1]
      eq(60, win_info.width)
      eq(20, win_info.height)
    end)

    it('gets written till grid width', function()
      insert(('a'):rep(60).."\n")
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa|
        ^                                                            |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
    end)

    it('"g$" works correctly with double-width characters and no wrapping', function()
      command('set nowrap')
      insert(('a'):rep(58) .. ('哦'):rep(3))
      feed('0')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        ^aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa哦|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
      feed('g$')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa^哦|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
    end)

    it('wraps with grid width', function()
      insert(('b'):rep(160).."\n")
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb|
        bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb|
        bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb                    |
        ^                                                            |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
      feed('2gk')
      command('setlocal cursorline cursorlineopt=screenline')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb|
        {23:^bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb}|
        bbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbbb                    |
                                                                    |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
    end)

    it('displays messages with default grid width', function()
      command('echomsg "this is a very very very very very very very very'..
        ' long message"')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name]                                            }|
        [3:-----------------------------------------------------]|
      ## grid 2
        ^                                                            |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
        this is a very very very...ry very very long message |
      ]]}
    end)

    it('creates folds with grid width', function()
      insert('this is a fold\nthis is inside fold\nthis is outside fold')
      feed('kzfgg')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        {13:^+--  2 lines: this is a fold································}|
        this is outside fold                                        |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
    end)

    it('anchored float window "bufpos"', function()
      insert(('c'):rep(1111))
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccc^c                             |
        {1:~                                                           }|
      ## grid 3
                                                             |
      ]]}
      local float_buf = meths.create_buf(false, false)
      meths.open_win(float_buf, false, {
        relative = 'win',
        win = curwin(),
        bufpos = {0, 1018},
        anchor = 'SE',
        width = 5,
        height = 5,
      })
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccccccccccccccccccccccccccccccccc|
        cccccccccccccccccccccccccccccc^c                             |
        {1:~                                                           }|
      ## grid 3
                                                             |
      ## grid 4
        {21:     }|
        {22:~    }|
        {22:~    }|
        {22:~    }|
        {22:~    }|
      ]], float_pos={
        [4] = {{id = 1001}, "SE", 2, 16, 58, true, 50};
      }}
    end)

    it('completion popup position', function()
      insert(('\n'):rep(14) .. ('foo bar '):rep(7))
      feed('A<C-X><C-N>')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
        foo bar foo bar foo bar foo bar foo bar foo bar foo bar foo^ |
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
        {1:~                                                           }|
      ## grid 3
        {7:-- Keyword Local completion (^N^P) }{15:match 1 of 2}      |
      ## grid 4
        {24: foo}|
        {21: bar}|
      ]], float_pos={
        [4] = {{id = -1}, "NW", 2, 15, 55, false, 100};
      }}
      feed('<C-E><Esc>')

      command('setlocal rightleft')
      feed('o<C-X><C-N>')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
             rab oof rab oof rab oof rab oof rab oof rab oof rab oof|
                                                                ^ oof|
        {1:                                                           ~}|
        {1:                                                           ~}|
        {1:                                                           ~}|
        {1:                                                           ~}|
      ## grid 3
        {7:-- Keyword Local completion (^N^P) }{15:match 1 of 2}      |
      ## grid 4
        {24:            oof}|
        {21:            rab}|
      ]], float_pos={
        [4] = {{id = -1}, "NW", 2, 16, 45, false, 100};
      }}
      feed('<C-E><Esc>')

      command('set wildoptions+=pum')
      feed(':sign un<Tab>')
      screen:expect{grid=[[
      ## grid 1
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        [2:-----------------------------------------------------]|
        {11:[No Name] [+]                                        }|
        [3:-----------------------------------------------------]|
      ## grid 2
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
                                                                    |
             rab oof rab oof rab oof rab oof rab oof rab oof rab oof|
                                                                    |
        {1:                                                           ~}|
        {1:                                                           ~}|
        {1:                                                           ~}|
        {1:                                                           ~}|
      ## grid 3
        :sign undefine^                                       |
      ## grid 4
        {24: undefine       }|
        {21: unplace        }|
      ]], float_pos={
        [4] = {{id = -1}, "SW", 1, 13, 5, false, 250};
      }}
    end)
  end)

  it('multiline messages scroll over windows', function()
    command('sp')
    command('vsp')
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    feed(":echoerr 'very' | echoerr 'much' | echoerr 'fail'<cr>")
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
      {14:very}                                                 |
      {14:much}                                                 |
      {14:fail}                                                 |
      {15:Press ENTER or type command to continue}^              |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    feed('<cr>')
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    command([[
      func! ErrMsg()
        for i in range(11)
          echoerr "error ".i
        endfor
      endfunc]])
    feed(":call ErrMsg()<cr>")
    screen:expect{grid=[[
    ## grid 1
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
      {14:Error detected while processing function ErrMsg:}     |
      {19:line    2:}                                           |
      {14:error 0}                                              |
      {14:error 1}                                              |
      {14:error 2}                                              |
      {14:error 3}                                              |
      {14:error 4}                                              |
      {14:error 5}                                              |
      {14:error 6}                                              |
      {14:error 7}                                              |
      {14:error 8}                                              |
      {14:error 9}                                              |
      {14:error 10}                                             |
      {15:Press ENTER or type command to continue}^              |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    feed("<c-c>")
    screen:expect{grid=[[
    ## grid 1
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      [5:--------------------------]│[4:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}
  end)

  it('handles switch tabs', function()
    command('vsp')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}


    command('tabnew')
    -- note the old grids aren't resized yet
    screen:expect{grid=[[
    ## grid 1
      {16: }{17:2}{16: [No Name] }{7: [No Name] }{12:                            }{16:X}|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('sp')
    screen:expect{grid=[[
    ## grid 1
      {16: }{17:2}{16: [No Name] }{7: }{18:2}{7: [No Name] }{12:                          }{16:X}|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 6
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('tabnext')
    screen:expect{grid=[[
    ## grid 1
      {7: }{18:2}{7: [No Name] }{16: }{17:2}{16: [No Name] }{12:                          }{16:X}|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5 (hidden)
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 6 (hidden)
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('tabnext')
    screen:expect{grid=[[
    ## grid 1
      {16: }{17:2}{16: [No Name] }{7: }{18:2}{7: [No Name] }{12:                          }{16:X}|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      [6:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      [5:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 6
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('tabnext')
    command('$tabnew')
    screen:expect{grid=[[
    ## grid 1
      {16: }{17:2}{16: [No Name]  }{17:2}{16: [No Name] }{7: [No Name] }{12:               }{16:X}|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      [7:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4 (hidden)
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5 (hidden)
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 6 (hidden)
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 7
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    command('tabclose')
    command('tabclose')
    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      {11:[No Name]                  }{12:[No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      ^                          |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}
  end)

  it('supports mouse', function()
    insert('some text\nto be clicked')
    screen:expect{grid=[[
    ## grid 1
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicke^d                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ]]}

    meths.input_mouse('left', 'press', '', 2, 0, 5)
    screen:expect{grid=[[
    ## grid 1
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some ^text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ]]}

    feed(':new<cr>')
    insert('Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo')

    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing el|
      it, sed do eiusm^o                                    |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    meths.input_mouse('left', 'press', '', 2, 1, 6)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be ^clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing el|
      it, sed do eiusmo                                    |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    meths.input_mouse('left', 'press', '', 4, 1, 4)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing el|
      it, ^sed do eiusmo                                    |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]]}

    screen:try_resize_grid(4, 80, 2)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, ^sed do eiusmo          |
      {1:~                                                                               }|
    ]]}

    meths.input_mouse('left', 'press', '', 4, 0, 64)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do ^eiusmo          |
      {1:~                                                                               }|
    ]]}

    meths.input_mouse('left', 'press', '', 1, 6, 20)
    -- TODO(bfredl): "batching" input_mouse is formally not supported yet.
    -- Normally it should work fine in async context when nvim is not blocked,
    -- but add a poke_eventloop be sure.
    poke_eventloop()
    meths.input_mouse('left', 'drag', '', 1, 4, 20)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                                            |
      to be clicked                                        |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do ^eiusmo          |
      {1:~                                                                               }|
    ]]}

    feed('<c-w><c-w><c-w>v')
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [5:--------------------------]│[2:--------------------------]|
      [5:--------------------------]│[2:--------------------------]|
      [5:--------------------------]│[2:--------------------------]|
      [5:--------------------------]│[2:--------------------------]|
      [5:--------------------------]│[2:--------------------------]|
      [5:--------------------------]│[2:--------------------------]|
      [5:--------------------------]│[2:--------------------------]|
      {11:[No Name] [+]              }{12:[No Name] [+]             }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text                 |
      to be clicked             |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                 |
      to be ^clicked             |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}

    meths.input_mouse('left', 'press', '', 1, 8, 26)
    poke_eventloop()
    meths.input_mouse('left', 'drag', '', 1, 6, 30)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {11:[No Name] [+]                  }{12:[No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be ^clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}

    command('aunmenu PopUp | vmenu PopUp.Copy y')

    funcs.setreg('"', '')
    meths.input_mouse('left', 'press', '2', 2, 1, 6)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  }{11:[No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be {20:clicke}^d         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be {20:clicked}                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    meths.input_mouse('right', 'press', '', 2, 1, 6)
    meths.input_mouse('right', 'release', '', 2, 1, 6)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  }{11:[No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be {20:clicke}^d         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be {20:clicked}                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ## grid 6
      {21: Copy }|
    ]], float_pos={
      [6] = {{id = -1}, "NW", 2, 2, 5, false, 250};
    }}
    feed('<Down><CR>')
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {12:[No Name] [+]                                        }|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  }{11:[No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be ^clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    eq('clicked', funcs.getreg('"'))

    funcs.setreg('"', '')
    meths.input_mouse('left', 'press', '2', 4, 0, 64)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do {20:eiusm}^o          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    meths.input_mouse('right', 'press', '', 4, 0, 64)
    meths.input_mouse('right', 'release', '', 4, 0, 64)
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do {20:eiusm}^o          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ## grid 6
      {21: Copy }|
    ]], float_pos={
      [6] = {{id = -1}, "NW", 4, 1, 63, false, 250};
    }}
    feed('<Down><CR>')
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem ipsum dolor sit amet, consectetur adipiscing elit, sed do ^eiusmo          |
      {1:~                                                                               }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    eq('eiusmo', funcs.getreg('"'))

    command('wincmd J')
    screen:try_resize_grid(4, 7, 10)
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       ^eiusmo|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}

    funcs.setreg('"', '')
    meths.input_mouse('left', 'press', '2', 4, 9, 1)
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       {20:eiusm}^o|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    meths.input_mouse('right', 'press', '', 4, 9, 1)
    meths.input_mouse('right', 'release', '', 4, 9, 1)
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       {20:eiusm}^o|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ## grid 6
      {21: Copy }|
    ]], float_pos={
      [6] = {{id = -1}, "SW", 4, 9, 0, false, 250};
    }}
    feed('<Down><CR>')
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       ^eiusmo|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    eq('eiusmo', funcs.getreg('"'))

    screen:try_resize_grid(4, 7, 11)
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      ^Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       eiusmo|
      {1:~      }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}

    funcs.setreg('"', '')
    meths.input_mouse('left', 'press', '2', 4, 9, 1)
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       {20:eiusm}^o|
      {1:~      }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    meths.input_mouse('right', 'press', '', 4, 9, 1)
    meths.input_mouse('right', 'release', '', 4, 9, 1)
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
      Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       {20:eiusm}^o|
      {1:~      }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ## grid 6
      {21: Copy }|
    ]], float_pos={
      [6] = {{id = -1}, "NW", 4, 10, 0, false, 250};
    }}
    feed('<Down><CR>')
    screen:expect{grid=[[
    ## grid 1
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      [5:------------------------------]│[2:----------------------]|
      {12:[No Name] [+]                  [No Name] [+]         }|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name] [+]                                        }|
      [3:-----------------------------------------------------]|
    ## grid 2
      some text             |
      to be clicked         |
      {1:~                     }|
      {1:~                     }|
      {1:~                     }|
    ## grid 3
                                                           |
    ## grid 4
      Lorem i|
      psum do|
      lor sit|
       amet, |
      consect|
      etur ad|
      ipiscin|
      g elit,|
       sed do|
       ^eiusmo|
      {1:~      }|
    ## grid 5
      some text                     |
      to be clicked                 |
      {1:~                             }|
      {1:~                             }|
      {1:~                             }|
    ]]}
    eq('eiusmo', funcs.getreg('"'))
  end)

  it('supports mouse drag with mouse=a', function()
    command('set mouse=a')
    command('vsplit')
    command('wincmd l')
    command('split')
    command('enew')
    feed('ifoo\nbar<esc>')

    meths.input_mouse('left', 'press', '', 5, 0, 0)
    poke_eventloop()
    meths.input_mouse('left', 'drag', '', 5, 1, 2)

    screen:expect{grid=[[
    ## grid 1
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│[5:--------------------------]|
      [4:--------------------------]│{11:[No Name] [+]             }|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      [4:--------------------------]│[2:--------------------------]|
      {12:[No Name]                  [No Name]                 }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 3
      {7:-- VISUAL --}                                         |
    ## grid 4
                                |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ## grid 5
      {20:foo}                       |
      {20:ba}^r                       |
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
      {1:~                         }|
    ]]}
  end)

  it('has viewport information', function()
    screen:try_resize(48, 8)
    screen:expect{grid=[[
    ## grid 1
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {11:[No Name]                                       }|
      [3:------------------------------------------------]|
    ## grid 2
      ^                                                |
      {1:~                                               }|
      {1:~                                               }|
      {1:~                                               }|
      {1:~                                               }|
      {1:~                                               }|
    ## grid 3
                                                      |
    ]], win_viewport={
      [2] = {win = { id = 1000 }, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0}
    }}
    insert([[
      Lorem ipsum dolor sit amet, consectetur
      adipisicing elit, sed do eiusmod tempor
      incididunt ut labore et dolore magna aliqua.
      Ut enim ad minim veniam, quis nostrud
      exercitation ullamco laboris nisi ut aliquip ex
      ea commodo consequat. Duis aute irure dolor in
      reprehenderit in voluptate velit esse cillum
      dolore eu fugiat nulla pariatur. Excepteur sint
      occaecat cupidatat non proident, sunt in culpa
      qui officia deserunt mollit anim id est
      laborum.]])

    screen:expect{grid=[[
    ## grid 1
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      ea commodo consequat. Duis aute irure dolor in  |
      reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
      occaecat cupidatat non proident, sunt in culpa  |
      qui officia deserunt mollit anim id est         |
      laborum^.                                        |
    ## grid 3
                                                      |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 5, botline = 11, curline = 10, curcol = 7, linecount = 11, sum_scroll_delta = 5},
    }}


    feed('<c-u>')
    screen:expect{grid=[[
    ## grid 1
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      incididunt ut labore et dolore magna aliqua.    |
      Ut enim ad minim veniam, quis nostrud           |
      exercitation ullamco laboris nisi ut aliquip ex |
      ea commodo consequat. Duis aute irure dolor in  |
      reprehenderit in voluptate velit esse cillum    |
      ^dolore eu fugiat nulla pariatur. Excepteur sint |
    ## grid 3
                                                      |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 2, botline = 9, curline = 7, curcol = 0, linecount = 11, sum_scroll_delta = 2},
    }}

    command("split")
    screen:expect{grid=[[
    ## grid 1
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {12:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
    ## grid 3
                                                      |
    ## grid 4
      ea commodo consequat. Duis aute irure dolor in  |
      reprehenderit in voluptate velit esse cillum    |
      ^dolore eu fugiat nulla pariatur. Excepteur sint |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 6, botline = 9, curline = 7, curcol = 0, linecount = 11, sum_scroll_delta = 6},
      [4] = {win = {id = 1001}, topline = 5, botline = 9, curline = 7, curcol = 0, linecount = 11, sum_scroll_delta = 5},
    }}

    feed("b")
    screen:expect{grid=[[
    ## grid 1
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {12:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
    ## grid 3
                                                      |
    ## grid 4
      ea commodo consequat. Duis aute irure dolor in  |
      reprehenderit in voluptate velit esse ^cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 6, botline = 9, curline = 7, curcol = 0, linecount = 11, sum_scroll_delta = 6},
      [4] = {win = {id = 1001}, topline = 5, botline = 9, curline = 6, curcol = 38, linecount = 11, sum_scroll_delta = 5},
    }}

    feed("2k")
    screen:expect{grid=[[
    ## grid 1
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {12:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
    ## grid 3
                                                      |
    ## grid 4
      exercitation ullamco laboris nisi ut a^liquip ex |
      ea commodo consequat. Duis aute irure dolor in  |
      reprehenderit in voluptate velit esse cillum    |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 6, botline = 9, curline = 7, curcol = 0, linecount = 11, sum_scroll_delta = 6},
      [4] = {win = {id = 1001}, topline = 4, botline = 8, curline = 4, curcol = 38, linecount = 11, sum_scroll_delta = 4},
    }}

    -- handles non-current window
    meths.win_set_cursor(1000, {1, 10})
    screen:expect{grid=[[
    ## grid 1
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {12:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      Lorem ipsum dolor sit amet, consectetur         |
      adipisicing elit, sed do eiusmod tempor         |
    ## grid 3
                                                      |
    ## grid 4
      exercitation ullamco laboris nisi ut a^liquip ex |
      ea commodo consequat. Duis aute irure dolor in  |
      reprehenderit in voluptate velit esse cillum    |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 0, botline = 3, curline = 0, curcol = 10, linecount = 11, sum_scroll_delta = 0},
      [4] = {win = {id = 1001}, topline = 4, botline = 8, curline = 4, curcol = 38, linecount = 11, sum_scroll_delta = 4},
    }}

    -- sum_scroll_delta works with folds
    feed('zfj')
    screen:expect{grid=[[
    ## grid 1
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {12:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      Lorem ipsum dolor sit amet, consectetur         |
      adipisicing elit, sed do eiusmod tempor         |
    ## grid 3
                                                      |
    ## grid 4
      {13:^+--  2 lines: exercitation ullamco laboris nisi }|
      reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 0, botline = 3, curline = 0, curcol = 10, linecount = 11, sum_scroll_delta = 0},
      [4] = {win = {id = 1001}, topline = 4, botline = 9, curline = 4, curcol = 38, linecount = 11, sum_scroll_delta = 4},
    }}

    feed('<c-e>')
    screen:expect{grid=[[
    ## grid 1
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      [4:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {12:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      Lorem ipsum dolor sit amet, consectetur         |
      adipisicing elit, sed do eiusmod tempor         |
    ## grid 3
                                                      |
    ## grid 4
      ^reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
      occaecat cupidatat non proident, sunt in culpa  |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 0, botline = 3, curline = 0, curcol = 10, linecount = 11, sum_scroll_delta = 0},
      [4] = {win = {id = 1001}, topline = 6, botline = 10, curline = 6, curcol = 0, linecount = 11, sum_scroll_delta = 5},
    }}
  end)

  it('does not crash when dragging mouse across grid boundary', function()
    screen:try_resize(48, 8)
    screen:expect{grid=[[
    ## grid 1
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {11:[No Name]                                       }|
      [3:------------------------------------------------]|
    ## grid 2
      ^                                                |
      {1:~                                               }|
      {1:~                                               }|
      {1:~                                               }|
      {1:~                                               }|
      {1:~                                               }|
    ## grid 3
                                                      |
    ]], win_viewport={
      [2] = {win = { id = 1000 }, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0}
    }}
    insert([[
      Lorem ipsum dolor sit amet, consectetur
      adipisicing elit, sed do eiusmod tempor
      incididunt ut labore et dolore magna aliqua.
      Ut enim ad minim veniam, quis nostrud
      exercitation ullamco laboris nisi ut aliquip ex
      ea commodo consequat. Duis aute irure dolor in
      reprehenderit in voluptate velit esse cillum
      dolore eu fugiat nulla pariatur. Excepteur sint
      occaecat cupidatat non proident, sunt in culpa
      qui officia deserunt mollit anim id est
      laborum.]])

    screen:expect{grid=[[
    ## grid 1
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      ea commodo consequat. Duis aute irure dolor in  |
      reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
      occaecat cupidatat non proident, sunt in culpa  |
      qui officia deserunt mollit anim id est         |
      laborum^.                                        |
    ## grid 3
                                                      |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 5, botline = 11, curline = 10, curcol = 7, linecount = 11, sum_scroll_delta = 5},
    }}

    meths.input_mouse('left', 'press', '', 1,5, 1)
    poke_eventloop()
    meths.input_mouse('left', 'drag', '', 1, 6, 1)

    screen:expect{grid=[[
    ## grid 1
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      [2:------------------------------------------------]|
      {11:[No Name] [+]                                   }|
      [3:------------------------------------------------]|
    ## grid 2
      reprehenderit in voluptate velit esse cillum    |
      dolore eu fugiat nulla pariatur. Excepteur sint |
      occaecat cupidatat non proident, sunt in culpa  |
      qui officia deserunt mollit anim id est         |
      l^aborum.                                        |
      {1:~                                               }|
    ## grid 3
      {7:-- VISUAL --}                                    |
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 6, botline = 12, curline = 10, curcol = 1, linecount = 11, sum_scroll_delta = 6},
    }}
  end)

  it('with winbar', function()
    command('split')
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0};
      [4] = {win = {id = 1001}, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0};
    }}

    -- XXX: hack to get notifications. Could use next_msg() also.
    local orig_handle_win_pos = screen._handle_win_pos
    local win_pos = {}
    function screen._handle_win_pos(self, grid, win, startrow, startcol, width, height)
      table.insert(win_pos, {grid, win, startrow, startcol, width, height})
      orig_handle_win_pos(self, grid, win, startrow, startcol, width, height)
    end

    command('setlocal winbar=very%=bar')
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      {7:very                                              bar}|
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0};
      [4] = {win = {id = 1001}, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0};
    }}
    eq({}, win_pos)

    command('setlocal winbar=')
    screen:expect{grid=[[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]], win_viewport={
      [2] = {win = {id = 1000}, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0};
      [4] = {win = {id = 1001}, topline = 0, botline = 2, curline = 0, curcol = 0, linecount = 1, sum_scroll_delta = 0};
    }}
    eq({}, win_pos)
  end)

  it('with winbar dragging statusline with mouse works correctly', function()
    meths.set_option('winbar', 'Set Up The Bars')
    command('split')
    screen:expect([[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
      {7:Set Up The Bars                                      }|
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      {7:Set Up The Bars                                      }|
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]])

    meths.input_mouse('left', 'press', '', 1, 6, 20)
    poke_eventloop()
    meths.input_mouse('left', 'drag', '', 1, 7, 20)
    screen:expect([[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
      {7:Set Up The Bars                                      }|
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      {7:Set Up The Bars                                      }|
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ]])

    meths.input_mouse('left', 'drag', '', 1, 4, 20)
    screen:expect([[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
      {7:Set Up The Bars                                      }|
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      {7:Set Up The Bars                                      }|
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
    ]])

    meths.input_mouse('left', 'press', '', 1, 12, 10)
    poke_eventloop()
    meths.input_mouse('left', 'drag', '', 1, 10, 10)
    screen:expect([[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
      [3:-----------------------------------------------------]|
    ## grid 2
      {7:Set Up The Bars                                      }|
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
                                                           |
                                                           |
    ## grid 4
      {7:Set Up The Bars                                      }|
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
    ]])
    eq(3, meths.get_option('cmdheight'))

    meths.input_mouse('left', 'drag', '', 1, 12, 10)
    screen:expect([[
    ## grid 1
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      [4:-----------------------------------------------------]|
      {11:[No Name]                                            }|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      [2:-----------------------------------------------------]|
      {12:[No Name]                                            }|
      [3:-----------------------------------------------------]|
    ## grid 2
      {7:Set Up The Bars                                      }|
                                                           |
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
      {1:~                                                    }|
    ## grid 3
                                                           |
    ## grid 4
      {7:Set Up The Bars                                      }|
      ^                                                     |
      {1:~                                                    }|
      {1:~                                                    }|
    ]])
    eq(1, meths.get_option('cmdheight'))
  end)
end)
